#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "global.h"
#include "cartridge.h"
#include "log.h"
#include "mapper.h"
#include "crc32.h"
#include "file.h"

/* iNES header
BYTE	BIT	DESC							cartridge->var
0-3		NES^Z header
4		number of 16kB PRG banks				prgbanks (8k here)
5		number of 8kB VROM banks				vrombanks (1k here)
6	0	mirroring: 1=vert, 0=hor				mirroring (1100=h, 1010=v)
	1	battery-backed RAM: 1=true, 0=false			battery
	2	512B trainer at 0x7000-0x71ff: 1=true, 0=false		trainer
	3	four-screen VRAM layout: 1=true, 0=false		mirroring=10000000, unused if not mmc3
	4-7	lower nibble of ROM mappertype				type
7	0	VS-system: 1=true, 0=false				!unused
	1-3	reserved, 0
	4-7	high nibble of ROM mappertype				type
8		number of 8kB (V)RAM? banks: 0=1, 1=1, 2=2, 3=3, etc.	!unused: mapper specific
9	0	video mode: 1=PAL, 0=NTSC				mode, unused if > 1
	1-7	reserved, 0
10-15		reserved, 0
16-...		(trainer)+PRG banks					trainer, prg[0..511]
...-end		VROM banks						vrom[0..2047] */

int cartridge_load(void)
{
	BYTE header[0x10];
	BYTE header_check[0x11];
	BYTE temp_type;
	char fn[STRING_SIZE];
	char fnb[STRING_SIZE];
	int i=TRUE;
	
	if (!file_open()) { LOG(LOG_MISC|LOG_WARNING,"ROM not found\n"); file_close(); return FALSE; }
	strcpy(fnb,file->filename);
	file_close();
	sprintf(fn,"%s\\%s.ips",file->patchdir,file->name);
	file_setfile(fn,NULL);
	if (!file_open()) {
		file_close();
		sprintf(fn,"%s\\%s.zip",file->patchdir,file->name);
		file_setfile(fn,".ips");
		if (!file_open()) { i=FALSE; file_close(); }
	}
	if (i) if (!file_patch_init()) file_patch_close();
	
	file_setfile(fnb,".nes");
	if (!file_open()) { LOG(LOG_MISC|LOG_WARNING,"ROM open error\n"); file_close(); return FALSE; }
		
	if (!file_read(header,0x10)) { LOG(LOG_MISC|LOG_WARNING,"ROM read error\n"); file_close(); return FALSE; }
	
	#if CARTRIDGE_CHECK_CRC32
	#define WRONGHEADER(x)	if (!i) LOG(LOG_MISC,"ROM with CRC32 0x%08x has wrong settings (%s",file->crc32,x); \
				else LOG(LOG_MISC,",%s",x); \
				i=TRUE; break
	i=FALSE;
	switch (file->crc32) { /* wrong mapper number */
		/* should be mapper 5 */
		case CRC32_JUST_BREED_J:
			header[6]=(header[6]&BIN8(00001111))|BIN8(01010000);
			header[7]&=BIN8(00001111);
			WRONGHEADER("mapper type");
		/* should be mapper 32 */
		case CRC32_AI_SENSEI_NO_OSHIETE_WATASHI_NO_HOSHI_J: case CRC32_AI_SENSEI_NO_OSHIETE_WATASHI_NO_HOSHI_J_A1:
			header[6]&=BIN8(00001111);
			header[7]=(header[7]&BIN8(00001111))|BIN8(00100000);
			WRONGHEADER("mapper type");
		/* should be mapper 87 */
		case CRC32_TWINBEE_J:
			header[6]=(header[6]&BIN8(00001111))|BIN8(01110000);
			header[7]=(header[7]&BIN8(00001111))|BIN8(01010000);
			WRONGHEADER("mapper type");
		/* should be mapper 118 */
		case CRC32_ALIEN_SYNDROME_U:
		case CRC32_GOAL_2_E:
		case CRC32_NES_PLAY_ACTION_FOOTBALL_U:
		case CRC32_RPG_JINSEI_GAME_J:
			header[6]=(header[6]&BIN8(00001111))|BIN8(01100000);
			header[7]=(header[7]&BIN8(00001111))|BIN8(01110000);
			WRONGHEADER("mapper type");
		/* should be mapper 144 */
		case CRC32_DEATH_RACE_U:
			header[6]&=BIN8(00001111);
			header[7]=(header[7]&BIN8(00001111))|BIN8(10010000);
			WRONGHEADER("mapper type");
		/* should be mapper 155 */
		case CRC32_TATAKAE_RAHMEN_MAN_SAKURETSU_CHOUJIN_102_GEI_J: case CRC32_TATAKAE_RAHMEN_MAN_SAKURETSU_CHOUJIN_102_GEI_J_A1:
		case CRC32_MONEY_GAME_THE_J:
			header[6]=(header[6]&BIN8(00001111))|BIN8(10110000);
			header[7]=(header[7]&BIN8(00001111))|BIN8(10010000);
			WRONGHEADER("mapper type");
		/* should be mapper 152 */
		case CRC32_ARKANOID_2_J:
		case CRC32_GEGEGE_NO_KITAROU_2_YOUKAI_GUNDAN_NO_CHOUSEN_J:
		case CRC32_POCKET_ZAURUS_JUU_OUKEN_NO_NAZO_J:
		case CRC32_SAINT_SEIYA_OUGON_DENSETSU_J:
			header[6]=(header[6]&BIN8(00001111))|BIN8(10000000);
			header[7]=(header[7]&BIN8(00001111))|BIN8(10010000);
			WRONGHEADER("mapper type");
		/* should be mapper 234 */
		case CRC32_MAXI_15_NTSC_P1: case CRC32_MAXI_15_NTSC_P1_A1:
			header[6]=(header[7]&BIN8(00001111))|BIN8(10100000);
			header[7]=(header[7]&BIN8(00001111))|BIN8(11100000);
			WRONGHEADER("mapper type");
		default: break;
	}
	switch (file->crc32) { /* wrong mirroring */		
		/* should be vertical mirroring */
		case CRC32_BLACK_BASS_2_J: case CRC32_BLACK_BASS_2_J_A1:
		case CRC32_DORAEMON_J:
		case CRC32_DRAGON_BALL_SHEN_LONG_NO_NAZO_J:
		case CRC32_DRAGON_QUEST_II_J:
		case CRC32_GILLIGANS_ISLAND_U:
		case CRC32_MISSISSIPPI_SATSUJIN_JIKEN_J: case CRC32_MISSISSIPPI_SATSUJIN_JIKEN_J_A1:
		case CRC32_MYSTERY_QUEST_U:
		case CRC32_RASAARU_ISHII_NO_CHILDS_QUEST_J:
		case CRC32_ROD_LAND_J:
		case CRC32_SHERLOCK_HOLMES_HAKUSHAKU_REIJOU_YUUKAI_JIKEN_J:
		case CRC32_SPACE_HUNTER_J:
		case CRC32_SUKEBAN_DEKA_3_J: case CRC32_SUKEBAN_DEKA_3_J_A1:
		case CRC32_SUPER_CARS_U:
		case CRC32_SUPER_DYNA_MIX_BADMINTON_J:
		case CRC32_TAKAHASHI_MEIJIN_NO_BUGUTTE_HONEY_J:
		case CRC32_TOTAL_RECALL_U:
			header[6]=(header[6]&BIN8(11110110))|1;
			WRONGHEADER("mirroring");
		/* should be horizontal mirroring */
		case CRC32_ELEVATOR_ACTION_J:
		case CRC32_EXED_EXES_J:
		case CRC32_SPY_HUNTER_U:
		case CRC32_TERRA_CRESTA_J:
		case CRC32_ZIPPY_RACE_J:
			header[6]&=BIN8(11110110);
			WRONGHEADER("mirroring");
		/* should be 4-screen */
		case CRC32_GAUNTLET_II_E:
		case CRC32_RAD_RACER_2_U:
			header[6]|=BIT(3);
			WRONGHEADER("mirroring");
		default: break;
	}
	switch (file->crc32) { /* should have battery backed SRAM (MMC6 games (and MMC1 SOROM/SUROM) too, but those are handled in the mapper module) */
		case CRC32_ADVANCED_DUNGEONS_AND_DRAGONS_DRAGONS_OF_FLAME_J:
		case CRC32_DOWNTOWN_SPECIAL_KUNIO_KUN_NO_JIDAIGEKI_DAYO_ZENIN_SHUUGOU_J:
		case CRC32_FAMICOM_WARS_J: case CRC32_FAMICOM_WARS_J_A1:
		case CRC32_GOD_SLAYER_HARUKA_TENKUU_NO_SONATA_J:
		case CRC32_JUST_BREED_J:
		case CRC32_KUJAKU_OU_2_J:
		case CRC32_NINJARA_HOI_J:
		case CRC32_OTAKU_NO_SEIZA_AN_ADVENTURE_IN_THE_OTAKU_GALAXY_J:
		case CRC32_OVERLORD_U:
		case CRC32_RPG_JINSEI_GAME_J:
		case CRC32_SD_GUNDAM_GACHAPON_SENSHI_2_CAPSULE_SENKI_J:
		case CRC32_SD_GUNDAM_GACHAPON_SENSHI_3_EIYUU_SENKI_J:
		case CRC32_SD_GUNDAM_GACHAPON_SENSHI_4_NEWTYPE_STORY_J:
		case CRC32_SD_GUNDAM_GACHAPON_SENSHI_5_BATTLE_OF_UNIVERSAL_CENTURY_J:
		case CRC32_SHIN_4_NIN_UCHI_MAHJONG_YAKUMAN_TENGOKU_J:
		case CRC32_TSUPPARI_WARS_J:
		case CRC32_ULTRAMAN_CLUB_3_J:
		case CRC32_UNINVITED_U:
		case CRC32_USHIO_TO_TORA_SHINEN_NO_DAIYOU_J:
		case CRC32_YAMAMURA_MISA_SUSPENSE_KYOUTO_ZAITEKU_SATSUJIN_JIKEN_J:
		case CRC32_YS_III_WANDERERS_FROM_YS_J:
			header[6]|=BIT(1);
			WRONGHEADER("SRAM");
		default: break;
	}
	switch (file->crc32) { /* garbage in (previously unused) bytes */
		/* bytes 7-15 */
		case CRC32_ASMIK_KUN_LAND_J:
		case CRC32_CAPTAIN_ED_J:
		case CRC32_CAT_NINDEN_TEYANDEE_J:
		case CRC32_CHOUJIN_ULTRA_BASEBALL_J:
		case CRC32_DRAGON_QUEST_II_J:
		case CRC32_HANJUKU_EIYUU:
		case CRC32_HANJUKU_EIYUU_A1:
		case CRC32_I_LOVE_SOFTBALL_A1:
		case CRC32_METRO_CROSS_J:
		case CRC32_MIRAI_SENSHI_LIOS_J:
			memset(header+7,0,9);
			WRONGHEADER("garbage");
		default: break;
	}
	#if CARTRIDGE_CHECK_CRC32_PAL
	switch (file->crc32) { /* should be PAL (" (E)" from GoodNES is autodetected as PAL) */
		case CRC32_BANANA_PRINCE_G:
		case CRC32_CHAMPIONSHIP_RALLY_A:
		case CRC32_DRAGON_BALL_LE_SECRET_DU_DRAGON_F:
		case CRC32_ELITE_PD:
		case CRC32_LEGEND_OF_ZELDA_THE_E:
		case CRC32_MARIO_BROS_CLASSIC_SERIES:
		case CRC32_MAXI_15_PAL_P1:
		case CRC32_RETROCODERS_MUSIC_ROMS_YEARS_BEHIND_PD:
		
		case CRC32_GAUNTLET_II_E:
		case CRC32_GOAL_2_E:
		case CRC32_STARTROPICS_E:
			header[9]=1;
			WRONGHEADER("PAL");
		default: break;
	}
	#endif /* CARTRIDGE_CHECK_CRC32_PAL */
	switch (file->crc32) { /* custom */
		case CRC32_FLINTSTONES_THE_THE_RESCUE_OF_DINO_AND_HOPPY_J: /* should have 32 VROM banks */
			header[5]=32;
			WRONGHEADER("VROM banks");
		case CRC32_GEGEGE_NO_KITAROU_2_YOUKAI_GUNDAN_NO_CHOUSEN_J: /* should have 8 PRG-ROM banks */
			header[4]=8;
			WRONGHEADER("PRG-ROM banks");
		case CRC32_LINUS_MUSIC_DEMO_PD: /* typo in NES^Z */
			memcpy(header,"NES\x1a",4);
			WRONGHEADER("NES signature");
		default: break;
	}
	if (i) LOG(LOG_MISC,"), solved\n");
	#endif /* CARTRIDGE_CHECK_CRC32 */
	
	if ((memcmp(header,"NES\x1a",4)!=0)|(header[7]&BIN8(00001110))|(header[9]>1)|(header[4]==0)|(memcmp(header+10,"\0\0\0\0\0\0",6)!=0)) {
		memset(header_check,0,0x11);
		for (i=0;i<0x10;i++) {
			header_check[i]=header[i];
			if (header_check[i]==0) header_check[i]='.';
			else if ((header_check[i]==255)|(header_check[i]<32)) header_check[i]=' ';
		}
		LOG(LOG_MISC|LOG_WARNING,"bad ROM header:[%s]%s",header_check,DEBUG_MISC?", ":"\n");

		/* no NES^Z in header check */
		if (memcmp(header,"NES\x1a",4)!=0) {
			LOG(LOG_MISC|LOG_WARNING,"fatal, no NES^Z signature\n");
			file_close();
			return FALSE;
		}
		/* 0 prg banks */
		else if (header[4]==0) {
			LOG(LOG_MISC|LOG_WARNING,"fatal, 0 prg banks\n");
			file_close();
			return FALSE;
		}
		
		/* Dis(kDude!) (things like DisNi330 are also available) header check/clean: Mega Man (U), Jackal (U), Rush'n Attack (U), Startropics (U), Karate Champ (U), etc. */
		else if (memcmp(header+7,"Dis",3)==0) {
			memset(header+7,0,9);
			LOG(LOG_MISC,"known, solved\n");
		}
		/* demiforce header check/clean: some translations by Neo Demiforce */
		else if (memcmp(header+7,"demiforce",9)==0) {
			memset(header+7,0,9);
			LOG(LOG_MISC,"known, solved\n");
		}
		/* fortune header check/clean: Super Contra 7 (Unl), etc(?) */
		else if (memcmp(header+9,"fortune",7)==0) {
			memset(header+9,0,7);
			LOG(LOG_MISC,"known, solved\n");
		}
		/* byKH+KEN header check/clean: Hot Slot (UE), etc(?) */
		else if (memcmp(header+7,"byKH+KEN",8)==0) {
			memset(header+7,0,9);
			LOG(LOG_MISC,"known, solved\n");
		}
		/* by PIRO header check/clean: Takahashi Meijin no Bouken Shima IV (J), Momotarou Densetsu (J), etc. */
		else if (memcmp(header+9,"by PIRO",7)==0) {
			memset(header+9,0,7);
			LOG(LOG_MISC,"known, solved\n");
		}
		/* GitM* header check/clean: M.U.L.E. (U), etc(?) */
		else if (memcmp(header+8,"GitM",4)==0) {
			memset(header+8,0,8);
			LOG(LOG_MISC,"known, solved\n");
		}
		/* something else: any bits set to 1 that should be 0 */
		else {
			LOG(LOG_MISC,"unknown, ");
			if ((header[7]&BIN8(00001110))|(header[9]>1)) LOG(LOG_MISC,"possibly harmful\n");
			else LOG(LOG_MISC,"harmless\n"); /* eg. "Ni330" or "aster" */
		}
	}
	
	temp_type=(header[7]&0xf0)|(header[6]>>4);

	/* bad header hack/test */
	/*temp_type=2;*/

	if (!mapper_supported(temp_type)) { LOG(LOG_MISC|LOG_WARNING,"iNES mapper %d is unsupported\n",temp_type); file_close(); return FALSE; }
	
	/* NES ROM header loaded successfully */
	
	cartridge_unload();
	
	cartridge->crc32=file->crc32;
	strcpy(cartridge->name,file->name);
	cartridge->prgbanks=(header[4]<<1)-1;
	cartridge->vrombanks=(header[5]<<3)-(header[5]!=0);
	cartridge->type=temp_type;
	cartridge->mirroring=((header[6]&1)?CARTRIDGE_MIRRORING_VERTICAL:CARTRIDGE_MIRRORING_HORIZONTAL)|(((header[6]>>3)&1)?CARTRIDGE_MIRRORING_4:0);
	cartridge->battery=(header[6]>>1)&1;
	cartridge->mode=(header[9]&1)&(header[9]<2);
	cartridge->trainer=(header[6]>>2)&1;
	
	/* bad header hack/test */
	/*cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL;*/
	/*cartridge->mirroring=CARTRIDGE_MIRRORING_HORIZONTAL;*/

	/* auto detect region */
	cartridge->region=CARTRIDGE_REGION_USA;
	if (strlen(cartridge->name)>3)
			for (i=0;i<strlen(cartridge->name)-3;i++) {
				if (memcmp(cartridge->name+i," (J)",4)==0) { cartridge->region=CARTRIDGE_REGION_JAPAN; break; }
				else if (memcmp(cartridge->name+i," (E)",4)==0) { cartridge->mode=TRUE; break; }
			}
	if (cartridge->mode) cartridge->region=CARTRIDGE_REGION_EUROPE;

	/* only MMC3 NES-TR1ROM/NES-TVROM supports 4-screen 'mirroring' */
	if ((cartridge->type!=4)&((cartridge->mirroring&CARTRIDGE_MIRRORING_4)==CARTRIDGE_MIRRORING_4)) {
		cartridge->mirroring=CARTRIDGE_MIRRORING_VERTICAL;
		LOG(LOG_MISC|LOG_WARNING,"board doesn't support 4-screen VRAM layout, guessed to vertical mirroring\n");
	}
	
	/* mapper type out of spec bounds, try to guess */
	if ((cartridge->type==0)&(cartridge->prgbanks>=4)) {
		if (cartridge->vrombanks) { cartridge->type=1; LOG(LOG_MISC|LOG_WARNING,"iNES mapper 0(NROM) is incorrect, guessed to iNES mapper 1\n"); }
		else { cartridge->type=2; LOG(LOG_MISC|LOG_WARNING,"iNES mapper 0(NROM) is incorrect, guessed to iNES mapper 2\n"); }
	}
	else if ((cartridge->type==0)&(cartridge->vrombanks>=8)) { cartridge->type=3; LOG(LOG_MISC|LOG_WARNING,"iNES mapper 0(NROM) is incorrect, guessed to iNES mapper 3\n"); }
	
	/* wrong prg/vrom size ? */
	if (((cartridge->vrombanks+1)>>3&1)&(cartridge->vrombanks!=7)) { LOG(LOG_MISC|LOG_WARNING,"uneven VROM banks, might cause problems"); LOG(LOG_MISC,": %"); LOG_BIN(LOG_MISC,cartridge->vrombanks,"\n"); }
	if ((cartridge->prgbanks!=1)&(cartridge->prgbanks!=3)&(cartridge->prgbanks!=7)&((cartridge->prgbanks&0xf)!=0xf)) { LOG(LOG_MISC|LOG_WARNING,"uneven PRG banks, might cause problems"); LOG(LOG_MISC,": %"); LOG_BIN(LOG_MISC,cartridge->prgbanks,"\n"); }

	
	/* allocate wram */
	if ((cartridge->wram=malloc(0x2000))==NULL) { LOG(LOG_MISC|LOG_ERROR,"WRAM allocation error!\n"); exit(1); }
	memset(cartridge->wram,0,0x2000);

	LOG(LOG_VERBOSE,"WRAM allocated\n");

	/* load trainer */
	if (cartridge->trainer) {
		if (!file_read(cartridge->wram+0x1000,0x200)) { LOG(LOG_MISC|LOG_WARNING,"ROM read error at trainer\n"); file_close(); return FALSE; }
		LOG(LOG_MISC,"trainer loaded\n");
	}

	/* load prg banks */
	LOG(LOG_MISC,"loading %dK PRG-ROM",(cartridge->prgbanks+1)<<3);
	if ((cartridge->dummy_rom=malloc(0x2000))==NULL) { LOG(LOG_MISC|LOG_ERROR,"ROM prgbanks[dummy] allocation error!\n"); exit(1); }
	memset(cartridge->dummy_rom,0,0x2000);
	for (i=0;i<=cartridge->prgbanks;i++) {
		if ((cartridge->prg[i]=malloc(0x2000))==NULL) { LOG(LOG_MISC|LOG_ERROR,"ROM prgbanks[%d] allocation error!\n",i); exit(1); }
		LOG(LOG_MISC,".");
		if (!file_read(cartridge->prg[i],0x2000)) { LOG(LOG_MISC|LOG_WARNING,"ROM read error at PRG bank %d\n",i); file_close(); return FALSE; }
	}
	LOG(LOG_MISC,"done\n");

	/* load vrom banks */
	if ((cartridge->dummy_vrom=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"ROM vrombanks[dummy] allocation error!\n"); exit(1); }
	memset(cartridge->dummy_vrom,0,0x400);
	if (cartridge->vrombanks) {
		LOG(LOG_MISC,"loading %dK VROM",cartridge->vrombanks+1);
		for (i=0;i<=cartridge->vrombanks;i++) {
			if ((cartridge->vrom[i]=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"ROM vrombanks[%d] allocation error!\n",i); exit(1); }
			if (((i&0xf)|8)==8) LOG(LOG_MISC,".");
			if (!file_read(cartridge->vrom[i],0x400)) { LOG(LOG_MISC|LOG_WARNING,"ROM read error at VROM bank %d\n",i); file_close(); return FALSE; }
		}
		LOG(LOG_MISC,"done\n");
	}
	else {
		for (i=0;i<8;i++) cartridge->vrom[i]=cartridge->dummy_vrom; /* point to dummies, just in case */
		
		/* allocate vram bank */
		for (i=0;i<8;i++) {
			if ((cartridge->vram[i]=malloc(0x400))==NULL) { LOG(LOG_MISC|LOG_ERROR,"ROM vrambanks[%d] allocation error!\n",i); exit(1); }
			memset(cartridge->vram[i],0,0x400);
		}
		LOG(LOG_VERBOSE,"VRAM allocated\n");
	}

	
	/* check loaded data size vs file size */
	i=((((cartridge->vrombanks+1)>>3)<<3)+((cartridge->prgbanks+1)<<3))*0x400+(cartridge->trainer*0x200);
	if (file->size-16!=i) LOG(LOG_MISC|LOG_WARNING,"file size-16:0x%x and raw size:0x%x don't match\n",file->size-16,i);
	
	
	LOG(LOG_MISC,"loaded '%s', %s standard\n",cartridge->name,cartridge->region==CARTRIDGE_REGION_USA?"NTSC-U":cartridge->region==CARTRIDGE_REGION_JAPAN?"NTSC-J":cartridge->region==CARTRIDGE_REGION_EUROPE?"PAL":"unknown");
	
	file_patch_close();
	file_close();
	return TRUE;
}

void cartridge_unload(void)
{
	int i;
	if (cartridge->dummy_rom!=NULL) { free(cartridge->dummy_rom); cartridge->dummy_rom=NULL; }
	if (cartridge->dummy_vrom!=NULL) {
		free(cartridge->dummy_vrom); cartridge->dummy_vrom=NULL;
		if (cartridge->vrombanks==0) for (i=0;i<8;i++) cartridge->vrom[i]=NULL;
	}
	if (cartridge->wram!=NULL) { free(cartridge->wram); cartridge->wram=NULL; }
	for (i=0;i<0x200;i++) { if (cartridge->prg[i]!=NULL) { free(cartridge->prg[i]); cartridge->prg[i]=NULL; } }
	for (i=0;i<0x800;i++) {	if (cartridge->vrom[i]!=NULL) { free(cartridge->vrom[i]); cartridge->vrom[i]=NULL; } }
	for (i=0;i<0x10;i++) { if (cartridge->vram[i]!=NULL) { free(cartridge->vram[i]); cartridge->vram[i]=NULL; } }

	LOG(LOG_VERBOSE,"unloaded cartridge\n");
}


void cartridge_battery_load(void)
{
	int size=cartridge->battery_size;
	char fn[STRING_SIZE];
	if (cartridge->battery) {
		sprintf(fn,"%s\\%s.sav",file->batterydir,cartridge->name);
		file_setfile(fn,NULL);
		if (file_open()) {
			if (size!=file->size) {
				if (size>file->size) size=file->size;
				LOG(LOG_MISC|LOG_WARNING,"battery backed SRAM source and destination sizes don't match\n"); /* no problem usually (eg. MMC5) */
			}
			if (file_read(cartridge->cur_wrambank,size)) LOG(LOG_MISC,"loaded battery backed SRAM\n");
			else LOG(LOG_MISC|LOG_WARNING,"battery backed SRAM read error!\n");
		}
		else LOG(LOG_MISC|LOG_WARNING,"couldn't load battery backed SRAM\n"); /* no problem usually */
		file_close();
		mapper_battery_custom_load();
	}
}

void cartridge_battery_save(void)
{
	char fn[STRING_SIZE];
	if (cartridge->battery) {
		if (!mapper_battery_custom_save()) {
			sprintf(fn,"%s\\%s.sav",file->batterydir,cartridge->name);
			file_setfile(fn,NULL);
			if (file_save()) {
				if (file_write(cartridge->cur_wrambank,cartridge->battery_size)) LOG(LOG_MISC,"saved battery backed SRAM\n");
				else LOG(LOG_MISC|LOG_WARNING,"battery backed SRAM write error!\n");
			}
			else LOG(LOG_MISC|LOG_WARNING,"couldn't save battery backed SRAM!\n");
			file_close();
		}
	}
}


void cartridge_init(void)
{
	if((cartridge=malloc(sizeof(Cartridge)))==NULL) { LOG(LOG_MISC|LOG_ERROR,"cartridge struct allocation error!\n"); exit(1); }
	memset(cartridge,0,sizeof(Cartridge));

	LOG(LOG_VERBOSE,"cartridge initialised\n");
}

void cartridge_clean(void)
{
	cartridge_unload();
	if (cartridge!=NULL) { free(cartridge); cartridge=NULL; }

	LOG(LOG_VERBOSE,"cartridge cleaned\n");
}
