/*
  Copyright (C) 2004 sanmaiwashi
*/

#define STRICT
#include <windows.h>
#include <mmsystem.h>
#include <dsound.h>
#include <process.h>
#include "winsound.h"
#include "winthread.h"

static struct SSoundThreadParams {
	WAVEFORMATEX *wfx;
	DWORD sb_size;
	int SoundBufferSize;
	int SoundMixInterval;
	void (*soundcallback)(void *buff, int len);
	BOOL SoundGlobalFocus;
	BOOL DirectSoundNotify;
	HANDLE hThread;
	HWND hWnd;
	volatile HANDLE hThreadEvent;
	volatile BOOL init_success;
	volatile BOOL exit;
} SoundThreadParams = {
	NULL,
	0,
	0,
	0,
	NULL,
	FALSE,
	FALSE,
	0,
	0,
	FALSE,
	FALSE,
	TRUE,
};
static HANDLE hEvent0 = FALSE;
static HANDLE hThreadEvent0 = FALSE;
static HANDLE hThreadEvent1 = FALSE;
static unsigned ThreadID;

static inline void dsRelease(HWND hWnd,
							 LPDIRECTSOUND *lpDS,
							 LPDIRECTSOUNDBUFFER *lpDSB,
							 LPDIRECTSOUNDBUFFER *lpDSBPrimary)
{
	if(*lpDS != NULL) {
		if(*lpDSB != NULL) {
	        (*lpDSB)->Stop();
			(*lpDSB)->Release();
			*lpDSB = NULL;
		}
		if(*lpDSBPrimary != NULL) {
	        (*lpDSBPrimary)->Stop();
			(*lpDSBPrimary)->Release();
			*lpDSBPrimary = NULL;
		}
        (*lpDS)->SetCooperativeLevel(hWnd, DSSCL_NORMAL);
		(*lpDS)->Release();
		*lpDS = NULL;
	}
}

static DWORD dsMinInterval(LPDIRECTSOUND lpds,
						   WAVEFORMATEX *wfx,
						   DWORD dwflags)
{
	if(lpds == NULL)
		return 0;

	LPDIRECTSOUNDBUFFER lpdsb;
	DSBUFFERDESC dsbd;

	ZeroMemory (&dsbd, sizeof (dsbd));
	dsbd.dwSize = sizeof(dsbd);
	dsbd.dwFlags = dwflags;
	dsbd.dwBufferBytes = wfx->nAvgBytesPerSec;//1000ms length buffer
	dsbd.lpwfxFormat = wfx;

    if(lpds->CreateSoundBuffer(&dsbd, &lpdsb, NULL) != DS_OK) {
        lpdsb = NULL;
		return 0;
	}

	lpdsb->Play (0, 0, DSBPLAY_LOOPING);

	DWORD play_pos = 0, write_pos = 0;
	HRESULT dErr = DS_OK;

	dErr = lpdsb->GetCurrentPosition (&play_pos, &write_pos);
	lpdsb->Stop (); 
	lpdsb->Release ();
	lpdsb = NULL;

	if (dErr != DS_OK)
		return 0;

	if (write_pos < play_pos)
		return (write_pos + dsbd.dwBufferBytes - play_pos);
	return (write_pos - play_pos);
}

static void ReleaseWaveHeader(WAVEHDR *wh, int c)
{
	if(wh) {
		for(int i = 0; i < c; i++) {
			if(wh[i].lpData) {
				HeapFree(GetProcessHeap(), 0, wh[i].lpData);
				wh[i].lpData = NULL;
			}
		}
		HeapFree(GetProcessHeap(), 0, wh);
	}
}

static WAVEHDR *ReserveWaveHeader(WAVEFORMATEX *wfx, DWORD sb_size, int c)
{
	if(wfx == NULL || sb_size == 0 || c == 0)
		return NULL;

	WAVEHDR *wh = (WAVEHDR *)HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof(WAVEHDR) * c);
	if(wh == NULL)
		return NULL;
	for(int i = 0; i < c; i++) {
		wh[i].lpData = (LPSTR)HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, sb_size);
		if(wh[i].lpData == NULL) {
			ReleaseWaveHeader(wh, c);
			return NULL;
		}
		if(wfx->wBitsPerSample == 8)
			FillMemory(wh[i].lpData, sb_size, 0x80);

		wh[i].dwBufferLength = sb_size;
		wh[i].dwBytesRecorded = 0;
		wh[i].dwUser = 0;
		wh[i].dwFlags = 0;
		wh[i].dwLoops = 1;//no loop
		wh[i].lpNext = NULL;
		wh[i].reserved = 0;
	}
	return wh;
}

static void dsThreadLoop(SSoundThreadParams *tp,
						 LPDIRECTSOUNDBUFFER lpDSB,
						 HANDLE hWaitEvent)
{
	HRESULT dErr = DS_OK;
	DWORD dsStatus = 0;
	int CurBlock = 0;
	int BlockBytes = (int)tp->sb_size;
	int nBlock = tp->SoundBufferSize;
	int play_block;
	int mix_block;
	DWORD play_pos, write_pos;
	LPVOID B1, B2;
	DWORD S1, S2;

	DWORD wait_ms = tp->SoundMixInterval * tp->SoundBufferSize;
	while (tp->exit == FALSE) {
		WaitForSingleObject(hWaitEvent, wait_ms);

		if(dsStatus == 0) {
			dErr = lpDSB->Play (0, 0, DSBPLAY_LOOPING);
			lpDSB->GetStatus (&dsStatus);
		}
		else {
			dErr = DS_OK;
		}

		if(dErr == DS_OK &&
		   lpDSB->GetCurrentPosition (&play_pos, &write_pos) == DS_OK &&
		   (play_block = (play_pos / BlockBytes)) != CurBlock) {
			
			CurBlock = play_block;
			mix_block = play_block == 0 ? (nBlock - 1) : (play_block - 1);

			dErr = lpDSB->Lock((DWORD)(BlockBytes * mix_block),
							   (DWORD)BlockBytes,
							   &B1, &S1,
							   &B2, &S2,
							   0);

			if(dErr == DSERR_BUFFERLOST) {
				lpDSB->Restore();
				dErr = lpDSB->Lock((DWORD)(BlockBytes * mix_block),
								   (DWORD)BlockBytes,
								   &B1, &S1,
								   &B2, &S2,
								   0);
			}
			if(dErr == DS_OK) {
				if(B1)
					tp->soundcallback(B1, (int)S1);
				if(B2)
					tp->soundcallback(B2, (int)S2);
				lpDSB->Unlock(B1, S1, B2, S2);
			}
			lpDSB->GetStatus(&dsStatus);
		}
	}
}

static unsigned __stdcall dsThread(void *tParams)
{
	SSoundThreadParams *tp = (SSoundThreadParams *)tParams;
	win_init_thread(tp->hWnd);

	//DirectSound
	LPDIRECTSOUND lpDS = NULL;
	LPDIRECTSOUNDBUFFER lpDSB = NULL;
	LPDIRECTSOUNDBUFFER lpDSBPrimary = NULL;
	HRESULT dErr = DS_OK;

	DSCAPS dscaps;
	ZeroMemory(&dscaps, sizeof(dscaps));
	dscaps.dwSize = sizeof(dscaps);

	//Create DirectSound
	if(DirectSoundCreate(NULL, &lpDS, NULL) != DS_OK ||
	   //Check Sound Device
	   lpDS->GetCaps(&dscaps) != S_OK) {

		dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
		SetEvent(tp->hThreadEvent);
		win_exit_thread();
		_endthreadex(0);
		return 0;
	}

	DWORD dwLocFlags;
	//ISA sound card?
	if(dscaps.dwMaxHwMixingStaticBuffers >= 1 &&
	   dscaps.dwMaxHwMixingStreamingBuffers == 0)
		dwLocFlags = DSBCAPS_STATIC;
	else
		dwLocFlags = 0;

	dErr = lpDS->SetCooperativeLevel(tp->hWnd, DSSCL_PRIORITY);
	if(dErr != DS_OK)
		dErr = lpDS->SetCooperativeLevel(tp->hWnd, DSSCL_NORMAL);
	if(dErr != DS_OK) {
		dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
		SetEvent(tp->hThreadEvent);
		win_exit_thread();
		_endthreadex(0);
		return 0;
	}

	// Create DSBPrimary
	DSBUFFERDESC dsbd;
	ZeroMemory (&dsbd, sizeof(DSBUFFERDESC));
	dsbd.dwSize = sizeof(dsbd);
	dsbd.dwFlags = DSBCAPS_PRIMARYBUFFER;
	dsbd.dwBufferBytes = 0;
	dsbd.lpwfxFormat = NULL;
	if (lpDS->CreateSoundBuffer(&dsbd, &lpDSBPrimary, NULL) != DS_OK) {
		lpDSBPrimary = NULL;
		dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
		SetEvent(tp->hThreadEvent);
		win_exit_thread();
		_endthreadex(0);
		return 0;
	}
	lpDSBPrimary->SetFormat(tp->wfx);
	lpDSBPrimary->GetFormat(tp->wfx, sizeof(WAVEFORMATEX), NULL);
	lpDSBPrimary->Play(0, 0, DSBPLAY_LOOPING);

	// Create DSB
	ZeroMemory (&dsbd, sizeof (dsbd));
	DWORD dwflags = dwLocFlags | DSBCAPS_GETCURRENTPOSITION2 |
		(tp->SoundGlobalFocus ? DSBCAPS_GLOBALFOCUS : 0) |
		(tp->DirectSoundNotify ? DSBCAPS_CTRLPOSITIONNOTIFY : 0);
	DWORD _1msc = 0;
	for (; (_1msc * 1000) < (DWORD)tp->wfx->nSamplesPerSec; _1msc++);
	DWORD buffer_bytes = _1msc * tp->SoundMixInterval * tp->wfx->nBlockAlign;
	DWORD minintarval = dsMinInterval(lpDS, tp->wfx, dwflags);
	tp->sb_size = buffer_bytes < minintarval ? minintarval : buffer_bytes;
	dsbd.dwSize = sizeof(dsbd);
	dsbd.dwFlags = dwflags;
	dsbd.dwBufferBytes = tp->sb_size * tp->SoundBufferSize;
	dsbd.lpwfxFormat = tp->wfx;
	if(lpDS->CreateSoundBuffer(&dsbd, &lpDSB, NULL) != DS_OK) {
		lpDSB = NULL;
		dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
		SetEvent(tp->hThreadEvent);
		win_exit_thread();
		_endthreadex(0);
		return 0;
	}

#if 0
	char dbg_string[256];
	wsprintf(dbg_string,
			 "min_block: %d\n min_interval: %d",
			 (int)minintarval, (int)(minintarval / (tp->wfx->nAvgBytesPerSec / 1000)));
	MessageBox(tp->hWnd,
			   dbg_string, "winsound",
			   MB_OK | MB_ICONINFORMATION);
#endif

	//OpenEvent
	HANDLE hEvent = OpenEvent(EVENT_MODIFY_STATE, FALSE, "winsound-Event");
	HANDLE hWaitEvent = OpenEvent(SYNCHRONIZE, FALSE, "winsound-Event");
	if(hEvent == FALSE || hWaitEvent == FALSE) {
		if(hWaitEvent != FALSE)
			CloseHandle(hWaitEvent);
		if(hEvent != FALSE)
			CloseHandle(hEvent);

		dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
		SetEvent(tp->hThreadEvent);
		win_exit_thread();
		_endthreadex(0);
		return 0;
	}
	
	//Use DirectSoundNotify
	if(tp->DirectSoundNotify) {
		LPDSBPOSITIONNOTIFY lpDSP = (LPDSBPOSITIONNOTIFY)GlobalAlloc(GPTR,
																	 (DWORD)(sizeof(DSBPOSITIONNOTIFY) *
																			 tp->SoundBufferSize));
		if(lpDSP == NULL) {
			CloseHandle(hWaitEvent);
			CloseHandle(hEvent);
			dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
			SetEvent(tp->hThreadEvent);
			win_exit_thread();
			_endthreadex(0);
			return 0;
		}

		int i;
		for(i = 0; i < tp->SoundBufferSize; i++) {
			lpDSP[i].dwOffset = tp->sb_size * i;
			lpDSP[i].hEventNotify = hEvent;
		}

		LPDIRECTSOUNDNOTIFY lpDSN = NULL;
		if(lpDSB->QueryInterface(IID_IDirectSoundNotify, (LPVOID*)&lpDSN) != S_OK ||
		   lpDSN->SetNotificationPositions(tp->SoundBufferSize, (LPCDSBPOSITIONNOTIFY)lpDSP) != DS_OK) {

			if(lpDSN) {
				lpDSN->Release();
				lpDSN = NULL;
			}
			lpDSB->Stop();
			lpDSB->Release();
			lpDSB = NULL;

			//Try DSBCAPS_LOCSOFTWARE
			dwflags |= DSBCAPS_LOCSOFTWARE;
			minintarval = dsMinInterval(lpDS, tp->wfx, dwflags);
			tp->sb_size = buffer_bytes < minintarval ? minintarval : buffer_bytes;

			ZeroMemory (&dsbd, sizeof (dsbd));
			dsbd.dwSize = sizeof(dsbd);
			dsbd.dwFlags = dwflags;
			dsbd.dwBufferBytes = tp->sb_size * tp->SoundBufferSize;
			dsbd.lpwfxFormat = tp->wfx;

			for(i = 0; i < tp->SoundBufferSize; i++) {
				lpDSP[i].dwOffset = tp->sb_size * i;
				lpDSP[i].hEventNotify = hEvent;
			}

			if(lpDS->CreateSoundBuffer(&dsbd, &lpDSB, NULL) != DS_OK ||
			   lpDSB->QueryInterface(IID_IDirectSoundNotify, (LPVOID*)&lpDSN) != S_OK ||
			   lpDSN->SetNotificationPositions(tp->SoundBufferSize, (LPCDSBPOSITIONNOTIFY)lpDSP) != DS_OK) {
				if(lpDSN) {
					lpDSN->Release();
					lpDSN = NULL;
				}
				GlobalFree((HGLOBAL)lpDSP);
				CloseHandle(hWaitEvent);
				CloseHandle(hEvent);
				dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
				SetEvent(tp->hThreadEvent);
				win_exit_thread();
				_endthreadex(0);
				return 0;
			}
		}

		tp->init_success = TRUE;
		SetEvent(tp->hThreadEvent);

		dsThreadLoop(tp, lpDSB, hWaitEvent);

		lpDSN->Release();
		lpDSN = NULL;
		GlobalFree((HGLOBAL)lpDSP);
	}
	//Use timeSetEvent()
	else {
		TIMECAPS tc;
		if(timeGetDevCaps(&tc, sizeof(TIMECAPS))== TIMERR_NOERROR)
			timeBeginPeriod(tc.wPeriodMin);
		else
			tc.wPeriodMin = 0;

		UINT interval = buffer_bytes < minintarval ?
			(UINT)(minintarval / (tp->wfx->nAvgBytesPerSec / 1000)) :
			(UINT)tp->SoundMixInterval;
		interval /= 3;

		UINT TimerID;
		if((TimerID = timeSetEvent((UINT)(interval < tc.wPeriodMin ? tc.wPeriodMin : interval),
								   0,
								   (LPTIMECALLBACK)hEvent,
								   0,
								   TIME_PERIODIC | TIME_CALLBACK_EVENT_SET)) == (UINT)NULL) {
		
			if(hWaitEvent != FALSE)
				CloseHandle(hWaitEvent);
			if(hEvent != FALSE)
				CloseHandle(hEvent);
			if(tc.wPeriodMin)
				timeEndPeriod(tc.wPeriodMin);

			dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);
			SetEvent(tp->hThreadEvent);
			win_exit_thread();
			_endthreadex(0);
			return 0;
		}

		tp->init_success = TRUE;
		SetEvent(tp->hThreadEvent);

		dsThreadLoop(tp, lpDSB, hWaitEvent);

		timeKillEvent(TimerID);
		if(tc.wPeriodMin)
			timeEndPeriod(tc.wPeriodMin);
	}

	// Close Event handle
	CloseHandle(hWaitEvent);
	CloseHandle(hEvent);

	dsRelease(tp->hWnd, &lpDS, &lpDSB, &lpDSBPrimary);

	win_exit_thread();
	_endthreadex(0);
	return 0;
}

static unsigned __stdcall waveOutThread(void *tParams)
{
	SSoundThreadParams *tp = (SSoundThreadParams *)tParams;
	win_init_thread(tp->hWnd);

	HANDLE hEvent = OpenEvent(EVENT_MODIFY_STATE, FALSE, "winsound-Event");
	HANDLE hWaitEvent = OpenEvent(SYNCHRONIZE, FALSE, "winsound-Event");

	// Calculate sound buffer size
	DWORD _1msc = 0;
	for (; (_1msc * 1000) < (DWORD)tp->wfx->nSamplesPerSec; _1msc++);
	tp->sb_size = _1msc * tp->SoundMixInterval * tp->wfx->nBlockAlign;

	WAVEHDR *wh = ReserveWaveHeader(tp->wfx, tp->sb_size, tp->SoundBufferSize);
	LPSTR *wb = (LPSTR *)GlobalAlloc(GPTR, (DWORD)(sizeof(LPSTR) * tp->SoundBufferSize));
	HWAVEOUT hwo;

	if(hEvent == FALSE ||
	   hWaitEvent == FALSE ||
	   wh == NULL ||
	   wb == NULL ||
	   /* waveOut open */
	   MMSYSERR_NOERROR != waveOutOpen(&hwo,
									   WAVE_MAPPER,
									   tp->wfx,
									   (DWORD)hEvent,
									   0,
									   CALLBACK_EVENT)) {

		if(hWaitEvent != FALSE)
			CloseHandle(hWaitEvent);
		if(hEvent != FALSE)
			CloseHandle(hEvent);
		if(wh)
			ReleaseWaveHeader(wh, tp->SoundBufferSize);
		if(wb)
			GlobalFree((HGLOBAL)wb);

		SetEvent(tp->hThreadEvent);
		win_exit_thread();
		_endthreadex(0);
		return 0;
	}
	int i;
	for(i = 0; i < tp->SoundBufferSize; i++)
		wb[i] = wh[i].lpData;

	//MM_WOM_OPEN
	WaitForSingleObject(hWaitEvent, INFINITE);

	tp->init_success = TRUE;
	SetEvent(tp->hThreadEvent);

	waveOutPause(hwo);
	for(i = 0; i < tp->SoundBufferSize; i++) {
		waveOutPrepareHeader(hwo, &wh[i], sizeof(WAVEHDR));
		waveOutWrite(hwo, &wh[i], sizeof(WAVEHDR));
	}
	waveOutRestart(hwo);

	DWORD wait_ms = tp->SoundMixInterval * tp->SoundBufferSize;
	i = 0;
	while (tp->exit == FALSE) {
		WaitForSingleObject(hWaitEvent, wait_ms);
		while(wh[i].dwFlags & WHDR_DONE) {
			waveOutUnprepareHeader(hwo, &wh[i], sizeof(WAVEHDR));

			tp->soundcallback(wb[i], (int)tp->sb_size);
			wh[i].lpData = wb[i];
			wh[i].dwBufferLength = tp->sb_size;
			wh[i].dwBytesRecorded = 0;
			wh[i].dwUser = 0;
			wh[i].dwFlags = 0;
			wh[i].dwLoops = 1;//no loop
			wh[i].lpNext = NULL;
			wh[i].reserved = 0;

			waveOutPrepareHeader(hwo, &wh[i], sizeof(WAVEHDR));
			waveOutWrite(hwo, &wh[i], sizeof(WAVEHDR));
			i = (i + 1) % tp->SoundBufferSize;
		}
	}

	//Wait that all buffers are set WHDR_DONE flags.
	int wh_c = 0;
	while(wh_c < tp->SoundBufferSize) {
		WaitForSingleObject(hWaitEvent, wait_ms);
		while(wh[i].dwFlags & WHDR_DONE &&
			  wh_c < tp->SoundBufferSize) {
			i = (i + 1) % tp->SoundBufferSize;
			wh_c++;
		}
	}

	//waveOutClose
	while(waveOutReset(hwo) != MMSYSERR_NOERROR);
	for(i = 0; i < tp->SoundBufferSize; i++) {
		waveOutUnprepareHeader(hwo, &wh[i], sizeof(WAVEHDR));
	}
	while(waveOutClose(hwo) != MMSYSERR_NOERROR);

	//MM_WOM_CLOSE
	WaitForSingleObject(hWaitEvent, INFINITE);

	//free Heap
	for(i = 0; i < tp->SoundBufferSize; i++)
		wh[i].lpData = wb[i];
	GlobalFree((HGLOBAL)wb);
	ReleaseWaveHeader(wh, tp->SoundBufferSize);

	// Close Event handle
	CloseHandle(hWaitEvent);
	CloseHandle(hEvent);

	win_exit_thread();
	_endthreadex(0);
	return 0;
}

void WinSoundDeinitialize()
{
	if(SoundThreadParams.exit == FALSE) {
		SetThreadPriority(SoundThreadParams.hThread, THREAD_PRIORITY_NORMAL);
		SoundThreadParams.exit = TRUE;
		WaitForSingleObject(SoundThreadParams.hThread, INFINITE);
		CloseHandle(SoundThreadParams.hThread);
		SoundThreadParams.hThread = 0;
	}
	if(SoundThreadParams.hThreadEvent != FALSE) {
		CloseHandle(SoundThreadParams.hThreadEvent);
		SoundThreadParams.hThreadEvent = FALSE;
	}
	if(hThreadEvent1 != FALSE) {
		CloseHandle(hThreadEvent1);
		hThreadEvent1 = FALSE;
	}
	if(hThreadEvent0 != FALSE) {
		CloseHandle(hThreadEvent0);
		hThreadEvent0 = FALSE;
	}
	if(hEvent0 != FALSE) {
		CloseHandle(hEvent0);
		hEvent0 = FALSE;
	}
}

BOOL WinSetupSound(long rate, BOOL sixteen_bit, BOOL stereo,
				   int SoundBufferSize, int SoundMixInterval,
				   void (*SoundCallback)(void *buff, int len),
				   WAVEFORMATEX *wfx, DWORD *buffersize,
				   HWND hWnd, BOOL DSuse,
				   BOOL SoundGlobalFocus, BOOL DirectSoundNotify)
{
	if(!hWnd || !SoundCallback || rate == 0 || !wfx)
		return false;

	WinSoundDeinitialize();

	hEvent0 = CreateEvent(NULL, FALSE, FALSE, "winsound-Event");
	hThreadEvent0 = CreateEvent(NULL, FALSE, FALSE, "winsound-ThreadEvent");

	if(hEvent0 == FALSE || hThreadEvent0 == FALSE) {
		WinSoundDeinitialize();
		return false;
	}

	hThreadEvent1 = OpenEvent(SYNCHRONIZE, FALSE, "winsound-ThreadEvent");
	SoundThreadParams.hThreadEvent = OpenEvent(EVENT_MODIFY_STATE,
											   FALSE,
											   "winsound-ThreadEvent");

	if(hThreadEvent1 == FALSE || SoundThreadParams.hThreadEvent == FALSE) {
		WinSoundDeinitialize();
		return false;
	}

	FillMemory((PVOID)wfx, sizeof(WAVEFORMATEX), 0);

	wfx->wFormatTag = WAVE_FORMAT_PCM;
	wfx->nChannels = stereo ? 2 : 1;
	wfx->nSamplesPerSec = (DWORD)rate;
	wfx->nBlockAlign = (sixteen_bit ? 2 : 1) * (stereo ? 2 : 1);
	wfx->wBitsPerSample = sixteen_bit ? 16 : 8;
	wfx->nAvgBytesPerSec = wfx->nSamplesPerSec * wfx->nBlockAlign;
	wfx->cbSize = 0;

	//Set thread params.
	SoundThreadParams.wfx = wfx;
	SoundThreadParams.sb_size = 0xffffffff;//This value will be store
	SoundThreadParams.SoundBufferSize = SoundBufferSize;
	SoundThreadParams.SoundMixInterval = SoundMixInterval;
	SoundThreadParams.soundcallback = SoundCallback;
	SoundThreadParams.SoundGlobalFocus = SoundGlobalFocus;
	SoundThreadParams.DirectSoundNotify = DirectSoundNotify;
	SoundThreadParams.init_success = FALSE;
	SoundThreadParams.exit = FALSE;
	SoundThreadParams.hWnd = hWnd;

	SoundThreadParams.hThread = (HANDLE)_beginthreadex(NULL,
													   0,
													   DSuse ? &dsThread : &waveOutThread,
													   &SoundThreadParams,
													   0,
													   &ThreadID);

	if((int)SoundThreadParams.hThread == 0) {
		SoundThreadParams.exit = TRUE;
		WinSoundDeinitialize();
		return false;
	}

	// init_success?
	WaitForSingleObject(hThreadEvent1, INFINITE);
	if(SoundThreadParams.init_success == FALSE) {
		//Wait thread end.
		WaitForSingleObject(SoundThreadParams.hThread, INFINITE);
		SoundThreadParams.exit = TRUE;
		CloseHandle(SoundThreadParams.hThread);
		SoundThreadParams.hThread = 0;
		WinSoundDeinitialize();
		return false;
	}
	SetThreadPriority(SoundThreadParams.hThread, THREAD_PRIORITY_TIME_CRITICAL);
	*buffersize = SoundThreadParams.sb_size * SoundBufferSize;
	return true;
}

