#define STRICT
#include <windows.h>
#include <mmsystem.h>
#include <dinput.h>
#include "wininput.h"
#include <math.h>

#define NUM_JOYSTICKS 16

typedef struct {
	BOOL Left;
	BOOL Right;
	BOOL Up;
	BOOL Down;
	BOOL PovLeft;
	BOOL PovRight;
	BOOL PovUp;
	BOOL PovDown;
	BOOL ZUp;
	BOOL ZDown;
	BOOL RUp;
	BOOL RDown;
	BOOL UUp;
	BOOL UDown;
	BOOL VUp;
	BOOL VDown;
	BOOL Button[32];
} SJoyState;

typedef struct {
	BOOL        m_bUseJoystick;
	UINT        m_uJoyID;

	JOYINFOEX   m_JoyInfo;
	JOYCAPS     m_JoyCaps;

	UINT        m_nDeadZone;
	DWORD       m_dwXDZmin;
	DWORD       m_dwXDZmax;
	DWORD       m_dwYDZmin;
	DWORD       m_dwYDZmax;
	DWORD       m_dwZDZmin;
	DWORD       m_dwZDZmax;
	DWORD       m_dwRDZmin;
	DWORD       m_dwRDZmax;
	DWORD       m_dwUDZmin;
	DWORD       m_dwUDZmax;
	DWORD       m_dwVDZmin;
	DWORD       m_dwVDZmax;

	double      m_dXScale;
	double      m_dYScale;
	SJoyState JoyState;
} mmJoystickInfo;

typedef struct {
	UINT                    m_nNumJoysticks;
	mmJoystickInfo          m_Joy[NUM_JOYSTICKS];
} mmJoystick_private;

typedef struct {
	byte m_key[256];
	BOOL KeyboardStatesOK;
} mmKeyboard_private;

#define MAX_PHYSICAL_JOYSTICKS 20
#define MAX_AXES 20

typedef struct {
	GUID guid;
	int offset;
} axis_type;

typedef struct {
	BOOL use_joystick;
	GUID guidDevice;
	LPDIRECTINPUTDEVICE2 did;
	DWORD num_axes;
	axis_type axes[MAX_AXES];
	DWORD num_pov;
	DWORD num_buttons;
	DIJOYSTATE dijs;
	SJoyState JoyState;
} dijoystick_type;

typedef struct {
	DWORD num_joysticks;
	dijoystick_type joysticks[MAX_PHYSICAL_JOYSTICKS];
} diJoystick_private;

typedef struct {
	LPDIRECTINPUTDEVICE2  m_didKeyboard;
	byte m_key[256];
	BOOL KeyboardStatesOK;
} diKeyboard_private;

typedef struct {
	LPDIRECTINPUTDEVICE2  m_didMouse;
	DIMOUSESTATE dims;
} diMouse_private;

static LPDIRECTINPUT lpDI = NULL;

static unsigned char DirectInputInitialize(HWND hWnd)
{
	if(!hWnd)
		return FALSE;
	if(lpDI != NULL)
		return TRUE;
	if(DI_OK != DirectInputCreate((HINSTANCE)GetWindowLong(hWnd, GWL_HINSTANCE),
								  DIRECTINPUT_VERSION,
								  &lpDI,
								  NULL) &&
	   DI_OK != DirectInputCreate((HINSTANCE)GetWindowLong(hWnd, GWL_HINSTANCE),
								  0x0300,
								  &lpDI,
								  NULL)) {
		lpDI = NULL;
		return FALSE;
	}
	return TRUE;
}

static void DirectInputDeInitialize()
{
	if(lpDI) {
		lpDI->Release();
		lpDI = NULL;
	}
}

static mmJoystick_private mmJoystick;
static mmKeyboard_private mmKeyboard;
static diJoystick_private diJoystick;
static diKeyboard_private diKeyboard;
//static diMouse_private diMouse;

#define OSD_ANALOGMAX       ( 127.0)
#define OSD_ANALOGMIN       (-128.0)
#ifdef PI
#undef PI
#endif
#define PI 3.1415926534

static DWORD Joystick_DeadZoneMin(DWORD dwMin, DWORD dwMax, UINT nDeadZone)
{
    return (DWORD)((dwMax - dwMin) * ((100.0 - nDeadZone) / 200.0));
}

static DWORD Joystick_DeadZoneMax(DWORD dwMin, DWORD dwMax, UINT nDeadZone)
{
    return (DWORD)((dwMax - dwMin) * ((100.0 + nDeadZone) / 200.0));
}

#define SetDeadZone(Joy, Axis)                                                       \
            Joy.m_dw##Axis##DZmin = Joystick_DeadZoneMin(Joy.m_JoyCaps.w##Axis##min, \
                                                Joy.m_JoyCaps.w##Axis##max,          \
                                                Joy.m_nDeadZone);                    \
            Joy.m_dw##Axis##DZmax = Joystick_DeadZoneMax(Joy.m_JoyCaps.w##Axis##min, \
                                                Joy.m_JoyCaps.w##Axis##max,          \
                                                Joy.m_nDeadZone);

static void mmJoystick_poll_joysticks(void)
{
	int i = NUM_JOYSTICKS;

	if (mmJoystick.m_nNumJoysticks == 0)
		return;

	while (i--)	{
		if (mmJoystick.m_Joy[i].m_bUseJoystick == TRUE)
			joyGetPosEx(mmJoystick.m_Joy[i].m_uJoyID, &mmJoystick.m_Joy[i].m_JoyInfo);
	}
}

static void mmClearJoyState(mmJoystickInfo *joystick)
{
	FillMemory((PVOID)&joystick->JoyState, sizeof(joystick->JoyState), 0);
}

static int mmJoystick_init()
{
	int         i;
	MMRESULT    mmResult;

	FillMemory((PVOID)&mmJoystick, sizeof(mmJoystick_private), 0);

	for (i = 0; i < NUM_JOYSTICKS; i++)	{
		mmJoystick.m_Joy[i].m_bUseJoystick = TRUE;
		mmJoystick.m_Joy[i].m_nDeadZone    = 75;    /* No way to set dead zone right now */
		mmJoystick.m_Joy[i].m_uJoyID       = (JOYSTICKID1 + i);
	}

	/* User turned off joy option or a joy driver is not installed. */
	if (joyGetNumDevs() == 0) {
		for(i = 0; i < NUM_JOYSTICKS; i++)
			mmJoystick.m_Joy[i].m_bUseJoystick = FALSE;
	}

	for (i = 0; i < NUM_JOYSTICKS; i++)	{
		if (mmJoystick.m_Joy[i].m_bUseJoystick == FALSE)
			continue;

		/* Determine if JOYSTICKID[1-16] is plugged in. */
		mmJoystick.m_Joy[i].m_JoyInfo.dwSize  = sizeof(JOYINFOEX);
		mmJoystick.m_Joy[i].m_JoyInfo.dwFlags = JOY_RETURNBUTTONS | JOY_RETURNX | JOY_RETURNY;

		mmResult = joyGetPosEx(mmJoystick.m_Joy[i].m_uJoyID, &mmJoystick.m_Joy[i].m_JoyInfo);

		if (mmResult == JOYERR_NOERROR) {
			joyGetDevCaps(mmJoystick.m_Joy[i].m_uJoyID, &mmJoystick.m_Joy[i].m_JoyCaps, sizeof(JOYCAPS));
			SetDeadZone(mmJoystick.m_Joy[i], X);
			SetDeadZone(mmJoystick.m_Joy[i], Y);

			/* Currently only analog support for X and Y axis. */
			mmJoystick.m_Joy[i].m_dXScale = ((OSD_ANALOGMAX - OSD_ANALOGMIN) /
											 (mmJoystick.m_Joy[i].m_JoyCaps.wXmax - mmJoystick.m_Joy[i].m_JoyCaps.wXmin));
			mmJoystick.m_Joy[i].m_dYScale = ((OSD_ANALOGMAX - OSD_ANALOGMIN) /
											 (mmJoystick.m_Joy[i].m_JoyCaps.wYmax - mmJoystick.m_Joy[i].m_JoyCaps.wYmin));

			/* Check for other axis */
			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASZ) {
				mmJoystick.m_Joy[i].m_JoyInfo.dwFlags |= JOY_RETURNZ;
				SetDeadZone(mmJoystick.m_Joy[i], Z);
			}

			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASR) {
				mmJoystick.m_Joy[i].m_JoyInfo.dwFlags |= JOY_RETURNR;
				SetDeadZone(mmJoystick.m_Joy[i], R);
			}

			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASU) {
				mmJoystick.m_Joy[i].m_JoyInfo.dwFlags |= JOY_RETURNU;
				SetDeadZone(mmJoystick.m_Joy[i], U);
			}

			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASV) {
				mmJoystick.m_Joy[i].m_JoyInfo.dwFlags |= JOY_RETURNV;
				SetDeadZone(mmJoystick.m_Joy[i], V);
			}

			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASPOV
			&&  mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_POV4DIR) {
				mmJoystick.m_Joy[i].m_JoyInfo.dwFlags |= JOY_RETURNPOV;
			}

			mmJoystick.m_nNumJoysticks++;
		}
		else if (mmResult == JOYERR_UNPLUGGED) {
			mmJoystick.m_Joy[i].m_bUseJoystick = FALSE;
		}
		else {
			/* Some other error with the joystick. Don't use it. */
			mmJoystick.m_Joy[i].m_bUseJoystick = FALSE;
		}
	}

	/* Initialize JOYINFOEX */      
	mmJoystick_poll_joysticks();

	return 0;
}

static WORD GetdiJoyButtonCode();
static WORD GetmmJoyButtonCode();
WORD (*GetJoyButtonCode)() = GetmmJoyButtonCode;

static WORD GetdiKeyButtonCode();
static WORD GetmmKeyButtonCode();
WORD (*GetKeyButtonCode)() = GetmmKeyButtonCode;

static void GetmmJoybuttonStates();
static void GetdiJoybuttonStates();
void (*GetJoybuttonStates)() = GetmmJoybuttonStates;

static void GetmmKeybuttonStates();
static void GetdiKeybuttonStates();
void (*GetKeybuttonStates)() = GetmmKeybuttonStates;

static BOOL GetJoyKeybuttonStatemm (WORD KeyIdent);
static BOOL GetJoyKeybuttonStatedi (WORD KeyIdent);
BOOL (*GetJoyKeybuttonState)(WORD KeyIdent) = GetJoyKeybuttonStatemm;

static BOOL GetJoyKeybuttonStringmm(WORD KeyIdent, char *strbuf, int strbuflen);
static BOOL GetJoyKeybuttonStringdi(WORD KeyIdent, char *strbuf, int strbuflen);
BOOL (*GetJoyKeybuttonString)(WORD KeyIdent, char *strbuf, int strbuflen) = GetJoyKeybuttonStringmm;

static void SwitchInputProc(BOOL DIuse)
{
	if(DIuse) {
		GetJoyButtonCode = GetdiJoyButtonCode;
		GetKeyButtonCode = GetdiKeyButtonCode;
		GetJoybuttonStates = GetdiJoybuttonStates;
		GetKeybuttonStates = GetdiKeybuttonStates;
		GetJoyKeybuttonState = GetJoyKeybuttonStatedi;
		GetJoyKeybuttonString = GetJoyKeybuttonStringdi;
	}
	else {
		GetJoyButtonCode = GetmmJoyButtonCode;
		GetKeyButtonCode = GetmmKeyButtonCode;
		GetJoybuttonStates = GetmmJoybuttonStates;
		GetKeybuttonStates = GetmmKeybuttonStates;
		GetJoyKeybuttonState = GetJoyKeybuttonStatemm;
		GetJoyKeybuttonString = GetJoyKeybuttonStringmm;
	}
	return;
}

static BOOL GetJoyKeybuttonStatemm (WORD KeyIdent)
{
	if (KeyIdent & 0x8000) {
		int i = (KeyIdent >> 8) & 15;

		switch (KeyIdent & 0xff) {
			case 0: return mmJoystick.m_Joy[i].JoyState.Left;
			case 1: return mmJoystick.m_Joy[i].JoyState.Right;
			case 2: return mmJoystick.m_Joy[i].JoyState.Up;
			case 3: return mmJoystick.m_Joy[i].JoyState.Down;
			case 4: return mmJoystick.m_Joy[i].JoyState.PovLeft;
			case 5: return mmJoystick.m_Joy[i].JoyState.PovRight;
			case 6: return mmJoystick.m_Joy[i].JoyState.PovUp;
			case 7: return mmJoystick.m_Joy[i].JoyState.PovDown;
			case 41:return mmJoystick.m_Joy[i].JoyState.ZUp;
			case 42:return mmJoystick.m_Joy[i].JoyState.ZDown;
			case 43:return mmJoystick.m_Joy[i].JoyState.RUp;
			case 44:return mmJoystick.m_Joy[i].JoyState.RDown;
			case 45:return mmJoystick.m_Joy[i].JoyState.UUp;
			case 46:return mmJoystick.m_Joy[i].JoyState.UDown;
			case 47:return mmJoystick.m_Joy[i].JoyState.VUp;
			case 48:return mmJoystick.m_Joy[i].JoyState.VDown;

			default:
				if ((KeyIdent & 0xff) > 40)
					return false;
				return mmJoystick.m_Joy[i].JoyState.Button [(KeyIdent & 0xff) - 8];
		}
	}
	return (mmKeyboard.KeyboardStatesOK && (mmKeyboard.m_key[KeyIdent & 0xff] & 0x80) != 0);
}

static void GetmmJoybuttonStates()
{
	int i;
	
	for (i = 0; i < NUM_JOYSTICKS; i++) {
		mmClearJoyState(&mmJoystick.m_Joy[i]);
		if (mmJoystick.m_Joy[i].m_bUseJoystick) {
			if (joyGetPosEx(mmJoystick.m_Joy[i].m_uJoyID, &mmJoystick.m_Joy[i].m_JoyInfo) != JOYERR_NOERROR) {
				mmJoystick.m_Joy[i].m_bUseJoystick = false;
				continue;
			}
			
			for (unsigned long B = 0; B < 32; B ++) {
				mmJoystick.m_Joy[i].JoyState.Button[B] = (mmJoystick.m_Joy[i].m_JoyInfo.dwButtons & (1 << B)) != 0;
			}

			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASPOV
			&&  mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_POV4DIR) {
				switch (mmJoystick.m_Joy[i].m_JoyInfo.dwPOV) {
					case JOY_POVFORWARD:
						mmJoystick.m_Joy[i].JoyState.PovUp = true;
						break;

					case 4500:
						mmJoystick.m_Joy[i].JoyState.PovUp = true;
						mmJoystick.m_Joy[i].JoyState.PovRight = true;
						break;

					case JOY_POVRIGHT:
						mmJoystick.m_Joy[i].JoyState.PovRight = true;
						break;

					case 13500:
						mmJoystick.m_Joy[i].JoyState.PovRight = true;
						mmJoystick.m_Joy[i].JoyState.PovDown = true;
						break;

					case JOY_POVBACKWARD:
						mmJoystick.m_Joy[i].JoyState.PovDown = true;
						break;

					case 22500:
						mmJoystick.m_Joy[i].JoyState.PovDown = true;
						mmJoystick.m_Joy[i].JoyState.PovLeft = true;
						break;

					case JOY_POVLEFT:
						mmJoystick.m_Joy[i].JoyState.PovLeft = true;
						break;

					case 31500:
						mmJoystick.m_Joy[i].JoyState.PovUp = true;
						mmJoystick.m_Joy[i].JoyState.PovLeft = true;
						break;

					default:
						break;
				}
			}
			if(mmJoystick.m_Joy[i].m_JoyInfo.dwXpos <= mmJoystick.m_Joy[i].m_dwXDZmin &&
				mmJoystick.m_Joy[i].m_dwXDZmin != mmJoystick.m_Joy[i].m_dwXDZmax)
				mmJoystick.m_Joy[i].JoyState.Left = TRUE;
			else if(mmJoystick.m_Joy[i].m_dwXDZmax <= mmJoystick.m_Joy[i].m_JoyInfo.dwXpos &&
				mmJoystick.m_Joy[i].m_dwXDZmin != mmJoystick.m_Joy[i].m_dwXDZmax)
				mmJoystick.m_Joy[i].JoyState.Right = TRUE;
			if(mmJoystick.m_Joy[i].m_JoyInfo.dwYpos <= mmJoystick.m_Joy[i].m_dwYDZmin &&
				mmJoystick.m_Joy[i].m_dwYDZmin != mmJoystick.m_Joy[i].m_dwYDZmax)
				mmJoystick.m_Joy[i].JoyState.Up = TRUE;
			else if(mmJoystick.m_Joy[i].m_dwYDZmax <= mmJoystick.m_Joy[i].m_JoyInfo.dwYpos &&
				mmJoystick.m_Joy[i].m_dwYDZmin != mmJoystick.m_Joy[i].m_dwYDZmax)
				mmJoystick.m_Joy[i].JoyState.Down = TRUE;

			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASZ) {
				if(mmJoystick.m_Joy[i].m_JoyInfo.dwZpos <= mmJoystick.m_Joy[i].m_dwZDZmin &&
					mmJoystick.m_Joy[i].m_dwZDZmin != mmJoystick.m_Joy[i].m_dwZDZmax)
					mmJoystick.m_Joy[i].JoyState.ZUp = TRUE;
				else if(mmJoystick.m_Joy[i].m_dwZDZmax <= mmJoystick.m_Joy[i].m_JoyInfo.dwZpos &&
					mmJoystick.m_Joy[i].m_dwZDZmin != mmJoystick.m_Joy[i].m_dwZDZmax)
					mmJoystick.m_Joy[i].JoyState.ZDown = TRUE;
			}
			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASR) {
				if(mmJoystick.m_Joy[i].m_JoyInfo.dwRpos <= mmJoystick.m_Joy[i].m_dwRDZmin &&
					mmJoystick.m_Joy[i].m_dwRDZmin != mmJoystick.m_Joy[i].m_dwRDZmax)
					mmJoystick.m_Joy[i].JoyState.RUp = TRUE;
				else if(mmJoystick.m_Joy[i].m_dwRDZmax <= mmJoystick.m_Joy[i].m_JoyInfo.dwRpos &&
					mmJoystick.m_Joy[i].m_dwRDZmin != mmJoystick.m_Joy[i].m_dwRDZmax)
					mmJoystick.m_Joy[i].JoyState.RDown = TRUE;
			}
			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASU) {
				if(mmJoystick.m_Joy[i].m_JoyInfo.dwUpos <= mmJoystick.m_Joy[i].m_dwUDZmin &&
					mmJoystick.m_Joy[i].m_dwUDZmin != mmJoystick.m_Joy[i].m_dwUDZmax)
					mmJoystick.m_Joy[i].JoyState.UUp = TRUE;
				else if(mmJoystick.m_Joy[i].m_dwUDZmax <= mmJoystick.m_Joy[i].m_JoyInfo.dwUpos &&
					mmJoystick.m_Joy[i].m_dwUDZmin != mmJoystick.m_Joy[i].m_dwUDZmax)
					mmJoystick.m_Joy[i].JoyState.UDown = TRUE;
			}
			if (mmJoystick.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_HASV) {
				if(mmJoystick.m_Joy[i].m_JoyInfo.dwVpos <= mmJoystick.m_Joy[i].m_dwVDZmin &&
					mmJoystick.m_Joy[i].m_dwVDZmin != mmJoystick.m_Joy[i].m_dwVDZmax)
					mmJoystick.m_Joy[i].JoyState.VUp = TRUE;
				else if(mmJoystick.m_Joy[i].m_dwVDZmax <= mmJoystick.m_Joy[i].m_JoyInfo.dwVpos &&
					mmJoystick.m_Joy[i].m_dwVDZmin != mmJoystick.m_Joy[i].m_dwVDZmax)
					mmJoystick.m_Joy[i].JoyState.VDown = TRUE;
			}
		}
	}
	return;
}

static void GetmmKeybuttonStates()
{
	if(!GetKeyboardState(mmKeyboard.m_key))
		mmKeyboard.KeyboardStatesOK = FALSE;
	else
		mmKeyboard.KeyboardStatesOK = TRUE;
}

static WORD GetmmJoyButtonCode()
{
	static SJoyState pmmJoyState[NUM_JOYSTICKS];
	int i;
	WORD B, jb;
	
	GetmmJoybuttonStates();
	
	jb = 0;
	
	//make joy button code..
	for(i = 0; i < NUM_JOYSTICKS; i++) {
		
		//Button
		for(B = 0; B < 32; B++) {
			if(mmJoystick.m_Joy[i].JoyState.Button[B] && !pmmJoyState[i].Button[B]) {
				jb = ((i << 8) | 0x8000 | (B + 8));
				break;
			}
		}
		if(jb)
			break;
		
		//Pov
		if(mmJoystick.m_Joy[i].JoyState.PovLeft && !pmmJoyState[i].PovLeft) {
			jb = ((i << 8) | 0x8000 | 4);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.PovRight && !pmmJoyState[i].PovRight) {
			jb = ((i << 8) | 0x8000 | 5);
			break;
		}
		if(mmJoystick.m_Joy[i].JoyState.PovUp && !pmmJoyState[i].PovUp) {
			jb = ((i << 8) | 0x8000 | 6);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.PovDown && !pmmJoyState[i].PovDown) {
			jb = ((i << 8) | 0x8000 | 7);
			break;
		}
		

		//axis
		//X
		if(mmJoystick.m_Joy[i].JoyState.Left && !pmmJoyState[i].Left) {
			jb = ((i << 8) | 0x8000 | 0);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.Right && !pmmJoyState[i].Right) {
			jb = ((i << 8) | 0x8000 | 1);
			break;
		}
		//Y
		if(mmJoystick.m_Joy[i].JoyState.Up && !pmmJoyState[i].Up) {
			jb = ((i << 8) | 0x8000 | 2);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.Down && !pmmJoyState[i].Down) {
			jb = ((i << 8) | 0x8000 | 3);
			break;
		}
		//Z
		if(mmJoystick.m_Joy[i].JoyState.ZUp && !pmmJoyState[i].ZUp) {
			jb = ((i << 8) | 0x8000 | 41);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.ZDown && !pmmJoyState[i].ZDown) {
			jb = ((i << 8) | 0x8000 | 42);
			break;
		}
		//R
		if(mmJoystick.m_Joy[i].JoyState.RUp && !pmmJoyState[i].RUp) {
			jb = ((i << 8) | 0x8000 | 43);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.RDown && !pmmJoyState[i].RDown) {
			jb = ((i << 8) | 0x8000 | 44);
			break;
		}
		//U
		if(mmJoystick.m_Joy[i].JoyState.UUp && !pmmJoyState[i].UUp) {
			jb = ((i << 8) | 0x8000 | 45);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.UDown && !pmmJoyState[i].UDown) {
			jb = ((i << 8) | 0x8000 | 46);
			break;
		}
		//V
		if(mmJoystick.m_Joy[i].JoyState.VUp && !pmmJoyState[i].VUp) {
			jb = ((i << 8) | 0x8000 | 47);
			break;
		}
		else if(mmJoystick.m_Joy[i].JoyState.VDown && !pmmJoyState[i].VDown) {
			jb = ((i << 8) | 0x8000 | 48);
			break;
		}
	}

	for(i = 0; i < NUM_JOYSTICKS; i++) {
		pmmJoyState[i] = mmJoystick.m_Joy[i].JoyState;
	}

	return jb;
}

static WORD GetmmKeyButtonCode()
{
	static BYTE pkeyboardstate[256];
	BYTE keyboardstate[256];
	WORD keybuttoncode;

	if(!GetKeyboardState(keyboardstate))
		return 0;

	for(keybuttoncode = 0;
		keybuttoncode < 256 && !((keyboardstate[keybuttoncode] & 0x80) &&
								 !(pkeyboardstate[keybuttoncode] & 0x80));
		keybuttoncode++);
	
	if(keybuttoncode >= 256 || keybuttoncode == VK_LBUTTON ||
	   keybuttoncode == VK_RBUTTON || keybuttoncode == VK_MBUTTON)
		keybuttoncode = 0;
	
	CopyMemory((PVOID)pkeyboardstate,
			   (const VOID*)keyboardstate,
			   (SIZE_T)sizeof(keyboardstate));
	
	return keybuttoncode;
}

static BOOL CALLBACK DIJoystick_EnumDeviceProc(LPDIDEVICEINSTANCE pdidi, LPVOID pv)
{
	if(!lstrcmp(pdidi->tszProductName, "ACT LABS GS") &&
	   !lstrcmp(pdidi->tszInstanceName, "ACT LABS GS"))
		return DIENUM_CONTINUE;
	diJoystick.joysticks[diJoystick.num_joysticks].guidDevice = pdidi->guidInstance;
	diJoystick.num_joysticks++;
	return DIENUM_CONTINUE;
}

static HRESULT SetDIDwordProperty(LPDIRECTINPUTDEVICE2 pdev, REFGUID guidProperty,
               DWORD dwObject, DWORD dwHow, DWORD dwValue)
{
    DIPROPDWORD dipdw;

    dipdw.diph.dwSize       = sizeof(dipdw);
    dipdw.diph.dwHeaderSize = sizeof(dipdw.diph);
    dipdw.diph.dwObj        = dwObject;
    dipdw.diph.dwHow        = dwHow;
    dipdw.dwData            = dwValue;

    return pdev->SetProperty(guidProperty, &dipdw.diph);

}

static BOOL CALLBACK DIJoystick_EnumAxisObjectsProc(LPCDIDEVICEOBJECTINSTANCE lpddoi,
                                             dijoystick_type *joystick)
{
	DIPROPRANGE diprg;
	HRESULT hr;

	joystick->axes[joystick->num_axes].guid = lpddoi->guidType;

	joystick->axes[joystick->num_axes].offset = lpddoi->dwOfs;

	//ErrorMsg("got axis %s, offset %i",lpddoi->tszName, lpddoi->dwOfs);
	diprg.diph.dwSize       = sizeof(diprg);
	diprg.diph.dwHeaderSize = sizeof(diprg.diph);
	diprg.diph.dwObj        = lpddoi->dwOfs;
	diprg.diph.dwHow        = DIPH_BYOFFSET;
	diprg.lMin              = 0;
	diprg.lMax              = 255;

	hr = joystick->did->SetProperty(DIPROP_RANGE, &diprg.diph);
	if (FAILED(hr)) /* if this fails, don't use this axis */
		return DIENUM_CONTINUE;

	/* Set axis dead zone to 0; we need accurate #'s for analog joystick reading. */
	hr = SetDIDwordProperty(joystick->did, DIPROP_DEADZONE, lpddoi->dwOfs, DIPH_BYOFFSET, 0);

	joystick->num_axes++;

	return DIENUM_CONTINUE;
}

static BOOL CALLBACK DIJoystick_EnumPOVObjectsProc(LPCDIDEVICEOBJECTINSTANCE lpddoi,
												   dijoystick_type *joystick)
{
   joystick->num_pov++;
   
   return DIENUM_CONTINUE;
}

static BOOL CALLBACK DIJoystick_EnumButtonObjectsProc(LPCDIDEVICEOBJECTINSTANCE lpddoi,
													  dijoystick_type *joystick)
{
   joystick->num_buttons++;
   
   return DIENUM_CONTINUE;
}

static void diClearJoyState(dijoystick_type *joystick)
{
	FillMemory((PVOID)&joystick->JoyState, sizeof(joystick->JoyState), 0);
	FillMemory((PVOID)&joystick->dijs, sizeof(joystick->dijs), 0);

	joystick->dijs.lX = 128;
	joystick->dijs.lY = 128;
	joystick->dijs.lZ = 128;
	joystick->dijs.lRx = 128;
	joystick->dijs.lRy = 128;
	joystick->dijs.lRz = 128;
	joystick->dijs.rglSlider[0] = 128;
	joystick->dijs.rglSlider[1] = 128;
	joystick->dijs.rgdwPOV[0] = (DWORD)-1;
	joystick->dijs.rgdwPOV[1] = (DWORD)-1;
	joystick->dijs.rgdwPOV[2] = (DWORD)-1;
	joystick->dijs.rgdwPOV[3] = (DWORD)-1;
}

static void dInitJoystick(dijoystick_type *joystick, HWND hWnd)
{
	LPDIRECTINPUTDEVICE didTemp;
	HRESULT hr;

	joystick->use_joystick = FALSE;

	joystick->did = NULL;
	joystick->num_axes = 0;

	/* get a did1 interface first... */
	hr = lpDI->CreateDevice(joystick->guidDevice, &didTemp, NULL);
	if (FAILED(hr))
		return;

	/* get a did2 interface to work with polling (most) joysticks */
	hr = didTemp->QueryInterface(IID_IDirectInputDevice2, (void **)&joystick->did);

	/* dispose of the temp interface */
	didTemp->Release();

	/* check result of getting the did2 */
	if (FAILED(hr))
	{
		/* no error message because this happens in dx3 */
		/* ErrorMsg("DirectInput QueryInterface joystick failed\n"); */
		joystick->did = NULL;
		return;
	}

	hr = joystick->did->SetCooperativeLevel(hWnd, DISCL_NONEXCLUSIVE | DISCL_BACKGROUND);
	if (FAILED(hr))
		return;

	hr = joystick->did->SetDataFormat(&c_dfDIJoystick);
	if (FAILED(hr))
		return;

	/* enumerate our axes */
	hr = joystick->did->EnumObjects((LPDIENUMDEVICEOBJECTSCALLBACK)DIJoystick_EnumAxisObjectsProc, joystick, DIDFT_AXIS);

	if (FAILED(hr))
		return;


	 /* enumerate our POV hats */
	joystick->num_pov = 0;
	hr = joystick->did->EnumObjects((LPDIENUMDEVICEOBJECTSCALLBACK)DIJoystick_EnumPOVObjectsProc, joystick, DIDFT_POV);
	if (FAILED(hr))
		return;

	/* enumerate our buttons */

	joystick->num_buttons = 0;
	hr = joystick->did->EnumObjects((LPDIENUMDEVICEOBJECTSCALLBACK)DIJoystick_EnumButtonObjectsProc, joystick, DIDFT_BUTTON);
	if (FAILED(hr))
		return;

	hr = joystick->did->Acquire();
	if (FAILED(hr)) 
		return;

	/* start by clearing the structures */

	diClearJoyState(joystick);

	joystick->use_joystick = TRUE;
}

static unsigned char DIJoystick_init(HWND hWnd)
{
	DWORD i;
	HRESULT hr;

	diJoystick.num_joysticks = 0;

	/* enumerate for joystick devices */
	hr = lpDI->EnumDevices(DIDEVTYPE_JOYSTICK,
								(LPDIENUMDEVICESCALLBACK)DIJoystick_EnumDeviceProc,
								NULL,
								DIEDFL_ALLDEVICES);
	if (FAILED(hr))
		return FALSE;

	/* create each joystick device, enumerate each joystick for axes, etc */
	for (i = 0; i < diJoystick.num_joysticks; i++) {
		dInitJoystick(&diJoystick.joysticks[i], hWnd);
	}
	return TRUE;
}

static unsigned char DIKeyboard_init(HWND hWnd)
{
	LPDIRECTINPUTDEVICE didTemp;
	HRESULT hr;
	int i;

	diKeyboard.m_didKeyboard  = NULL;

	for (i=0;i<256;i++)
		diKeyboard.m_key[i] = 0;


	if (lpDI == NULL)
		return FALSE;

	/* setup the keyboard */
	hr = lpDI->CreateDevice(GUID_SysKeyboard, &didTemp, NULL);

	if (FAILED(hr)) 
		return FALSE;

	/* get a did2 interface */
	hr = didTemp->QueryInterface(IID_IDirectInputDevice2, (void **)&diKeyboard.m_didKeyboard);

	/* dispose of the temp interface */
	didTemp->Release();

	if (FAILED(hr)) 
		return FALSE;

	hr = diKeyboard.m_didKeyboard->SetDataFormat(&c_dfDIKeyboard);

	if (FAILED(hr)) 
		return FALSE;

//	CEChEȊO֓̓tH[JXςAݒ_CAOœ͌osɂȂ
//	hr = diKeyboard.m_didKeyboard->SetCooperativeLevel(hWnd, DISCL_NONEXCLUSIVE | DISCL_FOREGROUND);
	hr = diKeyboard.m_didKeyboard->SetCooperativeLevel(hWnd, DISCL_NONEXCLUSIVE | DISCL_BACKGROUND);

	if (FAILED(hr)) 
		return FALSE;

	hr = diKeyboard.m_didKeyboard->Acquire();

	return TRUE;
}

#if 0
static unsigned char DIMouse_init(HWND hWnd)
{
	LPDIRECTINPUTDEVICE didTemp;
	HRESULT hr;

	diMouse.m_didMouse  = NULL;

	if (lpDI == NULL)
		return FALSE;

	hr = lpDI->CreateDevice(GUID_SysMouse, &didTemp, NULL);

	if (FAILED(hr)) 
		return FALSE;

	/* get a did2 interface */
	hr = didTemp->QueryInterface(IID_IDirectInputDevice2, (void **)&diMouse.m_didMouse);

	/* dispose of the temp interface */
	didTemp->Release();

	if (FAILED(hr)) 
		return FALSE;

	hr = diMouse.m_didMouse->SetDataFormat(&c_dfDIMouse);

	if (FAILED(hr)) 
		return FALSE;

	hr = diMouse.m_didMouse->SetCooperativeLevel(hWnd, DISCL_EXCLUSIVE | DISCL_FOREGROUND);

	if (FAILED(hr)) 
		return FALSE;

	hr = diMouse.m_didMouse->Acquire();

	return TRUE;
}
#endif

static BOOL DirectInput_init(HWND hWnd)
{
	if(!DirectInputInitialize(hWnd) ||
		!DIJoystick_init(hWnd) ||
		!DIKeyboard_init(hWnd) /*||
		!DIMouse_init(hWnd)*/)
		return FALSE;
	return TRUE;
}

static void ExitdiJoystick(dijoystick_type *joystick)
{
	if (joystick->did != NULL) {
		joystick->did->Unacquire();
		joystick->did->Release();
		joystick->did = NULL;
	}
}

static void DIJoystick_exit()
{
	DWORD i;
	for (i = 0; i < diJoystick.num_joysticks; i++)
		ExitdiJoystick(&diJoystick.joysticks[i]);
	diJoystick.num_joysticks = 0;
}

static void DIKeyboard_exit()
{
	if (!diKeyboard.m_didKeyboard)
		return;

	/*
	 Cleanliness is next to godliness.  Unacquire the device
	 one last time just in case we got really confused and tried
	 to exit while the device is still acquired.
	 */
	diKeyboard.m_didKeyboard->Unacquire();
	diKeyboard.m_didKeyboard->Release();
	diKeyboard.m_didKeyboard = NULL;
}

#if 0
static void DIMouse_exit()
{
	if (!diMouse.m_didMouse)
		return;

	diMouse.m_didMouse->Unacquire();
	diMouse.m_didMouse->Release();
	diMouse.m_didMouse = NULL;
}
#endif

static void DirectInput_exit()
{
	DIJoystick_exit();
	DIKeyboard_exit();
//	DIMouse_exit();
	DirectInputDeInitialize();
}

//Return DIuse setting
BOOL WinInputInitialize(HWND hWnd, BOOL DIuse)
{
	DirectInput_exit();
	if(DIuse) {
		if(!DirectInput_init(hWnd)) {
			if(hWnd)
				MessageBox(hWnd,
						   GetSystemDefaultLangID() == 0x0411 ?
						   "DirectInputɎs܂AWinAPIgp܂" :
						   "DirectInput failed to initialize, WinAPI will be used.", "DirectInput",
						   MB_OK | MB_ICONINFORMATION);
			DIuse = FALSE;
			DirectInput_exit();
			mmJoystick_init();
		}
		SwitchInputProc(DIuse);
	}
	else {
		mmJoystick_init();
		SwitchInputProc(DIuse);
	}
	return DIuse;
}

void WinInputDeinitialize()
{
	DirectInput_exit();
}

static BOOL GetJoyKeybuttonStatedi (WORD KeyIdent)
{
	if(!KeyIdent) {
		return false;
	}
	else if (KeyIdent & 0x8000) {
		int i = (KeyIdent >> 8) & 15;
		
		switch (KeyIdent & 0xff) {
			case 0: return diJoystick.joysticks[i].JoyState.Left;
			case 1: return diJoystick.joysticks[i].JoyState.Right;
			case 2: return diJoystick.joysticks[i].JoyState.Up;
			case 3: return diJoystick.joysticks[i].JoyState.Down;
			case 4: return diJoystick.joysticks[i].JoyState.PovLeft;
			case 5: return diJoystick.joysticks[i].JoyState.PovRight;
			case 6: return diJoystick.joysticks[i].JoyState.PovUp;
			case 7: return diJoystick.joysticks[i].JoyState.PovDown;
			case 41:return diJoystick.joysticks[i].JoyState.ZUp;
			case 42:return diJoystick.joysticks[i].JoyState.ZDown;
			case 43:return diJoystick.joysticks[i].JoyState.RUp;
			case 44:return diJoystick.joysticks[i].JoyState.RDown;
			case 45:return diJoystick.joysticks[i].JoyState.UUp;
			case 46:return diJoystick.joysticks[i].JoyState.UDown;
			case 47:return diJoystick.joysticks[i].JoyState.VUp;
			case 48:return diJoystick.joysticks[i].JoyState.VDown;

			default:
				if ((KeyIdent & 0xff) > 40)
					return false;
				return diJoystick.joysticks[i].JoyState.Button [(KeyIdent & 0xff) - 8];
		}
	}
	return (diKeyboard.KeyboardStatesOK && (diKeyboard.m_key[KeyIdent & 0xff] & 0x80) != 0);
}

#define DIDZ 60

static void GetdiJoybuttonStates()
{
	HRESULT hr;
	DWORD i, i2;
	unsigned long B;
	int pov_value;
	int angle;
	int axis_value_x, axis_value_y;
	
	for (i = 0; i < diJoystick.num_joysticks; i++) {
		diClearJoyState(&diJoystick.joysticks[i]);
		if (diJoystick.joysticks[i].did && diJoystick.joysticks[i].use_joystick) {
			hr = diJoystick.joysticks[i].did->Poll();
			hr = diJoystick.joysticks[i].did->GetDeviceState(sizeof(DIJOYSTATE), &diJoystick.joysticks[i].dijs);
			
			if (FAILED(hr)) {
				if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED) {
					hr = diJoystick.joysticks[i].did->Acquire();
					if (FAILED(diJoystick.joysticks[i].did->GetDeviceState(sizeof(DIJOYSTATE), &diJoystick.joysticks[i].dijs)))
						continue;
				}
				else {
					continue;
				}
			}
			
			//button
			for (B = 0; B < 32; B ++) {
				diJoystick.joysticks[i].JoyState.Button[B] = (diJoystick.joysticks[i].dijs.rgbButtons[B] & 0x80) != 0;
			}
			
			//pov
			if (diJoystick.joysticks[i].num_pov) {
				for(i2 = 0; i2 < diJoystick.joysticks[i].num_pov; i2++) {
					pov_value = (int)diJoystick.joysticks[i].dijs.rgdwPOV[i2];
					if(LOWORD(pov_value) != 0xffff) {
						angle = (pov_value + 27000) % 36000;
						angle = (36000 - angle) % 36000;
						angle /= 100;
						axis_value_x = 128 + (int)(127*cos(2*PI*angle/360.0)); // x
						axis_value_y = 128 + (int)(127*sin(2*PI*angle/360.0)); // y
						if(axis_value_x <= (128 - 128*DIDZ/100))
							diJoystick.joysticks[i].JoyState.PovLeft = TRUE;
						else if(axis_value_x >= (128 + 128*DIDZ/100))
							diJoystick.joysticks[i].JoyState.PovRight = TRUE;
						if(axis_value_y <= (128 - 128*DIDZ/100))
							diJoystick.joysticks[i].JoyState.PovUp = TRUE;
						else if(axis_value_y >= (128 + 128*DIDZ/100))
							diJoystick.joysticks[i].JoyState.PovDown = TRUE;
					}
				}
			}
			
			//axis
			//
			//X
			if(diJoystick.joysticks[i].num_axes >= 1 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[0].offset) <= (128 - 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.Left = TRUE;
			else if(diJoystick.joysticks[i].num_axes >= 1 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[0].offset) >= (128 + 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.Right = TRUE;
			
			//Y
			if(diJoystick.joysticks[i].num_axes >= 2 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[1].offset) <= (128 - 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.Up = TRUE;
			else if(diJoystick.joysticks[i].num_axes >= 2 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[1].offset) >= (128 + 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.Down = TRUE;
			
			//Z
			if(diJoystick.joysticks[i].num_axes >= 3 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[2].offset) <= (128 - 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.ZUp = TRUE;
			else if(diJoystick.joysticks[i].num_axes >= 3 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[2].offset) >= (128 + 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.ZDown = TRUE;
			
			//R
			if(diJoystick.joysticks[i].num_axes >= 4 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[3].offset) <= (128 - 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.RUp = TRUE;
			else if(diJoystick.joysticks[i].num_axes >= 4 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[3].offset) >= (128 + 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.RDown = TRUE;
			
			//U
			if(diJoystick.joysticks[i].num_axes >= 5 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[4].offset) <= (128 - 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.UUp = TRUE;
			else if(diJoystick.joysticks[i].num_axes >= 5 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[4].offset) >= (128 + 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.UDown = TRUE;
			
			//V
			if(diJoystick.joysticks[i].num_axes >= 6 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[5].offset) <= (128 - 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.VUp = TRUE;
			else if(diJoystick.joysticks[i].num_axes >= 6 &&
				*(int *)(((byte *)&diJoystick.joysticks[i].dijs) +
						 diJoystick.joysticks[i].axes[5].offset) >= (128 + 128*DIDZ/100))
				diJoystick.joysticks[i].JoyState.VDown = TRUE;
		}
	}
	return;
}

static void GetdiKeybuttonStates()
{
	HRESULT hr;

	hr = diKeyboard.m_didKeyboard->Poll();
	hr = diKeyboard.m_didKeyboard->GetDeviceState(sizeof(diKeyboard.m_key), (LPVOID)&diKeyboard.m_key);
	
	if (FAILED(hr)) {
		if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED) {
			hr = diKeyboard.m_didKeyboard->Acquire();
			if (FAILED(diKeyboard.m_didKeyboard->GetDeviceState(sizeof(diKeyboard.m_key), (LPVOID)&diKeyboard.m_key)))
				diKeyboard.KeyboardStatesOK = FALSE;
			else
				diKeyboard.KeyboardStatesOK = TRUE;
		}
		else {
			diKeyboard.KeyboardStatesOK = FALSE;
		}
	}
	else {
		diKeyboard.KeyboardStatesOK = TRUE;
	}
}

static WORD GetdiJoyButtonCode()
{
	static SJoyState pdiJoyState[MAX_PHYSICAL_JOYSTICKS];
	int i;
	WORD B, jb;
	
	GetdiJoybuttonStates();
	
	jb = 0;
	
	//make joy button code..
	for(i = 0; i < (int)diJoystick.num_joysticks; i++) {
		
		//Button
		for(B = 0; B < 32; B++) {
			if(diJoystick.joysticks[i].JoyState.Button[B] && !pdiJoyState[i].Button[B]) {
				jb = ((i << 8) | 0x8000 | (B + 8));
				break;
			}
		}
		if(jb)
			break;
		
		//Pov
		if(diJoystick.joysticks[i].JoyState.PovLeft && !pdiJoyState[i].PovLeft) {
			jb = ((i << 8) | 0x8000 | 4);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.PovRight && !pdiJoyState[i].PovRight) {
			jb = ((i << 8) | 0x8000 | 5);
			break;
		}
		if(diJoystick.joysticks[i].JoyState.PovUp && !pdiJoyState[i].PovUp) {
			jb = ((i << 8) | 0x8000 | 6);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.PovDown && !pdiJoyState[i].PovDown) {
			jb = ((i << 8) | 0x8000 | 7);
			break;
		}
		

		//axis
		//X
		if(diJoystick.joysticks[i].JoyState.Left && !pdiJoyState[i].Left) {
			jb = ((i << 8) | 0x8000 | 0);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.Right && !pdiJoyState[i].Right) {
			jb = ((i << 8) | 0x8000 | 1);
			break;
		}
		//Y
		if(diJoystick.joysticks[i].JoyState.Up && !pdiJoyState[i].Up) {
			jb = ((i << 8) | 0x8000 | 2);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.Down && !pdiJoyState[i].Down) {
			jb = ((i << 8) | 0x8000 | 3);
			break;
		}
		//Z
		if(diJoystick.joysticks[i].JoyState.ZUp && !pdiJoyState[i].ZUp) {
			jb = ((i << 8) | 0x8000 | 41);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.ZDown && !pdiJoyState[i].ZDown) {
			jb = ((i << 8) | 0x8000 | 42);
			break;
		}
		//R
		if(diJoystick.joysticks[i].JoyState.RUp && !pdiJoyState[i].RUp) {
			jb = ((i << 8) | 0x8000 | 43);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.RDown && !pdiJoyState[i].RDown) {
			jb = ((i << 8) | 0x8000 | 44);
			break;
		}
		//U
		if(diJoystick.joysticks[i].JoyState.UUp && !pdiJoyState[i].UUp) {
			jb = ((i << 8) | 0x8000 | 45);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.UDown && !pdiJoyState[i].UDown) {
			jb = ((i << 8) | 0x8000 | 46);
			break;
		}
		//V
		if(diJoystick.joysticks[i].JoyState.VUp && !pdiJoyState[i].VUp) {
			jb = ((i << 8) | 0x8000 | 47);
			break;
		}
		else if(diJoystick.joysticks[i].JoyState.VDown && !pdiJoyState[i].VDown) {
			jb = ((i << 8) | 0x8000 | 48);
			break;
		}
	}

	for(i = 0; i < (int)diJoystick.num_joysticks; i++) {
		pdiJoyState[i] = diJoystick.joysticks[i].JoyState;
	}

	return jb;
}

static WORD GetdiKeyButtonCode()
{
	static BYTE pkeyboardstate[256];
	WORD keybuttoncode;
	HRESULT hr;
	
	hr = diKeyboard.m_didKeyboard->Poll();
	hr = diKeyboard.m_didKeyboard->GetDeviceState(sizeof(diKeyboard.m_key), (LPVOID)&diKeyboard.m_key);
	
	if (FAILED(hr)) {
		if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED) {
			hr = diKeyboard.m_didKeyboard->Acquire();
			if (FAILED(diKeyboard.m_didKeyboard->GetDeviceState(sizeof(diKeyboard.m_key), (LPVOID)&diKeyboard.m_key)))
				return 0;
		}
		else {
			return 0;
		}
	}

	for(keybuttoncode = 0;
		keybuttoncode < 256 && !((diKeyboard.m_key[keybuttoncode] & 0x80) &&
								 !(pkeyboardstate[keybuttoncode] & 0x80));
		keybuttoncode++);
	
	if(keybuttoncode >= 256)
		keybuttoncode = 0;
	
	CopyMemory((PVOID)pkeyboardstate,
			   (const VOID*)diKeyboard.m_key,
			   (SIZE_T)sizeof(diKeyboard.m_key));
	
	return keybuttoncode;
}

//from "winuser.h"
#if (_WIN32_WINNT < 0x0500)
#define VK_XBUTTON1	5
#define VK_XBUTTON2	6

#define VK_BROWSER_BACK	0xA6
#define VK_BROWSER_FORWARD	0xA7
#define VK_BROWSER_REFRESH	0xA8
#define VK_BROWSER_STOP	0xA9
#define VK_BROWSER_SEARCH	0xAA
#define VK_BROWSER_FAVORITES	0xAB
#define VK_BROWSER_HOME	0xAC
#define VK_VOLUME_MUTE	0xAD
#define VK_VOLUME_DOWN	0xAE
#define VK_VOLUME_UP	0xAF
#define VK_MEDIA_NEXT_TRACK	0xB0
#define VK_MEDIA_PREV_TRACK	0xB1
#define VK_MEDIA_STOP	0xB2
#define VK_MEDIA_PLAY_PAUSE	0xB3
#define VK_LAUNCH_MAIL	0xB4
#define VK_LAUNCH_MEDIA_SELECT	0xB5
#define VK_LAUNCH_APP1	0xB6
#define VK_LAUNCH_APP2	0xB7

#define VK_OEM_PLUS	0xBB
#define VK_OEM_COMMA	0xBC
#define VK_OEM_MINUS	0xBD
#define VK_OEM_PERIOD	0xBE

#define VK_OEM_102	0xE2

#define VK_PACKET	0xE7

#endif

static BOOL GetJoyKeybuttonStringmm(WORD KeyIdent, char *strbuf, int strbuflen)
{
	if(!strbuf || strbuflen < 256) {
		if(strbuf)
			strbuf[0] = 0;
		return FALSE;
	}
	
	//Joystick
	if(KeyIdent & 0x8000) {
		strbuf[0] = 'J';
		wsprintf(&strbuf[1], "%02d ", ((KeyIdent >> 8) & 0x0f));
		switch(KeyIdent & 0xff) {
			case 0: lstrcat(strbuf, "Left"); break;
			case 1: lstrcat(strbuf, "Right"); break;
			case 2: lstrcat(strbuf, "Up"); break;
			case 3: lstrcat(strbuf, "Down"); break;
			case 4: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Left"); break;
			case 5: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Right"); break;
			case 6: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Up"); break;
			case 7: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Down"); break;
			case 41: lstrcat(strbuf, "Z "); lstrcat(strbuf, "Up"); break;
			case 42: lstrcat(strbuf, "Z "); lstrcat(strbuf, "Down"); break;
			case 43: lstrcat(strbuf, "R "); lstrcat(strbuf, "Up"); break;
			case 44: lstrcat(strbuf, "R "); lstrcat(strbuf, "Down"); break;
			case 45: lstrcat(strbuf, "U "); lstrcat(strbuf, "Up"); break;
			case 46: lstrcat(strbuf, "U "); lstrcat(strbuf, "Down"); break;
			case 47: lstrcat(strbuf, "V "); lstrcat(strbuf, "Up"); break;
			case 48: lstrcat(strbuf, "V "); lstrcat(strbuf, "Down"); break;

			default:
				if((KeyIdent & 0xff) > 40) {
					wsprintf(&strbuf[4], "%d", (KeyIdent & 0xff));
					break;
				}
				lstrcat(strbuf, "Button ");
				wsprintf(&strbuf[11], "%d", ((KeyIdent & 0xff) - 8));
				break;
		}
	}
	// 0~9, A~Z
	else if((KeyIdent >= '0' && KeyIdent <= '9') || (KeyIdent >= 'A' && KeyIdent <= 'Z')) {
		strbuf[0] = (char)KeyIdent;
		strbuf[1] = 0;
	}
	else {
		char *Result = NULL;

		switch(KeyIdent) {
		case 0: Result = "None"; break;
		case VK_LBUTTON: Result = "LBUTTON"; break;
		case VK_RBUTTON: Result = "RBUTTON"; break;
		case VK_CANCEL: Result = "CANCEL"; break;
		case VK_MBUTTON: Result = "MBUTTON"; break;

			//#if (_WIN32_WINNT >= 0x0500)
		case VK_XBUTTON1: Result = "XBUTTON1"; break;
		case VK_XBUTTON2: Result = "XBUTTON2"; break;

		case VK_BACK: Result = "BACK"; break;
		case VK_TAB: Result = "TAB"; break;
		case VK_CLEAR: Result = "CLEAR"; break;
		case VK_RETURN: Result = "RETURN"; break;
		case VK_SHIFT: Result = "SHIFT"; break;
		case VK_CONTROL: Result = "CONTROL"; break;
		case VK_MENU: Result = "MENU"; break;
		case VK_PAUSE: Result = "PAUSE"; break;
		case VK_CAPITAL: Result = "CAPITAL"; break;
		case VK_KANA: Result = "KANA"; break;
		//case VK_HANGEUL: Result = "HANGEUL"; break;
		//case VK_HANGUL: Result = "HANGUL"; break;
		case VK_JUNJA: Result = "JUNJA"; break;
		case VK_FINAL: Result = "FINAL"; break;
		//case VK_HANJA: Result = "HANJA"; break;
		case VK_KANJI: Result = "KANJI"; break;
		case VK_ESCAPE: Result = "ESCAPE"; break;
		case VK_CONVERT: Result = "CONVERT"; break;
		case VK_NONCONVERT: Result = "NONCONVERT"; break;
		case VK_ACCEPT: Result = "ACCEPT"; break;
		case VK_MODECHANGE: Result = "MODECHANGE"; break;
		case VK_SPACE: Result = "SPACE"; break;
		case VK_PRIOR: Result = "PRIOR"; break;
		case VK_NEXT: Result = "NEXT"; break;
		case VK_END: Result = "END"; break;
		case VK_HOME: Result = "HOME"; break;
		case VK_LEFT: Result = "LEFT"; break;
		case VK_UP: Result = "UP"; break;
		case VK_RIGHT: Result = "RIGHT"; break;
		case VK_DOWN: Result = "DOWN"; break;
		case VK_SELECT: Result = "SELECT"; break;
		case VK_PRINT: Result = "PRINT"; break;
		case VK_EXECUTE: Result = "EXECUTE"; break;
		case VK_SNAPSHOT: Result = "SNAPSHOT"; break;
		case VK_INSERT: Result = "INSERT"; break;
		case VK_DELETE: Result = "DELETE"; break;
		case VK_HELP: Result = "HELP"; break;
		case VK_LWIN: Result = "LWIN"; break;
		case VK_RWIN: Result = "RWIN"; break;
		case VK_APPS: Result = "APPS"; break;
		case VK_SLEEP: Result = "SLEEP"; break;
		case VK_NUMPAD0: Result = "NUMPAD0"; break;
		case VK_NUMPAD1: Result = "NUMPAD1"; break;
		case VK_NUMPAD2: Result = "NUMPAD2"; break;
		case VK_NUMPAD3: Result = "NUMPAD3"; break;
		case VK_NUMPAD4: Result = "NUMPAD4"; break;
		case VK_NUMPAD5: Result = "NUMPAD5"; break;
		case VK_NUMPAD6: Result = "NUMPAD6"; break;
		case VK_NUMPAD7: Result = "NUMPAD7"; break;
		case VK_NUMPAD8: Result = "NUMPAD8"; break;
		case VK_NUMPAD9: Result = "NUMPAD9"; break;
		case VK_MULTIPLY: Result = "MULTIPLY"; break;
		case VK_ADD: Result = "ADD"; break;
		case VK_SEPARATOR: Result = "SEPARATOR"; break;
		case VK_SUBTRACT: Result = "SUBTRACT"; break;
		case VK_DECIMAL: Result = "DECIMAL"; break;
		case VK_DIVIDE: Result = "DIVIDE"; break;
		case VK_F1: Result = "F1"; break;
		case VK_F2: Result = "F2"; break;
		case VK_F3: Result = "F3"; break;
		case VK_F4: Result = "F4"; break;
		case VK_F5: Result = "F5"; break;
		case VK_F6: Result = "F6"; break;
		case VK_F7: Result = "F7"; break;
		case VK_F8: Result = "F8"; break;
		case VK_F9: Result = "F9"; break;
		case VK_F10: Result = "F10"; break;
		case VK_F11: Result = "F11"; break;
		case VK_F12: Result = "F12"; break;
		case VK_F13: Result = "F13"; break;
		case VK_F14: Result = "F14"; break;
		case VK_F15: Result = "F15"; break;
		case VK_F16: Result = "F16"; break;
		case VK_F17: Result = "F17"; break;
		case VK_F18: Result = "F18"; break;
		case VK_F19: Result = "F19"; break;
		case VK_F20: Result = "F20"; break;
		case VK_F21: Result = "F21"; break;
		case VK_F22: Result = "F22"; break;
		case VK_F23: Result = "F23"; break;
		case VK_F24: Result = "F24"; break;
		case VK_NUMLOCK: Result = "NUMLOCK"; break;
		case VK_SCROLL: Result = "SCROLL"; break;
		case VK_LSHIFT: Result = "LSHIFT"; break;
		case VK_RSHIFT: Result = "RSHIFT"; break;
		case VK_LCONTROL: Result = "LCONTROL"; break;
		case VK_RCONTROL: Result = "RCONTROL"; break;
		case VK_LMENU: Result = "LMENU"; break;
		case VK_RMENU: Result = "RMENU"; break;

			//#if (_WIN32_WINNT >= 0x0500)
		case VK_BROWSER_BACK: Result = "BROWSER_BACK"; break;
		case VK_BROWSER_FORWARD: Result = "BROWSER_FORWARD"; break;
		case VK_BROWSER_REFRESH: Result = "BROWSER_REFRESH"; break;
		case VK_BROWSER_STOP: Result = "BROWSER_STOP"; break;
		case VK_BROWSER_SEARCH: Result = "BROWSER_SEARCH"; break;
		case VK_BROWSER_FAVORITES: Result = "BROWSER_FAVORITES"; break;
		case VK_BROWSER_HOME: Result = "BROWSER_HOME"; break;
		case VK_VOLUME_MUTE: Result = "VOLUME_MUTE"; break;
		case VK_VOLUME_DOWN: Result = "VOLUME_DOWN"; break;
		case VK_VOLUME_UP: Result = "VOLUME_UP"; break;
		case VK_MEDIA_NEXT_TRACK: Result = "MEDIA_NEXT_TRACK"; break;
		case VK_MEDIA_PREV_TRACK: Result = "MEDIA_PREV_TRACK"; break;
		case VK_MEDIA_STOP: Result = "MEDIA_STOP"; break;
		case VK_MEDIA_PLAY_PAUSE: Result = "MEDIA_PLAY_PAUSE"; break;
		case VK_LAUNCH_MAIL: Result = "LAUNCH_MAIL"; break;
		case VK_LAUNCH_MEDIA_SELECT: Result = "LAUNCH_MEDIA_SELECT"; break;
		case VK_LAUNCH_APP1: Result = "LAUNCH_APP1"; break;
		case VK_LAUNCH_APP2: Result = "LAUNCH_APP2"; break;

		case VK_OEM_1: Result = "OEM_1"; break;

			//#if (_WIN32_WINNT >= 0x0500)
		case VK_OEM_PLUS: Result = "OEM_PLUS"; break;
		case VK_OEM_COMMA: Result = "OEM_COMMA"; break;
		case VK_OEM_MINUS: Result = "OEM_MINUS"; break;
		case VK_OEM_PERIOD: Result = "OEM_PERIOD"; break;

		case VK_OEM_2: Result = "OEM_2"; break;
		case VK_OEM_3: Result = "OEM_3"; break;
		case VK_OEM_4: Result = "OEM_4"; break;
		case VK_OEM_5: Result = "OEM_5"; break;
		case VK_OEM_6: Result = "OEM_6"; break;
		case VK_OEM_7: Result = "OEM_7"; break;
		case VK_OEM_8: Result = "OEM_8"; break;

			//#if (_WIN32_WINNT >= 0x0500)
		case VK_OEM_102: Result = "OEM_102"; break;

		case VK_PROCESSKEY: Result = "PROCESSKEY"; break;

			//#if (_WIN32_WINNT >= 0x0500)
		case VK_PACKET: Result = "PACKET"; break;

		case VK_ATTN: Result = "ATTN"; break;
		case VK_CRSEL: Result = "CRSEL"; break;
		case VK_EXSEL: Result = "EXSEL"; break;
		case VK_EREOF: Result = "EREOF"; break;
		case VK_PLAY: Result = "PLAY"; break;
		case VK_ZOOM: Result = "ZOOM"; break;
		case VK_NONAME: Result = "NONAME"; break;
		case VK_PA1: Result = "PA1"; break;
		case VK_OEM_CLEAR: Result = "OEM_CLEAR"; break;
		}

		if(Result)
			lstrcpy(strbuf, Result);
		else
			wsprintf(&strbuf[0], "%d", KeyIdent);
	}
	return TRUE;
}

static BOOL GetJoyKeybuttonStringdi(WORD KeyIdent, char *strbuf, int strbuflen)
{
	if(!strbuf || strbuflen < 256) {
		if(strbuf)
			strbuf[0] = 0;
		return FALSE;
	}
	
	//Joystick
	if(KeyIdent & 0x8000) {
		strbuf[0] = 'J';
		wsprintf(&strbuf[1], "%02d ", ((KeyIdent >> 8) & 0x0f));
		switch(KeyIdent & 0xff) {
			case 0: lstrcat(strbuf, "Left"); break;
			case 1: lstrcat(strbuf, "Right"); break;
			case 2: lstrcat(strbuf, "Up"); break;
			case 3: lstrcat(strbuf, "Down"); break;
			case 4: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Left"); break;
			case 5: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Right"); break;
			case 6: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Up"); break;
			case 7: lstrcat(strbuf, "Pov "); lstrcat(strbuf, "Down"); break;
			case 41: lstrcat(strbuf, "Z "); lstrcat(strbuf, "Up"); break;
			case 42: lstrcat(strbuf, "Z "); lstrcat(strbuf, "Down"); break;
			case 43: lstrcat(strbuf, "R "); lstrcat(strbuf, "Up"); break;
			case 44: lstrcat(strbuf, "R "); lstrcat(strbuf, "Down"); break;
			case 45: lstrcat(strbuf, "U "); lstrcat(strbuf, "Up"); break;
			case 46: lstrcat(strbuf, "U "); lstrcat(strbuf, "Down"); break;
			case 47: lstrcat(strbuf, "V "); lstrcat(strbuf, "Up"); break;
			case 48: lstrcat(strbuf, "V "); lstrcat(strbuf, "Down"); break;

			default:
				if((KeyIdent & 0xff) > 40) {
					wsprintf(&strbuf[4], "%d", (KeyIdent & 0xff));
					break;
				}
				lstrcat(strbuf, "Button ");
				wsprintf(&strbuf[11], "%d", ((KeyIdent & 0xff) - 8));
				break;
		}
	}
	else {
		char *Result = NULL;

		switch(KeyIdent) {
		case 0: Result = "None"; break;
		case DIK_ESCAPE: Result = "ESCAPE"; break;
		case DIK_1: Result = "1"; break;
		case DIK_2: Result = "2"; break;
		case DIK_3: Result = "3"; break;
		case DIK_4: Result = "4"; break;
		case DIK_5: Result = "5"; break;
		case DIK_6: Result = "6"; break;
		case DIK_7: Result = "7"; break;
		case DIK_8: Result = "8"; break;
		case DIK_9: Result = "9"; break;
		case DIK_0: Result = "0"; break;
		case DIK_MINUS: Result = "MINUS"; break;
		case DIK_EQUALS: Result = "EQUALS"; break;
		case DIK_BACK: Result = "BACK"; break;
		case DIK_TAB: Result = "TAB"; break;
		case DIK_Q: Result = "Q"; break;
		case DIK_W: Result = "W"; break;
		case DIK_E: Result = "E"; break;
		case DIK_R: Result = "R"; break;
		case DIK_T: Result = "T"; break;
		case DIK_Y: Result = "Y"; break;
		case DIK_U: Result = "U"; break;
		case DIK_I: Result = "I"; break;
		case DIK_O: Result = "O"; break;
		case DIK_P: Result = "P"; break;
		case DIK_LBRACKET: Result = "LBRACKET"; break;
		case DIK_RBRACKET: Result = "RBRACKET"; break;
		case DIK_RETURN: Result = "RETURN"; break;
		case DIK_LCONTROL: Result = "LCONTROL"; break;
		case DIK_A: Result = "A"; break;
		case DIK_S: Result = "S"; break;
		case DIK_D: Result = "D"; break;
		case DIK_F: Result = "F"; break;
		case DIK_G: Result = "G"; break;
		case DIK_H: Result = "H"; break;
		case DIK_J: Result = "J"; break;
		case DIK_K: Result = "K"; break;
		case DIK_L: Result = "L"; break;
		case DIK_SEMICOLON: Result = "SEMICOLON"; break;
		case DIK_APOSTROPHE: Result = "APOSTROPHE"; break;
		case DIK_GRAVE: Result = "GRAVE"; break;
		case DIK_LSHIFT: Result = "LSHIFT"; break;
		case DIK_BACKSLASH: Result = "BACKSLASH"; break;
		case DIK_Z: Result = "Z"; break;
		case DIK_X: Result = "X"; break;
		case DIK_C: Result = "C"; break;
		case DIK_V: Result = "V"; break;
		case DIK_B: Result = "B"; break;
		case DIK_N: Result = "N"; break;
		case DIK_M: Result = "M"; break;
		case DIK_COMMA: Result = "COMMA"; break;
		case DIK_PERIOD: Result = "PERIOD"; break;
		case DIK_SLASH: Result = "SLASH"; break;
		case DIK_RSHIFT: Result = "RSHIFT"; break;
		case DIK_MULTIPLY: Result = "MULTIPLY"; break;
		case DIK_LMENU: Result = "LMENU"; break;
		case DIK_SPACE: Result = "SPACE"; break;
		case DIK_CAPITAL: Result = "CAPITAL"; break;
		case DIK_F1: Result = "F1"; break;
		case DIK_F2: Result = "F2"; break;
		case DIK_F3: Result = "F3"; break;
		case DIK_F4: Result = "F4"; break;
		case DIK_F5: Result = "F5"; break;
		case DIK_F6: Result = "F6"; break;
		case DIK_F7: Result = "F7"; break;
		case DIK_F8: Result = "F8"; break;
		case DIK_F9: Result = "F9"; break;
		case DIK_F10: Result = "F10"; break;
		case DIK_NUMLOCK: Result = "NUMLOCK"; break;
		case DIK_SCROLL: Result = "SCROLL"; break;
		case DIK_NUMPAD7: Result = "NUMPAD7"; break;
		case DIK_NUMPAD8: Result = "NUMPAD8"; break;
		case DIK_NUMPAD9: Result = "NUMPAD9"; break;
		case DIK_SUBTRACT: Result = "SUBTRACT"; break;
		case DIK_NUMPAD4: Result = "NUMPAD4"; break;
		case DIK_NUMPAD5: Result = "NUMPAD5"; break;
		case DIK_NUMPAD6: Result = "NUMPAD6"; break;
		case DIK_ADD: Result = "ADD"; break;
		case DIK_NUMPAD1: Result = "NUMPAD1"; break;
		case DIK_NUMPAD2: Result = "NUMPAD2"; break;
		case DIK_NUMPAD3: Result = "NUMPAD3"; break;
		case DIK_NUMPAD0: Result = "NUMPAD0"; break;
		case DIK_DECIMAL: Result = "DECIMAL"; break;
		case DIK_OEM_102: Result = "OEM_102"; break;
		case DIK_F11: Result = "F11"; break;
		case DIK_F12: Result = "F12"; break;
		case DIK_F13: Result = "F13"; break;
		case DIK_F14: Result = "F14"; break;
		case DIK_F15: Result = "F15"; break;
		case DIK_KANA: Result = "KANA"; break;
		case DIK_ABNT_C1: Result = "ABNT_C1"; break;
		case DIK_CONVERT: Result = "CONVERT"; break;
		case DIK_NOCONVERT: Result = "NOCONVERT"; break;
		case DIK_YEN: Result = "YEN"; break;
		case DIK_ABNT_C2: Result = "ABNT_C2"; break;
		case DIK_NUMPADEQUALS: Result = "NUMPADEQUALS"; break;
		case DIK_PREVTRACK: Result = "PREVTRACK"; break;
		case DIK_AT: Result = "AT"; break;
		case DIK_COLON: Result = "COLON"; break;
		case DIK_UNDERLINE: Result = "UNDERLINE"; break;
		case DIK_KANJI: Result = "KANJI"; break;
		case DIK_STOP: Result = "STOP"; break;
		case DIK_AX: Result = "AX"; break;
		case DIK_UNLABELED: Result = "UNLABELED"; break;
		case DIK_NEXTTRACK: Result = "NEXTTRACK"; break;
		case DIK_NUMPADENTER: Result = "NUMPADENTER"; break;
		case DIK_RCONTROL: Result = "RCONTROL"; break;
		case DIK_MUTE: Result = "MUTE"; break;
		case DIK_CALCULATOR: Result = "CALCULATOR"; break;
		case DIK_PLAYPAUSE: Result = "PLAYPAUSE"; break;
		case DIK_MEDIASTOP: Result = "MEDIASTOP"; break;
		case DIK_VOLUMEDOWN: Result = "VOLUMEDOWN"; break;
		case DIK_VOLUMEUP: Result = "VOLUMEUP"; break;
		case DIK_WEBHOME: Result = "WEBHOME"; break;
		case DIK_NUMPADCOMMA: Result = "NUMPADCOMMA"; break;
		case DIK_DIVIDE: Result = "DIVIDE"; break;
		case DIK_SYSRQ: Result = "SYSRQ"; break;
		case DIK_RMENU: Result = "RMENU"; break;
		case DIK_PAUSE: Result = "PAUSE"; break;
		case DIK_HOME: Result = "HOME"; break;
		case DIK_UP: Result = "UP"; break;
		case DIK_PRIOR: Result = "PRIOR"; break;
		case DIK_LEFT: Result = "LEFT"; break;
		case DIK_RIGHT: Result = "RIGHT"; break;
		case DIK_END: Result = "END"; break;
		case DIK_DOWN: Result = "DOWN"; break;
		case DIK_NEXT: Result = "NEXT"; break;
		case DIK_INSERT: Result = "INSERT"; break;
		case DIK_DELETE: Result = "DELETE"; break;
		case DIK_LWIN: Result = "LWIN"; break;
		case DIK_RWIN: Result = "RWIN"; break;
		case DIK_APPS: Result = "APPS"; break;
		case DIK_POWER: Result = "POWER"; break;
		case DIK_SLEEP: Result = "SLEEP"; break;
		case DIK_WAKE: Result = "WAKE"; break;
		case DIK_WEBSEARCH: Result = "WEBSEARCH"; break;
		case DIK_WEBFAVORITES: Result = "WEBFAVORITES"; break;
		case DIK_WEBREFRESH: Result = "WEBREFRESH"; break;
		case DIK_WEBSTOP: Result = "WEBSTOP"; break;
		case DIK_WEBFORWARD: Result = "WEBFORWARD"; break;
		case DIK_WEBBACK: Result = "WEBBACK"; break;
		case DIK_MYCOMPUTER: Result = "MYCOMPUTER"; break;
		case DIK_MAIL: Result = "MAIL"; break;
		case DIK_MEDIASELECT: Result = "MEDIASELECT"; break;
		}

		if(Result)
			lstrcpy(strbuf, Result);
		else
			wsprintf(&strbuf[0], "%d", KeyIdent);
	}
	return TRUE;
}

