/*
  Copyright (C) 2004 sanmaiwashi
*/

#include "snes9x.h"
#include "ppu.h"
#include "uosnesw.h"
#include "windraw.h"
#include "render.h"

#ifdef LSB_FIRST
#define READ_PIXEL_16(s) (*(uint16 *) (s))
#else
#define READ_PIXEL_16(s) (*(uint8 *)  (s) |\
					     (*((uint8 *) (s) + 1) << 8))
#endif

static void RenderNothing(SSurface *Src, SSurface *Dst, RECT *srcRect, RECT *dstRect)
{ // Just an empty procedure, so that it never happens that something 'bad' is rendered =)
}

START_EXTERN_C
// Contains the pointer to the now active render method
TRenderMethod OldRenderMethod = RenderNothing;
TRenderMethod RenderMethod = RenderNothing;
bool8 DoubleWidth = FALSE;
bool8 DoubleHeight = FALSE;
bool8 ExtendHeight = FALSE;
bool8 ExtendHeight_previos = FALSE;
extern uint8 S9xwPalette[];
uint8 snes9x_clear_change_log = 0;
END_EXTERN_C

#define SCREEN_BUFFER_PITCH_16 512*2
#define SCREEN_BUFFER_SIZE_16 (SCREEN_BUFFER_PITCH_16)*478
#define MAX_CHANGELOG 3

static uint8 ChangeLog1 [SCREEN_BUFFER_SIZE_16];
static uint8 ChangeLog2 [SCREEN_BUFFER_SIZE_16];
static uint8 ChangeLog3 [SCREEN_BUFFER_SIZE_16];

static uint8 *ChangeLog [MAX_CHANGELOG] = {
    ChangeLog1, ChangeLog2, ChangeLog3
};

static uint8 depthconv_buf[SCREEN_BUFFER_SIZE_16];

static uint8 render_palette[0x10000 * 4];

#define SNES_MULBRIGHTNESS 140

#define SRC_PIXSIZE_8to8   1
#define SRC_PIXSIZE_8to16  1
#define SRC_PIXSIZE_8to24  1
#define SRC_PIXSIZE_8to32  1
#define SRC_PIXSIZE_16to8  2
#define SRC_PIXSIZE_16to16 2
#define SRC_PIXSIZE_16to24 2
#define SRC_PIXSIZE_16to32 2
#define DST_PIXSIZE_8to8   1
#define DST_PIXSIZE_8to16  2
#define DST_PIXSIZE_8to24  3
#define DST_PIXSIZE_8to32  4
#define DST_PIXSIZE_16to8  1
#define DST_PIXSIZE_16to16 2
#define DST_PIXSIZE_16to24 3
#define DST_PIXSIZE_16to32 4


/*******************
 put pixel macros */

#define PUT_PIXEL_W1_8to8()\
	*(pDst + dst_index) = *(pSrc + src_index);

#define PUT_PIXEL_W2_8to8()\
	*(pDst + dst_index) =							\
	*(pDst + dst_index + 1) = *(pSrc + src_index);

#define PUT_PIXEL_W1_8to16()\
	*(uint16 *)(pDst + dst_index) = *(uint16 *)(&render_palette[*(pSrc + src_index) * 2]);

#define PUT_PIXEL_W2_8to16()\
	*(uint16 *)(pDst + dst_index) =										\
	*(uint16 *)(pDst + dst_index + 2) = *(uint16 *)(&render_palette[*(pSrc + src_index) * 2]);

#define PUT_PIXEL_W1_8to24()\
	int palette_index = *(pSrc + src_index) * 3;				\
	*(pDst + dst_index)     = render_palette[palette_index];		\
	*(pDst + dst_index + 1) = render_palette[palette_index + 1];	\
	*(pDst + dst_index + 2) = render_palette[palette_index + 2];

#define PUT_PIXEL_W2_8to24()\
	int palette_index = *(pSrc + src_index) * 3;					\
	*(pDst + dst_index)     = render_palette[palette_index];		\
	*(pDst + dst_index + 1) = render_palette[palette_index + 1];	\
	*(pDst + dst_index + 2) = render_palette[palette_index + 2];	\
	*(pDst + dst_index + 3) = render_palette[palette_index];		\
	*(pDst + dst_index + 4) = render_palette[palette_index + 1];	\
	*(pDst + dst_index + 5) = render_palette[palette_index + 2];

#define PUT_PIXEL_W1_8to32()\
	*(uint32 *)(pDst + dst_index) = *(uint32 *)(&render_palette[*(pSrc + src_index) * 4]);

#define PUT_PIXEL_W2_8to32()\
	*(uint32 *)(pDst + dst_index) =										\
	*(uint32 *)(pDst + dst_index + 4) = *(uint32 *)(&render_palette[*(pSrc + src_index) * 4]);

#define PUT_PIXEL_W1_16to16()\
	*(uint16 *)(pDst + dst_index) = *(uint16 *)(pSrc + src_index);

#define PUT_PIXEL_W2_16to16()\
	PUT_PIXEL_W1_16to16();												\
	*(uint16 *)(pDst + dst_index + 2) = *(uint16 *)(pSrc + src_index);

#define PUT_PIXEL_W1_16to8()\
	*(pDst + dst_index) = S9xwPalette [READ_PIXEL_16(pSrc + src_index)];

#define PUT_PIXEL_W2_16to8()\
	*(pDst + dst_index) =												\
	*(pDst + dst_index + 1) = S9xwPalette [READ_PIXEL_16(pSrc + src_index)];

#define PUT_PIXEL_W1_16to24()\
	int palette_index = READ_PIXEL_16(pSrc + src_index) * 3;		\
	*(pDst + dst_index)     = render_palette[palette_index];		\
	*(pDst + dst_index + 1) = render_palette[palette_index + 1];	\
	*(pDst + dst_index + 2) = render_palette[palette_index + 2];

#define PUT_PIXEL_W2_16to24()\
	int palette_index = READ_PIXEL_16(pSrc + src_index) * 3;		\
	*(pDst + dst_index)     = render_palette[palette_index];		\
	*(pDst + dst_index + 1) = render_palette[palette_index + 1];	\
	*(pDst + dst_index + 2) = render_palette[palette_index + 2];	\
	*(pDst + dst_index + 3) = render_palette[palette_index];		\
	*(pDst + dst_index + 4) = render_palette[palette_index + 1];	\
	*(pDst + dst_index + 5) = render_palette[palette_index + 2];

#define PUT_PIXEL_W1_16to32()\
	*(uint32 *)(pDst + dst_index) = *(uint32 *)(&render_palette[READ_PIXEL_16(pSrc + src_index) * 4]);

#define PUT_PIXEL_W2_16to32()\
	*(uint32 *)(pDst + dst_index) =										\
	*(uint32 *)(pDst + dst_index + 4) = *(uint32 *)(&render_palette[READ_PIXEL_16(pSrc + src_index) * 4]);


/****************************
 Render line helper macros */

#define RENDER_LINE_H1(w, depth)\
	int src_index = 0, dst_index = 0;									\
	for(width = (srcRect->right - srcRect->left); width; width--) {		\
		PUT_PIXEL_W##w##_##depth();										\
		src_index += SRC_PIXSIZE_##depth;								\
		dst_index += DST_PIXSIZE_##depth * (w);							\
	}																	\
	pSrc += Src->Pitch;													\
	pDst += Dst->Pitch;


#define RENDER_LINE_H2(w, depth)\
	int src_index = 0, dst_index = 0;									\
	for(width = (srcRect->right - srcRect->left); width; width--) {		\
		PUT_PIXEL_W##w##_##depth();										\
		src_index += SRC_PIXSIZE_##depth;								\
		dst_index += DST_PIXSIZE_##depth * (w);							\
	}																	\
	pDst += Dst->Pitch;													\
	src_index = dst_index = 0;											\
	for(width = (srcRect->right - srcRect->left); width; width--) {		\
		PUT_PIXEL_W##w##_##depth();										\
		src_index += SRC_PIXSIZE_##depth;								\
		dst_index += DST_PIXSIZE_##depth * (w);							\
	}																	\
	pSrc += Src->Pitch;													\
	pDst += Dst->Pitch;


/*********************
 Render line macros */

#define COPY_LINE_W1H1(depth)\
	memcpy(pDst, pSrc, (size_t)width * DST_PIXSIZE_##depth);	\
	pSrc += Src->Pitch;											\
	pDst += Dst->Pitch;

#define COPY_LINE_W1H2(depth)\
	memcpy(pDst, pSrc, (size_t)width * DST_PIXSIZE_##depth);	\
	pDst += Dst->Pitch;											\
	COPY_LINE_W1H1(depth)

#define RENDER_LINE_W1H1(depth)\
	RENDER_LINE_H1(1, depth);

#define RENDER_LINE_W2H1(depth)\
	RENDER_LINE_H1(2, depth);

#define RENDER_LINE_W1H2(depth)\
	RENDER_LINE_H2(1, depth);

#define RENDER_LINE_W2H2(depth)\
	RENDER_LINE_H2(2, depth);

#define COPY_SCANLINE_W1H2(depth)\
	COPY_LINE_W1H1(depth);												\
	memset(pDst, 0, (size_t)width * DST_PIXSIZE_##depth);				\
	pDst += Dst->Pitch;

#define RENDER_SCANLINE_W1H2(depth)\
	RENDER_LINE_H1(1, depth);											\
	width = (srcRect->right - srcRect->left);							\
	memset(pDst, 0, (size_t)width * DST_PIXSIZE_##depth);				\
	pDst += Dst->Pitch;

#define RENDER_SCANLINE_W2H2(depth)\
	RENDER_LINE_H1(2, depth);											\
	width = (srcRect->right - srcRect->left);							\
	memset(pDst, 0, (size_t)width * DST_PIXSIZE_##depth * 2);			\
	pDst += Dst->Pitch;

#define CLEAR_DEST(depth)\
	uint8 *dp = pDst;													\
	int dw = (dstRect->right - dstRect->left);							\
	for(int dh = (dstRect->bottom - dstRect->top); dh; dh--) {			\
		memset(dp, 0, (size_t)dw * DST_PIXSIZE_##depth);				\
		dp += Dst->Pitch;												\
	}

/***********************
 Render method macros */

#define RENDER_METHOD(name, w, h, depth, render_line)\
static void Render##name##W##w##H##h##depth(SSurface *Src, SSurface *Dst, RECT *srcRect, RECT *dstRect) \
{																	\
	uint8 *pSrc = (Src->Surface + srcRect->top * Src->Pitch + srcRect->left * SRC_PIXSIZE_##depth);	\
	uint8 *pDst = (Dst->Surface + dstRect->top * Dst->Pitch + dstRect->left * DST_PIXSIZE_##depth);	\
	int width = (srcRect->right - srcRect->left);						\
	int height = (srcRect->bottom - srcRect->top);						\
	for(; height; height--) {											\
		render_line(depth);												\
	}																	\
	ExtendHeight_previos = ExtendHeight;								\
}

// simply copy line render
RENDER_METHOD(, 1, 1, 8to8,   COPY_LINE_W1H1)
RENDER_METHOD(, 1, 1, 16to16, COPY_LINE_W1H1)

// simply copy H2 line render
RENDER_METHOD(, 1, 2, 8to8,   COPY_LINE_W1H2)
RENDER_METHOD(, 1, 2, 16to16, COPY_LINE_W1H2)

// simply W2 line render
RENDER_METHOD(, 2, 1, 8to8,   RENDER_LINE_W2H1)
RENDER_METHOD(, 2, 1, 16to16, RENDER_LINE_W2H1)

// simply W2H2 line render
RENDER_METHOD(, 2, 2, 8to8,   RENDER_LINE_W2H2)
RENDER_METHOD(, 2, 2, 16to16, RENDER_LINE_W2H2)

// simply copy scanline render
RENDER_METHOD(Scanline, 1, 2, 8to8,   COPY_SCANLINE_W1H2)
RENDER_METHOD(Scanline, 1, 2, 16to16, COPY_SCANLINE_W1H2)

// simply W2H2 scanline render
RENDER_METHOD(Scanline, 2, 2, 8to8,   RENDER_SCANLINE_W2H2)
RENDER_METHOD(Scanline, 2, 2, 16to16, RENDER_SCANLINE_W2H2)

// depth convert render
RENDER_METHOD(, 1, 1, 8to16,  RENDER_LINE_W1H1)
RENDER_METHOD(, 1, 1, 8to24,  RENDER_LINE_W1H1)
RENDER_METHOD(, 1, 1, 8to32,  RENDER_LINE_W1H1)
RENDER_METHOD(, 1, 1, 16to8,  RENDER_LINE_W1H1)
RENDER_METHOD(, 1, 1, 16to24, RENDER_LINE_W1H1)
RENDER_METHOD(, 1, 1, 16to32, RENDER_LINE_W1H1)

// depth convert W2 render 
RENDER_METHOD(, 2, 1, 8to16,  RENDER_LINE_W2H1)
RENDER_METHOD(, 2, 1, 8to24,  RENDER_LINE_W2H1)
RENDER_METHOD(, 2, 1, 8to32,  RENDER_LINE_W2H1)
RENDER_METHOD(, 2, 1, 16to8,  RENDER_LINE_W2H1)
RENDER_METHOD(, 2, 1, 16to24, RENDER_LINE_W2H1)
RENDER_METHOD(, 2, 1, 16to32, RENDER_LINE_W2H1)

// depth convert H2 render 
RENDER_METHOD(, 1, 2, 8to16,  RENDER_LINE_W1H2)
RENDER_METHOD(, 1, 2, 8to24,  RENDER_LINE_W1H2)
RENDER_METHOD(, 1, 2, 8to32,  RENDER_LINE_W1H2)
RENDER_METHOD(, 1, 2, 16to8,  RENDER_LINE_W1H2)
RENDER_METHOD(, 1, 2, 16to24, RENDER_LINE_W1H2)
RENDER_METHOD(, 1, 2, 16to32, RENDER_LINE_W1H2)

// depth convert W2H2 render 
RENDER_METHOD(, 2, 2, 8to16,  RENDER_LINE_W2H2)
RENDER_METHOD(, 2, 2, 8to24,  RENDER_LINE_W2H2)
RENDER_METHOD(, 2, 2, 8to32,  RENDER_LINE_W2H2)
RENDER_METHOD(, 2, 2, 16to8,  RENDER_LINE_W2H2)
RENDER_METHOD(, 2, 2, 16to24, RENDER_LINE_W2H2)
RENDER_METHOD(, 2, 2, 16to32, RENDER_LINE_W2H2)

// depth convert scanline render
RENDER_METHOD(Scanline, 1, 2, 8to16,  RENDER_SCANLINE_W1H2)
RENDER_METHOD(Scanline, 1, 2, 8to24,  RENDER_SCANLINE_W1H2)
RENDER_METHOD(Scanline, 1, 2, 8to32,  RENDER_SCANLINE_W1H2)
RENDER_METHOD(Scanline, 1, 2, 16to8,  RENDER_SCANLINE_W1H2)
RENDER_METHOD(Scanline, 1, 2, 16to24, RENDER_SCANLINE_W1H2)
RENDER_METHOD(Scanline, 1, 2, 16to32, RENDER_SCANLINE_W1H2)

// depth convert W2 scanline render
RENDER_METHOD(Scanline, 2, 2, 8to16,  RENDER_SCANLINE_W2H2)
RENDER_METHOD(Scanline, 2, 2, 8to24,  RENDER_SCANLINE_W2H2)
RENDER_METHOD(Scanline, 2, 2, 8to32,  RENDER_SCANLINE_W2H2)
RENDER_METHOD(Scanline, 2, 2, 16to8,  RENDER_SCANLINE_W2H2)
RENDER_METHOD(Scanline, 2, 2, 16to24, RENDER_SCANLINE_W2H2)
RENDER_METHOD(Scanline, 2, 2, 16to32, RENDER_SCANLINE_W2H2)


/***********************
 filter defines       */

void SuperEagle(uint8 *srcPtr, uint32 srcPitch, uint8 *deltaPtr,
				uint8 *dstPtr, uint32 dstPitch, int width, int height);

void _2xSaI(uint8 *srcPtr, uint32 srcPitch, uint8 *deltaPtr,
			uint8 *dstPtr, uint32 dstPitch, int width, int height);

void Super2xSaI(uint8 *srcPtr, uint32 srcPitch, uint8 *deltaPtr,
				uint8 *dstPtr, uint32 dstPitch, int width, int height);

void Scale_2xSaI (uint8 *srcPtr, uint32 srcPitch, uint8 *deltaPtr,
				  uint8 *dstPtr, uint32 dstPitch, 
				  uint32 dstWidth, uint32 dstHeight, int width, int height);

void TVMode(uint8 *srcPtr, uint32 srcPitch, uint8 *deltaPtr,
			uint8 *dstPtr, uint32 dstPitch, int width, int height);

void hq2x(uint8 *srcPtr, uint32 srcPitch, uint8 *deltaPtr,
		  uint8 *dstPtr, uint32 dstPitch, int width, int height);

void lq2x(uint8 *srcPtr, uint32 srcPitch, uint8 *deltaPtr,
		  uint8 *dstPtr, uint32 dstPitch, int width, int height);


/***********************
 filter method macros */

#define RENDER_SuperEagle(depth)\
	SuperEagle (pSrc, Src->Pitch,										\
				ChangeLog [GUI.FlipCounter % DirectX.NumFlipFrames],	\
				pDst, Dst->Pitch, width, height);

#define RENDER_2xSaI(depth)\
	_2xSaI (pSrc, Src->Pitch,										\
			ChangeLog [GUI.FlipCounter % DirectX.NumFlipFrames],	\
			pDst, Dst->Pitch, width, height);

#define RENDER_Super2xSaI(depth)\
	Super2xSaI (pSrc, Src->Pitch,										\
				ChangeLog [GUI.FlipCounter % DirectX.NumFlipFrames],	\
				pDst, Dst->Pitch, width, height);

#define RENDER_Scale_2xSaI(depth)\
	Scale_2xSaI (pSrc, Src->Pitch,										\
				 ChangeLog [GUI.FlipCounter % DirectX.NumFlipFrames],	\
				 pDst, Dst->Pitch,										\
				 dstRect->right - dstRect->left, dstRect->bottom - dstRect->top, \
				 width, height);

#define RENDER_TVMode(depth)\
	TVMode (pSrc, Src->Pitch,											\
			ChangeLog [GUI.FlipCounter % DirectX.NumFlipFrames],		\
			pDst, Dst->Pitch, width, height);

#define RENDER_hq2x(depth)\
	hq2x (pSrc, Src->Pitch,												\
		  ChangeLog [GUI.FlipCounter % DirectX.NumFlipFrames],			\
		  pDst, Dst->Pitch, width, height);

#define RENDER_lq2x(depth)\
	lq2x (pSrc, Src->Pitch,												\
		  ChangeLog [GUI.FlipCounter % DirectX.NumFlipFrames],			\
		  pDst, Dst->Pitch, width, height);

#define RENDER_FILTER(depth, filter, convert_depth)\
	if(convert_depth) {													\
		uint8 *pdst = pDst;												\
		SSurface *dst = Dst;											\
		SSurface ddst;													\
		pDst = ddst.Surface = depthconv_buf;							\
		ddst.Width = dstRect->right - dstRect->left;					\
		ddst.Height = dstRect->bottom - dstRect->top;					\
		ddst.Pitch = SCREEN_BUFFER_PITCH_16;							\
		Dst = &ddst;													\
		RECT *dstrect = dstRect;										\
		RECT ddstrect;													\
		ddstrect.top = 0;												\
		ddstrect.left = 0;												\
		ddstrect.right = ddst.Width;									\
		ddstrect.bottom = ddst.Height;									\
		dstRect = &ddstrect;											\
		filter(depth);														\
		Src = Dst;														\
		srcRect = dstRect;												\
		pSrc = pDst;													\
		Dst = dst;														\
		dstRect = dstrect;												\
		pDst = pdst;													\
		for(int height = (srcRect->bottom - srcRect->top); height; height--) { \
			RENDER_LINE_W1H1(depth);									\
		}																\
	}																	\
	else {																\
		filter(depth);														\
	}

#define FILTER_METHOD(name, w, h, depth, filter, convert_depth)\
static void Render##name##W##w##H##h##depth(SSurface *Src, SSurface *Dst, RECT *srcRect, RECT *dstRect) \
{																		\
	uint8 *pSrc = (Src->Surface + srcRect->top * Src->Pitch + srcRect->left * SRC_PIXSIZE_##depth);	\
	uint8 *pDst = (Dst->Surface + dstRect->top * Dst->Pitch + dstRect->left * DST_PIXSIZE_##depth);	\
	int width = (srcRect->right - srcRect->left);						\
	int height = (srcRect->bottom - srcRect->top);						\
	if(OldRenderMethod != RenderMethod || ExtendHeight_previos != ExtendHeight) {\
		for(int i = 0; i < MAX_CHANGELOG; i++)							\
			memset(ChangeLog[i], 0 , SCREEN_BUFFER_SIZE_16);			\
		memset(depthconv_buf, 0, SCREEN_BUFFER_SIZE_16);				\
		CLEAR_DEST(depth);												\
	}																	\
	RENDER_FILTER(depth, filter, convert_depth);						\
	ExtendHeight_previos = ExtendHeight;								\
}

// SuperEagle
FILTER_METHOD(SuperEagle, 2, 2, 16to8,  RENDER_SuperEagle, TRUE)
FILTER_METHOD(SuperEagle, 2, 2, 16to16, RENDER_SuperEagle, FALSE)
FILTER_METHOD(SuperEagle, 2, 2, 16to24, RENDER_SuperEagle, TRUE)
FILTER_METHOD(SuperEagle, 2, 2, 16to32, RENDER_SuperEagle, TRUE)

// 2xSaI
FILTER_METHOD(2xSaI, 2, 2, 16to8,  RENDER_2xSaI, TRUE)
FILTER_METHOD(2xSaI, 2, 2, 16to16, RENDER_2xSaI, FALSE)
FILTER_METHOD(2xSaI, 2, 2, 16to24, RENDER_2xSaI, TRUE)
FILTER_METHOD(2xSaI, 2, 2, 16to32, RENDER_2xSaI, TRUE)

// Super2xSaI
FILTER_METHOD(Super2xSaI, 2, 2, 16to8,  RENDER_Super2xSaI, TRUE)
FILTER_METHOD(Super2xSaI, 2, 2, 16to16, RENDER_Super2xSaI, FALSE)
FILTER_METHOD(Super2xSaI, 2, 2, 16to24, RENDER_Super2xSaI, TRUE)
FILTER_METHOD(Super2xSaI, 2, 2, 16to32, RENDER_Super2xSaI, TRUE)

// Scale2xSaI
FILTER_METHOD(Scale2xSaI, 2, 2, 16to8,  RENDER_Scale_2xSaI, TRUE)
FILTER_METHOD(Scale2xSaI, 2, 2, 16to16, RENDER_Scale_2xSaI, FALSE)
FILTER_METHOD(Scale2xSaI, 2, 2, 16to24, RENDER_Scale_2xSaI, TRUE)
FILTER_METHOD(Scale2xSaI, 2, 2, 16to32, RENDER_Scale_2xSaI, TRUE)

// TV
FILTER_METHOD(TV, 2, 2, 16to8,  RENDER_TVMode, TRUE)
FILTER_METHOD(TV, 2, 2, 16to16, RENDER_TVMode, FALSE)
FILTER_METHOD(TV, 2, 2, 16to24, RENDER_TVMode, TRUE)
FILTER_METHOD(TV, 2, 2, 16to32, RENDER_TVMode, TRUE)
FILTER_METHOD(TV, 1, 2, 16to8,  RENDER_TVMode, TRUE)
FILTER_METHOD(TV, 1, 2, 16to16, RENDER_TVMode, FALSE)
FILTER_METHOD(TV, 1, 2, 16to24, RENDER_TVMode, TRUE)
FILTER_METHOD(TV, 1, 2, 16to32, RENDER_TVMode, TRUE)

// HQ2X
FILTER_METHOD(HQ2X, 2, 2, 16to8,  RENDER_hq2x, TRUE)
FILTER_METHOD(HQ2X, 2, 2, 16to16, RENDER_hq2x, FALSE)
FILTER_METHOD(HQ2X, 2, 2, 16to24, RENDER_hq2x, TRUE)
FILTER_METHOD(HQ2X, 2, 2, 16to32, RENDER_hq2x, TRUE)
FILTER_METHOD(HQ2X, 1, 2, 16to8,  RENDER_hq2x, TRUE)
FILTER_METHOD(HQ2X, 1, 2, 16to16, RENDER_hq2x, FALSE)
FILTER_METHOD(HQ2X, 1, 2, 16to24, RENDER_hq2x, TRUE)
FILTER_METHOD(HQ2X, 1, 2, 16to32, RENDER_hq2x, TRUE)

// LQ2X
FILTER_METHOD(LQ2X, 2, 2, 16to8,  RENDER_lq2x, TRUE)
FILTER_METHOD(LQ2X, 2, 2, 16to16, RENDER_lq2x, FALSE)
FILTER_METHOD(LQ2X, 2, 2, 16to24, RENDER_lq2x, TRUE)
FILTER_METHOD(LQ2X, 2, 2, 16to32, RENDER_lq2x, TRUE)
FILTER_METHOD(LQ2X, 1, 2, 16to8,  RENDER_lq2x, TRUE)
FILTER_METHOD(LQ2X, 1, 2, 16to16, RENDER_lq2x, FALSE)
FILTER_METHOD(LQ2X, 1, 2, 16to24, RENDER_lq2x, TRUE)
FILTER_METHOD(LQ2X, 1, 2, 16to32, RENDER_lq2x, TRUE)

static inline void CreatePalette8to16()
{
	uint32 levels[32];
	uint32 pixel;
	for (int l = 0; l < 32; l++)
		levels [l] = (l * IPPU_MaxBrightness * SNES_MULBRIGHTNESS) >> 11;
	
#ifdef LSB_FIRST
	if (GUI.RedShift < GUI.BlueShift) {
#else	    
	if (GUI.RedShift > GUI.BlueShift) {
#endif
		// Order is RGB
		for (int p = 0; p < 256; p++) {
			pixel = PPU_CGDATA[p];
			*(uint16 *)(&render_palette[p * 2]) = (levels[(pixel >> 10) & 0x1f] << GUI.RedShift) |
			                                      (levels[(pixel >> 5) & 0x1f] << GUI.GreenShift) |
			                                      (levels[pixel & 0x1f] << GUI.BlueShift);
		}
	}
	else {
		// Order is BGR
		for (int p = 0; p < 256; p++) {
			pixel = PPU_CGDATA[p];
			*(uint16 *)(&render_palette[p * 2]) = (levels[pixel & 0x1f] << GUI.RedShift) |
			                                      (levels[(pixel >> 5) & 0x1f] << GUI.GreenShift) |
			                                      (levels[(pixel >> 10) & 0x1f] << GUI.BlueShift);
		}
	}
}

static inline void CreatePalette8to24()
{
	uint32 levels[32];
	uint32 pixel;
	for (int l = 0; l < 32; l++)
		levels [l] = (l * IPPU_MaxBrightness * SNES_MULBRIGHTNESS) >> 8;
	
#ifdef LSB_FIRST
	if (GUI.RedShift < GUI.BlueShift) {
#else	    
	if (GUI.RedShift > GUI.BlueShift) {
#endif
		// Order is RGB
		for (int p = 0; p < 256; p++) {
			pixel = PPU_CGDATA[p];
			render_palette[p * 3]     = levels [(pixel & 0x1f)];
			render_palette[p * 3 + 1] = levels [((pixel >> 5) & 0x1f)];
			render_palette[p * 3 + 2] = levels [((pixel >> 10) & 0x1f)];
		}
	}
	else {
		// Order is BGR
		for (int p = 0; p < 256; p++) {
			pixel = PPU_CGDATA[p];
			render_palette[p * 3]     = levels [((pixel >> 10) & 0x1f)];
			render_palette[p * 3 + 1] = levels [((pixel >> 5) & 0x1f)];
			render_palette[p * 3 + 2] = levels [(pixel & 0x1f)];
		}
	}
}

static inline void CreatePalette8to32()
{
	uint32 levels[32];
	uint32 pixel;
	for (int l = 0; l < 32; l++)
		levels [l] = (l * IPPU_MaxBrightness * SNES_MULBRIGHTNESS) >> 8;
	
#ifdef LSB_FIRST
	if (GUI.RedShift < GUI.BlueShift) {
#else	    
	if (GUI.RedShift > GUI.BlueShift) {
#endif
		// Order is RGB
		for (int p = 0; p < 256; p++) {
			pixel = PPU_CGDATA[p];
			render_palette[p * 4]     = levels [(pixel & 0x1f)];
			render_palette[p * 4 + 1] = levels [((pixel >> 5) & 0x1f)];
			render_palette[p * 4 + 2] = levels [((pixel >> 10) & 0x1f)];
		}
	}
	else {
		// Order is BGR
		for (int p = 0; p < 256; p++) {
			pixel = PPU_CGDATA[p];
			render_palette[p * 4]     = levels [((pixel >> 10) & 0x1f)];
			render_palette[p * 4 + 1] = levels [((pixel >> 5) & 0x1f)];
			render_palette[p * 4 + 2] = levels [(pixel & 0x1f)];
		}
	}
}

static inline void CreatePalette16to24()
{
	uint32 pixel;
#ifdef LSB_FIRST
	if (GUI.RedShift < GUI.BlueShift) {
#else	    
	if (GUI.RedShift > GUI.BlueShift) {
#endif
		// Order is RGB
		for(pixel = 0; pixel < 0x10000; pixel++) {
			render_palette[pixel * 3]     = (pixel >> (11 - 3)) & 0xf8;
			render_palette[pixel * 3 + 1] = (pixel >> (6 - 3)) & 0xf8;
			render_palette[pixel * 3 + 2] = (pixel & 0x1f) << 3;
		}
	}
	else {
		// Order is BGR
		for(pixel = 0; pixel < 0x10000; pixel++) {
			render_palette[pixel * 3]     = (pixel & 0x1f) << 3;
			render_palette[pixel * 3 + 1] = (pixel >> (6 - 3)) & 0xf8;
			render_palette[pixel * 3 + 2] = (pixel >> (11 - 3)) & 0xf8;
		}
	}
}

static inline void CreatePalette16to32()
{
	uint32 pixel;
#ifdef LSB_FIRST
	if (GUI.RedShift < GUI.BlueShift) {
#else	    
	if (GUI.RedShift > GUI.BlueShift) {
#endif
		// Order is RGB
		for(pixel = 0; pixel < 0x10000; pixel++) {
			render_palette[pixel * 4]     = (pixel >> (11 - 3)) & 0xf8;
			render_palette[pixel * 4 + 1] = (pixel >> (6 - 3)) & 0xf8;
			render_palette[pixel * 4 + 2] = (pixel & 0x1f) << 3;
			render_palette[pixel * 4 + 3] = 0;
		}
	}
	else {
		// Order is BGR
		for(pixel = 0; pixel < 0x10000; pixel++) {
			render_palette[pixel * 4]     = (pixel & 0x1f) << 3;
			render_palette[pixel * 4 + 1] = (pixel >> (6 - 3)) & 0xf8;
			render_palette[pixel * 4 + 2] = (pixel >> (11 - 3)) & 0xf8;
			render_palette[pixel * 4 + 3] = 0;
		}
	}
}

#define RENDER_METHOD_CASE_FILTER0(num, name, depth)\
	case num:													\
		if(softwarescale) {										\
			if(DoubleWidth && !DoubleHeight)					\
				RenderMethod = RenderW1H2##depth;				\
			else if(DoubleHeight && !DoubleWidth)				\
				RenderMethod = RenderW2H1##depth;				\
			else if(DoubleHeight && DoubleWidth)				\
				RenderMethod = RenderW1H1##depth;				\
			else												\
				RenderMethod = Render##name##W2H2##depth;		\
		}														\
		else {													\
			if(!DoubleWidth && !DoubleHeight)					\
				RenderMethod = Render##name##W2H2##depth;		\
			else												\
				RenderMethod = RenderW1H1##depth;				\
		}														\
	    break;

#define RENDER_METHOD_CASE_FILTER(num, name, depth)\
	case num:													\
		if(softwarescale) {										\
			if(DoubleWidth && !DoubleHeight)					\
				RenderMethod = Render##name##W1H2##depth;		\
			else if(DoubleHeight && !DoubleWidth)				\
				RenderMethod = RenderW2H1##depth;				\
			else if(DoubleHeight && DoubleWidth)				\
				RenderMethod = RenderW1H1##depth;				\
			else												\
				RenderMethod = Render##name##W2H2##depth;		\
		}														\
		else {													\
			if(DoubleWidth && !DoubleHeight)					\
				RenderMethod = Render##name##W1H2##depth;		\
			else if(!DoubleWidth && !DoubleHeight)				\
				RenderMethod = Render##name##W2H2##depth;		\
			else												\
				RenderMethod = RenderW1H1##depth;				\
		}														\
	    break;

#define RENDER_METHOD_CASE0123(depth)\
	case 0:										\
		RenderMethod = RenderW1H1##depth;		\
        break;										\
												\
    case 1:												\
		if(softwarescale) {								\
			if(DoubleWidth && !DoubleHeight)			\
				RenderMethod = RenderW1H2##depth;		\
			else if(DoubleHeight && !DoubleWidth)		\
				RenderMethod = RenderW2H1##depth;		\
			else if(DoubleHeight && DoubleWidth)		\
				RenderMethod = RenderW1H1##depth;		\
			else										\
				RenderMethod = RenderW2H2##depth;		\
		}												\
		else {											\
			RenderMethod = RenderW1H1##depth;			\
		}												\
		break;											\
  														\
    case 2:												\
		if(DoubleWidth && !DoubleHeight)				\
			RenderMethod = RenderW1H2##depth;			\
		else if(DoubleHeight && !DoubleWidth)			\
			RenderMethod = RenderW2H1##depth;			\
		else if(DoubleHeight && DoubleWidth)			\
			RenderMethod = RenderW1H1##depth;			\
		else											\
			RenderMethod = RenderW2H2##depth;			\
		break;											\
  														\
    case 3:												\
		if(softwarescale) {										\
			if(DoubleWidth && !DoubleHeight)					\
				RenderMethod = RenderScanlineW1H2##depth;		\
			else if(DoubleHeight && !DoubleWidth)				\
				RenderMethod = RenderW2H1##depth;				\
			else if(DoubleHeight && DoubleWidth)				\
				RenderMethod = RenderW1H1##depth;				\
			else												\
				RenderMethod = RenderScanlineW2H2##depth;		\
		}														\
		else {													\
			if(DoubleHeight)									\
				RenderMethod = RenderW1H1##depth;				\
			else												\
				RenderMethod = RenderScanlineW1H2##depth;		\
		}														\
		break;

#define RENDER_METHOD_CASE_NO_FILTER(depth)\
	RENDER_METHOD_CASE0123(depth)

#define RENDER_METHOD_CASE_YES_FILTER(depth)\
	RENDER_METHOD_CASE0123(depth);					\
	RENDER_METHOD_CASE_FILTER0(4, SuperEagle, depth);	\
	RENDER_METHOD_CASE_FILTER0(5, 2xSaI, depth);			\
	RENDER_METHOD_CASE_FILTER0(6, Super2xSaI, depth);	\
	RENDER_METHOD_CASE_FILTER0(7, Scale2xSaI, depth);	\
	RENDER_METHOD_CASE_FILTER(8, TV, depth);			\
	RENDER_METHOD_CASE_FILTER(9, HQ2X, depth);			\
	RENDER_METHOD_CASE_FILTER(10, LQ2X, depth);

#define RENDER_METHOD_SWITCH_NO_FILTER(depth)\
	switch(GUI.Scale) {						 \
		RENDER_METHOD_CASE_NO_FILTER(depth); \
	default:								 \
		RenderMethod = RenderNothing;		 \
		break;								 \
	}

#define RENDER_METHOD_SWITCH_YES_FILTER(depth)\
	switch(GUI.Scale) {						  \
		RENDER_METHOD_CASE_YES_FILTER(depth); \
	default:								  \
		RenderMethod = RenderNothing;		  \
		break;								  \
	}

EXTERN_C void RenderInit()
{
	switch(Settings.SixteenBit) {
	case TRUE:
		switch(GUI.ScreenDepth) {

		case 24:
			CreatePalette16to24();
			break;

		case 32:
			CreatePalette16to32();
			break;

		default:
			break;
		}
		break;
	default:
		break;
	}
	int i;
	for(i = 0; i < MAX_CHANGELOG; i++)
		memset(ChangeLog[i], 0 , SCREEN_BUFFER_SIZE_16);
	memset(depthconv_buf, 0, SCREEN_BUFFER_SIZE_16);
}

EXTERN_C void SelectRenderMethod(bool8 softwarescale)
{
	OldRenderMethod = RenderMethod;

	switch(Settings.SixteenBit) {
	case TRUE:
		switch(GUI.ScreenDepth) {

		case 8:
			RENDER_METHOD_SWITCH_YES_FILTER(16to8);
			break;

		case 16:
			RENDER_METHOD_SWITCH_YES_FILTER(16to16);
			break;

		case 24:
			RENDER_METHOD_SWITCH_YES_FILTER(16to24);
			break;

		case 32:
			RENDER_METHOD_SWITCH_YES_FILTER(16to32);
			break;

		default:
			RenderMethod = RenderNothing;
			break;
		}
		break;

	case FALSE:
		switch(GUI.ScreenDepth) {
		case 8:
			RENDER_METHOD_SWITCH_NO_FILTER(8to8);
			break;

		case 16:
			RENDER_METHOD_SWITCH_NO_FILTER(8to16);
			CreatePalette8to16();
			break;

		case 24:
			RENDER_METHOD_SWITCH_NO_FILTER(8to24);
			CreatePalette8to24();
			break;

		case 32:
			RENDER_METHOD_SWITCH_NO_FILTER(8to32);
			CreatePalette8to32();
			break;

		default:
			RenderMethod = RenderNothing;
			break;
		}
		break;

	default:
		RenderMethod = RenderNothing;
		break;
	}
}

EXTERN_C void SelectRenderingCodes()
{
    return;
}

EXTERN_C void ResetRenderMethod()
{
	OldRenderMethod = RenderMethod;
	RenderMethod = RenderNothing;
	return;
}

