/*******************************************************************************
  Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 
  (c) Copyright 1996 - 2002 Gary Henderson (gary.henderson@ntlworld.com) and
                            Jerremy Koot (jkoot@snes9x.com)

  (c) Copyright 2001 - 2004 John Weidman (jweidman@slip.net)

  (c) Copyright 2002 - 2004 Brad Jorsch (anomie@users.sourceforge.net),
                            funkyass (funkyass@spam.shaw.ca),
                            Joel Yliluoma (http://iki.fi/bisqwit/)
                            Kris Bleakley (codeviolation@hotmail.com),
                            Matthew Kendora,
                            Nach (n-a-c-h@users.sourceforge.net),
                            Peter Bortas (peter@bortas.org) and
                            zones (kasumitokoduck@yahoo.com)

  C4 x86 assembler and some C emulation code
  (c) Copyright 2000 - 2003 zsKnight (zsknight@zsnes.com),
                            _Demo_ (_demo_@zsnes.com), and Nach

  C4 C++ code
  (c) Copyright 2003 Brad Jorsch

  DSP-1 emulator code
  (c) Copyright 1998 - 2004 Ivar (ivar@snes9x.com), _Demo_, Gary Henderson,
                            John Weidman, neviksti (neviksti@hotmail.com),
                            Kris Bleakley, Andreas Naive

  DSP-2 emulator code
  (c) Copyright 2003 Kris Bleakley, John Weidman, neviksti, Matthew Kendora, and
                     Lord Nightmare (lord_nightmare@users.sourceforge.net

  OBC1 emulator code
  (c) Copyright 2001 - 2004 zsKnight, pagefault (pagefault@zsnes.com) and
                            Kris Bleakley
  Ported from x86 assembler to C by sanmaiwashi

  SPC7110 and RTC C++ emulator code
  (c) Copyright 2002 Matthew Kendora with research by
                     zsKnight, John Weidman, and Dark Force

  S-DD1 C emulator code
  (c) Copyright 2003 Brad Jorsch with research by
                     Andreas Naive and John Weidman
 
  S-RTC C emulator code
  (c) Copyright 2001 John Weidman
  
  ST010 C++ emulator code
  (c) Copyright 2003 Feather, Kris Bleakley, John Weidman and Matthew Kendora

  Super FX x86 assembler emulator code 
  (c) Copyright 1998 - 2003 zsKnight, _Demo_, and pagefault 

  Super FX C emulator code 
  (c) Copyright 1997 - 1999 Ivar, Gary Henderson and John Weidman


  SH assembler code partly based on x86 assembler code
  (c) Copyright 2002 - 2004 Marcus Comstedt (marcus@mc.pp.se) 

 
  Specific ports contains the works of other authors. See headers in
  individual files.
 
  Snes9x homepage: http://www.snes9x.com
 
  Permission to use, copy, modify and distribute Snes9x in both binary and
  source form, for non-commercial purposes, is hereby granted without fee,
  providing that this license information and copyright notice appear with
  all copies and any derived work.
 
  This software is provided 'as-is', without any express or implied
  warranty. In no event shall the authors be held liable for any damages
  arising from the use of this software.
 
  Snes9x is freeware for PERSONAL USE only. Commercial users should
  seek permission of the copyright holders first. Commercial use includes
  charging money for Snes9x or software derived from Snes9x.
 
  The copyright holders request that bug fixes and improvements to the code
  should be forwarded to them so everyone can benefit from the modifications
  in future versions.
 
  Super NES and Super Nintendo Entertainment System are trademarks of
  Nintendo Co., Limited and its subsidiary companies.
*******************************************************************************/
#ifdef __DJGPP__
#include <allegro.h>
#undef TRUE
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#define CLIP16(v) \
	if ((v) < -32768) \
    (v) = -32768; \
	else \
	if ((v) > 32767) \
(v) = 32767

#define CLIP16_latch(v,l) \
	if ((v) < -32768) \
{ (v) = -32768; (l)++; }\
	else \
	if ((v) > 32767) \
{ (v) = 32767; (l)++; }

#define CLIP24(v) \
	if ((v) < -8388608) \
    (v) = -8388608; \
	else \
	if ((v) > 8388607) \
(v) = 8388607

#define CLIP8(v) \
	if ((v) < -128) \
    (v) = -128; \
	else \
	if ((v) > 127) \
(v) = 127

#include "snes9x.h"
#include "soundux.h"
#include "apu.h"
#include "memmap.h"
#include "cpuexec.h"
#include "snesapu.h"

extern int32 Echo [ECHOBUF];
extern int32 Loop [FIRBUF];
extern int32 FilterTaps [8];

extern unsigned long Z;

extern long FilterValues[4][2];
extern int NoiseFreq [32];

#undef ABS
#define ABS(a) ((a) < 0 ? -(a) : (a))

#define VOL_DIV8  0x8000
#define VOL_DIV16 0x0080
#define ENVX_SHIFT 8

#if 1
#define FrqScl(D)	(int)(((int64)D * pfrq)>>16)
#define PITCH_MOD(F,M) ( (( (F<<0) * (1 +((FrqScl((M>>7)) + 32768)>>1)) )>>(14+0)) )
#else
// F is channel's current frequency and M is the 16-bit modulation waveform
// from the previous channel multiplied by the current envelope volume level.
#define PITCH_MOD(F,M) ((F) * ((((unsigned long) (M)) + 0x800000) >> 16) >> 7)
//#define PITCH_MOD(F,M) ((F) * ((((M) & 0x7fffff) >> 14) + 1) >> 8)
#endif

static int64 pfrq = (1 << 16);

#define LAST_SAMPLE 0xffffff
#define JUST_PLAYED_LAST_SAMPLE(c) ((c)->sample_pointer >= LAST_SAMPLE)

STATIC inline uint8 *S9xGetSampleAddress (int sample_number)
{
    uint32 addr = (((APU.DSP[APU_DIR] << 8) + (sample_number << 2)) & 0xffff);
    return (IAPU.RAM + addr);
}

void S9xAPUSetEndOfSample (int i, Channel *ch)
{
    ch->state = SOUND_SILENT;
    ch->mode = MODE_NONE;
    APU.DSP [APU_ENDX] |= 1 << i;
    APU.DSP [APU_KON] &= ~(1 << i);
    APU.DSP [APU_KOFF] &= ~(1 << i);
    APU.KeyedChannels &= ~(1 << i);
}
#ifdef __DJGPP
END_OF_FUNCTION (S9xAPUSetEndOfSample)
#endif

void S9xAPUSetEndX (int ch)
{
    APU.DSP [APU_ENDX] |= 1 << ch;
}
#ifdef __DJGPP
END_OF_FUNCTION (S9xAPUSetEndX)
#endif

void S9xSetEnvRate (Channel *ch, unsigned long rate, int direction, int target)
{
    ch->envx_target = target;
	
    if (rate == ~0UL)
    {
		ch->direction = 0;
		rate = 0;
    }
    else
		ch->direction = direction;
	
    static int steps [] =
    {
		//	0, 64, 1238, 1238, 256, 1, 64, 109, 64, 1238
		0, 64, 619, 619, 128, 1, 64, 55, 64, 619
    };
	
    if (rate == 0 || so.playback_rate == 0)
		ch->erate = 0;
    else
    {
		ch->erate = (unsigned long)
			(((int64) FIXED_POINT * 1000 * steps [ch->state]) /
			(rate * so.playback_rate));
    }
}

#ifdef __DJGPP
END_OF_FUNCTION(S9xSetEnvRate);
#endif

void S9xSetEnvelopeRate (int channel, unsigned long rate, int direction,
						 int target)
{
    S9xSetEnvRate (&SoundData.channels [channel], rate, direction, target);
}

#ifdef __DJGPP
END_OF_FUNCTION(S9xSetEnvelopeRate);
#endif

void S9xSetSoundVolume (int channel, short volume_left, short volume_right)
{
    Channel *ch = &SoundData.channels[channel];
    if (!so.stereo)
		volume_left = (ABS(volume_right) + ABS(volume_left)) / 2;
	
    ch->volume_left = volume_left;
    ch->volume_right = volume_right;
    ch-> left_vol_level = (ch->envx * volume_left) / 128;
    ch->right_vol_level = (ch->envx * volume_right) / 128;
}

void S9xSetMasterVolume (short volume_left, short volume_right)
{
    if (Settings.DisableMasterVolume || SNESGameFixes.EchoOnlyOutput)
    {
		SoundData.master_volume_left = 127;
		SoundData.master_volume_right = 127;
		SoundData.master_volume [0] = SoundData.master_volume [1] = 127;
    }
    else
    {
		if (!so.stereo)
			volume_left = (ABS (volume_right) + ABS (volume_left)) / 2;
		SoundData.master_volume_left = volume_left;
		SoundData.master_volume_right = volume_right;
		SoundData.master_volume [Settings.ReverseStereo] = volume_left;
		SoundData.master_volume [1 ^ Settings.ReverseStereo] = volume_right;
    }
}

void S9xSetEchoVolume (short volume_left, short volume_right)
{
	if (!so.stereo)
	volume_left = (ABS (volume_right) + ABS (volume_left)) / 2;
	SoundData.echo_volume_left = volume_left;
	SoundData.echo_volume_right = volume_right;
	SoundData.echo_volume [Settings.ReverseStereo] = volume_left;
	SoundData.echo_volume [1 ^ Settings.ReverseStereo] = volume_right;
}

void S9xSetEchoEnable (uint8 byte)
{
    SoundData.echo_channel_enable = byte;
    if (!SoundData.echo_write_enabled || Settings.DisableSoundEcho)
		byte = 0;
    if (byte && !SoundData.echo_enable)
    {
		memset (Echo, 0, sizeof(int32) * ECHOBUF);
		memset (Loop, 0, sizeof(int32) * FIRBUF);
    }
	
    SoundData.echo_enable = byte;
    for (int i = 0; i < 8; i++)
    {
#if 0
		if (byte & (1 << i))
			SoundData.channels [i].echo_buf_ptr = EchoBuffer;
		else
			SoundData.channels [i].echo_buf_ptr = DummyEchoBuffer;
#else
		SoundData.channels [i].echo_buf_ptr = NULL;
#endif
    }
}

void S9xSetEchoFeedback (int feedback)
{
	CLIP8(feedback);
	SoundData.echo_feedback = feedback;
}

void S9xSetEchoDelay (int delay)
{
	SoundData.echo_buffer_size = (512 * delay * so.playback_rate) / SNES_SAMPLE_RATE;
	if (so.stereo)
		SoundData.echo_buffer_size <<= 1;
	if (SoundData.echo_buffer_size)
		SoundData.echo_ptr %= SoundData.echo_buffer_size;
	else
		SoundData.echo_ptr = 0;
	S9xSetEchoEnable (APU.DSP [APU_EON]);
}

void S9xSetEchoWriteEnable (uint8 byte)
{
	SoundData.echo_write_enabled = byte;
	S9xSetEchoDelay (APU.DSP [APU_EDL] & 15);
}

void S9xSetFrequencyModulationEnable (uint8 byte)
{
	SoundData.pitch_mod = byte & ~1;
}

void S9xSetSoundKeyOff (int channel)
{
    Channel *ch = &SoundData.channels[channel];
	
    if (ch->state != SOUND_SILENT)
    {
		ch->state = SOUND_RELEASE;
		ch->mode = MODE_RELEASE;
		S9xSetEnvRate (ch, 8, -1, 0);
    }
}

void S9xFixSoundAfterSnapshotLoad ()
{
	SoundData.echo_write_enabled = !(APU.DSP [APU_FLG] & 0x20);
	SoundData.echo_channel_enable = APU.DSP [APU_EON];
	S9xSetEchoDelay (APU.DSP [APU_EDL] & 0xf);
	S9xSetEchoFeedback ((signed char) APU.DSP [APU_EFB]);

	S9xSetFilterCoefficient (0, (signed char) APU.DSP [APU_C0]);
	S9xSetFilterCoefficient (1, (signed char) APU.DSP [APU_C1]);
	S9xSetFilterCoefficient (2, (signed char) APU.DSP [APU_C2]);
	S9xSetFilterCoefficient (3, (signed char) APU.DSP [APU_C3]);
	S9xSetFilterCoefficient (4, (signed char) APU.DSP [APU_C4]);
	S9xSetFilterCoefficient (5, (signed char) APU.DSP [APU_C5]);
	S9xSetFilterCoefficient (6, (signed char) APU.DSP [APU_C6]);
	S9xSetFilterCoefficient (7, (signed char) APU.DSP [APU_C7]);

	S9xSetMasterVolume (SoundData.master_volume_left, SoundData.master_volume_right);
	S9xSetEchoVolume (SoundData.echo_volume_left, SoundData.echo_volume_right);
	//S9xSetEchoEnable (APU.DSP [APU_EON]);

    uint32	i;
	for (i = 0; i < 8; i++)
	{
	SoundData.channels[i].needs_decode = TRUE;
	S9xSetSoundFrequency (i, SoundData.channels[i].hertz);
	SoundData.channels [i].envxx = SoundData.channels [i].envx << ENVX_SHIFT;
	SoundData.channels [i].next_sample = 0;
	SoundData.channels [i].interpolate = 0;
	SoundData.channels [i].previous [0] = (int32) SoundData.channels [i].previous16 [0];
	SoundData.channels [i].previous [1] = (int32) SoundData.channels [i].previous16 [1];
	}
	SoundData.master_volume [Settings.ReverseStereo] = SoundData.master_volume_left;
	SoundData.master_volume [1 ^ Settings.ReverseStereo] = SoundData.master_volume_right;
	SoundData.echo_volume [Settings.ReverseStereo] = SoundData.echo_volume_left;
	SoundData.echo_volume [1 ^ Settings.ReverseStereo] = SoundData.echo_volume_right;
	IAPU.Scanline = 0;

	//SNESAPU
	RestoreAPUDSP();
}

void S9xSetFilterCoefficient (int tap, int value)
{
	FilterTaps [tap & 7] = (int32)value;
	SoundData.no_filter = (FilterTaps [0] == 127 || FilterTaps [0] == 0) &&
			   FilterTaps [1] == 0	 &&
			   FilterTaps [2] == 0	 &&
			   FilterTaps [3] == 0	 &&
			   FilterTaps [4] == 0	 &&
			   FilterTaps [5] == 0	 &&
			   FilterTaps [6] == 0	 &&
			   FilterTaps [7] == 0;
}
void S9xSetSoundADSR (int channel, int attack_rate, int decay_rate,
					  int sustain_rate, int sustain_level, int release_rate)
{
    SoundData.channels[channel].attack_rate = attack_rate;
    SoundData.channels[channel].decay_rate = decay_rate;
    SoundData.channels[channel].sustain_rate = sustain_rate;
    SoundData.channels[channel].release_rate = release_rate;
    SoundData.channels[channel].sustain_level = sustain_level + 1;
	
    switch (SoundData.channels[channel].state)
    {
    case SOUND_ATTACK:
		S9xSetEnvelopeRate (channel, attack_rate, 1, 127);
		break;
		
    case SOUND_DECAY:
		S9xSetEnvelopeRate (channel, decay_rate, -1,
			(MAX_ENVELOPE_HEIGHT * (sustain_level + 1)) >> 3);
		break;
    case SOUND_SUSTAIN:
		S9xSetEnvelopeRate (channel, sustain_rate, -1, 0);
		break;
    }
}

void S9xSetEnvelopeHeight (int channel, int level)
{
    Channel *ch = &SoundData.channels[channel];
	
    ch->envx = level;
    ch->envxx = level << ENVX_SHIFT;
	
    ch->left_vol_level = (level * ch->volume_left) / 128;
    ch->right_vol_level = (level * ch->volume_right) / 128;
	
    if (ch->envx == 0 && ch->state != SOUND_SILENT && ch->state != SOUND_GAIN)
    {
		S9xAPUSetEndOfSample (channel, ch);
    }
}

int S9xGetEnvelopeHeight (int channel)
{
    if ((Settings.SoundEnvelopeHeightReading ||
		SNESGameFixes.SoundEnvelopeHeightReading2) &&
        SoundData.channels[channel].state != SOUND_SILENT &&
        SoundData.channels[channel].state != SOUND_GAIN)
    {
        return (SoundData.channels[channel].envx);
    }
	
    //siren fix from XPP
    if (SNESGameFixes.SoundEnvelopeHeightReading2 &&
        SoundData.channels[channel].state != SOUND_SILENT)
    {
        return (SoundData.channels[channel].envx);
    }
	
    return (0);
}

#if 1
void S9xSetSoundSample (int, uint16) 
{
}
#else
void S9xSetSoundSample (int channel, uint16 sample_number)
{
    register Channel *ch = &SoundData.channels[channel];
	
    if (ch->state != SOUND_SILENT && 
		sample_number != ch->sample_number)
    {
		int keep = ch->state;
		ch->state = SOUND_SILENT;
		ch->sample_number = sample_number;
		ch->loop = FALSE;
		ch->needs_decode = TRUE;
		ch->last_block = FALSE;
		ch->previous [0] = ch->previous[1] = 0;
		uint8 *dir = S9xGetSampleAddress (sample_number);
		ch->block_pointer = READ_WORD (dir);
		ch->sample_pointer = 0;
		ch->state = keep;
    }
}
#endif

void S9xSetSoundFrequency (int channel, int hertz)
{
    if (so.playback_rate)
    {
		if (SoundData.channels[channel].type == SOUND_NOISE)
			hertz = NoiseFreq [APU.DSP [APU_FLG] & 0x1f];
		SoundData.channels[channel].frequency = (int)
			(((int64) hertz * FIXED_POINT) / so.playback_rate);
		if (Settings.FixFrequency)
		{
			SoundData.channels[channel].frequency = 
				(unsigned long) ((double)  SoundData.channels[channel].frequency * 0.980);
		}
    }
}

void S9xSetSoundHertz (int channel, int hertz)
{
    SoundData.channels[channel].hertz = hertz;
    S9xSetSoundFrequency (channel, hertz);
}

void S9xSetSoundType (int channel, int type_of_sound)
{
    SoundData.channels[channel].type = type_of_sound;
}

bool8 S9xSetSoundMute (bool8 mute)
{
    bool8 old = so.mute_sound;
    so.mute_sound = mute;
    return (old);
}

#ifdef __sun
extern uint8 int2ulaw (int);
#endif

// For backwards compatibility with older port specific code
void S9xMixSamples (uint8 *buffer, int sample_count)
{
    S9xMixSamplesO (buffer, sample_count, 0);
}
#ifdef __DJGPP
END_OF_FUNCTION(S9xMixSamples);
#endif

void S9xMixSamplesO (uint8 *buffer, int sample_count, int byte_offset)
{
	if (!so.mute_sound) {
		if(Settings.Mute) {
			EmuAPUDSP(NULL, sample_count >> (so.stereo ? 1 : 0));
			memset(buffer + byte_offset,
				   so.sixteen_bit ? 0 : 128,
				   sample_count << (so.sixteen_bit ? 1 : 0));
		}
		else {
			EmuAPUDSP(buffer + byte_offset,
					  sample_count >> (so.stereo ? 1 : 0));
		}
	}
	else {
		memset(buffer + byte_offset,
			   so.sixteen_bit ? 0 : 128,
			   sample_count << (so.sixteen_bit ? 1 : 0));
	}
	return;
}
#ifdef __DJGPP
END_OF_FUNCTION(S9xMixSamplesO);
#endif

void S9xFixSoundSyncCycles ()
{
	if(Memory.FillRAM[0x2133] & 1) {
		if(!(IPPU.FrameCount & 1))
			so.err_rate = so.err_rate_long;
		else
			so.err_rate = so.err_rate_normal;
	}
	else {
		if(!(IPPU.FrameCount & 1))
			so.err_rate = so.err_rate_normal;
		else
			so.err_rate = so.err_rate_short;
	}
}

static void S9xCalcSoundSyncRate()
{
	if(so.playback_rate == 0 || Memory.ROMFramesPerSecond == 0)
		return;

	if(!Settings.SPCPlaying) {
		so.err_rate_normal = (uint32)((((uint64)so.playback_rate << FIXED_POINT_SHIFT) *
									   (SNES_CYCLES_PER_SECOND / (SNES_CYCLES_PER_SCANLINE * SNES_MAX_VCOUNTER)))
									  / (64000 * Memory.ROMFramesPerSecond));
		so.err_rate_long = (uint32)((((uint64)so.playback_rate << FIXED_POINT_SHIFT) *
									 (SNES_CYCLES_PER_SECOND / (SNES_CYCLES_PER_SCANLINE * (SNES_MAX_VCOUNTER + 1))))
									/ (64000 * Memory.ROMFramesPerSecond));
		so.err_rate_short = (uint32)((((uint64)so.playback_rate << FIXED_POINT_SHIFT) *
									  (SNES_CYCLES_PER_SECOND / (SNES_CYCLES_PER_SCANLINE * SNES_MAX_VCOUNTER - DOT_CYCLE)))
									 / (64000 * Memory.ROMFramesPerSecond));
	}
	else {
		so.err_rate_normal = (uint32)(((uint64)so.playback_rate << FIXED_POINT_SHIFT) / 64000);
		so.err_rate_long = so.err_rate_normal;
		so.err_rate_short = so.err_rate_normal;
	}

	S9xFixSoundSyncCycles ();
}

void S9xResetSound (bool8 full)
{
	for (int i = 0; i < 8; i++)
	{
	SoundData.channels[i].state = SOUND_SILENT;
	SoundData.channels[i].mode = MODE_NONE;
	SoundData.channels[i].type = SOUND_SAMPLE;
	SoundData.channels[i].volume_left = 0;
	SoundData.channels[i].volume_right = 0;
	SoundData.channels[i].hertz = 0;
	SoundData.channels[i].count = 0;
	SoundData.channels[i].loop = FALSE;
	SoundData.channels[i].envx_target = 0;
	SoundData.channels[i].env_error = 0;
	SoundData.channels[i].erate = 0;
	SoundData.channels[i].envx = 0;
	SoundData.channels[i].envxx = 0;
	SoundData.channels[i].left_vol_level = 0;
	SoundData.channels[i].right_vol_level = 0;
	SoundData.channels[i].direction = 0;
	SoundData.channels[i].attack_rate = 0;
	SoundData.channels[i].decay_rate = 0;
	SoundData.channels[i].sustain_rate = 0;
	SoundData.channels[i].release_rate = 0;
	SoundData.channels[i].sustain_level = 0;
	SoundData.echo_ptr = 0;
	SoundData.echo_feedback = 0;
	SoundData.echo_buffer_size = 1;
	}
	//FilterTaps [0] = 127;
	FilterTaps [0] = 0;
	FilterTaps [1] = 0;
	FilterTaps [2] = 0;
	FilterTaps [3] = 0;
	FilterTaps [4] = 0;
	FilterTaps [5] = 0;
	FilterTaps [6] = 0;
	FilterTaps [7] = 0;
	so.mute_sound = TRUE;
	so.noise_gen = 1;
	S9xSetSoundControl (255);
	so.samples_mixed_so_far = 0;
	so.play_position = 0;
	so.err_counter = 0;
	so.t64Cnt = EXT.t64Cnt;

	if(full) {
		SoundData.master_volume_left = 0;
		SoundData.master_volume_right = 0;
		SoundData.echo_volume_left = 0;
		SoundData.echo_volume_right = 0;
		SoundData.echo_enable = 0;
		SoundData.echo_write_enabled = 0;
		SoundData.echo_channel_enable = 0;
		SoundData.pitch_mod = 0;
		SoundData.dummy[0] = 0;
		SoundData.dummy[1] = 0;
		SoundData.dummy[2] = 0;
		SoundData.master_volume[0] = 0;
		SoundData.master_volume[1] = 0;
		SoundData.echo_volume[0] = 0;
		SoundData.echo_volume[1] = 0;
		SoundData.noise_hertz = 0;

		memset (Echo, 0, sizeof(int32) * ECHOBUF);
		memset (Loop, 0, sizeof(int32) * FIRBUF);
	}

	S9xCalcSoundSyncRate();
	SoundData.no_filter = TRUE;
}

void S9xSetPlaybackRate (uint32 playback_rate)
{
	so.playback_rate = playback_rate;

	S9xCalcSoundSyncRate();
	S9xSetEchoDelay (APU.DSP [APU_EDL] & 0xf);
	for (int i = 0; i < 8; i++)
	S9xSetSoundFrequency (i, SoundData.channels [i].hertz);

	pfrq = ((so.playback_rate<<16) / (2048000/64));

	int opts =
		(Settings.ReverseStereo ? DSP_REVERSE : 0) |
		(Settings.DisableSoundEcho ? DSP_NOECHO : 0);

	//SNESAPU
    SetAPUDSPOpt(MIX_INT,
				 so.stereo ? 2 : 1,
				 so.sixteen_bit ? 16 : 8,
				 so.playback_rate,
				 Settings.InterpolatedSound,
				 opts);
}

bool8 S9xInitSound (int mode, bool8 stereo, int buffer_size)
{
    so.sound_fd = -1;
	S9xSetSoundControl (255);
	
    so.playback_rate = 0;
    so.buffer_size = 0;
    so.stereo = stereo;
    so.sixteen_bit = Settings.SixteenBitSound;
    so.encoded = FALSE;
    
    S9xResetSound (TRUE);
	
    if (!(mode & 7))
		return (1);
	
    S9xSetSoundMute (TRUE);
    if (!S9xOpenSoundDevice (mode, stereo, buffer_size))
    {
#ifdef NOSOUND
                S9xMessage (S9X_WARNING, S9X_SOUND_NOT_BUILT,
                            "No sound support compiled in");
#else
		S9xMessage (S9X_ERROR, S9X_SOUND_DEVICE_OPEN_FAILED,
                            "Sound device open failed");
#endif
		return (0);
    }
	
    return (1);
}

bool8 S9xSetSoundMode (int channel, int mode)
{
    Channel *ch = &SoundData.channels[channel];
	
    switch (mode)
    {
    case MODE_RELEASE:
		if (ch->mode != MODE_NONE)
		{
			ch->mode = MODE_RELEASE;
			return (TRUE);
		}
		break;
		
    case MODE_DECREASE_LINEAR:
    case MODE_DECREASE_EXPONENTIAL:
    case MODE_GAIN:
		if (ch->mode != MODE_RELEASE)
		{
			ch->mode = mode;
			if (ch->state != SOUND_SILENT)
				ch->state = mode;
			
			return (TRUE);
		}
		break;
		
    case MODE_INCREASE_LINEAR:
    case MODE_INCREASE_BENT_LINE:
		if (ch->mode != MODE_RELEASE)
		{
			ch->mode = mode;
			if (ch->state != SOUND_SILENT)
				ch->state = mode;
			
			return (TRUE);
		}
		break;
		
    case MODE_ADSR:
		if (ch->mode == MODE_NONE || ch->mode == MODE_ADSR)
		{
			ch->mode = mode;
			return (TRUE);
		}
    }
	
    return (FALSE);
}

void S9xSetSoundControl (int sound_switch)
{
    so.sound_switch = sound_switch;
    for(int i=0; i<8; i++){
        if(sound_switch&(1<<i)) mix[i].mFlg &= ~MFLG_MUTE;
        else mix[i].mFlg |= MFLG_MUTE;
    }
}

void S9xPlaySample (int channel)
{
    Channel *ch = &SoundData.channels[channel];
    
    ch->state = SOUND_SILENT;
    ch->mode = MODE_NONE;
    ch->envx = 0;
    ch->envxx = 0;
	
    S9xFixEnvelope (channel,
		APU.DSP [APU_GAIN  + (channel << 4)], 
		APU.DSP [APU_ADSR1 + (channel << 4)],
		APU.DSP [APU_ADSR2 + (channel << 4)]);
	
    ch->sample_number = APU.DSP [APU_SRCN + channel * 0x10];
    if (APU.DSP [APU_NON] & (1 << channel))
		ch->type = SOUND_NOISE;
    else
		ch->type = SOUND_SAMPLE;
	
    S9xSetSoundFrequency (channel, ch->hertz);
    ch->loop = FALSE;
    ch->needs_decode = TRUE;
    ch->last_block = FALSE;
    ch->previous [0] = ch->previous[1] = 0;
    uint8 *dir = S9xGetSampleAddress (ch->sample_number);
    ch->block_pointer = READ_WORD (dir);
    ch->sample_pointer = 0;
	ch->sample = 0;
    ch->env_error = 0;
    ch->next_sample = 0;
    ch->interpolate = 0;

	ch->Filter = 0;
	ch->chkMode = 0;

    switch (ch->mode)
    {
    case MODE_ADSR:
		if (ch->attack_rate == 0)
		{
			if (ch->decay_rate == 0 || ch->sustain_level == 8)
			{
				ch->state = SOUND_SUSTAIN;
				ch->envx = (MAX_ENVELOPE_HEIGHT * ch->sustain_level) >> 3;
				S9xSetEnvRate (ch, ch->sustain_rate, -1, 0);
			}
			else
			{
				ch->state = SOUND_DECAY;
				ch->envx = MAX_ENVELOPE_HEIGHT;
				S9xSetEnvRate (ch, ch->decay_rate, -1, 
					(MAX_ENVELOPE_HEIGHT * ch->sustain_level) >> 3);
			}
			ch-> left_vol_level = (ch->envx * ch->volume_left) / 128;
			ch->right_vol_level = (ch->envx * ch->volume_right) / 128;
		}
		else
		{
			ch->state = SOUND_ATTACK;
			ch->envx = 0;
			ch->left_vol_level = 0;
			ch->right_vol_level = 0;
			S9xSetEnvRate (ch, ch->attack_rate, 1, MAX_ENVELOPE_HEIGHT);
		}
		ch->envxx = ch->envx << ENVX_SHIFT;
		break;
		
    case MODE_GAIN:
		ch->state = SOUND_GAIN;
		break;
		
    case MODE_INCREASE_LINEAR:
		ch->state = SOUND_INCREASE_LINEAR;
		break;
		
    case MODE_INCREASE_BENT_LINE:
		ch->state = SOUND_INCREASE_BENT_LINE;
		break;
		
    case MODE_DECREASE_LINEAR:
		ch->state = SOUND_DECREASE_LINEAR;
		break;
		
    case MODE_DECREASE_EXPONENTIAL:
		ch->state = SOUND_DECREASE_EXPONENTIAL;
		break;
		
    default:
		break;
    }
	
    S9xFixEnvelope (channel,
		APU.DSP [APU_GAIN  + (channel << 4)], 
		APU.DSP [APU_ADSR1 + (channel << 4)],
		APU.DSP [APU_ADSR2 + (channel << 4)]);
}

void S9xFixEnvelope (int channel, uint8 gain, uint8 adsr1, uint8 adsr2)
{
    if (adsr1 & 0x80)
    {
		// ADSR mode
		static unsigned long AttackRate [16] = {
			4100, 2600, 1500, 1000, 640, 380, 260, 160,
				96, 64, 40, 24, 16, 10, 6, 1
		};
		static unsigned long DecayRate [8] = {
			1200, 740, 440, 290, 180, 110, 74, 37
		};
		static unsigned long SustainRate [32] = {
			~0, 38000, 28000, 24000, 19000, 14000, 12000, 9400,
				7100, 5900, 4700, 3500, 2900, 2400, 1800, 1500,
				1200, 880, 740, 590, 440, 370, 290, 220,
				180, 150, 110, 92, 74, 55, 37, 18
		};
		// XXX: can DSP be switched to ADSR mode directly from GAIN/INCREASE/
		// DECREASE mode? And if so, what stage of the sequence does it start
		// at?
		if (S9xSetSoundMode (channel, MODE_ADSR))
		{
			// Hack for ROMs that use a very short attack rate, key on a 
			// channel, then switch to decay mode. e.g. Final Fantasy II.
			
			int attack = AttackRate [adsr1 & 0xf];
			
			if (attack == 1 && (!Settings.SoundSync
#if 0
#ifdef __WIN32__
                || Settings.SoundDriver != WIN_SNES9X_DIRECT_SOUND_DRIVER
#endif
#endif
                ))
				attack = 0;
			
			S9xSetSoundADSR (channel, attack,
				DecayRate [(adsr1 >> 4) & 7],
				SustainRate [adsr2 & 0x1f],
				(adsr2 >> 5) & 7, 8);
		}
    }
    else
    {
		// Gain mode
		if ((gain & 0x80) == 0)
		{
			if (S9xSetSoundMode (channel, MODE_GAIN))
			{
				S9xSetEnvelopeRate (channel, 0, 0, gain & 0x7f);
				S9xSetEnvelopeHeight (channel, gain & 0x7f);
			}
		}
		else
		{
			static unsigned long IncreaseRate [32] = {
				~0, 4100, 3100, 2600, 2000, 1500, 1300, 1000,
					770, 640, 510, 380, 320, 260, 190, 160,
					130, 96, 80, 64, 48, 40, 32, 24,
					20, 16, 12, 10, 8, 6, 4, 2
			};
			static unsigned long DecreaseRateExp [32] = {
				~0, 38000, 28000, 24000, 19000, 14000, 12000, 9400,
					7100, 5900, 4700, 3500, 2900, 2400, 1800, 1500,
					1200, 880, 740, 590, 440, 370, 290, 220,
					180, 150, 110, 92, 74, 55, 37, 18
			};
			if (gain & 0x40)
			{
				// Increase mode
				if (S9xSetSoundMode (channel, (gain & 0x20) ?
									 MODE_INCREASE_BENT_LINE :
									 MODE_INCREASE_LINEAR))
					{
						S9xSetEnvelopeRate (channel, IncreaseRate [gain & 0x1f],
											1, 127);
					}
			}
			else
			{
				uint32 rate = (gain & 0x20) ? DecreaseRateExp [gain & 0x1f] / 2 :
			IncreaseRate [gain & 0x1f];
			int mode = (gain & 0x20) ? MODE_DECREASE_EXPONENTIAL
				: MODE_DECREASE_LINEAR;
			
			if (S9xSetSoundMode (channel, mode))
				S9xSetEnvelopeRate (channel, rate, -1, 0);
			}
		}
    }
}

