/*
  SNESAPU 
  copyright (C)2003-04 Alpha-II Productions
  http://www.alpha-ii.com/

  Ported from x86 assembler to C and implementation by Brad Jorsch

  Person who rewrites this.
  Copyright (C) 2004 sanmaiwashi
  http://toriniku.sourceforge.net/
*/

#include "snes9x.h"
#include "memmap.h"
#include "apu.h"
#include "soundux.h"
#include "snesapu.h"
#include <math.h>

/* For note-triggered SPC dump support */
#include "snapshot.h"

typedef float f32;
typedef double f64;
typedef long double f80;

const static struct {
	int o;
	uint8 m;
} chs[8] = {{0x00, 0x01},
			{0x10, 0x02},
			{0x20, 0x04},
			{0x30, 0x08},
			{0x40, 0x10},
			{0x50, 0x20},
			{0x60, 0x40},
			{0x70, 0x80}};

#ifdef __UOSNES__
extern const char *S9xGetSPCFilename ();
#else
extern "C" {const char *S9xGetFilenameInc (const char *);}
#endif

#undef M_PIl
# define M_PIl		3.1415926535897932384626433832795029L  /* pi */

#define FIR_LINEAR_INTERPOLATION
//#define CALC_T64CNTS
#define	SA_SNESINT

//#define APUDSP_DEBUG
#ifdef APUDSP_DEBUG
static const char apudsp_log_name[] = "apudsp.log";
static FILE *apudsp_fs = NULL;
extern FILE *trace;
#endif

/***********************************************************************
  Defines
***********************************************************************/

// Envelope mode masks
#define E_TYPE   0x01	  // Type of adj: Constant(1/64 or 1/256) / Exp.(255/256)
#define E_DIR    0x02     // Direction: Decrease / Increase
#define E_DEST   0x04     // Destination: Default(0 or 1) / Other(x/8 or .75)
#define E_ADSR   0x08     // Envelope mode: Gain/ADSR
#define E_IDLE   0x80     // Envelope speed is set to 0

#define E_DEC    0x00     // Linear decrease
#define E_EXP    0x01     // Exponential decrease
#define E_INC    0x02     // Linear increase
#define E_BENT   0x06     // Bent line increase
#define E_REL    0x08     // Release mode
#define E_SUST   0x09     // Sustain mode
#define E_ATT    0x0A     // Attack mode
#define E_DECAY  0x0D     // Decay mode
#define E_DIRECT 0x87     // Direct gain

// Envelope adjustment rates ------------------
#define A_GAIN   (1 << E_SHIFT)					// Amount to adjust envelope values
#define A_LIN    ((128*A_GAIN)/64)				// Linear rate to increase/decrease envelope
#define A_KOF    ((128*A_GAIN)/256)				// Rate to decrease envelope during release
#define A_BENT   ((128*A_GAIN)/256)				// Rate to increase envelope after bend
#define A_NOATT  (64*A_GAIN)					// Rate to increase if attack rate is set to 0ms
#define A_EXP    0								// Rate to decrease envelope exponentially (Not used)

// Envelope destination values ----------------
#define D_MAX    ((128*A_GAIN)-1)				// Maximum envelope value
#define D_ATTACK ((128*A_GAIN*127)/128)			// Destination of attack rate
#define D_BENT   ((128*A_GAIN*3)/4)				// First destination of bent line
#define D_EXP    ((128*A_GAIN)/8)				// Minimum decay destination value
#define D_MIN    0								// Minimum envelope value

// Others -------------------------------------
#define T_EXP    600							// 1->1/10, eVal*255/256, T_EXP times.

#undef ABS
#define ABS(a) ((a) < 0 ? -(a) : (a))

/***********************************************************************
  Variables
***********************************************************************/

static int32 brrTab[256];
static int16 cubicTab[1024];
static int16 gaussTab[1024];
#ifdef SA_SNESINT
#include "gausspre.h"
#endif


// Register write handler prototypes
static void RADSR1(int, uint8);
static void RADSR2(int, uint8);
static void REDl(int, uint8);
static void REFB(int, uint8);
static void REFBM(int, uint8);
static void REVolL(int, uint8);
static void REVolLM(int, uint8);
static void REVolR(int, uint8);
static void REVolRM(int, uint8);
static void RNOn(int, uint8);
static void RFCI(int, uint8);
static void RFlg(int, uint8);
static void RESA(int, uint8);
static void RGain(int, uint8);
static void RKOn(int, uint8);
static void RKOf(int, uint8);
static void RDir(int, uint8);
static void RMVolL(int, uint8);
static void RMVolLM(int, uint8);
static void RMVolR(int, uint8);
static void RMVolRM(int, uint8);
static void RNull(int, uint8);
static void RPMOn(int, uint8);
static void RPitch(int, uint8);
static void RSrcn(int, uint8);
static void RVolL(int, uint8);
static void RVolLM(int, uint8);
static void RVolR(int, uint8);
static void RVolRM(int, uint8);
static void REndX(int, uint8);

// Jump table for DSP register writes (see DSPIn)
static void (*dspRegs[0x80])(int, uint8) = {
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  RMVolL, REFB,   RNull,  RFCI,
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  RMVolR, RNull,  RNull,  RFCI,
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  REVolL, RPMOn,  RNull,  RFCI,
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  REVolR, RNOn,   RNull,  RFCI,
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  RKOn,   RNull,  RNull,  RFCI,
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  RKOf,   RDir,   RNull,  RFCI,
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  RFlg,   RESA,   RNull,  RFCI,
	RVolL,  RVolR,  RPitch, RPitch, RSrcn,  RADSR1, RADSR2, RGain,
	RNull,  RNull,  RNull,  RNull,  REndX,  REDl,   RNull,  RFCI
};

// Registers that require special handling for output type
static void (*dspRegsI[8])(int, uint8)= {  // Integer
    RVolL, RVolR, RMVolL, RMVolR, REVolL, REVolR, REFB, RFCI
};
static void (*dspRegsF[8])(int, uint8)= {  // Float
    RVolL, RVolR, RMVolL, RMVolR, REVolL, REVolR, REFB, RFCI
};
static void (*dspRegsM[8])(int, uint8)= {  // Monaural
    RVolLM, RVolRM, RMVolLM, RMVolRM, REVolLM, REVolRM, REFBM, RFCI
};

// Frequency table -------------------------
const static uint32 freqTab[32] = {
	   0,
	2048, 1536, 1280,// Number of samples between updates.  Used to determine
	1024,  768,  640,// envelope rates and noise frequencies
	 512,  384,  320,
	 256,  192,  160,
	 128,   96,   80,
	  64,   48,   40,
	  32,   24,   20,
	  16,   12,   10,
	   8,    6,    5,
	   4,    3,
	   2,
	   1
};

static uint64 max48;

// Prototypes for functions in mixRout
static bool8 EmuAPUDSPN(uint8 *&, int32 &);
static bool8 EmuAPUDSPI(uint8 *&, int32 &);
static bool8 EmuAPUDSPX(uint8 *&, int32 &);
static bool8 EmuAPUDSPF(uint8 *&, int32 &);

// Pointers to each mixing routine:  none, integer (386), mmx, and float
static bool8 (*mixRout[])(uint8 *&, int32 &) = {
	EmuAPUDSPN, EmuAPUDSPI, EmuAPUDSPX, EmuAPUDSPF
};

// -----------------------------------------
static uint8 *pAPURAM;             // -> APU RAM
static uint8 *pAPUExtraRAM;        // -> APU ExtraRAM 0xFFC0-0xFFFF
static uint8 src_buffer[9];        // Temporary

// Mixing ----------------------------------
VoiceMix mix[8];                   // Mixing settings for each voice and waveform playback
static uint8  voiceKon;            // Voices that are currently being key on, == APU.KeyedChannels
static uint32 rateTab[32];         // Update Rate Table

// DSP Options ----------------------
static int8   dspMix;              // Mixing routine
static int8   dspChn;              // Number of channels being output
static int8   dspSize;             // Size of samples in bytes
static int32  dspRate;             // Output sample rate
static int32  realRate;            // Real output sample rate, not emulated rate
static uint32 pitchAdj;            // Amount to adjust pitch rates [16.16]
static uint8  dspOpts;             // Option flags passed to SetDSPOpt

// Prototypes for functions that will go in dspDecmp
static inline void UnpckSrc(uint8, uint16 &, int16 *, int32 &, int32 &);

// Volume ----------------------------------
static int32  volAdj;              // Amount to adjust main volumes [-15.16]
static int32  volMainL;            // Main volume
static int32  volMainR;
static int32  volEchoL;            // Echo volume
static int32  volEchoR;

// Echo ------------------------------------
static uint32 echoStart;           // Echo Start Address
static uint32 echoDel;             // Size of delay (in bytes)
static uint32 echoCur;             // Current sample in echo area
static int32  echoFB;              // Echo feedback

// Noise -----------------------------------
static uint32 nRate;               // Noise sample rate reciprocal [.32]
static uint32 nDec;                // Noise accumulator [.32] (>= 1 generate a new sample)
static int32  nSmp;                // Current Noise sample

// 64kHz counter ---------------------------
static uint32 t64Rate;
static uint32 t64Dec;

// Echo filtering --------------------------
static uint32 firCur;              // Index of the first sample to feed into the filter
static uint32 firRate;             // Rate to feed samples into filter
static uint8  firEnabl;            // 0 if filtering is disabled (each bit C0-C7 != 0)
static uint8  disEcho;             // 0 if echo is enabled

// Storage buffers -------------------------
extern int32 Echo [ECHOBUF];
extern int32 Loop [FIRBUF];
extern int32 FilterTaps [8];

// Other
uint32 t64CntS;                    // 64kHz counter (increased every 64kHz pulse) by sample count
static uint32 konW;                // Last time the KON register was written to
static uint32 kofW;                // Last time the KOF register was written to
static bool8 konWw;
static SSoundData *sd;
static Channel *ch[NUM_CHANNELS];

//Used for converting envelope mode flags to Snes9x's enum
static int SAtoS9xEMode[256];

/***********************************************************************
  Code
***********************************************************************/

//The IPL region is excluded.
static inline uint8 DSP_GET_BYTE(uint16 addr)
{
	if(addr < 0xffc0)
		return pAPURAM[addr];
	return pAPUExtraRAM[addr - 0xffc0];
}

static inline uint16 DSP_GET_WORD(uint16 addr)
{
	if(addr < 0xffbf)
		return READ_WORD(&pAPURAM[addr]);
	if(addr == 0xffbf)
		return pAPURAM[addr] | (pAPUExtraRAM[0] << 8);
	if(addr < 0xffff)
		return READ_WORD(&pAPUExtraRAM[addr - 0xffc0]);
	return pAPUExtraRAM[0x3f] | (pAPURAM[0] << 8);
}

static inline uint16 DSP_GET_SRC(int i)
{
	return DSP_GET_WORD((APU.DSP[APU_DIR] << 8) +
						(APU.DSP[chs[i].o + APU_SRCN] << 2));
}

static inline uint16 DSP_GET_SRC_LOOP(int i)
{
	return DSP_GET_WORD((APU.DSP[APU_DIR] << 8) +
						(APU.DSP[chs[i].o + APU_SRCN] << 2) + 2);
}

static inline uint8 *DSP_GET_SRC_P(uint16 addr)
{
	if(addr <= (0xffc0 - 9))
		return &pAPURAM[addr];
	if(addr < 0xffc0) {
		int len = 0xffc0 - addr;
		memcpy(src_buffer, &pAPURAM[addr], (size_t)len);
		memcpy(src_buffer + len, pAPUExtraRAM, (size_t)(9 - len));
		return src_buffer;
	}
	if(addr <= (0x10000 - 9))
		return &pAPUExtraRAM[addr - 0xffc0];

	int len = 0x10000 - addr;
	memcpy(src_buffer, &pAPUExtraRAM[addr - 0xffc0], (size_t)len);
	memcpy(src_buffer + len, pAPURAM, (size_t)(9 - len));
	return src_buffer;
}

static inline uint16 DSP_GET_PITCH(int i)
{
	return ((APU.DSP[chs[i].o + APU_P_HIGH] << 8) |
			APU.DSP[chs[i].o + APU_P_LOW]) & FREQUENCY_MASK;
}

// Start Waveform Decompression
//  Called when a voice is keyed on to set up the internal data for waveform
//  mixing, and decompress the first block.
static inline void StartSrc(int i)
{
	// Initialize interpolation
	memset(&mix[i].sBuf[0x10], 0, sizeof(int16) * 16);

	mix[i].bCur = mix[i].bStart;
	mix[i].bMixStart = mix[i].bCur;
	mix[i].bHdr = DSP_GET_BYTE(mix[i].bCur);
	mix[i].sIdx = 0;
	mix[i].sP1 = 0;
	mix[i].sP2 = 0;

	// Start voice
	APU.DSP[APU_ENDX] &= ~chs[i].m;

	UnpckSrc(mix[i].bHdr,
			 mix[i].bCur,
			 mix[i].sBuf,
			 mix[i].sP1,
			 mix[i].sP2);
}

static inline void UpdateSrc(int i)
{
	if(mix[i].bHdr & 1) {// Is this the end block?
		APU.DSP[APU_ENDX] |= chs[i].m;
		if(mix[i].bHdr & 2) {// Loop?
			mix[i].bCur = DSP_GET_SRC_LOOP(i);
		}
		else {
			mix[i].mFlg |= MFLG_END;
			memset(&mix[i].sBuf[mix[i].sIdx & 0x10], 0, sizeof(int16) * 16);
			mix[i].eVal = D_MIN;
			return;
		}
	}

	// Save block header
	mix[i].bHdr = DSP_GET_BYTE(mix[i].bCur);

	UnpckSrc(mix[i].bHdr,
			 mix[i].bCur,
			 &mix[i].sBuf[mix[i].sIdx & 0x10],
			 mix[i].sP1,
			 mix[i].sP2);
}

static inline void SetNoiseHertz()
{
	int i = APU.DSP[APU_FLG] & 0x1f;
	if(i > 0) {
		nRate =  (uint32)(max48 / (uint64)(freqTab[i] << 16));
	}
	else {
		nRate = 0;
		nSmp = 0;
	}
}

//#define APU_VOL_LEFT 0x00
//#define APU_VOL_RIGHT 0x01
//#define APU_P_LOW 0x02
//#define APU_P_HIGH 0x03
//#define APU_SRCN 0x04
//#define APU_ADSR1 0x05
//#define APU_ADSR2 0x06
//#define APU_GAIN 0x07
static inline void InitReg(int reg, uint8 val)
{
	APU.DSP[reg] = val;
	int i = ((reg) & 0x70) >> 4;
	/* if(~reg & 0x08) the register is in dsp.voice */
	if(!((~(reg)) & MFLG_OFF & mix[i].mFlg))
		(*dspRegs[reg])(i, val);
}

void InitAPUDSP()
{
	int i, c;

	//max48
	max48 = 0x80000000;
	max48 <<= 17;
	max48 -= 1;

	pAPURAM = IAPU.RAM;// Save pointer to SPC RAM
	pAPUExtraRAM = APU.ExtraRAM;

	sd = &SoundData;
	for(i = 0; i < NUM_CHANNELS; i++) {
		ch[i] = &sd->channels[i];
	}

	// Reset values so SetDSPOpt will create new ones
	dspMix = -1;
	dspChn = -1;
	dspSize = -1;
    dspRate = -1;
	realRate = SNES_SAMPLE_RATE;
	dspOpts = 0;
    disEcho = 0;

     // Erase all mixer settings
    for(i = 0; i < 8; i++)
		memset(&mix[i], 0, sizeof(VoiceMix));

    // Set pointers to reasonable default values. Otherwise savestates cause
    // SEGVs!
    for(i = 0; i < 8; i++) {
		mix[i].sIdx = 0;
		mix[i].bStart = DSP_GET_SRC(i);
		mix[i].bMixStart = mix[i].bStart;
		mix[i].bCur = DSP_GET_SRC(i);
    }

	// Build a look-up table for all possible expanded values in a BRR block.
	// range 0-12
    for(i = 0; i < 13; i++) {
		for(c = 0; c < 16; c++)
			brrTab[(i << 4) + c] = (int32)(((((c ^ 8) - 8) << i) >> 1) << 1);
	}
	// range 13-15
    for(i = 13; i < 16; i++) {
		for(c = 0; c < 8; c++)
			brrTab[(i << 4) + c] = 0;
		for(c = 8; c < 16; c++)
			brrTab[(i << 4) + c] = -4096;
	}

	//Build a look-up table to calculate a cubic spline with only four integer multiplies.
	//The table is built from the following equation, simplified for s:
	//
	// y = ax^3 + bx^2 + cx + d
	//
	//     3 (s[0] - s[1]) - s[-1] + s[2]
	// a = ------------------------------
	//                   2
	//
	//                      5 s[0] + s[2]
	// b = 2 s[1] + s[-1] - -------------
	//                            2
	//
	//     s[1] - s[-1]
	// c = ------------
	//          2
	//
	// d = s[0]
	//
	//y is the return sample
	//x is the delta from current sample
	//s is a four sample array with [0] being the current sample
	f80 x1, x2, x3;
	for(i = 0; i < 256; i++) {
		//x1=(n/256)  x2=(n/256)^2  x3=(n/256)^3
		x1 = (f80)i / (f80)256.0;
		x2 = x1 * x1;
		x3 = x2 * x1;

		//s[-1] *= -.5(x^3) + (x^2) - .5x ------
		cubicTab[(i << 2) + 0] = (int16)(((f80)-0.5 * x1 + x2 + (f80)-0.5 * x3) * 32767.0);

		//s[0] *= 1.5(x^3) - 2.5(x^2) + 1 ------
		cubicTab[(i << 2) + 1] = (int16)(((f80)-2.5 * x2 + (f80)1.5 * x3 + 1.0) * 32767.0);

		//s[1] *= -1.5(x^3) + 2(x^2) + .5x -----
		cubicTab[(i << 2) + 2] = (int16)(((f80)0.5 * x1 + x2 + x2 + (f80)-1.5 * x3) * 32767.0);

		//s[2] *= .5(x^3) - .5(x^2) ------------
		cubicTab[(i << 2) + 3] = (int16)(((f80)-0.5 * x2 + (f80)0.5 * x3) * 32767.0);
	}

#ifdef SA_SNESINT
	//Interleave Gaussian table ---------------
    for(i = 0; i < 256; i++) {
		gaussTab[(i << 2) + 3] = gaussPre[i + 0];
		gaussTab[(i << 2) + 2] = gaussPre[i + 256];
		gaussTab[(i << 2) + 1] = gaussPre[i + 512];
		gaussTab[(i << 2) + 0] = gaussPre[i + 768];
	}
#else
	//Build a look-up table based on the Gaussian curve.
	//The table is built from the following equation:
	//
	//                  2
	//            (x/pi)
	//          - -------
	// y = A * e     2
	//
	//       ____
	//     \| 2pi * o
	// A = ----------
	//         n
	//
	//o is the mean value (2^15)
	//n is the number of points to interpolate between (4)
	f80 _512pi = (f80)512.0 / M_PIl;
	f32 fpA = (f32)((sqrt(M_PIl * 2) * 32768) / 4);

    for(i = 0; i < 256; i++) {
		f80 x;
		x = (f80)((-512 + i) + 0)   / _512pi;
		gaussTab[(i << 2) + 3] = (int16)(exp(-(x * x * 0.5)) * fpA);

		x = (f80)((-512 + i) + 256) / _512pi;
		gaussTab[(i << 2) + 2] = (int16)(exp(-(x * x * 0.5)) * fpA);

		x = (f80)((-512 + i) + 512) / _512pi;
		gaussTab[(i << 2) + 1] = (int16)(exp(-(x * x * 0.5)) * fpA);

		x = (f80)((-512 + i) + 768) / _512pi;
		gaussTab[(i << 2) + 0] = (int16)(exp(-(x * x * 0.5)) * fpA);
	}
#endif

	// Create SAtoS9xEMode table
    for(i = 0; i < 256; i++)
		SAtoS9xEMode[i] = SOUND_SILENT;
	SAtoS9xEMode[E_IDLE] = SOUND_SILENT;
	SAtoS9xEMode[E_ATT] = SOUND_ATTACK;
	SAtoS9xEMode[E_DECAY] = SOUND_DECAY;
	SAtoS9xEMode[E_SUST] = SOUND_SUSTAIN;
	SAtoS9xEMode[E_REL] = SOUND_RELEASE;
	SAtoS9xEMode[E_DIRECT] = SOUND_GAIN;
	SAtoS9xEMode[E_INC] = SOUND_INCREASE_LINEAR;
	SAtoS9xEMode[E_BENT] = SOUND_INCREASE_BENT_LINE;
	SAtoS9xEMode[E_DEC] = SOUND_DECREASE_LINEAR;
	SAtoS9xEMode[E_EXP] = SOUND_DECREASE_EXPONENTIAL;

	SAtoS9xEMode[E_ATT | E_IDLE] = SOUND_GAIN;
	SAtoS9xEMode[E_DECAY | E_IDLE] = SOUND_GAIN;
	SAtoS9xEMode[E_SUST | E_IDLE] = SOUND_GAIN;

	SAtoS9xEMode[E_DEC | E_IDLE] = SOUND_GAIN;
	SAtoS9xEMode[E_EXP | E_IDLE] = SOUND_GAIN;
	SAtoS9xEMode[E_INC | E_IDLE] = SOUND_GAIN;
	SAtoS9xEMode[E_BENT | E_IDLE] = SOUND_GAIN;

	SAtoS9xEMode[E_REL | E_IDLE] = SOUND_GAIN;
	
    SetAPUDSPOpt(MIX_NONE,
				 2,
				 16,
				 SNES_SAMPLE_RATE,
				 INT_GAUSS,
				 0);
}

void SetAPUDSPOpt(int mixType,
				  int numChn,
				  int bits,
				  int32 rate,
				  int inter,
				  int opts)
{
	int i;
	bool8 fixVol = FALSE;
	bool8 eraseBuf = FALSE;

	//=========================================
	// Verify parameters

	// mixType
	if(mixType == -1)
		mixType = dspMix;

	// numChn
	if(numChn == -1)
		numChn = dspChn;
	else
		numChn = numChn != 1 ? 2 : 1;

	// bits
	if(bits == -1)
		bits = dspSize;
	else {
		switch(bits) {
		case   8: break;
		case  16: break;
		case  24: break;
		case  32: break;
		case -32: break;
		default: bits = 16; break;
		}
		bits >>= 3;
	}

	// rate
	if(rate == -1)
		rate = (dspOpts & DSP_ANALOG) ? realRate : dspRate;
	else
		rate = (rate >= 8000 && rate <= 192000) ? rate : SNES_SAMPLE_RATE;

	// opts
	if(opts == -1)
		opts = dspOpts;

	// Verify mix type is capable of producing requested output
	if(mixType != 0 &&// MIX_NONE is unaffected by output settings
	   !(bits >= 2 && numChn >= 2 && mixType > MIX_INT)) {
		
		if(bits == 1) {
			mixType = MIX_INT;
		}
		else if(numChn == 1) {//numChn == mono
			mixType = MIX_INT;
			bits = 2;
		}
		else if(bits != 2) {
			mixType = MIX_FLOAT;
		}
	}

	// Reverse stereo
	if((dspOpts & DSP_REVERSE) ^ (opts & DSP_REVERSE))// Did reverse flag change?
		fixVol = TRUE;

	// Disable echo ----------------------------
	disEcho = (disEcho & ~DSP_NOECHO) | (opts & DSP_NOECHO);

	// Save option flags
	dspOpts = (uint8)opts;

	SetAPUDSPAmp(FIXED_POINT * Options.VolumeLevel);

	//=========================================
	// Calculate sample rate change
	realRate = rate;
	dspOpts = rate < SNES_SAMPLE_RATE ? (dspOpts & ~DSP_ANALOG) : dspOpts;

	// Has sample rate changed?
	if(rate != dspRate) {
		dspRate = rate;

		pitchAdj = (uint32)(((uint64)SNES_SAMPLE_RATE << (FIXED_POINT_SHIFT + 4)) / dspRate);

		uint64 r, rr;

		r = (uint64)0x1000 * pitchAdj;
		rateTab[0] = 0;
		for(i = 1; i < 32; i++) {
			rr = r / freqTab[i];
			rateTab[i] = (uint32)((rr >> FIXED_POINT_SHIFT) + ((rr & FIXED_POINT_REMAINDER) ? 1 : 0));
		}

		// Noise
		SetNoiseHertz();

		//Sample counter
		t64Rate = rateTab[31] << 1;
		t64Dec = 0;

		firCur = 0;
		firRate = (uint32)(((int64)dspRate << FIXED_POINT_SHIFT) / SNES_SAMPLE_RATE);

		// Adjust voice rates
		for(i = 0; i < 7; i++) {
			mix[i].mOrgP = DSP_GET_PITCH(i);
			r = (uint64)mix[i].mOrgP * pitchAdj;
			mix[i].mOrgRate = (uint32)((r >> FIXED_POINT_SHIFT) + ((r & FIXED_POINT_REMAINDER) ? 1 : 0));
			mix[i].mRate = mix[i].mOrgRate;
			mix[i].mDec = 0;
			mix[i].eRate = rateTab[mix[i].eRIdx];
			mix[i].eDec = 0;
		}

		// Adjust echo delay
		InitReg(APU_ESA, APU.DSP[APU_ESA]);
		InitReg(APU_EDL, APU.DSP[APU_EDL]);

		eraseBuf = TRUE;
	}

	//=========================================
	// Set sample size
	if(bits != dspSize)
		dspSize = bits;

	//=========================================
	// Set number of channels
	if(numChn != dspChn) {
		dspChn = numChn;
		fixVol = TRUE;
	}

	//=========================================
	// Update areas affected by the mix type
	if(mixType != dspMix) {
		dspMix = mixType;
		fixVol = TRUE;
		eraseBuf = TRUE;
	}

	//=========================================
	// Erase sample buffers
	if(eraseBuf) {
		memset(Echo, 0, sizeof(int32) * ECHOBUF);
		memset(Loop, 0, sizeof(int32) * FIRBUF);
	}

	//=========================================
	// Fixup volume handlers
	if(fixVol) {
		// Setup registers
		void (**regs)(int, uint8);
		regs = dspChn == 1 ? dspRegsM : (dspMix == MIX_FLOAT ? dspRegsF : dspRegsI);

		if(!(dspOpts & DSP_REVERSE)) {
			for(i = 0; i < 8; i++) {
				dspRegs[chs[i].o + APU_VOL_LEFT] = regs[0];  // VOLL handler
				dspRegs[chs[i].o + APU_VOL_RIGHT] = regs[1];  // VOLR handler
			}
			dspRegs[APU_MVOL_LEFT] = regs[2];   // MVOLL handler
			dspRegs[APU_MVOL_RIGHT] = regs[3];   // MVOLR handler
			dspRegs[APU_EVOL_LEFT] = regs[4];   // EVOLL handler
			dspRegs[APU_EVOL_RIGHT] = regs[5];   // EVOLR handler
		} else {
			// Reverse, e.g. put VOLL in VOLR slot
			for(i = 0; i < 8; i++) {
				dspRegs[chs[i].o + APU_VOL_LEFT] = regs[1];  // VOLR handler
				dspRegs[chs[i].o + APU_VOL_RIGHT] = regs[0];  // VOLL handler
			}
			dspRegs[APU_MVOL_LEFT] = regs[3];   // MVOLR handler
			dspRegs[APU_MVOL_RIGHT] = regs[2];   // MVOLL handler
			dspRegs[APU_EVOL_LEFT] = regs[5];   // EVOLR handler
			dspRegs[APU_EVOL_RIGHT] = regs[4];   // EVOLL handler
		}
		for(i = 0; i < 8; i++)
			dspRegs[chs[i].o + APU_C0] = regs[7];         // FC handler
		dspRegs[APU_EFB] = regs[6];  // EFB handler

		// Fix volumes and filter
		for(i = 0; i < 8; i++) {
			InitReg(chs[i].o + APU_VOL_LEFT, APU.DSP[chs[i].o + APU_VOL_LEFT]);
			InitReg(chs[i].o + APU_VOL_RIGHT, APU.DSP[chs[i].o + APU_VOL_RIGHT]);
			InitReg(chs[i].o + APU_C0, APU.DSP[chs[i].o + APU_C0]);
		}
		InitReg(APU_MVOL_LEFT, APU.DSP[APU_MVOL_LEFT]);
		InitReg(APU_MVOL_RIGHT, APU.DSP[APU_MVOL_RIGHT]);
		InitReg(APU_EVOL_LEFT, APU.DSP[APU_EVOL_LEFT]);
		InitReg(APU_EVOL_RIGHT, APU.DSP[APU_EVOL_RIGHT]);

		// Fix feedback level
		InitReg(APU_EFB, APU.DSP[APU_EFB]);
	}
}

void ResetAPUDSP(void)
{
	int i;

	for(i = 0; i < 8; i++) {
		// Voice is inactive
		mix[i].mFlg &= (MFLG_MUTE | MFLG_OFF);
		mix[i].mFlg |= MFLG_OFF;
	}

	// Disable echo
	disEcho &= DSP_NOECHO;

    // Disable echo if all taps are 0
	disEcho |= 1;

	// Erase DSP Registers ---------------------
	memset(APU.DSP, 0, 0x80);

	// Place DSP in power up mode
	APU.DSP[APU_FLG] = APU_SOFT_RESET | APU_MUTE;

    // Erase internal mixing settings
    // We'll just save-and-restore mFlg instead of selectively zeroing
	for(i = 0; i < 8; i++) {

		// Set pointers to reasonable default values. Otherwise savestates
		// cause SEGVs!
		mix[i].bStart = DSP_GET_SRC(i);
		mix[i].bMixStart = mix[i].bStart;
		mix[i].bCur = DSP_GET_SRC(i);
		mix[i].bHdr	= DSP_GET_BYTE(mix[i].bCur);

		// Envelope
		mix[i].eMode = E_IDLE;
		mix[i].eRIdx = 0x1f;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		mix[i].eDec = 0;
		mix[i].eVal = 0;
		mix[i].eAdj = A_KOF;
		mix[i].eDest = D_MIN;
		
		// Samples
		memset(mix[i].sBuf, 0, sizeof(int16) * 32);
        mix[i].sIdx = 0;
		mix[i].sP1 = 0;
		mix[i].sP2 = 0;

		// Mixing
		mix[i].mChnL = 0;
		mix[i].mChnR = 0;
		RPitch(i, APU.DSP[chs[i].o + APU_P_LOW]);
		mix[i].mDec = 0;
		mix[i].mOut = 0;
	}

    // Erase global volume settings
    volMainL = 0;
    volMainR = 0;
    volEchoL = 0;
    volEchoR = 0;

    // Erase noise settings
    nRate = 0;
    nSmp = 0;
    nDec = 0;

    // Erase echo region
    memset(Echo, 0, sizeof(int32) * ECHOBUF);

	// Reset echo variables
	echoStart = 0;
	echoCur = 0;
	echoDel = 2;  // Delay 1 sample
	echoFB = 0;

	// Echo filter
    memset(Loop, 0, sizeof(int32) * FIRBUF);// Erase filter memory
    memset(FilterTaps, 0, sizeof(int32) * 8);// Reset filter coefficients

	// Snes9x
	//FilterTaps [0] = 127;

	// Reset filter variables
    firCur = 0;
	firEnabl = 0;

    // Disable voices
	voiceKon = 0;

	// Reset times
	konW = (EXT.t64Cnt - 4);
	konWw = FALSE;
	kofW = (EXT.t64Cnt - 4);
}

static inline void ChkStartSrc(int i);

void StoreAPUDSP()
{
	int i;

	FLUSH_SAMPLES();

	APU.KeyedChannels = voiceKon;

	sd->master_volume_left  = (short)(int8)APU.DSP[APU_MVOL_LEFT];
	sd->master_volume_right = (short)(int8)APU.DSP[APU_MVOL_RIGHT];
	sd->echo_volume_left    = (short)(int8)APU.DSP[APU_EVOL_LEFT];
	sd->echo_volume_right   = (short)(int8)APU.DSP[APU_EVOL_RIGHT];
	sd->echo_enable         = APU.DSP[APU_EON];
	sd->echo_feedback       = (int)(int8)APU.DSP[APU_EFB];

	sd->echo_buffer_size    = (512 * (APU.DSP[APU_EDL] & 0x0f) * SNES_SAMPLE_RATE) / SNES_SAMPLE_RATE;
	sd->echo_buffer_size <<= 1;

	sd->echo_ptr            = 0;

	sd->echo_write_enabled  = (APU.DSP[APU_FLG] & APU_ECHO_DISABLED) == 0;
	sd->echo_channel_enable = APU.DSP[APU_EON];
	sd->pitch_mod           = APU.DSP[APU_PMON];

	sd->dummy[0] = 0;
	sd->dummy[1] = 0;
	sd->dummy[2] = 0;

	//no_filter
	S9xSetFilterCoefficient(0, (int)(signed char) APU.DSP [APU_C0]);

	sd->master_volume [0] = sd->master_volume_left;
	sd->master_volume [1] = sd->master_volume_right;
	sd->echo_volume [0] = sd->echo_volume_left;
	sd->echo_volume [1] = sd->echo_volume_right;

	uint32 freq, f;

	freq = freqTab[APU.DSP[APU_FLG] & 0x1f];
	if(freq > 0) {
		f = (SNES_SAMPLE_RATE << FIXED_POINT_SHIFT) / freq;
		sd->noise_hertz = (int)((f >> FIXED_POINT_SHIFT) + ((f & FIXED_POINT_REMAINDER) ? 1 : 0));
	}
	else
		sd->noise_hertz = 0;

	for (i = 0; i < 8; i++) {
		int32 eVal;

		ChkStartSrc(i);

		if (mix[i].mFlg & MFLG_OFF) {// Channel is inactive
			ch[i]->state = SOUND_SILENT;
			ch[i]->mode  = MODE_NONE;
		}
		else {
			ch[i]->state = SAtoS9xEMode[mix[i].eMode];
			if (mix[i].mFlg & MFLG_KOFF)// Key off state
				ch[i]->mode = MODE_RELEASE;
			else {
				if (APU.DSP[chs[i].o + APU_ADSR1] & 0x80)  // ADSR mode
					ch[i]->mode = MODE_ADSR;
				else {// GAIN mode
					if (!(APU.DSP[chs[i].o + APU_GAIN] & 0x80))  // Direct mode
						ch[i]->mode = MODE_GAIN;
					else {                                       // Indirect mode
						switch(APU.DSP[chs[i].o + APU_GAIN] & 0x60){
						case 0x00:
							ch[i]->mode = MODE_DECREASE_LINEAR;
							break;

						case 0x20:
							ch[i]->mode = MODE_DECREASE_EXPONENTIAL;
							break;

						case 0x40:
							ch[i]->mode = MODE_INCREASE_LINEAR;
							break;

						case 0x60:
							ch[i]->mode = MODE_INCREASE_BENT_LINE;
							break;
						}

					}
				}
            }
		}
			
		ch[i]->type            = (APU.DSP[APU_NON] & chs[i].m) ? SOUND_NOISE : SOUND_SAMPLE;
		ch[i]->volume_left     = (short)(int8)APU.DSP[chs[i].o + APU_VOL_LEFT];
		ch[i]->volume_right    = (short)(int8)APU.DSP[chs[i].o + APU_VOL_RIGHT];
		ch[i]->hertz           = ((uint32)DSP_GET_PITCH(i) * SNES_SAMPLE_RATE) >> 12;
		ch[i]->frequency       = (uint32)(((int64)(ch[i]->type == SOUND_NOISE ?
												   sd->noise_hertz : ch[i]->hertz) *
										   FIXED_POINT) / SNES_SAMPLE_RATE);

		ch[i]->count           = 0;

		eVal = mix[i].eVal;
		if(eVal > D_ATTACK)
			eVal = D_ATTACK;
		else if(eVal < 0)
			eVal = 0;

		ch[i]->envx            = eVal >> E_SHIFT;
		ch[i]->envxx           = (ch[i]->envx << 24);// #define ENVX_SHIFT 24
		ch[i]->left_vol_level  = (ch[i]->envx * ch[i]->volume_left) / 128;
		ch[i]->right_vol_level = (ch[i]->envx * ch[i]->volume_right) / 128;

		switch(mix[i].eMode) {
		case E_ATT:
			//Attack mode
			//ch[i]->envx_target = MAX_ENVELOPE_HEIGHT;
			ch[i]->envx_target = mix[i].eDest >> E_SHIFT;
			break;

		case E_DECAY:
			//Decay mode
			//ch[i]->envx_target = (MAX_ENVELOPE_HEIGHT * ((APU.DSP[chs[i].o + APU_ADSR2] >> 5) + 1)) >> 3;
			ch[i]->envx_target = mix[i].eDest >> E_SHIFT;
			break;

		case E_SUST:
			//Sustain mode
			//ch[i]->envx_target = MAX_ENVELOPE_HEIGHT / 10;
			ch[i]->envx_target = mix[i].eDest >> E_SHIFT;
			break;

		default:
			ch[i]->envx_target = 0;
			break;
		}

		ch[i]->env_error       = mix[i].eDec;
		if(so.playback_rate > 0)
			ch[i]->erate       = mix[i].eRate * SNES_SAMPLE_RATE / so.playback_rate;
		else
			ch[i]->erate = 0;

		ch[i]->direction       = mix[i].eRate == 0 ? 0 : ((mix[i].eMode & E_DIR) ? 1 : -1);

		if((APU.DSP[chs[i].o + APU_ADSR1] & 0xF) < 0x0F) {
			f = freqTab[((APU.DSP[chs[i].o + APU_ADSR1] & 0xF) << 1) + 1];
			ch[i]->attack_rate = (unsigned long)(f << 1);
		}
		else
			ch[i]->attack_rate = (unsigned long)1;

		// 1->1/10, eVal*255/256, T_EXP times.
		f = freqTab[((APU.DSP[chs[i].o + APU_ADSR1] & 0x70) >> 3) + 0x10];
		ch[i]->decay_rate = (unsigned long)(f * T_EXP / (SNES_SAMPLE_RATE / 1000));

		if((APU.DSP[chs[i].o + APU_ADSR2] & 0x1F) > 0) {
			f = freqTab[APU.DSP[chs[i].o + APU_ADSR2] & 0x1F];
			ch[i]->sustain_rate = (unsigned long)(f * T_EXP / (SNES_SAMPLE_RATE / 1000));
		}
		else
			ch[i]->sustain_rate = ~0;

		ch[i]->release_rate    = 8;
		ch[i]->sustain_level   = (APU.DSP[chs[i].o + APU_ADSR2] >> 5) + 1;
		ch[i]->interpolate     = 0;

		memcpy(ch[i]->decoded, &mix[i].sBuf[mix[i].sIdx & 0x10], sizeof(int16) * 16);

		ch[i]->sample          = mix[i].sBuf[mix[i].sIdx];

		if(mix[i].sIdx < 31)
			ch[i]->next_sample = mix[i].sBuf[mix[i].sIdx + 1];
		else
			ch[i]->next_sample = 0;

		ch[i]->sample_pointer  = (uint32)(mix[i].sIdx & 0x0f);
		ch[i]->Filter          = mix[i].bHdr;
		ch[i]->last_block      = (ch[i]->Filter & 1) != 0;
		ch[i]->loop            = (ch[i]->Filter & 2) != 0;

		ch[i]->previous[0]   = mix[i].sP1;
		ch[i]->previous[1]   = mix[i].sP2;
		ch[i]->previous16[0]   = (signed short)mix[i].sP1;
		ch[i]->previous16[1]   = (signed short)mix[i].sP2;
		ch[i]->block = (signed short *)&ch[i]->decoded[ch[i]->sample_pointer];
		ch[i]->sample_number   = APU.DSP [chs[i].o + APU_SRCN];
		ch[i]->needs_decode    = FALSE;
		ch[i]->block_pointer   = (uint32)mix[i].bCur;

		ch[i]->echo_buf_ptr = NULL;

		ch[i]->chkMode = 0;

		ch[i]->dummy[0] = 0;
		ch[i]->dummy[1] = 0;
		ch[i]->dummy[2] = 0;
		ch[i]->dummy[3] = 0;
		ch[i]->dummy[4] = 0;
		ch[i]->dummy[5] = 0;
		ch[i]->dummy[6] = 0;
		ch[i]->dummy[7] = 0;

		// Store OUTX ENVX
		APU.DSP[chs[i].o + APU_OUTX] = (uint8)((mix[i].mFlg & (MFLG_OFF | MFLG_END)) ? 0 : (mix[i].mOut >> 8));
		APU.DSP[chs[i].o + APU_ENVX] = (uint8)((mix[i].mFlg & (MFLG_OFF | MFLG_END)) ? 0 : ((eVal >> E_SHIFT) & 0x7f));
	}

	// Store 0xf3
	pAPURAM[0xf3] = APU.DSP[pAPURAM[0xf2] & 0x7f];
}

static const uint8 S9xtoSAEMode[10] = {
	E_IDLE,  // SOUND_SILENT
	E_ATT,   // SOUND_ATTACK
	E_DECAY, // SOUND_DECAY
	E_SUST,  // SOUND_SUSTAIN
	E_REL,   // SOUND_RELEASE
	E_DIRECT,// SOUND_GAIN
	E_INC,   // SOUND_INCREASE_LINEAR
	E_BENT,  // SOUND_INCREASE_BENT_LINE
	E_DEC,   // SOUND_DECREASE_LINEAR
	E_EXP    // SOUND_DECREASE_EXPONENTIAL
};

// Restore DSP's State of Operation
void RestoreAPUDSP()
{
	int i;

	voiceKon = APU.KeyedChannels;

	for (i = 0; i < 8; i++) {
		mix[i].bStart = DSP_GET_SRC(i);
		mix[i].bMixStart = mix[i].bStart;
		mix[i].bCur	= (uint16)(ch[i]->block_pointer & 0xffff);
		mix[i].bHdr	= (uint8)ch[i]->Filter;
		memset(&mix[i].sBuf[0x10], 0, sizeof(int16) * 16);
		memcpy(mix[i].sBuf, ch[i]->decoded, sizeof(int16) * 16);
		mix[i].sP1	= (int32)ch[i]->previous16[0];
		mix[i].sP2	= (int32)ch[i]->previous16[1];
		mix[i].sIdx = (uint8)(ch[i]->sample_pointer & 0x0f);

		uint8 mFlg = ((mix[i].mFlg & MFLG_MUTE) |
					  ((ch[i]->last_block && !ch[i]->loop) ? MFLG_END : 0) |
					  (ch[i]->state == SOUND_SILENT ? MFLG_OFF : 0) |
					  (ch[i]->mode == MODE_RELEASE ? MFLG_KOFF : 0));

		if(mFlg & MFLG_OFF)
			mFlg &= ~(MFLG_KOFF | MFLG_END);

		mix[i].mFlg	= mFlg;

		InitReg(chs[i].o + APU_VOL_LEFT, APU.DSP[chs[i].o + APU_VOL_LEFT]);
		InitReg(chs[i].o + APU_VOL_RIGHT, APU.DSP[chs[i].o + APU_VOL_RIGHT]);
		InitReg(chs[i].o + APU_P_LOW, APU.DSP[chs[i].o + APU_P_LOW]);
		
		mix[i].eMode = S9xtoSAEMode[ch[i]->state];

		// Default to release mode settings
		mix[i].eRIdx = 31;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		mix[i].eDec = 0;
		mix[i].eAdj = A_KOF;
		mix[i].eDest = D_MIN;
		mix[i].eVal = ch[i]->envx << E_SHIFT;
		InitReg(chs[i].o + APU_ADSR1, APU.DSP[chs[i].o + APU_ADSR1]);

		mix[i].mOut = (int32)(int8)APU.DSP[chs[i].o + APU_OUTX] << 8;
		mix[i].mDec = 0;
	}
    // Setup global paramaters
    InitReg(APU_MVOL_LEFT, APU.DSP[APU_MVOL_LEFT]);
    InitReg(APU_MVOL_RIGHT, APU.DSP[APU_MVOL_RIGHT]);
    InitReg(APU_EVOL_LEFT, APU.DSP[APU_EVOL_LEFT]);
    InitReg(APU_EVOL_RIGHT, APU.DSP[APU_EVOL_RIGHT]);

	// Don't "APU_SOFT_RESET"
	uint8 flg = APU.DSP[APU_FLG];
	RFlg((APU_FLG & 0x70) >> 4, flg & ~APU_SOFT_RESET);
	APU.DSP[APU_FLG] = flg;

    InitReg(APU_ESA, APU.DSP[APU_ESA]);
    InitReg(APU_EFB, APU.DSP[APU_EFB]);
    InitReg(APU_EDL, APU.DSP[APU_EDL]);

    for(i = 0; i < 8; i++)
		InitReg(chs[i].o + APU_C0, APU.DSP[chs[i].o + APU_C0]);
}

void RestoreSPCFile()
{
	int i;

	for (i = 0; i < 8; i++) {
		mix[i].eVal = APU.DSP[chs[i].o + APU_ENVX] << E_SHIFT;
		//mix[i].mOut = (int32)(int8)APU.DSP[chs[i].o + APU_OUTX] << 8;
	}

    // Setup global paramaters
    InitReg(APU_MVOL_LEFT, APU.DSP[APU_MVOL_LEFT]);
    InitReg(APU_MVOL_RIGHT, APU.DSP[APU_MVOL_RIGHT]);
    InitReg(APU_EVOL_LEFT, APU.DSP[APU_EVOL_LEFT]);
    InitReg(APU_EVOL_RIGHT, APU.DSP[APU_EVOL_RIGHT]);

	// Don't "APU_SOFT_RESET"
	uint8 flg = APU.DSP[APU_FLG];
	RFlg((APU_FLG & 0x70) >> 4, flg & ~APU_SOFT_RESET);
	APU.DSP[APU_FLG] = flg;

    InitReg(APU_ESA, APU.DSP[APU_ESA]);
    InitReg(APU_EFB, APU.DSP[APU_EFB]);
    InitReg(APU_EDL, APU.DSP[APU_EDL]);

    for(i = 0; i < 8; i++)
		InitReg(chs[i].o + APU_C0, APU.DSP[chs[i].o + APU_C0]);

	// Start Voices
	InitReg(APU_KON, APU.DSP[APU_KON]);
}

// DSP Pre-Amplification Level
void SetAPUDSPAmp(int32 amp)
{
	volAdj = amp;

	// Update global volumes
	InitReg(APU_MVOL_LEFT, APU.DSP[APU_MVOL_LEFT]);
	InitReg(APU_MVOL_RIGHT, APU.DSP[APU_MVOL_RIGHT]);
	InitReg(APU_EVOL_LEFT, APU.DSP[APU_EVOL_LEFT]);
	InitReg(APU_EVOL_RIGHT, APU.DSP[APU_EVOL_RIGHT]);
}

static inline void ChgSus(int i)
{
	mix[i].eRIdx = APU.DSP[chs[i].o + APU_ADSR2] & 0x1f;
	if(rateTab[mix[i].eRIdx] == 0 || mix[i].eVal <= D_MIN) {
		mix[i].eMode = E_IDLE | E_SUST;
		mix[i].eDec = 0;
	}
	else
		mix[i].eMode = E_SUST;
	mix[i].eRate = rateTab[mix[i].eRIdx];
	mix[i].eAdj = A_EXP;
	mix[i].eDest = D_MIN;
}

static inline void ChgDec(int i)
{
	uint8 sl = APU.DSP[chs[i].o + APU_ADSR2] >> 5;
	int32 eDest = (int32)(sl + 1) * D_EXP;
	if(sl == 7 || mix[i].eVal <= eDest) {
		ChgSus(i);
		return;
	}
	mix[i].eRIdx = ((APU.DSP[chs[i].o + APU_ADSR1] & 0x70) >> 3) + 0x10;
	mix[i].eRate = rateTab[mix[i].eRIdx];
	mix[i].eAdj = A_EXP;
	mix[i].eDest = eDest;
	mix[i].eMode = E_DECAY;
}

static inline void ChgAtt(int i)
{
	uint8 ar = APU.DSP[chs[i].o + APU_ADSR1] & 0x0f;

	if(ar == 0x0f) {//0->1 0ms ??
		if(mix[i].eVal < D_ATTACK)
			mix[i].eVal = D_ATTACK;
		ChgDec(i);
	}
	else {
		if(mix[i].eVal >= D_ATTACK) {
			ChgDec(i);
			return;
		}
		mix[i].eRIdx = (ar << 1) + 1;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		mix[i].eAdj = A_LIN;
		mix[i].eDest = D_ATTACK;
		mix[i].eMode = E_ATT;
	}
}

static inline void ChgADSR(int i)
{
	switch(mix[i].eMode) {
	case E_ATT:
		ChgAtt(i);
		break;

	case E_DECAY:
		ChgDec(i);
		break;

	case E_SUST:
	case (E_IDLE | E_SUST):
		if(mix[i].eVal > D_MIN)
			ChgSus(i);
		break;
	}
}

static inline void ChgGain(int i)
{
	// Is gain direct?
	if(!(APU.DSP[chs[i].o + APU_GAIN] & 0x80)) {
		mix[i].eMode = E_DIRECT;
		mix[i].eRIdx = 0;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		mix[i].eAdj = A_EXP;
		mix[i].eVal = mix[i].eDest = (int32)(APU.DSP[chs[i].o + APU_GAIN] & 0x7f) * A_GAIN;
		return;
	}

	switch(APU.DSP[chs[i].o + APU_GAIN] & 0x60) {
	case 0x00:
		// GainDec
		mix[i].eRIdx = APU.DSP[chs[i].o + APU_GAIN] & 0x1f;
		if(rateTab[mix[i].eRIdx] == 0 || mix[i].eVal <= D_MIN) {
			mix[i].eMode = E_IDLE | E_DEC;
			mix[i].eDec = 0;
		}
		else
			mix[i].eMode = E_DEC;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		mix[i].eAdj = A_LIN;
		mix[i].eDest = D_MIN;
		break;

	case 0x20:
		// GainExp
		mix[i].eRIdx = APU.DSP[chs[i].o + APU_GAIN] & 0x1f;
		if(rateTab[mix[i].eRIdx] == 0 || mix[i].eVal <= D_MIN) {
			mix[i].eMode = E_IDLE | E_EXP;
			mix[i].eDec = 0;
		}
		else
			mix[i].eMode = E_EXP;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		mix[i].eAdj = A_EXP;
		mix[i].eDest = D_MIN;
		break;

	case 0x40:
		// GainInc
		mix[i].eRIdx = APU.DSP[chs[i].o + APU_GAIN] & 0x1f;
		if(rateTab[mix[i].eRIdx] == 0 || mix[i].eVal >= D_ATTACK) {
			mix[i].eMode = E_IDLE | E_INC;
			mix[i].eDec = 0;
		}
		else
			mix[i].eMode = E_INC;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		mix[i].eAdj = A_LIN;
		mix[i].eDest = D_ATTACK;
		break;

	case 0x60:
		// GainBent
		mix[i].eRIdx = APU.DSP[chs[i].o + APU_GAIN] & 0x1f;
		if(rateTab[mix[i].eRIdx] == 0 || mix[i].eVal >= D_ATTACK) {
			mix[i].eMode = E_IDLE | E_BENT;
			mix[i].eDec = 0;
		}
		else
			mix[i].eMode = E_BENT;
		mix[i].eRate = rateTab[mix[i].eRIdx];
		if(mix[i].eVal < D_BENT) {
			mix[i].eAdj = A_LIN;
			mix[i].eDest = D_BENT;
		}
		else {
			mix[i].eAdj = A_BENT;
			mix[i].eDest = D_ATTACK;
		}
		break;
	}
}

// Start Envelope
//  Called when a voice is keyed on to set up the internal data to begin
//  envelope modification based on the values in ADSR/Gain.
static inline void StartEnv(int i)
{
	if((mix[i].mFlg & (MFLG_OFF | MFLG_END)) || (mix[i].eMode & E_IDLE))
		mix[i].eDec = 0;

	if(APU.DSP[chs[i].o + APU_ADSR1] & 0x80)
		ChgAtt(i);
	else
		ChgGain(i);
}

static inline void UpdateEnv(int i)
{
	// Is the adjustment exponential? (mode 1, 9, or D)
	// E_EXP, E_SUST, E_DECAY
	if(mix[i].eMode & E_TYPE) {
		if(mix[i].eVal > mix[i].eDest) {
			mix[i].eVal = (mix[i].eVal * 255) / 256;
			if(mix[i].eVal > mix[i].eDest)
				return;
		}
	}
	// Linear increase (mode 2, 6, or A)
	// E_INC, E_BENT, E_ATT
	else if(mix[i].eMode & E_DIR) {
		if(mix[i].eVal < mix[i].eDest) {
			mix[i].eVal += mix[i].eAdj;
			if(mix[i].eVal < mix[i].eDest)
				return;
		}
	}
	// Linear decrease (mode 0 or 8)
	// E_DEC, E_REL
	else {
		if(mix[i].eVal > mix[i].eDest) {
			if(mix[i].eVal >= mix[i].eAdj)
				mix[i].eVal -= mix[i].eAdj;
			else
				mix[i].eVal = D_MIN;
			if(mix[i].eVal > mix[i].eDest)
				return;
		}
		if(mix[i].eMode == E_REL) {// Is voice in release mode?
			mix[i].mFlg |= MFLG_OFF;
			mix[i].mFlg &= ~MFLG_KOFF;
		}
	}

	// Is envelope in ADSR mode?
	// E_REL, E_SUST, E_ATT, E_DECAY
	if(mix[i].eMode & E_ADSR) {
		// Change adjustment mode
		switch(mix[i].eMode) {
		case E_ATT:
			if(mix[i].eVal >= mix[i].eDest)
				ChgDec(i);
			return;

		case E_DECAY:
			if(mix[i].eVal <= mix[i].eDest)
				ChgSus(i);
			return;

		case E_SUST:
			break;

		case E_REL:
			break;
		}
	}
	// Is gain in bent line mode? No, it's constant
	else if(mix[i].eMode & E_DEST) {
		if(mix[i].eDest == D_BENT) {
			mix[i].eAdj = A_BENT;
			mix[i].eDest = D_ATTACK;
			return;
		}
	}
	mix[i].eMode |= E_IDLE;
	mix[i].eVal = mix[i].eDest;//clip eVal
	mix[i].eDec = 0;
}

static inline void CalcEnv(int i)
{
	if(!(mix[i].eMode & E_IDLE)) {
		uint32 e = (mix[i].eDec += mix[i].eRate) >> FIXED_POINT_SHIFT;
		mix[i].eDec &= FIXED_POINT_REMAINDER;
		for(; e; e--) {
			UpdateEnv(i);
			if(mix[i].eMode & E_IDLE)
				break;
		}
	}
}

static inline void ChkStartSrc(int i)
{
	if((mix[i].mFlg & MFLG_SSRC) || mix[i].bStart != mix[i].bMixStart) {
		mix[i].mFlg &= ~MFLG_SSRC;
		StartSrc(i);
		mix[i].mFlg &= ~MFLG_END;
		mix[i].mDec = 0;
	}
}

static inline void ProcessSrc(int i)
{
	// Waveform Resizing
	// MAX is 16 in 8000Hz Pitch 0x3FFF
	uint8 sc = (uint8)((mix[i].mDec += mix[i].mRate) >> FIXED_POINT_SHIFT);
	mix[i].mDec &= FIXED_POINT_REMAINDER;
	uint8 sIdx = (mix[i].sIdx + (sc & 0x0f)) & 0x1f;

	// It needs next 16 samples?
	if((mix[i].sIdx ^ sIdx) & 0x10) {
		mix[i].sIdx = sIdx;
		UpdateSrc(i);
	}
	else
		mix[i].sIdx = sIdx;

	if(!(mix[i].mFlg & MFLG_END)) {
		sc >>= 4;
		if(sc > 0) {
			mix[i].sIdx ^= 0x10;
			UpdateSrc(i);
		}
	}
}

// DSP Data Port
void APUDSPIn(uint8 address, uint8 data)
{
	// Writes to 80-FFh have no effect (reads are mirrored from lower mem)
	if(address & 0x80)
		return;

	switch(address) {
	case APU_KON:
	case APU_KOFF:
		FLUSH_SAMPLES();
		InitReg(address, data);
		break;

	case APU_FLG:
		FLUSH_SAMPLES();
		RFlg((address & 0x70) >> 4, data);
		break;

	case APU_ENDX:
		FLUSH_SAMPLES();
		REndX((address & 0x70) >> 4, data);
		break;

	default:
		if(data != APU.DSP[address]) {
			//#define APU_VOL_LEFT 0x00
			//#define APU_VOL_RIGHT 0x01
			//#define APU_P_LOW 0x02
			//#define APU_P_HIGH 0x03
			//#define APU_SRCN 0x04
			//#define APU_ADSR1 0x05
			//#define APU_ADSR2 0x06
			//#define APU_GAIN 0x07
			if(!(address & 0x08)) {
				APU.DSP[address] = data;
				int i = ((address) & 0x70) >> 4;
				if(!(mix[i].mFlg & MFLG_OFF)) {
					FLUSH_SAMPLES();
					if(!(mix[i].mFlg & MFLG_OFF))
						(*dspRegs[address])(i, data);
				}
			}
			else
				InitReg(address, data);
		}
		break;
	}
}

/***********************************************************************
  DSP Register Handlers
***********************************************************************/

// Channel volume
static void RVolL(int i, uint8 val)
{
	mix[i].mChnL = (int32)(int8)val;
}

static void RVolLM(int i, uint8 val)
{
	mix[i].mChnL = (int32)ABS((int8)val);
}

static void RVolR(int i, uint8 val)
{
	mix[i].mChnR = (int32)(int8)val;
}

static void RVolRM(int i, uint8 val)
{
	mix[i].mChnR = (int32)ABS((int8)val);
}

// Pitch
static void RPitch(int i, uint8)
{
	mix[i].mOrgP = DSP_GET_PITCH(i);
	uint64 r = (uint64)mix[i].mOrgP * pitchAdj;
	mix[i].mOrgRate = (uint32)((r >> FIXED_POINT_SHIFT) + ((r & FIXED_POINT_REMAINDER) ? 1 : 0));
}

static void RSrcn(int, uint8)
{
	/* Do nothing */
}

// Envelope
static void RADSR1(int i, uint8 val)
{
	// MFLG_OFF was checked in InitReg()
	if(mix[i].mFlg & MFLG_KOFF)
		return;

	if(APU.DSP[chs[i].o + APU_ADSR1] & 0x80)
		ChgADSR(i);
	else
		ChgGain(i);
}

static void RADSR2(int i, uint8 val)
{
	// MFLG_OFF was checked in InitReg()
	if((mix[i].mFlg & MFLG_KOFF) || !(APU.DSP[chs[i].o + APU_ADSR1] & 0x80))
		return;
	ChgADSR(i);
}

static void RGain(int i, uint8)
{
	// MFLG_OFF was checked in InitReg()
	if((mix[i].mFlg & MFLG_KOFF) || (APU.DSP[chs[i].o + APU_ADSR1] & 0x80))
		return;
	ChgGain(i);
}

// Main volumes
static void RMVolL(int, uint8 val)
{
	FLUSH_SAMPLES();

	if(Settings.DisableMasterVolume)
		val = 127;
    volMainL = ((int32)(int8)val * volAdj) >> FIXED_POINT_SHIFT;
}

static void RMVolLM(int, uint8 val)
{
	FLUSH_SAMPLES();

	if(Settings.DisableMasterVolume)
		val = 127;
	// Multiply by 1/2 sqrt 2 to help average
	volMainL = (int32)(((int64)ABS((int8)val) * 46341 * volAdj) >> (FIXED_POINT_SHIFT + 16));
}

static void RMVolR(int, uint8 val)
{
	FLUSH_SAMPLES();

	if(Settings.DisableMasterVolume)
		val = 127;
	volMainR = ((int32)(int8)val * volAdj) >> FIXED_POINT_SHIFT;
}

static void RMVolRM(int, uint8 val)
{
	FLUSH_SAMPLES();

	if(Settings.DisableMasterVolume)
		val = 127;
	// Multiply by 1/2 sqrt 2 to help average
	volMainR = (int32)(((int64)ABS((int8)val) * 46341 * volAdj) >> (FIXED_POINT_SHIFT + 16));
}

static void REVolL(int, uint8 val)
{
	FLUSH_SAMPLES();

	volEchoL = ((int32)(int8)val * volAdj) >> FIXED_POINT_SHIFT;
}

static void REVolLM(int, uint8 val)
{
	FLUSH_SAMPLES();

	// Multiply by 1/2 sqrt 2 to help average
    volEchoL = (int32)(((int64)ABS((int8)val) * 46341 * volAdj) >> (FIXED_POINT_SHIFT + 16));
}

static void REVolR(int, uint8 val)
{
	FLUSH_SAMPLES();

	volEchoR = ((int32)(int8)val * volAdj) >> FIXED_POINT_SHIFT;
}

static void REVolRM(int, uint8 val)
{
	FLUSH_SAMPLES();

	// Multiply by 1/2 sqrt 2 to help average
	volEchoR = (int32)(((int64)ABS((int8)val) * 46341 * volAdj) >> (FIXED_POINT_SHIFT + 16));
}

// Echo settings
static void REFB(int, uint8 val)
{
	FLUSH_SAMPLES();

	echoFB = (int32)(int8)val;
}

static void REFBM(int, uint8 val)
{
	FLUSH_SAMPLES();

	echoFB = (int32)(int8)val;
}

static void REDl(int, uint8 val)
{
	FLUSH_SAMPLES();

	val &= 0x0f;
	if(val == 0)
		echoDel = 2;
	else
		echoDel = ((uint32)(val << 4) * dspRate / 1000) << 1;
}

static inline void CheckFC(int i)
{
	// If first tap is 127, pretend it's 0
	if(!(i == 0 && APU.DSP[APU_C0] == 0x7f) && APU.DSP[chs[i].o + APU_C0])
		firEnabl |= (1 << i);
	else
		firEnabl &= ~(1 << i);
}

static void RFCI(int i, uint8 val)
{
	FLUSH_SAMPLES();

	FilterTaps[i] = (int32)(int8)val;
	CheckFC(i);
}

// Key On/Off
static void RKOn(int idx, uint8 val)
{
	if(idx != -1) {
		konW = EXT.t64Cnt;
		konWw = TRUE;
	}

	if (spc_is_dumping)
	{
		if (val & ~spc_is_dumping_temp)
		{
			APURegisters.PC = IAPU.PC - IAPU.RAM;
			S9xAPUPackStatus();
#ifdef __UOSNES__
  #ifdef __WIN32__
			extern void S9xwSPCDumpMessages(bool8 err);
			S9xwSPCDumpMessages(S9xSPCDump (S9xGetSPCFilename ()));
  #endif
#else
			S9xSPCDump (S9xGetFilenameInc (".spc"));
#endif
			spc_is_dumping = 0;
		}
	}
	spc_is_dumping_temp = val;

	//if((EXT.t64Cnt - kofW) < 4)
	//	val &= ~APU.DSP[APU_KOFF];

	if(val) {
		for(int i = 0; i < 8; i++) {
			if(!(val & chs[i].m))
				continue;

			// Set channel volume
			(*dspRegs[chs[i].o + APU_VOL_LEFT])(i, APU.DSP[chs[i].o + APU_VOL_LEFT]);
			(*dspRegs[chs[i].o + APU_VOL_RIGHT])(i, APU.DSP[chs[i].o + APU_VOL_RIGHT]);

			// Set pitch
			RPitch(i, APU.DSP[chs[i].o + APU_P_LOW]);

			StartEnv(i);

			// Start waveform decompression
			mix[i].bStart = DSP_GET_SRC(i);
			mix[i].mFlg |= MFLG_SSRC;
			mix[i].mFlg &= ~(MFLG_KOFF | MFLG_OFF);// Reset flags

			// Mark voice as being on internally
			voiceKon |= chs[i].m;
		}
	}
}

static void RKOf(int, uint8 val)
{
	kofW = EXT.t64Cnt;

	// Only check voices that are currently playing
	val &= voiceKon;

	if(val) {
		for(int i = 0; i < 8; i++) {
			if(!(val & chs[i].m))
				continue;

			mix[i].eRIdx = 0x1f;// 1->0 8ms
			mix[i].eRate = rateTab[mix[i].eRIdx];
			mix[i].eAdj = A_KOF;
			mix[i].eDest = D_MIN;
			mix[i].eMode = E_REL;
			mix[i].mFlg |= MFLG_KOFF;// Flag voice as keying off

			voiceKon &= ~chs[i].m;
		}
	}

	//Has it been at least 62.5us since KON was written to?
	if(konWw) {
		if((EXT.t64Cnt - konW) < 4)
			RKOn(-1, APU.DSP[APU_KON] & ~APU.DSP[APU_KOFF]);
		else
			konWw = FALSE;
	}
}

static void RDir(int, uint8)
{
	FLUSH_SAMPLES();
}

static void RPMOn(int, uint8)
{
	FLUSH_SAMPLES();
}

static void RNOn(int, uint8)
{
	FLUSH_SAMPLES();
}

static void RFlg(int, uint8 al)
{
	if(al & APU_SOFT_RESET) {// Has a soft reset been initialized?
		al |= APU_MUTE;

		// Reset internal voice settings
		for(int i = 0; i < 8; i++) {
			mix[i].mFlg |= MFLG_OFF;
            mix[i].mFlg &= (MFLG_MUTE | MFLG_OFF);
			mix[i].mDec = 0;
			mix[i].mOut = 0;
			mix[i].eMode = E_IDLE;
			mix[i].eDec = 0;
		}

		APU.DSP[APU_ENDX] = 0;// Clear end block flags
		APU.DSP[APU_KON] = 0;
		APU.DSP[APU_KOFF] = 0;
		voiceKon = 0;
	}

	// Disable echo
	disEcho = (disEcho & ~APU_ECHO_DISABLED) | (al & APU_ECHO_DISABLED);
	APU.DSP[APU_FLG] = al;
	SetNoiseHertz();
}

static void RESA(int, uint8 val)
{
	FLUSH_SAMPLES();
	echoStart = (val * 64 * dspRate / SNES_SAMPLE_RATE) << 1;
}

static void REndX(int, uint8)
{
	APU.DSP[APU_ENDX] = 0;
}

// Null register
static void RNull(int, uint8)
{
	/* Do nothing */
}

// Noise Generator
//  Generates white noise samples (into nSmp)
static inline void NoiseGen()
{
	nDec += nRate;
	if(nDec < nRate)
		nSmp = (int32)(int16)(nSmp * 27865 + 7263);
}

// Pitch Modulation
//
// Changes the pitch based on the output of the previous voice:
//
//  P' = (P * (OUTX + 32768)) >> 15
//
// Pitch modulation in the SNES uses the full 16-bit sample value, not the 8-bit value in OUTX as
// previously believed.
static inline void PitchMod(int i)
{
	int32 t = ((mix[i - 1].mOut + 32768) * mix[i].mOrgP) >> 15;

	if(t > 0x3FFF)
		t = 0x3FFF;
	else if(t < 0)
		t = 0;

	uint64 r = (uint64)t * pitchAdj;
	mix[i].mRate = (uint32)((r >> FIXED_POINT_SHIFT) + ((r & FIXED_POINT_REMAINDER) ? 1 : 0));
}

static inline int32 GetCurSample(int i)
{
	// 16bit sample data has the possibility of doubling
	// and therefore E_SHIFT is up to 8.

	switch(Settings.InterpolatedSound) {
	case INT_NONE:
		return (int32)mix[i].sBuf[mix[i].sIdx];

	case INT_LINEAR:
		{
			// Linear Interpolation
			int16 ps = mix[i].sBuf[(mix[i].sIdx - 1) & 0x1f];
			return (int32)((((((int32)mix[i].sBuf[mix[i].sIdx] - ps) >> 1) *
							 (int32)mix[i].mDec) >> 15) + ps);
		}

	case INT_CUBIC:
		{
			// Cubic Interpolation
			uint32 mDec = (mix[i].mDec >> 8) << 2;
			int32 cursample = (int32)(int16)((((int32)mix[i].sBuf[(mix[i].sIdx - 3) & 0x1f]
											   * cubicTab[mDec + 0]) >> 16)  //d0
											 +(((int32)mix[i].sBuf[(mix[i].sIdx - 2) & 0x1f]
												* cubicTab[mDec + 1]) >> 16)  //d1
											 +(((int32)mix[i].sBuf[(mix[i].sIdx - 1) & 0x1f]
												* cubicTab[mDec + 2]) >> 16)  //d2
											 +(((int32)mix[i].sBuf[mix[i].sIdx]
												* cubicTab[mDec + 3]) >> 16));//d3
			cursample += cursample;
			return cursample;
		}

	case INT_GAUSS:
		{
			// Gaussian Interpolation
			uint32 mDec = (mix[i].mDec >> 8) << 2;
			int32 cursample = (int32)(int16)((((int32)mix[i].sBuf[(mix[i].sIdx - 3) & 0x1f]
											   * gaussTab[mDec + 0]) >> 16)  //d0
											 +(((int32)mix[i].sBuf[(mix[i].sIdx - 2) & 0x1f]
												* gaussTab[mDec + 1]) >> 16)  //d1
											 +(((int32)mix[i].sBuf[(mix[i].sIdx - 1) & 0x1f]
												* gaussTab[mDec + 2]) >> 16)  //d2
											 +(((int32)mix[i].sBuf[mix[i].sIdx]
												* gaussTab[mDec + 3]) >> 16));//d3
			cursample += cursample;
			return cursample;
		}
	}
	return (int32)mix[i].sBuf[mix[i].sIdx];
}

static inline int32 ProcessSample(int i)
{
	int32 cursample;

	// Is there reproducing sound data?
	if(!(mix[i].mFlg & MFLG_END)) {

		// Pitch Modulation
		if(i > 0 && (APU.DSP[APU_PMON] & chs[i].m))
			PitchMod(i);
		else
			mix[i].mRate = mix[i].mOrgRate;

		// Waveform Resizing
		ProcessSrc(i);

		int32 eVal = mix[i].eVal;
		if(eVal > D_ATTACK)
			eVal = D_ATTACK;
		else if(eVal < 0)
			eVal = 0;

		if(!(APU.DSP[APU_NON] & chs[i].m)) {
			mix[i].mOut = ((int32)mix[i].sBuf[mix[i].sIdx] * eVal) >> (E_SHIFT + 7);
			cursample = (GetCurSample(i) * eVal) >> (E_SHIFT + 7);
		}
		else
			cursample = mix[i].mOut = (nSmp * eVal) >> (E_SHIFT + 7);
	}
	// Note this, the noise hears only the length of the sound source data.
	else
		cursample = mix[i].mOut = 0;
	return cursample;
}

// Finite Impulse Response Echo Filter
// 
// Filters the echo using an eight tap FIR filter:
// 
//         7
//        ---
//    x = \   c  * s
//        /    n    n
//        ---
//        n=0
// 
//    x = output sample
//    c = filter coefficient (-.7)
//    s = unfiltered sample
//    n = 0 is the oldest sample and 7 is the most recent
// 
// FIR filters are based on the sample rate.  This was fine in the SNES, because the sample rate was
// always 32kHz, but in the case of an emulator the sample rate can change.  So measures have to be
// taken to ensure the filter will have the same effect, regardless of the output sample rate.
// 
// To overcome this problem, I figured each tap of the filter is applied every 31250ns.  So the
// solution is to calculate when 31250ns have gone by, and use the sample at that point.  Of course
// this method really only works if the output rate is a multiple of 32k.  In order to get accurate
// results, some sort of interpolation method needs to be introduced.  I went the cheap route and used
// linear interpolation.
static inline void FIRFilter(int32 &l, int32 &r)
{
	l = 0;
	r = 0;

	// Reset decimal overflow, so filtering is consistant
	uint32 firDec = 0;
	uint32 fi = firCur;

	for (int i = 0; i < 8; i++) {
#ifdef FIR_LINEAR_INTERPOLATION
		// linear interpolation
		l += (((((Loop[(fi + 0 - 2) & (FIRBUF - 1)] - Loop[fi + 0]) *
				 (int32)firDec) >> FIXED_POINT_SHIFT) + Loop[fi + 0]) * FilterTaps[i]) >> 7;
		r += (((((Loop[(fi + 1 - 2) & (FIRBUF - 1)] - Loop[fi + 1]) *
				 (int32)firDec) >> FIXED_POINT_SHIFT) + Loop[fi + 1]) * FilterTaps[i]) >> 7;
#else
		l += (Loop[fi + 0] * FilterTaps[i]) >> 7;
		r += (Loop[fi + 1] * FilterTaps[i]) >> 7;
#endif
		fi = (fi - (((firDec += firRate) >> FIXED_POINT_SHIFT) << 1)) & (FIRBUF - 1);
		firDec &= FIXED_POINT_REMAINDER;
	}

	//l = (int32)(int16)l & ~1;
	//r = (int32)(int16)r & ~1;
}

static inline void ECHOFilter(int32 &l, int32 &r,
							  int32 ll, int32 rr)
{
	if(firEnabl) {
		Loop[firCur + 0] = Echo[echoStart + echoCur + 0];
		Loop[firCur + 1] = Echo[echoStart + echoCur + 1];

		// Filter echo
		FIRFilter(l, r);

		firCur = (firCur + 2) & (FIRBUF - 1);

		// Add feedback
		if(!(APU.DSP[APU_FLG] & APU_ECHO_DISABLED)) {
			Echo[echoStart + echoCur + 0] = ((l * echoFB) >> 7) + ll;
			Echo[echoStart + echoCur + 1] = ((r * echoFB) >> 7) + rr;
		}
	}
	// all taps are 0x00
	else if(/* !firEnabl && */APU.DSP[APU_C0] == 0x00) {
		Loop[firCur + 0] = Echo[echoStart + echoCur + 0];
		Loop[firCur + 1] = Echo[echoStart + echoCur + 1];

		l = 0;
		r = 0;

		firCur = (firCur + 2) & (FIRBUF - 1);

		if(!(APU.DSP[APU_FLG] & APU_ECHO_DISABLED)) {
			Echo[echoStart + echoCur + 0] = ll;
			Echo[echoStart + echoCur + 1] = rr;
		}
	}
	// taps "0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00"
	else {
		l = Loop[firCur + 0] = Echo[echoStart + echoCur + 0];
		r = Loop[firCur + 1] = Echo[echoStart + echoCur + 1];

		firCur = (firCur + 2) & (FIRBUF - 1);

		// Add feedback
		if(!(APU.DSP[APU_FLG] & APU_ECHO_DISABLED)) {
			Echo[echoStart + echoCur + 0] = ((l * echoFB) >> 7) + ll;
			Echo[echoStart + echoCur + 1] = ((r * echoFB) >> 7) + rr;
		}
	}

	echoCur = (echoCur + 2) % echoDel;
}

//**************************************************************************************************
// Emulate DSP
//
// Emulates the DSP of the SNES
//
// Notes:
//    If 'pBuf' is NULL, the routine MIX_NONE will be used
//    Range checking is performed on 'size'
//
//    Callers should use EmuAPU instead
//
// In:
//    pBuf-> Buffer to store output
//	  num  = Length of buffer (in samples, can be 0)
//
// Out:
//    -> End of buffer
//
// Destroys:
//    ST(0-7)
void *EmuAPUDSP(void *pBuf, int32 num)
{
	uint8 *curptr = (uint8 *)pBuf;

	if(mixRout[(pBuf == NULL) ? MIX_NONE : dspMix](curptr, num) == FALSE) {
		// Output silence
		num = num * dspSize * dspChn;

		// mixlen is already in bytes instead of u32's, so no multiply is needed
		memset(curptr, (dspSize != 1) ? 0 : 0x80, num);
		curptr += num;
	}
	return curptr;
}

static bool8 EmuAPUDSPN(uint8 *&pBuf, int32 &num)
{
	for(int cnt = 0; cnt < num; cnt++) {
#ifdef CALC_T64CNTS
		// Calculate Sample Counter
		t64CntS += ((t64Dec += t64Rate) >> FIXED_POINT_SHIFT);
		t64Dec &= FIXED_POINT_REMAINDER;
#endif
        // Generate Noise
		NoiseGen();

		// Erase current echo samples
		int32 EchoSampleL = 0, EchoSampleR = 0;

		for(int i = 0; i < 8; i++) {

			ChkStartSrc(i);

			// Is the current voice active?
			if(!(mix[i].mFlg & MFLG_OFF)) {

				int32 cursample = ProcessSample(i);

				// Mixing
				if(cursample != 0 && !(mix[i].mFlg & MFLG_MUTE)) {
					if(APU.DSP[APU_EON] & chs[i].m){// Is echo on?
						EchoSampleL += (cursample * mix[i].mChnL) >> 7;
						EchoSampleR += (cursample * mix[i].mChnR) >> 7;
					}
				}

				// Envelope Calculation
				CalcEnv(i);

			}//if(!(mix[i].mFlg & MFLG_OFF))
			else
				mix[i].mOut = 0;
        }//for(i = 0; i < 8; i++)

		int32 l, r;
		ECHOFilter(l, r,
				   EchoSampleL,
				   EchoSampleR);
	}
	return (pBuf == NULL);
}

// Emulate DSP (Integer)
//
// Emulates the DSP of the SNES using standard 80386 instructions.
// Samples can be 8 or 16-bit.
static bool8 EmuAPUDSPI(uint8 *&pBuf, int32 &num)
{
	for(int cnt = 0; cnt < num; cnt++, pBuf += (dspSize * dspChn)) {
#ifdef CALC_T64CNTS
		// Calculate Sample Counter
		t64CntS += ((t64Dec += t64Rate) >> FIXED_POINT_SHIFT);
		t64Dec &= FIXED_POINT_REMAINDER;
#endif
        // Generate Noise
		NoiseGen();

		// Erase current samples
		int32 MixSampleL = 0, MixSampleR = 0;
		int32 EchoSampleL = 0, EchoSampleR = 0;

		for(int i = 0; i < 8; i++) {

			ChkStartSrc(i);

			// Is the current voice active?
			if(!(mix[i].mFlg & MFLG_OFF)) {

				int32 cursample = ProcessSample(i);

				// Mixing
				if(cursample != 0 && !(mix[i].mFlg & MFLG_MUTE)) {
					MixSampleL += (cursample * mix[i].mChnL) >> 7;// Add to master samples
					MixSampleR += (cursample * mix[i].mChnR) >> 7;
					if(APU.DSP[APU_EON] & chs[i].m){// Is echo on?
						EchoSampleL += (cursample * mix[i].mChnL) >> 7;
						EchoSampleR += (cursample * mix[i].mChnR) >> 7;
					}
				}

				// Envelope Calculation
				CalcEnv(i);

			}//if(!(mix[i].mFlg & MFLG_OFF))
			else
				mix[i].mOut = 0;
        }//for(i = 0; i < 8; i++)

		int32 l, r;
		ECHOFilter(l, r,
				   EchoSampleL,
				   EchoSampleR);

		if(!(APU.DSP[APU_FLG] & APU_MUTE)) {// Is the DSP not muted?

			// Multiply samples by main volume
			MixSampleL = (MixSampleL * volMainL) >> 7;
			MixSampleR = (MixSampleR * volMainR) >> 7;

			if(!(disEcho & DSP_NOECHO)) {
				MixSampleL += (l * volEchoL) >> 7;
				MixSampleR += (r * volEchoR) >> 7;
			}

			// Render sound buffer
			switch(dspChn) {
			case 1:// mono
				switch(dspSize) {
				case 1:// 8bit
					l = (((MixSampleL + MixSampleR) * 181) >> 16) + 128;

					if(l < 0)
						l = 0;
					else if(l > 255)
						l = 255;

					pBuf[0] = (uint8)l;
					break;

				case 2:// 16bit
					l = ((MixSampleL + MixSampleR) * 181) >> 8;

					if(l < -32768)
						l = -32768;
					else if(l > 32767)
						l = 32767;

					((int16 *)pBuf)[0] = (int16)l;
					break;
				}
				break;

			case 2:// stereo
				switch(dspSize) {
				case 1:// 8bit
					l = (MixSampleL + 32768) >> 8;// Unsign sample
					r = (MixSampleR + 32768) >> 8;// Unsign sample

					if(l < 0)
						l = 0;
					else if(l > 255)
						l = 255;

					if(r < 0)
						r = 0;
					else if(r > 255)
						r = 255;

					pBuf[0] = (uint8)l;
					pBuf[1] = (uint8)r;
					break;

				case 2:// 16bit
					l = MixSampleL;
					r = MixSampleR;

					if(l < -32768)
						l = -32768;
					else if(l > 32767)
						l = 32767;


					if(r < -32768)
						r = -32768;
					else if(r > 32767)
						r = 32767;

					((int16 *)pBuf)[0] = (int16)l;
					((int16 *)pBuf)[1] = (int16)r;
					break;
				}
				break;
			}
		}
		else {//if(APU.DSP[APU_FLG] & APU_MUTE)
			// Clear sound buffer
			memset(pBuf, (dspSize != 1) ? 0 : 0x80, (size_t)(dspSize * dspChn));
		}
    }
	return TRUE;
}

static bool8 EmuAPUDSPX(uint8 *&pBuf, int32 &num)
{
	return EmuAPUDSPI(pBuf, num);
}

static bool8 EmuAPUDSPF(uint8 *&pBuf, int32 &num)
{
	return EmuAPUDSPI(pBuf, num);
}

//Decompress Sound Source
//
//Decompresses a 9-byte bit-rate reduced block into 16 16-bit samples
//
//In:
//  blk_hdr     = Block header
//  xsample_blk = Sample Block
//  output_buf  = Output buffer
//  smp_1       = Last sample of previous block
//  smp_2       = Next to last sample
//
//Out:
//  xsample_blk = Next Block
//  smp_1       = Last sample
//  smp_2       = Next to last sample
static void UnpckSrc(uint8 blk_hdr,
					 uint16 &xsample_blk,
					 int16 *output_buf,
					 int32 &smp_1,
					 int32 &smp_2)
{
	uint8 *sample_blk = DSP_GET_SRC_P(xsample_blk);
	xsample_blk += 9;

	//skip header
	sample_blk++;
	int32 *BRR_row = &brrTab[(blk_hdr & 0xf0)];

    switch(blk_hdr & 0x0c){
	case 0x00:
        // Method0 - [Smp]
        for(int i = 0; i < 8; i++){
			output_buf[0] = BRR_row[sample_blk[0] >> 4];
			output_buf[1] = BRR_row[sample_blk[0] & 0x0f];
			sample_blk++;
			output_buf += 2;
        }
		smp_2 = output_buf[-2];
		smp_1 = output_buf[-1];
		break;

	case 0x04:
        // Method1 - [Delta]+[Smp-1](15/16)
        // [Smp-1] here is 16 bits, which means we have to truncate smp_1.
        for(int i = 0; i < 8; i++){
			smp_2 = (int32)(int16)(BRR_row[sample_blk[0] >> 4] +
								   smp_1 + (((-smp_1) >> 5) << 1));
			output_buf[0] = smp_2;

			smp_1 = (int32)(int16)(BRR_row[sample_blk[0] & 0x0f] +
								   smp_2 + (((-smp_2) >> 5) << 1));
			output_buf[1] = smp_1;

			sample_blk++;
			output_buf += 2;
        }
		break;

	case 0x08:
        // Method2 - [Delta]+[Smp-1](61/32)-[Smp-2](15/16)
        // [Smp-1] is something more than 16 bits, 24 seems good. [Smp-2] is 16
        // bits though. Pay attention!
        for(int i = 0; i < 8; i++){
			int32 smp;
			smp = ((smp_2 >> 5) << 1) - smp_2;
			smp_2 = smp_1;
			smp_1 = (BRR_row[sample_blk[0] >> 4] + (smp_1 << 1) +
					 ((-((smp_1 << 1) + smp_1) >> 6) << 1) + smp);
			if(smp_1 < -65536)
				smp_1 = 0;
			else if(smp_1 > 65534)
				smp_1 = -2;
			else
				smp_1 = (int32)(int16)smp_1;
			output_buf[0] = smp_1;

			smp = ((smp_2 >> 5) << 1) - smp_2;
			smp_2 = smp_1;
			smp_1 = (BRR_row[sample_blk[0] & 0x0f] + (smp_1 << 1) +
					 ((-((smp_1 << 1) + smp_1) >> 6) << 1) + smp);
			if(smp_1 < -65536)
				smp_1 = 0;
			else if(smp_1 > 65534)
				smp_1 = -2;
			else
				smp_1 = (int32)(int16)smp_1;
			output_buf[1] = smp_1;

			sample_blk++;
			output_buf += 2;
		}
		break;
 
	case 0x0c:
        // Method3 - [Delta]+[Smp-1](115/64)-[Smp-2](13/16)
        // [Smp-1] is something more than 16 bits, 24 seems good. [Smp-2] is 16
        // bits though. Pay attention!
        for(int i = 0; i < 8; i++) {
			int32 smp;
			smp = ((((smp_2 << 1) + smp_2) >> 5) << 1) - smp_2;
			smp_2 = smp_1;
			smp_1 = (BRR_row[sample_blk[0] >> 4] + (smp_1 << 1) +
					 ((-((smp_1 << 3) + (smp_1 << 2) + smp_1) >> 7) << 1) + smp);
			if(smp_1 < -65536)
				smp_1 = 0;
			else if(smp_1 > 65534)
				smp_1 = -2;
			else
				smp_1 = (int32)(int16)smp_1;
			output_buf[0] = smp_1;

			smp = ((((smp_2 << 1) + smp_2) >> 5) << 1) - smp_2;
			smp_2 = smp_1;
			smp_1 = (BRR_row[sample_blk[0] & 0x0f] + (smp_1 << 1) +
					 ((-((smp_1 << 3) + (smp_1 << 2) + smp_1) >> 7) << 1) + smp);
			if(smp_1 < -65536)
				smp_1 = 0;
			else if(smp_1 > 65534)
				smp_1 = -2;
			else
				smp_1 = (int32)(int16)smp_1;
			output_buf[1] = smp_1;

			sample_blk++;
			output_buf += 2;
        }
		break;
    }
}

