#include "snes9x.h"
#include "ppu.h"
#include "bmpfutil.h"
#include "snesgd08.h"
#include "snesgd16.h"

#define MULBRIGHTNESS 140

static bool8 SNESWriteBMP(FILE *fp);
static bool8 SNESWritePCX(FILE *fp);
static bool8 SNESWriteGIF(FILE *fp);
static bool8 SNESWritePNG(FILE *fp);

static bool8 _ReduceTheColorTo256;
static bool8 _ReduceTheColorTo256IfWithin;
static bool8 _ImagefileHiresStretch;
static bool8 _RLEcompression256bitmap;

bool8 S9xTakeScreenShot(char *filename,
						bool8 ReduceTheColorTo256,
						bool8 ReduceTheColorTo256IfWithin,
						bool8 ImagefileHiresStretch,
						bool8 RLEcompression256bitmap)
{
	if(filename == NULL)
		return FALSE;

	//Check Resolution
	if(!(IPPU.RenderedScreenWidth == SNES_WIDTH && IPPU.RenderedScreenHeight == SNES_HEIGHT)
		&& !(IPPU.RenderedScreenWidth == SNES_WIDTH && IPPU.RenderedScreenHeight == SNES_HEIGHT_EXTENDED)
		&& !(IPPU.RenderedScreenWidth == SNES_WIDTH && IPPU.RenderedScreenHeight == (SNES_HEIGHT * 2))
		&& !(IPPU.RenderedScreenWidth == SNES_WIDTH && IPPU.RenderedScreenHeight == (SNES_HEIGHT_EXTENDED * 2))
		&& !(IPPU.RenderedScreenWidth == (SNES_WIDTH * 2) && IPPU.RenderedScreenHeight == SNES_HEIGHT)
		&& !(IPPU.RenderedScreenWidth == (SNES_WIDTH * 2) && IPPU.RenderedScreenHeight == SNES_HEIGHT_EXTENDED)
		&& !(IPPU.RenderedScreenWidth == (SNES_WIDTH * 2) && IPPU.RenderedScreenHeight == (SNES_HEIGHT * 2))
		&& !(IPPU.RenderedScreenWidth == (SNES_WIDTH * 2) && IPPU.RenderedScreenHeight == (SNES_HEIGHT_EXTENDED * 2))
		)
		return FALSE;
	
	_ReduceTheColorTo256 = ReduceTheColorTo256;
	_ReduceTheColorTo256IfWithin = ReduceTheColorTo256IfWithin;
	_ImagefileHiresStretch = ImagefileHiresStretch;
	_RLEcompression256bitmap = RLEcompression256bitmap;
	
	//Check ext
	static char extstr[_MAX_EXT];
	char *extptr = strrchr((const char *)filename, '.');
	if(extptr && strlen(extptr) < _MAX_EXT)
		strcpy(extstr, extptr);
	else
		return FALSE;
	
	//File open
	FILE *fp;
	fp = fopen(filename, "wb");
	if(!fp)
		return FALSE;

	bool8 Err;
	//Select output format
	if(!strcasecmp(extstr, ".pcx"))
		Err = SNESWritePCX(fp);
	else if(!strcasecmp(extstr, ".gif"))
		Err = SNESWriteGIF(fp);
	else if(!strcmp(extstr, ".png"))
		Err = SNESWritePNG(fp);
	else
		Err = SNESWriteBMP(fp);
	if(!Err)
		remove(filename);
	return Err;
}

static void *MemAllocFunc(uint32 bytecount)
{
	return malloc((size_t)bytecount);
}

static void MemFreeFunc(void *memptr)
{
	free(memptr);
	return;
}

static uint8 *SNESCreateBitmapImage(bool8 Force8)
{
	int SrcPixelByteSize = (Settings.SixteenBit ? 2 : 1);
	int DstPixelByteSize = (Settings.SixteenBit ? 3 : 1);
	uint32 bf_size = BITMAP_HEADER_SIZE + (Settings.SixteenBit ? 0 : 1024) +
		(IPPU.RenderedScreenWidth * IPPU.RenderedScreenHeight * DstPixelByteSize);
	uint8 *pbmp = (uint8 *)malloc((size_t)bf_size);
	
	if(!pbmp)
		return NULL;
	
	WRITE_WORD(pbmp + BF_TYPE, 0x4D42);
	WRITE_DWORD(pbmp + BF_SIZE, bf_size);
	WRITE_WORD(pbmp + BF_RESERVED1, 0);
	WRITE_WORD(pbmp + BF_RESERVED2, 0);
	WRITE_DWORD(pbmp + BF_OFFBITS, BITMAP_HEADER_SIZE + (Settings.SixteenBit ? 0 : 1024));
	
	WRITE_DWORD(pbmp + BI_SIZE, BITMAP_INFO_SIZE);
	WRITE_DWORD(pbmp + BI_WIDTH, (uint32)IPPU.RenderedScreenWidth);
	WRITE_DWORD(pbmp + BI_HEIGHT, (uint32)IPPU.RenderedScreenHeight);
	WRITE_WORD(pbmp + BI_PLANES, 1);
	WRITE_WORD(pbmp + BI_BITCOUNT, (Settings.SixteenBit ? 24 : 8));
	WRITE_DWORD(pbmp + BI_COMPRESSION, 0);
	WRITE_DWORD(pbmp + BI_SIZEIMAGE, IPPU.RenderedScreenWidth * IPPU.RenderedScreenHeight * DstPixelByteSize);
	WRITE_DWORD(pbmp + BI_XPELSPERMETER, 0);
	WRITE_DWORD(pbmp + BI_YPELSPERMETER, 0);
	WRITE_DWORD(pbmp + BI_CLRUSED, (Settings.SixteenBit ? 0 : 256));
	WRITE_DWORD(pbmp + BI_CLRIMPORTANT, (Settings.SixteenBit ? 0 : 256));
	
	uint8 *pScreenBuffer = GFX.Screen + GFX.RealPitch * (IPPU.RenderedScreenHeight - 1);
	
	int dibpinc = 0;
	
	if(!Settings.SixteenBit) {
		//Create palette
		uint16 Brightness = IPPU_MaxBrightness * MULBRIGHTNESS;
		for (int i = 0; i < 256; i++)
		{
			*((uint8 *)pbmp + BITMAP_DATA_OFFSET + i * 4) = (((PPU_CGDATA [i] >> 10) & 0x1F) * Brightness) >> 8;
			*((uint8 *)pbmp + BITMAP_DATA_OFFSET + i * 4 + 1) = (((PPU_CGDATA [i] >>  5) & 0x1F) * Brightness) >> 8;
			*((uint8 *)pbmp + BITMAP_DATA_OFFSET + i * 4 + 2) = (((PPU_CGDATA [i] >>  0) & 0x1F) * Brightness) >> 8;
			*((uint8 *)pbmp + BITMAP_DATA_OFFSET + i * 4 + 3) = 0;
		}
		for(int Height = IPPU.RenderedScreenHeight;
			Height;
			Height--, pScreenBuffer -= (GFX.RealPitch + IPPU.RenderedScreenWidth * SrcPixelByteSize))
			for(int Width = IPPU.RenderedScreenWidth;
				Width;
				Width--, pScreenBuffer += SrcPixelByteSize, dibpinc += DstPixelByteSize)
				*((uint8 *)pbmp + BITMAP_DATA_OFFSET + 1024 + dibpinc) = *pScreenBuffer;
	}
	else {
		uint32 p, r, g, b;
		for(int Height = IPPU.RenderedScreenHeight;
			Height;
			Height--, pScreenBuffer -= (GFX.RealPitch + IPPU.RenderedScreenWidth * SrcPixelByteSize)) {
			for(int Width = IPPU.RenderedScreenWidth;
				Width;
				Width--, pScreenBuffer += SrcPixelByteSize, dibpinc += DstPixelByteSize) {
				p = (uint32)READ_WORD(pScreenBuffer);
				DECOMPOSE_PIXEL(p, r, g, b);
				*((uint8 *)pbmp + BITMAP_DATA_OFFSET + dibpinc + 0) = (uint8)b << 3;
				*((uint8 *)pbmp + BITMAP_DATA_OFFSET + dibpinc + 1) = (uint8)g << 3;
				*((uint8 *)pbmp + BITMAP_DATA_OFFSET + dibpinc + 2) = (uint8)r << 3;
			}
		}
	}
	
	if((Force8 || _ReduceTheColorTo256) && READ_WORD(pbmp + BI_BITCOUNT) == 24) {
		uint8 *pbmp8 = 
			(uint8 *)malloc((size_t)((uint32)BITMAP_HEADER_SIZE + 1024 + READ_DWORD(pbmp + BI_WIDTH) * READ_DWORD(pbmp + BI_HEIGHT)));
//		if(pbmp8 && ConvertBitmapImage24To8(pbmp8, pbmp))
		if(pbmp8 && ConvertBitmapImage24To8IfWithin(pbmp8, pbmp)) {
			free(pbmp);
			pbmp = pbmp8;
		}
		else if(pbmp8) {
			free(pbmp8);
		}
	}
	else if(!Force8 && _ReduceTheColorTo256IfWithin && READ_WORD(pbmp + BI_BITCOUNT) == 24) {
		uint8 *pbmp8 = 
			(uint8 *)malloc((size_t)((uint32)BITMAP_HEADER_SIZE + 1024 + READ_DWORD(pbmp + BI_WIDTH) * READ_DWORD(pbmp + BI_HEIGHT)));
		if(pbmp8 && ConvertBitmapImage24To8IfWithin(pbmp8, pbmp)) {
			free(pbmp);
			pbmp = pbmp8;
		}
		else if(pbmp8) {
			free(pbmp8);
		}
	}
	if(_ImagefileHiresStretch &&
	   pbmp && IPPU.RenderedScreenWidth == (SNES_WIDTH * 2) &&
	   IPPU.RenderedScreenHeight < (SNES_HEIGHT * 2)) {

		uint8 *pbmpx2h = 
			(uint8 *)malloc((size_t)(READ_DWORD(pbmp + BF_SIZE) + READ_DWORD(pbmp + BI_SIZEIMAGE)));
		if(pbmpx2h && x2hBitmapImage(pbmpx2h, pbmp)) {
			free(pbmp);
			pbmp = pbmpx2h;
		}
		else if(pbmpx2h) {
			free(pbmpx2h);
		}
	}
	else if(_ImagefileHiresStretch &&
			pbmp && IPPU.RenderedScreenWidth == SNES_WIDTH &&
			IPPU.RenderedScreenHeight >= (SNES_HEIGHT * 2)) {
		uint8 *pbmpx2w =
			(uint8 *)malloc((size_t)(READ_DWORD(pbmp + BF_SIZE) + READ_DWORD(pbmp + BI_SIZEIMAGE)));
		if(pbmpx2w && x2wBitmapImage(pbmpx2w, pbmp)) {
			free(pbmp);
			pbmp = pbmpx2w;
		}
		else if(pbmpx2w) {
			free(pbmpx2w);
		}
	}
	return pbmp;
}

static bool8 SNESWriteBMP(FILE *fp)
{
	if(!fp)
		return FALSE;
	
	uint8 *pBitmapImage = SNESCreateBitmapImage(FALSE);
	
	if(!pBitmapImage) {
		fclose(fp);
		return FALSE;
	}
	
	//RLE compression, 8bit color only
	if(_RLEcompression256bitmap && READ_WORD(pBitmapImage + BI_BITCOUNT) == 8) {
		int num_palette = (int)READ_DWORD(pBitmapImage + BI_CLRUSED);
		int bmp_pitch = READ_DWORD(pBitmapImage + BI_WIDTH) * READ_WORD(pBitmapImage + BI_BITCOUNT) / 8;
		
		if(num_palette == 0)
			num_palette = (1 << READ_WORD(pBitmapImage + BI_BITCOUNT));
		
		uint8 *pBitmapImageRLE =
			(uint8 *)malloc((size_t)((uint32)BITMAP_HEADER_SIZE + (num_palette * 4) +
							 ((READ_DWORD(pBitmapImage + BI_HEIGHT) + 2) * (bmp_pitch + 2) * 2) + 2));
		
		if(pBitmapImageRLE && bmp_compress_dib_rle(pBitmapImageRLE, pBitmapImage)) {
			free(pBitmapImage);
			pBitmapImage = pBitmapImageRLE;
		}
		else if(pBitmapImageRLE) {
			free(pBitmapImageRLE);
		}
	}
	
	int Err = fwrite(pBitmapImage, (unsigned long)READ_DWORD(pBitmapImage + BF_SIZE), 1, fp);
	fclose(fp);
	free(pBitmapImage);
	if(Err)
		return TRUE;
	else
		return FALSE;
}

static bool8 SNESWritePCX(FILE *fp)
{
	if(!fp)
		return FALSE;
	
	uint8 *pBitmapImage = SNESCreateBitmapImage(FALSE);
	
	if(!pBitmapImage) {
		fclose(fp);
		return FALSE;
	}
	
	//Create PCX palette
	uint8 pcxpalette[1024];
	if(READ_WORD(pBitmapImage + BI_BITCOUNT) == 8) {
		for (int i = 0; i < 256; i++)
		{
			pcxpalette[i * 4 + 0] = *(pBitmapImage + BITMAP_HEADER_SIZE + i * 4 + 2) >> 2;
			pcxpalette[i * 4 + 1] = *(pBitmapImage + BITMAP_HEADER_SIZE + i * 4 + 1) >> 2;
			pcxpalette[i * 4 + 2] = *(pBitmapImage + BITMAP_HEADER_SIZE + i * 4 + 0) >> 2;
			pcxpalette[i * 4 + 3] = 0;
		}
	}
	else {
		uint16 Brightness = IPPU_MaxBrightness * MULBRIGHTNESS;
		for (int i = 0; i < 256; i++)
		{
			pcxpalette[i * 4 + 0] = (((PPU_CGDATA [i] >>  0) & 0x1F) * Brightness) >> 10;
			pcxpalette[i * 4 + 1] = (((PPU_CGDATA [i] >>  5) & 0x1F) * Brightness) >> 10;
			pcxpalette[i * 4 + 2] = (((PPU_CGDATA [i] >> 10) & 0x1F) * Brightness) >> 10;
			pcxpalette[i * 4 + 3] = 0;
		}
	}
	
	uint32 ImageDataSize;
	uint8 *pPCXImage = bmp2pcx(pBitmapImage,
							   pcxpalette,
							   &ImageDataSize,
							   MemAllocFunc,
							   MemFreeFunc);
	free(pBitmapImage);
	
	if(!pPCXImage) {
		fclose(fp);
		return FALSE;
	}
	
	int Err = fwrite((const void *)pPCXImage, ImageDataSize, 1, fp);
	
	fclose(fp);
	free(pPCXImage);
	if(Err)
		return TRUE;
	else
		return FALSE;
}

static bool8 SNESWriteGIF(FILE *fp)
{
	if(!fp)
		return FALSE;
	
	uint8 *pBitmapImage = SNESCreateBitmapImage(TRUE);
	
	if(!pBitmapImage) {
		fclose(fp);
		return FALSE;
	}
	
	if(READ_WORD(pBitmapImage + BI_BITCOUNT) != 8) {
		fclose(fp);
		free(pBitmapImage);
		return FALSE;
	}
	
	uint32 ImageDataSize;
	uint8 *pGIFImage = bmp2gif(pBitmapImage,
							   &ImageDataSize,
							   "Generated by uosnes",
							   MemAllocFunc,
							   MemFreeFunc);
	free(pBitmapImage);
	
	if(!pGIFImage) {
		fclose(fp);
		return FALSE;
	}
	
	int Err = fwrite((const void *)pGIFImage, ImageDataSize, 1, fp);
	fclose(fp);
	free(pGIFImage);
	
	if(Err)
		return TRUE;
	else
		return FALSE;
}

static bool8 SNESWritePNG(FILE *fp)
{
	if(!fp)
		return FALSE;
	
	uint8 *pBitmapImage = SNESCreateBitmapImage(FALSE);
	
	if(!pBitmapImage) {
		fclose(fp);
		return FALSE;
	}
	
	bool8 pngErr = bmp2png(pBitmapImage, fp, "Emulator", "uosnes");
	
	fclose(fp);
	free(pBitmapImage);
	return pngErr;
}

void S9xDrawScreenData()
{
	uint8 *sc = (uint8 *)(Settings.SixteenBit ? snesgd16 : snesgd08);
	int w = (int)READ_DWORD(&sc[0]);
	int h = (int)READ_DWORD(&sc[4]);
	sc += 8;
	
	if(Settings.SixteenBit) {
		for(int y = 0; y < h; y++) {
			for(int x = 0; x < w; x++) {
				register uint32 p, r, g, b;
				p = (uint32)READ_WORD(&sc[y * w * 2 + x * 2]);
				DECOMPOSE_PIXEL_RGB555(p, r, g, b);
				WRITE_WORD(&GFX.Screen[y * GFX.RealPitch + x * 2], (uint16)BUILD_PIXEL (r, g, b));
			}
		}
	}
	else {
		for(int y = 0; y < h; y++)
			for(int x = 0; x < w; x++)
				GFX.Screen[y * GFX.RealPitch + x] = sc[y * w + x];
		
		sc += w * h;
		
		for(int i = 0; i < 256; i++)
			PPU_CGDATA[i] = READ_WORD(sc + i * 2);

		sc += 256 * 2;
		
		IPPU_MaxBrightness = *sc;
	}
	
	IPPU.RenderedScreenWidth = w;
	IPPU.RenderedScreenHeight = h;
}

