/*
  This source code
  Copyright (C) 2004 sanmaiwashi
  http://toriniku.sourceforge.net/

  BS-X technical.txt
  http://bszelda.zeldalegends.net/
*/
#include "memmap.h"
#include "bsx.h"

//#define BSX_DEBUG
#ifdef BSX_DEBUG
static const char bsx_log_name[] = "bsx.log";
static FILE *bsx_fs = NULL;
extern FILE *trace;
#endif

static uint8 *bsxmmc = NULL;
static uint8 *bsxmmc_p = NULL;
static uint8 *bsxflash0000 = NULL;
static uint8 *bsx_rom = NULL;
static bool8 bsx_rom_hirom = TRUE;

uint8 GetBSX (uint32 Address)
{
	uint8 Byte;

	// flash hardware registers.
	// $C0:0000,2AAA,5555 and $C0:FF00-FF1F
	if((Address & 0xff0000) == 0xc00000) {
		uint16 address = (uint16)(Address & 0xffff);
		switch(address) {
		case 0x0000:
			Byte = 0x80;
			break;

		case 0x0002:
			Byte = bsxflash0000[0] >= 3 ? 0x80 : 0x00;
			break;

		case 0x0004:
			//Address C00000 w Byte 50
			//Address C00000 w Byte 71
			//Address C00004 r Byte 00
			//Address C00000 w Byte 97
			//Address C00000 w Byte D0
			//Address C00000 w Byte 71
			//Address C00004 r Byte 80
			Byte = bsxflash0000[0] >= 3 ? 0x80 : 0x00;
			break;

		case 0xff00:
			Byte = 0x4D;//M
			break;

		case 0xff02:
			Byte = 0x50;//P
			break;

		case 0xff04:
			//$80:C190 AF D8 99 7E LDA $7E99D8[$7E:99D8] $C0:FF04 data
			//$80:C194 29 81       AND #$81             
			//$80:C196 D0 26       BNE $26    [$C1BE]    Error?
			Byte = 0x00;
			break;

		case 0xff06:
			//$80:C19A AF D9 99 7E LDA $7E99D9[$7E:99D9] $C0:FF06 data
			//$80:C19E 4A          LSR A                 
			//$80:C19F 4A          LSR A                 
			//$80:C1A0 4A          LSR A                 
			//$80:C1A1 4A          LSR A                 
			//$80:C1A2 3A          DEC A                 
			//$80:C1A3 8D 41 14    STA $1441  [$80:1441] 
			//$80:C1A6 C9 04       CMP #$04              
			//$80:C1A8 B0 14       BCS $14    [$C1BE]    Error?
			//$80:C1AA AF D9 99 7E LDA $7E99D9[$7E:99D9] $C0:FF06 data
			//$80:C1AE 29 0F       AND #$0F              
			//$80:C1B0 38          SEC                   
			//$80:C1B1 E9 06       SBC #$06              
			//$80:C1B3 90 09       BCC $09    [$C1BE]    Error?
			//$80:C1B5 AA          TAX                   
			//$80:C1B6 A9 01       LDA #$01              
			//$80:C1B8 CA          DEX                   
			//$80:C1B9 F0 05       BEQ $05    [$C1C0]
			//$80:C1BB 0A          ASL A             
			//$80:C1BC 80 FA       BRA $FA    [$C1B8]
			//...
			//$80:C1A3 8D 41 14    STA $1440  [$80:1440]

			Byte = 0x4A;
			break;

		case 0xff08:
		case 0xff0A:
		case 0xff0C:
		case 0xff0E:
		case 0xff10:
		case 0xff12:
		case 0xff14:
		case 0xff16:
		case 0xff18:
		case 0xff1A:
		case 0xff1C:
		case 0xff1E:
			Byte = 0x00;
			break;

		default:
			{
#ifdef CPU_SHUTDOWN
				if (bsx_rom == Memory.PSRAM)
					CPU.WaitAddress = CPU.PCAtOpcodeStart;
#endif
				Byte = *(bsx_rom + (address & (bsx_rom_hirom ? 0xffff : 0x7fff)));
			}
			break;
		}
	}
	else {
		switch(Address & 0xffff) {
		case 0x2188://(r/w): Test register
		case 0x2189://(r/w): Test register
		case 0x218A:
		case 0x218B:
		case 0x218C:
		case 0x218D:
		case 0x218E://(r/w): Transmission number
		case 0x218F://(r/w): Transmission number
		case 0x2190://(r): Unknown [status register]
		case 0x2191://(w): Strobe assertion register
		case 0x2192://(r): Data register
		case 0x2193://(r): Transmission status (Data ready when bits 2 and 3 clear)
		case 0x2194://(w): Unknown
		case 0x2195:
		case 0x2196:
		case 0x2197:
		case 0x2198:
		case 0x2199://(r/w): Unknown [modem protocol strings]
			Byte = Memory.FillRAM[Address & 0xffff];
			break;

		case 0x5000:
			switch(Address & 0xff0000){
			case 0x010000:
				Byte = bsxmmc[1];
				break;

			case 0x020000:
				Byte = bsxmmc[2];
				break;

			case 0x030000:
				Byte = bsxmmc[3];
				break;

			case 0x040000:
				Byte = bsxmmc[4];
				break;

			case 0x050000:
				Byte = bsxmmc[5];
				break;

			case 0x060000:
				Byte = bsxmmc[6];
				break;

			case 0x070000:
				Byte = bsxmmc[7];
				break;

			case 0x080000:
				Byte = bsxmmc[8];
				break;

			case 0x090000:
				Byte = bsxmmc[9];
				break;

			case 0x0A0000:
				Byte = bsxmmc[10];
				break;

			case 0x0B0000:
				Byte = bsxmmc[11];
				break;

			case 0x0C0000:
				Byte = bsxmmc[12];
				break;

			case 0x0D0000:
				Byte = bsxmmc[13];
				break;

			case 0x0E0000:
				Byte = bsxmmc[14];
				break;

			default:
				Byte = Memory.FillRAM[Address & 0xffff];
				break;
			}
			break;

		default:
			Byte = Memory.FillRAM[Address & 0xffff];
			break;
		}
	}
#ifdef BSX_DEBUG
	if(!bsx_fs)
		bsx_fs = fopen(bsx_log_name, "w");
	fprintf(bsx_fs,
			"Address %06X r Byte %02X\n",
			(unsigned int)Address, (unsigned int)Byte);
#endif
	return Byte;
}

void SetBSX (uint8 Byte, uint32 Address)
{
#ifdef BSX_DEBUG
	if(!bsx_fs)
		bsx_fs = fopen(bsx_log_name, "w");
	fprintf(bsx_fs,
			"Address %06X w Byte %02X\n",
			(unsigned int)Address, (unsigned int)Byte);
#endif
	// flash hardware registers.
	// $C0:0000,2AAA,5555 and $C0:FF00-FF1F
	if((Address & 0xff0000) == 0xc00000) {
		uint16 address = (uint16)(Address & 0xffff);
		switch(address) {
		case 0x0000:
			//Address C00000 w Byte 38
			//Address C00000 w Byte D0
			//Address C00000 w Byte 71
			//Address C00002 r Byte 80
			//Address C00000 w Byte 72
			//Address C00000 w Byte 75
			//Address C0FF00 r Byte 00
			//Address C0FF02 r Byte 00
			//Address C0FF04 r Byte 00
			//Address C0FF06 r Byte 00
			//Address C0FF08 r Byte 00
			//Address C0FF0A r Byte 00
			//Address C0FF0C r Byte 00
			//Address C0FF0E r Byte 00
			//Address C0FF10 r Byte 00
			//Address C0FF12 r Byte 00
			//Address C00000 w Byte FF
			if(bsxflash0000[0] < 15) {
				bsxflash0000[bsxflash0000[0] + 1] = Byte;
				bsxflash0000[0]++;
			}
			return;

		default:
			break;
		}
		if (bsx_rom != Memory.PSRAM)
			return;

		uint8 *SetAddress = bsx_rom + (address & (bsx_rom_hirom ? 0xffff : 0x7fff));
#ifdef CPU_SHUTDOWN
		if (SetAddress == SA1.WaitByteAddress1 ||
			SetAddress == SA1.WaitByteAddress2) {
			SA1.Executing = SA1.S9xOpcodes != NULL;
			SA1.WaitCounter = 0;
		}
		*SetAddress = Byte;
#else
		*SetAddress = Byte;
#endif
		return;
	}
	else {
		switch(Address & 0xffff) {
		case 0x2188://(r/w): Test register
			break;

		case 0x2189://(r/w): Test register
			break;

		case 0x218A:
			break;

		case 0x218B:
			break;

		case 0x218C:
			break;

		case 0x218D:
			break;

		case 0x218E://(r/w): Transmission number
			break;

		case 0x218F://(r/w): Transmission number
			break;

		case 0x2190://(r): Unknown [status register]
			break;

		case 0x2191://(w): Strobe assertion register
			break;

		case 0x2192://(w): Strobe negation register
			break;

		case 0x2193://(r): Transmission status (Data ready when bits 2 and 3 clear)
			break;

		case 0x2194://(w): Unknown
			break;

		case 0x2195:
			break;

		case 0x2196:
			break;

		case 0x2197:
			break;

		case 0x2198:
			break;

		case 0x2199://(r/w): Unknown [modem protocol strings]
			break;

		case 0x5000:
			switch(Address & 0xff0000){
			case 0x010000:
				bsxmmc[1] = Byte;
				return;

			case 0x020000:
				bsxmmc[2] = Byte;
				return;

			case 0x030000:
				bsxmmc[3] = Byte;
				return;

			case 0x040000:
				bsxmmc[4] = Byte;
				return;

			case 0x050000:
				bsxmmc[5] = Byte;
				return;

			case 0x060000:	
				bsxmmc[6] = Byte;
				return;

			case 0x070000:
				bsxmmc[7] = Byte;
				return;

			case 0x080000:
				bsxmmc[8] = Byte;
				return;

			case 0x090000:
				bsxmmc[9] = Byte;
				return;

			case 0x0A0000:
				bsxmmc[10] = Byte;
				return;

			case 0x0B0000:
				bsxmmc[11] = Byte;
				return;

			case 0x0C0000:
				bsxmmc[12] = Byte;
				return;

			case 0x0D0000:
				bsxmmc[13] = Byte;
				return;

			case 0x0E0000:
				bsxmmc[14] = Byte;
				BSXMMC();
#if 0
				if(!trace) {
					trace = fopen ("bsxtrace.log", "w");
					CPU.Flags |= TRACE_FLAG;
				}
#endif
				return;

			default:
				break;
			}
			break;

		default:
			break;
		}
		Memory.FillRAM[Address & 0xffff] = Byte;
	}
}

/*
BS-X Check Memory Pack.

Address 0C5000 w Byte 80
Address 0D5000 w Byte 80
Address 0E5000 w Byte 80
Address C00000 w Byte 38
Address C00000 w Byte D0
Address C00000 w Byte 71
Address C00002 r Byte 80
Address C00000 w Byte 72
Address C00000 w Byte 75
Address C0FF00 r Byte 00
Address C0FF02 r Byte 00
Address C0FF04 r Byte 00
Address C0FF06 r Byte 00
Address C0FF08 r Byte 00
Address C0FF0A r Byte 00
Address C0FF0C r Byte 00
Address C0FF0E r Byte 00
Address C0FF10 r Byte 00
Address C0FF12 r Byte 00
Address C00000 w Byte FF
Address 055000 w Byte 00
Address 065000 w Byte 00
Address 045000 w Byte 00
Address 015000 w Byte 00
Address 0C5000 w Byte 00
Address 0D5000 w Byte 00
Address 025000 w Byte 80
Address 035000 w Byte 80
Address 0E5000 w Byte 80
*/

static uint8 ffff32[32] = {
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

void ResetBSX(bool8 full)
{
	//Reset MMC
	bsxmmc = &Memory.FillRAM[0x6000];
	bsxmmc_p = &Memory.FillRAM[0x6010];
	bsxflash0000 = &Memory.FillRAM[0x6020];

	if(Memory.BSXInitialized == BSX_NO_INITIALIZED) {
		memset(bsxmmc, 0, 16);
		memset(bsxmmc_p, 0, 16);
		memset(bsxflash0000, 0, 16);
	}

	if(Memory.BSXInitialized != BSX_LOAD_INITIALIZED) {
		if(Memory.BSXMemoryPack > 0 && (!full ||
										Memory.BSXInitialized == BSX_NO_INITIALIZED)) {

			uint8 pack_area = Memory.ROM[0x300000 + (Memory.HiROM ? 0xffd0 : 0x7fd0)];
			bool8 map_psram = (Memory.BSXMemoryPack <= 0x80000 && pack_area != 0xff);
			memcpy(Memory.PSRAM, Memory.ROM + 0x300000, 0x80000);

			bsxmmc[1] = map_psram ? 0x80 : 0x00;//flash card or PSRAM
			bsxmmc[2] = Memory.HiROM ? 0x80 : 0x00;//HiROM or LoROM
			bsxmmc[5] = 0x80;//$40-$4F:0000-FFFF ROM
			bsxmmc[6] = 0x80;//$50-$5F:0000-FFFF ROM
			bsxmmc[7] = 0x00;//$00-$1F:8000-FFFF ROM or BIOS
			bsxmmc[8] = 0x00;//$80-$9F:8000-FFFF ROM or BIOS
			bsxmmc[14] = 0x80;//updates the memory map, Commit
		}
		else {
			bsxmmc[1] = 0x00;//flash card
			bsxmmc[2] = 0x80;//HiROM
			bsxmmc[5] = 0x80;//$40-$4F:0000-FFFF ROM
			bsxmmc[6] = 0x80;//$50-$5F:0000-FFFF ROM
			bsxmmc[7] = 0x80;//$00-$1F:8000-FFFF BIOS
			bsxmmc[8] = 0x80;//$80-$9F:8000-FFFF BIOS
			bsxmmc[14] = 0x80;//updates the memory map, Commit
		}
	}

	for(int i = 0; i < 16; i++)
		bsxmmc_p[i] = bsxmmc[i] ^ 0x80;

	BSXMMC();

	//memmap was changed
    Registers.PC = S9xGetWord (0xFFFC);
    S9xSetPCBase (Registers.PC);
	Memory.BSXInitialized = BSX_YES_INITIALIZED;
}


typedef struct {
	bool8 hirom;
	uint8 *data;
	uint32 data_size;
	uint8 blockisrom;
	int offset_shift;
	int offset_16M;
	int offset_sub;
} Smemmap_params;

static inline void BSXMMC_20_3F__78_7F__A0_BF__C1_FF(Smemmap_params *mp)
{
	int i, c;
	//Banks 20->3f A0->Bf map ROM
	for (c = 0x200; c < 0x400; c += 16) {
		for (i = c + 8; i < c + 16; i++) {
			Memory.Map [i] = Memory.Map [i + 0x800] = &mp->data[(c << mp->offset_shift) % mp->data_size] - mp->offset_sub;
			Memory.BlockIsROM [i] = Memory.BlockIsROM [i + 0x800] = mp->blockisrom;
			Memory.BlockIsRAM [i] = Memory.BlockIsRAM [i + 0x800] = !mp->blockisrom;
		}
	}

	//Banks 78->7d map ROM
	for (c = 0x380; c < 0x3e0; c += 16) {
		for (i = c; i < c + 8; i++)
			Memory.Map [i + 0x400] = &mp->data[(c << mp->offset_shift) % mp->data_size];
		
		for (i = c + 8; i < c + 16; i++)
			Memory.Map [i + 0x400] = &mp->data[((c << mp->offset_shift) + mp->offset_16M) % mp->data_size] - mp->offset_sub;
		
		for (i = c; i < c + 16; i++) {
			Memory.BlockIsROM [i + 0x400] = mp->blockisrom;
			Memory.BlockIsRAM [i + 0x400] = !mp->blockisrom;
		}
	}

	// Banks c1->ff
	for (c = 0x10; c < 0x400; c += 16) {
		for (i = c; i < c + 8; i++)
			Memory.Map [i + 0xc00] = &mp->data[(c << mp->offset_shift) % mp->data_size];
		
		for (i = c + 8; i < c + 16; i++)
			Memory.Map [i + 0xc00] = &mp->data[((c << mp->offset_shift) + mp->offset_16M) % mp->data_size] - mp->offset_sub;
		
		for (i = c; i < c + 16; i++) {
			Memory.BlockIsROM [i + 0xc00] = mp->blockisrom;
			Memory.BlockIsRAM [i + 0xc00] = !mp->blockisrom;
		}
	}
}

static inline void BSXMMC_60_6F(Smemmap_params *mp)
{
	int i, c;
	for (c = 0x200; c < 0x300; c += 16) {
		for (i = c; i < c + 8; i++)
			Memory.Map [i + 0x400] = &mp->data[(c << mp->offset_shift) % mp->data_size];
		
		for (i = c + 8; i < c + 16; i++)
			Memory.Map [i + 0x400] = &mp->data[((c << mp->offset_shift) + mp->offset_16M) % mp->data_size] - mp->offset_sub;
		
		for (i = c; i < c + 16; i++) {
			Memory.BlockIsROM [i + 0x400] = mp->blockisrom;
			Memory.BlockIsRAM [i + 0x400] = !mp->blockisrom;
		}
	}
}

static inline void BSXMMC_00_1F(Smemmap_params *mp)
{
	int i, c;
	for (c = 0; c < 0x200; c += 16) {
		for (i = c + 8; i < c + 16; i++) {
			Memory.Map [i] = &mp->data[(c << mp->offset_shift) % mp->data_size] - mp->offset_sub;
			Memory.BlockIsROM [i] = mp->blockisrom;
			Memory.BlockIsRAM [i] = !mp->blockisrom;
		}
	}
}

static inline void BSXMMC_80_9F(Smemmap_params *mp)
{
	int i, c;
	for (c = 0; c < 0x200; c += 16) {
		for (i = c + 8; i < c + 16; i++) {
			Memory.Map [i + 0x800] = &mp->data[(c << mp->offset_shift) % mp->data_size] - mp->offset_sub;
			Memory.BlockIsROM [i + 0x800] = mp->blockisrom;
			Memory.BlockIsRAM [i + 0x800] = !mp->blockisrom;
		}
	}
}

static inline void BSXMMC_40_4F(Smemmap_params *mp)
{
	int i, c;
	for (c = 0; c < 0x100; c += 16) {
		for (i = c; i < c + 8; i++)
			Memory.Map [i + 0x400] = &mp->data[(c << mp->offset_shift) % mp->data_size];
		
		for (i = c + 8; i < c + 16; i++)
			Memory.Map [i + 0x400] = &mp->data[((c << mp->offset_shift) + mp->offset_16M) % mp->data_size] - mp->offset_sub;

		for (i = c; i < c + 16; i++) {
			Memory.BlockIsROM [i + 0x400] = mp->blockisrom;
			Memory.BlockIsRAM [i + 0x400] = !mp->blockisrom;
		}
	}
}

static inline void BSXMMC_50_5F(Smemmap_params *mp)
{
	int i, c;
	for (c = 0x100; c < 0x200; c += 16) {
		for (i = c; i < c + 8; i++)
			Memory.Map [i + 0x400] = &mp->data[(c << mp->offset_shift) % mp->data_size];
		
		for (i = c + 8; i < c + 16; i++)
			Memory.Map [i + 0x400] = &mp->data[((c << mp->offset_shift) + mp->offset_16M) % mp->data_size] - mp->offset_sub;

		for (i = c; i < c + 16; i++) {
			Memory.BlockIsROM [i + 0x400] = mp->blockisrom;
			Memory.BlockIsRAM [i + 0x400] = !mp->blockisrom;
		}
	}
}

static Smemmap_params mp;

static inline void get_bsx_memmap_params(Smemmap_params *mp, bool8 map_psram)
{
	mp->hirom = (bsxmmc[2] & 0x80) != 0;
	mp->data = map_psram ? Memory.PSRAM : &Memory.ROM[0x300000];
	mp->data_size = map_psram ? 0x80000 : 0x100000;
	mp->blockisrom = map_psram ? FALSE : TRUE;

	mp->offset_shift = mp->hirom ? 12 : 11;
	mp->offset_16M = mp->hirom ? 0 : 0x200000;
	mp->offset_sub = mp->hirom ? 0 : 0x8000;
}

void BSXMMC()
{
	if(!(bsxmmc[14] & 0x80))
		return;

	bool8 map_changed_1_2 = (bsxmmc_p[1] != bsxmmc[1] ||
							 bsxmmc_p[2] != bsxmmc[2]);
	if(map_changed_1_2) {
		get_bsx_memmap_params(&mp, (bsxmmc[1] & 0x80) != 0);
		bsx_rom = mp.data;
		bsx_rom_hirom = mp.hirom;

		BSXMMC_20_3F__78_7F__A0_BF__C1_FF(&mp);
	}

	bool8 map_changed_3 = (bsxmmc_p[3] != bsxmmc[3] ||
						   (!bsxmmc[3] && map_changed_1_2));

	if(map_changed_3) {
		bsxmmc_p[3] = bsxmmc[3];

		if(!bsxmmc[3]) {
			if(!map_changed_1_2) {
				get_bsx_memmap_params(&mp, (bsxmmc[1] & 0x80) != 0);
				bsx_rom = mp.data;
				bsx_rom_hirom = mp.hirom;
			}
			BSXMMC_60_6F(&mp);
		}
		else {
			//BS-X PSRAM
			int c;
			for(c = 0; c < 0x100; c++) {
				Memory.Map[c + 0x600] = &Memory.PSRAM[(c & 0x70) << 12];
				Memory.BlockIsRAM[c + 0x600] = TRUE;
				Memory.BlockIsROM[c + 0x600] = FALSE;
			}
		}
	}

	bool8 map_changed_7 = (bsxmmc_p[7] != bsxmmc[7] ||
						   (!bsxmmc[7] && map_changed_1_2));

	if(map_changed_7) {
		bsxmmc_p[7] = bsxmmc[7];

		if(!bsxmmc[7]) {
			if(!map_changed_1_2) {
				get_bsx_memmap_params(&mp, (bsxmmc[1] & 0x80) != 0);
				bsx_rom = mp.data;
				bsx_rom_hirom = mp.hirom;
			}
			BSXMMC_00_1F(&mp);
		}
		else {
			//BS-X BIOS
			int c, i;
			for (c = 0; c < 0x200; c += 16) {
				for (i = c + 8; i < c + 16; i++) {
					Memory.Map [i] = &Memory.ROM [(c << 11) % 0x100000] - 0x8000;
					Memory.BlockIsROM [i] = TRUE;
					Memory.BlockIsRAM [i] = FALSE;
				}
			}
		}
	}

	bool8 map_changed_8 = (bsxmmc_p[8] != bsxmmc[8] ||
						   (!bsxmmc[8] && map_changed_1_2));

	if(map_changed_8) {
		bsxmmc_p[8] = bsxmmc[8];

		if(!bsxmmc[8]) {
			if(!map_changed_1_2) {
				get_bsx_memmap_params(&mp, (bsxmmc[1] & 0x80) != 0);
				bsx_rom = mp.data;
				bsx_rom_hirom = mp.hirom;
			}
			BSXMMC_80_9F(&mp);
		}
		else {
			//BS-X BIOS
			int c, i;
			for (c = 0; c < 0x200; c += 16) {
				for (i = c + 8; i < c + 16; i++) {
					Memory.Map [i + 0x800] = &Memory.ROM [(c << 11) % 0x100000] - 0x8000;
					Memory.BlockIsROM [i + 0x800] = TRUE;
					Memory.BlockIsRAM [i + 0x800] = FALSE;
				}
			}
		}
	}

	bool8 map_changed_C = (bsxmmc_p[0x0c] != bsxmmc[0x0c] ||
						   map_changed_1_2);

	if(map_changed_C) {
		bsxmmc_p[0x0c] = bsxmmc[0x0c];
		if(!map_changed_1_2) {
			get_bsx_memmap_params(&mp, (bsxmmc[1] & 0x80) != 0);
			bsx_rom = mp.data;
			bsx_rom_hirom = mp.hirom;
		}
		if(bsxmmc[0x0c] & 0x80) {
			Memory.Map [0xc00] = (uint8 *) CMemory::MAP_BSX;
			Memory.BlockIsROM [0xc00] = FALSE;
			Memory.BlockIsRAM [0xc00] = FALSE;

			Memory.Map [0xc02] = (uint8 *) CMemory::MAP_BSX;
			Memory.BlockIsROM [0xc02] = FALSE;
			Memory.BlockIsRAM [0xc02] = FALSE;

			Memory.Map [0xc05] = (uint8 *) CMemory::MAP_BSX;
			Memory.BlockIsROM [0xc05] = FALSE;
			Memory.BlockIsRAM [0xc05] = FALSE;
		}
		else {
			Memory.Map [0xc00] = mp.data;
			Memory.BlockIsROM [0xc00] = mp.blockisrom;
			Memory.BlockIsRAM [0xc00] = !mp.blockisrom;

			Memory.Map [0xc02] = mp.data;
			Memory.BlockIsROM [0xc02] = mp.blockisrom;
			Memory.BlockIsRAM [0xc02] = !mp.blockisrom;

			Memory.Map [0xc05] = mp.data;
			Memory.BlockIsROM [0xc05] = mp.blockisrom;
			Memory.BlockIsRAM [0xc05] = !mp.blockisrom;
		}
		Memory.Map [0xc01] = mp.data;
		Memory.BlockIsROM [0xc01] = mp.blockisrom;
		Memory.BlockIsRAM [0xc01] = !mp.blockisrom;

		Memory.Map [0xc03] = mp.data;
		Memory.BlockIsROM [0xc03] = mp.blockisrom;
		Memory.BlockIsRAM [0xc03] = !mp.blockisrom;

		Memory.Map [0xc04] = mp.data;
		Memory.BlockIsROM [0xc04] = mp.blockisrom;
		Memory.BlockIsRAM [0xc04] = !mp.blockisrom;

		Memory.Map [0xc06] = mp.data;
		Memory.BlockIsROM [0xc06] = mp.blockisrom;
		Memory.BlockIsRAM [0xc06] = !mp.blockisrom;

		Memory.Map [0xc07] = mp.data;
		Memory.BlockIsROM [0xc07] = mp.blockisrom;
		Memory.BlockIsRAM [0xc07] = !mp.blockisrom;
	}


	bool8 map_changed_D = (bsxmmc_p[0x0d] != bsxmmc[0x0d] ||
						   map_changed_1_2);

	if(map_changed_D) {
		bsxmmc_p[0x0d] = bsxmmc[0x0d];
		if(!map_changed_1_2) {
			get_bsx_memmap_params(&mp, (bsxmmc[1] & 0x80) != 0);
			bsx_rom = mp.data;
			bsx_rom_hirom = mp.hirom;
		}
		if(bsxmmc[0x0d] & 0x80) {
			Memory.Map [0xc0f] = (uint8 *) CMemory::MAP_BSX;
			Memory.BlockIsROM [0xc0f] = FALSE;
			Memory.BlockIsRAM [0xc0f] = FALSE;
		}
		else {
			Memory.Map [0xc0f] = mp.data - mp.offset_sub;
			Memory.BlockIsROM [0xc0f] = mp.blockisrom;
			Memory.BlockIsRAM [0xc0f] = !mp.blockisrom;
		}
		Memory.Map [0xc08] = mp.data - mp.offset_sub;
		Memory.BlockIsROM [0xc08] = mp.blockisrom;
		Memory.BlockIsRAM [0xc08] = !mp.blockisrom;

		Memory.Map [0xc09] = mp.data - mp.offset_sub;
		Memory.BlockIsROM [0xc09] = mp.blockisrom;
		Memory.BlockIsRAM [0xc09] = !mp.blockisrom;

		Memory.Map [0xc0a] = mp.data - mp.offset_sub;
		Memory.BlockIsROM [0xc0a] = mp.blockisrom;
		Memory.BlockIsRAM [0xc0a] = !mp.blockisrom;

		Memory.Map [0xc0b] = mp.data - mp.offset_sub;
		Memory.BlockIsROM [0xc0b] = mp.blockisrom;
		Memory.BlockIsRAM [0xc0b] = !mp.blockisrom;

		Memory.Map [0xc0c] = mp.data - mp.offset_sub;
		Memory.BlockIsROM [0xc0c] = mp.blockisrom;
		Memory.BlockIsRAM [0xc0c] = !mp.blockisrom;

		Memory.Map [0xc0d] = mp.data - mp.offset_sub;
		Memory.BlockIsROM [0xc0d] = mp.blockisrom;
		Memory.BlockIsRAM [0xc0d] = !mp.blockisrom;

		Memory.Map [0xc0e] = mp.data - mp.offset_sub;
		Memory.BlockIsROM [0xc0e] = mp.blockisrom;
		Memory.BlockIsRAM [0xc0e] = !mp.blockisrom;
	}

	bool8 map_changed_5 = (bsxmmc_p[2] != bsxmmc[2] || /* HiROM or LoROM */
						   ((!bsxmmc_p[5] || (bsxmmc_p[1] & 0x80)) && !(!bsxmmc[5] || (bsxmmc[1] & 0x80))) ||
						   (!(!bsxmmc_p[5] || (bsxmmc_p[1] & 0x80)) && (!bsxmmc[5] || (bsxmmc[1] & 0x80))));/* PSRAM or Card */

	if(map_changed_5) {
		get_bsx_memmap_params(&mp, (!bsxmmc[5] || (bsxmmc[1] & 0x80)));
		BSXMMC_40_4F(&mp);
	}
	bsxmmc_p[5] = bsxmmc[5];


	bool8 map_changed_6 = (bsxmmc_p[2] != bsxmmc[2] || /* HiROM or LoROM */
						   ((!bsxmmc_p[6] || (bsxmmc_p[1] & 0x80)) && !(!bsxmmc[6] || (bsxmmc[1] & 0x80))) ||
						   (!(!bsxmmc_p[6] || (bsxmmc_p[1] & 0x80)) && (!bsxmmc[6] || (bsxmmc[1] & 0x80))));/* PSRAM or Card */

	if(map_changed_6) {
		get_bsx_memmap_params(&mp, (!bsxmmc[6] || (bsxmmc[1] & 0x80)));
		BSXMMC_50_5F(&mp);
	}
	bsxmmc_p[6] = bsxmmc[6];

	//store previos states.
	if(map_changed_1_2) {
		bsxmmc_p[1] = bsxmmc[1];
		bsxmmc_p[2] = bsxmmc[2];
	}

	Memory.WriteProtectROM ();
	bsxflash0000[0] = 0x00;

	if(map_changed_1_2 ||
	   map_changed_3 ||
	   map_changed_5 ||
	   map_changed_6 ||
	   map_changed_7 ||
	   map_changed_8 ||
	   map_changed_C ||
	   map_changed_D) {
#if defined(CPU_SHUTDOWN)
		CPU.WaitAddress = NULL;
		CPU.WaitCounter = 1;
#endif
	}
}

