/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *
 * (c) Copyright 1996 - 2001 Gary Henderson (gary@daniver.demon.co.uk) and
 *                           Jerremy Koot (jkoot@snes9x.com)
 *
 * Super FX C emulator code 
 * (c) Copyright 1997 - 1999 Ivar (Ivar@snes9x.com) and
 *                           Gary Henderson.
 * Super FX assembler emulator code (c) Copyright 1998 zsKnight and _Demo_.
 *
 * DSP1 emulator code (c) Copyright 1998 Ivar, _Demo_ and Gary Henderson.
 * C4 asm and some C emulation code (c) Copyright 2000 zsKnight and _Demo_.
 * C4 C code (c) Copyright 2001 Gary Henderson (gary@daniver.demon.co.uk).
 *
 * DOS port code contains the works of other authors. See headers in
 * individual files.
 *
 * Snes9x homepage: www.snes9x.com
 *
 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */
#ifdef __DJGPP
#include <allegro.h>
#undef TRUE
#endif

#include "snes9x.h"
#include "spc700.h"
#include "apu.h"
#include "soundux.h"
#include "snesapu.h"
#include "cpuexec.h"

/* For note-triggered SPC dump support */
#include "snapshot.h"

//#define APU_DEBUG
#ifdef APU_DEBUG
static const char apu_log_name[] = "apu.log";
static FILE *apu_fs = NULL;
extern FILE *trace;
#endif

#ifdef __UOSNES__
extern const char *S9xGetSPCFilename ();
#else
extern "C" {const char *S9xGetFilenameInc (const char *);}
#endif

int spc_is_dumping=0;
int spc_is_dumping_temp;
uint8 spc_dump_dsp[0x100]; 

extern int NoiseFreq [32];
#ifdef DEBUGGER
void S9xTraceSoundDSP (const char *s, int i1 = 0, int i2 = 0, int i3 = 0,
		       int i4 = 0, int i5 = 0, int i6 = 0, int i7 = 0);
#endif

bool8 S9xInitAPU ()
{
    IAPU.RAM = (uint8 *) malloc (0x10000);
    
    if (!IAPU.RAM)
    {
		S9xDeinitAPU ();
		return (FALSE);
    }

	memset(IAPU.RAM, 0, 0x10000);

	//SNESAPU
    InitAPUDSP();

    return (TRUE);
}

void S9xDeinitAPU ()
{
    if (IAPU.RAM)
    {
		free ((char *) IAPU.RAM);
		IAPU.RAM = NULL;
    }
}

EXTERN_C uint8 APUROM [64];

void S9xResetAPU ()
{
    int i;

    Settings.APUEnabled = Settings.NextAPUEnabled;
    if(SNESGameFixes.APU_Disable) Settings.APUEnabled = FALSE;

	ZeroMemory(spc_dump_dsp, 0x100);

	//ZeroMemory(IAPU.RAM, 0x100);
    memset (IAPU.RAM, Settings.APURAMInitialValueMaster, 0x100);

	memset(IAPU.RAM+0x20, 0xFF, 0x20);
	memset(IAPU.RAM+0x60, 0xFF, 0x20);
	memset(IAPU.RAM+0xA0, 0xFF, 0x20);
	memset(IAPU.RAM+0xE0, 0xFF, 0x20);

	for(i=1;i<256;i++)
	{
		memcpy(IAPU.RAM+(i<<8), IAPU.RAM, 0x100);
	}
	
    ZeroMemory (APU.OutPorts, 4);
    IAPU.DirectPage = IAPU.RAM;
#if 0
    memmove (APU.ExtraRAM, APUROM, sizeof (APUROM));
#else
	//SNESAPU ResetSPC
    memmove (APU.ExtraRAM, &IAPU.RAM [0xffc0], sizeof (APUROM));
#endif
    memmove (&IAPU.RAM [0xffc0], APUROM, sizeof (APUROM));
    IAPU.PC = IAPU.RAM + IAPU.RAM [0xfffe] + (IAPU.RAM [0xffff] << 8);

    APU.Cycles = CPU.Cycles;

    APURegisters.YA.W = 0;
    APURegisters.X = 0;
    APURegisters.S = 0xff;
    APURegisters.P = 0;

    S9xAPUUnpackStatus ();

    APURegisters.PC = 0;
    IAPU.APUExecuting = Settings.APUEnabled;

#ifdef SPC700_SHUTDOWN
    IAPU.WaitAddress1 = NULL;
    IAPU.WaitAddress2 = NULL;
    IAPU.WaitCounter = 1;
#endif

    EXT.NextAPUTimerPos = APU.Cycles;
	EXT.APUTimerCounter_err = 0;

	S9xSetAPUTimerCounter();

    EXT.t64Cnt &= ~7;

    EXT.NextAPUTimerPos += ((EXT.APUTimerCounter_err += EXT.APUTimerCounter) >> FIXED_POINT_SHIFT);
	EXT.APUTimerCounter_err &= FIXED_POINT_REMAINDER;

    APU.ShowROM = TRUE;
    IAPU.RAM [0xf1] = 0x80;
		
    for (i = 0; i < 3; i++)
    {
		APU.TimerEnabled [i] = FALSE;
		APU.TimerValueWritten [i] = 0;
		APU.TimerTarget [i] = 0;
		APU.Timer [i] = 0;
    }
    for (int j = 0; j < 0x80; j++)
		APU.DSP [j] = 0;
	
    IAPU.TwoCycles = IAPU.OneCycle * 2;
	
    for (i = 0; i < 256; i++)
		S9xAPUCycles [i] = S9xAPUCycleLengths [i] * IAPU.OneCycle;
	
    APU.DSP [APU_ENDX] = 0;
    APU.DSP [APU_KOFF] = 0;
    APU.DSP [APU_KON] = 0;
    //APU.DSP [APU_FLG] = APU_MUTE | APU_ECHO_DISABLED;
	APU.DSP [APU_FLG] = APU_SOFT_RESET | APU_MUTE;
	//APU.DSP [APU_C0] = 0x7f;
	APU.KeyedChannels = 0;
	
    S9xResetSound (TRUE);
    S9xSetEchoEnable (0);

	//SNESAPU
	ResetAPUDSP();
}

void S9xSetAPUDSP (uint8 byte)
{
#ifdef APU_DEBUG
	if(Options.toggle && !apu_fs)
		apu_fs = fopen(apu_log_name, "a");
	else if(!Options.toggle && apu_fs) {
		fclose(apu_fs);
		apu_fs = NULL;
	}
	if(apu_fs)
		fprintf(apu_fs,
				"Address %02X w Byte %02X\n",
				(unsigned int)IAPU.RAM [0xf2], (unsigned int)byte);
#endif
	APUDSPIn(IAPU.RAM [0xf2], byte);
}

void S9xSetAPUControl (uint8 byte)
{
	//if (byte & 0x40)
	//printf ("*** Special SPC700 timing enabled\n");
    if ((byte & 1) != 0 && !APU.TimerEnabled [0])
    {
		APU.Timer [0] = 0;
		IAPU.RAM [0xfd] = 0;
		if ((APU.TimerTarget [0] = IAPU.RAM [0xfa]) == 0)
			APU.TimerTarget [0] = 0x100;
    }
    if ((byte & 2) != 0 && !APU.TimerEnabled [1])
    {
		APU.Timer [1] = 0;
		IAPU.RAM [0xfe] = 0;
		if ((APU.TimerTarget [1] = IAPU.RAM [0xfb]) == 0)
			APU.TimerTarget [1] = 0x100;
    }
    if ((byte & 4) != 0 && !APU.TimerEnabled [2])
    {
		APU.Timer [2] = 0;
		IAPU.RAM [0xff] = 0;
		if ((APU.TimerTarget [2] = IAPU.RAM [0xfc]) == 0)
			APU.TimerTarget [2] = 0x100;
    }
    APU.TimerEnabled [0] = byte & 1;
    APU.TimerEnabled [1] = (byte & 2) >> 1;
    APU.TimerEnabled [2] = (byte & 4) >> 2;
	
    if (byte & 0x10)
		IAPU.RAM [0xF4] = IAPU.RAM [0xF5] = 0;
	
    if (byte & 0x20)
		IAPU.RAM [0xF6] = IAPU.RAM [0xF7] = 0;
	
    if (byte & 0x80)
    {
		if (!APU.ShowROM)
		{
			memmove (&IAPU.RAM [0xffc0], APUROM, sizeof (APUROM));
			APU.ShowROM = TRUE;
		}
    }
    else
    {
		if (APU.ShowROM)
		{
			APU.ShowROM = FALSE;
			memmove (&IAPU.RAM [0xffc0], APU.ExtraRAM, sizeof (APUROM));
		}
    }
    IAPU.RAM [0xf1] = byte;
}

void S9xSetAPUTimer (uint16 Address, uint8 byte)
{
    IAPU.RAM [Address] = byte;
	
    switch (Address)
    {
    case 0xfa:
		if ((APU.TimerTarget [0] = IAPU.RAM [0xfa]) == 0)
			APU.TimerTarget [0] = 0x100;
		APU.TimerValueWritten [0] = TRUE;
		break;
    case 0xfb:
		if ((APU.TimerTarget [1] = IAPU.RAM [0xfb]) == 0)
			APU.TimerTarget [1] = 0x100;
		APU.TimerValueWritten [1] = TRUE;
		break;
    case 0xfc:
		if ((APU.TimerTarget [2] = IAPU.RAM [0xfc]) == 0)
			APU.TimerTarget [2] = 0x100;
		APU.TimerValueWritten [2] = TRUE;
		break;
    }
}

void S9xSetAPUTimerCounter()
{
	EXT.APUTimerCounter =
		(long)((((uint64)SNES_CYCLES_PER_SECOND << FIXED_POINT_SHIFT) * Settings.H_Max_normal)
			   / (SNES_CYCLES_PER_SCANLINE * 64000));
	return;
}

uint8 S9xGetAPUDSP ()
{
	// Writes to 80-FFh have no effect (reads are mirrored from lower mem)
	uint8 reg = IAPU.RAM [0xf2] & 0x7f;
	uint8 byte;

	switch (reg) {
	case APU_ENDX:
		FLUSH_SAMPLES();
		byte = APU.DSP [reg];
		for(int i = 0; i < 8; i++) {
			if((mix[i].mFlg & MFLG_SSRC) || mix[i].bStart != mix[i].bMixStart)
				byte &= ~(1 << i);
		}
		break;

	default:
		switch (reg & 0x0f) {
		case APU_OUTX:
			{
				int i = reg >> 4;
				if(!(mix[i].mFlg & MFLG_OFF)) {
					FLUSH_SAMPLES();
					byte = (uint8)((mix[i].mFlg & (MFLG_OFF | MFLG_END)) ? 0x00 : (mix[i].mOut >> 8));
				}
				else
					byte = 0x00;
			}
			break;
		
		case APU_ENVX:
			if(Settings.SoundEnvelopeHeightReading) {
				int i = reg >> 4;
				if(!(mix[i].mFlg & MFLG_OFF)) {
					FLUSH_SAMPLES();
					// "Fuurai no siren" needs return (0) in case MFLG_END 
					if(!(mix[i].mFlg & (MFLG_OFF | MFLG_END))) {
						int32 eVal = mix[i].eVal >> E_SHIFT;
						if(eVal > 0x7f)
							eVal = 0x7f;
						else if(eVal < 0)
							eVal = 0;
						byte = (uint8)eVal;
					}
					else
						byte = 0x00;
				}
				else
					byte = 0x00;
			}
			else
				byte = 0x00;
			break;

		default:
			byte = APU.DSP [reg];
			break;
		}
		break;
	}
#ifdef APU_DEBUG
	if(Options.toggle && !apu_fs)
		apu_fs = fopen(apu_log_name, "a");
	else if(!Options.toggle && apu_fs) {
		fclose(apu_fs);
		apu_fs = NULL;
	}
	if(apu_fs)
		fprintf(apu_fs,
				"Address %02X r Byte %02X\n",
				(unsigned int)IAPU.RAM [0xf2], (unsigned int)byte);
#endif
	return byte;
}

void sapuport_reset()
{
	
}

void sport_reset()
{
	
}

uint8 sapuport_r(int n)
{
#ifdef SPC700_SHUTDOWN
	IAPU.WaitAddress2 = IAPU.WaitAddress1;
	IAPU.WaitAddress1 = IAPU.PC;
#endif
	return IAPU.RAM [0xf4 + n];
}

void sapuport_w(int n, uint8 val)
{
	IAPU.RAM [0xf4 + n] = val;
#ifdef SPC700_SHUTDOWN
	IAPU.APUExecuting = Settings.APUEnabled;
	IAPU.WaitCounter = 1;
#endif
}

uint8 sport_r(int n)
{
#ifdef SPC700_SHUTDOWN	
	//As for the reason for which this is necessary,
	//the system of SPC700_SHUTDOWN is not perfect yet.
	IAPU.APUExecuting = Settings.APUEnabled;
	IAPU.WaitCounter = 1;
#endif
	return APU.OutPorts [n];
}

void sport_w(int n, uint8 val)
{
	APU.OutPorts [n] = val;
}

