/*
  Copyright (C) 2004 sanmaiwashi
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef HAVE_IO_H
#include <io.h>
#elif HAVE_SYS_IO_H
#include <sys/io.h>
#endif
#include <unistd.h>
#include <sys/stat.h>

#include "cfgfile.h"

#define PATH_BUFFER_LENGTH 1025

/************************************
    internal variables
************************************/
static char *config_data = NULL;
static int config_data_size = 0;
static int config_data_memory_size = 0;
static void (*exit_configfile_func)() = NULL;
static u8 exit_func_added = FALSE;
static char configfilename[PATH_BUFFER_LENGTH] = {0};
static char nullchar = 0;

/************************************
  null function & global variables   
************************************/
u8 configfile_cannot_create = TRUE;
u8 configfile_no_section_and_name = FALSE;
u8 configfile_no_value = FALSE;

#define CONFIG_FLAG_CLR() \
configfile_no_section_and_name = FALSE;\
configfile_no_value = FALSE;

static char *get_config_string0(const char *section, const char *name)
{
	CONFIG_FLAG_CLR();
	return &nullchar;
}

static u8 set_config_string0(const char *section, const char *name, char *value)
{
	CONFIG_FLAG_CLR();
	return FALSE;
}

static int get_config_int0(int defaultreturnvalue, const char *section, const char *name)
{
	CONFIG_FLAG_CLR();
	return defaultreturnvalue;
}

static u8 set_config_int0(const char *section, const char *name, int value)
{
	CONFIG_FLAG_CLR();
	return FALSE;
}

static void exit_configfile0()
{
	CONFIG_FLAG_CLR();
	return;
}

static u8 flush_configfile0()
{
	CONFIG_FLAG_CLR();
	return FALSE;
}

char *(*get_config_string)(const char *section, const char *name) = get_config_string0;
u8 (*set_config_string)(const char *section, const char *name, char *value) = set_config_string0;
int (*get_config_int)(int defaultreturnvalue, const char *section, const char *name) = get_config_int0;
u8 (*set_config_int)(const char *section, const char *name, int value) = set_config_int0;
void (*exit_configfile)() = exit_configfile0;
u8 (*flush_configfile)() = flush_configfile0;

static u8 fflush_configfile()
{
	FILE *fp;
	CONFIG_FLAG_CLR();
	if(!configfile_cannot_create && config_data && (fp = fopen((const char *)configfilename, "w"))) {
		fprintf(fp, config_data);
		fclose(fp);
		return TRUE;
	}
	return FALSE;
}

static void eexit_configfile()
{
	CONFIG_FLAG_CLR();
	if(exit_configfile_func) {
		exit_configfile_func();
		exit_configfile_func = NULL;
	}
	fflush_configfile();
	if(config_data) {
		free(config_data);
		config_data = NULL;
		config_data_size = 0;
		config_data_memory_size = 0;
	}
	return;
}

static char *get_config_value_ptr(const char *section, const char *name)
{
	char *section_start;
	char *section_end;
	char section_string[256];
	char name_string[256];
	int section_string_len;
	int name_string_len;
	char *tmpcharptr;
	
	section_string_len = strlen(section) + 4;
	name_string_len = strlen(name) + 2;
	if(255 < section_string_len || 255 < name_string_len) {
		configfile_no_section_and_name = TRUE;
		return NULL;
	}
	/* create "[section]" */
	section_string[0] = '\n';
	section_string[1] = '[';
	strcpy(&section_string[2], section);
	strcat(section_string, "]\n");

	/* create "name=" */
	name_string[0] = '\n';
	strcpy(&name_string[1], name);
	strcat(name_string, "=");

	/* search [section] */
	tmpcharptr = strstr(config_data, section_string);
	if(!tmpcharptr) {
		configfile_no_section_and_name = TRUE;
		return NULL;
	}
	section_start = tmpcharptr + section_string_len - 1;
	/* search "\n[section]" or end of file */
	tmpcharptr = strstr(section_start, "\n[");
	if(!tmpcharptr)
		section_end = config_data + strlen(config_data) - 1;
	else
		section_end = tmpcharptr;
	/* search [section] "name=" */
	tmpcharptr = strstr(section_start, name_string);
	if(!tmpcharptr || section_end < tmpcharptr) {
		configfile_no_section_and_name = TRUE;
		return NULL;
	}
	return (tmpcharptr + name_string_len);
}

static char *gget_config_string(const char *section, const char *name)
{
	static char string[PATH_BUFFER_LENGTH];
	int strlength;
	char *tmpcharptr;
	CONFIG_FLAG_CLR();
	if(!(tmpcharptr = get_config_value_ptr(section, name))) {
		configfile_no_value = TRUE;
		return &nullchar;
	}
	for(strlength = 0;
		strlength < PATH_BUFFER_LENGTH && tmpcharptr[strlength] != '\n' && tmpcharptr[strlength] != 0;
		strlength++);
	if(strlength == 0 || strlength >= PATH_BUFFER_LENGTH) {
		configfile_no_value = TRUE;
		return &nullchar;
	}
	memcpy(string, tmpcharptr, strlength);
	string[strlength] = 0;
	return string;
}

static int gget_config_int(int defaultreturnvalue, const char *section, const char *name)
{
	char *tmpcharptr;
	CONFIG_FLAG_CLR();
	tmpcharptr = get_config_string(section, name);
	if(!(*tmpcharptr) || !(isdigit(*tmpcharptr))) {
		configfile_no_value = TRUE;
		return defaultreturnvalue;
	}
	return (atoi(tmpcharptr));
}

static u8 sset_config_string(const char *section, const char *name, char *value)
{
	int strlength;
	int old_string_length;
	int new_string_length;
	int move_back_size;
	int string_ofs;
	int newsize;
	char *tmpcharptr;
	char *newptr;

	CONFIG_FLAG_CLR();
	if(!(tmpcharptr = get_config_value_ptr(section, name)))
		return FALSE;
	for(strlength = 0; tmpcharptr[strlength] != '\n' && tmpcharptr[strlength] != 0;
		strlength++);
	old_string_length = strlength;
	new_string_length = strlen(value);
	string_ofs = tmpcharptr - config_data;
	move_back_size = config_data_size - (string_ofs + old_string_length);
	if(new_string_length) {
		if(old_string_length == new_string_length) {
			memcpy(tmpcharptr, value, (size_t)new_string_length);
		}
		else if(old_string_length  > new_string_length) {
			memmove(tmpcharptr + new_string_length, tmpcharptr + old_string_length, move_back_size);
			config_data_size = (config_data_size + new_string_length - old_string_length);
			*(config_data + config_data_size) = 0x00;
			memcpy(tmpcharptr, value, (size_t)new_string_length);
		}
		//old_string_length  < new_string_length
		else {
			if((config_data_size - old_string_length + new_string_length) > (config_data_memory_size - 1)) {
				newsize = (((config_data_size - old_string_length + new_string_length) / 0x1000) + 1) * 0x1000 + 1;
				if(!(newptr = (char *)calloc((unsigned int)newsize, (size_t)sizeof(char))))
					return FALSE;
				config_data_memory_size = newsize;
				memcpy(newptr, config_data, (size_t)string_ofs);
				memcpy(newptr + string_ofs + new_string_length,
					   config_data + string_ofs + old_string_length,
					   (size_t)move_back_size);
				tmpcharptr = newptr + string_ofs;
				free(config_data);
				config_data = newptr;
			}
			else {
				memmove(tmpcharptr + new_string_length, tmpcharptr + old_string_length, move_back_size);
			}
			config_data_size = (config_data_size + new_string_length - old_string_length);
			*(config_data + config_data_size) = 0x00;
			memcpy(tmpcharptr, value, (size_t)new_string_length);
		}
	}
	else {
		memmove(tmpcharptr, tmpcharptr + old_string_length, move_back_size);
		config_data_size = (config_data_size - old_string_length);
		*(config_data + config_data_size) = 0x00;
	}
	return TRUE;
}

static u8 sset_config_int(const char *section, const char *name, int value)
{
	char intstr[20];
	CONFIG_FLAG_CLR();
	sprintf(intstr, "%d", value);
	return (sset_config_string(section, name, intstr));
}

static void enable_functions()
{
	get_config_string = gget_config_string;
	set_config_string = sset_config_string;
	get_config_int = gget_config_int;
	set_config_int = sset_config_int;
	exit_configfile = eexit_configfile;
	flush_configfile = fflush_configfile;
	return;
}

static void disable_functions()
{
	get_config_string = get_config_string0;
	set_config_string = set_config_string0;
	get_config_int = get_config_int0;
	set_config_int = set_config_int0;
	exit_configfile = exit_configfile0;
	flush_configfile = flush_configfile0;
	return;
}

u8 load_configfile(char *filename, const char *ctemplate, void (*exitfunc)())
{
	FILE *fp;
	struct stat Stat;
	u8 newfile = FALSE;
	CONFIG_FLAG_CLR();
	configfile_cannot_create = TRUE;
	disable_functions();
	
	if(config_data) {
		free(config_data);
		config_data = NULL;
		config_data_size = 0;
		config_data_memory_size = 0;
	}
	
	if(!filename || strlen(filename) >= PATH_BUFFER_LENGTH)
		return FALSE;
	
	/* create new file */
	if(access(filename, 0)) {
		newfile = TRUE;
		if(!ctemplate)
			return FALSE;
		if(!(fp = fopen(filename, "w"))) {
			return FALSE;
		}
		fprintf(fp, ctemplate);
		fclose(fp);
	}
	
	strcpy(configfilename, filename);
	/* get filesize */
	stat(filename, &Stat);
	/* set configdata memory size */
	config_data_size = Stat.st_size;
	config_data_memory_size = ((config_data_size / 0x1000) + 1) * 0x1000 + 1;

	if(!(fp = fopen(filename, "r")))
		return FALSE;
	if(!(config_data = (char *)calloc((unsigned int)config_data_memory_size, (size_t)sizeof(char)))) {
		fclose(fp);
		return FALSE;
	}
	config_data_size = fread(config_data, sizeof(char), config_data_size, fp);
	fclose(fp);
	configfile_cannot_create = FALSE;
	exit_configfile_func = exitfunc;
	enable_functions();
	
	if(!exit_func_added) {
		atexit((void (*)())exit_configfile);
		exit_func_added = TRUE;
	}
	return (newfile ? FALSE : TRUE);
}

