#include <string.h>
#include <stdlib.h>
#include "bmpfutil.h"

static int palcompare(const void *x, const void *y)
{
	u32 xv = READ_32(x);
	u32 yv = READ_32(y);
	if(xv > yv)
		return 1;
	if(xv < yv)
		return -1;
	return 0;
}

static inline u8 Convert24bitDIBtoPalette(u8 *dst, u8 *dib24, u32 *palnum,
											 u32 width, u32 height)
{
	u32 dbxy, pali, dbi, i;
	u8 tmppalpix[4];
	
	if(!dst || !dib24 || !palnum)
		return FALSE;
	
	tmppalpix[3] = 0;
	dbxy = width * height;
	dbi = 0;
	
	for(pali = 0; pali < 256 && dbi < dbxy; dbi++) {
		tmppalpix [0] = *(dib24 + dbi * 3 + 0);
		tmppalpix [1] = *(dib24 + dbi * 3 + 1);
		tmppalpix [2] = *(dib24 + dbi * 3 + 2);
		for(i = 0; i <= pali; i++) {
			if(i == pali) {
				WRITE_32(dst + i * 4, READ_32(&tmppalpix[0]));
				pali++;
				break;
			}
			if(*(u32 *)(dst + i * 4) == *(u32 *)(&tmppalpix[0])) {
				break;
			}
		}
	}
	
	if(pali >= 256 && dbi < dbxy) {
		*palnum = 0xFFFFFFFF;
		return FALSE;
	}
	else if(pali < 256) {
		memset(dst + pali * 4, 0, (256 - pali) * 4);
	}
	*palnum = pali;
	//sort..
	qsort(dst, (size_t)pali, 4, palcompare);
	return TRUE;
}

static inline u8 Convert24bitDIBto8bitDIB(u8 *dst, u8 *dib24, u8 *palette,
											 u32 width, u32 height, u32 palnum)
{
	u32 dbxy, dbi;
	u8 tmppalpix[4];
	u8 *ppal;
	
	if(!dst || !dib24 || !palette)
		return FALSE;
	
	tmppalpix[3] = 0;
	dbxy = width * height;
	
	for(dbi = 0; dbi < dbxy; dbi++) {
		tmppalpix [0] = *(dib24 + dbi * 3);
		tmppalpix [1] = *(dib24 + dbi * 3 + 1);
		tmppalpix [2] = *(dib24 + dbi * 3 + 2);
		ppal = (u8 *)bsearch(tmppalpix, palette, (size_t)palnum, (size_t)4, palcompare);
		if(ppal)
			*(dst + dbi) = (u8)((u32)(ppal - palette) / 4);
	}
	return TRUE;
}

u8 ConvertBitmapImage24To8IfWithin(u8 *dst, u8 *src)
{
	u32 bmfWidth, bmfHeight, palnum;
	
	/* Check 24 bit bitmap file header. */
	if(!src || !dst ||
		READ_16(src + BF_TYPE) != 0x4D42 ||
		READ_16(src + BF_RESERVED1) != 0 ||
		READ_16(src + BF_RESERVED2) != 0 ||
		READ_32(src + BI_SIZE) != BITMAP_INFO_SIZE ||
		READ_16(src + BI_PLANES) != 1 ||
		READ_32(src + BI_COMPRESSION) != 0 ||
		READ_32(src + BI_XPELSPERMETER) != 0 ||
		READ_32(src + BI_YPELSPERMETER) != 0 ||
		!(READ_32(src + BF_OFFBITS) == BITMAP_HEADER_SIZE && READ_16(src + BI_BITCOUNT) == 24 &&
		READ_32(src + BI_CLRUSED) == 0 && READ_32(src + BI_CLRIMPORTANT) == 0))
		return FALSE;
	
	bmfWidth = READ_32(src + BI_WIDTH);
	bmfHeight = READ_32(src + BI_HEIGHT);
	
	WRITE_16(dst + BF_TYPE, 0x4D42);
	WRITE_32(dst + BF_SIZE, (u32)BITMAP_HEADER_SIZE + 1024 + bmfWidth * bmfHeight);
	WRITE_16(dst + BF_RESERVED1, 0);
	WRITE_16(dst + BF_RESERVED2, 0);
	WRITE_32(dst + BF_OFFBITS, (u32)BITMAP_HEADER_SIZE + 1024);
	
	WRITE_32(dst + BI_SIZE, BITMAP_INFO_SIZE);
	WRITE_32(dst + BI_WIDTH, bmfWidth);
	WRITE_32(dst + BI_HEIGHT, bmfHeight);
	WRITE_16(dst + BI_PLANES, 1);
	WRITE_16(dst + BI_BITCOUNT, 8);
	WRITE_32(dst + BI_COMPRESSION, 0);
	WRITE_32(dst + BI_SIZEIMAGE, bmfWidth * bmfHeight);
	WRITE_32(dst + BI_XPELSPERMETER, 0);
	WRITE_32(dst + BI_YPELSPERMETER, 0);
	WRITE_32(dst + BI_CLRUSED, 256);
	WRITE_32(dst + BI_CLRIMPORTANT, 256);
	
	if(!Convert24bitDIBtoPalette(dst + BITMAP_HEADER_SIZE,
								 src + BITMAP_HEADER_SIZE,
								 &palnum, bmfWidth, bmfHeight) ||
		!Convert24bitDIBto8bitDIB(dst + BITMAP_HEADER_SIZE + 1024,
								  src + BITMAP_HEADER_SIZE,
								  dst + BITMAP_HEADER_SIZE, bmfWidth, bmfHeight, palnum))
		return FALSE;
	
	return TRUE;
}

/* not supported yet. */
u8 ConvertBitmapImage24To8(u8 *dst, u8 *src)
{
	return FALSE;
}

u8 x2wBitmapImage(u8 *dst, u8 *src)
{
	u32 srci, dsti, srcbmpfsize;
	
	/* Check bitmap file header. */
	if(!src || !dst || !CheckBitmapHeader(src))
		return FALSE;
	
	//copy bitmap header.
	memcpy(dst, src, (size_t)READ_32(src + BF_OFFBITS));
	
	WRITE_32(dst + BF_SIZE, READ_32(src + BF_SIZE) + READ_32(src + BI_SIZEIMAGE));
	WRITE_32(dst + BI_SIZEIMAGE, READ_32(src + BI_SIZEIMAGE) << 1);
	WRITE_32(dst + BI_WIDTH, READ_32(src + BI_WIDTH) << 1);
	
	srci = READ_32(src + BF_OFFBITS);
	dsti = srci;
	srcbmpfsize = READ_32(src + BF_SIZE);
	
	if(READ_16(src + BI_BITCOUNT) == 24) {
		for(; srci < srcbmpfsize; srci += 3, dsti += 6) {
			*(dst + dsti + 0) = *(src + srci + 0);
			*(dst + dsti + 1) = *(src + srci + 1);
			*(dst + dsti + 2) = *(src + srci + 2);
			*(dst + dsti + 3) = *(src + srci + 0);
			*(dst + dsti + 4) = *(src + srci + 1);
			*(dst + dsti + 5) = *(src + srci + 2);
		}
	}
	else {
		for(; srci < srcbmpfsize; srci += 1, dsti += 2) {
			*(dst + dsti + 0) = *(src + srci);
			*(dst + dsti + 1) = *(src + srci);
		}
	}
	return TRUE;
}

u8 x2hBitmapImage(u8 *dst, u8 *src)
{
	u32 srci, dsti, srcbmpfsize, srcdibpitch, srcdibpitchx2;
	
	/* Check bitmap file header. */
	if(!src || !dst || !CheckBitmapHeader(src))
		return FALSE;
	
	//copy bitmap header.
	memcpy(dst, src, (size_t)READ_32(src + BF_OFFBITS));
	
	WRITE_32(dst + BF_SIZE, READ_32(src + BF_SIZE) + READ_32(src + BI_SIZEIMAGE));
	WRITE_32(dst + BI_SIZEIMAGE, READ_32(src + BI_SIZEIMAGE) << 1);
	WRITE_32(dst + BI_HEIGHT, READ_32(src + BI_HEIGHT) << 1);
	
	srci = READ_32(src + BF_OFFBITS);
	dsti = srci;
	srcbmpfsize = READ_32(src + BF_SIZE);
	srcdibpitch = READ_32(src + BI_SIZEIMAGE) / READ_32(src + BI_HEIGHT);
	srcdibpitchx2 = srcdibpitch << 1;
	
	for(; srci < srcbmpfsize; srci += srcdibpitch, dsti += srcdibpitchx2) {
		memcpy(dst + dsti, src + srci, srcdibpitch);
		memcpy(dst + dsti + srcdibpitch, src + srci, srcdibpitch);
	}
	
	return TRUE;
}


/**********************
  RLE compression code
***********************/

#define BI_RLE8       1
#define BI_RLE4       2

static u32 EncodeImage(u32 height, u32 pitch,
	const u8 *pixels, u8 *compressed_pixels)
{
	u32 y;
	register const u8 *p;
	register u32 i, x;
	register u8 *q;
	
	/*
	Runlength encode pixels.
	*/
	p = pixels;
	q = compressed_pixels;
	i = 0;
	for (y = 0; y < height; y++)
	{
		for (x = 0; x < pitch; x += i)
		{
			/*
			Determine runlength.
			*/
			for (i = 1; ((x + i) < pitch); i++)
				if ((*(p + i) != *p) || (i == 255))
					break;
			*q++ = (u8)i;
			*q++ = (*p);
			p += i;
		}
		/*
		End of line.
		*/
		*q++ = 0x00;
		*q++ = 0x00;
	}
	/*
	End of bitmap.
	*/
	*q++ = 0;
	*q++ = 0x01;
	return (u32)(q - compressed_pixels);
}

u8 bmp_compress_dib_rle(u8 *dst, u8 *src)
{
	u32 data_size;
	u8 *new_dib;
	u8 *RLEsrc, *RLEdest;
	u32 row_bytes, num_palette;
	
	if(READ_32(src + BI_COMPRESSION) != 0)
		return FALSE;
	
	if(READ_16(src + BI_BITCOUNT) != 8)
		return FALSE;
	
	row_bytes = READ_32(src + BI_WIDTH) * READ_16(src + BI_BITCOUNT) / 8;
	num_palette = READ_32(src + BI_CLRUSED);
	
	if(num_palette == 0)
		num_palette = (1 << READ_16(src + BI_BITCOUNT));
	
	new_dib = dst + BITMAP_FILE_HEADER_SIZE;
	
	memcpy(new_dib, src + BITMAP_FILE_HEADER_SIZE, BITMAP_INFO_SIZE + num_palette * 4);
	
	RLEsrc = src + BITMAP_HEADER_SIZE + num_palette * 4;
	RLEdest = new_dib + BITMAP_INFO_SIZE + num_palette * 4;

	WRITE_32(new_dib + BI_COMPRESSION - BITMAP_FILE_HEADER_SIZE, BI_RLE8);

	data_size = EncodeImage(READ_32(src + BI_HEIGHT),
							row_bytes, (const u8 *)RLEsrc, RLEdest);

	WRITE_32(new_dib + BI_CLRUSED - BITMAP_FILE_HEADER_SIZE, num_palette);
	WRITE_32(new_dib + BI_SIZEIMAGE - BITMAP_FILE_HEADER_SIZE, data_size);

	memcpy(dst, src, BITMAP_FILE_HEADER_SIZE);
	WRITE_32(dst + BF_SIZE, BITMAP_HEADER_SIZE + 1024 + data_size);
	
	return TRUE;
}

