#include <string.h>
#include "bmpfutil.h"

#undef EOF
#define EOF (-1)

/*****************************************************************************
 *
 * convert bmp2gif
 *                  return  pGIFfileimagedata, ImageDataSize
 *****************************************************************************/

static void *(*MemAllocFunc)(u32 bytecount);
static void (*MemFreeFunc)(void *memptr);

static u32 realsize = 0;

static void realsizeinit(u32 initsize)
{
	realsize = initsize;
}

static u8 cmap[256*4];	/* Output colormap */
static u8 *pbmpf = NULL;
static u32 pbmpf_width;
static u32 pbmpf_height;

static u8 *pwdata = NULL;
static u32 wdatasize = 0;

typedef u8 (*ifunptr)(int, int);

static u8 gifGetPixel(int x, int y)
{
	return *(pbmpf + BITMAP_DATA_OFFSET + 1024 + pbmpf_width * (pbmpf_height - 1 - y) + x);
}

static u8 *GetpGIFdataAndSize(u32 *ImageDataSize)
{
	*ImageDataSize = wdatasize;
	return pwdata;
}

static void formoutputcmap(u8 *pBitmapImage){
	int i;
	for(i = 0; i < 256; i++) {
		cmap[i * 3 + 0]= *(pBitmapImage + BITMAP_DATA_OFFSET + 4 * i + 2);
		cmap[i * 3 + 1]= *(pBitmapImage + BITMAP_DATA_OFFSET + 4 * i + 1);
		cmap[i * 3 + 2]= *(pBitmapImage + BITMAP_DATA_OFFSET + 4 * i + 0);
	}
	return;
}

static int GIFEncode(u32 GWidth, u32 GHeight,
	int GInterlace, int Background, int Transparent,
	int BitsPerPixel, u8 *gifpalette, ifunptr gifgetpixel, char *gifcomment);

u8 *bmp2gif(u8 *pBitmapImage, u32 *ImageDataSize, char *gifcomment,
	void *(*memallocfunc)(u32 bytecount), void(*memfreefunc)(void *memptr))
{
	int GInterlace, Background, Transparent, gifErr;
	
	if(!pBitmapImage || !ImageDataSize || !memallocfunc || !memfreefunc ||
		!CheckBitmapHeader(pBitmapImage) || READ_16(pBitmapImage + BI_BITCOUNT) != 8)
		return NULL;
	
	MemAllocFunc = memallocfunc;
	MemFreeFunc = memfreefunc;
	pbmpf = pBitmapImage;
	pbmpf_width = READ_32(pbmpf + BI_WIDTH);
	pbmpf_height = READ_32(pbmpf + BI_HEIGHT);
	
	formoutputcmap(pBitmapImage);
	
	GInterlace= 0;
	Background = 0;
	Transparent = -1;
	
	gifErr = GIFEncode(pbmpf_width, pbmpf_height, GInterlace, Background, Transparent,
		(int)READ_16(pBitmapImage + BI_BITCOUNT), cmap, gifGetPixel, gifcomment);
	
	if(!gifErr)
		return NULL;
	else
		return GetpGIFdataAndSize(ImageDataSize);
}

/*****************************************************************************
 *
 * GIFEncode(GHeight, GWidth, GInterlace, Background, Transparent,
 *	      BitsPerPixel, Palette, gifgetpixel, gifcomment)
 *
 *****************************************************************************/

static inline u8 mem_write_data(u8 *prdata, u32 rdatasize, u32 rdatacount)
{
	u8 *newptr;
	if((wdatasize + rdatasize * rdatacount) > realsize) {
		if(rdatasize * rdatacount <= 0x1000) {
			newptr = (u8 *)MemAllocFunc(realsize + 0x1000);
			realsize += 0x1000;
		}
		else {
			newptr = (u8 *)MemAllocFunc(realsize + rdatasize * rdatacount);
			realsize += (rdatasize * rdatacount);
		}
		if(!newptr) {
			if(pwdata)
				MemFreeFunc(pwdata);
			return FALSE;
		}
		if(pwdata) {
			memcpy(newptr, pwdata, wdatasize);
			MemFreeFunc(pwdata);
		}
		pwdata = newptr;
	}
	memcpy(pwdata + wdatasize, prdata, rdatasize * rdatacount);
	wdatasize += (rdatasize * rdatacount);
	return TRUE;
}

static u32 Width, Height;
static u32 curx, cury;
static u32 CountDown;
static int Pass = 0;
static int Interlace;

static int compress ( int init_bits, ifunptr ReadValue );
static int output ( int code );
static int cl_block ( void );
static void cl_hash ( long int hsize );
static void char_init ( void );
static int char_out ( int c );
static int flush_char ( void );


/*
 * Bump the 'curx' and 'cury' to point to the next pixel
 */
static void BumpPixel() {
	/* Bump the current X position */
	++curx;

	/*
	 * If we are at the end of a scan line, set curx back to the beginning
	 * If we are interlaced, bump the cury to the appropriate spot,
	 * otherwise, just increment it.
	 */
	if(curx == Width) {
		curx = 0;
		if(!Interlace)
			++cury;
		else {
			switch(Pass) {
			case 0:
				cury += 8;
				if(cury >= Height) {
					++Pass;
					cury = 4;
				} break;
			case 1:
				cury += 8;
				if(cury >= Height) {
					++Pass;
					cury = 2;
				} break;
			case 2:
				cury += 4;
				if(cury >= Height) {
					++Pass;
					cury = 1;
				} break;
			case 3:
				cury += 2;
				break;
			}
		}
	}
}

/*
 * Return the next pixel from the image
 */
static int GIFNextPixel(ifunptr gifgetpixel){
	int r;

	if( CountDown == 0 )
		return EOF;
	--CountDown;

	r = gifgetpixel(curx, cury);
	BumpPixel();
	return r;
}

static void gifencinit();
/* for endian.. */
static u8 worddata[2];
static u8 dworddata[4];

/* public */
static int GIFEncode(u32 GWidth, u32 GHeight,
	int GInterlace, int Background, int Transparent,
	int BitsPerPixel, u8 *gifpalette, ifunptr gifgetpixel, char *gifcomment ){

	int B;
	u32 RWidth, RHeight;
	int LeftOfs, TopOfs;
	int Resolution;
	int ColorMapSize;
	int InitCodeSize;
	int i;

	u8 putc;
	u16 putw;
	u32 putl;
	char gifcomm[0xff + 1];

	if(gifcomment)
		strncpy(gifcomm, gifcomment, 0xff);
	else
		gifcomm[0] = 0;
	gifcomm[0xff] = 0;
	gifencinit();
	
	Interlace = GInterlace;

	ColorMapSize = 1 << BitsPerPixel;

	RWidth = Width = GWidth;
	RHeight = Height = GHeight;
	LeftOfs = TopOfs = 0;

	Resolution = BitsPerPixel;

	/* Calculate number of bits we are expecting */
	CountDown = Width * Height;

	/* Indicate which pass we are on (if interlace) */
	Pass = 0;

	/* The initial code size */
	if( BitsPerPixel <= 1 )
		InitCodeSize = 2;
	else
		InitCodeSize = BitsPerPixel;

	/* Set up the current x and y position */
	curx = cury = 0;

	realsizeinit(0);

	/* Write the Magic header */
	if(!mem_write_data((u8 *)(Transparent < 0 ? "GIF87a" : "GIF89a"), 1, 6))
		return FALSE;

	/* Write out the screen width and height */
	WRITE_32(dworddata, RWidth);
	if(!mem_write_data(dworddata, 2, 1))
		return FALSE;
	WRITE_32(dworddata, RHeight);
	if(!mem_write_data(dworddata, 2, 1))
		return FALSE;

	/* Indicate that there is a global colour map */
	B = 0x80;	/* Yes, there is a color map */

	/* OR in the resolution */
	B |= (Resolution - 1) << 5;

	/* OR in the Bits per Pixel */
	B |= (BitsPerPixel - 1);

	/* Write it out */
	WRITE_16(worddata, B);
	if(!mem_write_data(worddata, 1, 1))
		return FALSE;

	/* Write out the Background colour */
	WRITE_16(worddata, Background);
	if(!mem_write_data(worddata, 1, 1))
		return FALSE;

	/* Byte of 0's (future expansion) */
	putc = 0;
	if(!mem_write_data(&putc, 1, 1))
		return FALSE;

	/* Write out the Global Colour Map */
	for( i=0; i<ColorMapSize; ++i ) {
		if(!mem_write_data((u8 *)gifpalette, 1, 3))
			return FALSE;
		// Red
		// Green
		// Blue
		gifpalette += 3;
	}

	/* Write out extension for transparent colour index, if necessary. */
	if ( Transparent >= 0 ) {
		putc = '!';
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
		putc = 0xf9;
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
		putc = 0x4;
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
		putc = 1;
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
		putw = 0;
		WRITE_16(worddata, putw);
		if(!mem_write_data(worddata, 2, 1))
			return FALSE;
		WRITE_16(worddata, Transparent);
		if(!mem_write_data(worddata, 1, 1))
			return FALSE;
		putc = 0;
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
	}

	if (gifcomm[0]){
		putc = '!';
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
		putc = 0xfe;
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
		putl = (u32)strlen(gifcomm);
		if(putl > 0xff)
			putl = 0xff;
		putc = (u8)(putl & 0xff);
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
		if(!mem_write_data((u8 *)gifcomm, 1, (u32)(putc + 1)))
			return FALSE;
	}

	/* Write an Image separator */
	putc = ',';
	if(!mem_write_data(&putc, 1, 1))
		return FALSE;

	/* Write the Image header */
	WRITE_16(worddata, LeftOfs);
	if(!mem_write_data(worddata, 2, 1))
		return FALSE;
	WRITE_16(worddata, TopOfs);
	if(!mem_write_data(worddata, 2, 1))
		return FALSE;
	WRITE_16(worddata, (u16)Width);
	if(!mem_write_data(worddata, 2, 1))
		return FALSE;
	WRITE_16(worddata, (u16)Height);
	if(!mem_write_data(worddata, 2, 1))
		return FALSE;

	/* Write out whether or not the image is interlaced */
	if( Interlace ) {
		putc = 0x40;
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
	}
	else {
		putc = 0;
		if(!mem_write_data(&putc, 1, 1))
			return FALSE;
	}

	/* Write out the initial code size */
	WRITE_16(worddata, InitCodeSize);
	if(!mem_write_data(worddata, 1, 1))
		return FALSE;

	/* Go and actually compress the data */
	if(!compress(InitCodeSize+1, gifgetpixel))
		return FALSE;

	/* Write out a Zero-length packet (to end the series) */
	putc = 0;
	if(!mem_write_data(&putc, 1, 1))
		return FALSE;

	/* Write the GIF file terminator */
	putc = ';';
	if(!mem_write_data(&putc, 1, 1))
		return FALSE;

	return TRUE;
}

/***************************************************************************
 *
 *  GIFCOMPR.C	     - GIF Image compression routines
 *
 *  Lempel-Ziv compression based on 'compress'.  GIF modifications by
 *  David Rowley (mgardi@watdcsu.waterloo.edu)
 *
 ***************************************************************************/

/* General DEFINEs  */

#define BITS	12
#define HSIZE  5003	       /* 80% occupancy */

/*
 *
 * GIF Image compression - modified 'compress'
 *
 * Based on: compress.c - File compression ala IEEE Computer, June 1984.
 *
 * By Authors:	Spencer W. Thomas	(decvax!harpo!utah-cs!utah-gr!thomas)
 *		Jim McKie		(decvax!mcvax!jim)
 *		Steve Davies		(decvax!vax135!petsd!peora!srd)
 *		Ken Turkowski		(decvax!decwrl!turtlevax!ken)
 *		James A. Woods		(decvax!ihnp4!ames!jaw)
 *		Joe Orost		(decvax!vax135!petsd!joe)
 *
 */

static int n_bits;			  /* number of bits/code */
static int maxbits = BITS;		  /* user settable max # bits/code */
static int maxcode;		  /* maximum code, given n_bits */
static int maxmaxcode = (int)1 << BITS; /* should NEVER generate this code */
#ifdef COMPATIBLE		/* But wrong! */
# define MAXCODE(n_bits)	((int) 1 << (n_bits) - 1)
#else /*COMPATIBLE*/
# define MAXCODE(n_bits)	(((int) 1 << (n_bits)) - 1)
#endif /*COMPATIBLE*/

static long int htab [HSIZE];
static u16 codetab [HSIZE];
#define HashTabOf(i)	   htab[i]
#define CodeTabOf(i)	codetab[i]

static int hsize = HSIZE;		       /* for dynamic table sizing */

/*
 * To save much memory, we overlay the table used by compress() with those
 * used by decompress().  The tab_prefix table is the same size and type
 * as the codetab.  The tab_suffix table needs 2**BITS characters.  We
 * get this from the beginning of htab.  The output stack uses the rest
 * of htab, and contains characters.  There is plenty of room for any
 * possible stack (stack used to be 8000 characters).
 */

static int free_ent = 0;		       /* first unused entry */

/*
 * block compression parameters -- after all codes are used up,
 * and compression rate changes, start over.
 */
static int clear_flg = 0;

static int offset;
static long int in_count = 1;		 /* length of input */
static long int out_count = 0;		 /* # of codes output (for debugging) */

/*
 * compress stdin to stdout
 *
 * Algorithm:  use open addressing double hashing (no chaining) on the
 * prefix code / next character combination.  We do a variant of Knuth's
 * algorithm D (vol. 3, sec. 6.4) along with G. Knott's relatively-prime
 * secondary probe.  Here, the modular division first probe is gives way
 * to a faster exclusive-or manipulation.  Also do block compression with
 * an adaptive reset, whereby the code table is cleared when the compression
 * ratio decreases, but after the table fills.	The variable-length output
 * codes are re-sized at this point, and a special CLEAR code is generated
 * for the decompressor.  Late addition:  construct the table according to
 * file size for noticeable speed improvement on small files.  Please direct
 * questions about this implementation to ames!jaw.
 */

static int g_init_bits;

static int ClearCode;
static int EOFCode;

static int compress(int init_bits, ifunptr ReadValue ){
    register long fcode;
    register int i /* = 0 */;
    register int c;
    register int ent;
    register int disp;
    register int hsize_reg;
    register int hshift;

    /*
     * Set up the globals:  g_init_bits - initial number of bits
     *			    g_outfile	- pointer to output file
     */
    g_init_bits = init_bits;

    /*
     * Set up the necessary values
     */
    offset = 0;
    out_count = 0;
    clear_flg = 0;
    in_count = 1;
    maxcode = MAXCODE(n_bits = g_init_bits);

    ClearCode = (1 << (init_bits - 1));
    EOFCode = ClearCode + 1;
    free_ent = ClearCode + 2;

    char_init();

    ent = GIFNextPixel(ReadValue);

    hshift = 0;
    for ( fcode = (long) hsize;  fcode < 65536L; fcode *= 2L )
	++hshift;
    hshift = 8 - hshift;		/* set hash code range bound */

    hsize_reg = hsize;
    cl_hash( (long int) hsize_reg);		/* clear hash table */

    if(!output( (int)ClearCode ))
    	return FALSE;

    while ( (c = GIFNextPixel( ReadValue )) != EOF ) {	/* } */

	++in_count;

	fcode = (long) (((long) c << maxbits) + ent);
	i = (((int)c << hshift) ^ ent);	/* xor hashing */

	if ( HashTabOf (i) == fcode ) {
	    ent = CodeTabOf (i);
	    continue;
	} else if ( (long)HashTabOf (i) < 0 )	   /* empty slot */
	    goto nomatch;
	disp = hsize_reg - i;		/* secondary hash (after G. Knott) */
	if ( i == 0 )
	    disp = 1;
probe:
	if ( (i -= disp) < 0 )
	    i += hsize_reg;

	if ( HashTabOf (i) == fcode ) {
	    ent = CodeTabOf (i);
	    continue;
	}
	if ( (long)HashTabOf (i) > 0 )
	    goto probe;
nomatch:
	if(!output ( (int) ent ))
		return FALSE;
	++out_count;
	ent = c;
	if ( free_ent < maxmaxcode ) {	/* } */
	    CodeTabOf (i) = free_ent++; /* code -> hashtable */
	    HashTabOf (i) = fcode;
	} 
	else {
		if(!cl_block())
			return FALSE;
	}
    }
    /*
     * Put out the final code.
     */
    if(!output( (int)ent ))
    	return FALSE;
    ++out_count;
    if(!output( (int) EOFCode ))
    	return FALSE;
	return TRUE;
}

/*****************************************************************
 * TAG( output )
 *
 * Output the given code.
 * Inputs:
 *	code:	A n_bits-bit integer.  If == -1, then EOF.  This assumes
 *		that n_bits =< (long)wordsize - 1.
 * Outputs:
 *	Outputs code to the file.
 * Assumptions:
 *	Chars are 8 bits long.
 * Algorithm:
 *	Maintain a BITS character long buffer (so that 8 codes will
 * fit in it exactly).	Use the VAX insv instruction to insert each
 * code in turn.  When the buffer fills up empty it and start over.
 */

static u32 cur_accum = 0;
static int cur_bits = 0;

static u32 masks[] = { 0x0000, 0x0001, 0x0003, 0x0007, 0x000F,
				  0x001F, 0x003F, 0x007F, 0x00FF,
				  0x01FF, 0x03FF, 0x07FF, 0x0FFF,
				  0x1FFF, 0x3FFF, 0x7FFF, 0xFFFF };

static int output(int code )
{
    cur_accum &= masks[ cur_bits ];

    if( cur_bits > 0 )
	cur_accum |= ((long)code << cur_bits);
    else
	cur_accum = code;

    cur_bits += n_bits;

    while( cur_bits >= 8 ) {
	if(!char_out( (unsigned int)(cur_accum & 0xff) ))
		return FALSE;
	cur_accum >>= 8;
	cur_bits -= 8;
    }

    /*
     * If the next entry is going to be too big for the code size,
     * then increase it, if possible.
     */
   if ( free_ent > maxcode || clear_flg ) {

	    if( clear_flg ) {

		maxcode = MAXCODE (n_bits = g_init_bits);
		clear_flg = 0;

	    } else {

		++n_bits;
		if ( n_bits == maxbits )
		    maxcode = maxmaxcode;
		else
		    maxcode = MAXCODE(n_bits);
	    }
	}

    if( code == EOFCode ) {
	/*
	 * At EOF, write the rest of the buffer.
	 */
		while( cur_bits > 0 ) {
			if(!char_out( (unsigned int)(cur_accum & 0xff) ))
				return FALSE;
			cur_accum >>= 8;
			cur_bits -= 8;
		}
	
		if(!flush_char())
			return FALSE;
	}

	return TRUE;
}

/*
 * Clear out the hash table
 */
static int cl_block ()		/* table clear for block compress */
{

	cl_hash ( (long int) hsize );
	free_ent = ClearCode + 2;
	clear_flg = 1;

	if(!output( (int)ClearCode ))
		return FALSE;
	return TRUE;
}

static void
cl_hash(long int hsize)		/* reset code table */
{
	register long i;
	register long int *htab_p = htab+hsize;
	register long m1 = -1;

	i = hsize - 16;
	do {				/* might use Sys V memset(3) here */
		*(htab_p-16) = m1;
		*(htab_p-15) = m1;
		*(htab_p-14) = m1;
		*(htab_p-13) = m1;
		*(htab_p-12) = m1;
		*(htab_p-11) = m1;
		*(htab_p-10) = m1;
		*(htab_p-9) = m1;
		*(htab_p-8) = m1;
		*(htab_p-7) = m1;
		*(htab_p-6) = m1;
		*(htab_p-5) = m1;
		*(htab_p-4) = m1;
		*(htab_p-3) = m1;
		*(htab_p-2) = m1;
		*(htab_p-1) = m1;
		htab_p -= 16;
	} while ((i -= 16) >= 0);

	for ( i += 16; i > 0; --i )
		*--htab_p = m1;
}

/******************************************************************************
 *
 * GIF Specific routines
 *
 ******************************************************************************/

/*
 * Number of characters so far in this 'packet'
 */
static int a_count;

/*
 * Set up the 'byte output' routine
 */
static void char_init(){
	a_count = 0;
}

/*
 * Define the storage for the packet accumulator
 */
static char accum[ 256 ];

static void
gifencinit(){
	int i;
	pwdata = NULL;
	wdatasize = 0;
	cur_accum = 0;
	cur_bits = 0;
	for(i = 0; i < 256; i++)
		accum[i] = 0;
}

/*
 * Add a character to the end of the current packet, and if it is 254
 * characters, flush the packet to disk.
 */
static int
char_out( int c ) {
	accum[ a_count++ ] = c;
	if( a_count >= 254 ) {
		if(!flush_char())
			return FALSE;
	}
	return TRUE;
}
/*
 * Flush the packet to disk, and reset the accumulator
 */
static int flush_char(){
	if( a_count > 0 ) {
		WRITE_16(worddata, a_count);
		if(!mem_write_data(worddata, 1, 1))
			return FALSE;
		if(!mem_write_data((u8 *)accum, 1, (u32)a_count))
			return FALSE;
		a_count = 0;
	}
	return TRUE;
}

