/*
** nester - NES emulator
** Copyright (C) 2000  Darren Ranalli
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
** Library General Public License for more details.  To obtain a 
** copy of the GNU Library General Public License, write to the Free 
** Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** Any permitted reproduction of these routines, in whole or in part,
** must bear this legend.
*/

#define _NES_MAPPER_CPP_

#include "NES_mapper.h"
#include "../NES.h"

#include "../../debug/debug.h"

#include "../../GBA_globals.h" 
#include "../../arm7_shared.h"

#define MAPPERSTATIC static

#define INLINE __inline//static

//appears to be unused
//MAPPERSTATIC uint32 num_16k_ROM_banks;
uint32 num_8k_ROM_banks NESTER_DTCM;
uint32 num_1k_VROM_banks NESTER_DTCM;
MAPPERSTATIC uint8* ROM_banks NESTER_DTCM;
MAPPERSTATIC uint8* VROM_banks NESTER_DTCM;
MAPPERSTATIC uint32 ROM_mask NESTER_DTCM;
MAPPERSTATIC uint32 VROM_mask NESTER_DTCM;

MAPPERSTATIC uint32 write_count NESTER_DTCM;
MAPPERSTATIC uint8  bits NESTER_DTCM;
//MAPPERSTATIC uint8  regs[4]; (use the largest regs globally)
MAPPERSTATIC uint32 last_write_addr NESTER_DTCM;
typedef enum
{
	MMC1_SMALL,
	MMC1_512K,
	MMC1_1024K
} MMC1_Size_t;
MAPPERSTATIC MMC1_Size_t MMC1_Size NESTER_DTCM;
MAPPERSTATIC uint32 MMC1_256K_base NESTER_DTCM;
MAPPERSTATIC uint32 MMC1_swap NESTER_DTCM;
// these are the 4 ROM banks currently selected
MAPPERSTATIC uint32 MMC1_bank1 NESTER_DTCM;
MAPPERSTATIC uint32 MMC1_bank2 NESTER_DTCM;
MAPPERSTATIC uint32 MMC1_bank3 NESTER_DTCM;
MAPPERSTATIC uint32 MMC1_bank4 NESTER_DTCM;
MAPPERSTATIC uint32 MMC1_HI1 NESTER_DTCM;
MAPPERSTATIC uint32 MMC1_HI2 NESTER_DTCM;

uint8  regs[8] NESTER_DTCM;

//uint32 prg0 NESTER_DTCM,prg1 NESTER_DTCM;
//uint32 chr01 NESTER_DTCM,chr23 NESTER_DTCM,chr4 NESTER_DTCM,chr5 NESTER_DTCM,chr6 NESTER_DTCM,chr7 NESTER_DTCM;
uint32 prg0 NESTER_DTCM;
uint32 prg1 NESTER_DTCM;
uint32 chr01 NESTER_DTCM;
uint32 chr23 NESTER_DTCM;
uint32 chr4 NESTER_DTCM;
uint32 chr5 NESTER_DTCM;
uint32 chr6 NESTER_DTCM;
uint32 chr7 NESTER_DTCM;

MAPPERSTATIC uint8 irq_enabled NESTER_DTCM; // IRQs enabled
MAPPERSTATIC uint8 irq_counter NESTER_DTCM; // IRQ scanline counter, decreasing
MAPPERSTATIC uint8 irq_latch NESTER_DTCM;   // IRQ scanline counter latch

//MAPPERSTATIC uint8 regs[6]; (use the largest regs globally)
MAPPERSTATIC uint8 latch_0000 NESTER_DTCM;
MAPPERSTATIC uint8 latch_1000 NESTER_DTCM;

//MAPPERSTATIC uint8 irq_enabled;
MAPPERSTATIC uint32 lines_to_irq NESTER_DTCM;


#ifndef ASSERT
#define ASSERT(EXPR)
#endif

#define MASK_BANK(bank,mask) (bank) = ((bank) & (mask))

#ifdef NESTER_DEBUG
  #define VALIDATE_ROM_BANK(bank) \
    MASK_BANK(bank,ROM_mask); \
    ASSERT((bank) < num_8k_ROM_banks) \
    if((bank) >= num_8k_ROM_banks) \
    { \
      LOG("Illegal ROM bank switch: " << (int)(bank) << "/" << (int)num_8k_ROM_banks << endl); \
      return; \
    }

  #define VALIDATE_VROM_BANK(bank) \
    MASK_BANK(bank,VROM_mask); \
    ASSERT((bank) < num_1k_VROM_banks) \
    if((bank) >= num_1k_VROM_banks) \
    { \
      LOG("Illegal VROM bank switch: " << (int)(bank) << "/" << (int)num_1k_VROM_banks << endl); \
      return; \
    }
#else
  #define VALIDATE_ROM_BANK(bank) \
    MASK_BANK(bank,ROM_mask); \
    if((bank) >= num_8k_ROM_banks) return;
  #define VALIDATE_VROM_BANK(bank) \
    MASK_BANK(bank,VROM_mask); \
    if((bank) >= num_1k_VROM_banks) return;
#endif

/////////////////////////////////////////////////////////////////////
// Mapper virtual base class
//NES_mapper_NES_mapper(NES* parent) : parent_NES(parent)
void NES_mapper_FakeConstructor()
{
  uint32 probe;

  //num_16k_ROM_banks = ROM_get_num_16k_ROM_banks();
  num_8k_ROM_banks = 2 * ROM_get_num_16k_ROM_banks();
  num_1k_VROM_banks = 8 * ROM_get_num_8k_VROM_banks();

  ROM_banks  = ROM_get_ROM_banks();
  VROM_banks = ROM_get_VROM_banks();

  ROM_mask  = 0xFFFF;
  VROM_mask = 0xFFFF;

  for(probe = 0x8000; probe; probe >>= 1)
  {
    if((num_8k_ROM_banks-1) & probe) break;
    ROM_mask >>= 1;
  }
  for(probe = 0x8000; probe; probe >>= 1)
  {
    if((num_1k_VROM_banks-1) & probe) break;
    VROM_mask >>= 1;
  }

//  LOG(HEX(ROM_mask,2) << " " << HEX(VROM_mask,2) << endl);
}

void NES_mapper_set_CPU_banks(uint32 bank4_num, uint32 bank5_num,
                               uint32 bank6_num, uint32 bank7_num)
{
  CPUINT_Context context;

  VALIDATE_ROM_BANK(bank4_num);
  VALIDATE_ROM_BANK(bank5_num);
  VALIDATE_ROM_BANK(bank6_num);
  VALIDATE_ROM_BANK(bank7_num);

/*
  LOG("Setting CPU banks " << bank4_num << " " << bank5_num << " " <<
                              bank6_num << " " << bank7_num << endl);
*/

#if 0
  {
	  char inf[1024];
	  sprintf(inf, "setcpubanks\n%i\n%i\n%i\n%i", bank4_num << 13, bank5_num << 13, bank6_num << 13, bank7_num << 13);
	  GBA_Log(inf);
  }
#endif

  CPUINT_GetContext(&context);
  context.mem_page[4] = ROM_banks + (bank4_num << 13); // * 0x2000
  context.mem_page[5] = ROM_banks + (bank5_num << 13);
  context.mem_page[6] = ROM_banks + (bank6_num << 13);
  context.mem_page[7] = ROM_banks + (bank7_num << 13);
  CPUINT_SetContext(&context);

#if 0
  {
	  char inf[1024];
	  int d = (int)ROM_banks;
	  int dx = (int)(ROM_banks+(bank4_num << 13));
	  sprintf(inf, "newslotthingisnow %i %i %i\n%i %i", context.mem_page[4][0], context.mem_page[4][1], context.mem_page[4][1810], d, dx);
	  GBA_Log(inf);
  }
#endif
}

void NES_mapper_set_CPU_bank4(uint32 bank_num)
{
  CPUINT_Context context;

  VALIDATE_ROM_BANK(bank_num);

  CPUINT_GetContext(&context);
  context.mem_page[4] = ROM_banks + (bank_num << 13); // * 0x2000
  CPUINT_SetContext(&context);
}

void NES_mapper_set_CPU_bank5(uint32 bank_num)
{
  CPUINT_Context context;

  VALIDATE_ROM_BANK(bank_num);

  CPUINT_GetContext(&context);
  context.mem_page[5] = ROM_banks + (bank_num << 13); // * 0x2000
  CPUINT_SetContext(&context);
}

void NES_mapper_set_CPU_bank6(uint32 bank_num)
{
  CPUINT_Context context;

  VALIDATE_ROM_BANK(bank_num);

  CPUINT_GetContext(&context);
  context.mem_page[6] = ROM_banks + (bank_num << 13); // * 0x2000
  CPUINT_SetContext(&context);
}

void NES_mapper_set_CPU_bank7(uint32 bank_num)
{
  CPUINT_Context context;

  VALIDATE_ROM_BANK(bank_num);

  CPUINT_GetContext(&context);
  context.mem_page[7] = ROM_banks + (bank_num << 13); // * 0x2000
  CPUINT_SetContext(&context);
}

// for mapper 40 /////////////////////////////////////////////////////////
void NES_mapper_set_CPU_banksM40(uint32 bank3_num,
                               uint32 bank4_num, uint32 bank5_num,
                               uint32 bank6_num, uint32 bank7_num)
{
  CPUINT_Context context;

  VALIDATE_ROM_BANK(bank3_num);
  VALIDATE_ROM_BANK(bank4_num);
  VALIDATE_ROM_BANK(bank5_num);
  VALIDATE_ROM_BANK(bank6_num);
  VALIDATE_ROM_BANK(bank7_num);

  CPUINT_GetContext(&context);
  context.mem_page[3] = ROM_banks + (bank3_num << 13); // * 0x2000
  context.mem_page[4] = ROM_banks + (bank4_num << 13);
  context.mem_page[5] = ROM_banks + (bank5_num << 13);
  context.mem_page[6] = ROM_banks + (bank6_num << 13);
  context.mem_page[7] = ROM_banks + (bank7_num << 13);
  CPUINT_SetContext(&context);
}

void NES_mapper_set_CPU_bank3(uint32 bank_num)
{
  CPUINT_Context context;

  VALIDATE_ROM_BANK(bank_num);

  CPUINT_GetContext(&context);
  context.mem_page[3] = ROM_banks + (bank_num << 13); // * 0x2000
  CPUINT_SetContext(&context);
}
//////////////////////////////////////////////////////////////////////////


void NES_mapper_set_PPU_banks(uint32 bank0_num, uint32 bank1_num,
                               uint32 bank2_num, uint32 bank3_num,
                               uint32 bank4_num, uint32 bank5_num,
                               uint32 bank6_num, uint32 bank7_num)
{
  VALIDATE_VROM_BANK(bank0_num);
  VALIDATE_VROM_BANK(bank1_num);
  VALIDATE_VROM_BANK(bank2_num);
  VALIDATE_VROM_BANK(bank3_num);
  VALIDATE_VROM_BANK(bank4_num);
  VALIDATE_VROM_BANK(bank5_num);
  VALIDATE_VROM_BANK(bank6_num);
  VALIDATE_VROM_BANK(bank7_num);

  PPU_VRAM_banks[0] = VROM_banks + (bank0_num << 10); // * 0x400
  PPU_VRAM_banks[1] = VROM_banks + (bank1_num << 10);
  PPU_VRAM_banks[2] = VROM_banks + (bank2_num << 10);
  PPU_VRAM_banks[3] = VROM_banks + (bank3_num << 10);
  PPU_VRAM_banks[4] = VROM_banks + (bank4_num << 10);
  PPU_VRAM_banks[5] = VROM_banks + (bank5_num << 10);
  PPU_VRAM_banks[6] = VROM_banks + (bank6_num << 10);
  PPU_VRAM_banks[7] = VROM_banks + (bank7_num << 10);
}

void NES_mapper_set_PPU_bank0(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[0] = VROM_banks + (bank_num << 10); // * 0x400
}

void NES_mapper_set_PPU_bank1(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[1] = VROM_banks + (bank_num << 10); // * 0x400
}

void NES_mapper_set_PPU_bank2(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[2] = VROM_banks + (bank_num << 10); // * 0x400
}

void NES_mapper_set_PPU_bank3(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[3] = VROM_banks + (bank_num << 10); // * 0x400
}

void NES_mapper_set_PPU_bank4(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[4] = VROM_banks + (bank_num << 10); // * 0x400
}

void NES_mapper_set_PPU_bank5(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[5] = VROM_banks + (bank_num << 10); // * 0x400
}

void NES_mapper_set_PPU_bank6(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[6] = VROM_banks + (bank_num << 10); // * 0x400
}

void NES_mapper_set_PPU_bank7(uint32 bank_num)
{
  VALIDATE_VROM_BANK(bank_num);
  PPU_VRAM_banks[7] = VROM_banks + (bank_num << 10); // * 0x400
}


void NES_mapper_set_mirroring(uint32 nt0, uint32 nt1, uint32 nt2, uint32 nt3)
{
  ASSERT(nt0 < 4);
  ASSERT(nt1 < 4);
  ASSERT(nt2 < 4);
  ASSERT(nt3 < 4);

  PPU_set_mirroring(nt0,nt1,nt2,nt3);
}

void NES_mapper_set_mirroringB(mirroring_type m)
{
  PPU_set_mirroringB(m);
}
/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
// Mapper 0
void NES_mapper0_Reset()
{
  // set CPU bank pointers
  if(num_8k_ROM_banks > 2)
  {
    NES_mapper_set_CPU_banks(0,1,2,3);
  }
  else if(num_8k_ROM_banks > 1)
  {
    NES_mapper_set_CPU_banks(0,1,0,1);
  }
  else
  {
    NES_mapper_set_CPU_banks(0,0,0,0);
  }

  if(num_1k_VROM_banks)
  {
    NES_mapper_set_PPU_banks(0,1,2,3,4,5,6,7);
  }
}
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 1
void NES_mapper1_Reset()
{
  write_count = 0;
  bits = 0x00;
  regs[0] = 0x0C; // reflects initial ROM state
  regs[1] = 0x00;
  regs[2] = 0x00;
  regs[3] = 0x00;

  {
    uint32 size_in_K = num_8k_ROM_banks * 8;

    if(size_in_K == 1024)
    {
      MMC1_Size = MMC1_1024K;
    }
    else if(size_in_K == 512)
    {
      MMC1_Size = MMC1_512K;
    }
    else
    {
      MMC1_Size = MMC1_SMALL;
    }
  }
  MMC1_256K_base = 0; // use first 256K
  MMC1_swap = 0;

  if(MMC1_Size == MMC1_SMALL)
  {
    // set two high pages to last two banks
    MMC1_HI1 = num_8k_ROM_banks-2;
    MMC1_HI2 = num_8k_ROM_banks-1;
  }
  else
  {
    // set two high pages to last two banks of current 256K region
    MMC1_HI1 = (256/8)-2;
    MMC1_HI2 = (256/8)-1;
  }

  // set CPU bank pointers
  MMC1_bank1 = 0;
  MMC1_bank2 = 1;
  MMC1_bank3 = MMC1_HI1;
  MMC1_bank4 = MMC1_HI2;

  NES_mapper1_MMC1_set_CPU_banks();
}

void NES_mapper1_MMC1_set_CPU_banks()
{
  NES_mapper_set_CPU_banks((MMC1_256K_base << 5) + (MMC1_bank1 & ((256/8)-1)),
                (MMC1_256K_base << 5) + (MMC1_bank2 & ((256/8)-1)),
                (MMC1_256K_base << 5) + (MMC1_bank3 & ((256/8)-1)),
                (MMC1_256K_base << 5) + (MMC1_bank4 & ((256/8)-1)));
}

void NES_mapper1_MemoryWrite(uint32 addr, uint8 data)
{
  uint32 reg_num;

  // if write is to a different reg, reset
  if((addr & 0x6000) != (last_write_addr & 0x6000))
  {
    write_count = 0;
    bits = 0x00;
  }
  last_write_addr = addr;

  // if bit 7 set, reset and return
  if(data & 0x80)
  {
    write_count = 0;
    bits = 0x00;
    return;
  }

  if(data & 0x01) bits |= (1 << write_count);
  write_count++;
  if(write_count < 5) return;

  reg_num = (addr & 0x7FFF) >> 13;
  regs[reg_num] = bits;

  write_count = 0;
  bits = 0x00;

//  LOG("MAP1 REG" << reg_num << ": " << HEX(regs[reg_num],2) << endl);

  switch(reg_num)
  {
    case 0:
      {
//        LOG("REG0: " << HEX(reg[0],2) << endl);

        // set mirroring
        if(regs[0] & 0x02)
        {
          if(regs[0] & 0x01)
          {
            NES_mapper_set_mirroringB(MIRROR_HORIZ);
          }
          else
          {
            NES_mapper_set_mirroringB(MIRROR_VERT);
          }
        }
        else
        {
          // one-screen mirroring
          if(regs[0] & 0x01)
          {
            NES_mapper_set_mirroring(1,1,1,1);
          }
          else
          {
            NES_mapper_set_mirroring(0,0,0,0);
          }
        }
      }
      break;

    case 1:
      {
        uint8 bank_num = regs[1];

//        LOG("REG1: " << HEX(reg[1],2) << endl);

        if(MMC1_Size == MMC1_1024K)
        {
          if(regs[0] & 0x10)
          {
            if(MMC1_swap)
            {
              MMC1_256K_base = (regs[1] & 0x10) >> 4;
              if(regs[0] & 0x08)
              {
                MMC1_256K_base |= ((regs[2] & 0x10) >> 3);
              }
              NES_mapper1_MMC1_set_CPU_banks();
              MMC1_swap = 0;
            }
            else
            {
              MMC1_swap = 1;
            }
          }
          else
          {
            // use 1st or 4th 256K banks
            MMC1_256K_base = (regs[1] & 0x10) ? 3 : 0;
            NES_mapper1_MMC1_set_CPU_banks();
          }
        }
        else if((MMC1_Size == MMC1_512K) && (!num_1k_VROM_banks))
        {
          MMC1_256K_base = (regs[1] & 0x10) >> 4;
          NES_mapper1_MMC1_set_CPU_banks();
        }
        else if(num_1k_VROM_banks)
        {
          // set VROM bank at $0000
          if(regs[0] & 0x10)
          {
            // swap 4K
            bank_num <<= 2;
            NES_mapper_set_PPU_bank0(bank_num+0);
            NES_mapper_set_PPU_bank1(bank_num+1);
            NES_mapper_set_PPU_bank2(bank_num+2);
            NES_mapper_set_PPU_bank3(bank_num+3);
          }
          else
          {
            // swap 8K
            bank_num <<= 2;
            NES_mapper_set_PPU_banks(bank_num+0,bank_num+1,bank_num+2,bank_num+3,
                          bank_num+4,bank_num+5,bank_num+6,bank_num+7);
          }
        }
      }
      break;

    case 2:
      {
        uint8 bank_num = regs[2];

//        LOG("REG2: " << HEX(reg[2],2) << endl);

        if((MMC1_Size == MMC1_1024K) && (regs[0] & 0x08))
        {
          if(MMC1_swap)
          {
            MMC1_256K_base =  (regs[1] & 0x10) >> 4;
            MMC1_256K_base |= ((regs[2] & 0x10) >> 3);
            NES_mapper1_MMC1_set_CPU_banks();
            MMC1_swap = 0;
          }
          else
          {
            MMC1_swap = 1;
          }
        }

        if(!num_1k_VROM_banks) break;

        // set 4K VROM bank at $1000
        if(regs[0] & 0x10)
        {
          // swap 4K
          bank_num <<= 2;
          NES_mapper_set_PPU_bank4(bank_num+0);
          NES_mapper_set_PPU_bank5(bank_num+1);
          NES_mapper_set_PPU_bank6(bank_num+2);
          NES_mapper_set_PPU_bank7(bank_num+3);
        }
      }
      break;

    case 3:
      {
        uint8 bank_num = regs[3];

//        LOG("REG3: " << HEX(reg[3],2) << endl);

        // set ROM bank
        if(regs[0] & 0x08)
        {
          // 16K of ROM
          bank_num <<= 1;

          if(regs[0] & 0x04)
          {
            // 16K of ROM at $8000
            MMC1_bank1 = bank_num;
            MMC1_bank2 = bank_num+1;
            MMC1_bank3 = MMC1_HI1;
            MMC1_bank4 = MMC1_HI2;
          }
          else
          {
            // 16K of ROM at $C000
            if(MMC1_Size == MMC1_SMALL)
            {
              MMC1_bank1 = 0;
              MMC1_bank2 = 1;
              MMC1_bank3 = bank_num;
              MMC1_bank4 = bank_num+1;
            }
          }
        }
        else
        {
          // 32K of ROM at $8000
          bank_num <<= 2;

          MMC1_bank1 = bank_num;
          MMC1_bank2 = bank_num+1;
          if(MMC1_Size == MMC1_SMALL)
          {
            MMC1_bank3 = bank_num+2;
            MMC1_bank4 = bank_num+3;
          }
        }

        NES_mapper1_MMC1_set_CPU_banks();
      }
      break;
  }
}
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 2
void NES_mapper2_Reset()
{
  // set CPU bank pointers
  NES_mapper_set_CPU_banks(0,1,num_8k_ROM_banks-2,num_8k_ROM_banks-1);
}

void NES_mapper2_MemoryWrite(uint32 addr, uint8 data)
{
  data &= num_8k_ROM_banks-1;
  NES_mapper_set_CPU_banks(data*2,(data*2)+1,num_8k_ROM_banks-2,num_8k_ROM_banks-1);
}
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 3
void NES_mapper3_Reset()
{
  // set CPU bank pointers
  if(num_8k_ROM_banks > 2)
  {
    NES_mapper_set_CPU_banks(0,1,2,3);
  }
  else
  {
    NES_mapper_set_CPU_banks(0,1,0,1);
  }

  // set VROM banks
  NES_mapper_set_PPU_banks(0,1,2,3,4,5,6,7);
}

void NES_mapper3_MemoryWrite(uint32 addr, uint8 data)
{
  uint32 base;

  data &= (num_1k_VROM_banks>>1)-1;

  base = ((uint32)data) << 3;
  NES_mapper_set_PPU_banks(base+0,base+1,base+2,base+3,base+4,base+5,base+6,base+7);
}
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 4
// much of this is based on the DarcNES source. thanks, nyef :)
void NES_mapper4_Reset()
{
  // clear registers FIRST!!!
  int i;
  for(i = 0; i < 8; i++) regs[i] = 0x00;

  // set CPU bank pointers
  prg0 = 0;
  prg1 = 1;
  NES_mapper4_MMC3_set_CPU_banks();

  // set VROM banks
  if(num_1k_VROM_banks)
  {
    chr01 = 0;
    chr23 = 2;
    chr4  = 4;
    chr5  = 5;
    chr6  = 6;
    chr7  = 7;
    NES_mapper4_MMC3_set_PPU_banks();
  }
  else
  {
    chr01 = chr23 = chr4 = chr5 = chr6 = chr7 = 0;
  }

  irq_enabled = 0;
  irq_counter = 0;
  irq_latch = 0;
}

void NES_mapper4_MemoryWrite(uint32 addr, uint8 data)
{
//  LOG("Write: " << HEX((addr & 0xE001),4) << "=" << HEX(data,2) << endl);
  switch(addr & 0xE001)
  {
    case 0x8000:
      {
        regs[0] = data;
        NES_mapper4_MMC3_set_PPU_banks();
        NES_mapper4_MMC3_set_CPU_banks();
      }
      break;

    case 0x8001:
      {
        uint32 bank_num;

        regs[1] = data;

        bank_num = regs[1];

//        LOG("CMD" << (int)(regs[0]&0x07) << ": " << HEX(bank_num,2) << endl);

        switch(regs[0] & 0x07)
        {
          case 0x00:
            {
              if(num_1k_VROM_banks)
              {
                bank_num &= 0xfe;
                chr01 = bank_num;
                NES_mapper4_MMC3_set_PPU_banks();
              }
            }
            break;

          case 0x01:
            {
              if(num_1k_VROM_banks)
              {
                bank_num &= 0xfe;
                chr23 = bank_num;
                NES_mapper4_MMC3_set_PPU_banks();
              }
            }
            break;

          case 0x02:
            {
              if(num_1k_VROM_banks)
              {
                chr4 = bank_num;
                NES_mapper4_MMC3_set_PPU_banks();
              }
            }
            break;

          case 0x03:
            {
              if(num_1k_VROM_banks)
              {
                chr5 = bank_num;
                NES_mapper4_MMC3_set_PPU_banks();
              }
            }
            break;

          case 0x04:
            {
              if(num_1k_VROM_banks)
              {
                chr6 = bank_num;
                NES_mapper4_MMC3_set_PPU_banks();
              }
            }
            break;

          case 0x05:
            {
              if(num_1k_VROM_banks)
              {
                chr7 = bank_num;
                NES_mapper4_MMC3_set_PPU_banks();
              }
            }
            break;

          case 0x06:
            {
              prg0 = bank_num;
              NES_mapper4_MMC3_set_CPU_banks();
            }
            break;

          case 0x07:
            {
              prg1 = bank_num;
              NES_mapper4_MMC3_set_CPU_banks();
            }
            break;
        }
      }
      break;

    case 0xA000:
      {
        regs[2] = data;

        if(data & 0x40)
        {
          LOG("MAP4 MIRRORING: 0x40 ???" << endl);
        }

        if(ROM_get_mirroring() != MIRROR_FOUR_SCREEN)
        {
          if(data & 0x01)
          {
            NES_mapper_set_mirroringB(MIRROR_HORIZ);
          }
          else
          {
            NES_mapper_set_mirroringB(MIRROR_VERT);
          }
        }
      }
      break;

    case 0xA001:
      {
        regs[3] = data;

        if(data & 0x80)
        {
          // enable save RAM $6000-$7FFF
        }
        else
        {
          // disable save RAM $6000-$7FFF
        }
      }
      break;

    case 0xC000:
//  LOG("counter = " << HEX(data,2) << endl);
      regs[4] = data;
      irq_counter = regs[4];
      break;

    case 0xC001:
//  LOG("latch = " << HEX(data,2) << endl);
      regs[5] = data;
      irq_latch = regs[5];
      break;

    case 0xE000:
//  LOG("enabled = 0" << endl);
      regs[6] = data;
      irq_enabled = 0;
      break;

    case 0xE001:
//  LOG("enabled = 1" << endl);
      regs[7] = data;
      irq_enabled = 1;
      break;

    default:
      LOG("MAP4: UNKNOWN: " << HEX(addr,4) << " = " << HEX(data) << endl);
      break;

  }
}

void NES_mapper4_HSync(uint32 scanline)
{
  if(irq_enabled)
  {
    if((scanline >= 0) && (scanline <= 239))
    {
      if(PPU_spr_enabled() || PPU_bg_enabled())
      {
        if(!(irq_counter--))
        {
          irq_counter = irq_latch;
          CPUINT_DoIRQ();
        }
      }
    }
  }
}

#define MAP4_ROM(ptr)  (((ptr)-ROM_get_ROM_banks())  >> 13)
#define MAP4_VROM(ptr) (((ptr)-ROM_get_VROM_banks()) >> 10)

void NES_mapper4_SNSS_fixup() // HACK HACK HACK HACK
{
  CPUINT_Context context;
  CPUINT_GetContext(&context);

  prg0 = MAP4_ROM(context.mem_page[prg_swap() ? 6 : 4]);
  prg1 = MAP4_ROM(context.mem_page[5]);
  if(num_1k_VROM_banks)
  {
    if(chr_swap())
    {
      chr01 = MAP4_VROM(PPU_VRAM_banks[4]);
      chr23 = MAP4_VROM(PPU_VRAM_banks[6]);
      chr4  = MAP4_VROM(PPU_VRAM_banks[0]);
      chr5  = MAP4_VROM(PPU_VRAM_banks[1]);
      chr6  = MAP4_VROM(PPU_VRAM_banks[2]);
      chr7  = MAP4_VROM(PPU_VRAM_banks[3]);
    }
    else
    {
      chr01 = MAP4_VROM(PPU_VRAM_banks[0]);
      chr23 = MAP4_VROM(PPU_VRAM_banks[2]);
      chr4  = MAP4_VROM(PPU_VRAM_banks[4]);
      chr5  = MAP4_VROM(PPU_VRAM_banks[5]);
      chr6  = MAP4_VROM(PPU_VRAM_banks[6]);
      chr7  = MAP4_VROM(PPU_VRAM_banks[7]);
    }
  }
}

/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 7
void NES_mapper7_Reset()
{
  // set CPU bank pointers
  NES_mapper_set_CPU_banks(0,1,2,3);
}

void NES_mapper7_MemoryWrite(uint32 addr, uint8 data)
{
  uint32 bank;

  bank = (data & 0x07) << 2;
  NES_mapper_set_CPU_banks(bank+0,bank+1,bank+2,bank+3);

  if(data & 0x10)
  {
    NES_mapper_set_mirroring(1,1,1,1);
  }
  else
  {
    NES_mapper_set_mirroring(0,0,0,0);
  }
}
/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 9
void NES_mapper9_Reset()
{
  int i;

  // set CPU bank pointers
  NES_mapper_set_CPU_banks(0,num_8k_ROM_banks-3,num_8k_ROM_banks-2,num_8k_ROM_banks-1);

  for(i = 0; i < (int)(sizeof(regs)/sizeof(regs[0])); i++)
    regs[i] = 0;

  regs[2] = 4;

  latch_0000 = 0xFE;
  latch_1000 = 0xFE;

  NES_mapper9_set_VROM_0000();
  NES_mapper9_set_VROM_1000();
}

void NES_mapper9_PPU_Latch_FDFE(uint32 addr)
{
  if(addr & 0x1000)
  {
    latch_1000 = (addr & 0x0FF0) >> 4;
    NES_mapper9_set_VROM_1000();
  }
  else
  {
    latch_0000 = (addr & 0x0FF0) >> 4;
    NES_mapper9_set_VROM_0000();
  }
}

void NES_mapper9_set_VROM_0000()
{
  int bank_num = (latch_0000 == 0xFD) ? regs[1] : regs[2];

  bank_num <<= 2;

  NES_mapper_set_PPU_bank0(bank_num+0);
  NES_mapper_set_PPU_bank1(bank_num+1);
  NES_mapper_set_PPU_bank2(bank_num+2);
  NES_mapper_set_PPU_bank3(bank_num+3);
}

void NES_mapper9_set_VROM_1000()
{
  int bank_num = (latch_1000 == 0xFD) ? regs[3] : regs[4];

  bank_num <<= 2;

  NES_mapper_set_PPU_bank4(bank_num+0);
  NES_mapper_set_PPU_bank5(bank_num+1);
  NES_mapper_set_PPU_bank6(bank_num+2);
  NES_mapper_set_PPU_bank7(bank_num+3);
}

void NES_mapper9_MemoryWrite(uint32 addr, uint8 data)
{
  switch(addr & 0xF000)
  {
    case 0xA000:
      {
		uint8 bank_num;

        regs[0] = data;

        // 8K ROM bank at $8000
        bank_num = regs[0];
        NES_mapper_set_CPU_bank4(bank_num);
      }
      break;

    case 0xB000:
      {
        // B000-BFFF: select 4k VROM for (0000) $FD latch
        regs[1] = data;
        NES_mapper9_set_VROM_0000();
      }
      break;

    case 0xC000:
      {
        // C000-CFFF: select 4k VROM for (0000) $FE latch
        regs[2] = data;
        NES_mapper9_set_VROM_0000();
      }
      break;

    case 0xD000:
      {
        // D000-DFFF: select 4k VROM for (1000) $FD latch
        regs[3] = data;
        NES_mapper9_set_VROM_1000();
      }
      break;

    case 0xE000:
      {
        // E000-EFFF: select 4k VROM for (1000) $FE latch
        regs[4] = data;
        NES_mapper9_set_VROM_1000();
      }
      break;

    case 0xF000:
      {
        regs[5] = data;

        if(regs[5] & 0x01)
        {
          NES_mapper_set_mirroringB(MIRROR_HORIZ);
        }
        else
        {
          NES_mapper_set_mirroringB(MIRROR_VERT);
        }
      }
      break;
  }
}

void NES_mapper9_SNSS_fixup()
{
  NES_mapper9_set_VROM_0000();
  NES_mapper9_set_VROM_1000();
}

/////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////
// Mapper 40 (smb2j)
void NES_mapper40_Reset()
{
  irq_enabled = 0;
  lines_to_irq = 0;

  // set CPU bank pointers
  NES_mapper_set_CPU_banksM40(6,4,5,0,7);

  // set VROM banks
  if(num_1k_VROM_banks)
  {
    NES_mapper_set_PPU_banks(0,1,2,3,4,5,6,7);
  }
}

void NES_mapper40_MemoryWrite(uint32 addr, uint8 data)
{
  switch(addr & 0xE000)
  {
    case 0x8000:
      irq_enabled = 0;
//      LOG("MAP40: [$8000] = " << HEX(data,2) << endl);
      break;

    case 0xA000:
      irq_enabled = 1;
      lines_to_irq = 37;

//      LOG("MAP40: [$A000] = " << HEX(data,2) << endl);
      break;

    case 0xC000:
//      LOG("MAP40: [$C000] = " << HEX(data,2) << endl);
//      LOG("MAP40: INVALID WRITE TO $C000" << endl);
      break;

    case 0xE000:
//      LOG("MAP40: [$E000] = " << HEX(data,2) << endl);

      NES_mapper_set_CPU_bank6(data & 0x07);
      break;

  }
}

void NES_mapper40_HSync(uint32 scanline)
{
  if(irq_enabled)
  {
    if((--lines_to_irq) <= 0)
    {
      CPUINT_DoIRQ();
    }
  }
}
/////////////////////////////////////////////////////////////////////

#include "NES_mapper_Konami.c"

/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
// mapper factory
MAPPERSTATIC void MapperFill_Default(nesMapper_t *m)
{
	m->FakeConstructor = NES_mapper_FakeConstructor;
	m->Reset = NES_mapper_Reset;

	m->MemoryWrite = NES_mapper_MemoryWrite;
	m->MemoryWriteLow = NES_mapper_MemoryWriteLow;
	m->MemoryWriteSaveRAM = NES_mapper_MemoryWriteSaveRAM;

	m->HSync = NES_mapper_HSync;
	m->VSync = NES_mapper_VSync;

	m->PPU_Latch_FDFE = NES_mapper_PPU_Latch_FDFE;

	m->set_CPU_banks = NES_mapper_set_CPU_banks;
	m->set_CPU_bank4 = NES_mapper_set_CPU_bank4;
	m->set_CPU_bank5 = NES_mapper_set_CPU_bank5;
	m->set_CPU_bank6 = NES_mapper_set_CPU_bank6;
	m->set_CPU_bank7 = NES_mapper_set_CPU_bank7;

	m->set_CPU_banksM40 = NES_mapper_set_CPU_banksM40;
	m->set_CPU_bank3 = NES_mapper_set_CPU_bank3;

	m->set_PPU_banks = NES_mapper_set_PPU_banks;
	m->set_PPU_bank0 = NES_mapper_set_PPU_bank0;
	m->set_PPU_bank1 = NES_mapper_set_PPU_bank1;
	m->set_PPU_bank2 = NES_mapper_set_PPU_bank2;
	m->set_PPU_bank3 = NES_mapper_set_PPU_bank3;
	m->set_PPU_bank4 = NES_mapper_set_PPU_bank4;
	m->set_PPU_bank5 = NES_mapper_set_PPU_bank5;
	m->set_PPU_bank6 = NES_mapper_set_PPU_bank6;
	m->set_PPU_bank7 = NES_mapper_set_PPU_bank7;

	m->set_mirroring = NES_mapper_set_mirroring;
	m->set_mirroringB = NES_mapper_set_mirroringB;
}

MAPPERSTATIC void MapperFill_Mapper0(nesMapper_t *m)
{
	m->Reset = NES_mapper0_Reset;
}

MAPPERSTATIC void MapperFill_Mapper1(nesMapper_t *m)
{
	m->Reset = NES_mapper1_Reset;
	m->MemoryWrite = NES_mapper1_MemoryWrite;
}

MAPPERSTATIC void MapperFill_Mapper2(nesMapper_t *m)
{
	m->Reset = NES_mapper2_Reset;
	m->MemoryWrite = NES_mapper2_MemoryWrite;
}

MAPPERSTATIC void MapperFill_Mapper3(nesMapper_t *m)
{
	m->Reset = NES_mapper3_Reset;
	m->MemoryWrite = NES_mapper3_MemoryWrite;
}

MAPPERSTATIC void MapperFill_Mapper4(nesMapper_t *m)
{
	m->Reset = NES_mapper4_Reset;
	m->MemoryWrite = NES_mapper4_MemoryWrite;
	m->HSync = NES_mapper4_HSync;
}

MAPPERSTATIC void MapperFill_Mapper7(nesMapper_t *m)
{
	m->Reset = NES_mapper7_Reset;
	m->MemoryWrite = NES_mapper7_MemoryWrite;
}

MAPPERSTATIC void MapperFill_Mapper9(nesMapper_t *m)
{
	m->Reset = NES_mapper9_Reset;
	m->MemoryWrite = NES_mapper9_MemoryWrite;
	m->PPU_Latch_FDFE = NES_mapper9_PPU_Latch_FDFE;
}

MAPPERSTATIC void MapperFill_Mapper24(nesMapper_t *m)
{
	m->Reset = NES_mapper24_Reset;
	m->MemoryWrite = NES_mapper24_MemoryWrite;
	m->HSync = NES_mapper24_HSync;
}

MAPPERSTATIC void MapperFill_Mapper40(nesMapper_t *m)
{
	m->Reset = NES_mapper40_Reset;
	m->MemoryWrite = NES_mapper40_MemoryWrite;
	m->HSync = NES_mapper40_HSync;
}

nesMapper_t* GetMapper()
{
	nesMapper_t *m = (nesMapper_t *)malloc(sizeof(nesMapper_t));
	MapperFill_Default(m);
	switch(ROM_get_mapper_num())
	{
	case 0:
		MapperFill_Mapper0(m);
		break;
	case 1:
		MapperFill_Mapper1(m);
		break;
	case 2:
		MapperFill_Mapper2(m);
		break;
	case 3:
		MapperFill_Mapper3(m);
		break;
	case 4:
		MapperFill_Mapper4(m);
		break;
	case 7:
		MapperFill_Mapper7(m);
		break;
	case 9:
		MapperFill_Mapper9(m);
		break;
	case 24:
		MapperFill_Mapper24(m);
		break;
	case 40:
		MapperFill_Mapper40(m);
		break;
	default:
		return 0;  // mapper not supported
	}
	return m;
}
/////////////////////////////////////////////////////////////////////
