/*  device.c
 *  Copyright (C) 2002-2005  CDVDlinuz Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <errno.h> // errno
#include <fcntl.h> // open()
#include <stddef.h> // NULL
#include <stdio.h> // printf()
#include <stdlib.h> // getenv()
#include <string.h> // strerror()
#include <sys/ioctl.h> // ioctl()
#include <sys/stat.h> // open()
#include <sys/types.h> // open()
#include <time.h> // time_t, time(), struct timeval
#include <unistd.h> // close(), select()

#include <linux/cdrom.h> // CD/DVD based ioctl() and defines.

#define CDVDdefs
#include "PS2Edefs.h"

#include "CDVD.h"
#include "misc.h"
#include "CD.h"
#include "DVD.h"
#include "device.h"


// Globals

CDVDconf conf;

int devicehandle; // File Handle for the device/drive
s32 devicecapability; // Capability Flags
time_t lasttime; // Time marker (in case something gets called too often)
s32 traystatus; // Is the CD/DVD tray open?

s32 disctype; // Type of disc in drive (Video DVD, PSX CD, etc.)
u8 tocbuffer[1024];


// Called by DeviceOpen(), DeviceGetDiskType()
void InitDisc() {
  int i;

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DiscInit()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */

  if((disctype == CDVD_TYPE_PS2DVD) || 
     (disctype == CDVD_TYPE_DVDV)) {
    InitDVDInfo();
  } // ENDIF- Clean out DVD Disc Info?

  if((disctype == CDVD_TYPE_PS2CD) || 
     (disctype == CDVD_TYPE_PS2CDDA) ||
     (disctype == CDVD_TYPE_PSCD) ||
     (disctype == CDVD_TYPE_PSCDDA) ||
     (disctype == CDVD_TYPE_CDDA)) {
    InitCDInfo();
  } // ENDIF- Clean out DVD Disc Info?

  disctype = CDVD_TYPE_NODISC;
  for(i = 0; i > sizeof(tocbuffer); i++)  tocbuffer[i] = 0x00;
} // END InitDisc()


s32 DiscInserted() {
  if(devicehandle == -1)  return(-1);
  if(traystatus == CDVD_TRAY_OPEN)  return(-1);
  if(disctype == CDVD_TYPE_ILLEGAL)  return(-1);
  // if(disctype == CDVD_TYPE_UNKNOWN)  return(-1); // Hmm. Let this one through?
  if(disctype == CDVD_TYPE_DETCTDVDD)  return(-1);
  if(disctype == CDVD_TYPE_DETCTDVDS)  return(-1);
  if(disctype == CDVD_TYPE_DETCTCD)  return(-1);
  if(disctype == CDVD_TYPE_DETCT)  return(-1);
  if(disctype == CDVD_TYPE_NODISC)  return(-1);

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DiscInserted()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */

  return(0);
} // END DiscInserted()


// Called by DeviceTrayStatus() and CDVDopen()
s32 DeviceOpen() {
  // s32 s32result;

  errno = 0;

  if(devicehandle != -1) {
#ifdef VERBOSE_FUNCTION_DEVICE
    printf("CDVD device:   Device already open\n");
#endif /* VERBOSE_FUNCTION_DEVICE */
    return(0);
  } // ENDIF- Is the CD/DVD already in use? That's fine.

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceOpen()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */

  LoadConf();

  devicehandle = open(conf.devicename, O_RDONLY | O_NONBLOCK);
  if(devicehandle == -1) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD device:   Error opening device: %i:%s\n", errno, strerror(errno));
#endif /* VERBOSE_WARNINGS */
    return(-1);
  } // ENDIF- Failed to open device? Abort

  // Note: Hmm. Need a minimum capability in case this fails?
  devicecapability = ioctl(devicehandle, CDROM_GET_CAPABILITY);
  if(errno != 0) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD device:   Error getting device capabilities: %i:%s\n", errno, strerror(errno));
#endif /* VERBOSE_WARNINGS */
    close(devicehandle);
    devicehandle = -1;
    devicecapability = 0;
    return(-1);
  } // ENDIF- Can't read drive capabilities? Close and Abort.

#ifdef VERBOSE_DISC_TYPE
  printf("CDVD device: Device Type(s)  ");
  if(devicecapability < CDC_CD_R)  printf(" CD");
  if(devicecapability & CDC_CD_R)  printf(" CD-R");
  if(devicecapability & CDC_CD_RW)  printf(" CD-RW");
  if(devicecapability & CDC_DVD)  printf(" DVD");
  if(devicecapability & CDC_DVD_R)  printf(" DVD-R");
  if(devicecapability & CDC_DVD_RAM)  printf(" DVD-RAM");
  printf("\n");
#endif /* VERBOSE_DISC_TYPE */
#ifdef VERBOSE_DISC_INFO
  printf("CDVD device: Device Capabilities:\n");
  if(devicecapability & CDC_CLOSE_TRAY)  printf("CDVD device:   Can close a tray\n");
  if(devicecapability & CDC_OPEN_TRAY)  printf("CDVD device:   Can open a tray\n");
  // if(devicecapability & CDC_LOCK)  printf("CDVD device:   Can lock the drive door\n");
  if(devicecapability & CDC_SELECT_SPEED)  printf("CDVD device:   Can change spin speed\n");
  // if(devicecapability & CDC_SELECT_DISC)  printf("CDVD device:   Can change disks (multi-disk tray)\n");
  // if(devicecapability & CDC_MULTI_SESSION)  printf("CDVD device:   Can read multi-session disks\n");
  // if(devicecapability & CDC_MCN)  printf("CDVD device:   Can read Medium Catalog Numbers (maybe)\n");
  if(devicecapability & CDC_MEDIA_CHANGED)  printf("CDVD device:   Can tell if the disc was changed\n");
  if(devicecapability & CDC_PLAY_AUDIO)  printf("CDVD device:   Can play audio disks\n");
  // if(devicecapability & CDC_RESET)  printf("CDVD device:   Can reset the device\n");
  if(devicecapability & CDC_IOCTLS)  printf("CDVD device:   Odd IOCTLs. Not sure of compatability\n");
  if(devicecapability & CDC_DRIVE_STATUS)  printf("CDVD device:   Can monitor the drive tray\n");

#endif /* VERBOSE_DISC_INFO */

  // if((devicecapability & CDC_SELECT_SPEED) && (conf.drivespeed > 0)) {
  //   s32result = 0;
  //   s32result = ioctl(devicehandle, CDROM_SELECT_SPEED, &conf.drivespeed);
#ifdef VERBOSE_WARNINGS
  //   if((s32result != 0) || (errno != 0)) {
  //     printf("CDVD device:   Could not adjust disc speed!\n");
  //     printf("CDVD device:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
  //   } // ENDIF- Report an invalid speed reference?
#endif /* VERBOSE_WARNINGS */
  //   errno = 0;
  // } // ENDIF- Can we tell the device it's maximum speed? Then do so.

  InitDisc();
  traystatus = CDVD_TRAY_OPEN; // Start with Tray Open
  DeviceTrayStatus(); // Now find out for sure.

  return(0); // Device opened and ready for use.
} // END DeviceOpen()


// Called by DeviceTrayStatus(), CDVDclose(), and CDVDshutdown()
void DeviceClose() {
  // s32 s32result;
  int zerospeed;
  
  zerospeed = 0;

  if(devicehandle == -1) {
#ifdef VERBOSE_FUNCTION_DEVICE
    printf("CDVD device:   Device already closed\n");
#endif /* VERBOSE_FUNCTION_DEVICE */
    return;
  } // ENDIF- Device already closed? Ok.

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceClose()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */

  // if((devicecapability & CDC_SELECT_SPEED) && (conf.drivespeed > 0)) {
  //   s32result = ioctl(devicehandle, CDROM_SELECT_SPEED, &zerospeed);
#ifdef VERBOSE_WARNINGS
  //   if((s32result != 0) || (errno != 0)) {
  //     printf("CDVD device:   Could not release disc speed!\n");
  //     printf("CDVD device:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
  //   } // ENDIF- Report an invalid speed reference?
#endif /* VERBOSE_WARNINGS */
    // ioctl(devicehandle, CDROMRESET); // Removed as unnecessary?
  //   errno = 0;
  // } // ENDIF- Can we tell the device it's maximum speed? Reset speed to max.

  InitDisc();
  close(devicehandle);
  devicehandle = -1;
  devicecapability = 0;
  return;
} // END CDVDclose()


s32 DeviceReadTrack(u32 lsn, int mode, u8 *buffer) {
  s32 s32result;

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceReadTrack(%i)\n", lsn);
#endif /* VERBOSE_FUNCTION_DEVICE */

  if(DiscInserted() == -1)  return(-1);

  // Get that data
  if((disctype == CDVD_TYPE_PS2DVD) || (disctype == CDVD_TYPE_DVDV)) {
    s32result = DVDreadTrack(lsn, mode, buffer);
  } else {
    s32result = CDreadTrack(lsn, mode, buffer);
  } //ENDIF- Read a DVD sector or a CD sector?

  return(s32result);
} // END DeviceReadTrack()


s32 DeviceBufferOffset() {
#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceReadTrack(%i)\n", lsn);
#endif /* VERBOSE_FUNCTION_DEVICE */

  if(DiscInserted() == -1)  return(-1);

  if((disctype == CDVD_TYPE_PS2DVD) || (disctype == CDVD_TYPE_DVDV)) {
    return(0);
  } else {
    return(CDgetBufferOffset());
  } // ENDIF- Is this a DVD?
} // END DeviceBufferOffset()

// Called by DeviceTrayStatus()
s32 DeviceGetDiskType() {
  s32 s32result;
  s32 ioctldisktype;
  
  errno = 0;

  if(devicehandle == -1) {
    return(-1);
  } // ENDIF- Someone forget to open the device?

  if(traystatus == CDVD_TRAY_OPEN) {
    return(disctype);
  } // ENDIF- Is the device tray open? No disc to check yet.

  if(disctype != CDVD_TYPE_NODISC) {
    return(disctype);
  } // ENDIF- Already checked? Drive still closed? Disc hasn't changed.

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceGetDiskType()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */
  disctype = CDVD_TYPE_DETCT;

  ioctldisktype = ioctl(devicehandle, CDROM_DISC_STATUS);
  if(errno != 0) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD device:   Trouble reading Disc Type!\n");
    printf("CDVD device:     Error: %i:%s\n", errno, strerror(errno));
#endif /* VERBOSE_WARNINGS */
    disctype = CDVD_TYPE_UNKNOWN;
    return(disctype);
  } // ENDIF- Trouble probing for a disc?

  s32result = CDgetDiskType(ioctldisktype);
  if(s32result != -1) {
    return(disctype);
  } // ENDIF- Did we find a disc type?

  s32result = DVDgetDiskType(ioctldisktype);
  if(s32result != -1) {
    return(disctype);
  } // ENDIF- Did we find a disc type?

  disctype = CDVD_TYPE_UNKNOWN; // Not a CD? Not a DVD? Is is peanut butter?
  return(disctype);
} // END CDVDgetDiskType()


// Called by PollLoop() and CDVDgetTrayStatus()
s32 DeviceTrayStatus() {
  s32 s32result;

  errno = 0;

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceTrayStatus()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */

  if(devicehandle == -1) {
    return(-1);
  } // ENDIF- Someone forget to open the device?

  if((devicecapability & CDC_DRIVE_STATUS) != 0) {
    s32result = ioctl(devicehandle, CDROM_DRIVE_STATUS);
    if(s32result < 0) {
#ifdef VERBOSE_WARNINGS
      printf("CDVD device:   Trouble reading Drive Status!\n");
      printf("CDVD device:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
#endif /* VERBOSE_WARNINGS */
      s32result = CDS_TRAY_OPEN;
    } // ENDIF- Failure to get status? Assume it's open.
    errno = 0;

  } else {
    s32result = ioctl(devicehandle, CDROM_DISC_STATUS);
    if(errno != 0) {
#ifdef VERBOSE_WARNINGS
      printf("CDVD device:   Trouble detecting Disc Status presense!\n");
      printf("CDVD device:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
#endif /* VERBOSE_WARNINGS */
      s32result = CDS_TRAY_OPEN;
      errno = 0;
    } // ENDIF- Trouble?
    if(s32result == CDS_NO_DISC) {
      s32result = CDS_TRAY_OPEN;
    } // ENDIF- Is there no disc in the device? Guess the tray is open
  } // ENDIF- Can we poll the tray directly? (Or look at disc status instead?)

  if(s32result == CDS_TRAY_OPEN) {
    traystatus = CDVD_TRAY_OPEN;
    if(disctype != CDVD_TYPE_NODISC) {
      DeviceClose();
      DeviceOpen();
      // InitDisc(); // Called by DeviceOpen()
    } // ENDIF- Tray just opened... clear disc info
  } else {
    traystatus = CDVD_TRAY_CLOSE;
    if(disctype == CDVD_TYPE_NODISC) {
      DeviceGetDiskType();
    } // ENDIF- Tray just closed? Get disc information
  } // ENDIF- Do we detect an open tray?
  return(traystatus);
} // END CDVD_getTrayStatus()


s32 DeviceTrayOpen() {
  s32 s32result;

  errno = 0;

  if(devicehandle == -1) {
    return(-1);
  } // ENDIF- Someone forget to open the device?

  if((devicecapability & CDC_OPEN_TRAY) == 0) {
    return(-1);
  } // ENDIF- Don't have open capability? Error out.

  // Tray already open? Exit.
  if(traystatus == CDVD_TRAY_OPEN)  return(0);

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceTrayOpen()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */

  s32result = ioctl(devicehandle, CDROMEJECT);
#ifdef VERBOSE_WARNINGS
  if((s32result != 0) || (errno != 0)) {
    printf("CDVD device:   Could not open the tray!\n");
    printf("CDVD device:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
  } // ENDIF- Trouble?
#endif /* VERBOSE_WARNINGS */
  TrueSleep(1, 0);
  return(s32result);
} // END DeviceTrayOpen()


s32 DeviceTrayClose() {
  s32 s32result;

  errno = 0;

  if(devicehandle == -1) {
    return(-1);
  } // ENDIF- Someone forget to open the device?

  if((devicecapability & CDC_CLOSE_TRAY) == 0) {
    return(-1);
  } // ENDIF- Don't have close capability? Error out.

  // Tray already closed? Exit.
  if(traystatus == CDVD_TRAY_CLOSE)  return(0);

#ifdef VERBOSE_FUNCTION_DEVICE
  printf("CDVD device: DeviceTrayClose()\n");
#endif /* VERBOSE_FUNCTION_DEVICE */

  s32result = ioctl(devicehandle, CDROMCLOSETRAY);
#ifdef VERBOSE_WARNINGS
  if((s32result != 0) || (errno != 0)) {
    printf("CDVD device:   Could not close the tray!\n");
    printf("CDVD device:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
  } // ENDIF- Trouble?
#endif /* VERBOSE_WARNINGS */
  TrueSleep(1, 0);
  return(s32result);
} // END DeviceTrayClose()


// Configuration functions

void LoadConf() {
  FILE *f;
  char cfg[255];

  sprintf(cfg, "%s/.PS2E/CDVDlinuz.cfg", getenv("HOME"));
  f = fopen(cfg, "r");
  if (f == NULL) {
    return;
  }

  fscanf(f, "Dev = %s\n", conf.devicename);
  fscanf(f, "ReadMode = %d\n", &conf.readmode);
  // fscanf(f, "DriveSpeed = %d\n", &conf.drivespeed);
  fclose(f);
}


void SaveConf() {
  FILE *f;
  char cfg[255];

  sprintf (cfg, "%s/.PS2E", getenv("HOME"));
  mkdir(cfg, 0755);

  sprintf(cfg, "%s/.PS2E/CDVDlinuz.cfg", getenv("HOME"));
  f = fopen(cfg, "w");
  if (f == NULL)  return;

  fprintf(f, "Dev = %s\n", conf.devicename);
  fprintf(f, "ReadMode = %d\n", conf.readmode);
  // fprintf(f, "DriveSpeed = %d\n", conf.drivespeed);
  fclose(f);
}
