/*  CDVD.c
 *  Copyright (C) 2002-2005  CDVDlinuz Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <errno.h> // errno
#include <fcntl.h> // open()
#include <pthread.h> // pthread_create(), pthread_exit(), pthread_join()
#include <stddef.h> // NULL
#include <stdio.h> // printf()
#include <stdlib.h> // getenv()
#include <string.h> // strerror()
#include <sys/ioctl.h> // ioctl()
#include <sys/stat.h> // open()
#include <sys/types.h> // open()
#include <time.h> // time_t, time(), struct timeval
#include <unistd.h> // close(), select()

#define CDVDdefs
#include "PS2Edefs.h"

#include "CDVD.h"
#include "misc.h"
#include "buffer.h"
#include "CD.h" // InitCDInfo()
#include "DVD.h" // InitDVDInfo()
#include "device.h"


// Globals
const unsigned char version = PS2E_CDVD_VERSION;
const unsigned char revision = 0;
const unsigned char build = 7;
static char *LibName = "EFP-modified Linuzappz CDVD Driver";


// Data seen by both threads

pthread_t pollthread;
pthread_attr_t pollthreadattr;
pthread_mutex_t pollthreadmutex; // Disc Status Protection

pthread_t readthread;
pthread_attr_t readthreadattr;
pthread_mutex_t readthreadmutex; // Read Buffer Protection

pthread_mutex_t main1threadmutex; // Loop Gate for ReadLoop from interface
pthread_mutex_t main2threadmutex; // Makes sure main1 mutex is ready for capture

int readmode; // Current mode flag
s32 trayrequest; // Flag the driver open/close the tray?

u32 cachehintlsn;
int cachehintmode;
u32 cachehintcount;


void *PollLoop(void *garbage) {
#ifdef VERBOSE_FUNCTION
  printf("CDVD Poll Loop started\n");
#endif /* VERBOSE FUNCTION */

  while(readmode == READMODE_NORMAL)  TrueSleep(1, 0);

  while(readmode == READMODE_THREADED) {
    // Lock the read thread out while polling...
#ifdef VERBOSE_MUTEX
    printf("CDVD Poll loop: Locks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_lock(&pollthreadmutex);
    if(trayrequest != 0xff) {
      if(trayrequest == CDVD_TRAY_OPEN) {
        DeviceTrayOpen();
      } else {
        DeviceTrayClose();
      } // ENDIF- Are we trying to open (or close) the tray?
      errno = 0;
      trayrequest = 0xff;
    } // ENDIF- Is there a request to open/close the tray?

    DeviceTrayStatus();
    errno = 0;

#ifdef VERBOSE_MUTEX
    printf("CDVD Poll loop: Unlocks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_unlock(&pollthreadmutex);

    TrueSleep(1, 0); // Don't want to poll too often. 1/sec should be okay

    pthread_testcancel();
  } // ENDWHILE- we still need a loop

#ifdef VERBOSE_FUNCTION
  printf("CDVD Poll Loop stopped\n");
#endif /* VERBOSE FUNCTION */
  pthread_exit(NULL);
} // END PollLoop()


void *ReadLoop(void *garbage) {
  s32 s32result;
  u32 tempbuffer;

#ifdef VERBOSE_FUNCTION
  printf("CDVD Read Loop started\n");
#endif /* VERBOSE FUNCTION */

#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
  pthread_mutex_lock(&pollthreadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks Read mutex\n");
#endif /* VERBOSE_MUTEX */
  pthread_mutex_lock(&readthreadmutex);

  while(readmode == READMODE_NORMAL)  TrueSleep(1, 0);

  while(readmode == READMODE_THREADED) {
    // Let the poll thread loop once... just in case.
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_unlock(&pollthreadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_lock(&pollthreadmutex);

    // Hinted at cache read requests
    if(cachehintcount > 0) {
      s32result = 0;
      tempbuffer = FindListBuffer(cachehintlsn);
      if((tempbuffer >= BUFFERMAX) || 
         (bufferlist[userbuffer].lsn != cachehintlsn) ||
         (bufferlist[userbuffer].mode != cachehintmode)) {
        replacebuffer++;
        if(replacebuffer >= BUFFERMAX)  replacebuffer = 0;
        if(userbuffer == replacebuffer) {
          replacebuffer++;
          if(replacebuffer >= BUFFERMAX)  replacebuffer = 0;
        } // ENDIF- Did we accidently point at the user buffer? Skip 1 more

        if(bufferlist[replacebuffer].upsort != 0xffff) {
          RemoveListBuffer(replacebuffer);
        } // ENDIF- Is there a refernce to a sorted entry? Remove the sort marker.

        s32result = DeviceReadTrack(cachehintlsn, 
                                    cachehintmode,
                                    bufferlist[replacebuffer].buffer);
        bufferlist[replacebuffer].lsn = cachehintlsn;
        bufferlist[replacebuffer].mode = cachehintmode;
        bufferlist[replacebuffer].offset = DeviceBufferOffset();
        if((s32result != 0) || (errno != 0)) {
          bufferlist[replacebuffer].mode = -1; // Error! flag buffer as such.
        } else {
          if((disctype != CDVD_TYPE_PS2DVD) && (disctype != CDVD_TYPE_DVDV)) {
            if(cachehintmode == CDVD_MODE_2352) {
              CDreadSubQ(cachehintlsn, &bufferlist[replacebuffer].subq);
              errno = 0;
            } // ENDIF- Read subq as well?
          } // ENDIF- Read a DVD buffer or a CD buffer?
        } // ENDIF-Read ok? Fill out rest of buffer info.
        AddListBuffer(replacebuffer);
      } // ENDIF- Is this record not already in the cache?

      cachehintlsn++;
      if(s32result == 0) {
        cachehintcount--;
      } else {
        cachehintcount = 0;
      } // ENDIF- Good sector read? Prep for next cache sector
      errno = 0;
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks Read mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&readthreadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks Read mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&readthreadmutex);
    } // ENDIF- Get another hinted-at sector?

    if(cachehintcount == 0) {
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&pollthreadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks Read mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&readthreadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks interface mutex #1\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&main1threadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks interface mutex #1\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&main1threadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks interface mutex #2\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&main2threadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks interface mutex #2\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&main2threadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks Read mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&readthreadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Locks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&pollthreadmutex);
    } // ENDIF- Out of sectors to read? Freeze until count changes

    pthread_testcancel();
  } // ENDWHILE- Not told to close down yet...

#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks Read mutex\n");
#endif /* VERBOSE_MUTEX */
  pthread_mutex_unlock(&readthreadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD Read loop: Unlocks Poll mutex\n");
#endif /* VERBOSE_MUTEX */
  pthread_mutex_unlock(&pollthreadmutex);

#ifdef VERBOSE_FUNCTION
  printf("CDVD Read Loop stopped\n");
#endif /* VERBOSE FUNCTION */
  pthread_exit(NULL);
} // END DriverLoop()


u32 CALLBACK PS2EgetLibType() {
  return(PS2E_LT_CDVD); // Library Type CDVD
} // END PS2EgetLibType()


u32 CALLBACK PS2EgetLibVersion2(u32 type) {
  return((version<<16)|(revision<<8)|build);
} // END PS2EgetLibVersion2()


char* CALLBACK PS2EgetLibName() {
  return(LibName);
} // END PS2EgetLibName()


s32 CALLBACK CDVDinit() {
  int i;
  const char defaultdevice[] = DEFAULT_DEVICE;

  errno = 0;

#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDinit()\n");
#endif /* VERBOSE_FUNCTION */

  i = 0;
  while((i < 255) && defaultdevice[i] != 0) {
    conf.devicename[i] = defaultdevice[i];
    i++;
  } // ENDWHILE- copying the default CD/DVD name in
  conf.devicename[i] = 0; // 0-terminate the device name
  conf.readmode = READMODE_NORMAL;
  conf.drivespeed = 0; // Don't change the speed...

  devicehandle = -1;
  devicecapability = 0;
  lasttime = time(NULL);

  readmode = READMODE_NORMAL;

  // Initialize DVD.c and CD.c as well
  InitDisc();
  InitDVDInfo();
  InitCDInfo();

  return(0);
} // END CDVDinit()


void CALLBACK CDVDshutdown() {
  s32 s32result;

  errno = 0;

#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDshutdown()\n");
#endif /* VERBOSE_FUNCTION */

  if(readmode == READMODE_THREADED) {
    readmode = READMODE_NORMAL;
#ifdef VERBOSE_MUTEX
    printf("CDVD interface: Unlocks interface mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_unlock(&main1threadmutex);
    pthread_mutex_unlock(&main2threadmutex);
    pthread_cancel(readthread);
    pthread_cancel(pollthread);
    errno = 0;
    s32result = pthread_join(readthread, NULL);
#ifdef VERBOSE_WARNINGS
    if(s32result != 0) {
      printf("CDVD interface:   Trouble closing down Read Thread!\n");
      printf("CDVD interface:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
    } // ENDIF- Trouble?
#endif /* VERBOSE_WARNINGS */
    errno = 0;
    s32result = pthread_join(pollthread, NULL);
#ifdef VERBOSE_WARNINGS
    if(s32result != 0) {
      printf("CDVD interface:   Trouble closing down Poll Thread!\n");
      printf("CDVD interface:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
    } // ENDIF- Trouble?
#endif /* VERBOSE_WARNINGS */

    pthread_attr_destroy(&readthreadattr);
    pthread_attr_destroy(&pollthreadattr);
    pthread_mutex_destroy(&readthreadmutex);
    pthread_mutex_destroy(&pollthreadmutex);
    pthread_mutex_destroy(&main1threadmutex);
    pthread_mutex_destroy(&main2threadmutex);
  } // ENDIF- Kill the Driver thread

  DeviceClose();
} // END CDVDshutdown()


s32 CALLBACK CDVDopen() {
  s32 s32result;

  errno = 0;
  InitBuffer();
  cachehintlsn = 0xffffffff;
  cachehintmode = 0;
  cachehintcount = 0;

#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDopen()\n");
#endif /* VERBOSE_FUNCTION */

  s32result = DeviceOpen();
  if((s32result != 0) || (errno != 0))  return(s32result);

  if(conf.readmode == READMODE_NORMAL)  return(0);

  if(readmode == READMODE_THREADED) {
      printf("CDVD interface:   Internal Error (Lost Threads)\n");
      printf("CDVD interface:   Please Restart.\n");
      return(-1);
  } // ENDIF

  trayrequest = 0xff; // Clear tray requests

  pthread_mutex_init(&main1threadmutex, NULL); // Default of 'fast' locks
  pthread_mutex_init(&main2threadmutex, NULL);
  pthread_mutex_init(&pollthreadmutex, NULL);
  pthread_mutex_init(&readthreadmutex, NULL);

  pthread_attr_init(&pollthreadattr);
  pthread_attr_setdetachstate(&pollthreadattr, PTHREAD_CREATE_JOINABLE);
  s32result = pthread_create(&pollthread, 
                             &pollthreadattr,
                             PollLoop,
                             NULL);
  // if((s32result != 0) || (errno != 0)) {
  if(s32result != 0) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD interface:   Trouble starting up Poll Thread!\n");
    printf("CDVD interface:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
#endif /* VERBOSE_WARNINGS */
    return(-1);
  } // ENDIF- Trouble?
  errno = 0;

  pthread_attr_init(&readthreadattr);
  pthread_attr_setdetachstate(&readthreadattr, PTHREAD_CREATE_JOINABLE);
  s32result = pthread_create(&readthread, 
                             &readthreadattr,
                             ReadLoop,
                             NULL);
  // if((s32result != 0) || (errno != 0)) {
  if(s32result != 0) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD interface:   Trouble starting up Read Thread!");
    printf("CDVD interface:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
#endif /* VERBOSE_WARNINGS */
    return(-1);
  } // ENDIF- Trouble?
  errno = 0;

#ifdef VERBOSE_MUTEX
    printf("CDVD interface: Locks interface mutex #1\n");
#endif /* VERBOSE_MUTEX */
  pthread_mutex_lock(&main1threadmutex);
#ifdef VERBOSE_MUTEX
    printf("CDVD interface: Locks interface mutex #2\n");
#endif /* VERBOSE_MUTEX */
  pthread_mutex_lock(&main2threadmutex);
  TrueSleep(1, 0); // Small pause to make sure read loop locks are in place
  readmode = READMODE_THREADED;
  TrueSleep(2, 0); // Let's the Read Loop mutex lock, waiting for a sector call

  return(0);
} // END CDVDopen();


void CALLBACK CDVDclose() {
  s32 s32result;

  errno = 0;

#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDclose()\n");
#endif /* VERBOSE_FUNCTION */

  if(readmode == READMODE_THREADED) {
    readmode = READMODE_NORMAL;
#ifdef VERBOSE_MUTEX
    printf("CDVD interface: Unlocks interface mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_unlock(&main1threadmutex);
    pthread_mutex_unlock(&main2threadmutex);
    pthread_cancel(readthread);
    pthread_cancel(pollthread);
    errno = 0;
    s32result = pthread_join(readthread, NULL);
#ifdef VERBOSE_WARNINGS
    if(s32result != 0) {
      printf("CDVD interface:   Trouble closing down Read Thread!\n");
      printf("CDVD interface:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
    } // ENDIF- Trouble?
#endif /* VERBOSE_WARNINGS */
    errno = 0;
    s32result = pthread_join(pollthread, NULL);
#ifdef VERBOSE_WARNINGS
    if(s32result != 0) {
      printf("CDVD interface:   Trouble closing down Poll Thread!\n");
      printf("CDVD interface:     Error: (%i) %i:%s\n", s32result, errno, strerror(errno));
    } // ENDIF- Trouble?
#endif /* VERBOSE_WARNINGS */

    pthread_attr_destroy(&readthreadattr);
    pthread_attr_destroy(&pollthreadattr);
    pthread_mutex_destroy(&readthreadmutex);
    pthread_mutex_destroy(&pollthreadmutex);
    pthread_mutex_destroy(&main1threadmutex);
    pthread_mutex_destroy(&main2threadmutex);
  } // ENDIF- Kill the Driver thread

  DeviceClose();
} // END CDVDclose()


s32 CALLBACK CDVDreadTrack(u32 lsn, int mode) {
  s32 s32result;

#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDreadTrack(%i)\n", lsn);
#endif /* VERBOSE_FUNCTION */

  s32result = 0;
  errno = 0;

  if(DiscInserted() == -1)  return(-1);

  if(userbuffer < BUFFERMAX) {
    if((bufferlist[userbuffer].lsn == lsn) &&
       (bufferlist[userbuffer].mode == mode)) {
      return(0);
    } // ENDIF- And it's the right one?
  } // ENDIF- Are we already pointing at the buffer?

  if(readmode == READMODE_THREADED) {
#ifdef VERBOSE_MUTEX
    printf("CDVD interface: Locks Read mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_lock(&readthreadmutex);
  } // ENDIF- Threaded? Lock the Read thread

  userbuffer = FindListBuffer(lsn);
  if(userbuffer < BUFFERMAX) {
    if((bufferlist[userbuffer].lsn == lsn) &&
       (bufferlist[userbuffer].mode == mode)) {
      if((cachehintlsn >= lsn) &&
         (cachehintmode == mode) &&
         (cachehintlsn - lsn + cachehintcount == READ_AHEAD_BUFFERS - 1)) {
#ifdef VERBOSE_MUTEX
        printf("CDVD interface:   Hint: %u - Current: %u + Count: %u = 7? Inc.\n",
                cachehintlsn, lsn, cachehintcount);
#endif /* VERBOSE_MUTEX */
        cachehintcount++;
      } else {
#ifdef VERBOSE_MUTEX
        printf("CDVD interface:   Hint: %u - Current: %u + Count: %u = 7? Flush\n",
               cachehintlsn, lsn, cachehintcount);
#endif /* VERBOSE_MUTEX */
        cachehintlsn = lsn;
        cachehintmode = mode;
        cachehintcount = READ_AHEAD_BUFFERS;
      } // ENDIF- Could we just ask for one more record (in the cache)?
      if(readmode == READMODE_THREADED) {
#ifdef VERBOSE_MUTEX
        printf("CDVD interface: Unlocks Read mutex\n");
#endif /* VERBOSE_MUTEX */
        pthread_mutex_unlock(&readthreadmutex);
#ifdef VERBOSE_MUTEX
        printf("CDVD interface: Unlocks interface mutex #1\n");
#endif /* VERBOSE_MUTEX */
        pthread_mutex_unlock(&main1threadmutex);
#ifdef VERBOSE_MUTEX
        printf("CDVD interface: Locks interface mutex #1\n");
#endif /* VERBOSE_MUTEX */
        pthread_mutex_lock(&main1threadmutex);
#ifdef VERBOSE_MUTEX
        printf("CDVD interface: Unlocks interface mutex #2\n");
#endif /* VERBOSE_MUTEX */
        pthread_mutex_unlock(&main2threadmutex);
#ifdef VERBOSE_MUTEX
        printf("CDVD interface: Locks interface mutex #2\n");
#endif /* VERBOSE_MUTEX */
        pthread_mutex_lock(&main2threadmutex);
      } // ENDIF- Signal the Read Thread to update the cache?
      return(0);
    } // ENDIF- And it was the right one?
  } // ENDIF- Was a buffer found in the cache?

  if(readmode == READMODE_THREADED) {
#ifdef VERBOSE_MUTEX
    printf("CDVD interface:   Hint: %u - Current: %u + Count: %u = 7? Reset\n",
           cachehintlsn, lsn, cachehintcount);
#endif /* VERBOSE_MUTEX */
    cachehintlsn = lsn;
    cachehintmode = mode;
    cachehintcount = READ_AHEAD_BUFFERS;

    while(cachehintlsn == lsn) {
#ifdef VERBOSE_MUTEX
      printf("CDVD interface: Unlocks interface mutex #1\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&main1threadmutex);
#ifdef VERBOSE_MUTEX
      printf("CDVD interface: Locks interface mutex #1\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&main1threadmutex);
#ifdef VERBOSE_MUTEX
      printf("CDVD interface: Unlocks interface mutex #2\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&main2threadmutex);
#ifdef VERBOSE_MUTEX
      printf("CDVD interface: Locks interface mutex #2\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&main2threadmutex);
#ifdef VERBOSE_MUTEX
      printf("CDVD interface: Unlocks Read mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_unlock(&readthreadmutex);
#ifdef VERBOSE_MUTEX
      printf("CDVD interface: Locks Read mutex\n");
#endif /* VERBOSE_MUTEX */
      pthread_mutex_lock(&readthreadmutex);
    } // ENDWHILE- Trying several times to get 1 record
    userbuffer = FindListBuffer(lsn);

#ifdef VERBOSE_MUTEX
    printf("CDVD interface: Unlocks Read mutex\n");
#endif /* VERBOSE_MUTEX */
    pthread_mutex_unlock(&readthreadmutex);

    if(userbuffer == 0xffff) {
#ifdef VERBOSE_WARNINGS
      printf("CDVD interface:   Haven't read in the record!\n");
#endif /* VERBOSE_WARNINGS */
      return(-1); // No buffer reference?
    } // ENDIF- user buffer not pointing at anything? Abort

    if(bufferlist[userbuffer].mode < 0) {
#ifdef VERBOSE_WARNINGS
      printf("CDVD interface:   Physical error reading sector!\n");
#endif /* VERBOSE_WARNINGS */
      return(-1); // Bad Sector?
    } // ENDIF- Trouble reading physical sector? Tell them.

    return(0);

  } else {
    replacebuffer++;
    if(replacebuffer >= BUFFERMAX)  replacebuffer = 0;
    userbuffer = replacebuffer;

    if(bufferlist[replacebuffer].upsort != 0xffff) {
      RemoveListBuffer(replacebuffer);
    } // ENDIF- Reference already in place? Remove it.

    s32result = DeviceReadTrack(lsn, mode, bufferlist[replacebuffer].buffer);
    bufferlist[replacebuffer].lsn = lsn;
    bufferlist[replacebuffer].mode = mode;
    bufferlist[replacebuffer].offset = DeviceBufferOffset();

    if((s32result != 0) || (errno != 0)) {
      bufferlist[replacebuffer].mode = -1; // Error! flag buffer as such.
    } else {
      if((disctype != CDVD_TYPE_PS2DVD) && (disctype != CDVD_TYPE_DVDV)) {
        if(mode == CDVD_MODE_2352) {
          CDreadSubQ(lsn, &bufferlist[replacebuffer].subq);
          errno = 0;
        } // ENDIF- Read subq as well?
      } // ENDIF- Read a DVD buffer or a CD buffer?
    } // ENDIF-Read ok? Fill out rest of buffer info.
    AddListBuffer(replacebuffer);
    return(s32result);
  } // ENDIF- Are we in a threaded state?
} // END CDVDreadTrack()


u8* CALLBACK CDVDgetBuffer() {
#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDgetBuffer()\n");
#endif /* VERBOSE_FUNCTION */

  if(DiscInserted() == -1)  return(NULL);

  if(userbuffer == 0xffff) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD interface:   Not pointing to a buffer!\n");
#endif /* VERBOSE_WARNINGS */
    return(NULL); // No buffer reference?
  } // ENDIF- user buffer not pointing at anything? Abort

  if(bufferlist[userbuffer].mode < 0) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD interface:   Error in buffer!\n");
#endif /* VERBOSE_WARNINGS */
    return(NULL); // Bad Sector?
  } // ENDIF- Trouble reading physical sector? Tell them.

  return(bufferlist[userbuffer].buffer + bufferlist[userbuffer].offset);
} // END CDVDgetBuffer()


s32 CALLBACK CDVDreadSubQ(u32 lsn, cdvdSubQ *subq) {
#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDreadSubQ(%i)\n", lsn);
#endif /* VERBOSE_FUNCTION */

  if(DiscInserted() == -1)  return(-1);

  // DVDs don't have SubQ data
  if(disctype == CDVD_TYPE_PS2DVD)  return(-1);
  if(disctype == CDVD_TYPE_DVDV)  return(-1);

  return(CDreadSubQ(lsn, subq));
} // END CDVDreadSubQ()


s32 CALLBACK CDVDgetTN(cdvdTN *cdvdtn) {
#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDgetTN()\n");
#endif /* VERBOSE_FUNCTION */

  if(DiscInserted() == -1)  return(-1);

  if((disctype == CDVD_TYPE_PS2DVD) || (disctype == CDVD_TYPE_DVDV)) {
    return(DVDgetTN(cdvdtn));
  } else {
    return(CDgetTN(cdvdtn));
  } // ENDIF- Are we looking at a DVD?
} // END CDVDgetTN()


s32 CALLBACK CDVDgetTD(u8 newtrack, cdvdTD *cdvdtd) {
#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDgetTD()\n");
#endif /* VERBOSE_FUNCTION */

  if(DiscInserted() == -1)  return(-1);

  if((disctype == CDVD_TYPE_PS2DVD) || (disctype == CDVD_TYPE_DVDV)) {
    return(DVDgetTD(newtrack, cdvdtd));
  } else {
    return(CDgetTD(newtrack, cdvdtd));
  } // ENDIF- Are we looking at a DVD?
} // END CDVDgetTD()


s32 CALLBACK CDVDgetTOC(void *toc) {
  // A structure to fill in, or at least some documentation on what
  // the PS2 expects from this call would be more helpful than a 
  // "void *".

  union {
    void *voidptr;
    u8 *u8ptr;
  } tocptr;
  s32 i;

#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDgetTOC()\n");
#endif /* VERBOSE_FUNCTION */

  if(toc == NULL)  return(-1);
  if(DiscInserted() == -1)  return(-1);

  tocptr.voidptr = toc;
  for(i = 0; i < 1024; i++)  *(tocptr.u8ptr + i) = tocbuffer[i];
  tocptr.voidptr = NULL;

  return(0);
} // END CDVDgetTOC()


s32 CALLBACK CDVDgetDiskType() {
#ifdef VERBOSE_FUNCTION
  // Called way too often in boot part of bios to be left in.
  // printf("CDVD interface: CDVDgetDiskType()\n");
#endif /* VERBOSE_FUNCTION */

  if(readmode == READMODE_NORMAL) {
    if(lasttime != time(NULL)) {
      lasttime = time(NULL);
      DeviceTrayStatus();
    } // ENDIF- Has enough time passed between calls?
  } // ENDIF- Are we in a non-threaded state?

  return(disctype);
} // END CDVDgetDiskType()


s32 CALLBACK CDVDgetTrayStatus() {
#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDgetTrayStatus()\n");
#endif /* VERBOSE_FUNCTION */

  if(readmode == READMODE_NORMAL) {
    if(lasttime != time(NULL)) {
      lasttime = time(NULL);
      DeviceTrayStatus();
    } // ENDIF- Has enough time passed between calls?
  } // ENDIF- Are we in a non-threaded state?

  return(traystatus);
} // END CDVDgetTrayStatus()


s32 CALLBACK CDVDctrlTrayOpen() {
#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDctrlTrayOpen()\n");
#endif /* VERBOSE_FUNCTION */

  if(readmode == READMODE_THREADED) {
    trayrequest = CDVD_TRAY_OPEN;
    return(0);
  } else {
    return(DeviceTrayOpen());
  } // ENDIF- Are we in a threaded state?
} // END CDVDctrlTrayOpen()


s32 CALLBACK CDVDctrlTrayClose() {
#ifdef VERBOSE_FUNCTION
  printf("CDVD interface: CDVDctrlTrayClose()\n");
#endif /* VERBOSE_FUNCTION */

  if(readmode == READMODE_THREADED) {
    trayrequest = CDVD_TRAY_CLOSE;
    return(0);
  } else {
    return(DeviceTrayClose());
  } // ENDIF- Are we in a threaded state?
} // END CDVDctrlTrayClose()


s32 CALLBACK CDVDtest() {
  s32 s32result;

  errno = 0;

  if(devicehandle != -1) {
#ifdef VERBOSE_WARNINGS
    printf("CDVD interface:   Device already open\n");
#endif /* VERBOSE_WARNINGS */
    return(0);
  } // ENDIF- Is the CD/DVD already in use? That's fine.

#ifdef VERBOSE_FUNCTION
  printf("CDVD driver: CDVDtest()\n");
#endif /* VERBOSE_FUNCTION */

  s32result = DeviceOpen();
  DeviceClose();
  return(s32result);
} // END CDVDtest()
