/* FCE Ultra - NES/Famicom Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Xodnizel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "main.h"
#include "video.h"
#include "opengl.h"
#include "../common/vidblit.h"

#include "icon.h"

SDL_Surface *screen;
SDL_Surface *IconSurface=NULL;

static int curbpp;
static int srendline,erendline;
static int tlines;
static int inited=0;

#ifdef OPENGL
extern int sdlhaveogl;
static int usingogl;
static double exs,eys;
#else
static int exs,eys;
#endif
static int eefx;

#define NWIDTH	(256-((eoptions&EO_CLIPSIDES)?16:0))
#define NOFFSET	(eoptions&EO_CLIPSIDES?8:0)


static int paletterefresh;


void ClearVideoSurfaces(void)
{
 SDL_Surface *scr = screen;
 int y;
 int b = 0;

 #ifdef OPENGL
 if(usingogl)
 {
  ClearGLSurfaces(scr);
  return;
 }
 #endif
 if(scr->flags & SDL_DOUBLEBUF)
  b++;

 do
 {
  if(SDL_MUSTLOCK(scr))
   if(SDL_LockSurface(scr))
    return;

  for(y=0;y<scr->h;y++)
   memset((uint8 *)scr->pixels + scr->pitch * y, 0, scr->w);

  if(SDL_MUSTLOCK(scr))
   SDL_UnlockSurface(scr);

  SDL_UpdateRect(scr, 0, 0, scr->w, scr->h);

  PumpWrap();

  if(scr->flags&SDL_DOUBLEBUF)
   SDL_Flip(scr);
 } while(b--);
}

/* Return 1 if video was killed, 0 otherwise(video wasn't initialized). */
int KillVideo(int subsys)
{
 if(IconSurface)
 {
  SDL_FreeSurface(IconSurface);
  IconSurface=0;
 }

 if(inited&1)
 {
  #ifdef OPENGL
  if(usingogl)
   KillOpenGL();
  else
  #endif
  if(curbpp>8)
   KillBlitToHigh();
  SDL_QuitSubSystem(SDL_INIT_VIDEO);
  inited&=~1;
  return(1);
 }

 inited=0;
 return(0);
}

static int sponge;

int InitVideo(FCEUGI *gi)
{
 const SDL_VideoInfo *vinf;
 int flags=0;

 FCEUI_printf(_("Initializing video..."));
 FCEU_indent(1);

 FCEUI_GetCurrentVidSystem(&srendline,&erendline);

 if(_fullscreen) sponge=Settings.specialfs;
 else sponge=Settings.special;


 #ifdef OPENGL
 usingogl=0;
 if(_opengl && sdlhaveogl && !sponge)	// Disable special scalers with OpenGL for now.  It's far too slow.
 {
  flags=SDL_OPENGL;
  usingogl=1;
 }
 #endif

 if(!(SDL_WasInit(SDL_INIT_VIDEO)&SDL_INIT_VIDEO))
  if(SDL_InitSubSystem(SDL_INIT_VIDEO)==-1)
  {
   FCEUD_PrintError(SDL_GetError());
   FCEU_indent(-1);
   return(0);
  }
 inited|=1;

 SDL_ShowCursor(0);
 tlines=erendline-srendline+1;

 vinf=SDL_GetVideoInfo();

 if(vinf->hw_available)
  flags|=SDL_HWSURFACE;

 if(_fullscreen)
  flags|=SDL_FULLSCREEN;

 flags|=SDL_HWPALETTE;

 #ifdef OPENGL
 if(usingogl)
 {
  FCEU_printf(_("\nInitializing with OpenGL(Use \"-opengl 0\" to disable).\n"));
  if(_doublebuf)
   SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
 }
 else
 #endif
  if(_doublebuf && (flags&SDL_HWSURFACE))
   flags|=SDL_DOUBLEBUF;

 if(_fullscreen)
 {
  int desbpp=_bpp;

  exs=_xscalefs;
  eys=_yscalefs;
  eefx=_efxfs;

  #ifdef OPENGL
  if(!usingogl) {exs=(int)exs;eys=(int)eys;}
  else desbpp=0;

  if(sponge)
  {
   if(!usingogl)
   {
    exs=eys=2;
    if(sponge == NTVB_HQ3X || sponge == NTVB_SCALE3X) exs = eys = 3;
    if(sponge == NTVB_2XSAI || sponge == NTVB_SUPER2XSAI || sponge == NTVB_SUPEREAGLE || sponge == NTVB_SCALE2XSAI) desbpp = 16;
   }
   eefx=0;
   if(sponge == NTVB_HQ2X || sponge == NTVB_HQ3X) desbpp = 32;
  }


  if( (usingogl && !_stretchx) || !usingogl)
  #endif
   if(_xres<NWIDTH*exs || exs <= 0.01)
   {
    FCEUD_PrintError(_("xscale out of bounds."));
    KillVideo(1);
    FCEU_indent(-1);
    return(0);
   }

  #ifdef OPENGL
  if( (usingogl && !_stretchy) || !usingogl)
  #endif
   if(_yres<tlines*eys || eys <= 0.01)
   {
    FCEUD_PrintError(_("yscale out of bounds."));
    KillVideo(1);
    FCEU_indent(-1);
    return(0);
   }

  #ifdef OPENGL
  if(!(screen = SDL_SetVideoMode(_xres, _yres, desbpp, flags)))
  #else
  if(!(screen = SDL_SetVideoMode(_xres, _yres, desbpp, flags)))
  #endif
  {
   FCEUD_PrintError(SDL_GetError());
   FCEU_indent(-1);
   return(0);
  }
 }
 else
 {
  int desbpp=0;

   exs=_xscale;
   eys=_yscale;
   eefx=_efx;

  if(sponge) 
  {
   #ifdef OPENGL
   if(!usingogl)
   #endif
   {
    exs=eys=2;
    if(sponge == NTVB_HQ3X || sponge == NTVB_SCALE3X) exs=eys=3;
   }
   eefx=0;
   // SDL's 32bpp->16bpp code is slighty faster than mine, at least :/
   if(sponge == NTVB_HQ2X || sponge == NTVB_HQ3X) desbpp=32;
   if(sponge == NTVB_2XSAI || sponge == NTVB_SUPER2XSAI || sponge == NTVB_SUPEREAGLE || sponge == NTVB_SCALE2XSAI) desbpp = 16;
  }

  #ifdef OPENGL
  if(!usingogl) {exs=(int)exs;eys=(int)eys;}
  if(exs <= 0.01) 
  {
   FCEUD_PrintError(_("xscale out of bounds."));
   KillVideo(1);
   FCEU_indent(-1);
   return(0);
  }
  if(eys <= 0.01)
  {
   FCEUD_PrintError(_("yscale out of bounds."));
   KillVideo(1);
   FCEU_indent(-1);
   return(0);
  }
  #endif

  screen = SDL_SetVideoMode((int)(NWIDTH*exs), (int)(tlines*eys), desbpp, flags);
 }
 curbpp=screen->format->BitsPerPixel;
 if(!screen)
 {
  FCEUD_PrintError(SDL_GetError());
  KillVideo(1);
  FCEU_indent(-1);
  return(0);
 }

 inited=1;

 FCEU_printf(_("Video Mode: %d x %d x %d bpp %s\n"),screen->w,screen->h,screen->format->BitsPerPixel,_fullscreen?"full screen":"");
 if(curbpp!=16 && curbpp!=24 && curbpp!=8 && curbpp!=32)
 {
  FCEU_printf(_("Sorry, %dbpp modes are not supported by Nintencer.  Supported bit depths are 8bpp, 16bpp, 24bpp, and 32bpp.\n"),curbpp);
  KillVideo(1);
  FCEU_indent(-1);
  return(0);
 }

 if(gi->name)
  SDL_WM_SetCaption((char *)gi->name,(char *)gi->name);
 else
  SDL_WM_SetCaption("Nintencer","Nintencer");

 #ifdef LSB_FIRST
 IconSurface=SDL_CreateRGBSurfaceFrom((void *)nintencer_playicon.pixel_data,32,32,32,32*4,0xFF,0xFF00,0xFF0000,0xFF000000);
 #else
 IconSurface=SDL_CreateRGBSurfaceFrom((void *)nintencer_playicon.pixel_data,32,32,32,32*4,0xFF000000,0xFF0000,0xFF00,0xFF);
 #endif

 SDL_WM_SetIcon(IconSurface,0);

 paletterefresh=1;

 if(curbpp>8)
 #ifdef OPENGL
  if(!usingogl)
 #endif
  InitBlitToHigh(curbpp>>3,screen->format->Rmask,screen->format->Gmask,screen->format->Bmask,eefx,sponge);
 #ifdef OPENGL
 if(usingogl)
 {
  int sw,sh;

  sw = sh = 1;

  if(sponge)
   sw = sh = 2;
  if(sponge == NTVB_HQ3X || sponge == NTVB_SCALE3X)
   sw = sh = 3;
  if(!InitOpenGL((eoptions&EO_CLIPSIDES)?8:0,256-((eoptions&EO_CLIPSIDES)?8:0),srendline,erendline+1,exs,eys,eefx,_openglip,_stretchx,_stretchy,sponge,sw,sh,screen))
  {
   KillVideo(1);
   FCEU_indent(-1);
   return(0);
  }
 }
 #endif


 ClearVideoSurfaces();

 FCEU_indent(-1);
 return 1;
}

static SDL_Color psdl[256];
void FCEUD_SetPalette(uint8 index, uint8 r, uint8 g, uint8 b)
{
 psdl[index].r=r;
 psdl[index].g=g;
 psdl[index].b=b;

 paletterefresh=1;
}

void FCEUD_GetPalette(uint8 index, uint8 *r, uint8 *g, uint8 *b)
{
 *r=psdl[index].r;
 *g=psdl[index].g;
 *b=psdl[index].b;
}

static void RedoPalette(void)
{
 #ifdef OPENGL
 if(usingogl)
  SetOpenGLPalette((uint8*)psdl);
 else
 #endif
 {
  if(curbpp>8)
   SetPaletteBlitToHigh((uint8*)psdl); 
  else
  {
   SDL_SetPalette(screen,SDL_PHYSPAL,psdl,0,256);
  }
 }
}

void BlitScreen(uint8 *XBuf)
{
 SDL_Surface *TmpScreen;
 uint8 *dest;
 int xo=0,yo=0;

 if(!screen) return;

 if(paletterefresh)
 {
  RedoPalette();
  paletterefresh=0;
 }

 #ifdef OPENGL
 if(usingogl)
 {
  BlitOpenGL(XBuf);
  return;
 }
 #endif

 XBuf+=srendline*256;

 TmpScreen=screen;

 if(SDL_MUSTLOCK(TmpScreen))
  if(SDL_LockSurface(TmpScreen))
  {   
   return;
  }

 dest=(uint8*)TmpScreen->pixels;

 if(_fullscreen)
 {
  xo=(int)(((TmpScreen->w-NWIDTH*exs))/2);
  dest+=xo*(curbpp>>3);
  if(TmpScreen->h>(tlines*eys))
  {
   yo=(int)((TmpScreen->h-tlines*eys)/2);
   dest+=yo*TmpScreen->pitch;
  }
 }

 if(curbpp>8)
 {
  Blit8ToHigh(XBuf+NOFFSET,dest, NWIDTH, tlines, TmpScreen->pitch,(int)exs,(int)eys);
 }
 else
 {
  Blit8To8(XBuf+NOFFSET,dest, NWIDTH, tlines, TmpScreen->pitch,(int)exs,(int)eys,eefx,sponge);
 }

 if(SDL_MUSTLOCK(TmpScreen))
  SDL_UnlockSurface(TmpScreen);

 SDL_UpdateRect(screen, xo, yo, (int)(NWIDTH*exs), (int)(tlines*eys));

 PumpWrap();

 if(screen->flags&SDL_DOUBLEBUF)
  SDL_Flip(screen);
}

uint32 PtoV(uint16 x, uint16 y)
{
 y=(uint16)((double)y/eys);
 x=(uint16)((double)x/exs);
 if(eoptions&EO_CLIPSIDES)
  x+=8;
 y+=srendline;
 return(x|(y<<16));
}
