/* FCE Ultra - NES/Famicom Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Xodnizel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdlib.h>

#include "main.h"
#include "input.h"
#include "input-config.h"
#include "sound.h"
#include "video.h"
#include "joystick.h"

#include "../common/cheat.h"

#include "keyscan.h"

int ConfigDevice(int which, int arg, int);
static void ConfigDeviceBegin(int ingame);
static void ConfigDeviceEnd(int ingame);

static void subcon_begin(void);
static int subcon(char *text, ButtConfig *bc, int commandkey, int ingame);

static char keys[MKK_COUNT];

/* UsrInputType[] is user-specified.  InputType[] is current
        (game loading can override user settings)
*/
int UsrInputType[3]={SI_GAMEPAD,SI_GAMEPAD,SIFC_NONE};
int InputType[3];
static int cspec;
   
static int gametype;

/* Necessary for proper GUI functioning(configuring when a game isn't loaded). */
void InputUserActiveFix(void)
{
 int x;
 for(x=0;x<3;x++) InputType[x]=UsrInputType[x];
}

void ParseGIInput(FCEUGI *gi)
{
 gametype=gi->type;
 
 InputType[0]=UsrInputType[0];
 InputType[1]=UsrInputType[1];
 InputType[2]=UsrInputType[2];
 
 if(gi->input[0]>=0)
  InputType[0]=gi->input[0];
 if(gi->input[1]>=0)
  InputType[1]=gi->input[1];
 if(gi->inputfc>=0)
  InputType[2]=gi->inputfc;
 cspec = gi->cspecial;
}


#define MK(x)		{{BUTTC_KEYBOARD},{0}, {MKK(x)},1}

#define MK_CK_SHIFT(x)	{{BUTTC_KEYBOARD}, {0}, {MKK(x) | (2<<24)}, 1}
#define MK_CK_ALT(x)	{{BUTTC_KEYBOARD}, {0}, {MKK(x) | (1<<24)}, 1}

#define MK2(x1,x2)      {{BUTTC_KEYBOARD},{0},{MKK(x1),MKK(x2)},2}

#define MKZ()   {{0},{0},{0},0}

typedef enum {
	_CK_FIRST = 0,
        CK_SAVE_STATE = 0,
	CK_LOAD_STATE,
	CK_SAVE_MOVIE,
	CK_LOAD_MOVIE,
	CK_CHEATS,
	CK_0,CK_1,CK_2,CK_3,CK_4,CK_5,CK_6,CK_7,CK_8,CK_9,
	CK_M0,CK_M1,CK_M2,CK_M3,CK_M4,CK_M5,CK_M6,CK_M7,CK_M8,CK_M9,
	CK_TAKE_SNAPSHOT,
	CK_HIDE_SPRITES,
	CK_HIDE_BG,
	CK_TOGGLE_FS,
	CK_FAST_FORWARD,

	CK_INSERT_COIN,
	CK_TOGGLE_DIPVIEW,
	CK_SELECT_DISK,
	CK_INSERTEJECT_DISK,
	CK_ACTIVATE_BARCODE,

	CK_TOGGLE_FKB,

	CK_INPUT_CONFIG1,
	CK_INPUT_CONFIG2,
	CK_INPUT_CONFIGFC,
	CK_INPUT_CONFIGC,
	CK_RESET,
	CK_POWER,
	CK_EXIT,
	_CK_COUNT	=	256	// Quick hack to allow adding new command keys to be configured in the future
					// without breaking old configuration files.
} CommandKey;

/* Note to self:  Always add new key mappings to the END, and never delete any out of the middle.  */
ButtConfig CKeys[_CK_COUNT]	=
{
	MK(F5),MK(F7),MK_CK_SHIFT(F5),MK_CK_SHIFT(F7),MK(F1),
	MK(0),MK(1),MK(2),MK(3),MK(4),MK(5),MK(6),MK(7),MK(8),MK(9),
	MK_CK_SHIFT(0),MK_CK_SHIFT(1),MK_CK_SHIFT(2),MK_CK_SHIFT(3),MK_CK_SHIFT(4),MK_CK_SHIFT(5),MK_CK_SHIFT(6),MK_CK_SHIFT(7),MK_CK_SHIFT(8),MK_CK_SHIFT(9),
	MK(F9),MK(F4),MK_CK_SHIFT(F4),MK_CK_ALT(RETURN), MK(BACKQUOTE),
	MK(F8),MK(F6),MK(F6),MK(F8),MK(F8),MK(SCROLLOCK),MK(F3),MK_CK_SHIFT(F3),MK_CK_ALT(F3),MK(F2),MK(F10),MK(F11),MK2(F12,ESCAPE)
};

static char *CKeysMagic[_CK_COUNT]	=
{
	"save_state",
	"load_state",
	"save_movide",
	"load_movie",
	"cheats",
	"0","1","2","3","4","5","6","7","8","9",
	"m0","m1","m2","m3","m4","m5","m6","m7","m8","m9",
	"take_snapshot",
	"hide_sprites",
	"hide_bg",
	"toggle_fs",
	"fast_forward",
	"insert_coin",
	"toggle_dipview",
	"select_disk",
	"insert_eject_disk",
	"activate_barcode",
	"toggle_fkb",
	"input_config1",
	"input_config2",
	"input_configfc",
	"input_configc",
	"reset",
	"power",
	"exit"
};

static int CKeysLastState[_CK_COUNT];

static int CK_Check(CommandKey which)
{
 int last = CKeysLastState[which];

 if((CKeysLastState[which] = DTestButtonCombo(&CKeys[which], keys)) && !last)
 {
  //printf("Rah: %d, %d\n", which,_CK_COUNT);
  return(1);
 }
 //printf("Nope: %d\n", which);
 return(0);
}

static int CK_CheckActive(CommandKey which)
{
 return(DTestButtonCombo(&CKeys[which], keys));
}

static uint8 QuizKingData;
static uint8 HyperShotData;
static uint32 MahjongData;
static uint32 FTrainerData;
static uint8 TopRiderData;

static uint8 BWorldData[1+13+1];

static void UpdateFKB(void);
static void UpdateGamepad(void);
static void UpdateQuizKing(void);
static void UpdateHyperShot(void);
static void UpdateMahjong(void);
static void UpdateFTrainer(void);
static void UpdateTopRider(void);

static uint32 JSreturn;
int NoWaiting = 0;

static void DoCheatSeq(void)
{
 SilenceSound(1);
 //KillVideo();

 DoConsoleCheatConfig();
 //InitVideo(CurGame);
 SilenceSound(0);
}

static int DIPS=0;

#define KEY(__a) keys[MKK(__a)]

static int cidisabled=0;
static int inff = 0;

typedef enum
{
	none,
	Port1,
	Port2,
	FCPort,
	Command
} ICType;

static ICType IConfig = none;
static int ICLatch;

static void KeyboardCommands(void)
{
  memcpy(keys, SDL_GetKeyState(0), MKK_COUNT);

  if(IConfig == Port1)
  {
   if(CK_Check(CK_INPUT_CONFIG1))
   {
    ConfigDeviceEnd(1);
    IConfig = none;
   }
   if(ConfigDevice(InputType[0], 0, 1))
    IConfig = none;
  }
  else if(IConfig == Port2)
  {
   if(CK_Check(CK_INPUT_CONFIG1))
   {
    ConfigDeviceEnd(1);
    IConfig = none;
   }
   if(ConfigDevice(InputType[1], 1, 1))
    IConfig = none;
  }
  else if(IConfig == FCPort)
  {
   if(CK_Check(CK_INPUT_CONFIG1))
   {
    ConfigDeviceEnd(1);
    IConfig = none;
   }
   if(ConfigDevice(InputType[2], 0, 1))
    IConfig = none;
  }
  else if(IConfig == Command)
  {
   if(ICLatch != -1)
   {
    if(subcon(CKeysMagic[ICLatch], &CKeys[ICLatch], 1, 1))
    {
     IConfig = none;
     CKeysLastState[ICLatch] = 1;	// We don't want to accidentally
     return;				// trigger the command. :b
    }
   }
   else
   {
    int x;
    FCEUI_DispMessage((UTF8*)_("Please press the command key to be mapped."));
    for(x = (int)_CK_FIRST; x < (int)_CK_COUNT; x++)
     if(CK_Check((CommandKey)x))
     {      
      ICLatch = x;
      subcon_begin();
      break;
     }
   }
  }

  if(InputType[2]==SIFC_FKB)
  {
   if(CK_Check(CK_TOGGLE_FKB))
   {
    cidisabled^=1;
    if(cidisabled)
     FCEUI_DispMessage((UTF8*)_("Family Keyboard enabled."));
    else
     FCEUI_DispMessage((UTF8*)_("Family Keyboard disabled."));
   }
   SDL_WM_GrabInput(cidisabled?SDL_GRAB_ON:SDL_GRAB_OFF);
   if(cidisabled) return;
  }

  if(IConfig != none)
   return;

  if(CK_Check(CK_INPUT_CONFIG1))
  {
   ConfigDeviceBegin(1);
   IConfig = Port1;
  }
  else if(CK_Check(CK_INPUT_CONFIG2))
  {
   ConfigDeviceBegin(1);
   IConfig = Port2;
  }
  else if(CK_Check(CK_INPUT_CONFIGFC))
  {
   ConfigDeviceBegin(1);
   IConfig = FCPort;
  }
  else if(CK_Check(CK_INPUT_CONFIGC))
  {
   ConfigDeviceBegin(1);
   ICLatch = -1;
   IConfig = Command;
  }

  if(CK_Check(CK_HIDE_SPRITES)) FCEUI_SetRenderDisable(2, -1);
  if(CK_Check(CK_HIDE_BG)) FCEUI_SetRenderDisable(-1, 2);
  if(CK_Check(CK_TOGGLE_FS)) ToggleFS();

  if(CK_CheckActive(CK_FAST_FORWARD))
  {
   if(!inff) 
   {
    NTCRI_SetSoundMultiplier(Settings.ffspeed);
    RefreshThrottleFPS(Settings.ffspeed);
    inff = 1;
   }
  }
  else
  {
   if(inff) 
   {
    NTCRI_SetSoundMultiplier(1);
    RefreshThrottleFPS(1);
   }
   inff = 0;
  }
  if(gametype==GIT_FDS)
  {
   if(CK_Check(CK_SELECT_DISK)) FCEUI_FDSSelect();
   if(CK_Check(CK_INSERTEJECT_DISK)) FCEUI_FDSInsert(0);
  }

  if(CK_Check(CK_TAKE_SNAPSHOT)) FCEUI_SaveSnapshot();
  if(gametype!=GIT_NSF)
  {
   if(CK_Check(CK_CHEATS)) DoCheatSeq();

   if(CK_Check(CK_SAVE_STATE))
	NTCRI_SaveState(NULL, NULL);
   if(CK_Check(CK_SAVE_MOVIE))
	FCEUI_SaveMovie(NULL);
   if(CK_Check(CK_LOAD_STATE))
	NTCRI_LoadState(NULL, NULL);
   if(CK_Check(CK_LOAD_MOVIE))
	FCEUI_LoadMovie(NULL);
  }

  //if(keyonly(F1)) FCEUI_ToggleTileView();

  if(CK_Check(CK_RESET))
	FCEUI_ResetNES();
  if(CK_Check(CK_POWER))
	FCEUI_PowerNES();
  if(CK_Check(CK_EXIT))
	CloseGame();

  if(gametype==GIT_VSUNI)
  {
	if(CK_Check(CK_INSERT_COIN))
		FCEUI_VSUniCoin();
	if(CK_Check(CK_TOGGLE_DIPVIEW))
        {
	 DIPS^=1;
	 FCEUI_VSUniToggleDIPView();
	}
	if(!(DIPS&1)) goto DIPSless;
	if(CK_Check(CK_1)) FCEUI_VSUniToggleDIP(0);
	if(CK_Check(CK_2)) FCEUI_VSUniToggleDIP(1);
	if(CK_Check(CK_3)) FCEUI_VSUniToggleDIP(2);
	if(CK_Check(CK_4)) FCEUI_VSUniToggleDIP(3);
	if(CK_Check(CK_5)) FCEUI_VSUniToggleDIP(4);
	if(CK_Check(CK_6)) FCEUI_VSUniToggleDIP(5);
	if(CK_Check(CK_7)) FCEUI_VSUniToggleDIP(6);
	if(CK_Check(CK_8)) FCEUI_VSUniToggleDIP(7);
  }
  else
  {
   static uint8 bbuf[32];
   static int bbuft;
   static int barcoder = 0;

   //if(keyonly(H)) FCEUI_NTSCSELHUE();
   //if(keyonly(T)) FCEUI_NTSCSELTINT();
   if(KEY(KP_MINUS) || KEY(MINUS)) FCEUI_NTSCDEC();
   if(KEY(KP_PLUS) || KEY(EQUALS)) FCEUI_NTSCINC();

   if((InputType[2] == SIFC_BWORLD) || (cspec == SIS_DATACH))
   {
    if(CK_Check(CK_ACTIVATE_BARCODE))
    {
     barcoder ^= 1;
     if(!barcoder)
     {
      if(InputType[2] == SIFC_BWORLD)
      {
       strcpy((char *)&BWorldData[1],(char *)bbuf);
       BWorldData[0]=1;
      }
      else
       FCEUI_DatachSet(bbuf);
      FCEUI_DispMessage((UTF8 *)_("Barcode Entered"));
     } 
     else { bbuft = 0; FCEUI_DispMessage((UTF8 *)_("Enter Barcode"));}
    }
   } else barcoder = 0;

   #define SSM(x) { if(bbuft < 13) {bbuf[bbuft++] = '0' + x; bbuf[bbuft] = 0;} FCEUI_DispMessage((UTF8 *)_("Barcode: %s"),bbuf); }
   DIPSless:

   if(barcoder)
   {
    if(CK_Check(CK_0)) SSM(0);
    if(CK_Check(CK_1)) SSM(1);
    if(CK_Check(CK_2)) SSM(2);
    if(CK_Check(CK_3)) SSM(3);
    if(CK_Check(CK_4)) SSM(4);
    if(CK_Check(CK_5)) SSM(5);
    if(CK_Check(CK_6)) SSM(6);
    if(CK_Check(CK_7)) SSM(7);
    if(CK_Check(CK_8)) SSM(8);
    if(CK_Check(CK_9)) SSM(9);
   }
   else
   {
    if(CK_Check(CK_0)) FCEUI_SelectState(0);
    if(CK_Check(CK_1)) FCEUI_SelectState(1);
    if(CK_Check(CK_2)) FCEUI_SelectState(2);
    if(CK_Check(CK_3)) FCEUI_SelectState(3);
    if(CK_Check(CK_4)) FCEUI_SelectState(4);
    if(CK_Check(CK_5)) FCEUI_SelectState(5);
    if(CK_Check(CK_6)) FCEUI_SelectState(6);
    if(CK_Check(CK_7)) FCEUI_SelectState(7);
    if(CK_Check(CK_8)) FCEUI_SelectState(8);
    if(CK_Check(CK_9)) FCEUI_SelectState(9);

    if(CK_Check(CK_M0)) FCEUI_SelectMovie(0);
    if(CK_Check(CK_M1)) FCEUI_SelectMovie(1);
    if(CK_Check(CK_M2)) FCEUI_SelectMovie(2);
    if(CK_Check(CK_M3)) FCEUI_SelectMovie(3);
    if(CK_Check(CK_M4)) FCEUI_SelectMovie(4);
    if(CK_Check(CK_M5)) FCEUI_SelectMovie(5);
    if(CK_Check(CK_M6)) FCEUI_SelectMovie(6);
    if(CK_Check(CK_M7)) FCEUI_SelectMovie(7);
    if(CK_Check(CK_M8)) FCEUI_SelectMovie(8);
    if(CK_Check(CK_M9)) FCEUI_SelectMovie(9);
   }
   #undef SSM
 }
}

#define GPZ()   {MKZ(), MKZ(), MKZ(), MKZ()}

ButtConfig GamePadConfig[4][10]={
        /* Gamepad 1 */
        { 
	 MK(KP3), MK(KP2), MK(TAB), MK(RETURN), MK(w),MK(z),
                MK(a), MK(s), MKZ(), MKZ()
	},

        /* Gamepad 2 */
        GPZ(),

        /* Gamepad 3 */
        GPZ(),
  
        /* Gamepad 4 */
        GPZ()
};


static void UpdateGamepad(void)
{
 static int rapid=0;
 uint32 JS=0;
 unsigned int x;
 unsigned int wg;

 rapid^=1;

 for(wg=0;wg<4;wg++)
 {
  for(x=0;x<8;x++)
   if(DTestButton(&GamePadConfig[wg][x], keys))
    JS|=(1<<x)<<(wg<<3);

  if(rapid)
   for(x=0;x<2;x++)
     if(DTestButton(&GamePadConfig[wg][8+x], keys))
      JS|=(1<<x)<<(wg<<3);
  }

  for(x=0;x<32;x+=8)	/* Now, test to see if anything weird(up+down at same time)
			   is happening, and correct */
  {
   if((JS & (0xC0<<x) ) == ((unsigned)0xC0<<x) ) JS&=~((unsigned)0xC0<<x);
   if((JS & (0x30<<x) ) == ((unsigned)0x30<<x) ) JS&=~((unsigned)0x30<<x);
  }

  JSreturn=JS;
}

ButtConfig powerpadsc[2][12]={
                              {
                            	MK(o),MK(p),MK(LEFTBRACKET),MK(RIGHTBRACKET),
				MK(k),MK(l),MK(SEMICOLON),MK(QUOTE),
                              	MK(M),MK(COMMA),MK(PERIOD),MK(SLASH)
                              },
                              {
                        	MK(o),MK(p),MK(LEFTBRACKET),MK(RIGHTBRACKET),
				MK(k),MK(l),MK(SEMICOLON),MK(QUOTE),
                               	MK(m),MK(COMMA),MK(PERIOD),MK(SLASH)
                              }
                             };

static uint32 powerpadbuf[2];

static uint32 UpdatePPadData(int w)
{
 uint32 r=0;
 ButtConfig *ppadtsc=powerpadsc[w];
 int x;

 for(x=0;x<12;x++)
  if(DTestButton(&ppadtsc[x], keys)) r|=1<<x;

 return r;
}

static uint32 MouseData[3];
static uint8 fkbkeys[0x48];

void FCEUD_UpdateInput(void)
{
  int x;
  int t=0;

  /*
  while(SDL_PollEvent(&event))
  {
   switch(event.type)
   {
    //case SDL_SYSWMEVENT: puts("Nifty keen");break;
    //case SDL_VIDEORESIZE: puts("Okie dokie");break;
    case SDL_QUIT: CloseGame();break;
   }
  }
  */

  KeyboardCommands();

  for(x=0;x<2;x++)
   switch(InputType[x])
   {
    case SI_GAMEPAD:t|=1;break;
    case SI_ARKANOID:t|=2;break;
    case SI_ZAPPER:t|=2;break;
    case SI_POWERPADA:
    case SI_POWERPADB:powerpadbuf[x]=UpdatePPadData(x);break;
   }

  switch(InputType[2])
  {
   case SIFC_ARKANOID:t|=2;break;
   case SIFC_SHADOW:t|=2;break;
   case SIFC_FKB:if(cidisabled) UpdateFKB();break;
   case SIFC_HYPERSHOT: UpdateHyperShot();break;
   case SIFC_MAHJONG: UpdateMahjong();break;
   case SIFC_QUIZKING: UpdateQuizKing();break;
   case SIFC_FTRAINERB:
   case SIFC_FTRAINERA: UpdateFTrainer();break;
   case SIFC_TOPRIDER: UpdateTopRider();break;
   case SIFC_OEKAKIDS:t|=2;break;

  }

  if(t&1)
   UpdateGamepad();

  if(t&2)
  {
   int x,y;
   uint32 t;

   t=SDL_GetMouseState(&x,&y);

   MouseData[2]=0;
   if(t&SDL_BUTTON(1))
    MouseData[2]|=1;
   if(t&SDL_BUTTON(3))
    MouseData[2]|=2;
   t=PtoV(x,y);
   MouseData[0]=t&0xFFFF;
   MouseData[1]=(t>>16)&0xFFFF;
  }
}

static void FixBCs(void);

void InitOtherInput(void)
{
   void *InputDPtr;

   int t;
   int x;
   int attrib;

   for(t=0,x=0;x<2;x++)
   {
    attrib=0;
    InputDPtr=0;
    switch(InputType[x])
    {
     case SI_POWERPADA:
     case SI_POWERPADB:InputDPtr=&powerpadbuf[x];break;
     case SI_GAMEPAD:InputDPtr=&JSreturn;break;     
     case SI_ARKANOID:InputDPtr=MouseData;t|=1;break;
     case SI_ZAPPER:InputDPtr=MouseData;
                                t|=1;
                                attrib=1;
                                break;
    }
    FCEUI_SetInput(x,InputType[x],InputDPtr,attrib);
   }

   attrib=0;
   InputDPtr=0;
   switch(InputType[2])
   {
    case SIFC_SHADOW:InputDPtr=MouseData;t|=1;attrib=1;break;
    case SIFC_OEKAKIDS:InputDPtr=MouseData;t|=1;attrib=1;break;
    case SIFC_ARKANOID:InputDPtr=MouseData;t|=1;break;
    case SIFC_FKB:InputDPtr=fkbkeys;break;
    case SIFC_HYPERSHOT:InputDPtr=&HyperShotData;break;
    case SIFC_MAHJONG:InputDPtr=&MahjongData;break;
    case SIFC_QUIZKING:InputDPtr=&QuizKingData;break;
    case SIFC_TOPRIDER:InputDPtr=&TopRiderData;break;
    case SIFC_BWORLD:InputDPtr=BWorldData;break;
    case SIFC_FTRAINERA:
    case SIFC_FTRAINERB:InputDPtr=&FTrainerData;break;
   }

   FCEUI_SetInputFC(InputType[2],InputDPtr,attrib);
   FCEUI_DisableFourScore(eoptions&EO_NOFOURSCORE);
 FixBCs();
}


ButtConfig fkbmap[0x48]=
{
 MK(F1),MK(F2),MK(F3),MK(F4),MK(F5),MK(F6),MK(F7),MK(F8),
 MK(1),MK(2),MK(3),MK(4),MK(5),MK(6),MK(7),MK(8),MK(9),MK(0),MK(MINUS),MK(EQUALS),MK(BACKSLASH),MK(BACKSPACE),
 MK(ESCAPE),MK(q),MK(w),MK(e),MK(r),MK(t),MK(y),MK(u),MK(i),MK(o),MK(p),MK(BACKQUOTE),MK(LEFTBRACKET),MK(RETURN),
 MK(LCTRL),MK(a),MK(s),MK(d),MK(f),MK(g),MK(h),MK(j),MK(k),MK(l),MK(SEMICOLON),MK(QUOTE),MK(RIGHTBRACKET),MK(INSERT),
 MK(LSHIFT),MK(z),MK(x),MK(c),MK(v),MK(b),MK(n),MK(m),MK(COMMA),MK(PERIOD),MK(SLASH),MK(RALT),MK(RSHIFT),MK(LALT),MK(SPACE),
 MK(DELETE),MK(END),MK(PAGEDOWN),MK(UP),MK(LEFT),MK(RIGHT),MK(DOWN)
};

static void UpdateFKB(void)
{
 int x;

 for(x=0;x<0x48;x++)
 {
  fkbkeys[x]=0;

  if(DTestButton(&fkbmap[x], keys))
   fkbkeys[x]=1;
 }
}

static ButtConfig HyperShotButtons[4]=
{
 MK(Q),MK(W),MK(E),MK(R)
};

static void UpdateHyperShot(void)
{
 int x;

 HyperShotData=0;
 for(x=0;x<0x4;x++)
 {
  if(DTestButton(&HyperShotButtons[x], keys))
   HyperShotData|=1<<x;
 }
}

static ButtConfig MahjongButtons[21]=
{
 MK(q),MK(w),MK(e),MK(r),MK(t),
 MK(a),MK(s),MK(d),MK(f),MK(g),MK(h),MK(j),MK(k),MK(l),
 MK(z),MK(x),MK(c),MK(v),MK(b),MK(n),MK(m)
};

static void UpdateMahjong(void)
{
 int x;
        
 MahjongData=0;
 for(x=0;x<21;x++)
 {  
  if(DTestButton(&MahjongButtons[x], keys))
   MahjongData|=1<<x;
 }
}

ButtConfig QuizKingButtons[6]=
{
 MK(q),MK(w),MK(e),MK(r),MK(t),MK(y)
};

static void UpdateQuizKing(void)
{
 int x;

 QuizKingData=0;

 for(x=0;x<6;x++)
 {
  if(DTestButton(&QuizKingButtons[x], keys))
   QuizKingData|=1<<x;
 }

}

ButtConfig TopRiderButtons[8]=
{
 MK(q),MK(w),MK(e),MK(r),MK(t),MK(y),MK(u),MK(i)
};

static void UpdateTopRider(void)
{
 int x;
 TopRiderData=0;
 for(x=0;x<8;x++)
  if(DTestButton(&TopRiderButtons[x], keys))
   TopRiderData|=1<<x;
}

ButtConfig FTrainerButtons[12]=
{
                               MK(o),MK(p),MK(LEFTBRACKET),
                               MK(RIGHTBRACKET),MK(k),MK(l),MK(SEMICOLON),
                                MK(QUOTE),
                               MK(m),MK(COMMA),MK(PERIOD),MK(SLASH)
};

static void UpdateFTrainer(void)
{
 int x;

 FTrainerData=0;

 for(x=0;x<12;x++)
 {
  if(DTestButton(&FTrainerButtons[x], keys))
   FTrainerData|=1<<x;
 }
}

static int subcon_tb;
static int subcon_wc;
static int jitter_correct;

static void subcon_begin(void)
{
 subcon_tb = -1;
 subcon_wc = 0;
 jitter_correct = 0;
}

/* Configures an individual virtual button. */
static int subcon(char *text, ButtConfig *bc, int commandkey, int ingame)
{
 char buf[256];

 while(subcon_wc < MAXBUTTCONFIG)
 {
  sprintf(buf,"%s (%d)",text,subcon_wc+1);
  //puts(buf);
  if(ingame)
  {
   FCEUI_DispMessage((UTF8*)buf);
   if(subcon_tb != subcon_wc)
   {
    DTryButtonBegin(bc, subcon_wc, commandkey);
    subcon_tb = subcon_wc;
    jitter_correct = 0;
   }
   if(!jitter_correct)
    if(!DTryButton())
     return(0);
  }
  else
   DWaitButton(buf,bc,subcon_wc, commandkey);

  if(ingame)
    DTryButtonEnd(bc);

  if(bc->ButtType[subcon_wc] != BUTTC_JOYSTICK)	// Only do jitter correction on joysticks.
   jitter_correct = 10;

  redojitter:	// for (!ingame) condition.

  if(jitter_correct < 10)
  {
   ButtConfig tmpbc;

   tmpbc.ButtType[0] = bc->ButtType[subcon_wc];
   tmpbc.DeviceNum[0] = bc->DeviceNum[subcon_wc];
   tmpbc.ButtonNum[0] = bc->ButtonNum[subcon_wc];
   tmpbc.DeviceID[0] = bc->DeviceID[subcon_wc];
   tmpbc.NumC = 1;

   if(!ingame)
    memcpy(keys, SDL_GetKeyState(0), MKK_COUNT);
   if((!commandkey && DTestButton(&tmpbc, keys)) || (commandkey && DTestButtonCombo(&tmpbc, keys)))
   {
    jitter_correct++;
    if(!ingame) goto redojitter;
   }
   else
   {
    puts("Jitter failure!");
    subcon_tb = subcon_wc - 1;	// Redo this physical button.
   }
   if(ingame)
    return(0);
   else
    continue;
  }

  if(subcon_wc && bc->ButtType[subcon_wc]==bc->ButtType[subcon_wc-1] && bc->DeviceNum[subcon_wc]==bc->DeviceNum[subcon_wc-1] &&
               bc->ButtonNum[subcon_wc]==bc->ButtonNum[subcon_wc-1])
        break;
  subcon_wc++;  
 }
 bc->NumC = subcon_wc;
 //puts("DONE");
 return(1);
}

void ConfigCommandKey(char *which)
{
 int which_n;
 for(which_n = (int)_CK_FIRST; which_n < (int)_CK_COUNT; which_n++)
 {
  if(!strcmp(CKeysMagic[which_n], which))
  {
   subcon_begin();
   subcon(CKeysMagic[which_n], &CKeys[which_n], 1, 0);
  }
 }
}

static void CKConfig(char *text)
{
 unsigned int slen = strlen(text);
 char *tmp = (char *)malloc(slen + 1);
 char* lastptr = tmp;
 unsigned int x;

 ButtonConfigBegin();
 strcpy(tmp, text);
 for(x=0;x<slen;x++)
 {
  if(tmp[x] == ',' || tmp[x] == ' ')
  {
   tmp[x] = 0;
   ConfigCommandKey(lastptr);
   lastptr = &tmp[x+1];
  }
 }
 ConfigCommandKey(lastptr);
 ButtonConfigEnd();
 free(tmp);
}

static int cd_x;
static int cd_lx = -1;
static void ConfigDeviceBegin(int ingame)
{
 cd_x = 0;
 cd_lx = -1;

 if(!ingame)
  ButtonConfigBegin();
}

static void ConfigDeviceEnd(int ingame)
{
 if(!ingame)
  ButtonConfigEnd();
}

int ConfigDevice(int which, int arg, int ingame)
{
 char buf[256];

 if(!ingame)
  ConfigDeviceBegin(0);

 switch(which)
 {
  case SIFC_QUIZKING:
   for(;cd_x<6;cd_x++)
   {
    sprintf(buf,"Quiz King Buzzer #%d", cd_x+1);
    if(cd_x != cd_lx)
    {
     cd_lx = cd_x;
     subcon_begin();
    }
    if(!subcon(buf,&QuizKingButtons[cd_x], 0, ingame))
     return(0);
   }
   break;
  case SIFC_HYPERSHOT:
   for(;cd_x<4;cd_x++)
   {
    sprintf(buf,"Hyper Shot %d: %s",((cd_x&2)>>1)+1,(cd_x&1)?"JUMP":"RUN");
    if(cd_x != cd_lx)
    {
     cd_lx = cd_x;
     subcon_begin();
    }
    if(!subcon(buf,&HyperShotButtons[cd_x], 0, ingame))
     return(0);
   } 
   break;
  case SI_POWERPADA:
  case SI_POWERPADB:
   for(;cd_x<12;cd_x++)
   {
    sprintf(buf,"PowerPad %d: %d", (arg&1)+1,cd_x+1);
    if(cd_x != cd_lx)
    {
     cd_lx = cd_x;
     subcon_begin();
    }
    if(!subcon(buf,&powerpadsc[arg&1][cd_x], 0, ingame))
     return(0);
   }
   break;

  case SIFC_FTRAINERA:
  case SIFC_FTRAINERB:
   for(;cd_x<12;cd_x++)
   {
    sprintf(buf,"Family Trainer: %d", cd_x+1);
    if(cd_x != cd_lx)
    {
     cd_lx = cd_x;
     subcon_begin();
    }
    if(!subcon(buf,&FTrainerButtons[cd_x], 0, ingame))
     return(0);
   }
   break;

  case SI_GAMEPAD:
  {
   char *str[10]={"A","B","SELECT","START","UP ↑","DOWN ↓","LEFT ←","RIGHT →","Rapid A","Rapid B"};
   for(;cd_x<10;cd_x++)
   {
    sprintf(buf,"GamePad #%d: %s",arg+1,str[cd_x]);
    if(cd_x != cd_lx)
    {
     cd_lx = cd_x;
     subcon_begin();
    }
    if(!subcon(buf,&GamePadConfig[arg][cd_x], 0, ingame))
     return(0);
   }
  }
  break;
 }
 if(!ingame)
  ConfigDeviceEnd(0);

 return(1);
}


CFGSTRUCT InputConfig[]={
	AC(CKeys),
        ACA(UsrInputType),
        AC(powerpadsc),
        AC(QuizKingButtons),
        AC(FTrainerButtons),
        AC(HyperShotButtons),
        AC(MahjongButtons),
        AC(GamePadConfig),
        AC(fkbmap),
        ENDCFGSTRUCT
};


static void InputCfg(char *text)
{
         if(!strncasecmp(text,"gamepad",strlen("gamepad")))
         {
          ConfigDevice(SI_GAMEPAD,(text[strlen("gamepad")]-'1')&3, 0);
         }
         else if(!strncasecmp(text,"powerpad",strlen("powerpad")))
         {
          ConfigDevice(SI_POWERPADA,(text[strlen("powerpad")]-'1')&1, 0);
         }
         else if(!strcasecmp(text,"hypershot"))         
	 {
	  ConfigDevice(SIFC_HYPERSHOT,0, 0);
	 }
	 else if(!strcasecmp(text, "ftrainer"))
	 {
	  ConfigDevice(SIFC_FTRAINERB, 0, 0);
	 }
         else if(!strcasecmp(text,"quizking"))
          ConfigDevice(SIFC_QUIZKING,0, 0);
}

static void FCExp(char *text)
{
        static char *fccortab[11]={"none","arkanoid","shadow","4player","fkb","hypershot",
                        "mahjong","quizking","ftrainera","ftrainerb","oekakids"};
           
        static int fccortabi[11]={SIFC_NONE,SIFC_ARKANOID,SIFC_SHADOW,
                                 SIFC_4PLAYER,SIFC_FKB,SIFC_HYPERSHOT,SIFC_MAHJONG,SIFC_QUIZKING,
                                 SIFC_FTRAINERA,SIFC_FTRAINERB,SIFC_OEKAKIDS};
	int y;
	for(y=0;y<11;y++)
	 if(!strcmp(fccortab[y],text))
	  UsrInputType[2]=fccortabi[y];
}

static char *cortab[6]={"none","gamepad","zapper","powerpada","powerpadb","arkanoid"};
static int cortabi[6]={SI_NONE,SI_GAMEPAD,
                               SI_ZAPPER,SI_POWERPADA,SI_POWERPADB,SI_ARKANOID};

static void Input1(char *text)
{
	int y;

	for(y=0;y<6;y++)
	 if(!strcmp(cortab[y],text))
	  UsrInputType[0]=cortabi[y];
}

static void Input2(char *text)
{
	int y;

	for(y=0;y<6;y++)
	 if(!strcmp(cortab[y],text))
	  UsrInputType[1]=cortabi[y];
}

ARGPSTRUCT InputArgs[]={
	{"-inputcfg",0,(void *)InputCfg,0x2000},
	{"-fcexp",0,(void *)FCExp,0x2000},
	{"-input1",0,(void *)Input1,0x2000},
	{"-input2",0,(void *)Input2,0x2000},
	{"-ckconfig",0,(void *)CKConfig, 0x2000},
	{0,0,0,0}
};

static void FixBCs(void)
{
 unsigned int x, y;

 for(x=0; x<_CK_COUNT;x++)
  JoyClearBC(&CKeys[x]);

   for(x=0; x<4; x++)
    for(y=0; y<10; y++)
     JoyClearBC(&GamePadConfig[x][y]);

   for(x=0; x<2; x++)
    for(y=0; y<12; y++)
     JoyClearBC(&powerpadsc[x][y]);

   for(x=0; x<0x48; x++)
    JoyClearBC(&fkbmap[x]);

   for(x=0; x<6; x++)
    JoyClearBC(&QuizKingButtons[x]);
   for(x=0; x<12; x++)
    JoyClearBC(&FTrainerButtons[x]);
   for(x=0; x<21; x++)
    JoyClearBC(&MahjongButtons[x]);
   for(x=0; x<4; x++)
    JoyClearBC(&HyperShotButtons[x]);
}
