/* FCE Ultra - NES/Famicom Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Xodnizel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "types.h"
#include "x6502.h"
#include "nintencer.h"
#include "video.h"
#include "sound.h"
#include "nsf.h"
#include "nsfe.h"
#include "general.h"
#include "memory.h"
#include "file.h"
#include "fds.h"
#include "cart.h"
#include "input.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

static uint8 SongReload;

static DECLFW(NSF_write);
static DECLFR(NSF_read);

static int vismode=1;
static NSFINFO *NSFInfo;

static uint8 NSFROM[0x30+6]=
{
/* 0x00 - NMI */
0x8D,0xF4,0x3F,                         /* Stop play routine NMIs. */
0xA2,0xFF,0x9A,                         /* Initialize the stack pointer. */
0xAD,0xF0,0x3F,                         /* See if we need to init. */
0xF0,0x09,                              /* If 0, go to play routine playing. */

0xAD,0xF1,0x3F,                         /* Confirm and load A      */
0xAE,0xF3,0x3F,                         /* Load X with PAL/NTSC byte */

0x20,0x00,0x00,                         /* JSR to init routine     */

0xA9,0x00,
0xAA,
0xA8,
0x20,0x00,0x00,                         /* JSR to play routine  */
0x8D,0xF5,0x3F,				/* Start play routine NMIs. */
0x90,0xFE,                               /* Loopie time. */

/* 0x20 */
0x8D,0xF3,0x3F,				/* Init init NMIs */
0x18,
0x90,0xFE				/* Loopie time. */
};

static DECLFR(NSFROMRead)
{
 return (NSFROM-0x3800)[A];
}

static int doreset=0;
static int NSFNMIFlags;
static uint8 BSon;

static uint8 *ExWRAM=0;

static void FreeNSF(void)
{
 if(NSFInfo)
 {
  if(NSFInfo->GameName) free(NSFInfo->GameName);
  if(NSFInfo->Artist) free(NSFInfo->Artist);
  if(NSFInfo->Copyright) free(NSFInfo->Copyright);
  if(NSFInfo->Ripper) free(NSFInfo->Ripper);
  if(NSFInfo->NSFDATA) free(NSFInfo->NSFDATA);
  if(ExWRAM) { free(ExWRAM); ExWRAM = NULL; }
  free(NSFInfo);
  NSFInfo = NULL;
 }
}

void NSFGI(int h)
{
 switch(h)
 {
 case GI_CLOSE:
  if(GameExpSound.Kill) GameExpSound.Kill();
  FreeNSF();
  break;
 case GI_RESETM2:
 case GI_POWER: NSF_init();break;
 }
}

// First 32KB is reserved for sound chip emulation in the iNES mapper code.

static INLINE void BANKSET(uint32 A, uint32 bank)
{
 bank &= NSFInfo->NSFMaxBank;
 if(NSFInfo->SoundChip&4)
  memcpy(ExWRAM+(A-0x6000),NSFInfo->NSFDATA+(bank<<12),4096);
 else 
  setprg4(A,bank);
}

int LoadNSF(FCEUFILE *fp)
{
 NSF_HEADER NSFHeader;
 FCEU_fseek(fp, 0, SEEK_SET);
 FCEU_fread(&NSFHeader, 1, 0x80, fp);

 // NULL-terminate strings just in case.
 NSFHeader.GameName[31] = NSFHeader.Artist[31] = NSFHeader.Copyright[31] = 0;

 NSFInfo->GameName = (UTF8*)NTCR_FixString(strdup((char *)NSFHeader.GameName));
 NSFInfo->Artist = (UTF8 *)NTCR_FixString(strdup((char *)NSFHeader.Artist));
 NSFInfo->Copyright = (UTF8 *)NTCR_FixString(strdup((char *)NSFHeader.Copyright));

 NSFInfo->LoadAddr = NSFHeader.LoadAddressLow | (NSFHeader.LoadAddressHigh << 8);
 NSFInfo->InitAddr = NSFHeader.InitAddressLow | (NSFHeader.InitAddressHigh << 8);
 NSFInfo->PlayAddr = NSFHeader.PlayAddressLow | (NSFHeader.PlayAddressHigh << 8);

 NSFInfo->NSFSize = FCEU_fgetsize(fp)-0x80;

 NSFInfo->NSFMaxBank = ((NSFInfo->NSFSize+(NSFInfo->LoadAddr&0xfff)+4095)/4096);
 NSFInfo->NSFMaxBank = uppow2(NSFInfo->NSFMaxBank);

 if(!(NSFInfo->NSFDATA=(uint8 *)malloc(NSFInfo->NSFMaxBank*4096)))
  return 0;

 FCEU_fseek(fp,0x80,SEEK_SET);
 memset(NSFInfo->NSFDATA, 0x00, NSFInfo->NSFMaxBank*4096);
 FCEU_fread(NSFInfo->NSFDATA+(NSFInfo->LoadAddr&0xfff), 1, NSFInfo->NSFSize, fp);
 
 NSFInfo->NSFMaxBank--;

 NSFInfo->VideoSystem = NSFHeader.VideoSystem;
 NSFInfo->SoundChip = NSFHeader.SoundChip;
 NSFInfo->TotalSongs = NSFHeader.TotalSongs;

 if(NSFHeader.StartingSong == 0)
  NSFHeader.StartingSong = 1;

 NSFInfo->StartingSong = NSFHeader.StartingSong - 1;
 memcpy(NSFInfo->BankSwitch, NSFHeader.BankSwitch, 8);

 return(1);
}

uint8 *FCEU_fgetmem(FCEUFILE *fp);
int NSFLoad(const char *name, FCEUFILE *fp)
{
 char magic[5];
 int x;

 if(!(NSFInfo = (NSFINFO *)malloc(sizeof(NSFINFO))))
 {
  FCEU_PrintError(_("Error allocating memory for NSF structure."));
  return(0);
 }
 memset(NSFInfo, 0, sizeof(NSFINFO));

 FCEU_fseek(fp,0,SEEK_SET);
 FCEU_fread(magic, 1, 5, fp);

 if(!memcmp(magic, "NESM\x1a", 5))
 {
  if(!LoadNSF(fp))
  {
   FreeNSF();
   return(0);
  }
 }
 else if(!memcmp(magic, "NSFE", 4))
 {
  if(!LoadNSFE(NSFInfo, FCEU_fgetmem(fp), FCEU_fgetsize(fp),0))
  {
   FreeNSF();
   return(0);
  }
 }
 else
 {
  FreeNSF();
  return(-1);
 }

 if(NSFInfo->LoadAddr < 0x6000)
 {
  FCEUI_printf("Load address is invalid!");
  FreeNSF();
  return(0);
 }

 if(NSFInfo->TotalSongs < 1)
 {
  FCEUI_printf("Total number of songs is less than 1!");
  FreeNSF();
  return(0);
 }

 BSon = 0;
 for(x=0;x<8;x++)
  BSon |= NSFInfo->BankSwitch[x];

 FCEUGameInfo->type = GIT_NSF;
 FCEUGameInfo->input[0] = FCEUGameInfo->input[1] = SI_GAMEPAD;
 FCEUGameInfo->cspecial = SIS_NSF;

 if(NSFInfo->GameName)
  FCEUGameInfo->name = (UTF8*)strdup((char*)NSFInfo->GameName);

 for(x=0;;x++)
 {
  if(NSFROM[x]==0x20)
  {
   NSFROM[x+1]=NSFInfo->InitAddr&0xFF;
   NSFROM[x+2]=NSFInfo->InitAddr>>8;
   NSFROM[x+8]=NSFInfo->PlayAddr&0xFF;
   NSFROM[x+9]=NSFInfo->PlayAddr>>8;
   break;
  }
 }

 if(NSFInfo->VideoSystem==0)
  FCEUGameInfo->vidsys=GIV_NTSC;
 else if(NSFInfo->VideoSystem==1)
  FCEUGameInfo->vidsys=GIV_PAL;

 GameInterface = NSFGI;

 FCEU_printf(_("NSF Loaded.  File information:\n\n"));
 FCEU_indent(1);
 if(NSFInfo->GameName)
  FCEU_printf(_("Game/Album Name:\t%s\n"), NSFInfo->GameName);
 if(NSFInfo->Artist)
  FCEU_printf(_("Music Artist:\t%s\n"), NSFInfo->Artist);
 if(NSFInfo->Copyright)
  FCEU_printf(_("Copyright:\t\t%s\n"), NSFInfo->Copyright);
 if(NSFInfo->Ripper)
  FCEU_printf(_("Ripper:\t\t%s\n"), NSFInfo->Ripper);

 if(NSFInfo->SoundChip)
 {
  static char *tab[6]={"Konami VRCVI","Konami VRCVII","Nintendo FDS","Nintendo MMC5","Namco 106","Sunsoft FME-07"};

  for(x=0;x<6;x++)
   if(NSFInfo->SoundChip&(1<<x))
   {
    FCEU_printf(_("Expansion hardware:  %s\n"),tab[x]);
    NSFInfo->SoundChip=1<<x;	/* Prevent confusing weirdness if more than one bit is set. */
    break;
   }
 }
 if(BSon)
  FCEU_printf(_("Bank-switched\n"));
 FCEU_printf(_("Load address:  $%04x\nInit address:  $%04x\nPlay address:  $%04x\n"),NSFInfo->LoadAddr,NSFInfo->InitAddr,NSFInfo->PlayAddr);
 FCEU_printf("%s\n",(NSFInfo->VideoSystem&1)?"PAL":"NTSC");
 FCEU_printf(_("Starting song:  %d / %d\n\n"),NSFInfo->StartingSong + 1,NSFInfo->TotalSongs);

 if(NSFInfo->SoundChip&4)
  ExWRAM=(uint8 *)malloc(32768+8192);
 else
  ExWRAM=(uint8 *)malloc(8192);

 FCEU_indent(-1);

 if(!ExWRAM)
 {
  FCEU_PrintError(_("Error allocating memory for NSF expansion RAM."));
  return(0);
 }

 return 1;
}

static DECLFR(NSFVectorRead)
{
 if(((NSFNMIFlags&1) && SongReload) || (NSFNMIFlags&2) || doreset)
 {
  if(A==0xFFFA) return(0x00);
  else if(A==0xFFFB) return(0x38);
  else if(A==0xFFFC) return(0x20);
  else if(A==0xFFFD) {doreset=0;return(0x38);}
  return(X.DB);
 }
 else
  return(CartBR(A));
}

int NSFFDS_Init(EXPSOUND *);
int NSFVRC6_Init(EXPSOUND *);
int NSFMMC5_Init(EXPSOUND *);
int NSFAY_Init(EXPSOUND *);
int NSFN106_Init(EXPSOUND *);
int NSFVRC7_Init(EXPSOUND *);

void NSF_init(void)
{
  doreset=1;

  ResetCartMapping();
  if(NSFInfo->SoundChip&4)
  {
   SetupCartPRGMapping(0,ExWRAM,32768+8192,1);
   setprg32(0x6000,0);
   setprg8(0xE000,4);
   memset(ExWRAM,0x00,32768+8192);
   SetWriteHandler(0x6000,0xDFFF,CartBW);
   SetReadHandler(0x6000,0xFFFF,CartBR);
  }
  else
  {
   memset(ExWRAM,0x00,8192);
   SetReadHandler(0x6000,0x7FFF,CartBR);
   SetWriteHandler(0x6000,0x7FFF,CartBW);
   SetupCartPRGMapping(0,NSFInfo->NSFDATA,((NSFInfo->NSFMaxBank+1)*4096),0);
   SetupCartPRGMapping(1,ExWRAM,8192,1);
   setprg8r(1,0x6000,0);
   SetReadHandler(0x8000,0xFFFF,CartBR);
  }

  if(BSon)
  {
   int32 x;
   for(x=0;x<8;x++)
   {
    if(NSFInfo->SoundChip&4 && x>=6)
     BANKSET(0x6000+(x-6)*4096,NSFInfo->BankSwitch[x]);
    BANKSET(0x8000+x*4096,NSFInfo->BankSwitch[x]);
   }
  }
  else
  {
   int32 x;
   for(x=(NSFInfo->LoadAddr&0xF000);x<0x10000;x+=0x1000)
    BANKSET(x,((x-(NSFInfo->LoadAddr&0xf000))>>12));
  }

  SetReadHandler(0xFFFA,0xFFFD,NSFVectorRead);

  SetWriteHandler(0x2000,0x3fff,0);
  SetReadHandler(0x2000,0x37ff,0);
  SetReadHandler(0x3836,0x3FFF,0);
  SetReadHandler(0x3800,0x3835,NSFROMRead);

  SetWriteHandler(0x5ff6,0x5fff,NSF_write);

  SetWriteHandler(0x3ff0,0x3fff,NSF_write);
  SetReadHandler(0x3ff0,0x3fff,NSF_read);


  if(NSFInfo->SoundChip&1) { 
   NSFVRC6_Init(&GameExpSound);
  } else if (NSFInfo->SoundChip&2) {
   NSFVRC7_Init(&GameExpSound);
  } else if (NSFInfo->SoundChip&4) {
   NSFFDS_Init(&GameExpSound);
  } else if (NSFInfo->SoundChip&8) {
   NSFMMC5_Init(&GameExpSound);
  } else if (NSFInfo->SoundChip&0x10) {
   NSFN106_Init(&GameExpSound);
  } else if (NSFInfo->SoundChip&0x20) {
   NSFAY_Init(&GameExpSound);
  }
  NSFInfo->CurrentSong=NSFInfo->StartingSong;
  SongReload=0xFF;
  NSFNMIFlags=0;
}

static DECLFW(NSF_write)
{
 switch(A)
 {
  case 0x3FF3:NSFNMIFlags|=1;break;
  case 0x3FF4:NSFNMIFlags&=~2;break;
  case 0x3FF5:NSFNMIFlags|=2;break;

  case 0x5FF6:
  case 0x5FF7:if(!(NSFInfo->SoundChip&4)) return;
  case 0x5FF8:
  case 0x5FF9:
  case 0x5FFA:
  case 0x5FFB:
  case 0x5FFC:
  case 0x5FFD:
  case 0x5FFE:
  case 0x5FFF:if(!BSon) return;
              A&=0xF;
              BANKSET((A*4096),V);
  	      break;
 } 
}

static DECLFR(NSF_read)
{
 int x;

 switch(A)
 {
 case 0x3ff0:x=SongReload;
	     if(!fceuindbg)
	      SongReload=0;
	     return x;
 case 0x3ff1:
	    if(!fceuindbg)
	    {
             memset(RAM,0x00,0x800);

             BWrite[0x4015](0x4015,0x0);
             for(x=0;x<0x14;x++)
              BWrite[0x4000+x](0x4000+x,0);
             BWrite[0x4015](0x4015,0xF);

	     if(NSFInfo->SoundChip&4) 
	     {
	      BWrite[0x4017](0x4017,0xC0);	/* FDS BIOS writes $C0 */
	      BWrite[0x4089](0x4089,0x80);
	      BWrite[0x408A](0x408A,0xE8);
	     }
	     else 
	     {
	      memset(ExWRAM,0x00,8192);
	      BWrite[0x4017](0x4017,0xC0);
              BWrite[0x4017](0x4017,0xC0);
              BWrite[0x4017](0x4017,0x40);
	     }

             if(BSon)
             {
              for(x=0;x<8;x++)
	       BANKSET(0x8000+x*4096,NSFInfo->BankSwitch[x]);
             }
             return (NSFInfo->CurrentSong);
 	     }
 case 0x3FF3:return PAL;
 }
 return 0;
}

uint8 FCEU_GetJoyJoy(void);

void DrawNSF(uint8 *XBuf)
{
 char snbuf[16];
 int x;

 if(vismode==0) return;

 memset(XBuf,3,256*240);


 {
  float *Bufpl;
  float mul=0;

  int l;
  l=GetSoundBuffer(&Bufpl);

  if(FSettings.SoundVolume)
   mul =  (240 / 2) * 300 / (float)FSettings.SoundVolume;

  for(x=0;x<256;x++)
  {
   float samp = Bufpl[(x * l) >> 8];
   uint32 y;
   y = (unsigned int)(120 + (samp - 0.5) * mul);
   if(y<240)
    XBuf[x+y*256]=5;
  }   
 }  

 if(NSFInfo->GameName)
  DrawTextTrans(XBuf+10*256, 256, NSFInfo->GameName, 1, 1);
 if(NSFInfo->Artist)
  DrawTextTrans(XBuf+26*256, 256, NSFInfo->Artist, 1, 1);
 if(NSFInfo->Copyright)
  DrawTextTrans(XBuf+42*256, 256, NSFInfo->Copyright, 1, 1);

 {
  UTF8 *tmpsong = NSFInfo->SongNames?NSFInfo->SongNames[NSFInfo->CurrentSong] : 0;

  if(!tmpsong)
   tmpsong = (UTF8 *)_("Song:");

  DrawTextTrans(XBuf+70*256, 256, tmpsong, 1, 1);
 }
 sprintf(snbuf,"<%d/%d>",NSFInfo->CurrentSong + 1,NSFInfo->TotalSongs);
 DrawTextTrans(XBuf+84*256, 256, (uint8*)snbuf, 1, 1);

 {
  static uint8 last=0;
  uint8 tmp;
  tmp=FCEU_GetJoyJoy();
  if((tmp&JOY_RIGHT) && !(last&JOY_RIGHT))
  {
   if(NSFInfo->CurrentSong<(NSFInfo->TotalSongs - 1))
   {
    NSFInfo->CurrentSong++;   
    SongReload=0xFF;
   }
  }
  else if((tmp&JOY_LEFT) && !(last&JOY_LEFT))
  {
   if(NSFInfo->CurrentSong>0)
   {
    NSFInfo->CurrentSong--;
    SongReload=0xFF;
   }
  }
  else if((tmp&JOY_UP) && !(last&JOY_UP))
  {
   NSFInfo->CurrentSong+=10;
   if(NSFInfo->CurrentSong>=NSFInfo->TotalSongs) NSFInfo->CurrentSong=NSFInfo->TotalSongs - 1;
   SongReload=0xFF;
  }
  else if((tmp&JOY_DOWN) && !(last&JOY_DOWN))
  {
   NSFInfo->CurrentSong-=10;
   if(NSFInfo->CurrentSong<0) NSFInfo->CurrentSong=0;
   SongReload=0xFF;
  }
  else if((tmp&JOY_START) && !(last&JOY_START))
   SongReload=0xFF;
  else if((tmp&JOY_A) && !(last&JOY_A))
  {

  }
  last=tmp;
 }
}

void DoNSFFrame(void)
{
 if(((NSFNMIFlags&1) && SongReload) || (NSFNMIFlags&2))
  TriggerNMI();
}

void FCEUI_NSFSetVis(int mode)
{
 vismode=mode;
}

int FCEUI_NSFChange(int amount)
{
   NSFInfo->CurrentSong+=amount;
   if(NSFInfo->CurrentSong<0) NSFInfo->CurrentSong=0;
   else if(NSFInfo->CurrentSong>=NSFInfo->TotalSongs) NSFInfo->CurrentSong=NSFInfo->TotalSongs - 1;
   SongReload=0xFF;

   return(NSFInfo->CurrentSong);
}

