/* FCE Ultra - NES/Famicom Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Xodnizel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef WIN32
#include <windows.h>
#endif

#include "main.h"

#include "../common/cheat.h"

#include "input.h"
#include "joystick.h"
#include "video.h"
#include "sound.h"
#ifdef NETWORK
#include "unix-netplay.h"
#endif
#ifdef OPENGL
int sdlhaveogl;
#endif

DSETTINGS Settings;

static void SetDefaults(void)
{
 Settings.special=Settings.specialfs=0;
 Settings.ffspeed = 4;
 Settings.fnscan = 1;
 _bpp=8;
 _xres=640;
 _yres=480;
 _fullscreen=0;
 _xscale=2.50;
 _yscale=2;
 _xscalefs=2.50;
 _yscalefs=2;
 _efx=_efxfs=0;
 //_fshack=_fshacksave=0;
 _doublebuf = 1;
#ifdef OPENGL
 _opengl=1;
 _stretchx=0;
 _stretchy=0;
 _openglip=1;
#endif
}

static uint8 *VTBuffer = NULL;
static SDL_mutex *VTMutex = NULL;
static SDL_cond *VTCond = NULL;
static volatile int VTReady;
static volatile int VTRunning;



static char *soundrecfn=0;	/* File name of sound recording. */

static int ntsccol=0,ntschue=0,ntsctint=0;
int soundvol=100;
long soundq=0;
int _sound=1;
long soundrate=48000;
#ifdef WIN32
long soundbufsize=52;
#else
long soundbufsize=32;
#endif

static int frameskip=0;
static int inited=0;
static int isloaded=0;	// Is game loaded?

int srendlinev[2]={8,0};
int erendlinev[2]={231,239};


static char *DrBaseDirectory;

int eoptions=0;

static void DriverKill(void);
static int DriverInitialize(FCEUGI *gi);
int gametype;

FCEUGI *CurGame=NULL;

static void ParseGI(FCEUGI *gi)
{
 ParseGIInput(gi);
 gametype=gi->type;
}

void FCEUD_PrintError(char *s)
{
 puts(s);
}

void FCEUD_Message(char *s)
{
 fputs(s,stdout);
}

static char *cpalette=0;
static void LoadCPalette(void)
{
 uint8 tmpp[192];
 FILE *fp;

 if(!(fp=fopen(cpalette,"rb")))
 {
  printf(_("Error loading custom palette from file: %s\n"),cpalette);
  return;
 }
 fread(tmpp,1,192,fp);
 FCEUI_SetPaletteArray(tmpp);
 fclose(fp);
}
static CFGSTRUCT fceuconfig[]={
	AC(soundrate),
	AC(soundq),
	AC(_sound),
	AC(soundvol),
	AC(soundbufsize),
	ACS(cpalette),
	AC(ntsctint),
	AC(ntschue),
	AC(ntsccol),
	AC(eoptions),
	ACA(srendlinev),
	ACA(erendlinev),
	ADDCFGSTRUCT(InputConfig),
        #ifdef OPENGL
        AC(_stretchx),
        AC(_stretchy),
        AC(_opengl),
        AC(_openglip),
        #endif
        AC(Settings.special),
        AC(Settings.specialfs),
        AC(_doublebuf),
        AC(_xscale),
        AC(_yscale),
        AC(_xscalefs),
        AC(_yscalefs),
        AC(_bpp),
        AC(_efx),
        AC(_efxfs),
        AC(_fullscreen),
        AC(_xres),
        AC(_yres),
        #ifdef NETWORK
        ACS(netplaynick),
        AC(netlocalplayers),
        AC(tport),
        ACS(netpassword),
        ACS(netgamekey),
        #endif
	AC(Settings.ffspeed),
	AC(Settings.fnscan),
	//ADDCFGSTRUCT(DriverConfig),
	ENDCFGSTRUCT
};

static void SaveConfig(void)
{	
	char tdir[2048];
	sprintf(tdir,"%s"PSS"nintencer.cfg",DrBaseDirectory);
	FCEUI_GetNTSCTH(&ntsctint, &ntschue);
        SaveFCEUConfig(tdir,fceuconfig);
}

static void LoadConfig(void)
{
	char tdir[2048];
        sprintf(tdir,"%s"PSS"nintencer.cfg",DrBaseDirectory);
	FCEUI_GetNTSCTH(&ntsctint, &ntschue);	/* Get default settings for if
					   no config file exists. */
        LoadFCEUConfig(tdir,fceuconfig);
	InputUserActiveFix();
}

static void CreateDirs(void)
{
 char *subs[6]={"ncs","ncm","snaps","gameinfo","sav","cheats"};
 char tdir[2048];
 int x;

 #ifdef WIN32
 mkdir(DrBaseDirectory);
 for(x=0;x<6;x++)
 {
  sprintf(tdir,"%s"PSS"%s",DrBaseDirectory,subs[x]);
  mkdir(tdir);
 }
 #else
 mkdir(DrBaseDirectory,S_IRWXU);
 for(x=0;x<6;x++)
 {
  sprintf(tdir,"%s"PSS"%s",DrBaseDirectory,subs[x]);
  mkdir(tdir,S_IRWXU);
 }
 #endif
}

#ifndef WIN32
static void SetSignals(void (*t)(int))
{
  int sigs[11]={SIGINT,SIGTERM,SIGHUP,SIGPIPE,SIGSEGV,SIGFPE,SIGKILL,SIGALRM,SIGABRT,SIGUSR1,SIGUSR2};
  int x;
  for(x=0;x<11;x++)
   signal(sigs[x],t);
}

static void CloseStuff(int signum)
{
	DriverKill();
        printf(_("\nSignal %d has been caught and dealt with...\n"),signum);
        switch(signum)
        {
         case SIGINT:printf(_("How DARE you interrupt me!\n"));break;
         case SIGTERM:printf(_("MUST TERMINATE ALL HUMANS\n"));break;
         case SIGHUP:printf(_("Reach out and hang-up on someone.\n"));break;
         case SIGPIPE:printf(_("The pipe has broken!  Better watch out for floods...\n"));break;
         case SIGSEGV:printf(_("Iyeeeeeeeee!!!  A segmentation fault has occurred.  Have a fluffy day.\n"));break;
	 /* So much SIGBUS evil. */
	 #ifdef SIGBUS
	 #if(SIGBUS!=SIGSEGV)
         case SIGBUS:printf(_("I told you to be nice to the driver.\n"));break;
	 #endif
	 #endif
         case SIGFPE:printf(_("Those darn floating points.  Ne'er know when they'll bite!\n"));break;
         case SIGALRM:printf(_("Don't throw your clock at the meowing cats!\n"));break;
         case SIGABRT:printf(_("Abort, Retry, Ignore, Fail?\n"));break;
         case SIGUSR1:
         case SIGUSR2:printf(_("Killing your processes is not nice.\n"));break;
        }
        exit(1);
}
#endif

static int DoArgs(int argc, char *argv[])
{
        static ARGPSTRUCT FCEUArgs[]={
	 {"-soundbufsize",0,&soundbufsize,0},
	 {"-soundrate",0,&soundrate,0},
	 {"-soundq",0,&soundq,0},
	 {"-frameskip",0,&frameskip,0},
	 {"-ffspeed", 0, &Settings.ffspeed, 0},
         {"-sound",0,&_sound,0},
         {"-soundvol",0,&soundvol,0},
         {"-cpalette",0,&cpalette,0x4001},
	 {"-soundrecord",0,&soundrecfn,0x4001},

         {"-ntsccol",0,&ntsccol,0},
         {"-pal",0,&eoptions,0x8000|EO_PAL},

	 {"-autosave",0,&eoptions,0x8000|EO_AUTOSAVE},
         {"-gg",0,&eoptions,0x8000|EO_GAMEGENIE},
         {"-no8lim",0,&eoptions,0x8001},
         {"-snapname",0,&eoptions,0x8000|EO_SNAPNAME},
	 {"-nofs",0,&eoptions,0x8000|EO_NOFOURSCORE},
         {"-clipsides",0,&eoptions,0x8000|EO_CLIPSIDES},
	 {"-nothrottle",0,&eoptions,0x8000|EO_NOTHROTTLE},
         {"-slstart",0,&srendlinev[0],0},{"-slend",0,&erendlinev[0],0},
         {"-slstartp",0,&srendlinev[1],0},{"-slendp",0,&erendlinev[1],0},
	 {0,(int *)InputArgs,0,0},
        #ifdef OPENGL
         {"-opengl",0,&_opengl,0},
         {"-openglip",0,&_openglip,0},
         {"-stretchx",0,&_stretchx,0},
         {"-stretchy",0,&_stretchy,0},
        #endif
         {"-special",0,&Settings.special,0},
         {"-specialfs",0,&Settings.specialfs,0},
         {"-doublebuf",0,&_doublebuf,0},
         {"-bpp",0,&_bpp,0},
         {"-xscale",0,&_xscale,2},
         {"-yscale",0,&_yscale,2},
         {"-efx",0,&_efx,0},
         {"-xscalefs",0,&_xscalefs,2},
         {"-yscalefs",0,&_yscalefs,2},
         {"-efxfs",0,&_efxfs,0},
         {"-xres",0,&_xres,0},
         {"-yres",0,&_yres,0},
         {"-fs",0,&_fullscreen,0},
	 {"-fnscan", 0, &Settings.fnscan, 0},
         //{"-fshack",0,&_fshack,0x4001},
         #ifdef NETWORK
         {"-connect",0,&netplayhost,0x4001},
         {"-netport",0,&tport,0},
         {"-netlocalplayers",0,&netlocalplayers,0},
         {"-netnick",0,&netplaynick,0x4001},
         {"-netpassword",0,&netpassword,0x4001},
         #endif

	 //{0,(int *)DriverArgs,0,0},
	 {0,0,0,0}
        };

	if(argc == 0)	// If argc == 0, does the world end?
	{
	 return(0);
	}
	else if(argc == 1)
	{
	 unsigned int x;

	 printf(_("No command-line arguments specified.\n\n"));
	 printf(_("Usage: %s [OPTION]... [FILE]\n"), argv[0]);
	 printf(_("Valid options are:  "));

	 for(x=0;x<sizeof(FCEUArgs) / sizeof(ARGPSTRUCT); x++)
	 {
	  if(FCEUArgs[x].name) printf("\"%s\" ", FCEUArgs[x].name);
	 }
	 printf(_("\n\nPlease refer to the documentation for option parameters and usage.\n\n"));
	 return(0);
	}
	else if(argc == 2)
	 return(1);
	else
	 ParseArguments(argc - 2, &argv[1], FCEUArgs);

	if(cpalette)
	{
  	 if(cpalette[0]=='0')
	  if(cpalette[1]==0)
	  {
	   free(cpalette);
	   cpalette=0;
	  }
	}
	return(1);
}

void DoPostArgs(void)
{
	int x;

	if(Settings.ffspeed < 1) Settings.ffspeed = 1;
	if(Settings.ffspeed > 10) Settings.ffspeed = 10;

	FCEUI_SetVidSystem((eoptions&EO_PAL)?1:0);
	FCEUI_SetGameGenie((eoptions&EO_GAMEGENIE)?1:0);
	//FCEUI_SetLowPass((eoptions&EO_LOWPASS)?1:0);

        FCEUI_DisableSpriteLimitation(eoptions&1);
	FCEUI_SetSnapName(eoptions&EO_SNAPNAME);

	for(x=0;x<2;x++)
	{
         if(srendlinev[x]<0 || srendlinev[x]>239) srendlinev[x]=0;
         if(erendlinev[x]<srendlinev[x] || erendlinev[x]>239) erendlinev[x]=239;
	}

        FCEUI_SetRenderedLines(srendlinev[0],erendlinev[0],srendlinev[1],erendlinev[1]);
}

/* Loads a game, given a full path/filename.  The driver code must be
   initialized after the game is loaded, because the emulator code
   provides data necessary for the driver code(number of scanlines to
   render, what virtual input devices to use, etc.).
*/
int LoadGame(const char *path)
{
	FCEUGI *tmp;

	CloseGame();

        if(Settings.fnscan)
        {
         if(strstr(path, "(U)"))
          FCEUI_SetVidSystem(0);
         else if(strstr(path, "(J)"))
          FCEUI_SetVidSystem(0);
         else if(strstr(path, "(E)"))
          FCEUI_SetVidSystem(1);
         else
          FCEUI_SetVidSystem((eoptions&EO_PAL)?1:0);
        }
        else
         FCEUI_SetVidSystem((eoptions&EO_PAL)?1:0);

        if(!(tmp=FCEUI_LoadGame(path)))
	 return 0;
	CurGame=tmp;
        ParseGI(tmp);
        RefreshThrottleFPS(1);

        if(!DriverInitialize(tmp))
         return(0);  

	if(eoptions & EO_AUTOSAVE)
	 NTCRI_LoadState(NULL, "ncq");

	if(soundrecfn)
	{
	 if(!FCEUI_BeginWaveRecord(soundrecfn))
	 {
 	  free(soundrecfn);
	  soundrecfn=0;
	 }
	}
	isloaded=1;

	#ifdef NETWORK
	FCEUD_NetworkConnect();
	#endif
	return 1;
}

/* Closes a game and frees memory. */
int CloseGame(void)
{
	if(!isloaded) return(0);
	
	if(eoptions & EO_AUTOSAVE)
	 NTCRI_SaveState(NULL, "ncq");

	FCEUI_CloseGame();
	//DriverKill();
	isloaded=0;
	CurGame=0;

	if(soundrecfn)
         FCEUI_EndWaveRecord();

	InputUserActiveFix();

	//puts("Closed");
	return(1);
}

void FCEUD_Update(uint8 *XBuf, float *Buffer, int Count);

static volatile int NeedExitNow = 0;
int CurGameSpeed = 1;
int GameLoop(void *arg)
{
	while(CurGame)
	{
         uint8 *gfx;  
         float *sound;
         int32 ssize;
         static int fskipc=0;
        
         if(CurGameSpeed > 1 && CurGameSpeed > (frameskip+1))
	  fskipc=(fskipc+1)%CurGameSpeed;
	 else
          fskipc=(fskipc+1)%(frameskip+1);
         
         if(NoWaiting) {gfx=0;}
         FCEUI_Emulate(&gfx, &sound, &ssize, fskipc);
         FCEUD_Update(gfx, sound, ssize);

	 if(NeedExitNow)
	  CloseGame();
	}

	SDL_mutexP(VTMutex);
	VTRunning = 0;
	SDL_mutexV(VTMutex);

	return(1);
}   

char *GetBaseDirectory(void)
{
 char *ol;
 char *ret;

 ol=getenv("HOME");

 if(ol)
 {
  ret=(char *)malloc(strlen(ol)+1+strlen("/.nintencer"));
  strcpy(ret,ol);
  strcat(ret,"/.nintencer");
 }
 else
 {
  #ifdef WIN32
  char *sa;

  ret=(char *)malloc(MAX_PATH+1);
  GetModuleFileName(NULL,ret,MAX_PATH+1);

  sa=strrchr(ret,'\\');
  if(sa)
   *sa = 0;
  #else
  ret=(char *)malloc(1);
  ret[0]=0;
  #endif
  printf("%s\n",ret);
 }
 return(ret);
}

static volatile int NeedVideoChange = 0;

void PumpWrap(void)
{
 SDL_Event event;

 while(SDL_PollEvent(&event))
 {
  switch(event.type)
  {
   //case SDL_SYSWMEVENT: puts("Nifty keen");break;
   //case SDL_VIDEORESIZE: puts("Okie dokie");break;
   case SDL_VIDEOEXPOSE: ClearVideoSurfaces();break;
   case SDL_QUIT: NeedExitNow = 1;break;
  }
 }
}

static SDL_Thread *GameThread;

int main(int argc, char *argv[])
{
	int ret;

	DrBaseDirectory=GetBaseDirectory();

	#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");
	#ifdef WIN32
        bindtextdomain(PACKAGE, DrBaseDirectory);
        bind_textdomain_codeset(PACKAGE, "UTF-8");
        textdomain(PACKAGE);
	#else
	bindtextdomain(PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);
	#endif
	#endif
        if(!(ret=FCEUI_Initialize()))
         return(-1);

	FCEUI_printf(_("Starting Nintencer %s\n"), NINTENCER_VERSION);
	FCEU_indent(1);

	if(SDL_Init(SDL_INIT_VIDEO)) /* SDL_INIT_VIDEO Needed for (joystick config) event processing? */
	{
	 printf("Could not initialize SDL: %s.\n", SDL_GetError());
	 return(-1);
	}

        #ifdef OPENGL
         #ifdef APPLEOPENGL
         sdlhaveogl = 1;        /* Stupid something...  Hack. */
         #else
         if(!SDL_GL_LoadLibrary(0)) sdlhaveogl=1;
         else sdlhaveogl=0;
         #endif
        #endif

	SetDefaults();

	FCEUI_SetBaseDirectory(DrBaseDirectory);

	CreateDirs();

        LoadConfig();

	if(!DoArgs(argc,argv))
	{
	 return(-1);
	}

	DoPostArgs();
	FCEUI_SetNTSCTH(ntsccol, ntsctint, ntschue);
	if(cpalette)
	 LoadCPalette();

       	if(!LoadGame(argv[argc-1]))
        {
         DriverKill();
	 SDL_Quit();
         return(-1);
        }

	/* Now the fun begins! */
	/* Run the video and event pumping in the main thread, and create a 
	   secondary thread to run the game in(and do sound output, since we use
	   separate sound code which should be thread safe(?)).
	*/

	VTRunning = 0;
	VTMutex = SDL_CreateMutex();
	VTCond = SDL_CreateCond();
	VTBuffer = (uint8 *)malloc(256 * 240);
	VTReady = 1;

	VTRunning = 1;

	GameThread = SDL_CreateThread(GameLoop, NULL);

	while(VTRunning)
	{
	 SDL_mutexP(VTMutex);	/* Lock mutex */

	 if(NeedVideoChange)
	 {
	  KillVideo(0);
	  _fullscreen=!_fullscreen;

	  if(!InitVideo(CurGame))
	  {
	   _fullscreen=!_fullscreen;
	   InitVideo(CurGame);
	  }
	  NeedVideoChange = 0;
	 }

	 PumpWrap();
	 if(!VTReady)
	 {
	  BlitScreen(VTBuffer);
	  VTReady = 1;
	 } 
	 SDL_mutexV(VTMutex);	/* Unlock mutex */
	 PumpWrap();
         SDL_Delay(1);
	}

	SDL_WaitThread(GameThread, NULL);

	//puts("Main loop ended");
	//SDL_mutexV(VTMutex);
	SDL_DestroyMutex(VTMutex);

	if(VTBuffer)
	{
	 free(VTBuffer);
	 VTBuffer = NULL;
	}

        CloseGame();
	SaveConfig();
	DriverKill();
        FCEUI_Kill();

	SDL_Quit();
        return(0);
}

void ToggleFS(void)
{
 SDL_mutexP(VTMutex);
 NeedVideoChange = 1;
 SDL_mutexV(VTMutex);
}



static int DriverInitialize(FCEUGI *gi)
{
	#ifndef WIN32
	SetSignals(CloseStuff);
	#endif

	/* Initialize video before all else, due to some wacko dependencies
	   in the SexyAL code(DirectSound) that need to be fixed.
	*/

        if(!InitVideo(gi)) return 0;
        inited|=4;

	if(InitSound(gi))
	 inited|=1;

	if(InitJoysticks())
	 inited|=2;

	InitOtherInput();
	return 1;
}

static void DriverKill(void)
{
 SaveConfig();

 #ifndef WIN32
 SetSignals(SIG_IGN);
 #endif

 if(inited&2)
  KillJoysticks();
 if(inited&4)
  KillVideo(1);
 if(inited&1)
  KillSound();
 inited=0;
}

static uint64 tfreq;
static uint64 desiredfps;

void RefreshThrottleFPS(int multiplier)
{
        desiredfps=(FCEUI_GetDesiredFPS() * multiplier)>>8;
        tfreq=10000000;
        tfreq<<=16;    /* Adjustment for fps returned from FCEUI_GetDesiredFPS(). */
	CurGameSpeed = multiplier;
}

void FCEUD_Update(uint8 *XBuf, float *Buffer, int Count)
{
 static uint64 ttime,ltime=0;
 static int skipcount = 0;
 int soundfull = 0;
 #ifdef NETWORK
 extern int FCEUDnetplay;
 #endif

 if(Count)
 {
  int32 max = GetWriteSound();
  if(Count > max)
  {
   if(NoWaiting)
    Count = max;
  }
  if(Count >= (max * 0.95))
  {
   soundfull = 1;
  }
  WriteSound(Buffer, Count);
 }

 waiter:

 ttime=SDL_GetTicks();
 ttime*=10000;

 if((ttime - ltime) < (tfreq / desiredfps ))
 {
  if(!Count && !NoWaiting && !(eoptions & EO_NOTHROTTLE))
  {
   int64 delay;
   delay=(tfreq/desiredfps)-(ttime-ltime);
   if(delay>0)
    SDL_Delay(delay/10000);
   goto waiter;
  }
 }

 if(soundfull)
  ltime = ttime;
 else if(((ttime-ltime) >= (1.3*tfreq/desiredfps)))
 {
  //puts("Oops");
  if(skipcount < 4)	// Only skip four frames in a row at maximum.
  {
   skipcount ++;
   XBuf = 0; // Skip frame.
  } else skipcount = 0;
  if(!Count)	// Only adjust base time if sound is disabled.
   ltime=ttime;
 }
 else 
  ltime+=tfreq/desiredfps;

 FCEUD_UpdateInput();

 if(XBuf && CurGame)
  FCEUI_DrawOverlay(XBuf);

 //DrawGUI(XBuf);

 if(VTReady)
 {
  if(XBuf)
  {
   skipcount = 0;
   memcpy(VTBuffer, XBuf, 256*240);
   VTReady = 0;
  }
 }
 else
 {

 }
}


FILE *FCEUD_UTF8fopen(const char *fn, const char *mode)
{
 return(fopen(fn,mode));
}

uint64 FCEUD_GetTime(void)
{
 return(SDL_GetTicks());
}

uint64 FCEUD_GetTimeFreq(void)
{
 return(1000);
}
