/* FCE Ultra - NES/Famicom Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Xodnizel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <zlib.h>

#include "types.h"
#include "video.h"
#include "nintencer.h"
#include "general.h"
#include "memory.h"
#include "crc32.h"
#include "state.h"
#include "movie.h"
#include "ppu/palette.h"
#include "nsf.h"
#include "input.h"
#include "vsuni.h"

#include "string/ConvertUTF.h"

uint8 *XBuf=NULL;
static uint8 *xbsave=NULL;

typedef struct
{
        uint16 glyph_num;
        uint8 data[13];
} font6x13;

typedef struct
{
        uint16 glyph_num;
        uint8 data[13 * 2];
} font12x13;

static font6x13 FontData6x13[]=
{
        #include "font6x13.h"
};

static font12x13 FontData12x13[]=
{
        #include "font12x13.h"
};

static uint8 FontDataCache6x13[65536][13];
static uint8 FontDataCache12x13[65536][13 * 2];

static uint8 FontDataHave6x13[65536];
static uint8 FontDataHave12x13[65536];

static void InitFontData(void)
{
 unsigned int x;
 unsigned int inx;

 for(inx=x=0;x<65536;x++)
 {
  if(inx < (sizeof(FontData6x13) / sizeof(font6x13)) && FontData6x13[inx].glyph_num == x)
  {
   memcpy(FontDataCache6x13[x], FontData6x13[inx].data, 13);
   FontDataHave6x13[x] = 1;
   inx++;
  }
  else
  {
   FontDataHave6x13[x] = 0;
   memset(FontDataCache6x13[x], 0, 13);
  }
 }


 for(inx=x=0;x<65536;x++)
 {
  if(inx < (sizeof(FontData12x13) / sizeof(font12x13)) && FontData12x13[inx].glyph_num == x)
  {
   memcpy(FontDataCache12x13[x], FontData12x13[inx].data, 13 * 2);
   FontDataHave12x13[x] = 1;
   inx++;
  }
  else
  {
   FontDataHave12x13[x] = 0;
   memset(FontDataCache12x13[x], 0, 13);
  }
 }

}

static long min(long a1, long a2)
{
 if(a1 < a2)
  return(a1);
 return(a2);
}

void DrawTextTrans(uint8 *dest, uint32 width, uint8 *msg, uint8 fgcolor, int centered)
{
 UTF32 utf32_buf[256];
 UTF32 *tstart = utf32_buf;
 uint32 slen;
 uint32 pixwidth;
 uint32 x;

 uint8 *glyph_ptrs[256];
 uint8 glyph_width[256];
 uint8 glyph_ov_width[256];

 ConvertUTF8toUTF32((const UTF8**)&msg, &msg[strlen((char *)msg)], &tstart, &tstart[256], lenientConversion);

 slen = (tstart - utf32_buf); 

 pixwidth = 0;

 for(x=0;x<slen;x++)
 {
  if(FontDataHave6x13[utf32_buf[x]])
  {
   glyph_ptrs[x] = FontDataCache6x13[utf32_buf[x]];
   glyph_width[x] = 6;
   pixwidth += 6;
  }
  else if(FontDataHave12x13[utf32_buf[x]])
  {
   glyph_ptrs[x] = FontDataCache12x13[utf32_buf[x]];
   glyph_width[x] = 12;
   pixwidth += 12;
  }
  else
  {
   glyph_ptrs[x] = FontDataCache6x13['?'];
   glyph_width[x] = 6;
   pixwidth += 6;
  }
  if(utf32_buf[x] >= 0x0300 && utf32_buf[x] <= 0x036F)
  {
   if(x != (slen-1)) 
   {
    pixwidth -= min(glyph_width[x], glyph_width[x + 1]);
   }
   glyph_ov_width[x] = 0;
  }
  else
   glyph_ov_width[x] = glyph_width[x];

  if(pixwidth > width) // Oopsies, it's too big for the screen!  Just truncate it for now.
  {
   slen = x;
   pixwidth -= glyph_width[x];
   break;
  }
 }

 if(centered)
 {
  int32 poot = width - pixwidth;

  dest += poot / 2;
 }

 for(x=0;x<slen;x++)
 {
  int gx, gy;
  uint8 *src_glyph = glyph_ptrs[x];

  if(glyph_width[x] > 8)
  {
   for(gy=0;gy<13;gy++)
   {
    for(gx=0;gx<glyph_width[x];gx++)
    {
      if((src_glyph[gy * 2 + (gx >> 3)] << (gx & 0x7)) & 0x80) dest[gy * 256 + gx] = fgcolor;
    }
   }
   //dest += glyph_ov_width[x];
  }
  else
  {
   for(gy=0;gy<13;gy++)
   {
    for(gx=0;gx<glyph_width[x];gx++)
    {
     if((src_glyph[gy] << gx) & 0x80) dest[gy * 256 + gx] = fgcolor;
    }
   }
   //dest += glyph_width[x];
  }
  dest += glyph_ov_width[x];
 }
}

void FCEU_KillVirtualVideo(void)
{
 if(xbsave)
 {
  free(xbsave);
  xbsave=0;
 }
}

int FCEU_InitVirtualVideo(void)
{
 InitFontData();

 if(!XBuf)		/* Some driver code may allocate XBuf externally. */
			/* 256 bytes per scanline, * 240 scanline maximum, +8 for alignment,
			*/
 if(!(XBuf= (uint8*) (FCEU_malloc(256 * 256 + 8))))
  return 0;
 xbsave=XBuf;

 if(sizeof(uint8*)==4)
 {
  uint32 m;
  m=(uint32)XBuf;
  m=(4-m)&3;
  XBuf+=m;
 } 
 memset(XBuf,128,256*256); //*240);
 return 1;
}

static int howlong;
static UTF8 errmsg[65];

#include "drawing.h"

//#define SHOWFPS
void ShowFPS(void);
void FCEU_PutImageDummy(void)
{
 #ifdef SHOWFPS
 ShowFPS();
 #endif
 if(FCEUGameInfo->type!=GIT_NSF)
 {
  FCEU_DrawNTSCControlBars(XBuf);
  FCEU_DrawSaveStates(XBuf);
  FCEU_DrawMovies(XBuf);
 }
 if(howlong) howlong--; /* DrawMessage() */
}

void FCEUI_SaveSnapshot(void)
{
 int x=SaveSnapshot();
 if(!x)
  FCEU_DispMessage((UTF8 *)_("Error saving screen snapshot."));
 else
  FCEU_DispMessage((UTF8 *)_("Screen snapshot %d saved."),x-1);
}

void FCEU_PutImage(void)
{
        if(FCEUGameInfo->type==GIT_NSF)
        {
         DrawNSF(XBuf);
         /* Save snapshot after NSF screen is drawn.  Why would we want to
            do it before?
         */
        } 
}

void FCEUI_DrawOverlay(uint8 *XBuf)
{
        if(FCEUGameInfo->type != GIT_NSF)
        {   
         if(FCEUGameInfo->type==GIT_VSUNI)
          FCEU_VSUniDraw(XBuf);
	 
	 FCEU_DrawSaveStates(XBuf);
	 FCEU_DrawMovies(XBuf);
         FCEU_DrawNTSCControlBars(XBuf);
	 
        }
        DrawMessage();
        FCEU_DrawInput(XBuf);
        #ifdef SHOWFPS
        ShowFPS();
        #endif
}

void FCEU_DispMessage(UTF8 *format, ...)
{
 va_list ap;

 va_start(ap,format);
 vsprintf((char *)errmsg,(char *)format,ap);
 va_end(ap);

 howlong=180;
}

void FCEU_ResetMessages(void)
{
 howlong=0;
}


static int WritePNGChunk(FILE *fp, uint32 size, char *type, uint8 *data)
{
 uint32 crc;

 uint8 tempo[4];

 tempo[0]=size>>24;
 tempo[1]=size>>16;
 tempo[2]=size>>8;
 tempo[3]=size;

 if(fwrite(tempo,4,1,fp)!=1)
  return 0;
 if(fwrite(type,4,1,fp)!=1)
  return 0;

 if(size)
  if(fwrite(data,1,size,fp)!=size)
   return 0;

 crc=CalcCRC32(0,(uint8 *)type,4);
 if(size)
  crc=CalcCRC32(crc,data,size);

 tempo[0]=crc>>24;
 tempo[1]=crc>>16;
 tempo[2]=crc>>8;
 tempo[3]=crc;

 if(fwrite(tempo,4,1,fp)!=1)
  return 0;
 return 1;
}

int SaveSnapshot(void)
{
 static unsigned int lastu=0;

 char *fn=0;
 int totallines=FSettings.LastSLine-FSettings.FirstSLine+1;
 int x,u,y;
 FILE *pp=NULL;
 uint8 *compmem=NULL;
 uLongf compmemsize=totallines*263+12;

 if(!(compmem=(uint8 *)FCEU_malloc(compmemsize)))
  return 0;

 for(u=lastu;u<99999;u++)
 {
  pp=FCEUD_UTF8fopen((fn=FCEU_MakeFName(FCEUMKF_SNAP,u,"png")),"rb");
  if(pp==NULL) break;
  fclose(pp);
 }

 lastu=u;

 if(!(pp=FCEUD_UTF8fopen(fn,"wb")))
 {
  free(fn);
  return 0;
 }
 free(fn);
 {
  static uint8 header[8]={137,80,78,71,13,10,26,10};
  if(fwrite(header,8,1,pp)!=1)
   goto PNGerr;
 }

 {
  uint8 chunko[13];

  chunko[0]=chunko[1]=chunko[3]=0;
  chunko[2]=0x1;			// Width of 256

  chunko[4]=chunko[5]=chunko[6]=0;
  chunko[7]=totallines;			// Height

  chunko[8]=8;				// bit depth
  chunko[9]=3;				// Color type; indexed 8-bit
  chunko[10]=0;				// compression: deflate
  chunko[11]=0;				// Basic adapative filter set(though none are used).
  chunko[12]=0;				// No interlace.

  if(!WritePNGChunk(pp,13,"IHDR",chunko))
   goto PNGerr;
 }

 {
  uint8 pdata[256*3];
  for(x=0;x<256;x++)
   FCEUD_GetPalette(x,pdata+x*3,pdata+x*3+1,pdata+x*3+2);
  if(!WritePNGChunk(pp,256*3,"PLTE",pdata))
   goto PNGerr;
 }

 {
  uint8 *tmp=XBuf+FSettings.FirstSLine*256;
  uint8 *dest,*mal,*mork;

  if(!(mal=mork=dest=(uint8 *)malloc((totallines<<8)+totallines)))
   goto PNGerr;
 //   mork=dest=XBuf;

  for(y=0;y<totallines;y++)
  {
   *dest=0;			// No filter.
   dest++;
   for(x=256;x;x--,tmp++,dest++)
    *dest=*tmp; 	
  }

  if(compress(compmem,&compmemsize,mork,(totallines<<8)+totallines)!=Z_OK)
  {
   if(mal) free(mal);
   goto PNGerr;
  }
  if(mal) free(mal);
  if(!WritePNGChunk(pp,compmemsize,"IDAT",compmem))
   goto PNGerr;
 }
 if(!WritePNGChunk(pp,0,"IEND",0))
  goto PNGerr;

 free(compmem);
 fclose(pp);

 return u+1;


 PNGerr:
 if(compmem)
  free(compmem);
 if(pp)
  fclose(pp);
 return(0);
}

#ifdef SHOWFPS
uint64 FCEUD_GetTime(void);
uint64 FCEUD_GetTimeFreq(void);

static uint64 boop[60];
static int boopcount = 0;

void ShowFPS(void)
{ 
 uint64 da = FCEUD_GetTime() - boop[boopcount];
 char fpsmsg[16];
 int booplimit = PAL?50:60;
 boop[boopcount] = FCEUD_GetTime();

 sprintf(fpsmsg, "%8.1f",(double)booplimit / ((double)da / FCEUD_GetTimeFreq()));
 DrawTextTrans(XBuf + (256-8-8*8) + (FSettings.FirstSLine+4)*256,256,(uint8 *)fpsmsg,4, 0);
 // It's not averaging FPS over exactly 1 second, but it's close enough.
 boopcount = (boopcount + 1) % booplimit;
}
#endif
