/* FCE Ultra - NES/Famicom Emulator
 *
 * Copyright notice for this file:
 *  Copyright (C) 2002 Xodnizel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <zlib.h>
#include "unzip.h"

#include "types.h"
#include "file.h"
#include "endian.h"
#include "memory.h"
#include "driver.h"
#include "general.h"

void ApplyIPS(FILE *ips, FCEUFILE *dest)
{
 uint8 header[5];
 uint32 count=0;
 
 FCEU_printf(" Applying IPS...\n");
 if(fread(header,1,5,ips)!=5)
 {
  fclose(ips);
  return;
 }
 if(memcmp(header,"PATCH",5))
 {
  fclose(ips);
  return;
 }

 while(fread(header,1,3,ips)==3)
 {
  uint32 offset=(header[0]<<16)|(header[1]<<8)|header[2];
  uint16 size;

  if(!memcmp(header,"EOF",3))
  {
   FCEU_printf(" IPS EOF:  Did %d patches\n\n",count);
   fclose(ips);
   return;
  }

  size=fgetc(ips)<<8;
  size|=fgetc(ips);
  if(!size)	/* RLE */
  {
   uint8 *start;
   uint8 b;
   size=fgetc(ips)<<8;
   size|=fgetc(ips);

   //FCEU_printf("  Offset: %8d  Size: %5d RLE\n",offset,size);

   if((offset+size)>dest->size)
   {
    uint8 *tmp;

    // Probably a little slow.
    tmp=(uint8 *)realloc(dest->data,offset+size);
    if(!tmp)
    {
     FCEU_printf("  Oops.  IPS patch %d(type RLE) goes beyond end of file.  Could not allocate memory.\n",count);
     fclose(ips);
     return;
    }
    dest->size=offset+size;
    dest->data=tmp;
    memset(dest->data+dest->size,0,offset+size-dest->size);
   }
   b=fgetc(ips);
   start=dest->data+offset;
   do
   {
    *start=b;
    start++;
   } while(--size);
  }
  else		/* Normal patch */
  {
   //FCEU_printf("  Offset: %8d  Size: %5d\n",offset,size);
   if((offset+size)>dest->size)
   {
    uint8 *tmp;
    
    // Probably a little slow.
    tmp=(uint8 *)realloc(dest->data,offset+size);
    if(!tmp)
    {
     FCEU_printf("  Oops.  IPS patch %d(type normal) goes beyond end of file.  Could not allocate memory.\n",count);
     fclose(ips);
     return;
    }
    dest->data=tmp;
    memset(dest->data+dest->size,0,offset+size-dest->size);
   }
   fread(dest->data+offset,1,size,ips);
  }
  count++;
 }
 fclose(ips);
 FCEU_printf(" Hard IPS end!\n");
}

static FCEUFILE *MakeMemWrap(void *tz, int type)
{
 FCEUFILE *tmp;

 if(!(tmp=(FCEUFILE *)FCEU_malloc(sizeof(FCEUFILE))))
  goto doret;
 tmp->location=0;

 if(type==0)
 {
  fseek((FILE *)tz,0,SEEK_END);
  tmp->size=ftell((FILE *)tz);
  fseek((FILE *)tz,0,SEEK_SET);
  if(!(tmp->data=(uint8*)FCEU_malloc(tmp->size)))
  {
   free(tmp);
   tmp=0;
   goto doret;
  }
  fread(tmp->data,1,tmp->size,(FILE *)tz);
 }
 else if(type==1)
 {
  /* Bleck.  The gzip file format has the size of the uncompressed data,
     but I can't get to the info with the zlib interface(?). */
  for(tmp->size=0; gzgetc(tz) != EOF; tmp->size++);
  gzseek(tz,0,SEEK_SET);
  if(!(tmp->data=(uint8 *)FCEU_malloc(tmp->size)))
  {
   free(tmp);
   tmp=0;
   goto doret;
  }
  gzread(tz,tmp->data,tmp->size);
 }
 else if(type==2)
 {
  unz_file_info ufo; 
  unzGetCurrentFileInfo(tz,&ufo,0,0,0,0,0,0);  

  tmp->size=ufo.uncompressed_size;
  if(!(tmp->data=(uint8 *)FCEU_malloc(ufo.uncompressed_size)))
  {
   free(tmp);
   tmp=0;
   goto doret;
  }
  unzReadCurrentFile(tz,tmp->data,ufo.uncompressed_size);
 }

 doret:
 if(type==0)
 {
  fclose((FILE *)tz);
 }
 else if(type==1)
 {
  gzclose(tz);
 }
 else if(type==2)
 {
  unzCloseCurrentFile(tz);
  unzClose(tz);
 }
 return tmp;
}

#ifndef __GNUC__
 #define strcasecmp strcmp
#endif


FCEUFILE * FCEU_fopen(const char *path, const char *ipsfn, char *mode, char *ext)
{
 FILE *ipsfile=0;
 FCEUFILE *fceufp = 0;
 void *t;

 if(strchr(mode,'r'))
  ipsfile=FCEUD_UTF8fopen(ipsfn,"rb");

 {
  unzFile tz;
  if((tz=unzOpen(path)))  // If it's not a zip file, use regular file handlers.
			  // Assuming file type by extension usually works,
			  // but I don't like it. :)
  {
   if(unzGoToFirstFile(tz)==UNZ_OK)
   {
    for(;;)
    {
     char tempu[512];	// Longer filenames might be possible, but I don't
		 	// think people would name files that long in zip files...
     unzGetCurrentFileInfo(tz,0,tempu,512,0,0,0,0);
     tempu[511]=0;
     if(strlen(tempu)>=4)
     {
      char *za=tempu+strlen(tempu)-4;

      if(!ext)
      {
       if(!strcasecmp(za,".nes") || !strcasecmp(za,".fds") ||
          !strcasecmp(za,".nsf") || !strcasecmp(za,".unf") ||
          !strcasecmp(za,".nez"))
        break;
      }
      else if(!strcasecmp(za,ext)) 
       break;
     }
     if(strlen(tempu)>=5)
     {
      if(!strcasecmp(tempu+strlen(tempu)-5,".unif"))
       break;
     }
     if(unzGoToNextFile(tz)!=UNZ_OK)
     { 
      if(unzGoToFirstFile(tz)!=UNZ_OK) goto zpfail;
      break;     
     }
    }
    if(unzOpenCurrentFile(tz)!=UNZ_OK)
     goto zpfail;       
   }
   else
   {
    zpfail:
    free(fceufp);
    unzClose(tz);
    return 0;
   }
   if(!(fceufp=MakeMemWrap(tz,2)))
   {
    return(0);
   }
   if(ipsfile)
    ApplyIPS(ipsfile,fceufp);
   return(fceufp);
  }
 }

 if((t=FCEUD_UTF8fopen(path,"rb")))
 {
  uint32 magic;

  magic=fgetc((FILE *)t);
  magic|=fgetc((FILE *)t)<<8;
  magic|=fgetc((FILE *)t)<<16;

  if(magic!=0x088b1f)   /* Not gzip... */
   fclose((FILE *)t);
  else                  /* Probably gzip */
  {
   int fd;

   fd = dup(fileno( (FILE *)t));

   fclose((FILE *)t);

   lseek(fd, 0, SEEK_SET);

   if((t=gzdopen(fd,mode)))
   {
    fceufp = MakeMemWrap(t, 1);

    if(ipsfile)
     ApplyIPS(ipsfile, fceufp);
    return(fceufp);
   }
   close(fd);
  }
 }

 if((t=FCEUD_UTF8fopen(path,mode)))
 {
  fseek((FILE *)t,0,SEEK_SET);
 
  fceufp = MakeMemWrap(t, 0);

  if(ipsfile)
   ApplyIPS(ipsfile,fceufp);

  return(fceufp);
 }

 return 0;
}

int FCEU_fclose(FCEUFILE *fp)
{
 free(fp->data);
 free(fp);
 return 1;
}

uint64 FCEU_fread(void *ptr, size_t size, size_t nmemb, FCEUFILE *fp)
{
 uint32 total=size*nmemb;

 if(fp->location>=fp->size) return 0;

 if((fp->location+total)>fp->size)
 {
  int ak=fp->size-fp->location;
  memcpy((uint8*)ptr,fp->data+fp->location,ak);
  fp->location=fp->size;
  return(ak/size);
 }
 else
 {
  memcpy((uint8*)ptr,fp->data+fp->location,total);
  fp->location+=total;
  return nmemb;
 }
}

uint64 FCEU_fwrite(void *ptr, size_t size, size_t nmemb, FCEUFILE *fp)
{
 return(0); // TODO
}

int FCEU_fseek(FCEUFILE *fp, int64 offset, int whence)
{
  switch(whence)
  {
   case SEEK_SET:if(offset>=fp->size)
                  return(-1);
                 fp->location=offset;break;
   case SEEK_CUR:if(offset+fp->location>fp->size)
                  return (-1);
                 fp->location+=offset;
                 break;
  }    
  return 0;
}

uint64 FCEU_ftell(FCEUFILE *fp)
{
 return(fp->location);
}

void FCEU_rewind(FCEUFILE *fp)
{
 fp->location = 0;
}

int FCEU_read16le(uint16 *val, FCEUFILE *fp)
{
 uint8 t[2];

 if(fp->location+2>fp->size)
 {return 0;}
 *(uint32 *)t=*(uint32 *)(fp->data+fp->location);
 fp->location+=2;

 *val=t[0]|(t[1]<<8);
 return(1);
}

int FCEU_read32le(uint32 *Bufo, FCEUFILE *fp)
{
 uint8 *t;
 uint8 *t_buf = (uint8 *)Bufo;

 if(fp->location+4>fp->size)
 {
  return 0;
 }

 t=fp->data + fp->location;
 fp->location+=4;

 #ifndef LSB_FIRST
 t_buf[0]=t[3];
 t_buf[1]=t[2];
 t_buf[2]=t[1];
 t_buf[3]=t[0];
 #else
 t_buf[0] = t[0];
 t_buf[1] = t[1];
 t_buf[2] = t[2];
 t_buf[3] = t[3];
 #endif

 return 1;
}

int FCEU_fgetc(FCEUFILE *fp)
{
 if(fp->location<fp->size)
  return fp->data[fp->location++];
 return EOF;
}

uint8 *FCEU_fgetmem(FCEUFILE *fp)
{
 return fp->data;
}

uint64 FCEU_fgetsize(FCEUFILE *fp)
{
 return fp->size;
}

int FCEU_fisarchive(FCEUFILE *fp)
{
 return 0;
}
