//===================================================================
//
// File:  vid_epoc.c
//
// OS-Dependent Video Routines
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Alastair Bridgewater (1998-2001)
//     - http://www.dridus.com/~nyef/darcnes
//     - Initial distribution use in this port
//
//===================================================================
//
// Ported By: Steve Fischer (steve@stevesprojects.com)
//
//	(c) Copyright 2005, Steve Fischer
//	All Rights Reserved
//
//===================================================================
//
// DarcNES is Copyright 1998, 1999, 2000, 2001 by Alastair Bridgewater.
//
// Commercial use prohibited without express written permission from
// the author. Everyone else can do whatever they want, provided that
// I am credited in the documentation (and, if released, the source code).
//
// See the "readme" file for complete DarcNES details
//
//===================================================================

#include <stdio.h>
#include <stdlib.h>

#include "types.h"
#include "vid.h"
#include "ui.h"
#include "nes_epoc.h"

//===================================================================
//
// Local Function Prototypes
//
//===================================================================

void video_merge_lines( unsigned short *src, unsigned short *dest );

//===================================================================
//
// Local Data
//
//===================================================================

#define KRedMax       (0x001F)
#define KGreenMax     (0x003F)
#define KBlueMax      (0x001F)
#define KRedShift     (11)
#define KGreenShift   (5)
#define KBlueShift    (0)

// Macro to convert colors values to a combined color assuming green is correctly mapped
#define KRGB(r,g,b)    ( (((unsigned short)((b) & KBlueMax )) << KBlueShift ) \
                       + (((unsigned short)((g) & KGreenMax)) << KGreenShift) \
                       + (((unsigned short)((r) & KRedMax  )) << KRedShift  ))
// Macro to convert colors values to a combined color assuming green is 6 bits in length
#define KRGB444(r,g,b) ( (((unsigned short)((b) & KBlueMax )) ) \
                       + (((unsigned short)((g) & KGreenMax)) << 4) \
                       + (((unsigned short)((r) & KRedMax  )) << 8))

//
// Sorry but this is hard coded for the A92x/A1000 UIQ display dimensions
//
// For speed purposes, the NES engine has been altered to write directly
// to the bitmap buffer which is passed on to the gameengine.  This implies
// that the NES engine has been changed to write the 16 bit color values
// directly, not the 8 bit palette indexes.  Also, this seemed to be a
// great place to perform the required scaling (232 lines > 208 lines)
//

#define PIXELS_PER_NES_LINE    (256)
#define LINES_PER_NES_DISPLAY  (232)
#define LEAD_LINES_TO_SKIP       (8)

static unsigned short temp_line[PIXELS_PER_NES_LINE];

#define MRG  -1
#define SKP  -2

static short fit_to_screen_buffer_index[] =
{   0, MRG,   1,   2,   3,   4,   5,   6,   7,   8  //   0
,   9, MRG,  10,  11,  12,  13,  14,  15,  16,  17  //  10
,  18, MRG,  19,  20,  21,  22,  23,  24,  25,  26  //  20
,  27, MRG,  28,  29,  30,  31,  32,  33,  34,  35  //  30
,  36, MRG,  37,  38,  39,  40,  41,  42,  43,  44  //  40
,  45, MRG,  46,  47,  48,  49,  50,  51,  52,  53  //  50
,  54, MRG,  55,  56,  57,  58,  59,  60,  61,  62  //  60
,  63, MRG,  64,  65,  66,  67,  68,  69,  70,  71  //  70
,  72, MRG,  73,  74,  75,  76,  77,  78,  79,  80  //  80
,  81, MRG,  82,  83,  84,  85,  86,  87,  88,  89  //  90
,  90, MRG,  91,  92,  93,  94,  95,  96,  97,  98  // 100
,  99, MRG, 100, 101, 102, 103, 104, 105, 106, 107  // 110
, 108, MRG, 109, 110, 111, 112, 113, 114, 115, 116  // 120
, 117, MRG, 118, 119, 120, 121, 122, 123, 124, 125  // 130
, 126, MRG, 127, 128, 129, 130, 131, 132, 133, 134  // 140
, 135, MRG, 136, 137, 138, 139, 140, 141, 142, 143  // 150
, 144, MRG, 145, 146, 147, 148, 149, 150, 151, 152  // 160
, 153, MRG, 154, 155, 156, 157, 158, 159, 160, 161  // 170
, 162, MRG, 163, 164, 165, 166, 167, 168, 169, 170  // 180
, 171, MRG, 172, 173, 174, 175, 176, 177, 178, 179  // 190
, 180, MRG, 181, 182, 183, 184, 185, 186, 187, 188  // 200
, 189, MRG, 190, 191, 192, 193, 194, 195, 196, 197  // 210
, 198, MRG, 199, 200, 201, 202, 203, 204, 205, 206  // 220
, 207, MRG                                          // 230
};

static short center_buffer_index[] =
{ SKP, SKP, SKP, SKP, SKP, SKP, SKP, SKP, SKP, SKP  //   0
, SKP, SKP,   0,   1,   2,   3,   4,   5,   6,   7  //  10
,   8,   9,  10,  11,  12,  13,  14,  15,  16,  17  //  20
,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27  //  30
,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37  //  40
,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47  //  50
,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57  //  60
,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67  //  70
,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77  //  80
,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87  //  90
,  88,  89,  90,  91,  92,  93,  94,  95,  96,  97  // 100
,  98,  99, 100, 101, 102, 103, 104, 105, 106, 107  // 110
, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117  // 120
, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127  // 130
, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137  // 140
, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147  // 150
, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157  // 160
, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167  // 170
, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177  // 180
, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187  // 190
, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197  // 200
, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207  // 210
, SKP, SKP, SKP, SKP, SKP, SKP, SKP, SKP, SKP, SKP  // 220
, SKP, SKP                                          // 230
};

static short *request_line_to_buffer_index = fit_to_screen_buffer_index;

unsigned short  vid_palette[MAX_PALETTE_ENTRIES];
unsigned char   vid_palette_xlat[MAX_PALETTE_ENTRIES];

unsigned char  *vid_pre_xlat = vid_palette_xlat;

unsigned long   vid_lines[LINES_PER_NES_DISPLAY];

unsigned short *screen     = NULL;
unsigned short  screen_x   = 0;
unsigned short  screen_y   = 0;
unsigned short  screen_bpp = 0;

static int  vid_enable = 0;

char d[100];

//===================================================================
//
// Global Functions
//
//===================================================================

//===================================================================
//
// Function: video_init()
//
//===================================================================

void video_init( unsigned short *pScreen
               , unsigned short  x
               , unsigned short  y
               , unsigned short  bpp
               , unsigned short  mode
               )
{
   int i;

   deb_printf( "video_init" );

   screen     = pScreen;
   screen_x   = x;
   screen_y   = y;
   screen_bpp = bpp;

   if ( mode == VIDEO_MODE_CENTER )
   {
      request_line_to_buffer_index = center_buffer_index;
   }
   else
   {
      request_line_to_buffer_index = fit_to_screen_buffer_index;
   }

   if ( screen )
   {
      for ( i = 0; i < LINES_PER_NES_DISPLAY; i++ )
      {
         vid_lines[i] = (unsigned long) &screen[ i * x ];
      }

      vid_enable = 1;
   }
}

//===================================================================
//
// Function: video_destroy()
//
//===================================================================

void video_destroy( void )
{
   deb_printf( "video_destroy" );

   vid_enable = 0;
}

//===================================================================
//
// Function: video_display_buffer()
//
// Called when the video buffer is ready to be copied to the screen 
//
//===================================================================

void video_display_buffer( void )
{
//   ProfilerMark( 1 );

   if ( vid_enable && !SkipThisFrame )
   {
      DisplayFrame();
   }

//   ProfilerMark( 2 );
}

//===================================================================
//
// Function: video_setsize()
//
// Called to provide video buffer size info (???)
//
//===================================================================

void video_setsize(int x, int y)
{   
}

//===================================================================
//
// Function: video_setpal()
//
// Called to create & populate the color palette 
// 
//===================================================================

void video_setpal(int num_colors, int *red, int *green, int *blue)
{
   int i;

   if ( num_colors > MAX_PALETTE_ENTRIES )
   {
      deb_printf( "num_colors too large!!" );
   }

   for ( i = 0; i < MAX_PALETTE_ENTRIES; i++ ) 
   {
      if ( screen_bpp == 12 )
      {
         vid_palette[i]  = KRGB444( (red[i] >> 4), (green[i] >> 4), (blue[i] >> 4) );
      }
      else // Assume 16 bpp
      {
         vid_palette[i]  = KRGB( (red[i] >> 3), (green[i] >> 2), (blue[i] >> 3) );
      }
      vid_pre_xlat[i] = i;
   }
}

//===================================================================
//
// Function: video_get_vbp()
//
// Called to get the starting address of a video buffer line
//
//===================================================================

unsigned short *video_get_vbp(int line)
{
   unsigned short *p = temp_line;
   unsigned short *m;
   int idx;
   int last_idx;
   int adj_line = line - LEAD_LINES_TO_SKIP;

   if ( vid_enable )
   {
      if ( adj_line >= 0 && adj_line < LINES_PER_NES_DISPLAY )
      {
         if ( request_line_to_buffer_index[ adj_line ] == SKP )
         {
            // For skipped lines, just let the line be drawn in temp_line
         }
         else
         {
            // Check if we need to merge the temp line with it's previous one
            last_idx = (adj_line >= 2) ? request_line_to_buffer_index[ adj_line - 1 ] : 0;
            if ( last_idx == MRG )
            {
               m = (unsigned short*) vid_lines[ request_line_to_buffer_index[ adj_line - 2 ] ];
               video_merge_lines( temp_line, m );
            }
            
            idx = request_line_to_buffer_index[ adj_line ];
            
            if ( idx != MRG )
            {
               p = (unsigned short*) vid_lines[ idx ];
            }
            
            // For the merge (MRG) case store the line data in temp_line and merge it later
         }
      }
      else if ( line == LINES_PER_NES_DISPLAY ) // make sure we merge the last line
      {
         m = (unsigned short*) vid_lines[ request_line_to_buffer_index[ adj_line - 2 ] ];
         video_merge_lines( temp_line, m );
      }
   }

   return ( p );
}

//===================================================================
//
// Function: video_skipframe()
//
// Called to check if the current frame should be ignored
//
//===================================================================

int video_skipframe( void )
{
   return ( SkipThisFrame );
}

//===================================================================
//
// Local Functions
//
//===================================================================

//===================================================================
//
// Function: video_merge_lines()
//
//===================================================================

void video_merge_lines( unsigned short *src, unsigned short *dest )
{
   int i = PIXELS_PER_NES_LINE;

   while( i > 0 )
   {
      *dest++ |= *src++;
      i--;
   }
}

