//===================================================================
//
// File:  usermenu.cpp
//
// User menu
//
// Credits:
//
//   Steve Fischer (2004-2005)
//     - http://stevesprojects.com
//
//===================================================================

#include "user_menu.h"
#include "string.h"

#define __UM_DEBUG(x...)

//===================================================================
//
//  Local Definitions
//
//===================================================================

const TInt KMaxEntriesDisplayed = 10;
const TInt KEntrySize           = 18;
const TInt KTextVOffset         = 8;
const TInt KTextHOffset         = 6;

#define COLOR_16_BIT_R5_G6_B5
#ifdef COLOR_15_BIT_R5_G5_B5
#define KRedMax       (0x001F)
#define KGreenMax     (0x001F)
#define KBlueMax      (0x001F)
#define KRedShift     (10)
#define KGreenShift   (5)
#define KBlueShift    (0)
#define KGreenMod     (1)
#endif
#ifdef COLOR_16_BIT_R5_G6_B5
#define KRedMax       (0x001F)
#define KGreenMax     (0x003F)
#define KBlueMax      (0x001F)
#define KRedShift     (11)
#define KGreenShift   (5)
#define KBlueShift    (0)
#define KGreenMod     (0)
#endif

// Macro to convert colors values to a combined color assuming green is correctly mapped
#define KRGB(r,g,b)    ( ((b & KBlueMax ) << KBlueShift ) \
                       + ((g & KGreenMax) << KGreenShift) \
                       + ((r & KRedMax  ) << KRedShift  ))
// Macro to convert colors values to a combined color assuming green is 6 bits in length
#define KRGB565(r,g,b) ( ((b & KBlueMax ) << KBlueShift ) \
                       + (((g >> KGreenMod) & KGreenMax) << KGreenShift) \
                       + ((r & KRedMax  ) << KRedShift  ))
// Macro's used to get the specific base color value from a color
#define KRGB_r(c)     ((c >> KRedShift)   & KRedMax)
#define KRGB_g(c)     ((c >> KGreenShift) & KGreenMax)
#define KRGB_b(c)     ((c >> KBlueShift)  & KBlueMax)

const TUint16 KBackgroundColor = KRGB565( 27, 53, 26 ); // 0xDEBA;
const TUint16 KTitleColor      = KRGB565( 0,  47, 10 ); // 0x21E6;
const TUint16 KTitleTextColor  = KRGB565( 31, 63, 31 ); // White
const TUint16 KSelectedColor   = KRGB565( 0,  63, 15 ); // 1/3 lighter than KTitleColor
const TUint16 KTextColor       = KRGB565( 0,   0,  0 ); // Black

#define KInvalidMenuIndex  (0xFFFF)

const char KReturnText[] = "Return";

// Forward references for the CMenu class
class TCurrentList;
class CDynamicMenu;
class CMenuList;
class CMenuObject;
class CMenuObjectData;
class CMsgObject;

//===================================================================
//
//  CMenu Implementation Class Declaration
//
//===================================================================

class CMenuImp : public CMenu
{
public:
   CMenuImp( TMenuEntry aMenu[], TInt aNumOfEntries, MMenuObserver &aObserver );
   ~CMenuImp( void );
   void ConstructL( void );

   void Draw( TUint16* aPtr );
   void KeyEvent( TInt aKeyCode );

   TInt GetEntryState( TUint16 aEntryID, TUint16 aEntrySubID = 0 );
   void SetEntryState( TUint16 aEntryID, TUint16 aEntrySubID, TInt aEntryState );

   void DisplayMsg( char *aMsgLine1, char *aMsgLine2, TUint aTimeoutInDrawCallCounts );

   void Reload( void );
   void Reset( void );

private:
   void DeselectGroupsRadioButtons( TInt aEntryIdx );
   void SetupMenu( TInt aStartingIndex );

   enum { KInvalidState = -1 };
   void SetState( TUint16 aEntryID, TUint16 aEntrySubID = 0, TInt aEntryState = KInvalidState );

   MMenuObserver  &iObserver;
   TMenuEntry     *iMenuEntries;
   TInt            iNumOfEntries;
   CDynamicMenu   *iDynamicMenuEntries;
   TUint16         iStartingIndex;
   
   CMenuList      *iCurrList;
   CMsgObject     *iCurrMsg;
   TMenuEntry      iReturnEntry;
   TInt            iHighlightedIdx;
};

//===================================================================
//
//  CFontData Class Declaration
//
//===================================================================

class CFontData
{
public:
   enum EFontIndexes
   { EFontIndexMenuReturn
   , EFontIndexSubMenu
   , EFontIndexUnselectedButton
   , EFontIndexSelectedRadioButton
   , EFontIndexSelectedCheckboxButton
   };

   enum { EFontCharWidth = 8, EFontCharHeight = 10 };

   enum { EFontNumberOfCharacters = 128 };

   static const TUint8 iChar[EFontNumberOfCharacters][EFontCharHeight];
};

const TUint8 CFontData::iChar[EFontNumberOfCharacters][EFontCharHeight] =
{ {0x10,0x38,0x7C,0xFE,0x30,0x30,0x30,0x1F,0x00,0x00}  // Return menu symbol (0/0x00)
, {0x00,0x00,0x04,0x06,0xFF,0x06,0x04,0x00,0x00,0x00}  // Sub menu symbol
, {0x00,0xFF,0x81,0x81,0x81,0x81,0x81,0xFF,0x00,0x00}  // Empty checkbox & radio button
, {0x00,0xFF,0x81,0xBD,0xBD,0xBD,0x81,0xFF,0x00,0x00}  // Selected radio button
, {0x00,0xFF,0xC3,0xA5,0x99,0xA5,0xC3,0xFF,0x00,0x00}  // Selected checkbox
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}  // (16/0x10)
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}  // Space (32/0x20)
, {0x30,0x78,0x78,0x78,0x30,0x00,0x30,0x00,0x00,0x00}  // !
, {0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00,0x00,0x00}  // "
, {0x6C,0x6C,0xFE,0x6C,0xFE,0x6C,0x6C,0x00,0x00,0x00}  // #
, {0x30,0x7C,0xC0,0x78,0x0C,0xF8,0x30,0x00,0x00,0x00}  // $
, {0x00,0xC6,0xCC,0x18,0x30,0x66,0xC6,0x00,0x00,0x00}  // %
, {0x38,0x6C,0x38,0x76,0xDC,0xCC,0x76,0x00,0x00,0x00}  // &
, {0x60,0x60,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00}  // '
, {0x18,0x30,0x60,0x60,0x60,0x30,0x18,0x00,0x00,0x00}  // (
, {0x60,0x30,0x18,0x18,0x18,0x30,0x60,0x00,0x00,0x00}  // )
, {0x00,0x66,0x3C,0xFF,0x3C,0x66,0x00,0x00,0x00,0x00}  // *
, {0x00,0x30,0x30,0xFC,0x30,0x30,0x00,0x00,0x00,0x00}  // +
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x60,0x00,0x00}  // ,
, {0x00,0x00,0x00,0xFC,0x00,0x00,0x00,0x00,0x00,0x00}  // -
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}  // .
, {0x06,0x0C,0x18,0x30,0x60,0xC0,0x80,0x00,0x00,0x00}  // /
, {0x7C,0xC6,0xCE,0xDE,0xF6,0xE6,0x7C,0x00,0x00,0x00}  // 0 (48/0x30)
, {0x30,0x70,0x30,0x30,0x30,0x30,0xFC,0x00,0x00,0x00}  // 1
, {0x78,0xCC,0x0C,0x38,0x60,0xCC,0xFC,0x00,0x00,0x00}  // 2
, {0x78,0xCC,0x0C,0x38,0x0C,0xCC,0x78,0x00,0x00,0x00}  // 3
, {0x1C,0x3C,0x6C,0xCC,0xFE,0x0C,0x1E,0x00,0x00,0x00}  // 4
, {0xFC,0xC0,0xF8,0x0C,0x0C,0xCC,0x78,0x00,0x00,0x00}  // 5
, {0x38,0x60,0xC0,0xF8,0xCC,0xCC,0x78,0x00,0x00,0x00}  // 6
, {0xFC,0xCC,0x0C,0x18,0x30,0x30,0x30,0x00,0x00,0x00}  // 7
, {0x78,0xCC,0xCC,0x78,0xCC,0xCC,0x78,0x00,0x00,0x00}  // 8
, {0x78,0xCC,0xCC,0x7C,0x0C,0x18,0x70,0x00,0x00,0x00}  // 9
, {0x00,0x30,0x30,0x00,0x00,0x30,0x30,0x00,0x00,0x00}  // :
, {0x00,0x30,0x30,0x00,0x00,0x30,0x30,0x60,0x00,0x00}  // ;
, {0x18,0x30,0x60,0xC0,0x60,0x30,0x18,0x00,0x00,0x00}  // <
, {0x00,0x00,0xFC,0x00,0x00,0xFC,0x00,0x00,0x00,0x00}  // =
, {0x60,0x30,0x18,0x0C,0x18,0x30,0x60,0x00,0x00,0x00}  // >
, {0x78,0xCC,0x0C,0x18,0x30,0x00,0x30,0x00,0x00,0x00}  // ?
, {0x7C,0xC6,0xDE,0xDE,0xDE,0xC0,0x78,0x00,0x00,0x00}  // @ (64/0x40)
, {0x30,0x78,0xCC,0xCC,0xFC,0xCC,0xCC,0x00,0x00,0x00}  // A
, {0xFC,0x66,0x66,0x7C,0x66,0x66,0xFC,0x00,0x00,0x00}  // B
, {0x3C,0x66,0xC0,0xC0,0xC0,0x66,0x3C,0x00,0x00,0x00}  // C
, {0xF8,0x6C,0x66,0x66,0x66,0x6C,0xF8,0x00,0x00,0x00}  // D
, {0x7E,0x60,0x60,0x78,0x60,0x60,0x7E,0x00,0x00,0x00}  // E
, {0x7E,0x60,0x60,0x78,0x60,0x60,0x60,0x00,0x00,0x00}  // F
, {0x3C,0x66,0xC0,0xC0,0xCE,0x66,0x3E,0x00,0x00,0x00}  // G
, {0xCC,0xCC,0xCC,0xFC,0xCC,0xCC,0xCC,0x00,0x00,0x00}  // H
, {0x78,0x30,0x30,0x30,0x30,0x30,0x78,0x00,0x00,0x00}  // I
, {0x1E,0x0C,0x0C,0x0C,0xCC,0xCC,0x78,0x00,0x00,0x00}  // J
, {0xE6,0x66,0x6C,0x78,0x6C,0x66,0xE6,0x00,0x00,0x00}  // K
, {0x60,0x60,0x60,0x60,0x60,0x60,0x7E,0x00,0x00,0x00}  // L
, {0xC6,0xEE,0xFE,0xFE,0xD6,0xC6,0xC6,0x00,0x00,0x00}  // M
, {0xC6,0xE6,0xF6,0xDE,0xCE,0xC6,0xC6,0x00,0x00,0x00}  // N
, {0x38,0x6C,0xC6,0xC6,0xC6,0x6C,0x38,0x00,0x00,0x00}  // O
, {0xFC,0x66,0x66,0x7C,0x60,0x60,0xF0,0x00,0x00,0x00}  // P (80/0x50)
, {0x78,0xCC,0xCC,0xCC,0xDC,0x78,0x1C,0x00,0x00,0x00}  // Q
, {0xFC,0x66,0x66,0x7C,0x6C,0x66,0xE6,0x00,0x00,0x00}  // R
, {0x78,0xCC,0xE0,0x70,0x1C,0xCC,0x78,0x00,0x00,0x00}  // S
, {0xFC,0x30,0x30,0x30,0x30,0x30,0x30,0x00,0x00,0x00}  // T
, {0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0xFC,0x00,0x00,0x00}  // U
, {0xCC,0xCC,0xCC,0xCC,0xCC,0x78,0x30,0x00,0x00,0x00}  // V
, {0xC6,0xC6,0xC6,0xD6,0xFE,0xEE,0xC6,0x00,0x00,0x00}  // W
, {0xC6,0xC6,0x6C,0x38,0x38,0x6C,0xC6,0x00,0x00,0x00}  // X
, {0xCC,0xCC,0xCC,0x78,0x30,0x30,0x78,0x00,0x00,0x00}  // Y
, {0xFE,0x06,0x0C,0x18,0x30,0x60,0xFE,0x00,0x00,0x00}  // Z
, {0x78,0x60,0x60,0x60,0x60,0x60,0x78,0x00,0x00,0x00}  // [
, {0xC0,0x60,0x30,0x18,0x0C,0x06,0x02,0x00,0x00,0x00}  // "\"
, {0x78,0x18,0x18,0x18,0x18,0x18,0x78,0x00,0x00,0x00}  // ]
, {0x10,0x38,0x6C,0xC6,0x00,0x00,0x00,0x00,0x00,0x00}  // ^
, {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0x00}  // _
, {0x30,0x30,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00}  // ` (96/0x60)
, {0x00,0x00,0x78,0x0C,0x7C,0xCC,0x76,0x00,0x00,0x00}  // a
, {0xE0,0x60,0x60,0x7C,0x66,0x66,0xDC,0x00,0x00,0x00}  // b
, {0x00,0x00,0x78,0xCC,0xC0,0xCC,0x78,0x00,0x00,0x00}  // c
, {0x1C,0x0C,0x0C,0x7C,0xCC,0xCC,0x76,0x00,0x00,0x00}  // d
, {0x00,0x00,0x78,0xCC,0xFC,0xC0,0x78,0x00,0x00,0x00}  // e
, {0x38,0x6C,0x60,0xF0,0x60,0x60,0xF0,0x00,0x00,0x00}  // f
, {0x00,0x00,0x76,0xCC,0xCC,0x7C,0x0C,0xF8,0x00,0x00}  // g
, {0xE0,0x60,0x6C,0x76,0x66,0x66,0xE6,0x00,0x00,0x00}  // h
, {0x30,0x00,0x70,0x30,0x30,0x30,0x78,0x00,0x00,0x00}  // i
, {0x0C,0x00,0x0C,0x0C,0x0C,0xCC,0xCC,0x78,0x00,0x00}  // j
, {0xE0,0x60,0x66,0x6C,0x78,0x6C,0xE6,0x00,0x00,0x00}  // k
, {0x70,0x30,0x30,0x30,0x30,0x30,0x78,0x00,0x00,0x00}  // l
, {0x00,0x00,0xCC,0xFE,0xFE,0xD6,0xC6,0x00,0x00,0x00}  // m
, {0x00,0x00,0xF8,0xCC,0xCC,0xCC,0xCC,0x00,0x00,0x00}  // n
, {0x00,0x00,0x78,0xCC,0xCC,0xCC,0x78,0x00,0x00,0x00}  // o
, {0x00,0x00,0xDC,0x66,0x66,0x7C,0x60,0xF0,0x00,0x00}  // p (112/0x70)
, {0x00,0x00,0x76,0xCC,0xCC,0x7C,0x0C,0x1E,0x00,0x00}  // q
, {0x00,0x00,0xDC,0x76,0x66,0x60,0xF0,0x00,0x00,0x00}  // r
, {0x00,0x00,0x7C,0xC0,0x78,0x0C,0xF8,0x00,0x00,0x00}  // s
, {0x10,0x30,0x7C,0x30,0x30,0x34,0x18,0x00,0x00,0x00}  // t
, {0x00,0x00,0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00}  // u
, {0x00,0x00,0xCC,0xCC,0xCC,0x78,0x30,0x00,0x00,0x00}  // v
, {0x00,0x00,0xC6,0xD6,0xFE,0xFE,0x6C,0x00,0x00,0x00}  // w
, {0x00,0x00,0xC6,0x6C,0x38,0x6C,0xC6,0x00,0x00,0x00}  // x
, {0x00,0x00,0xCC,0xCC,0xCC,0x7C,0x0C,0xF8,0x00,0x00}  // y
, {0x00,0x00,0xFC,0x98,0x30,0x64,0xFC,0x00,0x00,0x00}  // z
, {0x1C,0x30,0x30,0xE0,0x30,0x30,0x1C,0x00,0x00,0x00}  // {
, {0x18,0x18,0x18,0x00,0x18,0x18,0x18,0x00,0x00,0x00}  // /
, {0xE0,0x30,0x30,0x1C,0x30,0x30,0xE0,0x00,0x00,0x00}  // }
, {0x76,0xDC,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}  // ~
, {0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x00,0x00}
} ;

//===================================================================
//
//  CMsgObject Class Declaration
//
//===================================================================

class CMsgObject
{
public:
    CMsgObject( TUint16 aWinHeight
              , TUint16 aWinWidth
              , TUint16 aObjHeight
              , TUint16 aObjWidth
              , TPoint &aObjOffset
              );
   ~CMsgObject();

   void  ClearMsg( void );
   TBool IsMsgActive( void );
   void  SetMsg( char *aMsgLine1, char *aMsgLine2, TInt aDrawCount );

   void  Draw( TUint16 *aBuf );

private:
   void TextBlit( TUint16      *aPtr       // Buffer to draw into
                , char         *aString    // Text string
                , const TPoint &aPosition  // Position in the buffer
                );

   TUint16 iWindowHeight;   // Height of the window of the entire display area
   TUint16 iWindowWidth;    // Width of the window of the entire display area
   TUint16 iObjectHeight;   // Height of the object
   TUint16 iObjectWidth;    // Width of the object
   TPoint  iObjectOffset;   // Upper-left-hand point of the object within the display area

   char   *iMsgLine1;
   char   *iMsgLine2;
   TUint   iDrawCount;

   CFontData  iFont;
};

//===================================================================
//
//  CMenuList Class Declaration
//
//===================================================================

class CMenuList
{
public:
    CMenuList( TUint16 aWinHeight, TUint16 aWinWidth, TUint8 aMaxButtons );
   ~CMenuList( void );

   void  AppendMenuObject( CMenuObjectData &aData );
   TInt  Count( void );
   void  DeselectRadioButtons( void );
   void  Draw( TUint16 *aBuf );
   CMenuObjectData* GetHighlightedObjectData( void );
   CMenuObjectData* GetObjectData( TUint16 aEntryID, TUint16 aEntrySubID = 0 );
   void  NextButton( void );
   void  PreviousButton( void );
   void  ResetList( void );

private:
   CMenuObject *iObject;

   TUint16 iWindowHeight;
   TUint16 iWindowWidth;
   TUint16 iWindowOffset;
   TUint8  iMaxButtonsInWindow;

   TUint16 iButtonHeight;
};

//===================================================================
//
//  CMenuObjectData Class Declaration
//
//===================================================================

class CMenuObjectData
{
public:
   CMenuObjectData( void ) { Reset(); }
   void Reset( void )      { iEntry = NULL; iSubID = iMenuTableIdx = 0; iFrom = 0; }

   TMenuEntry *iEntry;
   TUint16     iSubID;

   enum { EFromMenuTable = 1, EFromDynamicEntry, EFromReturnEntry };
   TUint8      iFrom;
   TUint16     iMenuTableIdx;
};

//===================================================================
//
//  CMenuObject Class Declaration
//
//===================================================================

class CMenuObject
{
public:
    CMenuObject( TUint16 aWinHeight, TUint16 aWinWidth, TUint16 aObjHeight, TUint16 aObjWidth, TPoint &aObjOffset );
   ~CMenuObject( void );

   TInt  Count( void );
   void  DeselectRadioButtons( void );
   void  Draw( TUint16 *aBuf, TPoint aWinOffset );
   TInt  GetHighlightedIndex( void );
   CMenuObjectData* GetObjectData( TInt aIdx );
   CMenuObjectData* GetObjectData( TUint16 aEntryID, TUint16 aEntrySubID );
   void  ResetObject( void );
   void  SetHighlighted( TInt aIdx );
   void  SetObjectData( CMenuObjectData &aData );

private:
   enum { ENoControl
        , EReturnControl
        , ESubMenuControl
        , EUnselectedControl
        , ESelectedCheckbox
        , ESelectedRadioButton 
        };
   void TextBlit( TUint         aType      // Control type
                , TUint16      *aPtr       // Buffer to draw into
                , char         *aString    // Text string
                , const TPoint &aPosition  // Position in the buffer
                , TUint16       aColor     // Text color
                );

   TBool   iInUse;
   TBool   iHighlighted;
   
   TUint16 iWindowHeight;   // Height of the window of the entire display area
   TUint16 iWindowWidth;    // Width of the window of the entire display area
   TUint16 iObjectHeight;   // Height of the object
   TUint16 iObjectWidth;    // Width of the object
   TPoint  iObjectOffset;   // Upper-left-hand point of the object within the display area
   TPoint  iTextOffset;     // Upper-left-hand point of text related to object rectangle

   CMenuObjectData  iData;
   CMenuObject     *iNext;

   CFontData        iFont;
};

//===================================================================
//
//  CDynamicMenu Class Declaration
//
//===================================================================
class CDynamicMenu : public CDynamicMenuEntry
{
public:
    CDynamicMenu( void );
   virtual ~CDynamicMenu( void );

   // Implementation of the CDynamicMenuEntry interface
   void AppendEntry( TUint16 aSubEntryID, TUint aFlags, char *aName );

   // Local methods
   void AppendEntry( TMenuEntry *aEntry, TUint16 aSubEntryID, TUint16 aMenuIndex );
   void ClearEntries( void );
   TInt Count( void );
   TMenuEntry* GetEntry( TInt aIndex, TUint16 &aSubEntryID );
   TMenuEntry* GetEntry( TUint16 aEntryID, TUint16 aSubEntryID );
   void SetDefaultFields( TUint16 aEntryID, TUint8 aLevel, TUint16 aMenuIndex );

private:
   TUint16     iDefaultEntryID;
   TUint8      iDefaultLevel;
   TUint16     iDefaultMenuIndex;

   TBool       iUsed;
   TMenuEntry  iEntry;
   TUint16     iSubEntryID;
   TUint16     iMenuIndex;

   CDynamicMenu *iNext;
};

//===================================================================
//
//  CMenu Class Definitions
//
//===================================================================

//
// Construction Methods
//===================================================================

CMenu* CMenu::NewL( TMenuEntry aMenu[], TInt aNumOfEntries, MMenuObserver &aObserver )
{
	CMenuImp* self = new /*(ELeave)*/ CMenuImp( aMenu, aNumOfEntries, aObserver );
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();
	return ( (CMenu*) self );
}

CMenuImp::CMenuImp( TMenuEntry aMenu[], TInt aNumOfEntries, MMenuObserver &aObserver )
: iObserver( aObserver )
{
   iStartingIndex  = 0;
   iHighlightedIdx = 0;

   iMenuEntries    = aMenu;
   iNumOfEntries   = aNumOfEntries;

   iReturnEntry.iEntryID = 0;   // The ID doesn't matter, it's never used
   iReturnEntry.iType    = EMenuReturnEntry;
   iReturnEntry.iLevel   = 0;   // The level doesn't matter, it's never used
   iReturnEntry.iFlags   = 0;
   iReturnEntry.iName    = const_cast<char*>(KReturnText);

   iCurrList = NULL;
   iCurrMsg  = NULL;
}

CMenuImp::~CMenuImp()
{
   if ( iCurrList )
   {
      delete iCurrList;
   }
}

void CMenuImp::ConstructL( void )
{
   iCurrList = new CMenuList( KDisplayHeight, KDisplayWidth, KButtonCount );

   TUint16 h = CFontData::EFontCharHeight * 6;
   if ( h > ((KDisplayHeight/4) * 3) ) h = ((KDisplayHeight/4) * 3);

   TPoint t(KDisplayWidth/8, (KDisplayHeight - h)/2);
   iCurrMsg = new CMsgObject( KDisplayHeight
                            , KDisplayWidth
                            , h
                            , (KDisplayWidth/4) * 3
                            , t
                            );
   iDynamicMenuEntries = new CDynamicMenu;

   // Init the current menu
   SetupMenu( iStartingIndex );
}

//
// Public Interface Methods
//===================================================================

void CMenuImp::Draw( TUint16 *aPtr )
{
   if ( iCurrList )
   {
      // Draw the menu list
      iCurrList->Draw( aPtr );

      // Draw the msg if one exists
      if ( iCurrMsg && iCurrMsg->IsMsgActive() )
      {
         iCurrMsg->Draw( aPtr );
      }
   }
}

void CMenuImp::KeyEvent( TInt aKeyCode )
{
   if ( !iCurrList ) return;

   switch ( aKeyCode )
   {
   case KKeyEventUp1:
   case KKeyEventUp2:
      iCurrList->PreviousButton();
      break;

   case KKeyEventDown1:
   case KKeyEventDown2:
      iCurrList->NextButton();
      break;

   case KKeyEventSelect1:
   case KKeyEventSelect2:
      {
         CMenuObjectData *p = iCurrList->GetHighlightedObjectData();
         if ( !p ) break;

         switch ( p->iEntry->iType )
         {
         case EMenuSubMenu:
            // Configure for a new menu list
            // Chk if a sub-menu exists
            if ( iMenuEntries[ p->iMenuTableIdx + 1 ].iLevel > p->iEntry->iLevel )
            {
               // If so the create the new current menu
               SetupMenu( p->iMenuTableIdx + 1 );
            }
            break;

         case EMenuReturnEntry:
            // Return to the next higher menu list
            {
               if ( iStartingIndex == 0 )
               {
                  // Exit the menu
                  iObserver.EntrySelection( EEntryIDExitMenu, 0 );
               }
               else
               {
                  TInt  idx = iStartingIndex - 1;
                  TUint menu_level = iMenuEntries[ idx ].iLevel;

                  while ( idx >= 0 )
                  {
                     if ( iMenuEntries[ idx ].iLevel < menu_level )
                     {
                        SetupMenu( idx + 1 );
                        break;
                     }
                     else if ( idx == 0 )
                     {
                        SetupMenu( idx );
                        break;
                     }

                     idx --;
                  }
               }
            }
            break;

         case EMenuStaticEntry:
         case EMenuDynamicEntry:
            if ( p->iMenuTableIdx == KInvalidMenuIndex || p->iMenuTableIdx >= iNumOfEntries )
            {
               // Ignore errors like these
               break;
            }
            else
            {
               SetState( p->iEntry->iEntryID, p->iSubID );
               iObserver.EntrySelection( p->iEntry->iEntryID, p->iSubID );
            }
            break;

         case EMenuTitle:
         default:
            // Do nothing, this shouldn't occur
            break;
         }
      }
      break;

   default:
      break;
   }
}

TInt CMenuImp::GetEntryState( TUint16 aEntryID, TUint16 aEntrySubID )
{
   TInt rtn = EEntryStateNotSelected;
   TMenuEntry *e = iMenuEntries;

   if ( !iCurrList ) return ( rtn );

   // Look for the entry in the menu list table
   for ( TInt i = 0; i < iNumOfEntries; i++, e++ )
   {
      if ( e->iEntryID == aEntryID )
      {
         // Make sure the entry is selectable
         if ( e->iFlags & (EEntryCheckBox | EEntryRadioButton) )
         {
            // Check for dynamic entries since they must be in the active list
            if ( e->iType == EMenuDynamicEntry )
            {
               CMenuObjectData *p = iCurrList->GetObjectData( aEntryID, aEntrySubID );
               if ( p )
               {
                  rtn = (p->iEntry->iFlags & EEntrySelected) ? EEntryStateSelected : EEntryStateNotSelected;
               }
            }
            else // all other entries can be found in the table
            {
               rtn = (e->iFlags & EEntrySelected) ? EEntryStateSelected : EEntryStateNotSelected;
            }
         }
         break;
      }
   }

   return ( rtn );
}

void CMenuImp::SetEntryState( TUint16 aEntryID, TUint16 aEntrySubID, TInt aEntryState )
{
   if (  ( iCurrList )
      && (  ( aEntryState == EEntryStateNotSelected )
         || ( aEntryState == EEntryStateSelected )
         )
      )
   {
      SetState( aEntryID, aEntrySubID, aEntryState );
   }
}

void CMenuImp::DisplayMsg( char *aMsgLine1, char *aMsgLine2, TUint aTimeoutInDrawCallCounts )
{
   // Delete the current msg if there is one
   if ( iCurrMsg )
   {
      if ( iCurrMsg->IsMsgActive() )
      {
         iCurrMsg->ClearMsg();
      }

      iCurrMsg->SetMsg( aMsgLine1, aMsgLine2, aTimeoutInDrawCallCounts );
   }
}

void CMenuImp::Reload( void )
{
   iHighlightedIdx = 0;

   if ( iCurrList ) iCurrList->ResetList();

   SetupMenu( iStartingIndex );
}

void CMenuImp::Reset( void )
{
   iStartingIndex  = 0;
   iHighlightedIdx = 0;

   if ( iCurrList ) iCurrList->ResetList();

   SetupMenu( iStartingIndex );
}

//
// Private Methods
//===================================================================

void CMenuImp::DeselectGroupsRadioButtons( TInt aEntryIdx )
{
   TMenuEntry *e = &iMenuEntries[ aEntryIdx ];

   // Find this menu lists starting index
   TInt idx = aEntryIdx;
   TUint menu_level = e->iLevel;
   while ( idx >= 0 && e->iLevel >= menu_level )
   {
      idx --;
      e = &iMenuEntries[ idx ];
   }
   if ( idx != 0 ) idx ++;

   // Now deal with all the entries in this menu list
   e = &iMenuEntries[ idx ];
   while (  ( idx < iNumOfEntries )
         && ( e->iLevel >= menu_level )
         )
   {
      e = &iMenuEntries[ idx ];
      if (  ( e->iLevel == menu_level )
         && ( e->iFlags & EEntryRadioButton )
         )
      {
         if ( e->iType != EMenuDynamicEntry )
         {
            e->iFlags &= ~(EEntrySelected);
         }
      }
      idx ++;
   }
}

void CMenuImp::SetupMenu( TInt aStartingIndex )
{
   TInt idx = aStartingIndex;

   __UM_DEBUG( _L("CMenuImp::SetupMenu/%d"), idx );

   if ( idx < iNumOfEntries ) 
   {
      // Clear out the previous menu data
      iDynamicMenuEntries->ClearEntries();
      iCurrList->ResetList();

      // Build the new menu from the passed in index
      CMenuObjectData obj;
      TUint menu_level = iMenuEntries[ idx ].iLevel;
      while (  ( idx < iNumOfEntries )
            && ( iMenuEntries[ idx ].iLevel >= menu_level )
            )
      {
         if ( iMenuEntries[ idx ].iLevel == menu_level )
         {
            if ( iMenuEntries[ idx ].iType != EMenuDynamicEntry )
            {
               obj.Reset();
               obj.iEntry        = &iMenuEntries[ idx ];
               obj.iFrom         = CMenuObjectData::EFromMenuTable;
               obj.iMenuTableIdx = (TUint16) idx;
               iCurrList->AppendMenuObject( obj );
            }
            else // Dynamic entry
            {
               CDynamicMenu d;

               d.SetDefaultFields( iMenuEntries[idx].iEntryID, iMenuEntries[idx].iLevel, (TUint16) idx );
               iObserver.GetDynamicEntry( iMenuEntries[idx].iEntryID, d );

               TInt cnt = d.Count();
               for ( TInt i = 0; i < cnt; i++ )
               {
                  TUint16 sub_id;
                  TMenuEntry *e = d.GetEntry( i, sub_id );

                  // Store here first so we have a static entry
                  iDynamicMenuEntries->AppendEntry( e, sub_id, (TUint16) idx );

                  // Get the static entry and add to list
                  obj.Reset();
                  obj.iEntry        = iDynamicMenuEntries->GetEntry( e->iEntryID, sub_id );
                  obj.iFrom         = CMenuObjectData::EFromDynamicEntry;
                  obj.iSubID        = sub_id;
                  obj.iMenuTableIdx = (TUint16) idx;
                  iCurrList->AppendMenuObject( obj );
               }
            }
         }
         idx ++;
      }

      // Add on the return menu entry
      obj.Reset();
      obj.iEntry        = &iReturnEntry;
      obj.iFrom         = CMenuObjectData::EFromReturnEntry;
      obj.iMenuTableIdx = KInvalidMenuIndex;
      iCurrList->AppendMenuObject( obj );

      // Setup the list parameters
      iStartingIndex    = (TUint16) aStartingIndex;
      iHighlightedIdx   = -1;

      // Init the highlighted entry
      iCurrList->NextButton();
   }
}

void CMenuImp::SetState( TUint16 aEntryID, TUint16 aEntrySubID, TInt aEntryState )
{
   TMenuEntry *e = NULL;

   // Check if the entry is in the current menu list
   CMenuObjectData *p = iCurrList->GetObjectData( aEntryID, aEntrySubID );
   if ( p )
   {
      if ( p->iEntry->iFlags & EEntryCheckBox )
      {
         e = p->iEntry;
      }
      else if ( p->iEntry->iFlags & EEntryRadioButton )
      {
         iCurrList->DeselectRadioButtons();
         e = p->iEntry;
      }
   }
   else // The entry is not in the current menu list
   {
      TInt i;
      e = iMenuEntries;

      // Look for the entry in the menu list table
      for ( i = 0; i < iNumOfEntries; i++, e++ )
      {
         // Chk if its the desired entry
         if ( e->iEntryID == aEntryID )
         {
            // Ignore dynamic entries since they must be in the active list to be selectable
            if ( e->iType == EMenuDynamicEntry )
            {
               e = NULL;
            }
            // If radio button clear out the other buttons in it's group
            else if ( e->iFlags & EEntryRadioButton )
            {
               DeselectGroupsRadioButtons( i );
            }
            else if ( e->iFlags & EEntryCheckBox )
            {
               // Nothing to do
            }
            // Ignore everything else
            else
            {
               e = NULL;
            }
            break;
         }
      }

      // Check for a not found condition
      if ( i >= iNumOfEntries )
      {
         e = NULL;
      }
   }

   if ( e )
   {
      // Handle check boxes
      if ( e->iFlags & EEntryCheckBox )
      {
         // Check for default, which means to toggle
         if ( aEntryState == KInvalidState )
         {
            e->iFlags ^= EEntrySelected;
         }
         else if ( aEntryState == EEntryStateSelected )
         {
            e->iFlags |= EEntrySelected;
         }
         else if ( aEntryState == EEntryStateNotSelected )
         {
            e->iFlags &= ~(EEntrySelected);
         }
      }
      // Handle radio buttons
      else if ( e->iFlags & EEntryRadioButton )
      {
         // The other radio buttons in the group are already deselected above

         if ( aEntryState == EEntryStateNotSelected )
         {
            e->iFlags &= ~(EEntrySelected);
         }
         else
         {
            e->iFlags |= EEntrySelected;
         }
      }
   }
}

//===================================================================
//
//  Local CMsgObject Class Definition
//
//===================================================================

CMsgObject::CMsgObject
( TUint16 aWinHeight
, TUint16 aWinWidth
, TUint16 aObjHeight
, TUint16 aObjWidth
, TPoint &aObjOffset
)
{
   iWindowHeight  = aWinHeight;
   iWindowWidth   = aWinWidth;
   iObjectHeight  = (aObjHeight > 28) ? aObjHeight : 28;  // 14 is minimum height
   iObjectWidth   = aObjWidth;
   iObjectOffset  = aObjOffset;

   iDrawCount     = 0;
   iMsgLine1      = NULL;
   iMsgLine2      = NULL;
}

CMsgObject::~CMsgObject()
{
   ClearMsg();
}

void CMsgObject::ClearMsg( void )
{
   iDrawCount = 0;
   if ( iMsgLine1 ) delete iMsgLine1;
   if ( iMsgLine2 ) delete iMsgLine2;
}

TBool CMsgObject::IsMsgActive( void )
{
   return ( (iDrawCount) ? ETrue : EFalse );
}

void CMsgObject::SetMsg( char *aMsgLine1, char *aMsgLine2, TInt aDrawCount )
{
   TInt len = 0;
   TInt max_len = ((iObjectWidth/CFontData::EFontCharWidth) - 4);

   iDrawCount = aDrawCount;

   if ( aMsgLine1 )
   {
      len = strlen( aMsgLine1 );
      if ( len > max_len ) len = max_len;

      iMsgLine1 = new char[ len + 1 ];
      strncpy( iMsgLine1, aMsgLine1, len );
      iMsgLine1[len] = 0;
   }

   if ( aMsgLine2 )
   {
      len = strlen( aMsgLine2 );
      if ( len > max_len ) len = max_len;

      iMsgLine2 = new char[ len + 1 ];
      strncpy( iMsgLine2, aMsgLine2, len );
      iMsgLine2[len] = 0;
   }
}

void CMsgObject::Draw( TUint16 *aBuf )
{
   TInt     i, j;
   TUint16 *pleft, *p;

   // Calc the offset into the window
   TInt offset_x = iObjectOffset.iX;
   TInt offset_y = iObjectOffset.iY;

   // Set the background color for title and selected entries
   TUint16 bg = KTitleColor;

   // Draw the object
   pleft = aBuf + (iWindowWidth * offset_y) + offset_x;

   TInt shaded_lines = CFontData::EFontCharHeight;
   TUint16 r = KRGB_r(bg);
   TUint16 g = KRGB_g(bg);
   TUint16 b = KRGB_b(bg);

   for ( i = 0; i < iObjectHeight; i++ )
   {
      TUint16 c = bg;
      p = pleft;

      if ( i < shaded_lines )
      {
         TInt range_r = r - KRedMax;
         TInt range_g = g - KGreenMax;
         TInt range_b = b - KBlueMax;

         c = KRGB( (((i * range_r) / shaded_lines) + KRedMax)
                 , (((i * range_g) / shaded_lines) + KGreenMax)
                 , (((i * range_b) / shaded_lines) + KBlueMax)
                 );
      }
      else if ( i >= iObjectHeight - shaded_lines )
      {
         TInt range_r = 0 - r;
         TInt range_g = 0 - g;
         TInt range_b = 0 - b;
         TInt h = i - (iObjectHeight - shaded_lines);

         c = KRGB( (((h * range_r) / shaded_lines) + r)
                 , (((h * range_g) / shaded_lines) + g)
                 , (((h * range_b) / shaded_lines) + b)
                 );
      }

      for ( j = 0; j < iObjectWidth; j++ ) *p++ = c;

      pleft += iWindowWidth;
   }

   // Draw the text
   TInt x = iObjectOffset.iX + CFontData::EFontCharWidth;
   TInt y = iObjectOffset.iY + ((iObjectHeight - ((CFontData::EFontCharHeight * 2) + 5)) / 2);
   if ( iMsgLine1 ) TextBlit( aBuf, iMsgLine1, TPoint(x, y) );
   y += CFontData::EFontCharHeight + 5;
   if ( iMsgLine2 ) TextBlit( aBuf, iMsgLine2, TPoint(x, y) );

   // Decrement the draw count
   iDrawCount--;
   if ( !iDrawCount ) ClearMsg();
}

void CMsgObject::TextBlit
( TUint16      *aPtr       // Buffer to draw into
, char         *aString    // Text string
, const TPoint &aPosition  // Position in the buffer
)
{
   TInt len = strlen( aString );
   if ( len )
   {
      TUint8   d;
      TUint16 *p;
      TUint16 *pLeft = (TUint16*) ( aPtr
                                  + (aPosition.iY * iWindowWidth)
                                  + (aPosition.iX)
                                  );
      for ( TInt h = 0; h < CFontData::EFontCharHeight; h++ )
      {
         p = pLeft;
         for ( TInt i = 0; i < len; i++ )
         {
            if ( aString[i] < CFontData::EFontNumberOfCharacters )
            {
               d = iFont.iChar[aString[i]][h];

               for ( TInt w = 0; w < CFontData::EFontCharWidth; w++ )
               {
                  if ( d & 0x80 )
                  {
                     *p = KTitleTextColor;
                  }
                  p++;
                  d <<= 1;
               } 
            }
         }	
         pLeft += iWindowWidth;
      } 
   } 
} 

//===================================================================
//
//  Local CMenuList Class Definition
//
//===================================================================

CMenuList::CMenuList( TUint16 aWinHeight, TUint16 aWinWidth, TUint8 aMaxButtons )
{
   iObject = NULL;

   iWindowHeight       = aWinHeight;
   iWindowWidth        = aWinWidth;
   iWindowOffset       = 0;
   iMaxButtonsInWindow = aMaxButtons;

   iButtonHeight = iWindowHeight / iMaxButtonsInWindow;
}

CMenuList::~CMenuList( void )
{
   if ( iObject ) delete iObject;
}

void CMenuList::AppendMenuObject( CMenuObjectData &aData )
{
   if ( !iObject )
   {
      TPoint t( 0, 0 );
      iObject = new CMenuObject( iWindowHeight, iWindowWidth, iButtonHeight, iWindowWidth, t );
   }

   iObject->SetObjectData( aData );
}

TInt CMenuList::Count( void )
{
   return ( ( iObject ) ? iObject->Count() : 0 );
}

void CMenuList::DeselectRadioButtons( void )
{
   if ( iObject ) iObject->DeselectRadioButtons();
}

void CMenuList::Draw( TUint16 *aBuf )
{
   // Reset the entire menu window to black
   Mem::FillZ( aBuf, (iWindowHeight * iWindowWidth * 2) );

   if ( iObject ) iObject->Draw( aBuf, TPoint(0,iWindowOffset) );
}

CMenuObjectData* CMenuList::GetHighlightedObjectData( void )
{
   CMenuObjectData *p = NULL;

   if ( iObject )
   {
      p = iObject->GetObjectData( iObject->GetHighlightedIndex() );
   }

   return ( p );
}

CMenuObjectData* CMenuList::GetObjectData( TUint16 aEntryID, TUint16 aEntrySubID )
{
   CMenuObjectData *p = NULL;

   if ( iObject )
   {
      p = iObject->GetObjectData( aEntryID, aEntrySubID );
   }

   return ( p );
}

void CMenuList::NextButton( void )
{
   if ( iObject )
   {
      TInt cnt = iObject->Count();
      TInt idx = iObject->GetHighlightedIndex() + 1;
      CMenuObjectData *p;

      while ( cnt-- > 0 )
      {
         p = iObject->GetObjectData( idx );

         // Chk for wrap-around case
         if ( p == NULL )
         {
            // Restart looking from the top of the menu group entries
            idx = 0; 
            p = iObject->GetObjectData( idx );
         }

         if ( p->iEntry->iType != CMenu::EMenuTitle )
         {
            iObject->SetHighlighted( idx );
            break;
         }

         idx ++;
      }

      // Adjust the window offset as needed to be sure the selected entry is visable
      TInt first_entry_idx = (iWindowOffset / iButtonHeight);
      if ( idx < first_entry_idx )
      {
         // We must have wrapped
         iWindowOffset = 0;
      }
      else if ( (idx - first_entry_idx) > (iMaxButtonsInWindow - 1) )
      {
         first_entry_idx = idx - (iMaxButtonsInWindow - 1);
         iWindowOffset = (first_entry_idx * iButtonHeight);
      }
   }
}

void CMenuList::PreviousButton( void )
{
   if ( iObject )
   {
      TInt cnt = iObject->Count();
      TInt idx = iObject->GetHighlightedIndex() - 1;
      CMenuObjectData *p;

      while ( cnt-- > 0 )
      {
         // Chk for wrap-around
         if ( idx < 0 )
         {
            // Make sure we display the title before wrapping
            if ( iWindowOffset != 0 )
            {
               iWindowOffset = 0;
               idx = 0;
               break;
            }
            else
            {
               idx = iObject->Count() - 1;
            }
         }

         p = iObject->GetObjectData( idx );

         if ( p->iEntry->iType != CMenu::EMenuTitle )
         {
            iObject->SetHighlighted( idx );
            break;
         }

         idx --;
      }

      // Adjust the window offset as needed to be sure the selected entry is visable
      TInt first_entry_idx = (iWindowOffset / iButtonHeight);
      if ( idx < first_entry_idx )
      {
         iWindowOffset = (idx * iButtonHeight);
      }
      else if ( (idx - first_entry_idx) > (iMaxButtonsInWindow - 1) )
      {
         // Must have wrapped
         first_entry_idx = Count() - (iMaxButtonsInWindow);
         iWindowOffset = (first_entry_idx * iButtonHeight);
      }
   }
}

void CMenuList::ResetList( void )
{
   iWindowOffset = 0;
   if ( iObject ) iObject->ResetObject();
}

//===================================================================
//
//  Local CMenuObject Class Definition
//
//===================================================================

CMenuObject::CMenuObject( TUint16 aWinHeight, TUint16 aWinWidth, TUint16 aObjHeight, TUint16 aObjWidth, TPoint &aObjOffset )
{
   iWindowHeight  = aWinHeight;
   iWindowWidth   = aWinWidth;
   iObjectHeight  = (aObjHeight > 14) ? aObjHeight : 14;  // 14 is minimum height
   iObjectWidth   = aObjWidth;
   iObjectOffset  = aObjOffset;

   iTextOffset.iX = 8;
   iTextOffset.iY = ((iObjectHeight - 10) / 2) + 1;

   iInUse         = EFalse;
   iHighlighted   = EFalse;

   iNext          = NULL;
}

CMenuObject::~CMenuObject( void )
{
   if ( iNext ) delete iNext;
}

TInt CMenuObject::Count( void )
{
   TInt cnt = 0;

   if ( iNext  ) cnt = iNext->Count();
   if ( iInUse ) cnt ++;

   return ( cnt );
}

void CMenuObject::DeselectRadioButtons( void )
{
   if ( iData.iEntry->iFlags & CMenu::EEntryRadioButton )
   {
      iData.iEntry->iFlags &= ~(CMenu::EEntrySelected);
   }

   if ( iNext ) iNext->DeselectRadioButtons();
}

void CMenuObject::Draw( TUint16 *aBuf, TPoint aViewOffset )
{
   if ( iInUse )
   {
      // Chk if the entire object can be drawn, we only draw entire objects
      if (  ( aViewOffset.iX <= iObjectOffset.iX )
         && ( aViewOffset.iX + iWindowWidth >= iObjectOffset.iX + iObjectWidth )
         && ( aViewOffset.iY <= iObjectOffset.iY )
         && ( aViewOffset.iY + iWindowHeight >= iObjectOffset.iY + iObjectHeight )
         )
      {
         TInt     i, j;
         TUint16 *pleft, *p;

         // Calc the offset into the window
         TInt offset_x = iObjectOffset.iX - aViewOffset.iX;
         TInt offset_y = iObjectOffset.iY - aViewOffset.iY;

         // Set the background color for title and selected entries
         TUint16 bg = KBackgroundColor;
         if      ( iData.iEntry->iType == CMenu::EMenuTitle ) bg = KTitleColor;
         else if ( iHighlighted                             ) bg = KSelectedColor;

         // Draw the object
         pleft = aBuf + (iWindowWidth * offset_y) + offset_x;

         TInt shaded_lines = ((iObjectHeight - 2)/2);
         TUint16 r = KRGB_r(bg);
         TUint16 g = KRGB_g(bg);
         TUint16 b = KRGB_b(bg);

         for ( i = 0; i < iObjectHeight; i++ )
         {
            TUint16 c = bg;
            p = pleft;

            if ( i < shaded_lines )
            {
               TInt range_r = r - KRedMax;
               TInt range_g = g - KGreenMax;
               TInt range_b = b - KBlueMax;

               c = KRGB( (((i * range_r) / shaded_lines) + KRedMax)
                       , (((i * range_g) / shaded_lines) + KGreenMax)
                       , (((i * range_b) / shaded_lines) + KBlueMax)
                       );
            }
            else if ( i >= iObjectHeight - shaded_lines )
            {
               TInt range_r = 0 - r;
               TInt range_g = 0 - g;
               TInt range_b = 0 - b;
               TInt h = i - (iObjectHeight - shaded_lines);

               c = KRGB( (((h * range_r) / shaded_lines) + r)
                       , (((h * range_g) / shaded_lines) + g)
                       , (((h * range_b) / shaded_lines) + b)
                       );
            }

            for ( j = 0; j < iObjectWidth; j++ ) *p++ = c;

            pleft += iWindowWidth;
         }

         // Draw the text
         TUint type = ENoControl;
         if ( iData.iEntry->iType == CMenu::EMenuReturnEntry )
         {
            type = EReturnControl;
         }
         else if ( iData.iEntry->iType == CMenu::EMenuSubMenu )
         {
            type = ESubMenuControl;
         }
         else if ( iData.iEntry->iFlags & (CMenu::EEntryCheckBox | CMenu::EEntryRadioButton) )
         {
            if ( iData.iEntry->iFlags & CMenu::EEntrySelected )
            {
                    if ( iData.iEntry->iFlags & CMenu::EEntryCheckBox    ) type = ESelectedCheckbox;
               else if ( iData.iEntry->iFlags & CMenu::EEntryRadioButton ) type = ESelectedRadioButton;
            }
            else
            {
               type = EUnselectedControl;
            }
         }

         TUint16 color = ( iData.iEntry->iType == CMenu::EMenuTitle ) ? KTitleTextColor : KTextColor;

         TextBlit( type
                 , aBuf
                 , iData.iEntry->iName
                 , TPoint( offset_x + iTextOffset.iX, offset_y + iTextOffset.iY )
                 , color
                 );
      }
   }
   if ( iNext ) iNext->Draw( aBuf, aViewOffset );
}

TInt CMenuObject::GetHighlightedIndex( void )
{
   TInt idx = 0;

   if ( iInUse && !iHighlighted && iNext ) idx = iNext->GetHighlightedIndex() + 1;

   return ( idx );
}

CMenuObjectData* CMenuObject::GetObjectData( TInt aIdx )
{
   CMenuObjectData *p = NULL;

   if ( aIdx > 0 )
   {
      if ( iNext ) p = iNext->GetObjectData( --aIdx );
   }
   else if ( iInUse )
   {
      p = &iData;
   }

   return ( p );
}

CMenuObjectData* CMenuObject::GetObjectData( TUint16 aEntryID, TUint16 aEntrySubID )
{
   CMenuObjectData *p = NULL;

   if (  ( iData.iEntry->iEntryID == aEntryID )
      && (  ( iData.iSubID == aEntrySubID )
         || ( iData.iSubID == 0 )
         )
      && ( iInUse )
      )
   {
      p = &iData;
   }
   else
   {
      if ( iNext ) p = iNext->GetObjectData( aEntryID, aEntrySubID );
   }

   return ( p );
}

void CMenuObject::ResetObject( void )
{
   if ( iInUse ) 
   {
      iInUse = EFalse;
      iHighlighted = EFalse;
   }
   if ( iNext )
   {
      iNext->ResetObject();
   }
}

void CMenuObject::SetHighlighted( TInt aIdx )
{
   if ( iInUse )
   {
      iHighlighted = ( aIdx == 0 ) ? ETrue : EFalse;
      aIdx --;
   }
   if ( iNext ) iNext->SetHighlighted( aIdx );
}

void CMenuObject::SetObjectData( CMenuObjectData &aData )
{
   if ( !iInUse )
   {
      iData  = aData;
      iInUse = ETrue;
   }
   else 
   {
      if ( !iNext ) 
      {
         TPoint t( iObjectOffset.iX, iObjectOffset.iY + iObjectHeight );
         iNext = new CMenuObject( iWindowHeight
                                , iWindowWidth
                                , iObjectHeight
                                , iObjectWidth
                                , t
                                );
      }
      iNext->SetObjectData( aData );
   }
}

void CMenuObject::TextBlit
( TUint         aType      // Type of object
, TUint16      *aPtr       // Buffer to draw into
, char         *aString    // Text string
, const TPoint &aPosition  // Position in the buffer
, TUint16       aColor     // Text color
)
{
   TUint8   buf[ 200 ];

   buf[0] = ' ';
        if ( aType == EReturnControl       ) buf[0] = CFontData::EFontIndexMenuReturn;
   else if ( aType == ESubMenuControl      ) buf[0] = CFontData::EFontIndexSubMenu;
   else if ( aType == EUnselectedControl   ) buf[0] = CFontData::EFontIndexUnselectedButton;
   else if ( aType == ESelectedCheckbox    ) buf[0] = CFontData::EFontIndexSelectedCheckboxButton;
   else if ( aType == ESelectedRadioButton ) buf[0] = CFontData::EFontIndexSelectedRadioButton;
   buf[1] = ' ';

   TInt max_len = (iWindowWidth - aPosition.iX - (CFontData::EFontCharWidth * 2)) / CFontData::EFontCharWidth;
   strncpy( (char*) &buf[2], aString, max_len );

   TInt len = strlen( aString ) + 2;
   if ( len > 2 )
   {
      TUint8   d;
      TUint16 *p;
      TUint16 *pLeft = (TUint16*) ( aPtr
                                  + (aPosition.iY * iWindowWidth)
                                  + (aPosition.iX)
                                  );
      for ( TInt h = 0; h < CFontData::EFontCharHeight; h++ )
      {
         p = pLeft;
         for ( TInt i = 0; i < len; i++ )
         {
            if ( buf[i] < CFontData::EFontNumberOfCharacters )
            {
               d = iFont.iChar[buf[i]][h];

               for ( TInt w = 0; w < CFontData::EFontCharWidth; w++ )
               {
                  if ( d & 0x80 )
                  {
                     *p = aColor;
                  }
                  p++;
                  d <<= 1;
               } 
            }
         }	
         pLeft += iWindowWidth;
      } 
   } 
} 

//===================================================================
//
//  CDynamicMenuEntry Class Definitions
//
//===================================================================

CDynamicMenu::CDynamicMenu( void )
{
   iEntry.iType = CMenu::EMenuStaticEntry;
   iEntry.iName = NULL;
   iUsed        = EFalse;
   iNext        = NULL;
}

CDynamicMenu::~CDynamicMenu( void )
{
   if ( iEntry.iName ) delete iEntry.iName;
   if ( iNext ) delete iNext;
}

void CDynamicMenu::AppendEntry( TUint16 aSubEntryID, TUint aFlags, char *aName )
{
   if ( !iUsed )
   {
      iEntry.iEntryID = iDefaultEntryID;
      iEntry.iLevel   = iDefaultLevel;
      iEntry.iFlags   = aFlags;

      TInt len = strlen( aName );
      if ( len > 100 ) { len = 100; }
      iEntry.iName    = new /*(ELeave)*/ char[len + 1];
      strncpy( iEntry.iName, aName, len );
      iEntry.iName[ len ] = 0;

      iSubEntryID     = aSubEntryID;
      iMenuIndex      = iDefaultMenuIndex;

      iUsed = ETrue;
   }
   else
   {
      if ( !iNext )
      {
         iNext = new CDynamicMenu();
         iNext->SetDefaultFields( iDefaultEntryID, iDefaultLevel, iDefaultMenuIndex );
      }

      iNext->AppendEntry( aSubEntryID, aFlags, aName );
   }
}

void CDynamicMenu::AppendEntry( TMenuEntry *aEntry, TUint16 aSubEntryID, TUint16 aMenuIndex )
{
   if ( !iUsed )
   {
      iEntry.iEntryID = aEntry->iEntryID;
      iEntry.iType    = aEntry->iType;
      iEntry.iLevel   = aEntry->iLevel;
      iEntry.iFlags   = aEntry->iFlags;

      TInt len = strlen( aEntry->iName );
      iEntry.iName = new /*(ELeave)*/ char[len + 1];
      strcpy( iEntry.iName, aEntry->iName );
      iEntry.iName[ len ] = 0;

      iSubEntryID     = aSubEntryID;
      iMenuIndex      = aMenuIndex;

      iUsed = ETrue;
   }
   else
   {
      if ( !iNext )
      {
         iNext = new CDynamicMenu();
      }

      iNext->AppendEntry( aEntry, aSubEntryID, aMenuIndex );
   }
}

void CDynamicMenu::ClearEntries( void )
{
   if ( iUsed )
   {
      if ( iEntry.iName )
      {
         delete iEntry.iName;
         iEntry.iName = NULL;
      }
      iUsed = EFalse;
   }
   if ( iNext ) iNext->ClearEntries();
}

TInt CDynamicMenu::Count( void )
{
   TInt cnt = 0;

   if ( iNext ) cnt = iNext->Count();
   if ( iUsed ) cnt ++;

   return ( cnt );
}

TMenuEntry* CDynamicMenu::GetEntry( TInt aIndex, TUint16 &aSubEntryID )
{
   TMenuEntry *p  = NULL;
   TUint16     id = 0;

   if ( aIndex == 0 )
   {
      if ( iUsed )
      {
         p  = &iEntry;
         id = iSubEntryID;
      }
   }
   else if ( iNext )
   {
      p = iNext->GetEntry( --aIndex, id );
   }

   aSubEntryID = id;
   return ( p );
}

TMenuEntry* CDynamicMenu::GetEntry( TUint16 aEntryID, TUint16 aSubEntryID )
{
   TMenuEntry *p = NULL;

   if ( iUsed )
   {
      if ( iEntry.iEntryID == aEntryID && iSubEntryID == aSubEntryID )
      {
         p  = &iEntry;
      }
      else if ( iNext )
      {
         p = iNext->GetEntry( aEntryID, aSubEntryID );
      }
   }

   return ( p );
}

void CDynamicMenu::SetDefaultFields( TUint16 aEntryID, TUint8 aLevel, TUint16 aMenuIndex )
{
   if ( iUsed && iNext )
   {
      SetDefaultFields( aEntryID, aLevel, aMenuIndex );
   }
   else
   {
      iDefaultEntryID   = aEntryID;
      iDefaultLevel     = aLevel;
      iDefaultMenuIndex = aMenuIndex;
   }
}

