//===================================================================
//
// File:  ui_epoc.c
//
// Interface between the NES engine and the UI/GUI
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Alastair Bridgewater (1998-2001)
//     - http://www.dridus.com/~nyef/darcnes
//     - Initial distribution use in this port
//
//===================================================================
//
// Ported By: Steve Fischer (steve@stevesprojects.com)
//
//	(c) Copyright 2005, Steve Fischer
//	All Rights Reserved
//
//===================================================================
//
// DarcNES is Copyright 1998, 1999, 2000, 2001 by Alastair Bridgewater.
//
// Commercial use prohibited without express written permission from
// the author. Everyone else can do whatever they want, provided that
// I am credited in the documentation (and, if released, the source code).
//
// See the "readme" file for complete DarcNES details
//
//===================================================================

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include "types.h"
#include "ui.h"
#include "system.h"
#include "vid.h"
#include "snd.h"
#include "tool.h"
#include "nes_epoc.h"

//===================================================================
// Local Data
//===================================================================

int skip_next_frame( unsigned int tick );

// Shutdown Callback
shutdown_t dn_shutdown;

// Timeslicer data
void (*timeslice)(void *) = NULL;
void *timeslice_data;

// Joystick data
struct joypad *ui_joypad;

// Debug string buffer
char d[200];

//===================================================================
// Local Functions
//===================================================================

//===================================================================
//
// Function: skip_next_frame()
//
//===================================================================

#define FRAMES_PER_QUARTER_SECOND  15
#define TICKS_PER_QUARTER_SECOND   16
static unsigned int last_sys_tick = 0;
int skip_next_frame( unsigned int tick )
{
   static int skip_table_index = 4;
   static int table_index      = 0;

   static u8 skip_table[][FRAMES_PER_QUARTER_SECOND] =
   { { 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 1 }  // 40 fps
   , { 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1 }  // 36 fps 
   , { 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1 }  // 32 fps 
   , { 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1 }  // 28 fps 
   , { 0, 1, 0, 1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 1 }  // 24 fps
   , { 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1, 1 }  // 20 fps
   , { 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1 }  // 16 fps
   , { 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 1 }  // 12 fps
   , { 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1 }  //  8 fps
   };

   const int table_size = ( sizeof(skip_table) / sizeof( u8[15] ) );

   if ( table_index >= FRAMES_PER_QUARTER_SECOND )
   {
      unsigned int sys_tick = SystemTick();
      unsigned int diff = sys_tick - last_sys_tick;

      if ( diff > TICKS_PER_QUARTER_SECOND )
      {
         // Slow down
         skip_table_index++;

         if ( skip_table_index >= table_size ) skip_table_index = table_size - 1;
      }
      else if ( diff < TICKS_PER_QUARTER_SECOND )
      {
         // Speed up
         skip_table_index--;

         if ( skip_table_index < 0 ) skip_table_index = 0;      
      }

      last_sys_tick = sys_tick;
      table_index   = 0;
   }

   return ( skip_table[skip_table_index][table_index++] );
}

//===================================================================
// Global Functions
//===================================================================

//===================================================================
//
// Function: sys_init()
//
//===================================================================

void sys_init( const char     *pFilename
             , unsigned short *pScreen
             , unsigned short  video_x
             , unsigned short  video_y
             , unsigned short  video_bpp
             , unsigned short  video_mode
             , unsigned short  sound_rate
             )
{
   rom_file romfile;

//   deb_printf( "sys_init" );

   video_init( pScreen, video_x, video_y, video_bpp, video_mode );

   snd_init( sound_rate );
   
   romfile = read_romimage( pFilename );
   
   if (romfile) 
   {
//      deb_printf( "sys_init run" );
      nes_run( romfile );
   }
   else
   {
      deb_printf( "error reading romfile.\n" );
   }
   
//   deb_printf( "sys_init exit" );
}

//===================================================================
//
// Function: sys_destroy()
//
//===================================================================

void sys_destroy( void )
{
   deb_printf( "sys_destroy\n" );
   video_destroy();
}

//===================================================================
//
// Function: sys_execute()
//
//===================================================================

void sys_execute( void )
{
//   deb_printf( "sys_execute" );

   last_sys_tick = SystemTick();

   while ( timeslice )
   {
//      sprintf( d, "sys_execute %d", Iterations );
//      deb_printf( d );

ProfilerMark(10);

      // Perform frame throttling as needed
      SkipThisFrame = skip_next_frame( SystemTick() );

      // Execute the frame
      timeslice( timeslice_data );

      // Count drawn frames
      if ( !SkipThisFrame ) FrameCount++;

      Iterations++;

      // Enter the menu if necessary
      if ( system_flags & F_QUIT )
      {
         // The return is non-zero if we should exit the game
         if ( ExecuteMenu() )
         {
            break;
         }
         system_flags &= ~(F_QUIT);
      }
   }

   if (dn_shutdown)
   {
      dn_shutdown();
   }

//   deb_printf( "sys_execute" );
}

//===================================================================
//
// Function: sys_get_context_length()
//
//===================================================================

int sys_get_context_length( void )
{
   return ( nes_getcontextsize() );
}

//===================================================================
//
// Function: sys_get_context()
//
//===================================================================

int sys_get_context( unsigned char *buffer, int maxlength )
{
   return ( nes_getcontext( buffer, maxlength ) );
}

//===================================================================
//
// Function: sys_set_context()
//
//===================================================================

void sys_set_context( unsigned char *buffer, int length )
{
   nes_setcontext( buffer, length );
}

//===================================================================
//
// Function: deb_printf()
//
//===================================================================

void deb_printf(const char *fmt, ...)
{
#if 1
   char s[256];
   va_list args;
   va_start(args, fmt);
   sprintf( s, fmt, args);
   va_end(args);
   DebugPrint( s );
#endif
}

//===================================================================
// Timeslicer Interface Functions
//===================================================================
//===================================================================
//
// Function: set_timeslice()
//
//===================================================================

void set_timeslice(void (*proc)(void *), void *data)
{
   timeslice = proc;
   timeslice_data = data;
}

//===================================================================
//
// Function: unset_timeslice()
//
//===================================================================

void unset_timeslice(void)
{
   timeslice = NULL;
}

//===================================================================
// Joypad Interface Functions
//===================================================================
//===================================================================
//
// Function: ui_register_joypad()
//
//===================================================================

int ui_register_joypad( struct joypad *pad )
{
   int accepted = 0;

   if (!ui_joypad)
   {
      ui_joypad = pad;
      accepted = 1;
   } 
   
   return ( accepted );
}

//===================================================================
//
// Function: ui_update_joypad()
//
//===================================================================

void ui_update_joypad( struct joypad *pad )
{
   pad->data = GetKeyData();
}

//===================================================================
// Keypad Interface Functions
//===================================================================
//===================================================================
//
// Function: keypad_register()
//
//===================================================================

int keypad_register( struct keypad *pad )
{
   // NES does not have keyboard support

   return ( 0 );
}

//===================================================================
//
// Function: keypad_update()
//
//===================================================================

void keypad_update( struct keypad *pad )
{
   // NES does not have keyboard support
}

