//===================================================================
//
// File:  nes_epoc.cpp
//
// A920 NES Startup
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Created & integrated all the below code for the A920
//
//===================================================================
//
// Created By: Steve Fischer (steve@stevesprojects.com)
//
//	(c) Copyright 2005, Steve Fischer
//	All Rights Reserved
//
//===================================================================

#include <e32std.h>
#include <e32svr.h>
#include <string.h>

#include "audio.h"
#include "video.h"
#include "controller.h"
#include "navigation.h"
#include "interface.h"
#include "polledas.h"
#include "user_menu.h"
#include "user_menu_data.h"

//#define ENABLE_PROFILER
#ifdef ENABLE_PROFILER
#include "symprofiler.h"
_LIT( KProfileFile, "d:\\nes_profile.out" );
static CSymProfiler         *sProfiler;
#endif

extern "C" 
{
#include "tool.h"
#include "ui.h"
#include "vid.h"

int FrameCount;
int AbortedFrameCount;
int Iterations;
int SkipThisFrame;
}

#define __NES_DEBUG( x )

//===================================================================
//
// Local Data
//
//===================================================================

static RFs TheFs;

static CGameWindow            *sWindow = NULL;
static CMemGameScreenDevice   *sBitmap = NULL;
static CAudioWatcher          *sAudio  = NULL;
static CPolledActiveScheduler *sPolledActiveScheduler = NULL;

#define CLIENT_WINDOW_WIDTH   (256)
#define CLIENT_WINDOW_HEIGHT  (208)

_LIT( KNesName, "ENES" );

// Storage for the game filename
static TFileName sGameFilename;

// Keypad data
struct TNesKeyData
{
   unsigned short key;
   unsigned char  mask;
};

enum { EKeyButtonA, EKeyButtonB, EKeyComboAB, EKeySelect, EKeyStart, EKeyUp, EKeyDown, EKeyLeft, EKeyRight, EKeyTermination, NKeyMax };
static TNesKeyData sKeyDataTable[NKeyMax] =
{ { DEVICE_SCANCODE_BUTTON1,  JPD_BUTTON_A                  }  // Button A
, { DEVICE_SCANCODE_BUTTON2,  JPD_BUTTON_B                  }  // Button B
, { DEVICE_SCANCODE_EXTRA,    (JPD_BUTTON_A | JPD_BUTTON_B) }  // Combo Button A+B
, { DEVICE_SCANCODE_SELECT2,  JPD_SELECT                    }  // Select
, { DEVICE_SCANCODE_START,    JPD_START                     }  // Start
, { DEVICE_SCANCODE_UP,       JPD_UP                        }  // Up
, { DEVICE_SCANCODE_DOWN,     JPD_DOWN                      }  // Down
, { DEVICE_SCANCODE_LEFT,     JPD_LEFT                      }  // Left
, { DEVICE_SCANCODE_RIGHT,    JPD_RIGHT                     }  // Right
, { 0, 0 }
};

// Turbo Key Data
#define KEY_TURBO_TIME_UNIT  (3)   // Number of polls per turbo unit of time
static TBool sTurboGameA     = EFalse;
static TInt  sGameADownCount = 0;
static TBool sTurboGameB     = EFalse;
static TInt  sGameBDownCount = 0;
static TBool sTurboSpeed     = 3 * KEY_TURBO_TIME_UNIT;

// Variables for FPS calculation
static int sStartTick;

// User Menu Locals

// The intent of the CFilenameStorage class is to provide a boundless
//  array for storing the given command line filenames to compile
class CFilenameStorage : public CBase
{
public:
    CFilenameStorage();
   ~CFilenameStorage();

   void  StoreFilename( TDesC &fn ); 
   TBool GetFilename( TInt idx, TFileName &fn );
   TInt  GetCount( void );
   void  ResetList( void );

private:
   HBufC *pfn;
   CFilenameStorage *pfs;
};

class CMenuCallback : public MMenuObserver
{
public:
   virtual ~CMenuCallback() {};
	void EntrySelection( TUint16 aEntryID, TUint16 aSubEntryID );
   void GetDynamicEntry( TUint16 aEntryID, CDynamicMenuEntry& aEntry );

private:
   void SaveGameL( void );
   void LoadGameL( TInt aFileIndex );
   void DeleteGame( TInt aFileIndex );
   void LoadGameList( CDynamicMenuEntry& aEntry );
   void ShowNote( const TDesC & aTitle, const TDesC & aNote );

   CFilenameStorage iFileList;
};

static CMenuCallback  sMenuCallback;
static CMenu         *sMenu       = NULL;
static TBool          sExitMenu   = EFalse;
static TBool          sExitGame   = EFalse;

struct TMenuKeyData
{
   unsigned short scancode;  // The key scancode to look for
   unsigned short sendcode;  // The key scancode to send to the menu key event handler
   unsigned char  state;     // The current state of the key, 1 = pressed, 0 = released
};

enum { EMenuKeyUpMot, EMenuKeyDownMot, EMenuKeySelMot, EMenuKeyUpPxxx, EMenuKeyDownPxxx, EMenuKeySelPxxx, EMenuKeyEsc, EMenuKeyMax };
static TMenuKeyData sMenuKeys[EMenuKeyMax+1] =
{ { DEVICE_SCANCODE_UP,              DEVICE_SCANCODE_UP,              0 }
, { DEVICE_SCANCODE_DOWN,            DEVICE_SCANCODE_DOWN,            0 }
, { DEVICE_SCANCODE_BUTTON1,         DEVICE_SCANCODE_BUTTON1,         0 }
, { DEVICE_SCANCODE_PXXX_JOG_UP,     DEVICE_SCANCODE_PXXX_JOG_UP,     0 }
, { DEVICE_SCANCODE_PXXX_JOG_DOWN,   DEVICE_SCANCODE_PXXX_JOG_DOWN,   0 }
, { DEVICE_SCANCODE_PXXX_JOG_INWARD, DEVICE_SCANCODE_PXXX_JOG_INWARD, 0 }
, { DEVICE_SCANCODE_ESC,             DEVICE_SCANCODE_ESC,             0 }
, { 0, 0, 0 }
};

//===================================================================
//
// Local Function Prototypes
//
//===================================================================

LOCAL_C void ConfigKeyTable( TInt aRotation, TAppToExeInterface *aInterfaceTable );
LOCAL_C unsigned short KeyCodeToScanCode( TAppToExeButton aKeyCode );
LOCAL_C void StartL();

//===================================================================
//
// Global "C" Functions
//
//===================================================================

extern "C" void AudioWrite( unsigned short *aPtr, int aLen )
{
   if ( sPolledActiveScheduler ) 
   {
      sPolledActiveScheduler->Schedule();
   }

   if ( sAudio )
   {
      if ( sAudio->IsAudioActive() )
      {
         sAudio->GameAudio()->Write( aPtr, aLen );
      }
      else
      {
         sAudio->ActivateAudio( EFalse, 33 );
      }
   }
}

// Not in header file, extern'd where needed
#define PRINT_MAX_LENGTH (80)

extern "C" void DebugPrint( char *p )
{
   TBuf8<PRINT_MAX_LENGTH> tb8;
   TBuf16<PRINT_MAX_LENGTH> tb16;

   TInt len = strlen( p );
   if ( len > PRINT_MAX_LENGTH ) len = PRINT_MAX_LENGTH;
   tb8.Append( (TUint8*) p, len );
   tb16.Copy( tb8 );

   RDebug::Print( _L("%S"), &tb16 );
}

// Not in header file, extern'd where needed
extern "C" unsigned char GetKeyData( void )
{
   int i = 0;
   unsigned char key_data = 0;

   // Check for the exit key
   if ( sWindow->GameController().IsKeyPressed( DEVICE_SCANCODE_ESC ) )
   {
      system_flags |= F_QUIT;
   }

   // Get the game keys
   while ( sKeyDataTable[i].key )
   {
      if ( sWindow->GameController().IsKeyPressed( sKeyDataTable[i].key ) )
      {
          key_data |= sKeyDataTable[i].mask;
      }

      i++;
   }

   // Handle Turbo Game Keys
   if ( sTurboGameA )
   {
      // Check key state
      if ( key_data & JPD_BUTTON_A )
      {
         sGameADownCount ++;
         if ( sGameADownCount < 0 || sGameADownCount > sTurboSpeed )
         {
            // Force up
            key_data &= ~(JPD_BUTTON_A);

            if ( sGameADownCount > sTurboSpeed )
            {
               sGameADownCount = -KEY_TURBO_TIME_UNIT;
            }
         }
      }
      else
      {
         sGameADownCount = 0;
      }
   }

   if ( sTurboGameB )
   {
      // Check key state
      if ( key_data & JPD_BUTTON_B )
      {
         sGameBDownCount ++;
         if ( sGameBDownCount < 0 || sGameBDownCount > sTurboSpeed )
         {
            // Force up
            key_data &= ~(JPD_BUTTON_B);

            if ( sGameBDownCount > sTurboSpeed )
            {
               sGameBDownCount = -KEY_TURBO_TIME_UNIT;
            }
         }
      }
      else
      {
         sGameBDownCount = 0;
      }
   }

   return ( key_data );
}

extern struct cpu6502_context *nes_cpu;

// Not in header file, extern'd where needed
extern "C" void DisplayFrame( void )
{
   sWindow->GameFrameTick( sBitmap );
}

extern "C" void ProfilerMark( int type )
{
#ifdef ENABLE_PROFILER
   sProfiler->Mark( type );
#endif
}

extern "C" void ProfilerDump( unsigned int value )
{
#ifdef ENABLE_PROFILER
   sProfiler->Dump( value );
#endif
}

extern "C" unsigned int SystemTick( void )
{
   return ( User::TickCount() );
}

extern "C" int ExecuteMenu( void )
{
   int i = 0;

   sExitMenu = EFalse;
   sExitGame = EFalse;

   // Assume the exit key is still pressed
   sMenuKeys[EMenuKeyEsc].state = 1;

   while ( !sExitMenu )
   {
      // Check the menu keys
      i = 0;
      while ( sMenuKeys[i].scancode )
      {
         if ( sWindow->GameController().IsKeyPressed( sMenuKeys[i].scancode ) )
         {
            // Only report if its newly pressed
            if ( !sMenuKeys[i].state )
            {
               // Quick exit if the exit key is pressed while in the menu
               if ( i == EMenuKeyEsc )
               {
                  sExitMenu = ETrue;
                  sExitGame = ETrue;
               }
               else
               {
                  sMenu->KeyEvent( sMenuKeys[i].sendcode );
               }
            }
            sMenuKeys[i].state = 1;
         }
         else
         {
            sMenuKeys[i].state = 0;
         }
         i++;
      }

      // Draw and display the menu
      sMenu->Draw( (TUint16*) sBitmap->ScanLineAddress(0) );
      DisplayFrame();
   }

   return ( (sExitGame) ? 1 : 0 ); 
}

//===================================================================
//
// Global Functions
//
//===================================================================

//===================================================================
//
// Function: E32Main()
//
// Program entry point
//
//===================================================================

extern TInt E32Main( void )
{
   __UHEAP_MARK;		// mark heap state

   CTrapCleanup *TheTrapCleanup = CTrapCleanup::New();

   TRAPD(error, StartL());

   delete TheTrapCleanup;

   __UHEAP_MARKEND;

   return (error);
}

//===================================================================
//
// Local Functions
//
//===================================================================

//===================================================================
//
// Function: ConfigKeyTable()
//
//===================================================================

LOCAL_C void ConfigKeyTable( TInt aRotation, TAppToExeInterface *aInterfaceTable )
{
   sKeyDataTable[EKeyStart].key   = KeyCodeToScanCode( aInterfaceTable->iOptions.iButtonStart   );
   sKeyDataTable[EKeySelect].key  = KeyCodeToScanCode( aInterfaceTable->iOptions.iButtonSelect  );
   sKeyDataTable[EKeyButtonA].key = KeyCodeToScanCode( aInterfaceTable->iOptions.iButtonGameA   );
   sKeyDataTable[EKeyButtonB].key = KeyCodeToScanCode( aInterfaceTable->iOptions.iButtonGameB   );
   sKeyDataTable[EKeyComboAB].key = KeyCodeToScanCode( aInterfaceTable->iOptions.iButtonComboAB );

   switch ( aRotation )
   {
   case EAppToExeRotationNormal:
      sKeyDataTable[EKeyUp].key    = DEVICE_SCANCODE_UP;
      sKeyDataTable[EKeyDown].key  = DEVICE_SCANCODE_DOWN;
      sKeyDataTable[EKeyLeft].key  = DEVICE_SCANCODE_LEFT;
      sKeyDataTable[EKeyRight].key = DEVICE_SCANCODE_RIGHT;
      sMenuKeys[EMenuKeyUpMot].scancode   = DEVICE_SCANCODE_UP;
      sMenuKeys[EMenuKeyDownMot].scancode = DEVICE_SCANCODE_DOWN;
      break;

   case EAppToExeRotationRight:
      sKeyDataTable[EKeyUp].key    = DEVICE_SCANCODE_LEFT;
      sKeyDataTable[EKeyDown].key  = DEVICE_SCANCODE_RIGHT;
      sKeyDataTable[EKeyLeft].key  = DEVICE_SCANCODE_DOWN;
      sKeyDataTable[EKeyRight].key = DEVICE_SCANCODE_UP;
      sMenuKeys[EMenuKeyUpMot].scancode   = DEVICE_SCANCODE_LEFT;
      sMenuKeys[EMenuKeyDownMot].scancode = DEVICE_SCANCODE_RIGHT;
      break;

   case EAppToExeRotationLeft:
   default:
      sKeyDataTable[EKeyUp].key    = DEVICE_SCANCODE_RIGHT;
      sKeyDataTable[EKeyDown].key  = DEVICE_SCANCODE_LEFT;
      sKeyDataTable[EKeyLeft].key  = DEVICE_SCANCODE_UP;
      sKeyDataTable[EKeyRight].key = DEVICE_SCANCODE_DOWN;
      sMenuKeys[EMenuKeyUpMot].scancode   = DEVICE_SCANCODE_RIGHT;
      sMenuKeys[EMenuKeyDownMot].scancode = DEVICE_SCANCODE_LEFT;
      break;
   }
}

//===================================================================
//
// Function: KeyCodeToScanCode()
//
//===================================================================

LOCAL_C unsigned short KeyCodeToScanCode( TAppToExeButton aKeyCode )
{
   const unsigned short lookup_table[][2] =
   { { EAppToExeButtonVirtualBlue,       DEVICE_SCANCODE_SELECT2         }
   , { EAppToExeButtonVirtualGreen,      DEVICE_SCANCODE_START           }
   , { EAppToExeButtonVirtualPurple,     DEVICE_SCANCODE_EXTRA           }
   , { EAppToExeButtonA92xA1000GameA,    DEVICE_SCANCODE_BUTTON1         }
   , { EAppToExeButtonA92xA1000GameB,    DEVICE_SCANCODE_BUTTON2         }
   , { EAppToExeButtonA92xA1000Select,   DEVICE_SCANCODE_SELECT          }
   , { EAppToExeButtonA92xA1000Shortcut, DEVICE_SCANCODE_SHORTCUT        }
   , { EAppToExeButtonA92xA1000Browser,  DEVICE_SCANCODE_HOME            }
   , { EAppToExeButtonA1000Speakerphone, DEVICE_SCANCODE_VOICE           }
   , { EAppToExeButtonA92xVoiceNotes,    DEVICE_SCANCODE_VOICE           }
   , { EAppToExeButtonA92xSpeakerphone,  DEVICE_SCANCODE_CAMERA          }
   , { EAppToExeButtonA1000Camera,       DEVICE_SCANCODE_CAMERA          }
   , { EAppToExeButtonPxxxBrowser,       DEVICE_SCANCODE_PXXX_BROWSER    }
   , { EAppToExeButtonPxxxCamera,        DEVICE_SCANCODE_PXXX_CAMERA     }
   , { EAppToExeButtonPxxxJogUp,         DEVICE_SCANCODE_PXXX_JOG_UP     }
   , { EAppToExeButtonPxxxJogDown,       DEVICE_SCANCODE_PXXX_JOG_DOWN   }
   , { EAppToExeButtonPxxxJogLeft,       DEVICE_SCANCODE_PXXX_JOG_LEFT   }
   , { EAppToExeButtonPxxxJogRight,      DEVICE_SCANCODE_PXXX_JOG_RIGHT  }
   , { EAppToExeButtonPxxxJogInward,     DEVICE_SCANCODE_PXXX_JOG_INWARD }
   , { 0, 0 }
   };

   unsigned short scan_code = 0;
   TInt i = 0;
   while ( lookup_table[i][0] )
   {
      if ( lookup_table[i][0] == aKeyCode )
      {
         scan_code = lookup_table[i][1];
      }
      i++;
   }

   return ( scan_code );
}

//===================================================================
//
// Function: StartL()
//
//===================================================================

LOCAL_C void StartL()
{
   __NES_DEBUG( _L("StartL 1") );
   
   // Start by renaming our thread
   RThread thread;
   thread.Rename(_L("NESEXE"));

   TheFs.Connect();

   // Get the options & filename of the rom to execute
   RChunk chunk;
   TInt err = chunk.OpenGlobal(_L("ENES_CHUNK"), ETrue );
   User::LeaveIfError( err );

   __NES_DEBUG( _L("StartL 2") );

   TAppToExeInterface *p = (TAppToExeInterface*) chunk.Base();

   DebugPrint( p->iFilename );
   TBuf8<260> name( (TUint8*) p->iFilename );
   sGameFilename.Copy( name );

   TOrientation rotation;
   switch ( p->iOptions.iRotation )
   {
   case EAppToExeRotationNormal:
      rotation = EOrientationNormal;
      break;

   case EAppToExeRotationRight:
      rotation = EOrientationRotRight;
      break;

   default:
      rotation = EOrientationRotLeft;
      break;
   }

   ConfigKeyTable( rotation, p );

   sTurboGameA = p->iOptions.iTurboGameA;
   sTurboGameB = p->iOptions.iTurboGameB;
   sTurboSpeed = p->iOptions.iTurboSpeed * KEY_TURBO_TIME_UNIT;

   TUint16 display_mode = VIDEO_MODE_FIT_TO_SCREEN;
   if ( p->iOptions.iDisplayMode == EAppToExeDisplayCenter )
   {
      display_mode = VIDEO_MODE_CENTER;
   }

   // Create the main objects
   sWindow = CGameWindow::NewL( KNesName );
   CleanupStack::PushL( sWindow );

   __NES_DEBUG( _L("StartL 3") );

   sBitmap = CMemGameScreenDevice::NewL( TSize( CLIENT_WINDOW_WIDTH, CLIENT_WINDOW_HEIGHT )
                                       , EColor64K
                                       , rotation
                                       );
   CleanupStack::PushL( sBitmap );

   sMenu = CMenu::NewL( MenuTable, EMenuIDMaxValue, sMenuCallback );
   CleanupStack::PushL( sMenu );

   if ( p->iOptions.iSound )
   {
      sPolledActiveScheduler = CPolledActiveScheduler::NewL();
      CleanupStack::PushL( sPolledActiveScheduler );

      sAudio = new CAudioWatcher();
      if ( sAudio )
      {
         CleanupStack::PushL( sAudio );
         sAudio->ActivateAudio( EFalse, 33 );
      }
   }

   if ( p->iOptions.iDisableSendEnd )
   {
      sWindow->DisableSendEndKeys();
   }

   __NES_DEBUG( _L("StartL 4") );

#ifdef ENABLE_PROFILER
   sProfiler = CSymProfiler::NewL( KProfileFile, (50 * 1024) );
   CleanupStack::PushL( sProfiler );
   sProfiler->Start();
#endif

   // Detemine how many bit-per-pixel
   TInt bpp = 16;
   switch( sWindow->GameScreenDevice()->DisplayMode() )
   {
   case EColor4K:
      bpp = 12;
      break;

   case EColor64K:
   default:
      bpp = 16;
      break;
   }

   // Config the sound
   TUint sound_rate = ( p->iOptions.iSound ) ? 8000 : 0;

   // Run the game
   sys_init( p->iFilename
           , (unsigned short *) sBitmap->ScanLineAddress(0)
           , CLIENT_WINDOW_WIDTH
           , CLIENT_WINDOW_HEIGHT
           , bpp
           , display_mode
           , sound_rate
           );

   __NES_DEBUG( _L("StartL 5") );

   FrameCount = 0;
   AbortedFrameCount = 0;
   Iterations = 0;
   sStartTick = User::TickCount();

   __NES_DEBUG( _L("StartL 6") );

   sys_execute();
   sys_destroy();

   // FPS Debug
/*   TUint endtick = User::TickCount();
   TUint secs = (endtick - sStartTick) / 64;
   TUint fps  = (secs > 0) ? (FrameCount / secs) : 0;
   RDebug::Print( _L("Iterations : %d"), Iterations );
   RDebug::Print( _L("Frames ....: %d"), FrameCount );
   RDebug::Print( _L("Aborted ...: %d"), AbortedFrameCount );
   RDebug::Print( _L("StartTick .: %d"), sStartTick );
   RDebug::Print( _L("EndTick ...: %d"), endtick );
   RDebug::Print( _L("Seconds ...: %d"), secs );
   RDebug::Print( _L("FPS .......: %d"), fps );
*/

#ifdef ENABLE_PROFILER
   sProfiler->Stop();
   sProfiler->Save();
   CleanupStack::Pop();
   delete sProfiler;
#endif

   // Clean up
   CleanupStack::Pop( 3 );
   delete sMenu;
   delete sWindow;
   delete sBitmap;

   if ( p->iOptions.iSound )
   {
      CleanupStack::Pop( 1 );
      delete sPolledActiveScheduler;

      if( sAudio )
      {
         CleanupStack::Pop( 1 );
         delete sAudio;
      }
   }

   TheFs.Close();
}

//===================================================================
//
// Function: CMenuCallback::EntrySelection()
//
//===================================================================

void CMenuCallback::EntrySelection( TUint16 aEntryID, TUint16 aSubEntryID )
{
   TInt err;

   switch ( aEntryID )
   {
   case CMenu::EEntryIDExitMenu:
      sExitMenu = ETrue;
      sExitGame = EFalse;
      break;

   case EMenuIDQuitMenu:
      sExitMenu = ETrue;
      sExitGame = ETrue;
      break;

   case EMenuIDFileSave:
      TRAP( err, SaveGameL() );
      if ( err != KErrNone )
      {
         ShowNote( _L("Save Failed"), _L("") );
      }
      break;

   case EMenuIDLoad:
      TRAP( err, LoadGameL( (aSubEntryID - 1) ) );
      if ( err != KErrNone )
      {
         ShowNote( _L("Load Failed"), _L("") );
      }
      else
      {
         // Entry the game immediately
         sExitMenu = ETrue;
         sExitGame = EFalse;
      }
      break;

   case EMenuIDDelete:
      DeleteGame( (aSubEntryID - 1) );
      sMenu->Reload();
      break;

   default:
      break;
   }
}

//===================================================================
//
// Function: CMenuCallback::GetDynamicEntry()
//
//===================================================================

void CMenuCallback::GetDynamicEntry( TUint16 aEntryID, CDynamicMenuEntry& aEntry )
{
   if (  ( aEntryID == EMenuIDLoad )
      || ( aEntryID == EMenuIDDelete )
      )
   {
      LoadGameList( aEntry );
   }
}

//===================================================================
//
// Function: CMenuCallback::SaveGameL()
//
//===================================================================

void CMenuCallback::SaveGameL()
{
   // Create the name for saving
   TInt      index = 1;
   TBool     isUniqueName = EFalse;
   TFileName name( sGameFilename );

   TInt len = name.Length();
   while (!isUniqueName)
   {
      name.AppendFormat( _L(".%03d"), index );
      TEntry entry;

      if ( TheFs.Entry(name, entry) == KErrNone )
      {
         index++;
         name.SetLength( len );
      }
      else
      {
         isUniqueName = ETrue;
      }
   }
   len = name.Length();

   // Create a buffer for the save data
   TInt size = sys_get_context_length();
   if ( size > (512 * 1024) )
   {
      ShowNote( _L("Save Data"), _L("Too Large") );
   }
   else
   {
      HBufC8 *file_buffer = HBufC8::NewL( size + 1 );
      TPtr8 file_ptr = file_buffer->Des();
      file_ptr.SetLength( size );
      TUint8 *buf = const_cast<TUint8*>(file_ptr.PtrZ());
      
      // Get the emulator state information
      TInt sav_size = sys_get_context( (unsigned char*) buf, size );
      
      // Save the data in a file
      if ( sav_size > 0 )
      {
         file_ptr.SetLength( sav_size );
         RFile file;
         TInt err = file.Create( TheFs, name, EFileWrite|EFileStream );
         if ( err == KErrNone )
         {
            file.Write( file_ptr );
            file.Close();
         }
      }
      delete file_buffer;
      
      // Display the info to the user
      TBuf < 20 > buffer;
      buffer.Format(_L("Game %03d"), index);

      ShowNote( buffer, _L("Saved") );
   }
}

//===================================================================
//
// Function: CMenuCallback::LoadGameL()
//
//===================================================================

void CMenuCallback::LoadGameL( TInt aFileIndex )
{
   if ( iFileList.GetCount() < aFileIndex )
   {
      ShowNote( _L("Invalid Index"), _L("") );
   }
   else
   {
      TFileName name;
      iFileList.GetFilename( aFileIndex, name );

      // Open the file and get it's size in bytes
      RFile file;
      TInt err = file.Open( TheFs, name, EFileRead|EFileStream );
      if ( err == KErrNone )
      {
         TInt size;
         err = file.Size( size );

         if ( err == KErrNone )
         {
            // Create a buffer for the file data
            HBufC8 *file_buffer = HBufC8::NewL( size + 1 );
            TPtr8 file_ptr = file_buffer->Des();
            
            // Read in the data
            err = file.Read( file_ptr );
            if ( err == KErrNone )
            {
               TUint8 *buf = const_cast<TUint8*>(file_ptr.PtrZ());
               
               // Restore the emulator state information
               sys_set_context( (unsigned char*) buf, size );
            }
            delete file_buffer;
         }
         file.Close();
      }
   }
}

//===================================================================
//
// Function: CMenuCallback::DeleteGame()
//
//===================================================================

void CMenuCallback::DeleteGame( TInt aFileIndex )
{
   if ( iFileList.GetCount() < aFileIndex )
   {
      ShowNote( _L("Invalid Index"), _L("") );
   }
   else
   {
      TFileName name;
      iFileList.GetFilename( aFileIndex, name );
      
      TheFs.Delete( name );
      
      TBuf<80> buffer(_L("Game "));
      TParsePtrC parse( name );
      buffer.Append( parse.Ext().Mid(1) );
      ShowNote( buffer, _L("Deleted") );      
   }
}

//===================================================================
//
// Function: CMenuCallback::LoadGameList()
//
//===================================================================

void CMenuCallback::LoadGameList( CDynamicMenuEntry& aEntry )
{
   iFileList.ResetList();
   
   // Create the directory & filename to search for
   TParsePtrC parse( sGameFilename );
   TFileName directory(parse.DriveAndPath());
   TFileName game(parse.NameAndExt());
   game.Append(_L(".???"));
   
   // Open the directory
   RDir dir;
   dir.Open(TheFs, directory, 0);
   
   // Read in the filenames
   TEntry  entry;
   TUint16 sub_id = 1;
   char    tmp_name[256];
   TInt    len, i;
   while (dir.Read(entry) == KErrNone)
   {
      if (entry.iName.Match(game) == 0)
      {
         // Store the actual filename
         TFileName fullname;
         fullname.Append( directory );
         fullname.Append( entry.iName );           
         iFileList.StoreFilename( fullname );
         
         // Mangle the name to a short display-able form
         TBuf < 80 > listItem;
         listItem.Append(entry.iName.Mid(entry.iName.Length() - 3));
         TBuf < 40 > fmt;
         _LIT(KFormatTxt, " %H%:1%T %/0%4%/1%5%/3");
         entry.iModified.FormatL(fmt, KFormatTxt);
         listItem.Append(fmt);
         
         // Store the display-able version in the dynamic display list
         len = listItem.Length();
         for ( i = 0; i < len; i++ ) tmp_name[i] = (char) listItem[i];
         tmp_name[ len ] = 0;
         aEntry.AppendEntry( sub_id++, 0, tmp_name );
      }
   }
   dir.Close();
}

//===================================================================
//
// Function: CMenuCallback::ShowNote()
//
//===================================================================

void CMenuCallback::ShowNote(const TDesC & aTitle, const TDesC & aNote)
{
   TInt len, i;
   char *line1, *line2;
   
   len = aTitle.Length();
   line1 = new char[ len + 1 ];
   for ( i = 0; i < len; i++ ) line1[i] = (char) aTitle[i];
   line1[ len ] = 0;
   
   len = aNote.Length();
   line2 = new char[ len + 1 ];
   for ( i = 0; i < len; i++ ) line2[i] = (char) aNote[i];
   line2[ len ] = 0;
   
   sMenu->DisplayMsg( line1, line2, 200 );
   
   delete line1;
   delete line2;
}

//===================================================================
//
//  CFilenameStorage Class Definition
//
//===================================================================

// The intent of the CFilenameStorage class is to provide a boundless
//  array for storing the command line given filenames to compile

CFilenameStorage::CFilenameStorage() 
{ 
   pfn = NULL;
   pfs = NULL;
}

CFilenameStorage::~CFilenameStorage() 
{ 
   if (pfn) delete pfn;
   if (pfs) delete pfs;
}

void CFilenameStorage::StoreFilename( TDesC &fn ) 
{ 
   if (!pfn)      
   {
      pfn = fn.Alloc();
   }
   else if (!pfs) 
   { 
      pfs = new CFilenameStorage;
      if ( pfs )
      {
         pfs->StoreFilename( fn );
      }
   }
   else 
   { 
      pfs->StoreFilename( fn );
   }
}

TBool CFilenameStorage::GetFilename( TInt idx, TFileName &fn ) 
{
   TBool rtn = EFalse;

   if ( idx != 0 )
   {
      if ( pfs )
      {
         rtn = pfs->GetFilename( --idx, fn );
      }
   }
   else
   {
      if (pfn) 
      {
         fn = pfn->Des();
         rtn = ETrue;
      }
   }

   return ( rtn );
}

TInt CFilenameStorage::GetCount( void )
{ 
   TInt count = 0;

   if (pfs) count = pfs->GetCount();
   if (pfn) count++;

   return (count);
}

void CFilenameStorage::ResetList( void )
{
   if (pfs) { pfs->ResetList();       }
   if (pfn) { delete pfn; pfn = NULL; }
}
