//===================================================================
//
// File:  video.h
//
// From EDoom, moved some video.cpp/h code out of here
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Peter van Sebille
//     - http://mobile.yipton.net
//     - Author of EMame for P800/P900 from which this port 
//       came from
//
//===================================================================
//
// Modifed By: Steve Fischer (steve@stevesprojects.com)
//
// (c) Copyright 2004-2005, Steve Fischer
// All Rights Reserved
//
//===================================================================
//
// Author:  Peter van Sebille (peter@yipton.net)
//
// (c) Copyright 2001, Peter van Sebille
// All Rights Reserved
//
//===================================================================

#ifndef __VIDEO_H
#define __VIDEO_H

#include "options.h"

#include <e32base.h>
#include <w32std.h>

#include "navigation.h"

//===============================================
//
// Forward References
//
//===============================================

class CGameWindow;
class CGameController;
class CInfoMgr;
class CSmartPenInputHandler;
class CSmartPenToKeyInputHandler;
class CGameDirectScreenAccess;
class CGameScreenDevice;
class CGamePalette;
class CExtraFbsBitmap;

#ifdef USE_GAME_INTRO
class CGameIntro;
#endif

#ifdef USE_WINDOW_PANER
class TGamePaner;
#endif

#ifdef USE_VIRTUAL_KEYBOARD
class CVirtualKeyboard;
#endif

#ifdef USE_VIRTUAL_BUTTONS
class CVirtualButtons;
#endif

//===============================================
//
// Defines/Enums/Etc.
//
//===============================================

template<class T>
void RemoveFromPointerArrayList(CArrayPtrFlat<T>& aList, T* aT)
{
   TInt count = aList.Count();

   for ( TInt i = count - 1; i >= 0; i-- )
   {
      if ( (aList)[i] == aT )
      {
         aList.Delete(i);
      }
   }
}

enum TOrientation
{
   EOrientationNormal,
   EOrientationRotLeft,
   EOrientationRotRight,
   EOrientationRotUpsideDown,
};

const TInt KMaxCapturedButtons = 20;
const TInt KVideoModes         = EColor4K +1;

//===============================================
//
// MGameScreenDrawObserver
//
//===============================================

class MGameScreenDrawObserver
{
public:
   /*
    * WantDrawToScreen indicates whether the observer will modify the screen
    * in PostDrawFrameToScreen. If so, then the game frame will be drawn to
    * a memory screen device. The same memory screen device will then be
    * passed in to PreDrawFrameToScreen and PostDrawFrameToScreen for 
    * the observers to modify. Afterwards the memory screen device is blitted
    * to screen.
    */
   virtual TBool WantDrawToScreen()                                                 { return EFalse; }
   virtual void  PreDrawFrameToScreen(CGameScreenDevice* /*aDstGameScreenDevice*/)  {};
   virtual void  PostDrawFrameToScreen(CGameScreenDevice* /*aDstGameScreenDevice*/) {};
};

//===============================================
//
// MScreenOrientationObserver
//
//===============================================

class MScreenOrientationObserver
{
public:
   virtual void ScreenOrientationChanged(TOrientation aOrientation) = 0;
};

//===============================================
//
// MGameScreenDrawer
//
//===============================================

class MGameScreenDrawer
{
public:
   virtual void   DrawFrame(CGameScreenDevice* aDstGameScreenDevice, const CGameScreenDevice* aSrcGameScreenDevice) = 0;
   virtual TRect  GameVisibleArea() const = 0;
   virtual TSize  GameSize() const = 0;
   virtual TPoint GameOrigin() const = 0;
   virtual void   SetGameOrigin(const TPoint& aOrigin) = 0;
};

//===============================================
//
// TGameBitmapUtil
//
//===============================================

class TGameBitmapUtil
{
public:
   static TDisplayMode GetDisplayMode(TInt aBpp, TBool aColor = ETrue);
   static TInt         GetBpp(TDisplayMode aDisplayMode);
   static TInt         GetColorDepth(TDisplayMode aDisplayMode);
   static TUint8*      PixelAddress(const CGameScreenDevice* aSrc, const TPoint& aPoint);
   static TRect        VisibleIntersection( const TSize& aScreenSize
                                          , const TPoint& aOrigin
                                          , const TSize& aBitmapSize
                                          , const TRect& aBitmapRect
                                          );
   static TUint        GetColorValue(const TRgb& aColor, const TDisplayMode& aDisplayMode);
};

//===============================================
//
// TDirectBitmapDrawer
//
//===============================================

class TDirectBitmapDrawer : public MGameScreenDrawer
{
public:
   virtual void   DrawFrame(CGameScreenDevice* aDstGameScreenDevice, const CGameScreenDevice* aSrcGameScreenDevice);
   virtual TRect  GameVisibleArea() const;
   virtual TSize  GameSize() const;
   virtual TPoint GameOrigin() const;
   virtual void   SetGameOrigin(const TPoint& aOrigin);

protected:
   TPoint iGameOrigin;
   TRect  iGameSize;
   TRect  iGameVisibleArea;
   TBool  iSetup;
};

//===============================================
//
// TScaledBitmapContext
//
//===============================================
#ifdef USE_SCALED_DRAWER

class TScaledBitmapContext
{
public:
   TScaledBitmapContext();
   TScaledBitmapContext(const CGameScreenDevice& aDstGameScreenDevice, const CGameScreenDevice& aSrcGameScreenDevice, const TSize& aBitmapSize, TBool aCenter);
   TInt   SkipX() const {return iSkipX;}
   TInt   SkipY() const {return iSkipY;}
   TPoint Point() const {return iPoint;}
   TSize  Size() const  {return iSize;}
   const CGameScreenDevice& SrcGameScreenDevice() const {return *iSrcGameScreenDevice;}
   TUint* ScanLine() const {return iScanLine;}
   TInt   ScanLineBytes() const {return iScanLineBytes;}
   TBool  HasSetup()const {return iSrcGameScreenDevice!=NULL;}

protected:
   TInt   iSkipX;
   TInt   iSkipY;
   TPoint iPoint;
   TSize  iSize;
   const CGameScreenDevice* iSrcGameScreenDevice;
   mutable TUint            iScanLine[1024];
   TInt                     iScanLineBytes;
};

#endif
//===============================================
//
// TScaledBitmapDrawer
//
//===============================================
#ifdef USE_SCALED_DRAWER

class TScaledBitmapDrawer : public MGameScreenDrawer
{
public:
   virtual void   DrawFrame(CGameScreenDevice* aDstGameScreenDevice, const CGameScreenDevice* aSrcGameScreenDevice);
   virtual TRect  GameVisibleArea() const;
   virtual TSize  GameSize() const;
   virtual TPoint GameOrigin() const;
   virtual void   SetGameOrigin(const TPoint& /*aOrigin*/){};

protected:
   TScaledBitmapContext   iScaledBitmapContext;
};

#endif
//===============================================
//
// CGameScreenDevice
//
//===============================================

class CGameScreenDevice : public CBase
{
public:
   ~CGameScreenDevice();

   // getters
   virtual TUint8* ScanLineAddress(TInt aLine) const = 0;
   TInt            BytesPerLine() const { return iBytesPerLine; }
   TSize           Size() const         { return iSize;         }
   TDisplayMode    DisplayMode() const  { return iDisplayMode;  }
   TOrientation    Orientation() const  { return iOrientation;  }

   // palette operations
   CGamePalette* Palette() const                        { return iPalette;         }
   void          SetPalette(CGamePalette* aGamePalette) { iPalette = aGamePalette; }

   // draw operations
   void Clear();
   void DirectDraw(const TPoint& aPoint, const CGameScreenDevice& aSrcDevice, const TRect& aRect);
   void AlphaBlend(const TPoint& aPoint, const CGameScreenDevice& aSrcDevice, const TRect& aRect, TInt8 aAlpha);

#ifdef USE_SCALED_DRAWER
   void ScaledDrawBitmap(const TScaledBitmapContext& aScaledBitmapContext);
#endif

protected:
   CGameScreenDevice(const TSize& aSize, TDisplayMode aDisplayMode, TOrientation aOrientation);

   void DirectDraw_8bpp_8bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize);
   void DirectDraw_12or16bpp_8bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize);
   void DirectDraw_12or16bpp_12or16bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize);
   void DirectDraw_Unsupported(const TPoint& /*aDstOrigin*/, const CGameScreenDevice& /*aSrcDevice*/, const TPoint& /*aSrcOrigin*/, const TSize& /*aSize*/){ASSERT(EFalse);}

   void AlphaBlend_12bpp_12bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize, TUint8 aAlpha);
   void AlphaBlend_16bpp_16bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize, TUint8 aAlpha);
   void AlphaBlend_Unsupported(const TPoint& /*aDstOrigin*/, const CGameScreenDevice& /*aSrcDevice*/, const TPoint& /*aSrcOrigin*/, const TSize& /*aSize*/, TUint8 /*aAlpha*/){ASSERT(EFalse);}

#ifdef USE_SCALED_DRAWER
   void ScaledScanLineDraw_Unsupported(const TScaledBitmapContext& /*aScaledBitmapContext*/, TUint* /*aSrcLine*/, TBool /*aMerge*/){ASSERT(EFalse);}
   void ScaledScanLineDraw_12or16bpp_8bpp(const TScaledBitmapContext& aScaledBitmapContext, TUint* aSrcLine, TBool aMerge);
#endif

   void DirectLeftDraw_12or16bpp_12or16bpp( const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize );
   void DirectLeftDraw_Unsupported(const TPoint& /*aDstOrigin*/, const CGameScreenDevice& /*aSrcDevice*/, const TPoint& /*aSrcOrigin*/, const TSize& /*aSize*/){ASSERT(EFalse);}

   void DirectRightDraw_12or16bpp_12or16bpp( const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize );
   void DirectRightDraw_Unsupported(const TPoint& /*aDstOrigin*/, const CGameScreenDevice& /*aSrcDevice*/, const TPoint& /*aSrcOrigin*/, const TSize& /*aSize*/){ASSERT(EFalse);}

   TInt          iBytesPerLine;
   TSize         iSize;
   TDisplayMode  iDisplayMode;
   CGamePalette *iPalette;
   TOrientation  iOrientation;

protected:
   typedef void (CGameScreenDevice::*TDirectDrawer)(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize);
   static const TDirectDrawer KDirectDrawers12or16bpp[KVideoModes];
   static const TDirectDrawer KDirectDrawers8bpp[KVideoModes];

   typedef void (CGameScreenDevice::*TAlphaBlender)(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize, TUint8 aAlpha);
   static const TAlphaBlender KAlphaBlenders12bpp[KVideoModes];
   static const TAlphaBlender KAlphaBlenders16bpp[KVideoModes];

#ifdef USE_SCALED_DRAWER
   typedef void (CGameScreenDevice::*TScaledScanLineDrawer)(const TScaledBitmapContext& aScaledBitmapContext, TUint* aSrcLine, TBool aMerge);
   static const TScaledScanLineDrawer KScaledScanLineDrawers12or16bpp[KVideoModes];
#endif

   typedef void (CGameScreenDevice::*TDirectLeftDrawer)(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize);
   static const TDirectDrawer KDirectLeftDrawers12or16bpp[KVideoModes];

   typedef void (CGameScreenDevice::*TDirectRightDrawer)(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize);
   static const TDirectDrawer KDirectRightDrawers12or16bpp[KVideoModes];
};

//===============================================
//
// CFbsGameScreenDevice
//
//===============================================

class CFbsGameScreenDevice : public CGameScreenDevice
{
public:
   ~CFbsGameScreenDevice();
   static CFbsGameScreenDevice* NewL(const TSize& aSize, TDisplayMode aDisplayMode);
   virtual TUint8*   ScanLineAddress(TInt aLine) const;

   CFbsBitmap*       FbsBitmap()       { return (CFbsBitmap*) iBitmap; }
   CFbsBitGc*        FbsBitGc()        { return iBitmapContext;        }
   CFbsBitmapDevice* FbsBitmapDevice() { return iBitmapDevice;         }

protected:
   CFbsGameScreenDevice(const TSize& aSize, TDisplayMode aDisplayMode);
   void ConstructL();

   CFbsBitGc        *iBitmapContext;
   CExtraFbsBitmap  *iBitmap;
   CFbsBitmapDevice *iBitmapDevice;

   TUint32          *iDataAddress;
   CBitwiseBitmap   *iBitwiseBitmap;
};

//===============================================
//
// CHwFbGameScreenDevice
//
//===============================================

class CHwFbGameScreenDevice : public CGameScreenDevice
{
public:
   static CHwFbGameScreenDevice* NewL(const TSize& aWindowSize, TDisplayMode aDisplayMode);

   virtual TUint8*   ScanLineAddress(TInt aLine) const;

protected:
   CHwFbGameScreenDevice(const TSize& aWindowSize, TDisplayMode aDisplayMode);
   void ConstructL(const TScreenInfoV01& aScreenInfo);
   TUint8 *iFrameBuffer;
};

//===============================================
//
// CMemGameScreenDevice
//
//===============================================

class CMemGameScreenDevice : public CGameScreenDevice
{
public:
   ~CMemGameScreenDevice();
   static CMemGameScreenDevice* NewL(const TSize& aSize, TDisplayMode aDisplayMode, TOrientation aOrientation);

   virtual TUint8* ScanLineAddress(TInt aLine) const;

protected:
   CMemGameScreenDevice(const TSize& aSize, TDisplayMode aDisplayMode, TOrientation aOrientation);
   void ConstructL();

   TUint8 *iMemBuffer;
};

//===============================================
//
// CGameWindow
//
//===============================================

class CGameWindow : public CBase
{
public:
   ~CGameWindow();
   static CGameWindow* NewL(const TDesC& aGameName);

   void GameFrameTick(CGameScreenDevice* aSrcGameScreenDevice);

   CGameScreenDevice* GameScreenDevice()       { return iGameScreenDevice; }
   CGameController&   GameController() const   { return *iGameController;  }
   RWsSession&        WsSession()              { return iWsSession;        }

#ifdef USE_VIRTUAL_KEYBOARD
   CVirtualKeyboard&  VirtualKeyboard() const  { return *iVirtualKeyboard; }
#endif

   void ClearScreen();

   // area on screen "covered" by the game; this is subject to scaling etc
   TRect  GameVisibleArea() const              { return iGameScreenDrawer->GameVisibleArea(); }
   TSize  GameSize() const                     { return iGameScreenDrawer->GameSize();        }
   TPoint GameOrigin() const                   { return iGameScreenDrawer->GameOrigin();      }
   void   SetGameOrigin(const TPoint& aOrigin) { iGameScreenDrawer->SetGameOrigin(aOrigin);   }

   void AddScreenOrientationObserverL(MScreenOrientationObserver& aObserver);
   void SetScreenOrientation(TOrientation aOrientation);

   void AddGameScreenDrawObserverL(MGameScreenDrawObserver& aObserver)   { iGameScreenDrawObservers->AppendL(&aObserver);}
   void RemoveGameScreenDrawObserver(MGameScreenDrawObserver& aObserver) { RemoveFromPointerArrayList<MGameScreenDrawObserver>(*iGameScreenDrawObservers, &aObserver);}
   void SetGameScreenDrawer(MGameScreenDrawer* aGameScreenDrawer);

   void ConstructSmartPenToKeyInputHandlerL();
   void CanDo_X_And_Y_AtTheSameTime(TBool aCanDo);

   void Hide();
   void Show();

   void DisableSendEndKeys();

   void CaptureKeyEvents();
   void CaptureKeyUpDowns();
   void ReleaseKeyEvents();
   void ReleaseKeyUpDowns();

protected:
   CGameWindow();
   void ConstructL(const TDesC& aGameName);
   void ConstructScreenDeviceL();
   void DestructScreenDevice();

   CGameScreenDevice   *iGameScreenDevice;
   MGameScreenDrawer   *iGameScreenDrawer;
   TDirectBitmapDrawer  iDirectBitmapDrawer;

#ifdef USE_SCALED_DRAWER
   TScaledBitmapDrawer  iScaledBitmapDrawer;
#endif

#ifdef USE_WINDOW_PANER
   TGamePaner          *iGamePaner;
#endif

   // Window Server stuff
   void WsBlitBitmap(CFbsGameScreenDevice* aFbsGameScreenDevice);
   void WsBlitBitmap(CFbsGameScreenDevice* aFbsGameScreenDevice, const TRect& aRect);
   void ConstructWsResourcesL();
   void DestructWsResources();
   CGameDirectScreenAccess *iGameDirectScreenAccess;
   RWsSession               iWsSession;
   RWindowGroup             iWsWindowGroup;
   RWindow                  iWsWindow;
   CWsScreenDevice         *iWsScreen;
   CWindowGc               *iWindowGc;
   TBool                    iUseWServ;

   // Game controller stuff
   void ConstructGameControllerL();
   void DestructGameController();
   CGameController         *iGameController;

   // Virtual key stuff
   void ConstructVirtualKeysL();
   void DestructVirtualKeys();
#ifdef USE_VIRTUAL_KEYBOARD
   CVirtualKeyboard        *iVirtualKeyboard;
#endif
#ifdef USE_VIRTUAL_BUTTONS
   CVirtualButtons         *iVirtualButtons;
#endif

   CInfoMgr                *iInfoMgr;
   CSmartPenToKeyInputHandler *iSmartPenToKeyInputHandler;

#ifdef USE_GAME_INTRO
   CGameIntro              *iGameIntro;
#endif

   CArrayPtrFlat<MGameScreenDrawObserver>    *iGameScreenDrawObservers;
   CArrayPtrFlat<MScreenOrientationObserver> *iScreenOrientationObservers;

   TOrientation iOrientation;

   TBuf<32>     iGameName;

   friend CGameDirectScreenAccess;

   CMemGameScreenDevice *iMemoryGameScreenDevice;
   TBool                 iUsedMemoryScreenDevice;
   TBool                 iWasPaused;

   TBool  iDisableSendEnd;
   TInt32 iCapturedButtons[KMaxCapturedButtons];
   TInt32 iCapturedUpDowns[KMaxCapturedButtons];
};

//===============================================
//
// CGamePalette
//
//===============================================

class CGamePalette : public CBase
{
public:
   ~CGamePalette();
   static CGamePalette* NewL(TDisplayMode aDisplayMode);

   // palette operations
   TInt         AllocateEntry(const TRgb& aColor);
   TUint*       PaletteBuffer() const { return (TUint*) iPalette; }
   TDisplayMode DisplayMode() const   { return iDisplayMode;      }

protected:
   CGamePalette(TDisplayMode aDisplayMode);
   void ConstructL();

   TUint        iPalette[256];
   TDisplayMode iDisplayMode;
   TInt         iPaletteLastIndexUsed;
   TInt         iPaletteMaxIndex;
};

//===============================================
//
// CExtraFbsBitmap
//
//===============================================

class CExtraFbsBitmap : public CFbsBitmap
{
public:
   void GameBitmapLockHeap()                   { CFbsBitmap::LockHeap(); }
   CBitwiseBitmap* BitmapAdressAddress() const { return iAddressPointer; }
};

#endif         /* __VIDEO_H */
