//===================================================================
//
// File: video.cpp
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Peter van Sebille
//     - http://mobile.yipton.net
//     - Author of EMame for P800/P900 from which this port 
//       came from
//
//===================================================================
//
// Modifed By: Steve Fischer (steve@stevesprojects.com)
//
// (c) Copyright 2004-2005, Steve Fischer
// All Rights Reserved
//
//===================================================================
//
// Author:  Peter van Sebille (peter@yipton.net)
//
// (c) Copyright 2001, Peter van Sebille
// All Rights Reserved
//
//===================================================================

#include "options.h"

#include <math.h>
#include <stdlib.h>
#undef NULL  // Gets rid of the redefinition compiler warning
#include <coedef.h>      // for ECoeWinPriorityNeverAtFront
#include <QuartzKeys.h>
#include <gdi.h>
#include <hal.h>
#include "video.h"
#include "navigation.h"
#include "keybrd.h"
#include "buttons.h"
#include "controller.h"


_LIT(KFontName,"Courier");
const TInt KOneSecondInMicroSeconds =   1000 * 1000;

//===================================================================
//
// Local Functions
//
//===================================================================

void my_memcpy(void* out, const void* in, size_t length)
{
   TUint16 *dst = (TUint16*) out;
   TUint16 *src = (TUint16*) in;
   
   length >>= 1;
   
   while (length--)
   {
      *dst++ = *src++;
   }
}

//===================================================================
//
// Local Class Declarations
//
//===================================================================

//===============================================
//
// CGameDirectScreenAccess
//
//===============================================

class CGameDirectScreenAccess : public CBase, public MGameScreenDrawObserver
{
public:
   ~CGameDirectScreenAccess();
   static CGameDirectScreenAccess* NewL(CGameWindow& aGameWindow);

   // Implementation of MGameScreenDrawObserver virtual methods
   virtual void PreDrawFrameToScreen( CGameScreenDevice *aDstGameScreenDevice );

protected:
   CGameDirectScreenAccess(CGameWindow& aGameWindow);
   void ConstructL();

   void ClaimScreenL();
   void PauseGameL();
   void ResumeGameL();

   RDirectScreenAccess iDirectScreenAccess;
   RSemaphore          iSemaphore;
   TRequestStatus      iRequestStatus;
   CGameWindow&        iGameWindow;
};

//===============================================
//
// CGameIntro
//
//===============================================
#ifdef USE_GAME_INTRO

class CGameIntro : public CBase, public MGameControlInputHandler, MGameScreenDrawObserver
{
public:
   ~CGameIntro();
   static CGameIntro* NewL(CGameWindow& aGameWindow, CGameController& aGameController);

   // Implementation of MGameScreenDrawObserver virtual methods
   virtual TBool WantDrawToScreen();
   virtual void  PostDrawFrameToScreen(CGameScreenDevice* aDstGameScreenDevice);

   // Implementation of MGameControlInputHandler virtual methods
   virtual void HandleKeyEvent(TInt aType, const TKeyEvent& aKeyEvent);
   virtual void HandlePointerEvent(const TPointerEvent& /*aPointerEvent*/){};

protected:
   CGameIntro(CGameWindow& aGameWindow, CGameController& aGameController);
   void ConstructL();
   void Destruct();

   void ActivateL();
   void Deactivate();

   void DrawBitmap();
   void BitmapRecolor();

   CFbsGameScreenDevice *iBitmap;
   CFont                *iFont;
   TPoint                iPosition;
   TSize                 iSize;
   CGameWindow&          iGameWindow;
   CGameController&      iGameController;
   TBool                 iIsActive;
   TRgb                  iFgColor;
   TRgb                  iBgColor;
   TInt                  iFrames;
};

#endif
//===============================================
//
// TGamePaner
//
//===============================================
#ifdef USE_WINDOW_PANER

class TGamePaner : MGameControlInputHandler, MGameScreenDrawObserver
{
public:
   void ActivateL(CGameWindow* aGameWindow, CGameController* aGameController, TOrientation aOrientation);

   // Implementation of MGameScreenDrawObserver virtual methods
   virtual void PostDrawFrameToScreen(CGameScreenDevice* /*aDstGameScreenDevice*/);

   // Implementation of MGameControlInputHandler virtual methods
   virtual void HandleKeyEvent(TInt aType, const TKeyEvent& aKeyEvent);
   virtual void HandlePointerEvent(const TPointerEvent& /*aPointerEvent*/){};

protected:
   TInt             iDeltaX;
   TInt             iDeltaY;
   TPoint           iMaxDelta;
   TPoint           iCurDelta;
   TOrientation     iOrientation;
   CGameWindow     *iGameWindow;
   CGameController *iGameController;
};

#endif
//===============================================
//
// CInfoMgr
//
//===============================================

class CInfoMgr : public CBase, public MGameControlInputHandler, MGameScreenDrawObserver
{
public:
   ~CInfoMgr();
   static CInfoMgr* NewL(CGameWindow& aGameWindow, CGameController& aGameController);

   // Implementation of MGameScreenDrawObserver virtual methods
   virtual void PostDrawFrameToScreen(CGameScreenDevice* aDstGameScreenDevice);

   // Implementation of MGameControlInputHandler virtual methods
   virtual void HandleKeyEvent(TInt aType, const TKeyEvent& aKeyEvent);
   virtual void HandlePointerEvent(const TPointerEvent& /*aPointerEvent*/){};

protected:
   CInfoMgr(CGameWindow& aGameWindow, CGameController& aGameController);
   void ConstructL();

   void ActivateL();
   void Deactivate();

   void DrawBitmap();
   void GameTick();

   TInt                  iVideoFps;
   TInt                  iPeriod;
   TInt                  iPrevTicks;
   TInt                  iFrames;

   CFbsGameScreenDevice *iBitmap;
   CFont                *iFont;
   TPoint                iPosition;
   TSize                 iSize;
   CGameWindow&          iGameWindow;
   CGameController&      iGameController;
   TBool                 iIsActive;
   TRgb                  iFgColor;
   TRgb                  iBgColor;
};

//===================================================================
//
// Global Class Definitions
//
//===================================================================

//===============================================
//
// TGameBitmapUtil
//
//===============================================

TRect TGameBitmapUtil::VisibleIntersection
( const TSize&  aScreenSize
, const TPoint& aOrigin
, const TSize&  aBitmapSize
, const TRect&  aBitmapRect
)
{
   TRect screenRect = TRect(aScreenSize);
   TRect bitmapRect = TRect(aBitmapSize);
   bitmapRect.Intersection(aBitmapRect);
   bitmapRect.Move(aOrigin);
   screenRect.Intersection(bitmapRect);
   return screenRect;
}

TUint8* TGameBitmapUtil::PixelAddress(const CGameScreenDevice* aSrc, const TPoint& aPoint)
{
   return aSrc->ScanLineAddress(aPoint.iY) + ((aPoint.iX * GetBpp(aSrc->DisplayMode())) / 8);
}

TDisplayMode TGameBitmapUtil::GetDisplayMode(TInt aBpp, TBool aColor)
{
   TDisplayMode rtn = ENone;

   switch ( aBpp )
   {
   case 1:
      rtn = EGray2;
      break;

   case 2:
      rtn = EGray4;
      break;

   case 4:
      rtn = aColor ? EColor16 : EGray16;
      break;

   case 8:
      rtn = aColor ? EColor256 : EGray256;
      break;

   case 16:
      rtn = EColor64K;
      break;

   case 12:
      rtn = EColor4K;
      break;

   case 24:
      rtn = EColor16M;
      break;

   case 32:
      rtn = ERgb;
      break;

   default:
      ASSERT(EFalse);
      break;
   }

   return ( rtn );
}

/*
 * GetBpp returns the number of bits we need per pixel in the given
 * display mode. It is different from GetColorDepth in that GetBpp can be
 * used to calculate byte offsets in a bitmap/framebuffer.
 * For example, GetBpp(EColor4K) returns 16 because each pixel requires
 * 16 bits (of which only the least significant 12 bits are used).
 * GetColorDepth(EColor4K) returns 12.
 */
TInt TGameBitmapUtil::GetBpp(TDisplayMode aDisplayMode)
{
   const TInt bpp[] = { -1, 1, 2, 4, 8, 4, 8, 16, 32, 32, 16 };
   return bpp[aDisplayMode];
}

TInt TGameBitmapUtil::GetColorDepth(TDisplayMode aDisplayMode)
{
   const TInt bpp[] = { -1, 1, 2, 4, 8, 4, 8, 16, 24, 32, 12 };
   return bpp[aDisplayMode];
}

TUint TGameBitmapUtil::GetColorValue(const TRgb& aColor, const TDisplayMode& aDisplayMode)
{
   TUint rtn = 0;

   switch (aDisplayMode)
   {
   case EColor256:
      rtn = aColor.Color256();
      break;

   case EColor4K:
      rtn = aColor.Color4K();
      break;

   case EColor16M:
      rtn = aColor.Color16M();
      break;

   case EColor64K:
      rtn = aColor.Color64K();
      break;

   case ERgb:
      rtn = aColor.Value();
      break;

   default:
      ASSERT(EFalse);
      break;
   }

   return ( rtn );
}

//===============================================
//
// TDirectBitmapDrawer
//
//===============================================

void TDirectBitmapDrawer::DrawFrame
(       CGameScreenDevice* aDstGameScreenDevice
, const CGameScreenDevice* aSrcGameScreenDevice
)
{
   if (!iSetup)
   {
      iGameSize = aSrcGameScreenDevice->Size();

      TSize rotation_size;
      switch( aSrcGameScreenDevice->Orientation() )
      {
      case EOrientationRotLeft:
      case EOrientationRotRight:
         rotation_size = TSize( aSrcGameScreenDevice->Size().iHeight, aSrcGameScreenDevice->Size().iWidth );
         break;
         
      default:
         rotation_size = aSrcGameScreenDevice->Size();
         break;
      }

      TSize screen_size = aDstGameScreenDevice->Size();
      TInt diffX = screen_size.iWidth  - rotation_size.iWidth;
      TInt diffY = screen_size.iHeight - rotation_size.iHeight;

      iGameOrigin.iX = (diffX > 0) ? diffX/2 : 0;
      iGameOrigin.iY = (diffY > 0) ? diffY/2 : 0;

      iSetup = ETrue;

      iGameVisibleArea = TGameBitmapUtil::VisibleIntersection(screen_size, iGameOrigin, rotation_size, TRect(rotation_size));
   }

   aDstGameScreenDevice->DirectDraw(iGameOrigin, *aSrcGameScreenDevice, iGameSize);
}

TRect TDirectBitmapDrawer::GameVisibleArea() const
{
   return iGameVisibleArea;
}

TSize TDirectBitmapDrawer::GameSize() const
{
   return iGameSize.Size();
}

TPoint TDirectBitmapDrawer::GameOrigin() const
{
   return iGameOrigin;
}

void TDirectBitmapDrawer::SetGameOrigin(const TPoint& aOrigin)
{
   iGameOrigin = aOrigin;
}

//===============================================
//
// TScaledBitmapContext
//
//===============================================
#ifdef USE_SCALED_DRAWER

TScaledBitmapContext::TScaledBitmapContext() : iSkipX(-1), iSkipY(-1), iSrcGameScreenDevice(NULL)
{
}

TScaledBitmapContext::TScaledBitmapContext
( const CGameScreenDevice& aDstGameScreenDevice
, const CGameScreenDevice& aSrcGameScreenDevice
, const TSize& aBitmapSize
, TBool aCenter
) : iSrcGameScreenDevice(&aSrcGameScreenDevice)
{
   float   scaleX = ((float) aDstGameScreenDevice.Size().iWidth) / ((float) aBitmapSize.iWidth);
   float   scaleY = ((float) aDstGameScreenDevice.Size().iHeight) / ((float) aBitmapSize.iHeight);
   if ((scaleX < 1) || (scaleY < 1))
   {
      float scale = (scaleX < scaleY) ? scaleX : scaleY;

      // ideal size
      iSize.iWidth = (TInt) (scale * aBitmapSize.iWidth);
      iSize.iHeight = (TInt) (scale * aBitmapSize.iHeight);

      iSkipX = (TInt) floor (((float) aBitmapSize.iWidth) / (((float) aBitmapSize.iWidth) - ((float)iSize.iWidth)));
      iSkipY = (TInt) floor (((float) aBitmapSize.iHeight) / (((float) aBitmapSize.iHeight) - ((float) iSize.iHeight)));

      // real size based on the pixels we are going to skip
      iSize.iWidth = aBitmapSize.iWidth - (aBitmapSize.iWidth / iSkipX);
      iSize.iHeight = aBitmapSize.iHeight - (aBitmapSize.iHeight / iSkipY);

      iSkipY--;   // we start counting at zero
      iSkipX--;
   }

   if (aCenter)
   {
      iPoint.iX = (aDstGameScreenDevice.Size().iWidth - iSize.iWidth) / 2;
      iPoint.iY = (aDstGameScreenDevice.Size().iHeight - iSize.iHeight) / 2;
   }

   iScanLineBytes = iSize.iWidth * (TGameBitmapUtil::GetBpp(aDstGameScreenDevice.DisplayMode())/8);
   ASSERT(iScanLineBytes < (1024 * 4));
}

#endif
//===============================================
//
// TScaledBitmapDrawer
//
//===============================================
#ifdef USE_SCALED_DRAWER

void TScaledBitmapDrawer::DrawFrame
(       CGameScreenDevice *aDstGameScreenDevice
, const CGameScreenDevice *aSrcGameScreenDevice
)
{
   if (!iScaledBitmapContext.HasSetup())
   {
      iScaledBitmapContext = TScaledBitmapContext( *aDstGameScreenDevice
                                                 , *aSrcGameScreenDevice
                                                 , aSrcGameScreenDevice->Size()
                                                 , ETrue
                                                 );
   }
   aDstGameScreenDevice->ScaledDrawBitmap(iScaledBitmapContext);
}

TRect TScaledBitmapDrawer::GameVisibleArea() const
{
   return TRect(iScaledBitmapContext.Point(), iScaledBitmapContext.Size());
}

TSize TScaledBitmapDrawer::GameSize() const
{
   return iScaledBitmapContext.Size();
}

TPoint TScaledBitmapDrawer::GameOrigin() const
{
   return iScaledBitmapContext.Point();
}

#endif
//===============================================
//
// CGameScreenDevice
//
//===============================================

const CGameScreenDevice::TDirectDrawer CGameScreenDevice::KDirectDrawers12or16bpp[KVideoModes] = 
{
   &CGameScreenDevice::DirectDraw_Unsupported,         /* ENone */
   &CGameScreenDevice::DirectDraw_Unsupported,         /* EGray2 */
   &CGameScreenDevice::DirectDraw_Unsupported,         /* EGray4 */
   &CGameScreenDevice::DirectDraw_Unsupported,         /* EGray16 */
   &CGameScreenDevice::DirectDraw_12or16bpp_8bpp,      /* EGray256 */
   &CGameScreenDevice::DirectDraw_Unsupported,         /* EColor16 */
   &CGameScreenDevice::DirectDraw_12or16bpp_8bpp,      /* EColor256 */
   &CGameScreenDevice::DirectDraw_12or16bpp_12or16bpp, /* EColor64K */
   &CGameScreenDevice::DirectDraw_Unsupported,         /* EColor16M */
   &CGameScreenDevice::DirectDraw_Unsupported,         /* ERgb */
   &CGameScreenDevice::DirectDraw_12or16bpp_12or16bpp  /* EColor4K */
};

const CGameScreenDevice::TDirectDrawer CGameScreenDevice::KDirectDrawers8bpp[KVideoModes] = 
{
   &CGameScreenDevice::DirectDraw_Unsupported,   /* ENone */
   &CGameScreenDevice::DirectDraw_Unsupported,   /* EGray2 */
   &CGameScreenDevice::DirectDraw_Unsupported,   /* EGray4 */
   &CGameScreenDevice::DirectDraw_Unsupported,   /* EGray16 */
   &CGameScreenDevice::DirectDraw_8bpp_8bpp,     /* EGray256 */
   &CGameScreenDevice::DirectDraw_Unsupported,   /* EColor16 */
   &CGameScreenDevice::DirectDraw_8bpp_8bpp,     /* EColor256 */
   &CGameScreenDevice::DirectDraw_Unsupported,   /* EColor64K */
   &CGameScreenDevice::DirectDraw_Unsupported,   /* EColor16M */
   &CGameScreenDevice::DirectDraw_Unsupported,   /* ERgb */
   &CGameScreenDevice::DirectDraw_Unsupported    /* EColor4K */
};

const CGameScreenDevice::TAlphaBlender CGameScreenDevice::KAlphaBlenders12bpp[KVideoModes] = 
{
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* ENone */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray2 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray4 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray16 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray256 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EColor16 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EColor256 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EColor64K */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EColor16M */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* ERgb */
   &CGameScreenDevice::AlphaBlend_12bpp_12bpp    /* EColor4K */
};

const CGameScreenDevice::TAlphaBlender CGameScreenDevice::KAlphaBlenders16bpp[KVideoModes] = 
{
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* ENone */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray2 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray4 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray16 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EGray256 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EColor16 */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EColor256 */
   &CGameScreenDevice::AlphaBlend_16bpp_16bpp,   /* EColor64K */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* EColor16M */
   &CGameScreenDevice::AlphaBlend_Unsupported,   /* ERgb */
   &CGameScreenDevice::AlphaBlend_Unsupported    /* EColor4K */
};

#ifdef USE_SCALED_DRAWER
const CGameScreenDevice::TScaledScanLineDrawer CGameScreenDevice::KScaledScanLineDrawers12or16bpp[KVideoModes] = 
{
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* ENone */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* EGray2 */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* EGray4 */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* EGray16 */
   &CGameScreenDevice::ScaledScanLineDraw_12or16bpp_8bpp,   /* EGray256 */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* EColor16 */
   &CGameScreenDevice::ScaledScanLineDraw_12or16bpp_8bpp,   /* EColor256 */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* EColor64K */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* EColor16M */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported,      /* ERgb */
   &CGameScreenDevice::ScaledScanLineDraw_Unsupported       /* EColor4K */
};
#endif

const CGameScreenDevice::TDirectLeftDrawer CGameScreenDevice::KDirectLeftDrawers12or16bpp[KVideoModes] = 
{
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* ENone */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* EGray2 */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* EGray4 */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* EGray16 */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* EGray256 */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* EColor16 */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* EColor256 */
   &CGameScreenDevice::DirectLeftDraw_12or16bpp_12or16bpp, /* EColor64K */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* EColor16M */
   &CGameScreenDevice::DirectLeftDraw_Unsupported,         /* ERgb */
   &CGameScreenDevice::DirectLeftDraw_12or16bpp_12or16bpp  /* EColor4K */
};

const CGameScreenDevice::TDirectRightDrawer CGameScreenDevice::KDirectRightDrawers12or16bpp[KVideoModes] = 
{
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* ENone */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* EGray2 */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* EGray4 */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* EGray16 */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* EGray256 */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* EColor16 */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* EColor256 */
   &CGameScreenDevice::DirectRightDraw_12or16bpp_12or16bpp,/* EColor64K */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* EColor16M */
   &CGameScreenDevice::DirectRightDraw_Unsupported,        /* ERgb */
   &CGameScreenDevice::DirectRightDraw_12or16bpp_12or16bpp /* EColor4K */
};


CGameScreenDevice::CGameScreenDevice
( const TSize& aSize
, TDisplayMode aDisplayMode
, TOrientation aOrientation
) : iSize(aSize)
  , iDisplayMode(aDisplayMode)
  , iOrientation(aOrientation)
{
}

CGameScreenDevice::~CGameScreenDevice()
{
}

void CGameScreenDevice::Clear()
{
   TInt height = iSize.iHeight;
   for ( TInt h = 0; h < height; h++ )
   {
      TUint8 *line = ScanLineAddress(h);
      Mem::FillZ(line, BytesPerLine());
   }
}

void CGameScreenDevice::DirectDraw(const TPoint& aPoint, const CGameScreenDevice& aSrcDevice, const TRect& aRect)
{
   TOrientation src_orient = aSrcDevice.Orientation();
   TSize rotation_size;
   switch( src_orient )
   {
   case EOrientationRotLeft:
   case EOrientationRotRight:
      rotation_size = TSize( aSrcDevice.Size().iHeight, aSrcDevice.Size().iWidth );
      break;
         
   default:
      rotation_size = aSrcDevice.Size();
      break;
   }

   TRect   rect = TGameBitmapUtil::VisibleIntersection(Size(), aPoint, rotation_size, aRect);
   TPoint  dstPoint = aRect.iTl;

   if (aPoint.iX < 0)
   {
      dstPoint.iX -= aPoint.iX;
   }

   if (aPoint.iY < 0)
   {
      dstPoint.iY -= aPoint.iY;
   }

   switch (DisplayMode())
   {
      case EColor256:
      case EGray256:
         switch( src_orient )
         {
         case EOrientationNormal:
            (this->*KDirectDrawers8bpp[aSrcDevice.DisplayMode()])(rect.iTl, aSrcDevice, dstPoint, rect.Size());
            break;

         default:
            ASSERT(EFalse);
            break;
         }
         break;

      case EColor4K:
      case EColor64K:
         switch( src_orient )
         {
         case EOrientationNormal:
            (this->*KDirectDrawers12or16bpp[aSrcDevice.DisplayMode()])(rect.iTl, aSrcDevice, dstPoint, rect.Size());
            break;

         case EOrientationRotLeft:
            (this->*KDirectLeftDrawers12or16bpp[aSrcDevice.DisplayMode()])(rect.iTl, aSrcDevice, dstPoint, aRect.Size());
            break;

         case EOrientationRotRight:
            (this->*KDirectRightDrawers12or16bpp[aSrcDevice.DisplayMode()])(rect.iTl, aSrcDevice, dstPoint, aRect.Size());
            break;

         default:
            ASSERT(EFalse);
            break;
         }
         break;

      default:
         ASSERT(EFalse);
         break;
   }
}

void CGameScreenDevice::AlphaBlend(const TPoint& aPoint, const CGameScreenDevice& aSrcDevice, const TRect& aRect, TInt8 aAlpha)
{
   TRect   rect = TGameBitmapUtil::VisibleIntersection(Size(), aPoint, aSrcDevice.Size(), aRect);
   TPoint  dstPoint = aRect.iTl;
   if (aPoint.iX < 0)
      dstPoint.iX -= aPoint.iX;
   if (aPoint.iY < 0)
      dstPoint.iY -= aPoint.iY;

   switch (DisplayMode())
   {
      case EColor4K:
         (this->*KAlphaBlenders12bpp[aSrcDevice.DisplayMode()])(rect.iTl, aSrcDevice, dstPoint, rect.Size(), aAlpha);
         break;
      case EColor64K:
         (this->*KAlphaBlenders16bpp[aSrcDevice.DisplayMode()])(rect.iTl, aSrcDevice, dstPoint, rect.Size(), aAlpha);
         break;
      default:
         ASSERT(EFalse);
         break;
   }
}

#ifdef USE_SCALED_DRAWER
void CGameScreenDevice::ScaledDrawBitmap(const TScaledBitmapContext& aScaledBitmapContext)
{
   CGameScreenDevice::TScaledScanLineDrawer scaledScanLineDrawer= NULL;

   switch (DisplayMode())
   {
      case EColor4K:
      case EColor64K:
         scaledScanLineDrawer = KScaledScanLineDrawers12or16bpp[aScaledBitmapContext.SrcGameScreenDevice().DisplayMode()];
         break;
      default:
         break;
   }
   if (!scaledScanLineDrawer)
   {
      ASSERT(EFalse);
      return;
   }

   TUint8 *dst_line = (TUint8*) TGameBitmapUtil::PixelAddress(this, aScaledBitmapContext.Point());
   const CGameScreenDevice& srcGameScreenDevice = aScaledBitmapContext.SrcGameScreenDevice();

   TInt   height = aScaledBitmapContext.SrcGameScreenDevice().Size().iHeight;
   TInt   lineCount = 0;
   TInt   temp=0;

   for (TInt h=0 ; h<height ; h++)
   {
      if (lineCount == aScaledBitmapContext.SkipY())
      {
         (this->*scaledScanLineDrawer)(aScaledBitmapContext, (TUint*)srcGameScreenDevice.ScanLineAddress(h), ETrue);
         lineCount = 0;
      }
      else
      {
         (this->*scaledScanLineDrawer)(aScaledBitmapContext, (TUint*) srcGameScreenDevice.ScanLineAddress(h), EFalse);
         lineCount++;
      }
      if (lineCount != aScaledBitmapContext.SkipY())
      {
         my_memcpy(dst_line, aScaledBitmapContext.ScanLine(), aScaledBitmapContext.ScanLineBytes());
         dst_line += BytesPerLine();
         temp++;
      }
   }
}
#endif

void CGameScreenDevice::DirectLeftDraw_12or16bpp_12or16bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize)
{
   TInt width  = aSize.iWidth;
   TInt height = aSize.iHeight;

   register TUint16 *dstLine = (TUint16*) TGameBitmapUtil::PixelAddress(this, TPoint( aDstOrigin.iX + (aSize.iHeight - 1), aDstOrigin.iY ));
   register TUint16 *srcLine = (TUint16*) TGameBitmapUtil::PixelAddress(&aSrcDevice, aSrcOrigin);

   for( int i = 0; i < height; i++ )
   {
      TUint16 *dst = dstLine;
      TUint16 *src = srcLine;
      for( int j = 0; j < width; j++ )
      {
         *dst = *src++;
         dst += BytesPerLine() >> 1;
      }
      dstLine --;
      srcLine += aSrcDevice.BytesPerLine() >> 1;
   }
}

void CGameScreenDevice::DirectRightDraw_12or16bpp_12or16bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize)
{
   TInt width  = aSize.iWidth;
   TInt height = aSize.iHeight;

   register TUint16 *dstLine = (TUint16*) TGameBitmapUtil::PixelAddress(this, TPoint(aDstOrigin.iX, aDstOrigin.iY + (aSize.iWidth - 1)));
   register TUint16 *srcLine = (TUint16*) TGameBitmapUtil::PixelAddress(&aSrcDevice, aSrcOrigin);

   for( int i = 0; i < height; i++ )
   {
      TUint16 *dst = dstLine;
      TUint16 *src = srcLine;
      for( int j = 0; j < width; j++ )
      {
         *dst = *src++;
         dst -= BytesPerLine() >> 1;
      }
      dstLine ++;
      srcLine += aSrcDevice.BytesPerLine() >> 1;
   }
}

void CGameScreenDevice::DirectDraw_8bpp_8bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize)
{
   ASSERT(aSrcDevice.Palette());
   ASSERT(iDisplayMode == aSrcDevice.Palette()->DisplayMode());

   TUint8*         dst_line = (TUint8*) TGameBitmapUtil::PixelAddress(this, aDstOrigin);
   const TUint8*   src_line = TGameBitmapUtil::PixelAddress(&aSrcDevice, aSrcOrigin);

   TInt   height = aSize.iHeight;
   TInt   width = aSize.iWidth;

   for(TInt h=0 ; h<height ; h++)
   {
      Mem::Copy(dst_line, src_line, width);
      dst_line += (BytesPerLine() >> 1);
      src_line += aSrcDevice.BytesPerLine();
   }
}

void CGameScreenDevice::DirectDraw_12or16bpp_8bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize)
{
   ASSERT(aSrcDevice.Palette());
   ASSERT(iDisplayMode == aSrcDevice.Palette()->DisplayMode());

   TUint*         pal = aSrcDevice.Palette()->PaletteBuffer();
   TUint16*      dst_line = (TUint16*) TGameBitmapUtil::PixelAddress(this, aDstOrigin);
   const TUint8*   src_line = TGameBitmapUtil::PixelAddress(&aSrcDevice, aSrcOrigin);

   if (!pal)
      return;

   TInt   height = aSize.iHeight;
   TInt   width = aSize.iWidth;

   for(TInt h=0 ; h<height ; h++)
   {
      TUint16*      dst = dst_line;
      const TUint8*   src = src_line;
      for (TInt w=0 ; w<width ; w++)
      {
         *dst++ = (TUint16) pal[*src++];
      }
      dst_line += (BytesPerLine() >> 1);
      src_line += aSrcDevice.BytesPerLine();
   }
}

void CGameScreenDevice::DirectDraw_12or16bpp_12or16bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize)
{
   TUint16*      dst_line = (TUint16*) TGameBitmapUtil::PixelAddress(this, aDstOrigin);
   const TUint16*   src_line = (TUint16*) TGameBitmapUtil::PixelAddress(&aSrcDevice, aSrcOrigin);

   TInt   height = aSize.iHeight;
   TInt   width = aSize.iWidth;

   for(TInt h=0 ; h<height ; h++)
   {
      TUint16*      dst = dst_line;
      const TUint16*   src = src_line;
      for (TInt w=0 ; w<width ; w++)
      {
         *dst++ = *src++;
      }
      dst_line += (BytesPerLine() >> 1);
      src_line += (aSrcDevice.BytesPerLine() >> 1);
   }
}

void CGameScreenDevice::AlphaBlend_12bpp_12bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize, TUint8 aAlpha)
{
   TUint16*      dst_line = (TUint16*) TGameBitmapUtil::PixelAddress(this, aDstOrigin);
   const TUint16*   src_line = (TUint16*) TGameBitmapUtil::PixelAddress(&aSrcDevice, aSrcOrigin);

   TInt   height = aSize.iHeight;
   TInt   width = aSize.iWidth;

   for(TInt h=0 ; h<height ; h++)
   {
      TUint16*      dst = dst_line;
      const TUint16*   src = src_line;
      for (TInt w=0 ; w<width ; w++)
      {
         // alpha blending algorithm "borrowed" from MicroWindows ;-)
         unsigned int s = *src++;
         unsigned int d = *dst;
         unsigned int t = d & 0xf00;
         unsigned int m1, m2, m3;
         m1 = (((((s & 0xf00) - t)*aAlpha)>>8) & 0xf00) + t;
         t = d & 0x0f0;
         m2 = (((((s & 0xf0) - t)*aAlpha)>>8) & 0xf0) + t;
         t = d & 0xf;
         m3 = (((((s & 0xf) - t)*aAlpha)>>8) & 0xf) + t;
         *dst++ = (TUint16) (m1 | m2 | m3);
      }
      dst_line += (BytesPerLine() >> 1);
      src_line += (aSrcDevice.BytesPerLine() >> 1);
   }
}

void CGameScreenDevice::AlphaBlend_16bpp_16bpp(const TPoint& aDstOrigin, const CGameScreenDevice& aSrcDevice, const TPoint& aSrcOrigin, const TSize& aSize, TUint8 aAlpha)
{
   TUint16*      dst_line = (TUint16*) TGameBitmapUtil::PixelAddress(this, aDstOrigin);
   const TUint16*   src_line = (TUint16*) TGameBitmapUtil::PixelAddress(&aSrcDevice, aSrcOrigin);

   TInt   height = aSize.iHeight;
   TInt   width = aSize.iWidth;

   for(TInt h=0 ; h<height ; h++)
   {
      TUint16*      dst = dst_line;
      const TUint16*   src = src_line;
      for (TInt w=0 ; w<width ; w++)
      {
         // alpha blending algorithm "borrowed" from MicroWindows ;-)
         unsigned int s = *src++;
         unsigned int d = *dst;
         unsigned int t = d & 0xf800;
         unsigned int m1, m2, m3;
         m1 = (((((s & 0xf800) - t)*aAlpha)>>8) & 0xf800) + t;
         t = d & 0x07e0;
         m2 = (((((s & 0x07e0) - t)*aAlpha)>>8) & 0x07e0) + t;
         t = d & 0x001f;
         m3 = (((((s & 0x001f) - t)*aAlpha)>>8) & 0x001f) + t;
         *dst++ = (TUint16) (m1 | m2 | m3);
      }
      dst_line += (BytesPerLine() >> 1);
      src_line += (aSrcDevice.BytesPerLine() >> 1);
   }
}

#ifdef USE_SCALED_DRAWER
void CGameScreenDevice::ScaledScanLineDraw_12or16bpp_8bpp(const TScaledBitmapContext& aScaledBitmapContext, TUint* aSrcLine, TBool aMerge)
{
   TInt      pixelCount = 0;
   TUint8*      src = (TUint8*) aSrcLine;
   TUint16*   dst = (TUint16*) aScaledBitmapContext.ScanLine();
   TUint*      pal = aScaledBitmapContext.SrcGameScreenDevice().Palette()->PaletteBuffer();
   TInt      width = aScaledBitmapContext.SrcGameScreenDevice().Size().iWidth;

   for (TInt w=0 ; w<width ; w++)
   {
      if (pixelCount == aScaledBitmapContext.SkipX())
      {
         pixelCount = 0;
         *(dst -1) |= pal[*src];
      }
      else
      {
         if (aMerge)
         {
            *dst++ |= pal[*src];
         }
         else
         {
            *dst++ = (TInt16) pal[*src];
         }
         pixelCount++;
      }
      src++;
   }
}
#endif

//===============================================
//
// CFbsGameScreenDevice
//
//===============================================

CFbsGameScreenDevice* CFbsGameScreenDevice::NewL(const TSize& aSize, TDisplayMode aDisplayMode)
{
   CFbsGameScreenDevice *self = new (ELeave) CFbsGameScreenDevice(aSize, aDisplayMode);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CFbsGameScreenDevice::CFbsGameScreenDevice
( const TSize& aSize
, TDisplayMode aDisplayMode
) : CGameScreenDevice(aSize, aDisplayMode, EOrientationNormal)
{
   iBytesPerLine = ((TGameBitmapUtil::GetBpp(aDisplayMode) * aSize.iWidth ) + 7) / 8;
}

CFbsGameScreenDevice::~CFbsGameScreenDevice()
{
   delete iBitmapContext;
   iBitmapContext = NULL;
   delete iBitmapDevice;
   iBitmapDevice = NULL;
   delete iBitmap;
   iBitmap = NULL;
}

void CFbsGameScreenDevice::ConstructL()
{
   iBitmap = new (ELeave) CExtraFbsBitmap;
   iBitmap->Create(iSize, iDisplayMode);

   iBitmapDevice = CFbsBitmapDevice::NewL(iBitmap);
   iBitmapDevice->CreateContext(iBitmapContext);

   iDataAddress = iBitmap->DataAddress();
   iBitwiseBitmap =iBitmap->BitmapAdressAddress();
}

TUint8* CFbsGameScreenDevice::ScanLineAddress(TInt aLine) const
{
   return (TUint8 *) iBitwiseBitmap->ScanLineAddress(iBitmap->DataAddress(), aLine);
}

//===============================================
//
// CHwFbGameScreenDevice
//
//===============================================

CHwFbGameScreenDevice* CHwFbGameScreenDevice::NewL(const TSize& aWindowSize, TDisplayMode aDisplayMode)
{
   TScreenInfoV01         screenInfo;
   TPckg<TScreenInfoV01>  sI(screenInfo);
   UserSvr::ScreenInfo(sI);

   if (!screenInfo.iScreenAddressValid)
   {
      User::Leave(KErrNotSupported);
   }

      /*
       * Note that aWindowSize may differ from screenInfo.iScreenSize.
       * screenInfo.iScreenSize reflects the video controller's configured pixel dimensions, 
       * whereas aWindowSize is the LCD's screen's maximim dimension.
       */
   CHwFbGameScreenDevice *self = new (ELeave) CHwFbGameScreenDevice(aWindowSize, aDisplayMode);
   CleanupStack::PushL(self);
   self->ConstructL(screenInfo);
   CleanupStack::Pop();
   return self;
}

CHwFbGameScreenDevice::CHwFbGameScreenDevice
( const TSize& aWindowSize
, TDisplayMode aDisplayMode
) : CGameScreenDevice(aWindowSize, aDisplayMode, EOrientationNormal)
{
}

void CHwFbGameScreenDevice::ConstructL(const TScreenInfoV01& aScreenInfo)
{
   iFrameBuffer = (TUint8*) aScreenInfo.iScreenAddress;
   iBytesPerLine = ((TGameBitmapUtil::GetBpp(iDisplayMode) * aScreenInfo.iScreenSize.iWidth ) + 7) / 8;

   TInt value = 0;
   HAL::Get(HALData::EMachineUid,value);
   TInt offset = 0;
   switch(value)
   {
   case 0x101f6b26: // A9xx & A10xx
   case 0x101fd279: // P3x
      offset = 32;
      break;
   case 0x101f408b: // P800
   case 0x101fb2ae: // P900
   case 0x10200ac6: // P910
   default:	
      break;
   }

   iFrameBuffer += offset;
}

TUint8* CHwFbGameScreenDevice::ScanLineAddress(TInt aLine) const
{
   return iFrameBuffer + (BytesPerLine() * aLine);
}

//===============================================
//
// CMemGameScreenDevice
//
//===============================================

CMemGameScreenDevice* CMemGameScreenDevice::NewL
( const TSize& aSize
, TDisplayMode aDisplayMode
, TOrientation aOrientation
)
{
   CMemGameScreenDevice *self = new (ELeave) CMemGameScreenDevice(aSize, aDisplayMode, aOrientation);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CMemGameScreenDevice::CMemGameScreenDevice
( const TSize& aSize
, TDisplayMode aDisplayMode
, TOrientation aOrientation
) : CGameScreenDevice(aSize, aDisplayMode, aOrientation)
{
   iMemBuffer = NULL;
   iBytesPerLine = ((TGameBitmapUtil::GetBpp(aDisplayMode) * aSize.iWidth ) + 7) / 8;
}

CMemGameScreenDevice::~CMemGameScreenDevice()
{
   User::Free(iMemBuffer);
   iMemBuffer = NULL;
}

void CMemGameScreenDevice::ConstructL()
{
   TInt size  = iBytesPerLine * iSize.iHeight;
   iMemBuffer = (TUint8*) User::AllocL(size);
   Mem::FillZ( iMemBuffer, size );
}

TUint8* CMemGameScreenDevice::ScanLineAddress(TInt aLine) const
{
   return ( iMemBuffer + (BytesPerLine() * aLine) );
}

//===============================================
//
// CGameWindow
//
//===============================================

CGameWindow* CGameWindow::NewL(const TDesC& aGameName)
{
   CGameWindow *self = new (ELeave) CGameWindow;
   CleanupStack::PushL(self);
   self->ConstructL(aGameName);
   CleanupStack::Pop();
   return self;
}

CGameWindow::CGameWindow()
{
#ifdef USE_WINDOW_PANER
   iGamePaner = NULL;
#endif

   iDisableSendEnd = EFalse;
}

CGameWindow::~CGameWindow()
{
#ifdef USE_WINDOW_PANER
   delete iGamePaner;
   iGamePaner = NULL;
#endif

#ifdef USE_GAME_INTRO
   delete iGameIntro;
   iGameIntro = NULL;
#endif

   delete iInfoMgr;
   iInfoMgr = NULL;

   delete iSmartPenToKeyInputHandler;
   iSmartPenToKeyInputHandler = 0;

   DestructVirtualKeys();

   DestructGameController();
   DestructScreenDevice();
   DestructWsResources();

   delete iScreenOrientationObservers;
   iScreenOrientationObservers = NULL;

   delete iGameScreenDrawObservers;
   iGameScreenDrawObservers = NULL;
}

void CGameWindow::ConstructL(const TDesC& aGameName)
{
   iGameName = aGameName.Left(iGameName.MaxLength());

   iGameScreenDrawObservers = new(ELeave)CArrayPtrFlat<MGameScreenDrawObserver>(5);
   iScreenOrientationObservers = new(ELeave)CArrayPtrFlat<MScreenOrientationObserver>(5);

   ConstructWsResourcesL();
   ConstructScreenDeviceL();
   ConstructGameControllerL();
   ConstructVirtualKeysL();

#ifdef USE_POINTER_KEY_INPUT
   ConstructSmartPenToKeyInputHandlerL();
#endif

   ClearScreen();

   SetGameScreenDrawer(&iDirectBitmapDrawer);

#ifdef USE_WINDOW_PANER
   iGamePaner = new (ELeave) TGamePaner();
   iGamePaner->ActivateL(this, iGameController, EOrientationNormal);
#endif

   iInfoMgr = CInfoMgr::NewL(*this, *iGameController);

#ifdef USE_GAME_INTRO
   iGameIntro = CGameIntro::NewL(*this, *iGameController);
#endif

   iMemoryGameScreenDevice = CMemGameScreenDevice::NewL( iGameScreenDevice->Size()
                                                       , iGameScreenDevice->DisplayMode()
                                                       , EOrientationNormal
                                                       );

   CaptureKeyEvents();
   CaptureKeyUpDowns();
}

void CGameWindow::ConstructSmartPenToKeyInputHandlerL()
{
   if (!iSmartPenToKeyInputHandler)
   {
      iSmartPenToKeyInputHandler = CSmartPenToKeyInputHandler::NewL(*this, *iGameController);
   }
}

void CGameWindow::CanDo_X_And_Y_AtTheSameTime(TBool aCanDo)
{
   if (iSmartPenToKeyInputHandler)
      iSmartPenToKeyInputHandler->CanDo_X_And_Y_AtTheSameTime(aCanDo);
}

void CGameWindow::DisableSendEndKeys()
{
   iDisableSendEnd = ETrue;

   ReleaseKeyEvents();
   ReleaseKeyUpDowns();
   CaptureKeyEvents();
   CaptureKeyUpDowns();
}

void CGameWindow::CaptureKeyEvents()
{
   const TUint keycodes[] =
   { DEVICE_KEYCODE_BUTTON1
   , DEVICE_KEYCODE_BUTTON2
   , DEVICE_KEYCODE_HOME
   , DEVICE_KEYCODE_SHORTCUT
   , DEVICE_KEYCODE_VOICE
   , DEVICE_KEYCODE_CAMERA
   , DEVICE_KEYCODE_VOLUP
   , DEVICE_KEYCODE_VOLDN
   , DEVICE_KEYCODE_PXXX_JOG_UP
   , DEVICE_KEYCODE_PXXX_JOG_DOWN
   , DEVICE_KEYCODE_PXXX_JOG_LEFT
   , DEVICE_KEYCODE_PXXX_JOG_RIGHT
   , 0
   };

   TInt i = 0;
   while ( keycodes[i] != 0 )
   {
      if (!iCapturedButtons[i])
      {
         iCapturedButtons[i] = iWsWindowGroup.CaptureKey( keycodes[i], 0, 0, 1 );
      }
      i++;
   }

   if ( iDisableSendEnd )
   {
      iCapturedButtons[i++] = iWsWindowGroup.CaptureKey( DEVICE_KEYCODE_SEND, 0, 0, 1 );
      iCapturedButtons[i]   = iWsWindowGroup.CaptureKey( DEVICE_KEYCODE_END, 0, 0, 1 );
   }
}

void CGameWindow::CaptureKeyUpDowns()
{
   const TUint scancodes[] =
   { DEVICE_SCANCODE_BUTTON1
   , DEVICE_SCANCODE_BUTTON2
   , DEVICE_SCANCODE_HOME
   , DEVICE_SCANCODE_SHORTCUT
   , DEVICE_SCANCODE_VOICE
   , DEVICE_SCANCODE_CAMERA
   , DEVICE_SCANCODE_VOLUP
   , DEVICE_SCANCODE_VOLDN
   , DEVICE_SCANCODE_PXXX_JOG_UP
   , DEVICE_SCANCODE_PXXX_JOG_DOWN
   , DEVICE_SCANCODE_PXXX_JOG_LEFT
   , DEVICE_SCANCODE_PXXX_JOG_RIGHT
   , 0
   };

   TInt i = 0;
   while ( scancodes[i] != 0 )
   {
      if (!iCapturedUpDowns[i])
      {
         iCapturedUpDowns[i] = iWsWindowGroup.CaptureKeyUpAndDowns( scancodes[i], 0, 0, 1 );
      }
      i++;
   }

   if ( iDisableSendEnd )
   {
      iCapturedUpDowns[i++] = iWsWindowGroup.CaptureKeyUpAndDowns( DEVICE_SCANCODE_SEND, 0, 0, 1 );
      iCapturedUpDowns[i]   = iWsWindowGroup.CaptureKeyUpAndDowns( DEVICE_SCANCODE_END, 0, 0, 1 );
   }
}

void CGameWindow::ReleaseKeyEvents()
{
   for (TInt i=0 ; i<KMaxCapturedButtons ; i++)
   {
      if (iCapturedButtons[i])
      {
         iWsWindowGroup.CancelCaptureKey(iCapturedButtons[i]);
         iCapturedButtons[i] = 0;
      }
   }
}

void CGameWindow::ReleaseKeyUpDowns()
{
   for (TInt i=0 ; i<KMaxCapturedButtons ; i++)
   {
      if (iCapturedUpDowns[i])
      {
         iWsWindowGroup.CancelCaptureKeyUpAndDowns(iCapturedUpDowns[i]);
         iCapturedUpDowns[i] = 0;
      }
   }
}

void CGameWindow::Hide()
{
   ReleaseKeyEvents();
   ReleaseKeyUpDowns();
   iGameController->ResetKeys();
   iWsWindowGroup.SetOrdinalPosition(KMinTInt, KMinTInt);
   iWsWindow.SetVisible(EFalse);

   TRect   rect(iWsWindow.Position(), iWsWindow.Size());
   iWindowGc->Activate(iWsWindow);
   iWsWindow.Invalidate();
   iWsWindow.BeginRedraw(rect);
   iWsWindow.EndRedraw();
   iWindowGc->Deactivate();
   
   iWsSession.Flush();
   iWasPaused = ETrue;
}

void CGameWindow::Show()
{
   iWsWindowGroup.SetOrdinalPosition(0, 0);
   iWsWindow.SetVisible(ETrue);
   CaptureKeyEvents();
   CaptureKeyUpDowns();
}

void CGameWindow::ConstructWsResourcesL()
{
   TInt   error;

   error = iWsSession.Connect();
   User::LeaveIfError(error);
   iWsScreen = new (ELeave) CWsScreenDevice(iWsSession);
   User::LeaveIfError(iWsScreen->Construct());
   User::LeaveIfError(iWsScreen->CreateContext(iWindowGc));

   iWsWindowGroup = RWindowGroup(iWsSession);
   User::LeaveIfError(iWsWindowGroup.Construct((TUint32)&iWsWindowGroup));
   iWsWindowGroup.SetOrdinalPosition(0);
   iWsWindowGroup.SetName(iGameName);

   iWsWindow = RWindow(iWsSession);
   User::LeaveIfError(iWsWindow.Construct(iWsWindowGroup, (TUint32)&iWsWindow));
   iWsWindow.Activate();
   iWsWindow.SetSize(iWsScreen->SizeInPixels());

#ifdef __TEST_DIRECT_SCREEN_ACCESS__
   // you can then click on the app picker to force something to move in front of the game 
   iWsWindow.SetPosition(TPoint(0, 4));
#endif

   iWsWindow.SetVisible(ETrue);
   iWsWindow.SetPointerGrab(ETrue);
   iWsWindow.PointerFilter(EPointerFilterDrag, 0);

      /*
       * Tell the Window Server not to mess about with our process priority
       * Also, because of the way legacy games are written, they never sleep
       * and thus never voluntarily yield the CPU. We set our process priority
       * to EPriorityForeground and hope that a Telephony application on
       * this device runs at EPriorityForeground as well. If not, tough! ;-)
       */
   iWsSession.ComputeMode(RWsSession::EPriorityControlDisabled);
   RProcess      me;
   me.SetPriority(EPriorityForeground);
}

void CGameWindow::DestructWsResources()
{
   if (iWsWindow.WsHandle())
      iWsWindow.Close();

   if (iWsWindowGroup.WsHandle())
      iWsWindowGroup.Close();

   delete iWindowGc;
   iWindowGc = NULL;

   delete iWsScreen;
   iWsScreen = NULL;

   if (iWsSession.WsHandle())
      iWsSession.Close();
}


void CGameWindow::ConstructGameControllerL()
{
   iGameController = CGameController::NewL(iWsSession);
   AddGameScreenDrawObserverL(*iGameController);
   AddScreenOrientationObserverL(*iGameController);
}

void CGameWindow::DestructGameController()
{
   delete iGameController;
   iGameController = NULL;
}

void CGameWindow::ConstructScreenDeviceL()
{
   TDisplayMode mode = iWsScreen->DisplayMode();
   TSize        size = iWsScreen->SizeInPixels();

#ifdef __WINS__
   iUseWServ = ETrue;
#endif

   if (iUseWServ)
   {
         /*
          * For WINS we emulate using a Fbs bitmap which we blit to the window's device
          * context. You can test most behaviour by:
          * 1) adjusting the size and display mode in the CFbsGameScreenDevice::NewL
          * 2) configuring the emulator for the appropriate size and mode
          */
      iGameScreenDevice = CFbsGameScreenDevice::NewL(size, mode);
   }
   else
   {
      iGameScreenDevice = CHwFbGameScreenDevice::NewL(size, mode);
   }

   iGameDirectScreenAccess = CGameDirectScreenAccess::NewL(*this);
   AddGameScreenDrawObserverL(*iGameDirectScreenAccess);
}

void CGameWindow::DestructScreenDevice()
{
   delete iGameScreenDevice;
   iGameScreenDevice = NULL;

   delete iGameDirectScreenAccess;
   iGameDirectScreenAccess = NULL;
}

#define BUTTON_X_BASE   (49)
#define BUTTON_X_HEIGHT (27)
void CGameWindow::ConstructVirtualKeysL()
{
#ifdef USE_VIRTUAL_KEYBOARD
   iVirtualKeyboard = CVirtualKeyboard::NewL( *this, *iGameController );
#endif

#ifdef USE_VIRTUAL_BUTTONS
   static TVirtualButton buttons[] =
   { { DEVICE_SCANCODE_ESC,     BUTTON_X_BASE,                     3, (BUTTON_X_BASE+(1*BUTTON_X_HEIGHT))-1, 30, 0, 1 }
   , { DEVICE_SCANCODE_SELECT2, BUTTON_X_BASE+(1*BUTTON_X_HEIGHT), 3, (BUTTON_X_BASE+(2*BUTTON_X_HEIGHT))-1, 30, 2, 3 }
   , { DEVICE_SCANCODE_START,   BUTTON_X_BASE+(2*BUTTON_X_HEIGHT), 3, (BUTTON_X_BASE+(3*BUTTON_X_HEIGHT))-1, 30, 4, 5 }
   , { DEVICE_SCANCODE_EXTRA,   BUTTON_X_BASE+(3*BUTTON_X_HEIGHT), 3, (BUTTON_X_BASE+(4*BUTTON_X_HEIGHT))-1, 30, 6, 7 }
   , { 0, 0, 0, 0, 0, 0, 0 }
   };

   iVirtualButtons = CVirtualButtons::NewL( *this, *iGameController, buttons );

   iVirtualButtons->ActivateL();
#endif
}

void CGameWindow::DestructVirtualKeys()
{
#ifdef USE_VIRTUAL_BUTTONS
   delete iVirtualButtons;
   iVirtualButtons = NULL;
#endif

#ifdef USE_VIRTUAL_KEYBOARD
   delete iVirtualKeyboard;
   iVirtualKeyboard = NULL;
#endif
}

void CGameWindow::GameFrameTick(CGameScreenDevice* aSrcGameScreenDevice)
{
   CGameScreenDevice *screen_device = iGameScreenDevice;
   
#ifdef ALLOW_VIDEO_MEM_BUFFERING
   /*
   * If any of the screen draw observers want to draw something to screen
   * we will use the memory screen device first. This will prevent the usual
   * screen flicker.
   * Also, if we used to use the memory screen device and this frame we don't
   * then we use it one more time to properly clear the screen.
   */
   TBool              use_mem_screen_device = EFalse;
   for (TInt ii = 0; ii < iGameScreenDrawObservers->Count() && !use_mem_screen_device; ii++)
   {
      use_mem_screen_device = (*iGameScreenDrawObservers)[ii]->WantDrawToScreen();
   }
   
   if (use_mem_screen_device || iUsedMemoryScreenDevice || iWasPaused)
   {
      screen_device = iMemoryGameScreenDevice;
      iMemoryGameScreenDevice->Clear();
      iWasPaused = EFalse;
   }
#endif
   
   for (TInt i = 0; i < iGameScreenDrawObservers->Count(); i++)
   {
      (*iGameScreenDrawObservers)[i]->PreDrawFrameToScreen(screen_device);
   }
   
   if (iGameScreenDrawer)
   {
      iGameScreenDrawer->DrawFrame(screen_device, aSrcGameScreenDevice);
   }
   
   for (TInt j = 0; j < iGameScreenDrawObservers->Count(); j++)
   {
      (*iGameScreenDrawObservers)[j]->PostDrawFrameToScreen(screen_device);
   }
   
#ifdef ALLOW_VIDEO_MEM_BUFFERING
   if (use_mem_screen_device || iUsedMemoryScreenDevice)
   {
      iGameScreenDevice->DirectDraw(TPoint(0,0), *iMemoryGameScreenDevice, iGameScreenDevice->Size());
   }
   
   iUsedMemoryScreenDevice = use_mem_screen_device;

   if (iUseWServ)
   {
      WsBlitBitmap((CFbsGameScreenDevice*) iGameScreenDevice);
   }
#endif
}

void CGameWindow::ClearScreen()
{
   iGameScreenDevice->Clear();

   if (iUseWServ)
      WsBlitBitmap((CFbsGameScreenDevice*) iGameScreenDevice, TRect(iGameScreenDevice->Size()));
}

void CGameWindow::SetGameScreenDrawer(MGameScreenDrawer* aGameScreenDrawer)
{
   if (aGameScreenDrawer)
      iGameScreenDrawer = aGameScreenDrawer;
   else
      iGameScreenDrawer = &iDirectBitmapDrawer;
}

void CGameWindow::WsBlitBitmap(CFbsGameScreenDevice* aFbsGameScreenDevice)
{
   WsBlitBitmap(aFbsGameScreenDevice, GameVisibleArea());
}

void CGameWindow::WsBlitBitmap(CFbsGameScreenDevice* aFbsGameScreenDevice, const TRect& aRect)
{
   iWindowGc->Activate(iWsWindow);

   iWsWindow.Invalidate(aRect);
   iWsWindow.BeginRedraw(aRect);

   if (aFbsGameScreenDevice)
   {
      iWindowGc->BitBlt(aRect.iTl, aFbsGameScreenDevice->FbsBitmap(), aRect);
   }

   iWsWindow.EndRedraw();
   iWindowGc->Deactivate();
   iWsSession.Flush();
}

void CGameWindow::AddScreenOrientationObserverL(MScreenOrientationObserver& aObserver)
{
   iScreenOrientationObservers->AppendL(&aObserver);
   aObserver.ScreenOrientationChanged(iOrientation);
}

void CGameWindow::SetScreenOrientation(TOrientation aOrientation)
{
   iOrientation = aOrientation;
   for (TInt i=0 ; i<iScreenOrientationObservers->Count() ; i++)
      (*iScreenOrientationObservers)[i]->ScreenOrientationChanged(iOrientation);
}

//===============================================
//
// CGamePalette
//
//===============================================

CGamePalette::CGamePalette(TDisplayMode aDisplayMode) : iDisplayMode(aDisplayMode)
{
}

CGamePalette::~CGamePalette()
{
}

CGamePalette* CGamePalette::NewL(TDisplayMode aDisplayMode)
{
   CGamePalette *self = new (ELeave) CGamePalette(aDisplayMode);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

void CGamePalette::ConstructL()
{
   iPaletteMaxIndex = 256;
   Mem::FillZ(iPalette, sizeof(iPalette));
   iPaletteLastIndexUsed = 0;
}

TInt CGamePalette::AllocateEntry(const TRgb& aColor)
{
   TUint   colorValue = TGameBitmapUtil::GetColorValue(aColor, iDisplayMode);
   for (TInt i=0 ; i<iPaletteLastIndexUsed; i++)
      if (iPalette[i] == colorValue)
         return i;
   ASSERT(iPaletteLastIndexUsed<iPaletteMaxIndex);
   iPalette[iPaletteLastIndexUsed] = colorValue;
   return (iPaletteLastIndexUsed++);
}

//===================================================================
//
// Local Class Definitions
//
//===================================================================

//===============================================
//
// CGameDirectScreenAccess
//
//===============================================

CGameDirectScreenAccess::~CGameDirectScreenAccess()
{
   if (iRequestStatus == KRequestPending)
      iDirectScreenAccess.Cancel();
   if (iDirectScreenAccess.WsHandle() != -1)
      iDirectScreenAccess.Close();

   if (iSemaphore.Handle() != -1)
      iSemaphore.Close();
}

CGameDirectScreenAccess* CGameDirectScreenAccess::NewL(CGameWindow& aGameWindow)
{
   CGameDirectScreenAccess*   self = new(ELeave)CGameDirectScreenAccess(aGameWindow);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CGameDirectScreenAccess::CGameDirectScreenAccess(CGameWindow& aGameWindow) :
   iGameWindow(aGameWindow)
{
}

void CGameDirectScreenAccess::ConstructL()
{
   iDirectScreenAccess = RDirectScreenAccess(iGameWindow.iWsSession);
   User::LeaveIfError(iDirectScreenAccess.Construct());
   TBuf<32>   semaphoreName;
   semaphoreName.Format(_L("%S_SEM"), &iGameWindow.iGameName);
   User::LeaveIfError(iSemaphore.CreateGlobal(semaphoreName, 0));

   ClaimScreenL();
}

void CGameDirectScreenAccess::ClaimScreenL()
{
   iRequestStatus = KRequestPending;

   RRegion*   region;
   User::LeaveIfError(iDirectScreenAccess.Request(region, iRequestStatus, iGameWindow.iWsWindow));
   TBool   isFullScreen = region->IsContainedBy(TRect(iGameWindow.iWsWindow.Position(), iGameWindow.iWsWindow.Size()));
   if (!isFullScreen)
      User::Leave(KErrInUse);
}

void CGameDirectScreenAccess::PauseGameL()
{
   iDirectScreenAccess.Completed();

   User::After(TTimeIntervalMicroSeconds32(1000000));
   iGameWindow.Hide();

   iSemaphore.Wait();
   ResumeGameL();
}

void CGameDirectScreenAccess::ResumeGameL()
{
   iGameWindow.Show();
   ClaimScreenL();
   iGameWindow.ClearScreen();
}

void CGameDirectScreenAccess::PreDrawFrameToScreen(CGameScreenDevice* /*aSrcGameScreenDevice*/)
{
   if (iRequestStatus != KRequestPending)
      PauseGameL();
}

//===============================================
//
// CGameIntro
//
//===============================================
#ifdef USE_GAME_INTRO

static TRgb sRgbTable[] = 
{
   TRgb(81, 81, 236),
   TRgb(78, 138, 239),
   TRgb(77, 171, 240),
   TRgb(77, 203, 240),
   TRgb(77, 240, 227),
   TRgb(75, 241, 175),
   TRgb(75, 241, 130),
   TRgb(79, 242, 74),
   TRgb(133, 243, 73),
   TRgb(188, 243, 73),
   TRgb(243, 243, 73),
   TRgb(244, 188, 72),
   TRgb(245, 132, 71),
   TRgb(245, 71, 71),
   TRgb(247, 70, 132),
   TRgb(248, 69, 194),
   TRgb(243, 69, 248),
   TRgb(181, 68, 249),
   TRgb(177, 67, 250),
   TRgb(70, 66, 251)
};

static TInt iRgbIndex = 0;
#define MAX_RGB_TABLE   (sizeof(sRgbTable)/sizeof(TRgb))

CGameIntro* CGameIntro::NewL(CGameWindow& aGameWindow, CGameController& aGameController)
{
   CGameIntro*   self = new(ELeave)CGameIntro(aGameWindow, aGameController);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CGameIntro::CGameIntro
( CGameWindow& aGameWindow
, CGameController& aGameController
) : iPosition(0, 0)
  , iSize(200, 40)
  , iGameWindow(aGameWindow)
  , iGameController(aGameController)
  , iIsActive(ETrue)
  , iFgColor(KRgbWhite)
  , iBgColor(KRgbBlack)
{
}

CGameIntro::~CGameIntro()
{
   Destruct();
}

void CGameIntro::ConstructL()
{
   iBitmap = CFbsGameScreenDevice::NewL(iSize, iGameWindow.GameScreenDevice()->DisplayMode());

   TFontSpec fontSpec(KFontName, 20);
   User::LeaveIfError(iBitmap->FbsBitmapDevice()->GetNearestFontInPixels(iFont, fontSpec));

   CFbsBitGc*   gc = iBitmap->FbsBitGc();
   gc->UseFont(iFont);

   iGameController.AddGameControlInputHandlerL(*this);
   iGameWindow.AddGameScreenDrawObserverL(*this);

   DrawBitmap();
   BitmapRecolor();
}

void CGameIntro::Destruct()
{
   if (iFont)
   {
      iBitmap->FbsBitmapDevice()->ReleaseFont(iFont);
      iFont = NULL;
   }
   if (iBitmap)
   {
      delete iBitmap;
      iBitmap = NULL;
   }
}

void CGameIntro::PostDrawFrameToScreen(CGameScreenDevice* aDstGameScreenDevice)
{
   TPoint p = iGameWindow.GameOrigin() + iPosition;
   if (p.iX < 0)
      p.iX = 0;
   if (p.iY < 0)
      p.iY = 0;

   if (++iFrames == 6)
   {
      iFrames = 0;
      if (++iRgbIndex == MAX_RGB_TABLE)
         iRgbIndex =0;
      DrawBitmap();
      BitmapRecolor();
   }
   aDstGameScreenDevice->DirectDraw(p, *iBitmap, TRect(iSize));
}

TBool CGameIntro::WantDrawToScreen()
{
   return iIsActive;
}

void CGameIntro::HandleKeyEvent(TInt aType, const TKeyEvent& /*aKeyEvent*/)
{
   if (aType == EEventKey)
   {
      iGameWindow.RemoveGameScreenDrawObserver(*this);
      iIsActive = EFalse;
   }
}

void CGameIntro::BitmapRecolor()
{
   TDisplayMode displayMode = iGameWindow.GameScreenDevice()->DisplayMode();
   TUint16      white = (TUint16) TGameBitmapUtil::GetColorValue(KRgbWhite.Color64K(), displayMode);
   TInt      index = iRgbIndex;
   TUint16      color;

   for (TInt h=0 ; h<iSize.iHeight ; h++)
   {
      TUint16*   p = (TUint16*) iBitmap->ScanLineAddress(h);

      TInt      index1 = index;
      TInt      count = 0;
      color = (TUint16) TGameBitmapUtil::GetColorValue(sRgbTable[index1], displayMode);
      for (TInt w=0 ; w<iSize.iWidth ; w++)
      {
         if (*p == white)
            *p = color;
         p++;

         if (++count == 16)
         {
            if (++index1 == MAX_RGB_TABLE)
               index1 = 0;
            count = 0;
            color = (TUint16) TGameBitmapUtil::GetColorValue(sRgbTable[index1], displayMode);
         }
      }
      if (++index == MAX_RGB_TABLE)
      {
         index = 0;
      }
   }
}

void CGameIntro::DrawBitmap()
{
   TBuf<32>   text;

   CFbsBitGc*   gc = iBitmap->FbsBitGc();
   gc->SetBrushColor(iBgColor);
   gc->SetBrushStyle(CGraphicsContext::ENullBrush);
   gc->SetPenColor(iFgColor);
   gc->SetPenSize(TSize(1,1));
   gc->Clear();


   // draw text
   TRect   rect(TSize(iSize.iWidth, 10));
   text.Format(_L("Symbian OS port by"));
   TInt baselineOffset=(rect.Size().iHeight + iFont->AscentInPixels()) / 2;
   gc->DrawText(text, rect, baselineOffset, CGraphicsContext::ECenter);

   text.Format(_L("Steve Fischer"));
   rect.Move(0, 12);
   baselineOffset=(rect.Size().iHeight + iFont->AscentInPixels()) / 2;
   gc->DrawText(text, rect, baselineOffset, CGraphicsContext::ECenter);
   
   text.Format(_L("Modified for A920!"));
   rect.Move(0, 12);
   baselineOffset=(rect.Size().iHeight + iFont->AscentInPixels()) / 2;
   gc->DrawText(text, rect, baselineOffset, CGraphicsContext::ECenter);
}

#endif
//===============================================
//
// TGamePaner
//
//===============================================
#ifdef USE_WINDOW_PANER

void TGamePaner::ActivateL(CGameWindow* aGameWindow, CGameController* aGameController, TOrientation aOrientation)
{
   iGameWindow = aGameWindow;
   iGameController = aGameController;
   iOrientation = aOrientation;

   // wait for the first frame to see if it needs paning
   iGameWindow->AddGameScreenDrawObserverL(*this);
}

void TGamePaner::PostDrawFrameToScreen(CGameScreenDevice* /*aDstGameScreenDevice*/)
{
   TRect   rect = iGameWindow->GameVisibleArea();
   TSize   size = iGameWindow->GameSize() - rect.Size();

   if (size.iWidth > 0)
   {
      iDeltaX = 2;
   }
   if (size.iHeight > 0)
   {
      iDeltaY = 2;
   }
   if (iDeltaX || iDeltaY)
   {
      // yup, needs paning, so make sure we get key events
      iGameController->AddGameControlInputHandlerL(*this);
   }

   iCurDelta = iGameWindow->GameOrigin();
   iMaxDelta = size.AsPoint();
   iGameWindow->RemoveGameScreenDrawObserver(*this);
}

void TGamePaner::HandleKeyEvent(TInt aType, const TKeyEvent& aKeyEvent)
{
   TPoint   delta(0, 0);
   
   if (aType == EEventKey)
   {
      switch (aKeyEvent.iCode)
      {
      case GAME_RIGHT_KEYCODE:
         delta.iX = (iCurDelta.iX + iDeltaX > iMaxDelta.iX) ? iMaxDelta.iX - iCurDelta.iX : iDeltaX;
         break;
         
      case GAME_LEFT_KEYCODE:
         delta.iX = (iCurDelta.iX < iDeltaX) ? iCurDelta.iX*-1 : iDeltaX*-1;
         break;
         
      case GAME_UP_KEYCODE:
         if (delta.iY)
            delta.iY = (iCurDelta.iY < iDeltaY) ? iCurDelta.iY*-1 : iDeltaY*-1;
         break;
         
      case GAME_DOWN_KEYCODE:
         if (delta.iY)
            delta.iY = (iCurDelta.iY + iDeltaY > iMaxDelta.iY) ? iMaxDelta.iY - iCurDelta.iY : iDeltaY;
         break;
      }
   }
   if (delta.iX || delta.iY)
   {
      iCurDelta += delta;
      iGameWindow->SetGameOrigin(iGameWindow->GameOrigin() - delta);
   }
}

#endif
//===============================================
//
// CInfoMgr
//
//===============================================

CInfoMgr* CInfoMgr::NewL(CGameWindow& aGameWindow, CGameController& aGameController)
{
   CInfoMgr *self = new (ELeave) CInfoMgr(aGameWindow, aGameController);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CInfoMgr::CInfoMgr
( CGameWindow& aGameWindow
, CGameController& aGameController
) : iPosition(20, 0)
  , iSize(50, 12)
  , iGameWindow(aGameWindow)
  , iGameController(aGameController)
  , iFgColor(KRgbWhite)
  , iBgColor(KRgbBlack)
{
}

CInfoMgr::~CInfoMgr()
{
   if (iFont)
      iBitmap->FbsBitmapDevice()->ReleaseFont(iFont);
   if (iBitmap)
      delete iBitmap;
}

void CInfoMgr::ConstructL()
{
   TTimeIntervalMicroSeconds32   period;
   UserHal::TickPeriod(period);
   iPeriod   = period.Int();

   iBitmap = CFbsGameScreenDevice::NewL(iSize, iGameWindow.GameScreenDevice()->DisplayMode());

   TFontSpec fontSpec(KFontName, 20);
   User::LeaveIfError(iBitmap->FbsBitmapDevice()->GetNearestFontInPixels(iFont, fontSpec));

   CFbsBitGc*   gc = iBitmap->FbsBitGc();
   gc->UseFont(iFont);

   iGameController.AddGameControlInputHandlerL(*this);
}

void CInfoMgr::PostDrawFrameToScreen(CGameScreenDevice* aDstGameScreenDevice)
{
   TPoint p = iGameWindow.GameOrigin() + iPosition;
   if (p.iX < 0)
      p.iX = 0;
   if (p.iY < 0)
      p.iY = 0;

   DrawBitmap();
   aDstGameScreenDevice->DirectDraw(p, *iBitmap, TRect(iSize));
   GameTick();
}

void CInfoMgr::HandleKeyEvent(TInt aType, const TKeyEvent& aKeyEvent)
{
   if (aType == EEventKey)
   {
      if (aKeyEvent.iCode == '#')
      {
         if (!iIsActive)
         {
            ActivateL();
         }
         else
         {
            Deactivate();
         }
      }
   }
}

void CInfoMgr::DrawBitmap()
{
   TBuf<32>   text;

   CFbsBitGc*   gc = iBitmap->FbsBitGc();
   gc->SetBrushColor(iBgColor);
   gc->SetBrushStyle(CGraphicsContext::ENullBrush);
   gc->SetPenColor(iFgColor);
   gc->SetPenSize(TSize(1,1));
   gc->Clear();

   text.Format(_L("%d fps"), iVideoFps);
               
   // draw text
   TInt baselineOffset=(iSize.iHeight + iFont->AscentInPixels()) / 2;
   gc->DrawText(text, TRect(iSize), baselineOffset, CGraphicsContext::ECenter);
}

void CInfoMgr::ActivateL()
{
   iIsActive = ETrue;
   iVideoFps = 0;
   iFrames = 0;
   iGameWindow.AddGameScreenDrawObserverL(*this);
}

void CInfoMgr::Deactivate()
{
   iGameWindow.RemoveGameScreenDrawObserver(*this);
   iIsActive = EFalse;
}

void CInfoMgr::GameTick()
{
   iFrames++;
   if (iFrames > 50)
   {
      TInt ticks = User::TickCount();
      TInt diff = ticks - iPrevTicks;
      iPrevTicks = ticks;

      TInt time = diff * iPeriod;
      if (!time)
         time = 1;
      iVideoFps = (iFrames * KOneSecondInMicroSeconds) / time;
      iFrames = 0;
   }
}


