//===================================================================
//
// File: keybrd.cpp
//
// From EDoom, moved some video.cpp/h code here for the A920
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Peter van Sebille
//     - http://mobile.yipton.net
//     - Author of EMame for P800/P900 from which this port 
//       came from
//
//===================================================================
//
// Modifed By: Steve Fischer (steve@stevesprojects.com)
//
// (c) Copyright 2004-2005, Steve Fischer
// All Rights Reserved
//
//===================================================================
//
// Author:  Peter van Sebille (peter@yipton.net)
//
// (c) Copyright 2001, Peter van Sebille
// All Rights Reserved
//
//===================================================================

#include "options.h"

#include "keybrd.h"
#include "controller.h"

#ifdef USE_VIRTUAL_KEYBOARD

_LIT(KFontName,"Courier");

const TInt KVkRows           = 5;
const TInt KVkCols           = 12;

const TInt KKeyCodeSym       = 2;
const TInt KKeyCodeAlphaInc  = 4;
const TInt KKeyCodeAlphaDec  = 5;

const TInt KVkXMargin        = 1;
const TInt KVkYMargin        = 1;
const TInt KVkStandardWidth  = 16;
const TInt KVkStandardHeight = 16;

#define IS_VALID_VIRTUAL_KEY(vk) ((vk)->iWidth)

const TVirtualKey iDefaultVirtualKeyboardTable[KVkRows][KVkCols] = 
{
{   /*                                         SHIFT  
    width           scan    label  key    label  key
   code          code          code */
   {KVkStandardWidth,   2,     "1", '1',     "!", '!'},
   {KVkStandardWidth,   3,     "2", '2',     "@", '@'},
   {KVkStandardWidth,   4,     "3", '3',     "#", '#'},
   {KVkStandardWidth,   5,     "4", '4',     "$", '$'},
   {KVkStandardWidth,   6,     "5", '5',     "%", '%'},
   {KVkStandardWidth,   7,     "6", '6',     "^", '^'},
   {KVkStandardWidth,   8,     "7", '7',     "&", '&'},
   {KVkStandardWidth,   9,     "8", '8',     "*", '*'},
   {KVkStandardWidth,  10,     "9", '9',     "(", '('},
   {KVkStandardWidth,  11,     "0", '0',     ")", ')'},
   {KVkStandardWidth + (KVkStandardWidth/2),   EStdKeyBackspace,     "<-", EKeyBackspace,     "<-", EKeyBackspace}
},
{
   {KVkStandardWidth + (KVkStandardWidth/2),  EStdKeyTab,     "tab", EKeyTab,     "TAB", EKeyTab},
   {KVkStandardWidth,  16,     "q", 'q',     "Q", 'Q'},
   {KVkStandardWidth,  17,     "w", 'w',     "W", 'W'},
   {KVkStandardWidth,  18,     "e", 'e',     "E", 'E'},
   {KVkStandardWidth,  19,     "r", 'r',     "R", 'R'},
   {KVkStandardWidth,  20,     "t", 't',     "T", 'T'},
   {KVkStandardWidth,  21,     "y", 'y',     "Y", 'Y'},
   {KVkStandardWidth,  22,     "u", 'u',     "U", 'U'},
   {KVkStandardWidth,  23,     "i", 'i',     "I", 'I'},
   {KVkStandardWidth,  24,     "o", 'o',     "O", 'O'},
   {KVkStandardWidth,  25,     "p", 'p',     "P", 'P'}
},
{
   {KVkStandardWidth + (KVkStandardWidth/2),  EStdKeyCapsLock,     "cps", EKeyCapsLock,     "CPS", EKeyCapsLock},
   {KVkStandardWidth,  30,     "a", 'a',     "A", 'A'},
   {KVkStandardWidth,  31,     "s", 's',     "S", 'S'},
   {KVkStandardWidth,  32,     "d", 'd',     "D", 'D'},
   {KVkStandardWidth,  33,     "f", 'f',     "F", 'F'},
   {KVkStandardWidth,  34,     "g", 'g',     "G", 'G'},
   {KVkStandardWidth,  35,     "h", 'h',     "H", 'H'},
   {KVkStandardWidth,  36,     "j", 'j',     "J", 'J'},
   {KVkStandardWidth,  37,     "k", 'k',     "K", 'K'},
   {KVkStandardWidth,  38,     "l", 'l',     "L", 'L'},
   {KVkStandardWidth,  59,     "@", KKeyCodeSym,     "@", KKeyCodeSym}      // 59 == scan code F1
},
{
   {KVkStandardWidth + (KVkStandardWidth/2),  EStdKeyLeftShift,     "sft", EKeyLeftShift,     "SFT", EKeyLeftShift},
   {KVkStandardWidth,  44,     "z", 'z',     "Z", 'Z'},
   {KVkStandardWidth,  45,     "x", 'x',     "X", 'X'},
   {KVkStandardWidth,  46,     "c", 'c',     "C", 'C'},
   {KVkStandardWidth,  47,     "v", 'v',     "V", 'V'},
   {KVkStandardWidth,  48,     "b", 'b',     "B", 'B'},
   {KVkStandardWidth,  49,     "n", 'n',     "N", 'N'},
   {KVkStandardWidth,  50,     "m", 'm',     "M", 'M'},
   {(KVkStandardWidth * 3) + (KVkXMargin*2), DEVICE_SCANCODE_SELECT,     "enter", DEVICE_KEYCODE_SELECT,     "ENTER", DEVICE_KEYCODE_SELECT},
   {0, 0,     NULL, 0,     NULL, 0},
   {0, 0,     NULL, 0,     NULL, 0}
},
{
   {KVkStandardWidth + (KVkStandardWidth/2),  DEVICE_SCANCODE_ESC,     "esc", DEVICE_KEYCODE_ESC,     "ESC", DEVICE_KEYCODE_ESC},
   {(KVkStandardWidth * 3) + (KVkXMargin*2), 60,     "---", KKeyCodeAlphaDec,     "---", KKeyCodeAlphaDec},
   {(KVkStandardWidth * 3) + (KVkXMargin*2), EStdKeySpace,     "space", ' ',     "space", ' '},
   {(KVkStandardWidth * 3) + (KVkXMargin*2), 61,     "+++", KKeyCodeAlphaInc,     "+++", KKeyCodeAlphaInc},
   {0, 0,     NULL, 0,     NULL, 0},
   {0, 0,     NULL, 0,     NULL, 0},
   {0, 0,     NULL, 0,     NULL, 0},
   {0, 0,     NULL, 0,     NULL, 0},
   {0, 0,     NULL, 0,     NULL, 0},
   {0, 0,     NULL, 0,     NULL, 0},
   {0, 0,     NULL, 0,     NULL, 0}
}
};


CVirtualKeyboard::CVirtualKeyboard
( CGameWindow& aGameWindow
 , CGameController& aGameController
 )
 : iRect(TSize(200, 100))
 , iGameWindow(aGameWindow)
 , iGameController(aGameController)
 , iActiveRect(TSize(-1, -1))
{
   iBgColor =KRgbWhite;
   iFgColor =KRgbRed;
}

CVirtualKeyboard::~CVirtualKeyboard()
{
   if (iFont)
      iBitmap->FbsBitmapDevice()->ReleaseFont(iFont);
   if (iBitmap)
      delete iBitmap;
}

CVirtualKeyboard* CVirtualKeyboard::NewL(CGameWindow& aGameWindow, CGameController& aGameController)
{
   CVirtualKeyboard*   self = new(ELeave)CVirtualKeyboard(aGameWindow, aGameController);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

void CVirtualKeyboard::ConstructL()
{
   iBitmap = CFbsGameScreenDevice::NewL(iRect.Size(), iGameWindow.GameScreenDevice()->DisplayMode());
   
   TFontSpec fontSpec(KFontName, 20);
   User::LeaveIfError(iBitmap->FbsBitmapDevice()->GetNearestFontInPixels(iFont, fontSpec));
   
   CFbsBitGc *gc = iBitmap->FbsBitGc();
   gc->UseFont(iFont);
   
   // sothat we can detect our hotkey
   iGameController.AddGameControlInputHandlerL(*this);
   iGameWindow.AddScreenOrientationObserverL(*this);
   iGameController.AddGameControlRawInputHandlerL(*this);
}

void CVirtualKeyboard::SetPosition(const TPoint& aPosition)
{
   TSize  curSize = iRect.Size();
   TSize  screenSize = iGameWindow.GameScreenDevice()->Size();
   TPoint newPos = aPosition;
   if (newPos.iX < 0)
      newPos.iX = 0;
   if (newPos.iX + curSize.iWidth > screenSize.iWidth)
      newPos.iX = screenSize.iWidth - curSize.iWidth;
   if (newPos.iY < 0)
      newPos.iY = 0;
   if (newPos.iY + curSize.iHeight> screenSize.iHeight)
      newPos.iY = screenSize.iHeight - curSize.iHeight;
   
   iRect = TRect(newPos, curSize);
}

void CVirtualKeyboard::ScreenOrientationChanged(TOrientation aOrientation)
{
   iOrientation = aOrientation;
   
   DrawKeyboard();
}

void CVirtualKeyboard::CheckForVirtualKeyboardHotKey(TInt aKeyCode)
{
   if (IsHotKeyForVirtualKeyboard(aKeyCode))
   {
      if (!iIsActive)
         ActivateL();
      else
         Deactivate();
   }
}

void CVirtualKeyboard::ActivateL()
{
   iIsActive = ETrue;
   iAlphaInc = 0;
   iAlpha = 175;
   iKey = NULL;
   
   iGameWindow.AddGameScreenDrawObserverL(*this);
   
   DrawKeyboard();
}

void CVirtualKeyboard::Deactivate()
{
   iIsActive = EFalse;
   
   // no longer need to steal pointer events
   iGameWindow.RemoveGameScreenDrawObserver(*this);
   iKey = NULL;
   ClearKeyboard();
}

void CVirtualKeyboard::PostDrawFrameToScreen(CGameScreenDevice* aDstGameScreenDevice)
{
   if (iAlphaInc)
   {
      iAlpha += iAlphaInc;
      if (iAlpha > 255)
         iAlpha = 255;
      if (iAlpha < 0)
         iAlpha = 0;
   }
   
   aDstGameScreenDevice->AlphaBlend(iRect.iTl, *iBitmap, TRect(iBitmap->Size()), (TUint8) iAlpha);
}

TBool CVirtualKeyboard::WantDrawToScreen()
{
   return iIsActive;
}

TBool CVirtualKeyboard::ConsumeRawPointerEvent(const TPointerEvent& aPointerEvent)
{
   TPoint   p = aPointerEvent.iPosition;
   TBool   consumed = EFalse;
   if (iIsActive && iRect.Contains(p))
   {
      consumed = ETrue;
      if (aPointerEvent.iType == TPointerEvent::EButton1Down)
      {
         const TVirtualKey* virtualKey = FindVirtualKey(p - iRect.iTl);
         if (virtualKey)
            HandleVirtualKeyDown(virtualKey);
      }
      else if (aPointerEvent.iType == TPointerEvent::EButton1Up)
      {
         HandleVirtualKeyUp();
      }
   }
   
   return consumed;
}

TInt CVirtualKeyboard::KeyCode(const TVirtualKey* aVirtualKey)
{
   TBool shift = iCapsMode ^ iShiftMode;
   return shift? aVirtualKey->iShiftKeyCode: aVirtualKey->iKeyCode;
}

TInt CVirtualKeyboard::ScanCode(const TVirtualKey* aVirtualKey)
{
   return iKeyCodesAreScanCodes ? aVirtualKey->iKeyCode : aVirtualKey->iScanCode;
}

void CVirtualKeyboard::HandleVirtualKeyDown(const TVirtualKey* aVirtualKey)
{
   TBool generateKeyEvent = !((aVirtualKey->iKeyCode == EKeyLeftShift) 
      || (aVirtualKey->iKeyCode == EKeyCapsLock) || 
      (aVirtualKey->iKeyCode == KKeyCodeSym));
   if (generateKeyEvent)
   {
      iGameController.InjectKeyEvent(EEventKeyDown, ScanCode(aVirtualKey));
      iGameController.InjectKeyEvent(EEventKey, KeyCode(aVirtualKey));
   }
   iKey = aVirtualKey;
   DrawKeyboard();
   
   if (aVirtualKey->iKeyCode == KKeyCodeAlphaInc)
      iAlphaInc = 1;
   else if (aVirtualKey->iKeyCode == KKeyCodeAlphaDec)
      iAlphaInc = -1;
}

void CVirtualKeyboard::HandleVirtualKeyUp()
{
   if (iKey)
   {
      if (iKey->iKeyCode == EKeyLeftShift)
         iShiftMode = !iShiftMode;
      else if (iKey->iKeyCode == EKeyCapsLock)
         iCapsMode = !iCapsMode;
      else
      {
         iGameController.InjectKeyEvent(EEventKeyUp, ScanCode(iKey));
         iKey = NULL;
         if (iShiftMode)
            iShiftMode = EFalse;
      }
      DrawKeyboard();
   }
   iAlphaInc = 0;
}

void CVirtualKeyboard::HandleKeyEvent(TInt aType, const TKeyEvent& aKeyEvent)
{
   if (aType == EEventKey)
      CheckForVirtualKeyboardHotKey(aKeyEvent.iCode);
}

void CVirtualKeyboard::HandlePointerEvent(const TPointerEvent& aPointerEvent)
{
   if ((aPointerEvent.iType == TPointerEvent::EButton1Down) && iActiveRect.Contains(aPointerEvent.iPosition))
   {
      if (!iIsActive)
         ActivateL();
      else
         Deactivate();
   }
}

const TVirtualKey* CVirtualKeyboard::FindVirtualKey(const TPoint& aPoint)
{
   TInt   row = aPoint.iY / (KVkStandardHeight + KVkYMargin);
   TPoint   p(KVkXMargin, row * (KVkStandardHeight + KVkYMargin));
   if (row>=0 && row<KVkRows)
   {
      for (TInt col = 0; col<KVkCols ; col++)
      {
         const TVirtualKey& vk = iDefaultVirtualKeyboardTable[row][col];
         if (IS_VALID_VIRTUAL_KEY(&vk))
         {
            TRect   rect(p, p + TPoint (vk.iWidth, KVkStandardHeight));
            if (rect.Contains(aPoint))
               return &vk;
         }
         p.iX += KVkXMargin + vk.iWidth;
      }
   }
   return NULL;
}

void CVirtualKeyboard::ClearKeyboard()
{
   CFbsBitGc*   gc = iBitmap->FbsBitGc();
   gc->SetBrushColor(KRgbBlack);
   gc->SetBrushStyle(CGraphicsContext::ENullBrush);
   gc->Clear();
}

CFbsBitGc::TGraphicsOrientation ToGraphicsOrientation(TOrientation aOrientation)
{
   static const CFbsBitGc::TGraphicsOrientation orientatationList[] = 
   { CFbsBitGc::EGraphicsOrientationNormal
   , CFbsBitGc::EGraphicsOrientationRotated90
   , CFbsBitGc::EGraphicsOrientationRotated270
   , CFbsBitGc::EGraphicsOrientationRotated180
   };
   return orientatationList[aOrientation];
}

void CVirtualKeyboard::DrawKeyboard()
{
   CFbsBitGc*   gc = iBitmap->FbsBitGc();
   gc->SetBrushColor(iBgColor);
   gc->SetBrushStyle(CGraphicsContext::ENullBrush);
   gc->Clear();
   gc->SetPenColor(iFgColor);
   gc->SetPenSize(TSize(1,1));
   gc->UseFont(iFont);
   
   TBool shift = iCapsMode ^ iShiftMode;
   TPoint   p(KVkYMargin, KVkXMargin);
   for (TInt row = 0; row<KVkRows ; row++)
   {
      p.iX = KVkXMargin;
      for (TInt col = 0; col<KVkCols ; col++)
      {
         const TVirtualKey& vk = iDefaultVirtualKeyboardTable[row][col];
         if (IS_VALID_VIRTUAL_KEY(&vk))
         {
            TRect   rect(p, p + TPoint (vk.iWidth, KVkStandardHeight));
            if (vk.iText)
            {
               TBool isReversed = (  ((vk.iKeyCode == EKeyLeftShift) && (iShiftMode))
                                  || ((vk.iKeyCode == EKeyCapsLock) && (iCapsMode))
                                  );
               if (iKey == &vk)
                  isReversed = ETrue;
               
               if (isReversed)
               {
                  gc->SetBrushColor(iFgColor);
                  gc->SetPenColor(iBgColor);
                  gc->Clear(rect);
               }
               
               // draw border
               gc->DrawRect(rect);
               
               // draw text
               TPtrC8      text8((TUint8*) (shift ? vk.iShiftText : vk.iText));
               TBuf<32>   text;
               text.Copy(text8);
               TInt baselineOffset=(rect.Height()+iFont->AscentInPixels())/2; 
               gc->DrawText(text, rect, baselineOffset, CGraphicsContext::ECenter);
               
               if (isReversed)
               {
                  gc->SetBrushColor(iBgColor);
                  gc->SetPenColor(iFgColor);
               }
            }
         }
         p.iX += KVkXMargin + vk.iWidth;
      }
      p.iY += KVkYMargin + KVkStandardHeight;
   }
   gc->SetOrientation(ToGraphicsOrientation(iOrientation));
}

#endif // USE_VIRTUAL_KEYBOARD



