//===================================================================
//
// File:  controller.cpp
//
// From EDoom, moved some video.cpp/h code here for the A920
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Peter van Sebille
//     - http://mobile.yipton.net
//     - Author of EMame for P800/P900 from which this port 
//       came from
//
//===================================================================
//
// Modifed By: Steve Fischer (steve@stevesprojects.com)
//
// (c) Copyright 2005, Steve Fischer
// All Rights Reserved
//
//===================================================================
//
// Author:  Peter van Sebille (peter@yipton.net)
//
// (c) Copyright 2001, Peter van Sebille
// All Rights Reserved
//
//===================================================================

#include "options.h"
#include <hal.h>

#include "controller.h"
#include "navigation.h"

//===============================================
//
// CGameController
//
//===============================================

// Default no translation tables
static TGameKeyCodeTable  sGameKeyCodeTableNone[]  = { {0, 0} };  // No translation
static TGameScanCodeTable sGameScanCodeTableNone[] = { {0, 0} };  // No translation

CGameController* CGameController::NewL(RWsSession& aWsSession)
{
   CGameController *self = new(ELeave)CGameController(aWsSession);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CGameController::~CGameController()
{
   delete iGameControlInputHandlers;
   delete iGameControlRawInputHandlers;
}

void CGameController::ConstructL()
{
   iGameControlInputHandlers    = new (ELeave) CArrayPtrFlat<MGameControlInputHandler>(5);
   iGameControlRawInputHandlers = new (ELeave) CArrayPtrFlat<MGameControlRawInputHandler>(5);

   SetGameKeymapping(sGameScanCodeTableNone[0], sGameKeyCodeTableNone[0]);

   StartWServEvents();
}

void CGameController::AddGameControlRawInputHandlerL(MGameControlRawInputHandler& aControlImputConsumser)
{
   iGameControlRawInputHandlers->AppendL(&aControlImputConsumser);
}

void CGameController::AddGameControlInputHandlerL(MGameControlInputHandler& aGameControlObserver)
{
   iGameControlInputHandlers->AppendL(&aGameControlObserver);
}

void CGameController::RemoveGameControlRawInputHandler(MGameControlRawInputHandler& aControlImputConsumser)
{
   RemoveFromPointerArrayList<MGameControlRawInputHandler>(*iGameControlRawInputHandlers, &aControlImputConsumser);
}

void CGameController::RemoveGameControlInputHandler(MGameControlInputHandler& aGameControlObserver)
{
   RemoveFromPointerArrayList<MGameControlInputHandler>(*iGameControlInputHandlers, &aGameControlObserver);
}

void CGameController::SetGameKeymapping(TGameScanCodeTable& aGameScanCodeTable, TGameKeyCodeTable& aGameKeyCodeTable)
{
   iGameScanCodeTable = &aGameScanCodeTable;
   iGameKeyCodeTable  = &aGameKeyCodeTable;
}

int CGameController::IsKeyPressed( unsigned short aScanCode )
{
   int pressed = EFalse;

   for ( TInt i = 0; i < KMaxKeysPressed; i++ )
   {
      if ( iKeysPressed[i].iScanCode == aScanCode )
      {
         pressed = ETrue;
         break;
      }
   }
   
   return ( pressed );
}

void CGameController::ResetKeys()
{
   for ( TInt i = 0; i < KMaxKeysPressed; i++ )
   {
      if ( iKeysPressed[i].iScanCode != 0 )
      {
         HandleKeyEvent( EEventKeyUp, iKeysPressed[i] );
         iKeysPressed[i].iScanCode = 0;
         break;
      }
   }  
}

void CGameController::StartWServEvents()
{
   iWsEventStatus = KRequestPending;
   iWsSession.EventReady(&iWsEventStatus);

   iRedrawEventStatus = KRequestPending;
   iWsSession.RedrawReady(&iRedrawEventStatus);
}

void CGameController::CancelWServEvents()
{
   if (iWsEventStatus != KRequestPending)
      iWsSession.EventReadyCancel();

   if (iRedrawEventStatus != KRequestPending)
      iWsSession.RedrawReadyCancel();
}

void CGameController::PollWServEvents()
{
   while (iWsEventStatus != KRequestPending)
   {
      iWsSession.GetEvent(iWsEvent);
      HandleWsEvent(iWsEvent);
      iWsEventStatus = KRequestPending;
      iWsSession.EventReady(&iWsEventStatus);
   }
}

void CGameController::HandleWsEvent(const TWsEvent& aWsEvent)
{
   if (  (aWsEvent.Type() == EEventKeyDown) 
      || (aWsEvent.Type() == EEventKey)
      || (aWsEvent.Type() == EEventKeyUp)
      )
   {
      HandleKeyEvent( aWsEvent.Type(), *aWsEvent.Key() );
   } 
   else if (aWsEvent.Type() == EEventPointer)
   {
      HandlePointerEvent( *aWsEvent.Pointer() );
   }
}

void CGameController::ScreenOrientationChanged(TOrientation aOrientation)
{
   iOrientation = aOrientation;
}

void CGameController::PostDrawFrameToScreen(CGameScreenDevice* /*aDstGameScreenDevice*/)
{
   PollWServEvents();
}

// Raw pointer events
void CGameController::HandlePointerEvent(const TPointerEvent& aPointerEvent)
{
   TBool consumed = EFalse;
   TInt  count    = iGameControlRawInputHandlers->Count();
   TInt  i = 0;

   for (i = count - 1; i >= 0 && !consumed; i-- )
   {
      consumed = (*iGameControlRawInputHandlers)[i]->ConsumeRawPointerEvent(aPointerEvent);
   }

   if (!consumed)
   {
      count = iGameControlInputHandlers->Count();
      for ( i = 0; i < count; i++ )
      {
         (*iGameControlInputHandlers)[i]->HandlePointerEvent(aPointerEvent);
      }
   }
}

void CGameController::MapToGameEvents(TInt aType, TKeyEvent& aKeyEvent)
{
   if ((aType == EEventKeyDown) || (aType == EEventKeyUp))
   {
      TGameScanCodeTable *gameScanCodeEntry = iGameScanCodeTable;
      while (gameScanCodeEntry->iGameScanCode)
      {
         if (gameScanCodeEntry->iPhysicalScanCode == aKeyEvent.iScanCode)
         {
            aKeyEvent.iScanCode = gameScanCodeEntry->iGameScanCode;
            break;
         }
         gameScanCodeEntry++;
      }
   }
   else if ( aType == (TInt) EEventKey )
   {
      TGameKeyCodeTable *gameKeyCodeEntry = iGameKeyCodeTable;
      while (gameKeyCodeEntry->iGameKey)
      {
         if (gameKeyCodeEntry->iPhysicalKey == (TInt) aKeyEvent.iCode)
         {
            aKeyEvent.iCode = gameKeyCodeEntry->iGameKey;
            break;
         }
         gameKeyCodeEntry++;
      }
   }
}

void CGameController::HandleKeyEvent(TInt aType, const TKeyEvent& aKeyEvent)
{
   TKeyEvent keyEvent = aKeyEvent;

   MapToGameEvents(aType, keyEvent);

   TBool consumed = EFalse;
   TInt  count = iGameControlRawInputHandlers->Count();
   TInt  i = 0;

   for ( i = count - 1; i >= 0 && !consumed; i-- )
   {
      consumed = (*iGameControlRawInputHandlers)[i]->ConsumeRawKeyEvent(aType, keyEvent);
   }

   if (!consumed)
   {
      StoreKeyEvent( aType, keyEvent );

      // Call all input handlers
      count = iGameControlInputHandlers->Count();
      for (i=0 ; i<count ; i++)
      {
         (*iGameControlInputHandlers)[i]->HandleKeyEvent(aType, keyEvent);
      }
   }
}

void CGameController::InjectKeyEvent(TInt aEventType, TInt aKeyOrScanKey, TInt aRepeat)
{
   TKeyEvent   keyEvent;

   keyEvent.iCode      = (aEventType == EEventKey) ? aKeyOrScanKey: 0;
   keyEvent.iScanCode  = (aEventType != EEventKey) ? aKeyOrScanKey: 0;
   keyEvent.iModifiers = 0;
   keyEvent.iRepeats   = aRepeat;

   HandleKeyEvent(aEventType, keyEvent);
}

void CGameController::StoreKeyEvent(TInt aType, const TKeyEvent& aKeyEvent)
{
   // Track pressed keys so we can release them if focus is lost
   if ( aType == EEventKeyDown )
   {
      for ( TInt i = 0; i < KMaxKeysPressed; i++ )
      {
         if ( iKeysPressed[i].iScanCode == 0 )
         {
            iKeysPressed[i].iCode     = aKeyEvent.iCode;
            iKeysPressed[i].iScanCode = aKeyEvent.iScanCode;
            break;
         }
      }
   }
   else if ( aType == EEventKeyUp )
   {
      for ( TInt i = 0; i < KMaxKeysPressed; i++ )
      {
         if ( iKeysPressed[i].iScanCode == aKeyEvent.iScanCode )
         {
            iKeysPressed[i].iScanCode = 0;
            break;
         }
      }
   }
}

//===============================================
//
// CSmartPenInputHandler
//
//===============================================

CSmartPenInputHandler::CSmartPenInputHandler
( CGameWindow& aGameWindow
, CGameController& aGameController
) : iAutoX(ETrue)
  , iAutoY(ETrue)
  , iFramesRepeatThreshHold(1)
  , iFramesMoveThreshHold(1)
  , iGameController(aGameController)
  , iGameWindow(aGameWindow)
{
}

CSmartPenInputHandler::~CSmartPenInputHandler()
{
}

void CSmartPenInputHandler::ConstructL()
{
   iGameController.AddGameControlInputHandlerL(*this);
   iGameWindow.AddGameScreenDrawObserverL(*this);
   iGameWindow.AddScreenOrientationObserverL(*this);
}


void CSmartPenInputHandler::HandlePointerEvent(const TPointerEvent& aPointerEvent)
{
   if (aPointerEvent.iType == TPointerEvent::EButton1Down)
      PenDownEvent(aPointerEvent.iPosition);
   else if (aPointerEvent.iType == TPointerEvent::EButton1Up)
      PenUpEvent(aPointerEvent.iPosition);
   else if (aPointerEvent.iType == TPointerEvent::EDrag)
      PenMoveEvent(aPointerEvent.iPosition);
}

void CSmartPenInputHandler::PostDrawFrameToScreen(CGameScreenDevice* /*aDstGameScreenDevice*/)
{
   if (iPenDown)
   {
         /*
          * The first threashold is low so that we start moving relatively quick.
          */
      if (++iFrameCount > iFramesThreshHold)
      {
         iFrameCount = 0;
         SimulateAutoMove();
         if (iEventCount == 2)
         {
            iFramesThreshHold = iFramesRepeatThreshHold;
         }
      }
   }
   else if (iEventCount)
   {
         /*
          * Generate the up event for a fast pen stroke. For such a stroke
          * we have generated the key down event on the pen-up. We have waited
          * a couple of frames to generate the key-up event.
          */
      if (++iFrameCount > iFramesThreshHold)
      {
         iFrameCount = 0;
         PenUpEvent(iLastPenPosition);
      }
   }
}

void CSmartPenInputHandler::SimulateAutoMove()
{
#ifdef PROJECT_EDOOM
   if (1)
#else
   if (iLastDelta.iX || iLastDelta.iY)
#endif
   {
      if (iEventCount == 0)
         HandleDownEvent();
      HandleMoveEvent(iLastDelta.iX, iLastDelta.iY);
      iEventCount++;
      if (!iAutoY)
         iLastDelta.iY = 0;
      if (!iAutoX)
         iLastDelta.iX = 0;
   }
}

void CSmartPenInputHandler::PenDownEvent(const TPoint& aPosition)
{
      /*
       * if iEventCount !=0 then we got a new down event when were handling
       * a special fast short moved pen stroke. For these strokes we will
       * wait a couple of frames after the pen-up event.
       */
   if (iEventCount)
      PenUpEvent(iLastPenPosition);

   iDownPenPosition = iLastPenPosition = aPosition;
   iLastDelta = TPoint(0,0);
   iPenDown = ETrue;

   iFramesThreshHold = iFramesMoveThreshHold;
   iEventCount = iFrameCount = 0;

   HandleDownEvent();
}

void CSmartPenInputHandler::PenUpEvent(const TPoint& aPosition)
{
   if (!iEventCount)
   {
         /*
          * If we have not generated any events during the pen move
          * then this maybe a very quick pen stroke. To allow for high
          * precision movements we will consider all delta's to be a valid
          * move (EFalse in PenMoveEvent). We set iFramesThreshHold so that
          * we can generate the according up event a couple of frames later.
          * Doom will ignore handle key-down and key-up events that happen 
          * too short after one another.
          */
      if (!iLastDelta.iX && !iLastDelta.iY)
         PenMoveEvent(aPosition, EFalse);
      HandleMoveEvent(iLastDelta.iX, iLastDelta.iY);
      iFramesThreshHold = iFramesMoveThreshHold;
   }
   else
   {
      HandleUpEvent();
      iFramesThreshHold = iEventCount = iLastDelta.iX = iLastDelta.iY = 0;
   }
   iPenDown = EFalse;
}

TPoint CSmartPenInputHandler::TransformDeltaForOrientation(const TPoint& aPoint)
{
   switch (iOrientation)
   {
   case EOrientationNormal:
      break;
      case EOrientationRotLeft:
         return TPoint(aPoint.iY, aPoint.iX * -1);
      case EOrientationRotRight:
         return TPoint(aPoint.iY * -1, aPoint.iX);
      case EOrientationRotUpsideDown:
         return TPoint(aPoint.iX * -1, aPoint.iY * -1);
   }
   return aPoint;
}

void CSmartPenInputHandler::PenMoveEvent(const TPoint& aPosition, TBool aStrict)
{
   TPoint   deltaRel = aPosition - iLastPenPosition;

   if (aStrict)
   {
         /*
          * Filter out jitter
          */
      if ((deltaRel.iX < 10) && (deltaRel.iX > -10))
         deltaRel.iX = 0;
      if ((deltaRel.iY < 10) && (deltaRel.iY > -10))
         deltaRel.iY = 0;
   }

   if (deltaRel.iX || deltaRel.iY)
   {
      deltaRel = TransformDeltaForOrientation(deltaRel);
      iLastPenPosition = aPosition;
      iLastDelta = deltaRel;
   }
}

//===============================================
//
// CSmartPenToKeyInputHandler
//
//===============================================

CSmartPenToKeyInputHandler* CSmartPenToKeyInputHandler::NewL(CGameWindow& aGameWindow, CGameController& aGameController)
{
   CSmartPenToKeyInputHandler *self = new (ELeave) CSmartPenToKeyInputHandler(aGameWindow, aGameController);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CSmartPenToKeyInputHandler::CSmartPenToKeyInputHandler
( CGameWindow& aGameWindow
, CGameController& aGameController
) : CSmartPenInputHandler(aGameWindow, aGameController)
{
}

void CSmartPenToKeyInputHandler::ConstructL()
{
   CSmartPenInputHandler::ConstructL();
}

void CSmartPenToKeyInputHandler::HandleDownEvent()
{
   iHorMove = iVertMove = 0;
   iIsMovingVert = EFalse;
   iLastPenPosition1 = TPoint(0,0);
}

void CSmartPenToKeyInputHandler::HandleHorMoveEvent(TInt aNewHorMove)
{
   if (aNewHorMove != iHorMove)
   {
      if (iHorMove)
      {
         if (iHorMove==-1)
         {
            InjectKeyEvent(EEventKeyUp, DEVICE_SCANCODE_LEFT);
            iLeftCount = 0;
         }
         else
         {
            InjectKeyEvent(EEventKeyUp, DEVICE_SCANCODE_RIGHT);
            iRightCount = 0;
         }
      }

      if (aNewHorMove < 0)
      {
         InjectKeyEvent(EEventKeyDown, DEVICE_SCANCODE_LEFT);
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_LEFT);
      } 
      else if (aNewHorMove>0)
      {
         InjectKeyEvent(EEventKeyDown, DEVICE_SCANCODE_RIGHT);
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_RIGHT);
      }
   }
   else
   {
      if (aNewHorMove < 0)
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_LEFT, ++iLeftCount);
      else if (aNewHorMove>0)
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_RIGHT, ++iRightCount);
   }

   iHorMove = aNewHorMove;
}

void CSmartPenToKeyInputHandler::HandleVertMoveEvent(TInt aNewVertMove)
{
   if (aNewVertMove != iVertMove)
   {
      if (iVertMove)
      {
         if (iVertMove==-1)
         {
            InjectKeyEvent(EEventKeyUp, DEVICE_SCANCODE_UP);
            iUpCount = 0;
         }
         else
         {
            InjectKeyEvent(EEventKeyUp, DEVICE_SCANCODE_DOWN);
            iDownCount = 0;
         }
      }

      if (aNewVertMove < 0)
      {
         InjectKeyEvent(EEventKeyDown, DEVICE_SCANCODE_UP);
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_UP);
      } 
      else if (aNewVertMove>0)
      {
         InjectKeyEvent(EEventKeyDown, DEVICE_SCANCODE_DOWN);
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_DOWN);
      }
   }
   else
   {
      if (aNewVertMove < 0)
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_UP, ++iUpCount);
      else if (aNewVertMove>0)
         InjectKeyEvent(EEventKey, DEVICE_SCANCODE_DOWN, ++iDownCount);
   }

   iVertMove = aNewVertMove;
}

void CSmartPenToKeyInputHandler::HandleMoveEvent(TInt aX, TInt aY)
{
#ifndef __WINS__
   TInt absX = abs(aX);
   TInt absY = abs(aY);
#else
   TInt absX = 0;
   TInt absY = 0;
#endif

   if (!iCanDo_X_And_Y_AtTheSameTime)
   {
         /*
          * If we want to move in one direction only, ignore either aX or aY
          */
      if (absX > absY)
         aY = 0;
      else 
         aX = 0;
   }

   TInt newHorMove = (aX < 0)? -1 : (aX>0) ? 1 : 0;
   TInt newVertMove = (aY < 0)? -1 : (aY>0) ? 1 : 0;

   if (iEnableAutoForward)
   {
//      TPoint deltaSinceLastVertMove = iLastPenPosition1 - iLastPenPosition;
//      deltaSinceLastVertMove = this->TransformDeltaForOrientation(deltaSinceLastVertMove);
//      if (newVertMove!=0)
//         iLastPenPosition1 = iLastPenPosition;

      iIsMovingVert = (iVertMove != 0);
//      iAutoX = iIsMovingVert ? EFalse : ETrue;
      if (iIsMovingVert && (newVertMove == 0))
         newVertMove = iVertMove;

      if (iHorMove && newHorMove && (iHorMove != newHorMove))
      {
         newHorMove = 0;
         this->iLastDelta.iX = 0;
      }
//      if (newHorMove)
//         InjectKeyEvent(EEventKey, (newHorMove > 0) ? 1 : 2);
   }

   if (absX > absY)
   {
      HandleHorMoveEvent(newHorMove);
      HandleVertMoveEvent(newVertMove);
   }
   else
   {
      HandleVertMoveEvent(newVertMove);
      HandleHorMoveEvent(newHorMove);
   }
}

void CSmartPenToKeyInputHandler::HandleUpEvent()
{
   if (iHorMove)
   {
      InjectKeyEvent(EEventKeyUp, (iHorMove==-1)? DEVICE_SCANCODE_LEFT : DEVICE_SCANCODE_RIGHT );
   }
   if (iVertMove)
   {
      InjectKeyEvent(EEventKeyUp, (iVertMove==-1)? DEVICE_SCANCODE_UP : DEVICE_SCANCODE_DOWN );
   }
   iLeftCount = iRightCount = iUpCount = iDownCount = 0;
}

void CSmartPenToKeyInputHandler::InjectKeyEvent(TInt aEventType, TInt aKeyOrScanKey, TInt aRepeat)
{
   iGameController.InjectKeyEvent(aEventType, aKeyOrScanKey, aRepeat);
   iEventCount++;
}

