//===================================================================
//
// File: buttons.cpp
//
// From EDoom, moved some video.cpp/h code here for the A920
//
// Credits:
//
//   Steve Fischer (2005)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Peter van Sebille
//     - http://mobile.yipton.net
//     - Author of EMame for P800/P900 from which this port 
//       came from
//
//===================================================================
//
// Modifed By: Steve Fischer (steve@stevesprojects.com)
//
// (c) Copyright 2005, Steve Fischer
// All Rights Reserved
//
//===================================================================
//
// Author:  Peter van Sebille (peter@yipton.net)
//
// (c) Copyright 2001, Peter van Sebille
// All Rights Reserved
//
//===================================================================

#include "options.h"

#include "buttons.h"
#include "controller.h"

#ifdef USE_VIRTUAL_BUTTONS

//===================================================================
// Local Data
//===================================================================

CFbsBitGc::TGraphicsOrientation ToGraphicsOrientation(TOrientation aOrientation)
{
   static const CFbsBitGc::TGraphicsOrientation orientatationList[] = 
   { CFbsBitGc::EGraphicsOrientationNormal
   , CFbsBitGc::EGraphicsOrientationRotated90
   , CFbsBitGc::EGraphicsOrientationRotated270
   , CFbsBitGc::EGraphicsOrientationRotated180
   };

   return ( orientatationList[aOrientation] );
}

//===================================================================
// Construction/Destruction Methods
//===================================================================

CVirtualButtons* CVirtualButtons::NewL
( CGameWindow          &aGameWindow
, CGameController      &aGameController
, const TVirtualButton *aVirtualButtons
)
{
   CVirtualButtons *self = new (ELeave) CVirtualButtons(aGameWindow, aGameController, aVirtualButtons);
   CleanupStack::PushL(self);
   self->ConstructL();
   CleanupStack::Pop();
   return self;
}

CVirtualButtons::CVirtualButtons
( CGameWindow          &aGameWindow
, CGameController      &aGameController
, const TVirtualButton *aVirtualButtons
) : iGameWindow(aGameWindow)
  , iGameController(aGameController)
{
   iVirtualButtons = aVirtualButtons;
}

CVirtualButtons::~CVirtualButtons()
{
   if (iBitmap)
   {
      delete iBitmap;
   }

   if ( iButtonBitmaps )
   {
      while ( iButtonBitmaps->Count() )
      {
         delete (*iButtonBitmaps)[0];
         iButtonBitmaps->Delete(0);
      }

      delete iButtonBitmaps;
   }
}

void CVirtualButtons::ConstructL()
{
   DetermineButtonRectL();

   iBitmap = CFbsGameScreenDevice::NewL( iRect.Size(), iGameWindow.GameScreenDevice()->DisplayMode());

   iButtonBitmaps = new (ELeave) CArrayPtrFlat<CFbsBitmap>(5);

   // Get the bitmaps
   CFbsBitmap *image;
   TInt        err;
   RProcess    process;
   TFileName   appName = process.FileName();
   TParsePtrC  parse(appName);
   TFileName   name(parse.DriveAndPath());
   name.Append( _L("nes_buttons.mbm") );
   
   TInt done = EFalse;
   TInt i = 0;
   while ( !done )
   {
      // Read in and store the bitmap
      image = new (ELeave) CFbsBitmap();
      err = image->Load(name, i);
      if (err != KErrNone)
      {
         done = ETrue;
      }
      else
      {
         iButtonBitmaps->AppendL( image );
      }
      i++;
   }

   // So that we can detect our hotkey
   iGameWindow.AddScreenOrientationObserverL(*this);
   iGameController.AddGameControlRawInputHandlerL(*this);
}

//===================================================================
// Public Methods
//===================================================================

void CVirtualButtons::ActivateL()
{
   iIsActive = ETrue;

   iGameWindow.AddGameScreenDrawObserverL(*this);
   
   DrawButtons();
}

void CVirtualButtons::Deactivate()
{
   iIsActive = EFalse;
   
   iGameWindow.RemoveGameScreenDrawObserver(*this);

   ClearButtons();
}

TBool CVirtualButtons::WantDrawToScreen()
{
   return ( iIsActive );
}

void CVirtualButtons::PostDrawFrameToScreen(CGameScreenDevice* aDstGameScreenDevice)
{
   aDstGameScreenDevice->DirectDraw( iRect.iTl, *iBitmap, TRect(iBitmap->Size()) );
}

TBool CVirtualButtons::ConsumeRawPointerEvent(const TPointerEvent& aPointerEvent)
{
   TBool consumed = EFalse;
   
   if ( iIsActive )
   {
      if (aPointerEvent.iType == TPointerEvent::EButton1Down)
      {
         const TVirtualButton* virtualButton = FindVirtualButton( aPointerEvent.iPosition );
         if (virtualButton)
         {
            HandleVirtualButtonDown(virtualButton);
            consumed = ETrue;
         }
      }
      else if (aPointerEvent.iType == TPointerEvent::EButton1Up)
      {
         if (iButton)
         {
            HandleVirtualButtonUp();
            consumed = ETrue;
         }
      }
   }

   return ( consumed );
}

void CVirtualButtons::ScreenOrientationChanged(TOrientation aOrientation)
{
   iOrientation = aOrientation;
   
   if ( iIsActive ) DrawButtons();
}

//===================================================================
// Private Methods
//===================================================================

void CVirtualButtons::ClearButtons()
{
   CFbsBitGc *gc = iBitmap->FbsBitGc();
   gc->SetBrushColor(KRgbBlack);
   gc->SetBrushStyle(CGraphicsContext::ENullBrush);
   gc->Clear();
}

void CVirtualButtons::DetermineButtonRectL( void )
{
   const TVirtualButton *p = iVirtualButtons;

   if ( !p ) User::Leave( KErrCancel );

   TInt tl_x = 10000;
   TInt tl_y = 10000;
   TInt br_x = 0;
   TInt br_y = 0;

   // Find the biggest rect containing all buttons
   while ( p->iScanCode )
   {
      if ( p->iTlX < tl_x ) tl_x = p->iTlX;
      if ( p->iTlY < tl_y ) tl_y = p->iTlY;
      if ( p->iBrX > br_x ) br_x = p->iBrX;
      if ( p->iBrY > br_y ) br_y = p->iBrY;

      p++;
   }

   iRect.SetRect( tl_x, tl_y, (br_x + 1), (br_y + 1) );
}

void CVirtualButtons::DrawButtons()
{
   CFbsBitGc *gc = iBitmap->FbsBitGc();
   gc->SetBrushColor(KRgbBlack);
   gc->SetBrushStyle(CGraphicsContext::ENullBrush);
   gc->Clear();
   gc->SetBrushStyle(CGraphicsContext::ESolidBrush);

   const TVirtualButton *p = iVirtualButtons;
   TInt button_image_cnt = iButtonBitmaps->Count();
   while ( p->iScanCode )
   {
      TInt image_idx = ( p == iButton ) ? p->iBitmapIdxPressed : p->iBitmapIdxReleased;

      TInt x = p->iTlX - iRect.iTl.iX;
      TInt y = p->iTlY - iRect.iTl.iY;
      if ( image_idx < button_image_cnt )
      {
         gc->BitBlt( TPoint(x,y), (*iButtonBitmaps)[image_idx] );
      }
      else
      {
         TRect r( TPoint(x,y), TSize(p->iBrX - p->iTlX, p->iBrY - p->iTlY) );
         gc->SetBrushColor(KRgbRed);
         gc->DrawRect( r );
      }
      p++;
   }

   gc->SetOrientation(ToGraphicsOrientation(iOrientation));
}

const TVirtualButton* CVirtualButtons::FindVirtualButton(const TPoint& aPoint)
{
   const TVirtualButton *p      = &iVirtualButtons[0];
   const TVirtualButton *button = NULL;

   while ( p->iScanCode )
   {
      TRect rect( p->iTlX, p->iTlY, p->iBrX, p->iBrY);

      if ( rect.Contains(aPoint) )
      {
         button = p;
         break;
      }

      p++;
   }

   return ( button );
}

void CVirtualButtons::HandleVirtualButtonDown(const TVirtualButton* aVirtualButton)
{
   if ( iButton )
   {
      HandleVirtualButtonUp();
   }

   iGameController.InjectKeyEvent(EEventKeyDown, aVirtualButton->iScanCode);
   iGameController.InjectKeyEvent(EEventKey,     aVirtualButton->iScanCode);
   iButton = aVirtualButton;
   DrawButtons();
}

void CVirtualButtons::HandleVirtualButtonUp()
{
   if (iButton)
   {
      iGameController.InjectKeyEvent(EEventKeyUp, iButton->iScanCode);
      iButton = NULL;
      DrawButtons();
   }
}

#endif // USE_VIRTUAL_BUTTONS


