//===================================================================
//
// File:  audio_motorola.cpp
//
// Nearly entirely re-written for EMame/EDoom 
//
// Credits:
//
//   Steve Fischer (2004)
//     - http://stevesprojects.com
//     - Ported & integrated all the below code to the A920
//
//   Peter van Sebille
//     - http://mobile.yipton.net
//     - Author of EMame for P800/P900 from which this port 
//       came from
//
//===================================================================
//
// Modifed By: Steve Fischer (steve@stevesprojects.com)
//
//	(c) Copyright 2004-2005, Steve Fischer
//	All Rights Reserved
//
//===================================================================
//
// Author:  Peter van Sebille (peter@yipton.net)
//
//	(c) Copyright 2001, Peter van Sebille
//	All Rights Reserved
//
//===================================================================

#include <e32svr.h>
#include <cmaudiofb.h>
#include "audio_motorola.h"

#define __AUD_DEBUG(x...)

//===================================================================
//
//  CGameAudio Class Definition
//
//===================================================================

CGameAudio* CGameAudio::NewL( TBool aStereo, TInt aSamplesPerFrame )
{
   __AUD_DEBUG( _L("CGameAudio::NewL") );

   CGameAudio* self = new (ELeave) CGameAudio( aStereo, aSamplesPerFrame );

   CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();

   __AUD_DEBUG( _L("CGameAudio::NewL - exit") );
   
	return self;
}

CGameAudio::CGameAudio( TBool aStereo, TInt aSamplesPerFrame )
{
   __AUD_DEBUG( _L("CGameAudio::CGameAudio") );

   iStereo    = aStereo;
   iSamplesPerFrame = aSamplesPerFrame;

   iPcmServer = NULL;
   iDeleteMe  = NULL;
   iAudioCtrl = NULL;

   iACConnect = EFalse;
	iConnected = EFalse;
	iDecoding  = EFalse;

   iQueueCnt  = 0;
   iWatchDog  = 0;

   iSoundOn   = ETrue;  // Initially forced on

   iQueueAO.SetParent( this );
   iDestroyAO.SetParent( this );
   iRestartAO.SetParent( this );

   __AUD_DEBUG( _L("CGameAudio::CGameAudio exit") );
}

CGameAudio::~CGameAudio()
{
   __AUD_DEBUG( _L("CGameAudio::~CGameAudio") );

   DestroySoundApi();

   iQueueAO.Cancel();
   iDestroyAO.Cancel();
   iRestartAO.Cancel();

   delete iPcmServer;

	for (TInt i = 0; i < KSoundBuffers; i++)
   {
      delete iPcmBufPtrs[i];
		delete iPcmBuffers[i];
   }

   __AUD_DEBUG( _L("CGameAudio::~CGameAudio") );
}

void CGameAudio::ConstructL()
{
   __AUD_DEBUG( _L("CGameAudio::ConstructL") );

   TInt frames_per_buffer = ((TInt)(KSamplesPerBuffer/iSamplesPerFrame));
   iBufferSize = frames_per_buffer * iSamplesPerFrame;
   iBufferSize = (iStereo) ? iBufferSize << 2 : iBufferSize << 1;

	for (TInt i = 0; i < KSoundBuffers; i++)
	{
		iPcmBuffers[i] = HBufC8::NewL(iBufferSize);
		iPcmBuffers[i]->Des().SetLength(iBufferSize);
      iPcmBufPtrs[i] = new TPtr8( iPcmBuffers[i]->Des() );
      iPcmBufFlag[i] = EBufferFree;
	}
   iCurrentBuffer = 0;
   iCurrentOffset = 0;

   iAudioCtrl = CMAudioAC::NewL( *this );

   iQueueAO.Activate();

   CreateSoundApiL();

   __AUD_DEBUG( _L("CGameAudio::ConstructL exit") );
}

void CGameAudio::CreateSoundApiL()
{
   __AUD_DEBUG( _L("CreateSoundApiL") );

   if ( iDeleteMe )
   {
      __AUD_DEBUG( _L("CreateSoundApiL - delete me") );
      delete iDeleteMe;  // Delete the old API if needed.
      iDeleteMe = NULL;
   }

   iConnected = EFalse;
   iDecoding  = EFalse;
   iQueueCnt  = 0;

   TMAudioFBBufSettings settings;

   settings.iPCMSettings.iSamplingFreq = (TMSampleRate) KAudioSampleRate;
   settings.iPCMSettings.iStereo       = iStereo;

   iPcmServer = CMAudioFB::NewL( EMAudioFBRequestTypeDecode
                               , EMAudioFBFormatPCM
                               , settings
                               , *this
                               , EMAudioPriorityPlaybackGeneric
                               );
}

void CGameAudio::DestroySoundApi( void )
{
   __AUD_DEBUG( _L("+DestroySoundApi") );

   iConnected = EFalse;
   iDecoding  = EFalse;

   if ( iDeleteMe )
   {
      delete iDeleteMe;
      iDeleteMe = NULL;
   }

   if ( iPcmServer )
   {
      delete iPcmServer;
      iPcmServer = NULL;
   }

   // Free all buffer flags
 	for (TInt i = 0; i < KSoundBuffers; i++)
	{
      iPcmBufFlag[i] = EBufferFree;
	}
   iCurrentBuffer = 0;
   iCurrentOffset = 0;

   __AUD_DEBUG( _L("-DestroySoundApi") );
}

TInt CGameAudio::Write( TUint16 *aPtr, TInt aLen)
{
   __AUD_DEBUG( _L("AudW/%d/%d/%d"), aLen, iCurrentBuffer, iCurrentOffset );

   TInt rtn = aLen;

   if ( iSoundOn && !iPcmServer )
   {
      __AUD_DEBUG( _L("AudW/recreate") );
      iRestartAO.Restart();
   }
   else if ( iSoundOn && iConnected )
   {
      // Skip buffer if not returned yet
      if ( iPcmBufFlag[iCurrentBuffer] == EBufferInUse )
      {
         __AUD_DEBUG( _L("AudW/buf not rtn'd/%d"), iCurrentBuffer );

         // Just an attempt to recover from lockup conditions
         iWatchDog ++;
         if ( iWatchDog > KAudioWatchdogCount )
         {
            iDestroyAO.Destroy();
         }
      }
      else
      {
//         sProfiler->Mark(0);


         iWatchDog = 0;

         // Determine the data length in bytes
         TInt byte_size = iStereo ? aLen << 2 : aLen << 1;
         TInt remaining = iBufferSize - iCurrentOffset;

         // Copy the data
         TUint8 *dest = (((TUint8*)(iPcmBuffers[iCurrentBuffer]->Ptr())) + iCurrentOffset);
         __AUD_DEBUG( _L("AudWDes/%08X"), (TUint32) dest );

         if ( byte_size > remaining )
         {
            Mem::Copy(dest, aPtr, remaining);
            iPcmBufFlag[iCurrentBuffer] = EBufferInUse;
//         sProfiler->Mark(1);
            iQueueAO.Queue( iCurrentBuffer );
            iCurrentBuffer ++;
            if (!iDecoding) { iQueueCnt++; }
            if (iCurrentBuffer >= KSoundBuffers)
            {
               iCurrentBuffer = 0;
            }
            dest = (TUint8*) (iPcmBuffers[iCurrentBuffer]->Ptr());
            __AUD_DEBUG( _L("AudWDest2/%08X"), (TUint32) dest );

            Mem::Copy(dest, (((TUint8*)(aPtr))+remaining), (byte_size - remaining) );
            iCurrentOffset = byte_size - remaining;
            __AUD_DEBUG( _L("AudWQ/%08X")
                      , (TUint32)iPcmBufPtrs[iCurrentBuffer]
                      );
         }
         else
         {
            __AUD_DEBUG( _L("AudW20") );
            Mem::Copy(dest, aPtr, byte_size);
            iCurrentOffset += byte_size;

            if ( iCurrentOffset == iBufferSize )
            {
               iPcmBufFlag[iCurrentBuffer] = EBufferInUse;
//         sProfiler->Mark(2);
               iQueueAO.Queue( iCurrentBuffer );
               iCurrentBuffer ++;
               if (!iDecoding) { iQueueCnt++; }
               if (iCurrentBuffer >= KSoundBuffers)
               {
                  iCurrentBuffer = 0;
               }

               iCurrentOffset = 0;
               __AUD_DEBUG( _L("AudWQ/%08X")
                         , (TUint32)iPcmBufPtrs[iCurrentBuffer]
                         );
            }
         }
      }
	}

   __AUD_DEBUG( _L("-AudW/%d"), iCurrentBuffer );

   return ( rtn );
}

TBool CGameAudio::IsSoundOn()
{
   return (iConnected && iSoundOn);
}

void CGameAudio::SetSoundOn(TBool aEnable)
{
   __AUD_DEBUG( _L("SetSoundOn/%d"), aEnable );

   if ( iSoundOn != aEnable )
   {
      iSoundOn = aEnable;

      if ( iSoundOn )
      {
         iRestartAO.Restart();
      }
      else
      {
         iDestroyAO.Destroy();
      }
   }
}

void CGameAudio::SetMasterVolume( TInt aUpDn )
{
   __AUD_DEBUG( _L("SetMasterVolume/%d/%d"), aUpDn, iACConnect );

   // Prevent the annoying volume bar from appearing
   if ( iACConnect )
   {
      TInt vol = iAudioCtrl->GetMasterVolume();
      TInt max = iAudioCtrl->GetMaxMasterVolume();

      if ( aUpDn == EIncreaseMasterVolume )
      {
         // Adjust the volume up
         vol++;
      }
      else
      {
         // Adjust the volume down
         vol--;
      }

      if ( ( vol >= 0 ) && ( vol <= max ) )
      {
         iAudioCtrl->SetMasterVolume( vol );
      }
   }
}

void CGameAudio::OnEvent
( TMAudioFBCallbackState aState
, TInt                   aError
)
{
   __AUD_DEBUG( _L("OnEvent/%d/%d"), aState, aError );

   switch ( aState )
   {
   case EMAudioFBCallbackStateReady:
      iConnected = ETrue;
      iWatchDog  = 0;
      iDestroyAO.Activate();
      iRestartAO.Activate();
      break;

   case EMAudioFBCallbackStateDecodeCompleteStopped:
      break;

   case EMAudioFBCallbackStateDecodeError:
      switch( aError )
      {
      case EMAudioFBCallbackErrorBufferFull:
      case EMAudioFBCallbackErrorForcedStop:
      case EMAudioFBCallbackErrorForcedClose:
      case EMAudioFBCallbackErrorPriorityRejection:
      case EMAudioFBCallbackErrorResourceRejection:
      case EMAudioFBCallbackErrorAlertModeRejection:
      case EMAudioFBCallbackErrorUnknown:
         // Setup the API to be deleted outside of the OnEvent() callback
         iDeleteMe  = iPcmServer;
         iPcmServer = NULL;
         iConnected = EFalse;
         iDecoding  = EFalse;
         break;

      default:
         break;
      }
      break;

   default:
      break;
   }
}

void CGameAudio::OnEvent
( TMAudioFBCallbackState aState
, TInt                   aError
, TDes8*                 aBuffer
)
{
   __AUD_DEBUG( _L("OnEvent/%d/%d/%08X"), aState, aError, (TUint32) aBuffer );

   switch( aState )
   {
   case EMAudioFBCallbackStateDecodeError:
      switch( aError )
      {
      case EMAudioFBCallbackErrorBufferFull:
      case EMAudioFBCallbackErrorForcedStop:
      case EMAudioFBCallbackErrorForcedClose:
      case EMAudioFBCallbackErrorPriorityRejection:
      case EMAudioFBCallbackErrorResourceRejection:
      case EMAudioFBCallbackErrorAlertModeRejection:
      case EMAudioFBCallbackErrorUnknown:
         // Setup the API to be deleted outside of the OnEvent() callback
         iDeleteMe  = iPcmServer;
         iPcmServer = NULL;
         iConnected = EFalse;
         iDecoding  = EFalse;
         break;

      default:
         break;
      }
      break;

   case EMAudioFBCallbackStateDecodeBufferDecoded:
      {
         for (TInt i = 0; i < KSoundBuffers; i++)
         {
            if ( aBuffer == iPcmBufPtrs[i] )
            {
               iPcmBufFlag[i] = EBufferFree;
               __AUD_DEBUG( _L("OnEvent - freed buffer %d"), i );
               break;
            }
         }
      }
      break;

   default:
      OnEvent( aState, aError );
      break;
   }
}

void CGameAudio::OnEvent( TMAudioACCallbackState aState, TInt aError )
{
   if ( !iACConnect && aState == EMAudioACCallbackStateReady )
   {
      iACConnect = ETrue;
   }
}

void CGameAudio::QueueBuffer( TInt aBufferIdx )
{
//   sProfiler->Mark(4);
   iPcmServer->QueueBufferL(iPcmBufPtrs[aBufferIdx]);

   if ( !iDecoding && iQueueCnt >= 2 )
   {
      iPcmServer->DecodeL();
      iDecoding = ETrue;
   }
}

//===================================================================
//
//  CQueueBuffer Class Definition
//
//===================================================================

CGameAudio::CQueueBuffer::CQueueBuffer() : CActive( EPriorityLow )
{
   CActiveScheduler::Add(this);
   iParent    = NULL;
   iThreadId  = RThread().Id();
}

void CGameAudio::CQueueBuffer::SetParent( CGameAudio *aParent )
{
   iParent = aParent;
}

void CGameAudio::CQueueBuffer::Activate()
{
   if ( !IsActive() )
   {
      SetActive();
      iStatus = KRequestPending;
   }
}

void CGameAudio::CQueueBuffer::Queue( TInt aBufIdx )
{
   if ( IsActive() && iStatus == KRequestPending )
   {
      RThread thread;

      if ( thread.Open(iThreadId) == KErrNone )
      {
         iBufferIdx = aBufIdx;
         iStatusPtr = &iStatus;
         thread.RequestComplete( iStatusPtr, KErrNone );
      }
   }
}

void CGameAudio::CQueueBuffer::RunL( void )
{
//         sProfiler->Mark(3);
   iParent->QueueBuffer( iBufferIdx );
   Activate();
}

void CGameAudio::CQueueBuffer::DoCancel( void )
{
   User::RequestComplete( iStatusPtr, KErrNone );
}

TInt CGameAudio::CQueueBuffer::RunError( TInt aError )
{
   User::Panic( _L("CQueueBuffer"), aError );
   return ( ETrue );
}

//===================================================================
//
//  CDestroyApi Class Definition
//
//===================================================================

CGameAudio::CDestroyApi::CDestroyApi() : CActive( EPriorityLow )
{
   CActiveScheduler::Add(this);
   iParent    = NULL;
   iThreadId  = RThread().Id();
}

void CGameAudio::CDestroyApi::SetParent( CGameAudio *aParent )
{
   iParent = aParent;
}

void CGameAudio::CDestroyApi::Activate()
{
   if ( !IsActive() )
   {
      SetActive();
      iStatus = KRequestPending;
   }
}

void CGameAudio::CDestroyApi::Destroy()
{
   if ( IsActive() && iStatus == KRequestPending )
   {
      RThread thread;

      if ( thread.Open(iThreadId) == KErrNone )
      {
         iStatusPtr = &iStatus;
         thread.RequestComplete( iStatusPtr, KErrNone );
      }
   }
}

void CGameAudio::CDestroyApi::RunL( void )
{
   if ( iParent ) iParent->DestroySoundApi();
}

void CGameAudio::CDestroyApi::DoCancel( void )
{
   User::RequestComplete( iStatusPtr, KErrNone );
}

TInt CGameAudio::CDestroyApi::RunError( TInt aError )
{
   User::Panic( _L("CDestroyApi"), aError );
   return ( ETrue );
}
 
//===================================================================
//
//  CRestartApi Class Definition
//
//===================================================================

CGameAudio::CRestartApi::CRestartApi() : CActive( EPriorityLow )
{
   CActiveScheduler::Add(this);
   iParent    = NULL;
   iThreadId  = RThread().Id();
}

void CGameAudio::CRestartApi::SetParent( CGameAudio *aParent )
{
   iParent = aParent;
}

void CGameAudio::CRestartApi::Activate()
{
   if ( !IsActive() )
   {
      SetActive();
      iStatus = KRequestPending;
   }
}

void CGameAudio::CRestartApi::Restart()
{
   if ( IsActive() && iStatus == KRequestPending )
   {
      RThread thread;

      if ( thread.Open(iThreadId) == KErrNone )
      {
         iStatusPtr = &iStatus;
         thread.RequestComplete( iStatusPtr, KErrNone );
      }
   }
}

void CGameAudio::CRestartApi::RunL( void )
{
   if ( iParent ) iParent->CreateSoundApiL();
}

void CGameAudio::CRestartApi::DoCancel( void )
{
   User::RequestComplete( iStatusPtr, KErrNone );
}

TInt CGameAudio::CRestartApi::RunError( TInt aError )
{
   User::Panic( _L("CRestartApi"), aError );
   return ( ETrue );
}
