//===================================================================
//
// File:  engine.cpp
//
// From EDoom & EMame, heavily modified
//
// Credits:
//
//   Steve Fischer (2005) 
//     - Ported & integrated all the below code to ENes for the A920
//
//   Peter van Sebille (2002-2003)
//     - http://mobile.yipton.net
//     - Author of EDoom & EMame for P800/P900 for which much of the
//       front-end GUI application came from
//
//===================================================================
//
//	Author:	Peter van Sebille (peter@yipton.net)
//
//	(c) Copyright 2002, Peter van Sebille
//	All Rights Reserved
//
//===================================================================

#include "Engine.h"
#include <e32std.h>
#include <w32std.h>
#include <eikenv.h>
#include <eikdll.h>
#include <f32file.h>
#include <hal.h>

#include "Interface.h"

#define __APP_DEBUG(x...)

_LIT( KGameName_C, "c:\\system\\Apps\\enes\\nes.exe" );
_LIT( KGameName_D, "d:\\system\\Apps\\enes\\nes.exe" );
_LIT( KPathOnC,    "c:\\Documents\\Media Files\\Document\\NESROMS\\" );
_LIT( KPathOnD,    "d:\\Media Files\\Document\\NESROMS\\" );
_LIT( KExtNES,      "*.nes" );
_LIT( KGameIni,    "c:\\system\\Apps\\enes\\enes.ini" );

_LIT( KGameSemaphore, "ENES_SEM" );
_LIT( KGameChunk,     "ENES_CHUNK" );

#define CHUNK_SIZE   (1024)

//===================================================================
//
//  CGameRunner Class Definition
//
//===================================================================

CGameRunner* CGameRunner::NewL( MGameWatcher& aGameWatcher, CGameEngine& aGameEngine )
{
	CGameRunner*	self = new(ELeave) CGameRunner(aGameWatcher, aGameEngine);
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();
	return self;
}

CGameRunner::CGameRunner
( MGameWatcher& aGameWatcher
, CGameEngine&  aGameEngine
) : CActive(CActive::EPriorityStandard)
  , iGameWatcher(aGameWatcher)
  , iGameEngine(aGameEngine)
{
   // Create the global chunk used to pass the game filename
   iChunk.CreateGlobal( KGameChunk, CHUNK_SIZE, CHUNK_SIZE );
}

CGameRunner::~CGameRunner()
{
   iChunk.Close();
	TerminateGame(EFalse);
}

void CGameRunner::ConstructL()
{
	CActiveScheduler::Add(this);
}

void CGameRunner::RunL()
{
	iGameWatcher.NotifyGameFinished();
}

void CGameRunner::DoCancel()
{
	iThread.LogonCancel(iStatus);
}

void CGameRunner::RunGameL( const TDesC& aFilename )
{
   // Configure the interface chunk
   TAppToExeInterface *p = (TAppToExeInterface*) iChunk.Base();

   TGameOptions game_options = iGameEngine.GameOptions();

   p->iOptions = game_options;

   iGameFilename.Copy( aFilename );
   TInt len = iGameFilename.Length(); 
   const TUint16 *src = iGameFilename.Ptr();

   TInt i;
   for ( i = 0; i < len; i++ )
   {
      p->iFilename[i] = (TUint8) (*src++ & 0x00FF);
   }
   p->iFilename[i] = 0;

   // Launch the EXE
	TThreadId threadId;
	TRAPD(err, threadId = EikDll::StartExeL(KGameName_C));
	if (err)
	{
		TRAP(err, threadId = EikDll::StartExeL(KGameName_D));
	}
	User::LeaveIfError(err);

	iThread.Open(threadId);
	iThread.Logon(iStatus);
	SetActive();
}

void CGameRunner::TerminateGame(TBool aNotifyObserver)
{
	Cancel();
	ShutdownGame();
	if (aNotifyObserver)
   {
		RunL();
   }
}

void CGameRunner::ResumeGame()
{
	RSemaphore	semaphore;

	if (semaphore.OpenGlobal(KGameSemaphore) == KErrNone)
	{
		semaphore.Signal();
		semaphore.Close();
	}
	else
	{
		User::InfoPrint(_L("Could not resume"));
	}
}

void CGameRunner::ShutdownGame()
{
	if (iThread.Handle())
	{
		iThread.Terminate(0);
		iThread.Close();
	}
}

//===================================================================
//
//  CGameEngine Class Declaration
//
//===================================================================

CGameEngine::CGameEngine* CGameEngine::NewL( RFs& aFs )
{
	CGameEngine*	self = new (ELeave) CGameEngine( aFs );
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();
	return self;
}

CGameEngine::CGameEngine( RFs& aFs ) : iFs(aFs)
{
   iRomNameArray = NULL;
}

CGameEngine::~CGameEngine()
{
   iRomNameArray->Reset();
   delete iRomNameArray;
}

void CGameEngine::ConstructL()
{
   iRomNameArray = new (ELeave) CDesCArrayFlat(10);
   LoadGameOptions();
}

void CGameEngine::FindRomsL()
{
   __APP_DEBUG( _L("+FindRomsL") );

   iRomNameArray->Reset();
   CheckDirL( KPathOnC );
   CheckDirL( KPathOnD );

   __APP_DEBUG( _L("-FindRomsL") );
}

void CGameEngine::CheckDirL( const TDesC& aDir )
{
   __APP_DEBUG( _L("CheckDirL/%S"), &aDir );

   TInt i;
   TInt cnt;
   TInt match_nes;
   TBuf<256> rom_name;

   TBuf<256> path;
   path.Copy( aDir );
   path.Append( '*' );

   CDir *dir = NULL;

   if ( iFs.GetDir( path, KEntryAttMaskSupported, ESortByName, dir ) == KErrNone )
   {
      CleanupStack::PushL( dir );

      cnt = dir->Count();
      for ( i = 0; i < cnt; i++ )
      {
         const TEntry& entry = (*dir)[i];
         rom_name.Copy( aDir );
         rom_name.Append( entry.iName );
         rom_name.LowerCase();

         match_nes = rom_name.Match( KExtNES );
         if (  match_nes != KErrNotFound )
         {
            iRomNameArray->AppendL( rom_name );
            __APP_DEBUG( _L("CheckDirL/DIR[%d]=\"%S\""), i, &rom_name );
         }
      }

		CleanupStack::PopAndDestroy( dir );
   }
}

TInt CGameEngine::GetRomName( TInt aIndex, TDes& aName )
{
   // NOTE: the "filename" contains the entire path and 
   //       the "name" contains only the rom's filename (no path) 

   __APP_DEBUG( _L("+GetRomName") );

   TInt err = KErrNone;
   TInt len = aName.MaxLength();

   // Verify the passed in index
   if ( aIndex >= NoOfRoms() )
   {
      err = KErrArgument;
   }
   else
   {
      // Determine where the "name" is from the filename
      TInt filename_len  = (*iRomNameArray)[ aIndex ].Length();
      TInt start_of_name = (*iRomNameArray)[ aIndex ].LocateReverse( '\\' ) + 1;
      TInt len_of_name   = filename_len - start_of_name;

      // Verify the passed descriptor can hold the "name"
      if ( len < len_of_name )
      {
         err = KErrOverflow;
      }
      else
      {
         aName.Copy( (*iRomNameArray)[ aIndex ].Right( len_of_name ) );
      }
   }

   __APP_DEBUG( _L("-GetRomName/%d/%S"), err, &aName );

   return err;
}

TInt CGameEngine::GetRomFilename( TInt aIndex, TDes& aFilename )
{
   // NOTE: the "filename" contains the entire path and 
   //       the "name" contains only the rom's filename (no path) 

   __APP_DEBUG( _L("+GetRomFilename") );

   TInt err = KErrNone;
   TInt len = aFilename.MaxLength();

   // Verify the passed in index
   if ( aIndex >= NoOfRoms() )
   {
      err = KErrArgument;
   }
   // Verify the passed descriptor can hold the "filename"
   else if ( len < (*iRomNameArray)[ aIndex ].Length() )
   {
      err = KErrOverflow;
   }
   // Copy the "filename"
   else
   {
      aFilename.Copy( (*iRomNameArray)[ aIndex ] );
   }

   __APP_DEBUG( _L("-GetRomFilename/%d/%S"), err, &aFilename );

   return err;
}

TInt CGameEngine::GetRomIndex( const TDesC& aName )
{
   __APP_DEBUG( _L("+GetRomIndex") );

   TInt idx;
   TInt cnt = NoOfRoms();
   TBuf<256> rom_name;

   for ( idx = 0; idx < cnt; idx++ )
   {
      // Determine where the "name" is from the filename
      TInt filename_len  = (*iRomNameArray)[ idx ].Length();
      TInt start_of_name = (*iRomNameArray)[ idx ].LocateReverse( '\\' ) + 1;
      TInt len_of_name   = filename_len - start_of_name;

      rom_name.Copy( (*iRomNameArray)[ idx ].Right( len_of_name ) );

      if ( rom_name == aName )
      {
         break;
      }
   }

   if ( idx >= cnt )
   {
      idx = KErrNotFound;
   }

   __APP_DEBUG( _L("-GetRomIndex/%d"), idx );

   return ( idx );
}

TInt CGameEngine::NoOfRoms()
{
   return ( iRomNameArray->Count() );
}

void CGameEngine::SetGameOptions( TGameOptions &aOptions )
{
   iOptions = aOptions;

   SaveGameOptions();
}

void CGameEngine::DefaultOptions( void )
{
   iOptions.iVersion       = GameOptionVersion;
   iOptions.iRotation      = EAppToExeRotationLeft;
   iOptions.iDisplayMode   = EAppToExeDisplayFitToScreen;
   iOptions.iSound         = ETrue;

   iOptions.iButtonStart   = EAppToExeButtonVirtualGreen;
   iOptions.iButtonSelect  = EAppToExeButtonVirtualBlue;
   iOptions.iButtonComboAB = EAppToExeButtonVirtualPurple;

   TInt value = 0;
   HAL::Get( HALData::EMachineUid, value );
   switch(value)
   {
   case 0x101f408b: // P800
   case 0x101fb2ae: // P900
   case 0x10200ac6: // P910
   case 0x101fd279: // P3x
      iOptions.iButtonGameA = EAppToExeButtonPxxxBrowser;
      iOptions.iButtonGameB = EAppToExeButtonPxxxCamera;
      break;

   case 0x101f6b26: // A9xx & A10xx
   default:	
      iOptions.iButtonGameA = EAppToExeButtonA92xA1000GameA;
      iOptions.iButtonGameB = EAppToExeButtonA92xA1000GameB;
      break;
   }

   iOptions.iTurboGameA     = EFalse;
   iOptions.iTurboGameB     = EFalse;
   iOptions.iTurboSpeed     = 3;

   iOptions.iDisableSendEnd = EFalse;
}

void CGameEngine::LoadGameOptions( void )
{
   TPckg<TGameOptions>   pkg(iOptions);

   RFile   file;
   TInt err = file.Open(iFs, KGameIni, 0);
   if (!err)
   {
      err = file.Read(pkg);
      if (err == KErrNone)
      {
         if (iOptions.iVersion != GameOptionVersion)
         {
            err = -1; // force DefaultOptions()
         }
      }
      file.Close();
   }

   if (err)
   {
      DefaultOptions();
   }
}

void CGameEngine::SaveGameOptions( void )
{
   TPckgC<TGameOptions>   pkg(iOptions);

   RFile file;
   TInt err = file.Open(iFs, KGameIni, EFileWrite);
   if (err)
   {
      err = file.Create(iFs, KGameIni, EFileWrite);
   }

   if (!err)
   {
      err = file.Write(pkg);
      file.Close();
   }
}

