//===================================================================
//
// File:  dialogs.cpp
//
// From EDoom & EMame, heavily modified
//
// Credits:
//
//   Steve Fischer (2005) 
//     - Ported & integrated all the below code to ENes for the A920
//
//   Peter van Sebille (2002-2003)
//     - http://mobile.yipton.net
//     - Author of EDoom & EMame for P800/P900 for which much of the
//       front-end GUI application came from
//
//===================================================================
//
//	Author:	Peter van Sebille (peter@yipton.net)
//
//	(c) Copyright 2002, Peter van Sebille
//	All Rights Reserved
//
//===================================================================

#include "Dialogs.h"
#include "Engine.h"
#include "enes.hrh"
#include <enes.rsg>

#include "version.h"
#include "CSimpleTextParser.h"
#include <txtrich.h>				// CRichText
#include <eikrted.h>				// CEikRichTextEditor
#include <qikvertoptionbuttonlist.h>  // CQikVertOptionButtonList
#include <qikslider.h>

#define __APP_DEBUG(x...)

//===================================================================
//
//  CGameRunningDialog Class Definition
//
//===================================================================

CGameRunningDialog::CGameRunningDialog(CGameRunner& aGameRunner) : iGameRunner(aGameRunner)
{
}

void CGameRunningDialog::PreLayoutDynInitL()
{
}

void CGameRunningDialog::HandleControlStateChangeL(TInt aControlId)
{
	switch (aControlId)
	{
		case ECtlCmdButPanelRunningGame:
		{
			const TInt ctrlId=STATIC_CAST(CEikDialogToolBar*, Control(aControlId))->StateChangeControlId();
			switch (ctrlId)
			{
				case ECtlCmdButStopGame:
					iGameRunner.TerminateGame();
					break;

				case ECtlCmdButResumeGame:
					iGameRunner.ResumeGame();
					break;
				default:
					break;
			}
			break;
		}
		default:
			break;
	}
}

//===================================================================
//
//  CSimpleTextInfoDialog Class Definition
//
//===================================================================

CSimpleTextInfoDialog::CSimpleTextInfoDialog
( TInt aTextIdOne
, TInt aRichTextCtlIdOne
, TInt aTextIdTwo
, TInt aRichTextCtlIdTwo
, TBool aSimpleTextResIdOneIsArray
, TBool aSimpleTextResIdTwoIsArray
) : iSimpleTextResIdOne(aTextIdOne)
  , iSimpleTextResIdTwo(aTextIdTwo)
  , iRichTextCtlIdOne(aRichTextCtlIdOne)
  , iRichTextCtlIdTwo(aRichTextCtlIdTwo)
  , iSimpleTextResIdOneIsArray(aSimpleTextResIdOneIsArray)
  , iSimpleTextResIdTwoIsArray(aSimpleTextResIdTwoIsArray)
  , iSetDialogBackground(ETrue)
{
}

void CSimpleTextInfoDialog::PreLayoutDynInitL()
{
	CEikRichTextEditor*		richTextEditor;
	
	if (iRichTextCtlIdOne != -1)
	{
		richTextEditor=STATIC_CAST(CEikRichTextEditor*, Control(iRichTextCtlIdOne));
		PreLayoutDynInitRichTextL(*richTextEditor, iRichTextCtlIdOne, iSimpleTextResIdOne);
	}

	if (iRichTextCtlIdTwo != -1)
	{
		richTextEditor=STATIC_CAST(CEikRichTextEditor*, Control(iRichTextCtlIdTwo));
		richTextEditor->Border().SetType(ENone);
		PreLayoutDynInitRichTextL(*richTextEditor, iRichTextCtlIdTwo, iSimpleTextResIdTwo);
	}
}

void CSimpleTextInfoDialog::PreLayoutDynInitRichTextL(CEikRichTextEditor& aRichTextEditor, TInt aRichTextCtlId, TInt aSimpleTextResId)
{
	iEikonEnv->BusyMsgL(_L("Opening"));
	aRichTextEditor.SetEdwinSizeObserver(this);
	if (iSetDialogBackground)
		aRichTextEditor.SetBackgroundColorL(iEikonEnv->Color(EColorDialogBackground));
	aRichTextEditor.SetSize(aRichTextEditor.Size()); // Set the size of the edwin

		// no scrollbars for us
	aRichTextEditor.CreateScrollBarFrameL(); // Create the scrollbars
	aRichTextEditor.ScrollBarFrame()->SetScrollBarVisibilityL(CEikScrollBarFrame::EOff, iWantVertScrollbar ? CEikScrollBarFrame::EAuto: CEikScrollBarFrame::EOff);

	ShowTextL(*aRichTextEditor.RichText(), aRichTextCtlId, aSimpleTextResId);

	aRichTextEditor.UpdateAllFieldsL(); // Updates all the fields in the document

	aRichTextEditor.UpdateScrollBarsL();
}

void CSimpleTextInfoDialog::PostLayoutDynInitL()
{
	Layout();
	iEikonEnv->BusyMsgCancel();
}

TBool CSimpleTextInfoDialog::HandleEdwinSizeEventL(CEikEdwin* aEdwin, TEdwinSizeEvent aEventType, TSize aDesirableEdwinSize)
{
	if ((aEventType == EEventSizeChanging))
		aEdwin->SetSize(aDesirableEdwinSize);
	return EFalse;
}

void CSimpleTextInfoDialog::ShowTextL(CRichText& aRichText, TInt /*aRichTextCtlId*/, TInt aResId)
{
	if (aResId != -1)
	{
		if (  ((aResId == iSimpleTextResIdOne) && (iSimpleTextResIdOneIsArray))
		   || ((aResId == iSimpleTextResIdTwo) && (iSimpleTextResIdTwoIsArray))
		   )
		{
			CDesCArrayFlat* desArray = CEikonEnv::Static()->ReadDesCArrayResourceL(aResId);
			CleanupStack::PushL(desArray);

			CSimpleTextFormatParser* parser = CSimpleTextFormatParser::NewLC();

			TInt	count = desArray->Count();
			for (TInt i=0 ;i<count ; i++)
				parser->ParseL(desArray->operator[](i), aRichText);

			CleanupStack::PopAndDestroy(parser);
			CleanupStack::PopAndDestroy(desArray);
		}
		else
		{
			HBufC* text = CEikonEnv::Static()->AllocReadResourceLC(aResId);
			ShowSimpleTextL(*text, aRichText);
			CleanupStack::PopAndDestroy(text);
		}
	}
}

void CSimpleTextInfoDialog::ShowSimpleTextL(const TDesC& aSimpleText, CRichText& aRichText)
{
	CSimpleTextFormatParser* parser = CSimpleTextFormatParser::NewLC();
	parser->ParseL(aSimpleText, aRichText);

	CleanupStack::PopAndDestroy(parser);
}

//===================================================================
//
//  CGameSetupDialog Class Definition
//
//===================================================================

CGameSetupDialog::CGameSetupDialog
( CGameEngine& aGameEngine
) : iGameEngine( aGameEngine )
{
}

void CGameSetupDialog::PreLayoutDynInitL()
{
   CEikHorOptionButtonList *button_list  = (CEikHorOptionButtonList*) Control( ECtlOptRotation );
   CEikHorOptionButtonList *display_mode = (CEikHorOptionButtonList*) Control( ECtlOptDisplayFit );
   CEikCheckBox            *sound_chkbox = (CEikCheckBox*)            Control( ECtlOptSound );
   CEikChoiceListBase      *start_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonStart );
   CEikChoiceListBase      *select_key   = (CEikChoiceListBase*)      Control( ECtlOptButtonSelect );
   CEikChoiceListBase      *game1_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonGameA );
   CEikChoiceListBase      *game2_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonGameB );
   CEikChoiceListBase      *combo_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonCombo );
   CEikCheckBox            *turbo_game1  = (CEikCheckBox*)            Control( ECtlOptTurboGameA );
   CEikCheckBox            *turbo_game2  = (CEikCheckBox*)            Control( ECtlOptTurboGameB );
   CQikSlider              *turbo_speed  = (CQikSlider*)              Control( ECtlOptTurboSpeed );
   CEikCheckBox            *disable_s_e  = (CEikCheckBox*)            Control( ECtlOptDisableSendEnd );

   TGameOptions game_options = iGameEngine.GameOptions();

   switch ( game_options.iRotation )
   {
   case EAppToExeRotationRight:
      button_list->SetButtonById( ECtlOptRotationRight );
      break;

   case EAppToExeRotationLeft:
   default:
      button_list->SetButtonById( ECtlOptRotationLeft );
      break;
   }

   switch ( game_options.iDisplayMode )
   {
   case EAppToExeDisplayFitToScreen:
      display_mode->SetButtonById( ECtlOptDisplayFitToScreen );
      break;

   case EAppToExeDisplayCenter:
   default:
      display_mode->SetButtonById( ECtlOptDisplayCenter );
      break;
   }

  	sound_chkbox->SetState( game_options.iSound ? CEikButtonBase::ESet : CEikButtonBase::EClear );

   start_key->SetCurrentItem( game_options.iButtonStart );
   select_key->SetCurrentItem( game_options.iButtonSelect );
   game1_key->SetCurrentItem( game_options.iButtonGameA );
   game2_key->SetCurrentItem( game_options.iButtonGameB );
   combo_key->SetCurrentItem( game_options.iButtonComboAB );

  	turbo_game1->SetState( game_options.iTurboGameA ? CEikButtonBase::ESet : CEikButtonBase::EClear );
  	turbo_game2->SetState( game_options.iTurboGameB ? CEikButtonBase::ESet : CEikButtonBase::EClear );

   turbo_speed->SetMinValueL(1);
   turbo_speed->SetMaxValueL(6);
   turbo_speed->SetNumberOfMarkersL(6);
   turbo_speed->SetValue( game_options.iTurboSpeed );

   disable_s_e->SetState( game_options.iDisableSendEnd ? CEikButtonBase::ESet : CEikButtonBase::EClear );
}

void CGameSetupDialog::SetSizeAndPositionL(const TSize& aSize)
{
	TSize size = aSize;
	if (size.iWidth < 400) size.iWidth = 400;
	SetCornerAndSize(EHCenterVCenter, size);
}
   
TBool CGameSetupDialog::OkToExitL(TInt aKeycode)
{
   if ( aKeycode )
   {
      CEikHorOptionButtonList *button_list  = (CEikHorOptionButtonList*) Control( ECtlOptRotation );
      CEikHorOptionButtonList *display_mode = (CEikHorOptionButtonList*) Control( ECtlOptDisplayFit );
      CEikCheckBox            *sound_chkbox = (CEikCheckBox*)            Control( ECtlOptSound );
      CEikChoiceListBase      *start_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonStart );
      CEikChoiceListBase      *select_key   = (CEikChoiceListBase*)      Control( ECtlOptButtonSelect );
      CEikChoiceListBase      *game1_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonGameA );
      CEikChoiceListBase      *game2_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonGameB );
      CEikChoiceListBase      *combo_key    = (CEikChoiceListBase*)      Control( ECtlOptButtonCombo );
      CEikCheckBox            *turbo_game1  = (CEikCheckBox*)            Control( ECtlOptTurboGameA );
      CEikCheckBox            *turbo_game2  = (CEikCheckBox*)            Control( ECtlOptTurboGameB );
      CQikSlider              *turbo_speed  = (CQikSlider*)              Control( ECtlOptTurboSpeed );
      CEikCheckBox            *disable_s_e  = (CEikCheckBox*)            Control( ECtlOptDisableSendEnd );

      TGameOptions game_options = iGameEngine.GameOptions();

      switch ( button_list->LabeledButtonId() )
      {
      case ECtlOptRotationRight:
         game_options.iRotation = EAppToExeRotationRight;
         break;

      case ECtlOptRotationLeft:
      default:
         game_options.iRotation = EAppToExeRotationLeft;
         break;
      }

      switch ( display_mode->LabeledButtonId() )
      {
      case ECtlOptDisplayFitToScreen:
         game_options.iDisplayMode = EAppToExeDisplayFitToScreen;
         break;

      case ECtlOptDisplayCenter:
      default:
         game_options.iDisplayMode = EAppToExeDisplayCenter;
         break;
      }

      game_options.iSound = (sound_chkbox->State() == CEikButtonBase::ESet) ? (TBool) ETrue : (TBool) EFalse;

      game_options.iButtonStart   = (TAppToExeButton) start_key->CurrentItem();
      game_options.iButtonSelect  = (TAppToExeButton) select_key->CurrentItem();
      game_options.iButtonGameA   = (TAppToExeButton) game1_key->CurrentItem();
      game_options.iButtonGameB   = (TAppToExeButton) game2_key->CurrentItem();
      game_options.iButtonComboAB = (TAppToExeButton) combo_key->CurrentItem();

      game_options.iTurboGameA = (turbo_game1->State() == CEikButtonBase::ESet) ? (TBool) ETrue : (TBool) EFalse;
      game_options.iTurboGameB = (turbo_game2->State() == CEikButtonBase::ESet) ? (TBool) ETrue : (TBool) EFalse;

      game_options.iTurboSpeed = turbo_speed->CurrentValue();

      game_options.iDisableSendEnd = (disable_s_e->State() == CEikButtonBase::ESet) ? (TBool) ETrue : (TBool) EFalse;

      iGameEngine.SetGameOptions( game_options );
   }

	return ( ETrue );
}

//===================================================================
//
//  CAboutDialog Class Definition
//
//===================================================================

CAboutDialog::CAboutDialog() : CSimpleTextInfoDialog(-1, ECtlAboutVersion, R_SIMPLE_TEXT_ABOUT_LINKS, ECtlAboutLinks)
{
}

void CAboutDialog::ShowTextL(CRichText& aRichText, TInt aRichTextCtlId, TInt aResId)
{
	if (aRichTextCtlId == ECtlAboutLinks)
		CSimpleTextInfoDialog::ShowTextL(aRichText, aRichTextCtlId, aResId);
	else
	{
		TBuf<16>	versionText;
		TBuf<512>	text;

#if (ENES_VERSION_PATCH > 0)
		versionText.Format(_L("%d.%d.%d"), ENES_VERSION_MAJOR, ENES_VERSION_MINOR, ENES_VERSION_PATCH);
#else
		versionText.Format(_L("%d.%d"), ENES_VERSION_MAJOR, ENES_VERSION_MINOR);
#endif

		HBufC*		aboutFormat =	CEikonEnv::Static()->AllocReadResourceLC(R_SIMPLE_TEXT_ABOUT);
		text.Format(*aboutFormat, &versionText);

		ShowSimpleTextL(text, aRichText);

		CleanupStack::PopAndDestroy(aboutFormat);
	}
}

//===================================================================
//
//  CPlayListDialog Class Definition
//
//===================================================================

CPlayListDialog::CPlayListDialog
( CGameEngine* aEngine
, TInt& aGameToRun
) : iEngine(aEngine)
  , iGameToRun(aGameToRun)
{
}

void CPlayListDialog::SetSizeAndPositionL(const TSize& aSize)
{
   __APP_DEBUG( _L("Dlg:SetSizeAndPositionL") );

   TSize dialogSize(400, aSize.iHeight);
   SetCornerAndSize(EHCenterVCenter,dialogSize);
}

void CPlayListDialog::HandleControlStateChangeL(TInt aControlId)
{
   __APP_DEBUG( _L("Dlg:HandleControlStateChangeL/%d"), aControlId );

   switch(aControlId)
   {
   case ECtlListboxPlayList:
      break;

   case ECtlCmdButPanel:
      {
         const TInt ctrlId = STATIC_CAST(CEikDialogToolBar*, Control(aControlId))->StateChangeControlId();
         switch (ctrlId)
         {
         case ECtlCmdButRun:
            TryExitL(EEikBidOk);
            break;

         case ECtlCmdButCancel:
            TryExitL(EEikBidCancel);
            break;
         }
      }
      break;
   }
}

void CPlayListDialog::PreLayoutDynInitL()
{
   __APP_DEBUG( _L("+Dlg:PreLayoutDynInitL") );

   TBuf<256> rom_name;

   CEikTextListBox* listbox = (CEikTextListBox*) Control(ECtlListboxPlayList);
   CDesCArray* listboxArray = ((CDesCArray*) listbox->Model()->ItemTextArray());

   // Append ROM names to listbox for display
   TInt count = iEngine->NoOfRoms();
   for ( TInt i = 0; i < count; i++ )
   {
      iEngine->GetRomName( i, rom_name );
      listboxArray->AppendL( rom_name );
   }
   listboxArray->Sort();

   CEikScrollBarFrame* scrollbarFrame = listbox->CreateScrollBarFrameL();

   // Set horizontal and vertical scroll bars to appear when required
   scrollbarFrame->SetScrollBarVisibilityL( CEikScrollBarFrame::EAuto
                                          , CEikScrollBarFrame::EAuto );

   SetTitleL(R_TEXT_PLAY_LIST_ROMS);

   __APP_DEBUG( _L("-Dlg:PreLayoutDynInitL") );
}

void CPlayListDialog::PostLayoutDynInitL()
{
   __APP_DEBUG( _L("Dlg:PostLayoutDynInitL/%d"), iGameToRun );

   TBuf<256> rom_name;

   CEikTextListBox* listbox = (CEikTextListBox*) Control(ECtlListboxPlayList);
   CDesCArray* listboxArray = ((CDesCArray*) listbox->Model()->ItemTextArray());

   if (iGameToRun != -1)
   {
      if ( iEngine->GetRomName( iGameToRun, rom_name ) == KErrNone )
      {
         TInt index = 0;
         if ( listboxArray->FindIsq( rom_name, index ) == KErrNone )
         {
            listbox->HandleItemAdditionL();
            listbox->SetCurrentItemIndex( index );
            listbox->View()->ScrollToMakeItemVisible( index );
            listbox->UpdateScrollBarsL();
         }
      }
   }
}

TInt CPlayListDialog::GameIndex()
{
   __APP_DEBUG( _L("Dlg:GameIndex") );

   CEikTextListBox* listbox = (CEikTextListBox*) Control(ECtlListboxPlayList);
   CDesCArray* listboxArray = ((CDesCArray*) listbox->Model()->ItemTextArray());

   TInt game = iEngine->GetRomIndex( listboxArray->operator[](listbox->CurrentItemIndex()) );

   __APP_DEBUG( _L("Dlg:GameIndex/%d"), game );

   return ( game );
}

TBool CPlayListDialog::OkToExitL( TInt aKeycode )
{
   __APP_DEBUG( _L("Dlg:OkToExitL/%d"), aKeycode );

   iGameToRun = GameIndex();

   return ( ETrue );
}

