#include <GC.h>
#include "shared.h"
#include "checker.raw.c"
#include "genplustxt.raw.c"
#include "controltxt.raw.c"
#include "controller.raw.c"

extern uint8 cart_rom[ ];
extern uint8 seccode[ ];

int ingame = 0;
int paused = 0;
GC_Pad gc_Pad;
int numpads; //Number of currently in use game pads

int update_input(void); //Updates input for the emulator
void emulatorloop(void); //The main emulator loop
void menu(void); //The menu
void rom_loader(void); //TODO: SD card loading
void gc_credits(void); //The credits
void pad_setup(void); //Controller Config
void load_pad_config(void); //Default Pad config, maybe a settings loader in future?
void gc_pause(void); //Time wasting loop. (Poor coding that never got replaced)
char* btnname(u16 button); //Returns the name of a button
void add_ggcode(void); //Add a game genie code screen
void wait_for_nobuttons(void); //Wait for no controller buttons to be pressed
void drawimage(int x, int y, int width, int height, const unsigned long *picture);
unsigned char *keystate;
unsigned long *FrameBuffer;
#define PALSIZE ((1<<15)-1)
unsigned long palette[PALSIZE];
void SetPalette();

void ChecksumCheck(); //Disclaimer screen and checksum check

#define TIME_ONE_MS 40000
#define TIME_SEC(s) ((s)*TIME_ONE_MS*1000)
#define TIME_FRAME 666666

#define REG_HOTRESET             (0xCC003024)
#define REG_HOTRESET_PTR         ((volatile u32*)REG_HOTRESET)
#define M_HOTRESET               *REG_HOTRESET_PTR = 0x00000000;

struct{
	u16 btnA;
	u16 btnB;
	u16 btnC;
	u16 btnX;
	u16 btnY;
	u16 btnZ;
	u16 btnStart;
	u16 btnMode;
} controller[2];

int main (int argc, char **argv)
{
	//Copy rom to safety
	//GC_Memory_memset(cart_rom, 0, 0x400000);
	//GC_Memory_memcpy(cart_rom, cart_file, 0x400000);

	/* Initialize GC stuff */
	// Initialize the system.
	GC_System_Init ();

	// Video and framebuffer initialization.
	FrameBuffer = (unsigned long*)0xC1000000;
	// Set NTSC mode.
	GC_Video_Init (VI_NTSC_640x480);
	GC_Debug_Init (FrameBuffer, VI_NTSC_640x480);
	// Set the framebuffer.
	//GC_Video_SetFrameBuffer ((void*)0x10080000, VI_FRAMEBUFFER_BOTH);
	*(unsigned long*)0xCC00201C = 0x10080000;
	*(unsigned long*)0xCC002024 = 0x10080000;
	// Clear framebuffer to black.
	GC_Video_ClearFrameBuffer (FrameBuffer, 0x00800080);
	// Wait two frames to allow TV to sync.
	GC_Video_WaitForVBlank ();
	GC_Video_WaitForVBlank ();
	GC_Memory_InitHeap((void*)0x81100000,(void*)0x81500000);
	/* Done with GC stuff */

	//Display the disclaimer
	ChecksumCheck();

	//Initialize the controllers, just in case.
	GC_PAD_Init();

	//The main emulator loop.
	emulatorloop();

    system_shutdown();

    return 0;
}

void emulatorloop(void){
	//u32 vidbuffer[153600];
	u16 genvideo[307200];
	u16 cenx, ceny;
	u16 lastw=0, lasth=0;
	u32 vidw, vidh;
	u32 vidx, vidy;
	u16 *mpointer;
	u8 audiostarted = 0;
	unsigned long tempcolor,pos,pos2;

	numpads = 1;

	GC_Time lasttime=0;
	GC_Time offby=0;
	u16 framecheck = 0;
	//Every second the auto frameskip will be behind 40 ticks.
	//This is used to sync it back with Vsync

	load_pad_config(); //Load the pad configurations
	SetPalette(); //Setup color pallete

	menu(); //Call the main menu up
	
	int running = 1;

	GC_Memory_memset((char*)&bitmap, 0, sizeof(t_bitmap));
    bitmap.width  = 640;
    bitmap.height = 480;
    bitmap.depth  = 15;
    bitmap.granularity = 2; /* Bytes per pixel */
    bitmap.pitch = (bitmap.width * bitmap.granularity);
    bitmap.data   = (unsigned char *)genvideo; /* Pointer to your bitmap */
    bitmap.viewport.w = 256; /* Initial size of display on power-up (do not change) */
    bitmap.viewport.h = 224;
    bitmap.viewport.x = 0x20; /* Offset used for rendering (do not change) */
    bitmap.viewport.y = 0;
    bitmap.remap = 1;

    system_init();
	audio_init(32000); //Genesis Plus Audio Init
	GC_AI_Init (); //GC audio init
	GC_AI_SetFreq32KHz(); //Set frequency

    system_reset();

	ingame = 1;
	//Setup frameskip
	GC_Video_WaitForVBlank();
	lasttime = GC_Time_GetTime(); //This will be the time synced to VBlank

	while(running)
	{
		running = update_input();

		if(!paused)
		{
			framecheck++;
			if(framecheck==15){
				lasttime -= 10; //This makes it wait 10ms longer to update next frame
				framecheck=0;
			}
            update_input();

			if((GC_Time_GetTime() - lasttime) > TIME_FRAME){
				//It's behind...
				offby = GC_Time_GetTime() - lasttime - TIME_FRAME;
				lasttime = GC_Time_GetTime() - offby;
				if(!system_frame(1)) //Skip frame
					system_reset();
			}
			else{
				while((GC_Time_GetTime() - lasttime) < TIME_FRAME);
				//Dont do anything till it's time to update video
				lasttime = GC_Time_GetTime();

				if(!system_frame(0)) //Update frame
					system_reset();

				//Video was resized, black out the background
				if ((lastw != bitmap.viewport.w) || (lasth != bitmap.viewport.h)) {
					GC_Video_ClearFrameBuffer(FrameBuffer, 0x00800080);
					lastw = bitmap.viewport.w;
					lasth = bitmap.viewport.h;
				}
				 
				cenx = (320 - bitmap.viewport.w)>>1; 
				ceny = (240 - bitmap.viewport.h)>>1;
				
				vidw = ((bitmap.viewport.w > 320) ? 320 : bitmap.viewport.w);
				vidh = ((bitmap.viewport.h > 224) ? 224 : bitmap.viewport.h);
				for (vidy = 0; vidy < vidh; vidy++) {
					pos = ((320*((vidy+ceny)<<1))+cenx);
					pos2 = (0x20+(vidy*640));
					for (vidx = 0; vidx < vidw; vidx++) { 
						tempcolor = palette[genvideo[pos2]];
						FrameBuffer[pos] = tempcolor;
						FrameBuffer[pos+320] = tempcolor;
						pos++;
						pos2++;
					}
				}
				//Triple Buffering :-(
				//GC_Memory_memcpy((void*)FrameBuffer,(void*)vidbuffer,614400);
			}

			mpointer = (u16*)0x81710000;
			//Merge left and right sound channels into a stereo sample
			for(vidx=0; vidx<snd.buffer_size;vidx++){
				 *mpointer++ = snd.buffer[0][vidx];
				 *mpointer++ = snd.buffer[1][vidx];
			}

			GC_Memory_DCFlushRange ((void*)0x81710000, snd.buffer_size*4);

			GC_AI_StopSample();
			GC_AI_LoadSample((u32*)0x81710000,snd.buffer_size*4);
			GC_AI_StartSample();
			if(!audiostarted){
				audiostarted = 1;
			}
        }

		if(paused){
			GC_AI_StopSample(); //Stop Audio
			audiostarted = 0;
			menu();
			GC_Video_ClearFrameBuffer (FrameBuffer, 0x00800080);
			paused = 0;
			framecheck = 0;
			lasttime = GC_Time_GetTime();
		}
		//End of While loop
    }
}

void menu(void){
	int checkerx, checkery;
	u8 redraw = 1;
	u8 sdload=0;

	if((*(char*)0x80001800==124) && (*(char*)0x80001801==96) && 
		(*(char*)0x80001802==0) && (*(char*)0x80001803==166)){
		sdload=1;
		//PSUL = 223 235 247 254
		//PSUL twice = 223 235 247 222
		//SDload = 124 96 0 166
	}


	GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad
	//Spit out the Demo Notice.
	while(!(gc_Pad.button & PAD_START))
	{
		GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad

		if(redraw){
			for (checkerx=0;checkerx<320;checkerx+=20)
				for (checkery=0;checkery<480;checkery+=40)
					drawimage(checkerx,checkery,20,40,checker_Bitmap);

			drawimage(15,25,85,40,genplus_Bitmap);

			//Main screen options
			GC_Debug_RStringT ("Start: Start Game", 0xFF80FF80, 160, 170);
			GC_Debug_RStringT ("Y:     Configure Controls", 0xFF80FF80, 160, 190);
			GC_Debug_RStringT ("X:     Credits", 0xFF80FF80, 160, 210);
			GC_Debug_RStringT ("A:     Add Game Genie code", 0xFF80FF80, 160, 230);
			if(ingame)
				GC_Debug_RStringT ("Z:     Reset the emulator", 0xFF80FF80, 160, 250);
			if(sdload)
				GC_Debug_RStringT ("L+R and C-stick Up: Quit to SDload", 0xFF80FF80, 160, 270);
			else
				GC_Debug_RStringT ("L+R and C-stick Up: Reboot Gamecube", 0xFF80FF80, 160, 270);

			GC_Debug_RStringT ("Genesis Plus v1.2 - GCN v1.1.1", 0xFF80FF80, 35, 420);

			redraw = 0;
		}
		GC_Video_WaitForVBlank ();
		if(gc_Pad.button & PAD_Y){
			pad_setup();
			redraw = 1;
		}
		if(gc_Pad.button & PAD_X){
			gc_credits();
			redraw=1;
		}
		if(gc_Pad.button & PAD_A){
			wait_for_nobuttons();
			add_ggcode();
			redraw = 1;
		}
		//Secret reset combo.
		if(gc_Pad.cy < -30){
			if((gc_Pad.button & PAD_L) && (gc_Pad.button & PAD_R))
			{
				if(sdload){
					asm ("lis 3, 0x8000");
					asm ("ori 3, 3, 0x1800");
					asm ("mtlr 3");
					asm ("blr");
				}
				else{
					M_HOTRESET;
				}
			}
		}
		if(ingame){
			if(gc_Pad.button & PAD_Z){
				while(gc_Pad.button & PAD_Z){
					GC_PAD_Read (&gc_Pad, GC_PAD1);	//Only reset once
				}
				system_reset();
			}
		}
	}

	
	while(gc_Pad.button & PAD_START){
		GC_PAD_Read (&gc_Pad, GC_PAD1);
	}
	//Pause till they let go.
}


void gc_credits(void){
	int checkerx, checkery;
	u8 redraw = 1;

	GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad
	//Spit out the Demo Notice.
	while(!(gc_Pad.button & PAD_B))
	{
		GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad

		GC_Video_WaitForVBlank ();
		if(redraw){
			for (checkerx=0;checkerx<320;checkerx+=20)
				for (checkery=0;checkery<480;checkery+=40)
					drawimage(checkerx,checkery,20,40,checker_Bitmap);

			drawimage(15,25,85,40,genplus_Bitmap);

			//Temporary Video output
			GC_Debug_RStringT ("Original Emulator:", 0xFF80FF80, 212, 100);
			GC_Debug_RStringT ("Charles Mac Donald", 0xFF80FF80, 212, 120);
			GC_Debug_RStringT ("The following people contributed", 0xFF80FF80, 134, 180);
			GC_Debug_RStringT ("code to the Gamecube port:", 0xFF80FF80, 164, 200);
			GC_Debug_RStringT ("Sappharad", 0xFF80FF80, 266, 240);
			GC_Debug_RStringT ("Parasyte", 0xFF80FF80, 272, 260);
			GC_Debug_RStringT ("Richard Bannister", 0xFF80FF80, 220, 280);
			GC_Debug_RStringT ("Costis", 0xFF80FF80, 284, 300);
			GC_Debug_RStringT ("Green Darkness", 0xFF80FF80, 236, 320);
			GC_Debug_RStringT ("BlackAura", 0xFF80FF80, 266, 340);

			redraw = 0;
		}
	}
}


void pad_setup(void){
	int checkerx, checkery;
	int mnuitem = 0;
	u8 pausenext = 0;
	int mypad = 0;
	u8 redraw = 1;

	GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad
	while(!(gc_Pad.button & PAD_B))
	{
		if(redraw){
			GC_Video_WaitForVBlank (); //Wait BEFORE you draw.
			for (checkerx=0;checkerx<320;checkerx+=20)
				for (checkery=0;checkery<480;checkery+=40)
					drawimage(checkerx,checkery,20,40,checker_Bitmap);

			drawimage(15,25,145,50,control_Bitmap);
			drawimage(61,228,200,220,cntrlr_Bitmap);

			if(numpads == 1){
				GC_Debug_RStringT ("Player 1", 0x952B9515, 165, 80);
			}
			else{
				//Show selected pad
				if(mypad==0){
					GC_Debug_RStringT ("Player 1", 0x952B9515, 165, 80);
					GC_Debug_RStringT ("Player 2", 0xFF80FF80, 365, 80);
				}
				else{
					GC_Debug_RStringT ("Player 1", 0xFF80FF80, 165, 80);
					GC_Debug_RStringT ("Player 2", 0x952B9515, 365, 80);
				}
			}

			
			if(mnuitem == 0)
				GC_Debug_RStringT ("A:", 0x952B9515, 50, 120);
			else
				GC_Debug_RStringT ("A:", 0xFF80FF80, 50, 120);
			GC_Debug_RStringT (btnname(controller[mypad].btnA), 0xFF80FF80, 134, 120);
			if(mnuitem == 1)
				GC_Debug_RStringT ("B:", 0x952B9515, 50, 140);
			else
				GC_Debug_RStringT ("B:", 0xFF80FF80, 50, 140);
			GC_Debug_RStringT (btnname(controller[mypad].btnB), 0xFF80FF80, 134, 140);
			if(mnuitem == 2)
				GC_Debug_RStringT ("C:", 0x952B9515, 50, 160);
			else
				GC_Debug_RStringT ("C:", 0xFF80FF80, 50, 160);
			GC_Debug_RStringT (btnname(controller[mypad].btnC), 0xFF80FF80, 134, 160);
			if(mnuitem == 3)
				GC_Debug_RStringT ("X:", 0x952B9515, 225, 120);
			else
				GC_Debug_RStringT ("X:", 0xFF80FF80, 225, 120);
			GC_Debug_RStringT (btnname(controller[mypad].btnX), 0xFF80FF80, 309, 120);
			if(mnuitem == 4)
				GC_Debug_RStringT ("Y:", 0x952B9515, 225, 140);
			else
				GC_Debug_RStringT ("Y:", 0xFF80FF80, 225, 140);
			GC_Debug_RStringT (btnname(controller[mypad].btnY), 0xFF80FF80, 309, 140);
			if(mnuitem == 5)
				GC_Debug_RStringT ("Z:", 0x952B9515, 225, 160);
			else
				GC_Debug_RStringT ("Z:", 0xFF80FF80, 225, 160);
			GC_Debug_RStringT (btnname(controller[mypad].btnZ), 0xFF80FF80, 309, 160);
			if(mnuitem == 6)
				GC_Debug_RStringT ("Start:", 0x952B9515, 400, 120);
			else
				GC_Debug_RStringT ("Start:", 0xFF80FF80, 400, 120);
			GC_Debug_RStringT (btnname(controller[mypad].btnStart), 0xFF80FF80, 484, 120);
			if(mnuitem == 7)
				GC_Debug_RStringT ("Mode:", 0x952B9515, 400, 140);
			else
				GC_Debug_RStringT ("Mode:", 0xFF80FF80, 400, 140);
			GC_Debug_RStringT (btnname(controller[mypad].btnMode), 0xFF80FF80, 484, 140);
			
			redraw = 0;
		}
		
		if(pausenext){
			gc_pause();
			pausenext=0;
		}

		GC_PAD_Read (&gc_Pad, GC_PAD2);
		if((numpads == 1) && (gc_Pad.button & PAD_START)){
			//They pressed start on player 2 (Thanks for the idea GrnDrk)
			numpads = 2;
			redraw = 1;
		}

		for(checkerx=0; checkerx<numpads;checkerx++){
			//Check Menu move for both players
			//Note: Not checking right now. Bad values when controller not plugged in.
			GC_PAD_Read (&gc_Pad, checkerx);
			if ((gc_Pad.button & PAD_UP) || (gc_Pad.y < -15)){
				if(mnuitem > 0)
					mnuitem--;
				pausenext=1;
				redraw=1;
			}
			if ((gc_Pad.button & PAD_DOWN) || (gc_Pad.y > 15)){
				if(mnuitem < 7)
					mnuitem++;
				pausenext=1;
				redraw=1;
			}
			if ((gc_Pad.button & PAD_LEFT) || (gc_Pad.x < -15)){
				mypad = 0;
				redraw=1;
			}
			
			if (numpads == 2){
				if ((gc_Pad.button & PAD_RIGHT) || (gc_Pad.x > 15)){
					mypad = 1;
					redraw=1;
				}
			}
		}
		
		GC_PAD_Read (&gc_Pad, mypad); //Read the pad

		if (gc_Pad.button & PAD_A){
			//They want to set a new button
			GC_Debug_RStringT ("Press the new button to set", 0x952B9515, 150, 185);
			GC_Video_WaitForVBlank ();
			while(gc_Pad.button & PAD_A){
				GC_PAD_Read (&gc_Pad, mypad); //Read the pad
			}
			while (!(gc_Pad.button & (PAD_A | PAD_B | PAD_X | PAD_Y | PAD_Z | PAD_START | PAD_L | PAD_R))){
				//No digital buttons being pressed.
				GC_PAD_Read (&gc_Pad, mypad); //Read the pad
				GC_Video_WaitForVBlank ();
			}
			u16 mybutton;
			if(gc_Pad.button & PAD_A)
				mybutton = PAD_A;
			else if (gc_Pad.button & PAD_B)
				mybutton = PAD_B;
			else if (gc_Pad.button & PAD_X)
				mybutton = PAD_X;
			else if (gc_Pad.button & PAD_Y)
				mybutton = PAD_Y;
			else if (gc_Pad.button & PAD_Z)
				mybutton = PAD_Z;
			else if (gc_Pad.button & PAD_L)
				mybutton = PAD_L;
			else if (gc_Pad.button & PAD_R)
				mybutton = PAD_R;
			else
				mybutton = PAD_START;
			switch (mnuitem){
			case 0:
				controller[mypad].btnA = mybutton;
				break;
			case 1:
				controller[mypad].btnB = mybutton;
				break;
			case 2:
				controller[mypad].btnC = mybutton;
				break;
			case 3:
				controller[mypad].btnX = mybutton;
				break;
			case 4:
				controller[mypad].btnY = mybutton;
				break;
			case 5:
				controller[mypad].btnZ = mybutton;
				break;
			case 6:
				controller[mypad].btnStart = mybutton;
				break;
			case 7:
				controller[mypad].btnMode = mybutton;
				break;
			}
			GC_PAD_Read (&gc_Pad, mypad);
			while (gc_Pad.button & (PAD_A | PAD_B | PAD_X | PAD_Y | PAD_Z | PAD_START | PAD_L | PAD_R)){
				//A button is still being pressed
				GC_PAD_Read (&gc_Pad, mypad);
				GC_Video_WaitForVBlank ();
			}
			redraw=1;
		}
		
		GC_PAD_Read (&gc_Pad, GC_PAD1); //Only Controller 1 can exit config
	}

}

void wait_for_nobuttons(void){
	GC_PAD_Read (&gc_Pad, 0);
	while (gc_Pad.button & (PAD_A | PAD_B | PAD_X | PAD_Y | PAD_Z | PAD_START | PAD_L | PAD_R)){
		//A button is still being pressed
		GC_PAD_Read (&gc_Pad, 0);
		GC_Video_WaitForVBlank ();
	}
}

void drawimage(int x, int y, int width, int height, const unsigned long *picture)
{
	int counter = 0;
	int thex;
	int they;

	for (thex = 0; thex < height; thex++)
		for (they = 0; they < width; they++)
		{
			if((x+they >= 0) && (x+they < 319))
				if((y+thex >=0) && (y+thex < 479)){
					FrameBuffer[(320*(y+thex))+(((they*2)+(x*2))>>1)] = picture[counter];
					FrameBuffer[(320*(y+thex))+(((they*2)+(x*2))>>1)+1] = picture[counter];
				}
			counter++;
		}
}

void gc_pause(void){
	u32 i,x;

	x=0;
	for (i=0;i<0x3FFFFFF;i++)
		x=i-x;
}

int update_input(void)
{
    int running = 1;
	int padno;

    memset((char*)&input, 0, sizeof(t_input));

	for(padno=0; padno<numpads; padno++){
		GC_PAD_Read (&gc_Pad, padno);
		if((gc_Pad.button & PAD_UP) || (gc_Pad.y < -30))	input.pad[padno] |= INPUT_UP;
		else if((gc_Pad.button & PAD_DOWN) || (gc_Pad.y > 30))	input.pad[padno] |= INPUT_DOWN;
		if((gc_Pad.button & PAD_LEFT) || (gc_Pad.x < -30))	input.pad[padno] |= INPUT_LEFT;
		else if((gc_Pad.button & PAD_RIGHT) || (gc_Pad.x > 30))	input.pad[padno] |= INPUT_RIGHT;
		if(gc_Pad.button & controller[padno].btnA)	input.pad[padno] |= INPUT_A;
		if(gc_Pad.button & controller[padno].btnB)	input.pad[padno] |= INPUT_B;
		if(gc_Pad.button & controller[padno].btnC)	input.pad[padno] |= INPUT_C;
		if(gc_Pad.button & controller[padno].btnStart)	input.pad[padno] |= INPUT_START;
		if(gc_Pad.button & controller[padno].btnX)	input.pad[padno] |= INPUT_X;
		if(gc_Pad.button & controller[padno].btnY)	input.pad[padno] |= INPUT_Y;
		if(gc_Pad.button & controller[padno].btnZ)	input.pad[padno] |= INPUT_Z;
		if(gc_Pad.button & controller[padno].btnMode)	input.pad[padno] |= INPUT_MODE;
	}

	GC_PAD_Read (&gc_Pad, 0);
	if(gc_Pad.cy > 30) //Down on Cstick
		paused = 1;

    return (running);
}

void add_ggcode(void){
	u8 redraw = 1;
	u16 checkerx, checkery;
	char ggletters[32];
	char code[8];
	u8 count, selection, entry, myval;
	u8 bitline[40];
	u32 address;
	u16 value;

	for(count=0; count<8;count++)
		code[count] = '_';

	selection = 0;
	entry=0;
	ggletters[0] = 'A'; ggletters[16] = 'T';
	ggletters[1] = 'B'; ggletters[17] = 'V';
	ggletters[2] = 'C'; ggletters[18] = 'W';
	ggletters[3] = 'D'; ggletters[19] = 'X';
	ggletters[4] = 'E'; ggletters[20] = 'Y';
	ggletters[5] = 'F'; ggletters[21] = 'Z';
	ggletters[6] = 'G'; ggletters[22] = '0';
	ggletters[7] = 'H'; ggletters[23] = '1';
	ggletters[8] = 'J'; ggletters[24] = '2';
	ggletters[9] = 'K'; ggletters[25] = '3';
	ggletters[10] = 'L'; ggletters[26] = '4';
	ggletters[11] = 'M'; ggletters[27] = '5';
	ggletters[12] = 'N'; ggletters[28] = '6';
	ggletters[13] = 'P'; ggletters[29] = '7';
	ggletters[14] = 'R'; ggletters[30] = '8';
	ggletters[15] = 'S'; ggletters[31] = '9';

	GC_PAD_Read (&gc_Pad, 0);
	while(!(gc_Pad.button & PAD_START)){
		GC_Video_WaitForVBlank ();//Wait BEFORE you draw.
		if(redraw){
			for (checkerx=0;checkerx<320;checkerx+=20)
				for (checkery=0;checkery<480;checkery+=40)
					drawimage(checkerx,checkery,20,40,checker_Bitmap);

			GC_Debug_RStringT ("Add a Game Genie code", 0xFF80FF80, 25, 25);

			count = 0;
			for (checkery=0; checkery<4; checkery++)
				for (checkerx=0; checkerx < 8; checkerx++){
					if(count==selection)
						GC_Debug_RCharT(ggletters[count], 0x952B9515, 180+(checkerx*36),80+(checkery*40));
					else
						GC_Debug_RCharT(ggletters[count], 0xFF80FF80, 180+(checkerx*36),80+(checkery*40));
					count++;

				}

			for(count=0; count<4;count++)
				if(entry==count)
					GC_Debug_RCharT (code[count], 0x952B9515, 250+(count*12), 250);
				else
					GC_Debug_RCharT (code[count], 0xFF80FF80, 250+(count*12), 250);
			GC_Debug_RCharT ('-', 0xFF80FF80, 298, 250);
			for(count=4; count<8;count++)
				if(entry==count)
					GC_Debug_RCharT (code[count], 0x952B9515, 262+(count*12), 250);
				else
					GC_Debug_RCharT (code[count], 0xFF80FF80, 262+(count*12), 250);

			GC_Debug_RStringT("Use the Dpad to select a letter, press A to", 0xFF80FF80, 62, 300);
			GC_Debug_RStringT("enter it. Press B to delete a letter. Press", 0xFF80FF80, 62, 320);
			GC_Debug_RStringT("Start to add the code. If no code is entered", 0xFF80FF80, 56, 340);
			GC_Debug_RStringT("pressing Start will return you to the menu.", 0xFF80FF80, 62, 360);
			redraw = 0;
		}

		GC_PAD_Read (&gc_Pad, 0);
		if ((gc_Pad.button & PAD_UP) || (gc_Pad.y < -20)){
			if(selection > 7)
				selection-=8;
			//Wait for the user to let go
			redraw=1;
			while((gc_Pad.button & PAD_UP) || (gc_Pad.y < -20)){
				GC_PAD_Read (&gc_Pad, 0);
				GC_Video_WaitForVBlank ();
			}
		}
		if ((gc_Pad.button & PAD_DOWN) || (gc_Pad.y > 20)){
			if(selection < 24)
				selection+=8;
			redraw=1;
			while((gc_Pad.button & PAD_DOWN) || (gc_Pad.y > 20)){
				GC_PAD_Read (&gc_Pad, 0);
				GC_Video_WaitForVBlank ();
			}
		}
		if ((gc_Pad.button & PAD_LEFT) || (gc_Pad.x < -20)){
			if(selection > 0)
				selection--;
			redraw=1;
			while((gc_Pad.button & PAD_LEFT) || (gc_Pad.x < -20)){
				GC_PAD_Read (&gc_Pad, 0);
				GC_Video_WaitForVBlank ();
			}
		}
		if ((gc_Pad.button & PAD_RIGHT) || (gc_Pad.x > 20)){
			if(selection < 31)
				selection++;
			redraw=1;
			while((gc_Pad.button & PAD_RIGHT) || (gc_Pad.x > 20)){
				GC_PAD_Read (&gc_Pad, 0);
				GC_Video_WaitForVBlank ();
			}
		}
		if (gc_Pad.button & PAD_A){
			code[entry] = ggletters[selection];
			if(entry<7)
				entry++;
			wait_for_nobuttons();
			redraw=1;
		}
		if (gc_Pad.button & PAD_B){
			code[entry] = '_';
			if(entry>0)
				entry--;
			wait_for_nobuttons();
			redraw=1;
		}

		//This is just here to remind me that this is the end of the wait for start button loop
		GC_PAD_Read (&gc_Pad, 0);
	}

	while(gc_Pad.button & PAD_START){
		GC_PAD_Read (&gc_Pad, 0);
	}

	myval=0; //So the compiler wont complain
	if(code[7] != '_'){
		//The code was entered, decrypt and add it.
		for(count=0; count<8; count++){
			for(checkerx=0; checkerx<32; checkerx++)
				if(code[count] == ggletters[checkerx]){
					myval = checkerx;
					break; //There's not really a need to speed this up
				}
			bitline[(count*5)+0] = myval >> 4;
			bitline[(count*5)+1] = (myval >> 3) & 1;
			bitline[(count*5)+2] = (myval >> 2) & 1;
			bitline[(count*5)+3] = (myval >> 1) & 1;
			bitline[(count*5)+4] = myval & 1;
		}
		//Everything should be in the bitline. Now we just need to arrange it.
		address = bitline[16] << 23;
		address += (bitline[17] << 22);
		address += (bitline[18] << 21);
		address += (bitline[19] << 20);
		address += (bitline[20] << 19);
		address += (bitline[21] << 18);
		address += (bitline[22] << 17);
		address += (bitline[23] << 16);
		address += (bitline[8] << 15);
		address += (bitline[9] << 14);
		address += (bitline[10] << 13);
		address += (bitline[11] << 12);
		address += (bitline[12] << 11);
		address += (bitline[13] << 10);
		address += (bitline[14] << 9);
		address += (bitline[15] << 8);
		address += (bitline[32] << 7);
		address += (bitline[33] << 6);
		address += (bitline[34] << 5);
		address += (bitline[35] << 4);
		address += (bitline[36] << 3);
		address += (bitline[37] << 2);
		address += (bitline[38] << 1);
		address += bitline[39];

		value = bitline[29] << 15;
		value += (bitline[30] << 14);
		value += (bitline[31] << 13);
		value += (bitline[24] << 12);
		value += (bitline[25] << 11);
		value += (bitline[26] << 10);
		value += (bitline[27] << 9);
		value += (bitline[28] << 8);
		value += (bitline[0] << 7);
		value += (bitline[1] << 6);
		value += (bitline[2] << 5);
		value += (bitline[3] << 4);
		value += (bitline[4] << 3);
		value += (bitline[5] << 2);
		value += (bitline[6] << 1);
		value += bitline[7];

		//Address and value decrypted.

		cart_rom[address] = value;
	}

}

char* btnname(u16 button){
	if(button & PAD_A)
		return "A";
	else if(button & PAD_B)
		return "B";
	else if(button & PAD_X)
		return "X";
	else if(button & PAD_Y)
		return "Y";
	else if(button & PAD_Z)
		return "Z";
	else if(button & PAD_L)
		return "L";
	else if(button & PAD_R)
		return "R";
	else if(button & PAD_START)
		return "Start";
	else
		return "???";
}

void SetPalette() {
	int i;
	unsigned char r, g, b;

	for (i = 0; i < PALSIZE; i++) {
		r = ((i >> 7) & 0xF8);
		g = ((i >> 2) & 0xF8);
		b = ((i << 3) & 0xF8);
		palette[i] = GC_Video_RGBToYCbCr(r, g, b);
	}
}

void load_pad_config(void){
	int x;

	for (x=0; x<2; x++){
		controller[x].btnA = PAD_B;
		controller[x].btnB = PAD_A;
		controller[x].btnC = PAD_X;
		controller[x].btnX = PAD_L;
		controller[x].btnY = PAD_R;
		controller[x].btnZ = PAD_Y;
		controller[x].btnStart = PAD_START;
		controller[x].btnMode = PAD_Z;
	}
}

void ChecksumCheck(){
	int romsize = 0;
	int temp;
	uint8 mycode[12];
	int matching = 1;

	romsize += (seccode[0] << 24);
	romsize += (seccode[1] << 16);
	romsize += (seccode[2] << 8);
	romsize += seccode[3];

	//Generate local checksum code
	for(temp = 0; temp < 12; temp++)
		mycode[temp]=8*(temp+4);
	for(temp = 0; temp < romsize; temp++)
		mycode[temp%12] ^= cart_rom[temp];
	for(temp=0; temp<12; temp++)
		mycode[temp] ^= (0xFF - ((temp+4)*8));

	//Check codes to see if they match
	for(temp=0; temp<12; temp++){
		if(mycode[temp] != seccode[temp+4]){
			matching = 0;
			break;
		}
	}

	if(romsize < 0x1000)
		matching=0; //Prevent cheating the checksum

	GC_Debug_RStringT ("This emulator is free, and open source.", 0xFF80FF80, 40, 60);
	GC_Debug_RStringT ("It should not be sold, or bundled with", 0xFF80FF80, 40, 80);
	GC_Debug_RStringT ("illegal software or rom images. Also, if", 0xFF80FF80, 40, 100);
	GC_Debug_RStringT ("you suffer from health problems because", 0xFF80FF80, 40, 120);
	GC_Debug_RStringT ("of playing games in this emulator, it's", 0xFF80FF80, 40, 140);
	GC_Debug_RStringT ("your fault because you dont know how to", 0xFF80FF80, 40, 160);
	GC_Debug_RStringT ("read the health and saftey warnings in your", 0xFF80FF80, 40, 180);
	GC_Debug_RStringT ("Gamecube's instruction manual. To agree", 0xFF80FF80, 40, 200);
	GC_Debug_RStringT ("that you are using this emulator fairly", 0xFF80FF80, 40, 220);
	GC_Debug_RStringT ("hold down the L and R triggers on your", 0xFF80FF80, 40, 240);
	GC_Debug_RStringT ("controller. Thank you.", 0xFF80FF80, 40, 260);

	if(matching != 1){
		GC_Debug_RStringT ("Also, it appears that this DOL is", 0xFF80FF80, 40, 300);
		GC_Debug_RStringT ("either corrupted, or has been modified", 0xFF80FF80, 40, 320);
		GC_Debug_RStringT ("from its original form. It may not function", 0xFF80FF80, 40, 340);
		GC_Debug_RStringT ("as intended. Please be aware of this.", 0xFF80FF80, 40, 360);
	}
	
	GC_Video_WaitForVBlank();

	GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad
	//Wait to make sure they're not holding it down when the emulator starts
	while(((gc_Pad.button & PAD_L) && (gc_Pad.button & PAD_R))){
		GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad
	}
	//This is the real read
	GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad
	while(!((gc_Pad.button & PAD_L) && (gc_Pad.button & PAD_R))){
		GC_PAD_Read (&gc_Pad, GC_PAD1); //Read the pad
	}

}

