//-----------------------------------------------------------------------------
// MEKA - keyboard.c
// SK-1100 (Sega Keyboard) / SC-3000 Keyboard Emulation - Code
//-----------------------------------------------------------------------------

#include "shared.h"
#include "inputs_t.h"
#include "keyboard.h"
#include "skin_bg.h"

//-----------------------------------------------------------------------------
// Data
//-----------------------------------------------------------------------------

typedef struct
{
    int     row;
    int     bit;
    char    desc[16];
} t_sk1100_key;

#define SK1100_KEYS_NUM (62)

//-----------------------------------------------------------------------------

static t_sk1100_key SK1100_Keys [SK1100_KEYS_NUM] =
{
  // Row: 0 -------------------------------------------------------------------
  {  0, 0x0001, "1"              }, //  0
  {  0, 0x0002, "Q"              }, //  1
  {  0, 0x0004, "A"              }, //  2
  {  0, 0x0008, "Z"              }, //  3
  {  0, 0x0010, "Eng Dier's"     }, //  4
  {  0, 0x0020, ","              }, //  5
  {  0, 0x0040, "K"              }, //  6
  {  0, 0x0080, "I"              }, //  7
  {  0, 0x0100, "8"              }, //  8
  // Row: 1 -------------------------------------------------------------------
  {  1, 0x0001, "2"              }, //  9
  {  1, 0x0002, "W"              }, // 10
  {  1, 0x0004, "S"              }, // 11
  {  1, 0x0008, "X"              }, // 12
  {  1, 0x0010, "Space"          }, // 13
  {  1, 0x0020, "."              }, // 14
  {  1, 0x0040, "L"              }, // 15
  {  1, 0x0080, "O"              }, // 16
  {  1, 0x0100, "9"              }, // 17
  // Row: 2 -------------------------------------------------------------------
  {  2, 0x0001, "3"              }, // 18
  {  2, 0x0002, "E"              }, // 19
  {  2, 0x0004, "D"              }, // 20
  {  2, 0x0008, "C"              }, // 21
  {  2, 0x0010, "Home/Clear"     }, // 22
  {  2, 0x0020, "/"              }, // 23
  {  2, 0x0040, ";"              }, // 24
  {  2, 0x0080, "P"              }, // 25
  {  2, 0x0100, "0"              }, // 26
  // Row: 3 -------------------------------------------------------------------
  {  3, 0x0001, "4"              }, // 27
  {  3, 0x0002, "R"              }, // 28
  {  3, 0x0004, "F"              }, // 29
  {  3, 0x0008, "V"              }, // 30
  {  3, 0x0010, "Insert/Delete"  }, // 31
  {  3, 0x0020, "Pi"             }, // 32
  {  3, 0x0040, ":"              }, // 33
  {  3, 0x0080, "@"              }, // 34
  {  3, 0x0100, "-"              }, // 35
  // Row: 4 -------------------------------------------------------------------
  {  4, 0x0001, "5"              }, // 36
  {  4, 0x0002, "T"              }, // 37
  {  4, 0x0004, "G"              }, // 38
  {  4, 0x0008, "B"              }, // 39
  // 4, 0x0010, <UNUSED>
  {  4, 0x0020, "Down Arrow"     }, // 40
  {  4, 0x0040, "]"              }, // 41
  {  4, 0x0080, "["              }, // 42
  {  4, 0x0100, "^"              }, // 43
  // Row: 5 -------------------------------------------------------------------
  {  5, 0x0001, "6"              }, // 44
  {  5, 0x0002, "Y"              }, // 45
  {  5, 0x0004, "H"              }, // 46
  {  5, 0x0008, "N"              }, // 47
  // 5, 0x0010, <UNUSED>
  {  5, 0x0020, "Left Arrow"     }, // 48
  {  5, 0x0040, "Return"         }, // 49
  // 5, 0x0080, <UNUSED>
  {  5, 0x0100, ""              }, // 50
  // 5, 0x0200, <UNUSED>
  // 5, 0x0400, <UNUSED>
  {  5, 0x0800, "Func"           }, // 51
  // Row: 6 -------------------------------------------------------------------
  {  6, 0x0001, "7"              }, // 52
  {  6, 0x0002, "U"              }, // 53
  {  6, 0x0004, "J"              }, // 54
  {  6, 0x0008, "M"              }, // 55
  // 6, 0x0010, <UNUSED>
  {  6, 0x0020, "Right Arrow"    }, // 56
  {  6, 0x0040, "Up Arrow"       }, // 57
  // 6, 0x0080, <UNUSED>
  {  6, 0x0100, "Break"          }, // 58
  {  6, 0x0200, "Graph"          }, // 59
  {  6, 0x0400, "Ctrl"           }, // 60
  {  6, 0x0800, "Shift"          }  // 61
};

//-----------------------------------------------------------------------------

typedef struct
{
    int   key_pc;
    int   key_sk1100;
} t_sk1100_map;

#define SK1100_MAPPING_NUM (62+5)

static  t_sk1100_map SK1100_Mapping [SK1100_MAPPING_NUM] =
{
    // Row: 0 -------------------------------------------------------------------
    { KEY_1,               0      },
    { KEY_Q,               1      },
    { KEY_A,               2      },
    { KEY_Z,               3      },
    { KEY_ALT,             4      }, // Eng Dier's
    { KEY_COMMA,           5      },
    { KEY_K,               6      },
    { KEY_I,               7      },
    { KEY_8,               8      },
    // Row: 1 -------------------------------------------------------------------
    { KEY_2,               9      },
    { KEY_W,              10      },
    { KEY_S,              11      },
    { KEY_X,              12      },
    { KEY_SPACE,          13      },
    { KEY_STOP,           14      }, // .
    { KEY_L,              15      },
    { KEY_O,              16      },
    { KEY_9,              17      },
    // Row: 2 -------------------------------------------------------------------
    { KEY_3,              18      },
    { KEY_E,              19      },
    { KEY_D,              20      },
    { KEY_C,              21      },
    { KEY_ALTGR,          22      }, // Home Clear
    { KEY_SLASH,          23      },
    { KEY_COLON,          24      },
    { KEY_P,              25      },
    { KEY_0,              26      },
    // Row: 3 -------------------------------------------------------------------
    { KEY_4,              27      },
    { KEY_R,              28      },
    { KEY_F,              29      },
    { KEY_V,              30      },
    { KEY_RCONTROL,       31      }, // Insert/Delete
    { KEY_BACKSPACE,      31      }, // Insert/Delete ++
    { KEY_ABNT_C1,        32      }, // Pi
    { KEY_SLASH_PAD,      32      }, // Pi ++
    { KEY_BACKSLASH2,     32      }, // Pi ++
    { KEY_QUOTE,          33      }, // :
    { KEY_OPENBRACE,      34,     }, // @
    { KEY_MINUS,          35      },
    // Row: 4 -------------------------------------------------------------------
    { KEY_5,              36      },
    { KEY_T,              37      },
    { KEY_G,              38      },
    { KEY_B,              39      },
    { KEY_DOWN,           40      }, // Down Arrow
    { KEY_BACKSLASH,      41      }, // ]
    { KEY_CLOSEBRACE,     42      }, // [
    { KEY_EQUALS,         43      }, // ^
    // Row: 5 -------------------------------------------------------------------
    { KEY_6,              44      },
    { KEY_Y,              45      },
    { KEY_H,              46      },
    { KEY_N,              47      },
    { KEY_LEFT,           48      }, // Left Arrow
    { KEY_ENTER,          49      }, // Return
    { KEY_YEN,            50,     }, // Yen
    { KEY_TILDE,          50,     }, // Yen ++
    { KEY_TAB,            51      }, // Func
    // Row: 6 -------------------------------------------------------------------
    { KEY_7,              52      },
    { KEY_U,              53      },
    { KEY_J,              54      },
    { KEY_M,              55      },
    { KEY_RIGHT,          56      }, // Right Arrow
    { KEY_UP,             57      }, // Up Arrow
    { KEY_END,            58      }, // Break             (!!?)
    { KEY_LCONTROL,       59      }, // Graph
    { KEY_CAPSLOCK,       60      }, // Ctrl
    { KEY_LSHIFT,         61      }, // Shift
    { KEY_RSHIFT,         61      }, // Shift ++
};

//-----------------------------------------------------------------------------
// Functions
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Keyboard_Switch ()
// Enable/disable SK-1100 emulation
//-----------------------------------------------------------------------------
void    Keyboard_Switch (void)
{
    Inputs.Keyboard_Enabled ^= 1;
    gui_menu_inverse_check (menus_ID.inputs, 5);
    Skins_Background_Redraw();
    gui.info.must_redraw = TRUE;
    if (Inputs.Keyboard_Enabled)
        Msg (MSGT_USER, Msg_Get (MSG_Inputs_SK1100_Enabled));
    else
        Msg (MSGT_USER, Msg_Get (MSG_Inputs_SK1100_Disabled));
}

//-----------------------------------------------------------------------------
// Keyboard_Emulation_Clear ()
// Clear SK-1100 data for emulation
//-----------------------------------------------------------------------------
void    Keyboard_Emulation_Clear (void)
{
    int i;
    for (i = 0; i < 7; i ++)
        tsms.Control [i] = 0xFFFF;
}

//-----------------------------------------------------------------------------
// Keyboard_Emulation_Update ()
// Update SK-1100 data for emulation
//-----------------------------------------------------------------------------
void    Keyboard_Emulation_Update (void)
{
    int i;
    Keyboard_Emulation_Clear();
    for (i = 0; i != SK1100_MAPPING_NUM; i ++)
    {
        if (key [SK1100_Mapping[i].key_pc])
        {
            t_sk1100_key *sk1100_key = &SK1100_Keys[SK1100_Mapping[i].key_sk1100];
            tsms.Control [sk1100_key->row] &= (~sk1100_key->bit);
            tsms.Control_Check_GUI = FALSE;
        }
    }
}

//-----------------------------------------------------------------------------

