/*///////////////////////////////////////////////////////////////////////////
// Name:        configuration.cpp
// Purpose:     wxWindows wxOsmose Front-End to the Osmose Emulator
// Author:      Ferreira Yohann
// Modified by:
// Created:     25/01/2005
// Copyright:   (c) Ferreira Yohann
// Licence:     GPL
///////////////////////////////////////////////////////////////////////////*/

#include "configuration.h" /* Header */
#include <wx/textfile.h> /* For wxTextFile class support */

#include "../SDL/SDL_keysym.h" /* For SDL Keys interpretations */

#include <iostream>

Configuration::Configuration()
{
	/* In case of Init() fails : */
	
	/* Video */
	FullScreenEnabled = false;
	TVLikeEnabled = false;
	DarkPaletteEnabled = false;
	MonochromEnabled = false;
	Scale2xEnabled = false;
	FPSDisplayEnabled = false;
	BilinearFilteringEnabled = false;
	
	/* Audio */
	SoundEnabled = true;
	
	/* Zone */
	ExportedModeEnabled = true;
	JapaneseModeEnabled = false;
	PALModeEnabled = false;
	
	/* Options */
	GamepadEnabled = false;
	MousepadEnabled = false;
	MousepadAcceleration = 0.75;
	
	/* Paths */
	User = UserHomeDirectory = IniFilename = _T("");
	
	Screenshot = SDLK_s;
	Tileshot = SDLK_t;
	Soundshot = SDLK_F1;
	Pause = SDLK_p;
	Quit = SDLK_ESCAPE;
	
	P1_Up = SDLK_UP;
	P1_Down = SDLK_DOWN;
	P1_Left = SDLK_LEFT;
	P1_Right = SDLK_RIGHT;
	P1_A = SDLK_LCTRL;
	P1_B = SDLK_LALT;
	
	P2_Up = SDLK_KP5;
	P2_Down = SDLK_KP2;
	P2_Left = SDLK_KP1;
	P2_Right = SDLK_KP3;
	P2_A = SDLK_n;
	P2_B = SDLK_b;
	
	#ifdef _DEBUG_CONSOLE_
	DebugKey = SDLK_d;
	#endif
	
	Init();
}

void Configuration::Init()
{
	/* Let's load preferences in the file which is in the ~/.osmose directory under Unix/Linux or in the same
	// directory under Windows and else
	// For Unix / Linux, we are after the user home directory. */
	#ifdef __USE_UNIX98
		UserHomeDirectory = wxGetHomeDir();
		User = wxGetUserId();
		IniFilename = UserHomeDirectory;
		IniFilename += _T("/.osmose/wxosmose.ini");
	#else /* Win and others */
		IniFilename = _T("wxosmose.ini");
	#endif
	
	#ifdef _DEBUG_
		wxLogMessage(IniFilename);
	#endif
	
	/* Preferences at start */
	wxTextFile MyConfigFile(IniFilename); /* The File Pointer */
	MyConfigFile.Open();
	wxString CurrentLine; /* The Line wich is being read. */

	if( !MyConfigFile.IsOpened() )
	{
		#ifdef _DEBUG_
			wxLogMessage( _T("No Ini File. Using default configuration..."));
		#endif
		MyConfigFile.Close();
		WriteConfig();
	}
	else /* Config existing */
	{
		unsigned int LineIndex = 0;
		while ( LineIndex < MyConfigFile.GetLineCount() ) 
		{
			CurrentLine = MyConfigFile.GetLine(LineIndex);
			LineIndex++;
			
			/* Video */
			if ( !(CurrentLine.substr(0,10)).compare(_T("FullScreen")) )
			{
				/* Fullscreen */
				if ( !(CurrentLine.substr(11,5)).compare(_T("true")) )  /* fullscreen on */
				{
					FullScreenEnabled = true;
				}
				else
				{
					FullScreenEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,6)).compare(_T("TVMode")) )
			{
				/* TV-Mode */
				if ( !(CurrentLine.substr(7,5)).compare(_T("true")) )  /* on */
				{
					TVLikeEnabled = true;
				}
				else
				{
					TVLikeEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,7)).compare(_T("Scale2x")) )
			{
				/* Scale2x */
				if ( !(CurrentLine.substr(8,5)).compare(_T("true")) )  /* on */
				{
					Scale2xEnabled = true;
				}
				else
				{
					Scale2xEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,11)).compare(_T("DarkPalette")) )
			{
				/* Dark palette */
				if ( !(CurrentLine.substr(12,5)).compare(_T("true")) )  /* on */
				{
					DarkPaletteEnabled = true;
				}
				else
				{
					DarkPaletteEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,10)).compare(_T("Monochrom")) )
			{
				/* Monochrome */
				if ( !(CurrentLine.substr(11,5)).compare(_T("true")) )  /* on */
				{
					MonochromEnabled = true;
				}
				else
				{
					MonochromEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,10)).compare(_T("FPSDisplay")) )
			{
				/* FPS Display */
				if ( !(CurrentLine.substr(11,5)).compare(_T("true")) )  /* on */
				{
					FPSDisplayEnabled = true;
				}
				else
				{
					FPSDisplayEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,17)).compare(_T("BilinearFiltering")) )
			{
				/* Bilinear Filtering */
				if ( !(CurrentLine.substr(18,5)).compare(_T("true")) )  /* on */
				{
					BilinearFilteringEnabled = true;
				}
				else
				{
					BilinearFilteringEnabled = false;
				}
			}
			
			/* Audio */
			if ( !(CurrentLine.substr(0,5)).compare(_T("Sound")) )
			{
				/* Sound */
				if ( !(CurrentLine.substr(6,5)).compare(_T("true")) )  /* Music on */
				{
					SoundEnabled = true;
				}
				else
				{
					SoundEnabled = false;
				}
			}

			/* Options */
			if ( !(CurrentLine.substr(0,7)).compare(_T("Gamepad")) )
			{
				/* Gamepad */
				if ( !(CurrentLine.substr(8,5)).compare(_T("true")) )  /* Gamepad on */
				{
					GamepadEnabled = true;
				}
				else
				{
					GamepadEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,8)).compare(_T("MousePad")) )
			{
				/* Mousepad */
				if ( !(CurrentLine.substr(9,5)).compare(_T("true")) )  /* Mousepad on */
				{
					MousepadEnabled = true;
				}
				else
				{
					MousepadEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,20)).compare(_T("MousepadAcceleration")) )
			{
				/* MousepadAcceleration */
				wxString TempStr = CurrentLine.substr(21,5);
				TempStr.ToDouble(&MousepadAcceleration);   /* Gamepad Acceleration */
			}


			/* Paths */
			if ( !(CurrentLine.substr(0,8)).compare(_T("ExecPath")) )
			{
				ExecutableFilename = CurrentLine.substr(9, ( CurrentLine.length() - 9 ) );
			}
			if ( !(CurrentLine.substr(0,13)).compare(_T("LastOpenedDir")) )
			{
				LastOpenedDirectory = CurrentLine.substr(14, ( CurrentLine.length() - 14 ) );
			}
			
			/* Zone */
			if ( !(CurrentLine.substr(0,12)).compare(_T("ExportedMode")) )
			{
				/* Exported Mode */
				if ( !(CurrentLine.substr(13,5)).compare(_T("true")) )  /* on */
				{
					ExportedModeEnabled = true;
				}
				else
				{
					ExportedModeEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,7)).compare(_T("PALMode")) )
			{
				/* PAL Mode */
				if ( !(CurrentLine.substr(8,5)).compare(_T("true")) )  /* on */
				{
					PALModeEnabled = true;
				}
				else
				{
					PALModeEnabled = false;
				}
			}
			if ( !(CurrentLine.substr(0,12)).compare(_T("JapaneseMode")) )
			{
				/* Japanese Mode */
				if ( !(CurrentLine.substr(13,5)).compare(_T("true")) )  /* on */
				{
					JapaneseModeEnabled = true;
				}
				else
				{
					JapaneseModeEnabled = false;
				}
			}
			
			/* Osmose Specific */
			if ( !(CurrentLine.substr(0,7)).compare(_T("PAD1_UP")) )
			{
				P1_Up = GetSDLKeyFromStringKey( CurrentLine.substr(10,15) );
			}
			if ( !(CurrentLine.substr(0,9)).compare(_T("PAD1_DOWN")) )
			{
				P1_Down = GetSDLKeyFromStringKey( CurrentLine.substr(12,15) );
			}
			if ( !(CurrentLine.substr(0,9)).compare(_T("PAD1_LEFT")) )
			{
				P1_Left = GetSDLKeyFromStringKey( CurrentLine.substr(12,15) );
			}
			if ( !(CurrentLine.substr(0,10)).compare(_T("PAD1_RIGHT")) )
			{
				P1_Right = GetSDLKeyFromStringKey( CurrentLine.substr(13,15) );
			}
			if ( !(CurrentLine.substr(0,13)).compare(_T("PAD1_BUTTON_A")) )
			{
				P1_A = GetSDLKeyFromStringKey( CurrentLine.substr(16,15) );
			}
			if ( !(CurrentLine.substr(0,13)).compare(_T("PAD1_BUTTON_B")) )
			{
				P1_B = GetSDLKeyFromStringKey( CurrentLine.substr(16,15) );
			}
			if ( !(CurrentLine.substr(0,7)).compare(_T("PAD2_UP")) )
			{
				P2_Up = GetSDLKeyFromStringKey( CurrentLine.substr(10,15) );
			}
			if ( !(CurrentLine.substr(0,9)).compare(_T("PAD2_DOWN")) )
			{
				P2_Down = GetSDLKeyFromStringKey( CurrentLine.substr(12,15) );
			}
			if ( !(CurrentLine.substr(0,9)).compare(_T("PAD2_LEFT")) )
			{
				P2_Left = GetSDLKeyFromStringKey( CurrentLine.substr(12,15) );
			}
			if ( !(CurrentLine.substr(0,10)).compare(_T("PAD2_RIGHT")) )
			{
				P2_Right = GetSDLKeyFromStringKey( CurrentLine.substr(13,15) );
			}
			if ( !(CurrentLine.substr(0,13)).compare(_T("PAD2_BUTTON_A")) )
			{
				P2_A = GetSDLKeyFromStringKey( CurrentLine.substr(16,15) );
			}
			if ( !(CurrentLine.substr(0,13)).compare(_T("PAD2_BUTTON_B")) )
			{
				P2_B = GetSDLKeyFromStringKey( CurrentLine.substr(16,15) );
			}
			
			if ( !(CurrentLine.substr(0,10)).compare(_T("SCREENSHOT")) )
			{
				Screenshot = GetSDLKeyFromStringKey( CurrentLine.substr(13,15) );
			}
			if ( !(CurrentLine.substr(0,8)).compare(_T("TILESHOT")) )
			{
				Tileshot = GetSDLKeyFromStringKey( CurrentLine.substr(11,15) );
			}
			if ( !(CurrentLine.substr(0,9)).compare(_T("SOUNDSHOT")) )
			{
				Soundshot = GetSDLKeyFromStringKey( CurrentLine.substr(12,15) );
			}
			if ( !(CurrentLine.substr(0,5)).compare(_T("PAUSE")) )
			{
				Pause = GetSDLKeyFromStringKey( CurrentLine.substr(8,15) );
			}
			if ( !(CurrentLine.substr(0,4)).compare(_T("QUIT")) )
			{
				Quit = GetSDLKeyFromStringKey( CurrentLine.substr(7,15) );
			}
			#ifdef _DEBUG_CONSOLE_
			if ( !(CurrentLine.substr(0,5)).compare(_T("DEBUGGER")) )
			{
				DebugKey = GetSDLKeyFromStringKey( CurrentLine.substr(8,15) );
			}
			#endif
		} /* End While not eof */
		MyConfigFile.Close();
	} /* End if Config exists */
	
}


Configuration::~Configuration()
{
	WriteConfig();
}

void Configuration::WriteConfig()
{
	#ifdef __USE_UNIX98 // Create a directory in order to put saves, screenshots, ini file and waves shot. */
		if ( !wxDirExists( wxString::Format(_T("%s/.osmose"), UserHomeDirectory.c_str()) ) )
		{
			wxMkdir( wxString::Format(_T("%s/.osmose"), UserHomeDirectory.c_str()), S_IRWXU | S_IRWXG | S_IXOTH);
		}
	#else /* Win and others */
		/* Nothing */
	#endif
		
	/* Ini File creation and writing */
	wxTextFile FirstIniFile;
	FirstIniFile.Open(IniFilename);
	if ( FirstIniFile.IsOpened() )
	{
		FirstIniFile.Clear();

		FirstIniFile.AddLine(_T("[---Osmose---]"));
		
		FirstIniFile.AddLine(_T(""));
		FirstIniFile.AddLine(_T("[Gamepad Keys]"));
		FirstIniFile.AddLine(wxString::Format(_T("PAD1_UP = %s"), (GetStringKeyFromSDLKey(P1_Up)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD1_DOWN = %s"), (GetStringKeyFromSDLKey(P1_Down)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD1_LEFT = %s"), (GetStringKeyFromSDLKey(P1_Left)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD1_RIGHT = %s"), (GetStringKeyFromSDLKey(P1_Right)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD1_BUTTON_A = %s"), (GetStringKeyFromSDLKey(P1_A)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD1_BUTTON_B = %s"), (GetStringKeyFromSDLKey(P1_B)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD2_UP = %s"), (GetStringKeyFromSDLKey(P2_Up)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD2_DOWN = %s"), (GetStringKeyFromSDLKey(P2_Down)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD2_LEFT = %s"), (GetStringKeyFromSDLKey(P2_Left)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD2_RIGHT = %s"), (GetStringKeyFromSDLKey(P2_Right)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD2_BUTTON_A = %s"), (GetStringKeyFromSDLKey(P2_A)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAD2_BUTTON_B = %s"), (GetStringKeyFromSDLKey(P2_B)).c_str() ));
		
		FirstIniFile.AddLine(_T(""));
		FirstIniFile.AddLine(_T("[Emulator Keys]"));
		FirstIniFile.AddLine(wxString::Format(_T("SCREENSHOT = %s"), (GetStringKeyFromSDLKey(Screenshot)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("SOUNDSHOT = %s"), (GetStringKeyFromSDLKey(Soundshot)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("TILESHOT = %s"), (GetStringKeyFromSDLKey(Tileshot)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("PAUSE = %s"), (GetStringKeyFromSDLKey(Pause)).c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("QUIT = %s"), (GetStringKeyFromSDLKey(Quit)).c_str() ));
		#ifdef _DEBUG_CONSOLE_
		FirstIniFile.AddLine(wxString::Format(_T("DEBUGGER = %s"), (GetStringKeyFromSDLKey(DebugKey)).c_str() ));
		#endif
		
		FirstIniFile.AddLine(wxString::Format(_T("")));
		FirstIniFile.AddLine(wxString::Format(_T("[---wxOsmose---]")));
		FirstIniFile.AddLine(wxString::Format(_T("")));
		FirstIniFile.AddLine(wxString::Format(_T("[Video]")));
		if ( FullScreenEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("FullScreen=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("FullScreen=false")));
		}
		
		if ( Scale2xEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("Scale2x=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("Scale2x=false")));
		}
		
		if ( TVLikeEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("TVMode=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("TVMode=false")));
		}
		
		if ( DarkPaletteEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("DarkPalette=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("DarkPalette=false")));
		}
		
		if ( MonochromEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("Monochrom=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("Monochrom=false")));
		}
		
		if ( FPSDisplayEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("FPSDisplay=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("FPSDisplay=false")));
		}

		if ( BilinearFilteringEnabled )
		{
			FirstIniFile.AddLine(wxString::Format(_T("BilinearFiltering=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("BilinearFiltering=false")));
		}
		
		FirstIniFile.AddLine(wxString::Format(_T("")));
		FirstIniFile.AddLine(wxString::Format(_T("[Audio]")));
		if ( SoundEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("Sound=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("Sound=false")));
		}
		
		FirstIniFile.AddLine(wxString::Format(_T("")));
		FirstIniFile.AddLine(wxString::Format(_T("[Options]")));
		if ( GamepadEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("Gamepad=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("Gamepad=false")));
		}
		if ( MousepadEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("MousePad=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("MousePad=false")));
		}
		
		FirstIniFile.AddLine(wxString::Format(_T("MousepadAcceleration=%f"), MousepadAcceleration ));
		
		FirstIniFile.AddLine(wxString::Format(_T("")));
		FirstIniFile.AddLine(wxString::Format(_T("[Paths]")));
		FirstIniFile.AddLine(wxString::Format(_T("ExecPath=%s"), ExecutableFilename.c_str() ));
		FirstIniFile.AddLine(wxString::Format(_T("LastOpenedDir=%s"), LastOpenedDirectory.c_str() ));
		
		FirstIniFile.AddLine(wxString::Format(_T("")));
		FirstIniFile.AddLine(wxString::Format(_T("[Zone]")));
		if ( ExportedModeEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("ExportedMode=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("ExportedMode=false")));
		}
		
		if ( PALModeEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("PALMode=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("PALMode=false")));
		}
		
		if ( JapaneseModeEnabled )  
		{
			FirstIniFile.AddLine(wxString::Format(_T("JapaneseMode=true")));
		}
		else
		{
			FirstIniFile.AddLine(wxString::Format(_T("JapaneseMode=false")));
		}
		
		FirstIniFile.Write();
		
		#ifdef _DEBUG_
			wxLogMessage(_T("Wrote configuration file..."));
		#endif
	}
	else
	{
		#ifdef _DEBUG_
			wxLogMessage(_T("ini File creation failed !"));
		#endif
		FirstIniFile.Close();
	}
}

/* Options */
void Configuration::SetMousepadAcceleration( double Acceleration )
{
	if ( Acceleration < 0 ) Acceleration = 0;
	if ( Acceleration > 2 ) Acceleration = 2;
	MousepadAcceleration = Acceleration; 
}

/* Zone */
bool Configuration::IsExportedModeEnabled()
{
	return ExportedModeEnabled;
}

void Configuration::SetExportedModeEnabled( bool Enabled )
{
	ExportedModeEnabled = Enabled;
	if ( Enabled )
	{
		PALModeEnabled = false;
		JapaneseModeEnabled = false;
	}
}

bool Configuration::IsJapaneseModeEnabled()
{ 
	return JapaneseModeEnabled; 
}

void Configuration::SetJapaneseModeEnabled( bool Enabled )
{
	JapaneseModeEnabled = Enabled;
	if ( Enabled )
	{
		ExportedModeEnabled = false;
		PALModeEnabled = false;
	}
}

bool Configuration::IsPALModeEnabled()
{
	return PALModeEnabled;
}

void Configuration::SetPALModeEnabled( bool Enabled )
{
	PALModeEnabled = Enabled;
	if ( Enabled )
	{
		ExportedModeEnabled = false;
		JapaneseModeEnabled = false;
	}
}

wxString Configuration::GetStringKeyFromSDLKey(int MySDLKey)
{
	
	switch ( MySDLKey )
	{
		default:
			return wxString::Format(_T("<Nothing>"));
		break;
		
		case SDLK_BACKSPACE:
			return wxString::Format(_T("SDLK_BACKSPACE"));
		break;
		case SDLK_TAB:
			return wxString::Format(_T("SDLK_TAB"));
		break;
		case SDLK_CLEAR:
			return wxString::Format(_T("SDLK_CLEAR"));
		break;
		case SDLK_RETURN:
			return wxString::Format(_T("SDLK_RETURN"));
		break;
		case SDLK_PAUSE:
			return wxString::Format(_T("SDLK_PAUSE"));
		break;
		case SDLK_ESCAPE:
			return wxString::Format(_T("SDLK_ESCAPE"));
		break;
		case SDLK_SPACE:
			return wxString::Format(_T("SDLK_SPACE"));
		break;
		case SDLK_EXCLAIM:
			return wxString::Format(_T("SDLK_EXCLAIM"));
		break;
		case SDLK_QUOTEDBL:
			return wxString::Format(_T("SDLK_QUOTEDBL"));
		break;
		case SDLK_HASH:
			return wxString::Format(_T("SDLK_HASH"));
		break;
		case SDLK_DOLLAR:
			return wxString::Format(_T("SDLK_DOLLAR"));
		break;
		case SDLK_AMPERSAND:
			return wxString::Format(_T("SDLK_AMPERSAND"));
		break;
		case SDLK_QUOTE:
			return wxString::Format(_T("SDLK_QUOTE"));
		break;
		case SDLK_LEFTPAREN:
			return wxString::Format(_T("SDLK_LEFTPAREN"));
		break;
		case SDLK_RIGHTPAREN:
			return wxString::Format(_T("SDLK_RIGHTPAREN"));
		break;
		case SDLK_ASTERISK:
			return wxString::Format(_T("SDLK_ASTERISK"));
		break;
		case SDLK_PLUS:
			return wxString::Format(_T("SDLK_PLUS"));
		break;
		case SDLK_COMMA:
			return wxString::Format(_T("SDLK_COMMA"));
		break;
		case SDLK_MINUS:
			return wxString::Format(_T("SDLK_MINUS"));
		break;
		case SDLK_PERIOD:
			return wxString::Format(_T("SDLK_PERIOD"));
		break;
		case SDLK_SLASH:
			return wxString::Format(_T("SDLK_SLASH"));
		break;
		case SDLK_0:
			return wxString::Format(_T("SDLK_0"));
		break;
		case SDLK_1:
			return wxString::Format(_T("SDLK_1"));
		break;
		case SDLK_2:
			return wxString::Format(_T("SDLK_2"));
		break;
		case SDLK_3:
			return wxString::Format(_T("SDLK_3"));
		break;
		case SDLK_4:
			return wxString::Format(_T("SDLK_4"));
		break;
		case SDLK_5:
			return wxString::Format(_T("SDLK_5"));
		break;
		case SDLK_6:
			return wxString::Format(_T("SDLK_6"));
		break;
		case SDLK_7:
			return wxString::Format(_T("SDLK_7"));
		break;
		case SDLK_8:
			return wxString::Format(_T("SDLK_8"));
		break;
		case SDLK_9:
			return wxString::Format(_T("SDLK_9"));
		break;
		case SDLK_COLON:
			return wxString::Format(_T("SDLK_COLON"));
		break;
		case SDLK_SEMICOLON:
			return wxString::Format(_T("SDLK_SEMICOLON"));
		break;
		case SDLK_LESS:
			return wxString::Format(_T("SDLK_LESS"));
		break;
		case SDLK_EQUALS:
			return wxString::Format(_T("SDLK_EQUALS"));
		break;
		case SDLK_GREATER:
			return wxString::Format(_T("SDLK_GREATER"));
		break;
		case SDLK_QUESTION:
			return wxString::Format(_T("SDLK_QUESTION"));
		break;
		case SDLK_AT:
			return wxString::Format(_T("SDLK_AT"));
		break;
		
		case SDLK_LEFTBRACKET:
			return wxString::Format(_T("SDLK_LEFTBRACKET"));
		break;
		case SDLK_BACKSLASH:
			return wxString::Format(_T("SDLK_BACKSLASH"));
		break;
		case SDLK_RIGHTBRACKET:
			return wxString::Format(_T("SDLK_RIGHTBRACKET"));
		break;
		case SDLK_CARET:
			return wxString::Format(_T("SDLK_CARET"));
		break;
		case SDLK_UNDERSCORE:
			return wxString::Format(_T("SDLK_UNDERSCORE"));
		break;
		case SDLK_BACKQUOTE:
			return wxString::Format(_T("SDLK_BACKQUOTE"));
		break;
		case SDLK_a:
			return wxString::Format(_T("SDLK_a"));
		break;
		case SDLK_b:
			return wxString::Format(_T("SDLK_b"));
		break;
		case SDLK_c:
			return wxString::Format(_T("SDLK_c"));
		break;
		case SDLK_d:
			return wxString::Format(_T("SDLK_d"));
		break;
		case SDLK_e:
			return wxString::Format(_T("SDLK_e"));
		break;
		case SDLK_f:
			return wxString::Format(_T("SDLK_f"));
		break;
		case SDLK_g:
			return wxString::Format(_T("SDLK_g"));
		break;
		case SDLK_h:
			return wxString::Format(_T("SDLK_h"));
		break;
		case SDLK_i:
			return wxString::Format(_T("SDLK_i"));
		break;
		case SDLK_j:
			return wxString::Format(_T("SDLK_j"));
		break;
		case SDLK_k:
			return wxString::Format(_T("SDLK_k"));
		break;
		case SDLK_l:
			return wxString::Format(_T("SDLK_l"));
		break;
		case SDLK_m:
			return wxString::Format(_T("SDLK_m"));
		break;
		case SDLK_n:
			return wxString::Format(_T("SDLK_n"));
		break;
		case SDLK_o:
			return wxString::Format(_T("SDLK_o"));
		break;
		case SDLK_p:
			return wxString::Format(_T("SDLK_p"));
		break;
		case SDLK_q:
			return wxString::Format(_T("SDLK_q"));
		break;
		case SDLK_r:
			return wxString::Format(_T("SDLK_r"));
		break;
		case SDLK_s:
			return wxString::Format(_T("SDLK_s"));
		break;
		case SDLK_t:
			return wxString::Format(_T("SDLK_t"));
		break;
		case SDLK_u:
			return wxString::Format(_T("SDLK_u"));
		break;
		case SDLK_v:
			return wxString::Format(_T("SDLK_v"));
		break;
		case SDLK_w:
			return wxString::Format(_T("SDLK_w"));
		break;
		case SDLK_x:
			return wxString::Format(_T("SDLK_x"));
		break;
		case SDLK_y:
			return wxString::Format(_T("SDLK_y"));
		break;
		case SDLK_z:
			return wxString::Format(_T("SDLK_z"));
		break;
		case SDLK_DELETE:
			return wxString::Format(_T("SDLK_DELETE"));
		break;
		case SDLK_KP0:
			return wxString::Format(_T("SDLK_KP0"));
		break;
		case SDLK_KP1:
			return wxString::Format(_T("SDLK_KP1"));
		break;
		case SDLK_KP2:
			return wxString::Format(_T("SDLK_KP2"));
		break;
		case SDLK_KP3:
			return wxString::Format(_T("SDLK_KP3"));
		break;
		case SDLK_KP4:
			return wxString::Format(_T("SDLK_KP4"));
		break;
		case SDLK_KP5:
			return wxString::Format(_T("SDLK_KP5"));
		break;
		case SDLK_KP6:
			return wxString::Format(_T("SDLK_KP6"));
		break;
		case SDLK_KP7:
			return wxString::Format(_T("SDLK_KP7"));
		break;
		case SDLK_KP8:
			return wxString::Format(_T("SDLK_KP8"));
		break;
		case SDLK_KP9:
			return wxString::Format(_T("SDLK_KP9"));
		break;
		case SDLK_KP_PERIOD:
			return wxString::Format(_T("SDLK_KP_PERIOD"));
		break;
		case SDLK_KP_DIVIDE:
			return wxString::Format(_T("SDLK_KP_DIVIDE"));
		break;
		case SDLK_KP_MULTIPLY:
			return wxString::Format(_T("SDLK_KP_MULTIPLY"));
		break;
		case SDLK_KP_MINUS:
			return wxString::Format(_T("SDLK_KP_MINUS"));
		break;
		case SDLK_KP_PLUS:
			return wxString::Format(_T("SDLK_KP_PLUS"));
		break;
		case SDLK_KP_ENTER:
			return wxString::Format(_T("SDLK_KP_ENTER"));
		break;
		case SDLK_KP_EQUALS:
			return wxString::Format(_T("SDLK_KP_EQUALS"));
		break;
		case SDLK_UP:
			return wxString::Format(_T("SDLK_UP"));
		break;
		case SDLK_DOWN:
			return wxString::Format(_T("SDLK_DOWN"));
		break;
		case SDLK_RIGHT:
			return wxString::Format(_T("SDLK_RIGHT"));
		break;
		case SDLK_LEFT:
			return wxString::Format(_T("SDLK_LEFT"));
		break;
		case SDLK_INSERT:
			return wxString::Format(_T("SDLK_INSERT"));
		break;
		case SDLK_HOME:
			return wxString::Format(_T("SDLK_HOME"));
		break;
		case SDLK_END:
			return wxString::Format(_T("SDLK_END"));
		break;
		case SDLK_PAGEUP:
			return wxString::Format(_T("SDLK_PAGEUP"));
		break;
		case SDLK_PAGEDOWN:
			return wxString::Format(_T("SDLK_PAGEDOWN"));
		break;
		case SDLK_F1:
			return wxString::Format(_T("SDLK_F1"));
		break;
		case SDLK_F2:
			return wxString::Format(_T("SDLK_F2"));
		break;
		case SDLK_F3:
			return wxString::Format(_T("SDLK_F3"));
		break;
		case SDLK_F4:
			return wxString::Format(_T("SDLK_F4"));
		break;
		case SDLK_F5:
			return wxString::Format(_T("SDLK_F5"));
		break;
		case SDLK_F6:
			return wxString::Format(_T("SDLK_F6"));
		break;
		case SDLK_F7:
			return wxString::Format(_T("SDLK_F7"));
		break;
		case SDLK_F8:
			return wxString::Format(_T("SDLK_F8"));
		break;
		case SDLK_F9:
			return wxString::Format(_T("SDLK_F9"));
		break;
		case SDLK_F10:
			return wxString::Format(_T("SDLK_F10"));
		break;
		case SDLK_F11:
			return wxString::Format(_T("SDLK_F11"));
		break;
		case SDLK_F12:
			return wxString::Format(_T("SDLK_F12"));
		break;
		case SDLK_F13:
			return wxString::Format(_T("SDLK_F13"));
		break;
		case SDLK_F14:
			return wxString::Format(_T("SDLK_F14"));
		break;
		case SDLK_F15:
			return wxString::Format(_T("SDLK_F15"));
		break;
		case SDLK_RSHIFT:
			return wxString::Format(_T("SDLK_RSHIFT"));
		break;
		case SDLK_LSHIFT:
			return wxString::Format(_T("SDLK_LSHIFT"));
		break;
		case SDLK_RCTRL:
			return wxString::Format(_T("SDLK_RCTRL"));
		break;
		case SDLK_LCTRL:
			return wxString::Format(_T("SDLK_LCTRL"));
		break;
		case SDLK_RALT:
			return wxString::Format(_T("SDLK_RALT"));
		break;
		case SDLK_LALT:
			return wxString::Format(_T("SDLK_LALT"));
		break;
	
	}
	return wxString::Format(_T("<Nothing>"));
}

int Configuration::GetSDLKeyFromStringKey(wxString MyStringKey)
{	 
	if ( !(MyStringKey.compare(_T("SDLK_BACKSPACE"))) ) return SDLK_BACKSPACE;
	if ( !(MyStringKey.compare(_T("SDLK_TAB"))) ) return SDLK_TAB;
	if ( !(MyStringKey.compare(_T("SDLK_CLEAR"))) ) return SDLK_CLEAR;
	if ( !(MyStringKey.compare(_T("SDLK_RETURN"))) ) return SDLK_RETURN;
	if ( !(MyStringKey.compare(_T("SDLK_PAUSE"))) ) return SDLK_PAUSE;
	if ( !(MyStringKey.compare(_T("SDLK_ESCAPE"))) ) return SDLK_ESCAPE;
	if ( !(MyStringKey.compare(_T("SDLK_SPACE"))) ) return SDLK_SPACE;
	if ( !(MyStringKey.compare(_T("SDLK_EXCLAIM"))) ) return SDLK_EXCLAIM;
	if ( !(MyStringKey.compare(_T("SDLK_QUOTEDBL"))) ) return SDLK_QUOTEDBL;
	if ( !(MyStringKey.compare(_T("SDLK_HASH"))) ) return SDLK_HASH;
	if ( !(MyStringKey.compare(_T("SDLK_DOLLAR"))) ) return SDLK_DOLLAR;
	if ( !(MyStringKey.compare(_T("SDLK_AMPERSAND"))) ) return SDLK_AMPERSAND;
	if ( !(MyStringKey.compare(_T("SDLK_QUOTE"))) ) return SDLK_QUOTE;
	if ( !(MyStringKey.compare(_T("SDLK_LEFTPAREN"))) ) return SDLK_LEFTPAREN;
	if ( !(MyStringKey.compare(_T("SDLK_RIGHTPAREN"))) ) return SDLK_RIGHTPAREN;
	if ( !(MyStringKey.compare(_T("SDLK_ASTERISK"))) ) return SDLK_ASTERISK;
	if ( !(MyStringKey.compare(_T("SDLK_PLUS"))) ) return SDLK_PLUS;
	if ( !(MyStringKey.compare(_T("SDLK_COMMA"))) ) return SDLK_COMMA;
	if ( !(MyStringKey.compare(_T("SDLK_MINUS"))) ) return SDLK_MINUS;
	if ( !(MyStringKey.compare(_T("SDLK_PERIOD"))) ) return SDLK_PERIOD;
	if ( !(MyStringKey.compare(_T("SDLK_SLASH"))) ) return SDLK_SLASH;
	if ( !(MyStringKey.compare(_T("SDLK_0"))) ) return SDLK_0;
	if ( !(MyStringKey.compare(_T("SDLK_1"))) ) return SDLK_1;
	if ( !(MyStringKey.compare(_T("SDLK_2"))) ) return SDLK_2;
	if ( !(MyStringKey.compare(_T("SDLK_3"))) ) return SDLK_3;
	if ( !(MyStringKey.compare(_T("SDLK_4"))) ) return SDLK_4;
	if ( !(MyStringKey.compare(_T("SDLK_5"))) ) return SDLK_5;
	if ( !(MyStringKey.compare(_T("SDLK_6"))) ) return SDLK_6;
	if ( !(MyStringKey.compare(_T("SDLK_7"))) ) return SDLK_7;
	if ( !(MyStringKey.compare(_T("SDLK_8"))) ) return SDLK_8;
	if ( !(MyStringKey.compare(_T("SDLK_9"))) ) return SDLK_9;
	if ( !(MyStringKey.compare(_T("SDLK_COLON"))) ) return SDLK_COLON;
	if ( !(MyStringKey.compare(_T("SDLK_SEMICOLON"))) ) return SDLK_SEMICOLON;
	if ( !(MyStringKey.compare(_T("SDLK_LESS"))) ) return SDLK_LESS;
	if ( !(MyStringKey.compare(_T("SDLK_EQUALS"))) ) return SDLK_EQUALS;
	if ( !(MyStringKey.compare(_T("SDLK_GREATER"))) ) return SDLK_GREATER;
	if ( !(MyStringKey.compare(_T("SDLK_QUESTION"))) ) return SDLK_QUESTION;
	if ( !(MyStringKey.compare(_T("SDLK_AT"))) ) return SDLK_AT;
	if ( !(MyStringKey.compare(_T("SDLK_LEFTBRACKET"))) ) return SDLK_LEFTBRACKET;
	if ( !(MyStringKey.compare(_T("SDLK_BACKSLASH"))) ) return SDLK_BACKSLASH;
	if ( !(MyStringKey.compare(_T("SDLK_RIGHTBRACKET"))) ) return SDLK_RIGHTBRACKET;
	if ( !(MyStringKey.compare(_T("SDLK_CARET"))) ) return SDLK_CARET;
	if ( !(MyStringKey.compare(_T("SDLK_UNDERSCORE"))) ) return SDLK_UNDERSCORE;
	if ( !(MyStringKey.compare(_T("SDLK_BACKQUOTE"))) ) return SDLK_BACKQUOTE;
	if ( !(MyStringKey.compare(_T("SDLK_a"))) ) return SDLK_a;
	if ( !(MyStringKey.compare(_T("SDLK_b"))) ) return SDLK_b;
	if ( !(MyStringKey.compare(_T("SDLK_c"))) ) return SDLK_c;
	if ( !(MyStringKey.compare(_T("SDLK_d"))) ) return SDLK_d;
	if ( !(MyStringKey.compare(_T("SDLK_e"))) ) return SDLK_e;
	if ( !(MyStringKey.compare(_T("SDLK_f"))) ) return SDLK_f;
	if ( !(MyStringKey.compare(_T("SDLK_g"))) ) return SDLK_g;
	if ( !(MyStringKey.compare(_T("SDLK_h"))) ) return SDLK_h;
	if ( !(MyStringKey.compare(_T("SDLK_i"))) ) return SDLK_i;
	if ( !(MyStringKey.compare(_T("SDLK_j"))) ) return SDLK_j;
	if ( !(MyStringKey.compare(_T("SDLK_k"))) ) return SDLK_k;
	if ( !(MyStringKey.compare(_T("SDLK_l"))) ) return SDLK_l;
	if ( !(MyStringKey.compare(_T("SDLK_m"))) ) return SDLK_m;
	if ( !(MyStringKey.compare(_T("SDLK_n"))) ) return SDLK_n;
	if ( !(MyStringKey.compare(_T("SDLK_o"))) ) return SDLK_o;
	if ( !(MyStringKey.compare(_T("SDLK_p"))) ) return SDLK_p;
	if ( !(MyStringKey.compare(_T("SDLK_q"))) ) return SDLK_q;
	if ( !(MyStringKey.compare(_T("SDLK_r"))) ) return SDLK_r;
	if ( !(MyStringKey.compare(_T("SDLK_s"))) ) return SDLK_s;
	if ( !(MyStringKey.compare(_T("SDLK_t"))) ) return SDLK_t;
	if ( !(MyStringKey.compare(_T("SDLK_u"))) ) return SDLK_u;
	if ( !(MyStringKey.compare(_T("SDLK_v"))) ) return SDLK_v;
	if ( !(MyStringKey.compare(_T("SDLK_w"))) ) return SDLK_w;
	if ( !(MyStringKey.compare(_T("SDLK_x"))) ) return SDLK_x;
	if ( !(MyStringKey.compare(_T("SDLK_y"))) ) return SDLK_y;
	if ( !(MyStringKey.compare(_T("SDLK_z"))) ) return SDLK_z;
	if ( !(MyStringKey.compare(_T("SDLK_DELETE"))) ) return SDLK_DELETE;
	if ( !(MyStringKey.compare(_T("SDLK_KP0"))) ) return SDLK_KP0;
	if ( !(MyStringKey.compare(_T("SDLK_KP1"))) ) return SDLK_KP1;
	if ( !(MyStringKey.compare(_T("SDLK_KP2"))) ) return SDLK_KP2;
	if ( !(MyStringKey.compare(_T("SDLK_KP3"))) ) return SDLK_KP3;
	if ( !(MyStringKey.compare(_T("SDLK_KP4"))) ) return SDLK_KP4;
	if ( !(MyStringKey.compare(_T("SDLK_KP5"))) ) return SDLK_KP5;
	if ( !(MyStringKey.compare(_T("SDLK_KP6"))) ) return SDLK_KP6;
	if ( !(MyStringKey.compare(_T("SDLK_KP7"))) ) return SDLK_KP7;
	if ( !(MyStringKey.compare(_T("SDLK_KP8"))) ) return SDLK_KP8;
	if ( !(MyStringKey.compare(_T("SDLK_KP9"))) ) return SDLK_KP9;
	if ( !(MyStringKey.compare(_T("SDLK_KP_PERIOD"))) ) return SDLK_KP_PERIOD;
	if ( !(MyStringKey.compare(_T("SDLK_KP_DIVIDE"))) ) return SDLK_KP_DIVIDE;
	if ( !(MyStringKey.compare(_T("SDLK_KP_MULTIPLY"))) ) return SDLK_KP_MULTIPLY;
	if ( !(MyStringKey.compare(_T("SDLK_KP_MINUS"))) ) return SDLK_KP_MINUS;
	if ( !(MyStringKey.compare(_T("SDLK_KP_PLUS"))) ) return SDLK_KP_PLUS;
	if ( !(MyStringKey.compare(_T("SDLK_KP_ENTER"))) ) return SDLK_KP_ENTER;
	if ( !(MyStringKey.compare(_T("SDLK_KP_EQUALS"))) ) return SDLK_KP_EQUALS;
	if ( !(MyStringKey.compare(_T("SDLK_UP"))) ) return SDLK_UP;
	if ( !(MyStringKey.compare(_T("SDLK_DOWN"))) ) return SDLK_DOWN;
	if ( !(MyStringKey.compare(_T("SDLK_RIGHT"))) ) return SDLK_RIGHT;
	if ( !(MyStringKey.compare(_T("SDLK_LEFT"))) ) return SDLK_LEFT;
	if ( !(MyStringKey.compare(_T("SDLK_INSERT"))) ) return SDLK_INSERT;
	if ( !(MyStringKey.compare(_T("SDLK_HOME"))) ) return SDLK_HOME;
	if ( !(MyStringKey.compare(_T("SDLK_END"))) ) return SDLK_END;
	if ( !(MyStringKey.compare(_T("SDLK_PAGEUP"))) ) return SDLK_PAGEUP;
	if ( !(MyStringKey.compare(_T("SDLK_PAGEDOWN"))) ) return SDLK_PAGEDOWN;
	if ( !(MyStringKey.compare(_T("SDLK_F1"))) ) return SDLK_F1;
	if ( !(MyStringKey.compare(_T("SDLK_F2"))) ) return SDLK_F2;
	if ( !(MyStringKey.compare(_T("SDLK_F3"))) ) return SDLK_F3;
	if ( !(MyStringKey.compare(_T("SDLK_F4"))) ) return SDLK_F4;
	if ( !(MyStringKey.compare(_T("SDLK_F5"))) ) return SDLK_F5;
	if ( !(MyStringKey.compare(_T("SDLK_F6"))) ) return SDLK_F6;
	if ( !(MyStringKey.compare(_T("SDLK_F7"))) ) return SDLK_F7;
	if ( !(MyStringKey.compare(_T("SDLK_F8"))) ) return SDLK_F8;
	if ( !(MyStringKey.compare(_T("SDLK_F9"))) ) return SDLK_F9;
	if ( !(MyStringKey.compare(_T("SDLK_F10"))) ) return SDLK_F10;
	if ( !(MyStringKey.compare(_T("SDLK_F11"))) ) return SDLK_F11;
	if ( !(MyStringKey.compare(_T("SDLK_F12"))) ) return SDLK_F12;
	if ( !(MyStringKey.compare(_T("SDLK_F13"))) ) return SDLK_F13;
	if ( !(MyStringKey.compare(_T("SDLK_F14"))) ) return SDLK_F14;
	if ( !(MyStringKey.compare(_T("SDLK_F15"))) ) return SDLK_F15;
	if ( !(MyStringKey.compare(_T("SDLK_RSHIFT"))) ) return SDLK_RSHIFT;
	if ( !(MyStringKey.compare(_T("SDLK_LSHIFT"))) ) return SDLK_LSHIFT;
	if ( !(MyStringKey.compare(_T("SDLK_RCTRL"))) ) return SDLK_RCTRL;
	if ( !(MyStringKey.compare(_T("SDLK_LCTRL"))) ) return SDLK_LCTRL;
	if ( !(MyStringKey.compare(_T("SDLK_RALT"))) ) return SDLK_RALT;
	if ( !(MyStringKey.compare(_T("SDLK_LALT"))) ) return SDLK_LALT;

	return -1;
}
