/*///////////////////////////////////////////////////////////////////////////
// Name:        myframe.cpp
// Purpose:     wxWindows wxOsmose Front-End to the Osmose Emulator
// Author:      Ferreira Yohann
// Modified by:
// Created:     25/01/2005
// Copyright:   (c) Ferreira Yohann
// Licence:     GPL
///////////////////////////////////////////////////////////////////////////*/

/* Application's Main Window */
#include "myframe.h" /* Header */
#include "keydialog.h" /* For KeyDialog Class */
#include "myaboutdialog.h" /* For AboutDialog Class */

#include "../md5/md5.h" /* For MD5 Sum Functions Support */
#include "../md5/romslist.h" /* For Sms Properties auto-detection. */

#ifdef _DEBUG_CONSOLE_
#include "debugconsole.h"
#endif

#include <wx/filename.h> /* For wxFileName support */

/* the application icon (under Windows and OS/2 it is in resources) */
#if defined(__WXGTK__) || defined(__WXMOTIF__) || defined(__WXMAC__) || defined(__WXMGL__) || defined(__WXX11__)
    #include "../ressources/wxosmose.xpm"
#endif 

#include "enum.h" /* For constants enumeration constitency */

/* ----------------------------------------------------------------------------*/
/* event tables and other macros for wxWindows                                 */
/* ----------------------------------------------------------------------------*/

/* the event tables connect the wxWindows events with the functions (event    */
/* handlers) which process them. It can be also done at run-time, but for the */
/* simple menu events like this the static method is much simpler.            */
BEGIN_EVENT_TABLE(MyFrame, wxFrame)
    
    /* File */
    EVT_MENU(wxOsmose_OpenRom, MyFrame::OnOpenRom)
    EVT_MENU(wxOsmose_Quit,  MyFrame::OnQuit)
    
    /* Video */
    EVT_MENU(wxOsmose_FullScreen, MyFrame::OnChangeFullScreen)
    EVT_MENU(wxOsmose_Scale2x, MyFrame::OnChangeScale2x)
    EVT_MENU(wxOsmose_TVLike, MyFrame::OnChangeTVMode)
    EVT_MENU(wxOsmose_DarkPalette, MyFrame::OnChangeDarkPaletteMode)
    EVT_MENU(wxOsmose_FPS, MyFrame::OnChangeFPSDisplay)
    EVT_MENU(wxOsmose_Monochrom, MyFrame::OnChangeMonochromMode)
    EVT_MENU(wxOsmose_Bilinear, MyFrame::OnChangeBilinearFiltering)
    
    /* Audio */
    EVT_MENU(wxOsmose_Sound, MyFrame::OnChangeSound)
    EVT_MENU(wxOsmose_TVLike, MyFrame::OnChangeTVMode)
    
    /* Options */
    EVT_MENU(wxOsmose_Keys, MyFrame::OnChangeKeys)
    EVT_MENU(wxOsmose_ExecutablePath, MyFrame::OnChangeExecutablePath)
    #ifdef _DEBUG_CONSOLE_
    EVT_MENU(wxOsmose_DebugConsole, MyFrame::OnOpenDebugConsole)
    #endif
    
    /* Zone */
    EVT_MENU(wxOsmose_Exported, MyFrame::OnChangeExportedMode)
    EVT_MENU(wxOsmose_Pal, MyFrame::OnChangePALMode)
    EVT_MENU(wxOsmose_Jap, MyFrame::OnChangeJapaneseMode)
    
    /* Help */
    EVT_MENU(wxOsmose_About, MyFrame::OnAbout)
END_EVENT_TABLE()

/* ---------------------------------------------------------------------------- */
/* main frame                                                                   */
/* ---------------------------------------------------------------------------- */

/* frame constructor */
MyFrame::MyFrame(const wxString& title, const wxPoint& pos, const wxSize& size, long style)
       : wxFrame(NULL, wxOsmose_MainFrame, title, pos, size, style)
{
    /* set the frame icon */
    SetIcon(wxICON(wxosmose));

#if wxUSE_MENUS
    // create a menu bar
    
    // File Menu
    // ---------
    FileMenu = 0;
    FileMenu = new wxMenu;
    // Open ROM
    // File Menu Set
    FileMenu->Append(wxOsmose_OpenRom, _T("&Open ROM...\tCtrl-O"), _T("Open a Rom and Play."));
    
    if ( (Config.GetExecutableFilename() == _T("")) || !( wxFileExists( Config.GetExecutableFilename() ) ) )
    {
    // We search for the most logical of the emulator
    #ifdef __USE_UNIX98
	if ( wxFileExists( wxString::Format(_T("/usr/games/osmose")) ) )
    #else
        if ( wxFileExists( wxString::Format(_T("osmose.exe")) ) )
    #endif
	{
		Config.SetExecutableFilename(wxString::Format(_T("/usr/games/osmose")));
		FileMenu->Enable(wxOsmose_OpenRom, TRUE);
	}
	else
	{
		FileMenu->Enable(wxOsmose_OpenRom, FALSE);
		wxString Msg;
    		Msg.Printf( _T("The Osmose executable hasn't been found or the Executable Path is Empty.\n Please Set the Executable Path in Paths/Executable Path...\n"));

    		wxMessageBox(Msg, _T("About Osmose"), wxOK | wxICON_INFORMATION, this);
	}
	
    }
    
    FileMenu->Append(wxOsmose_Quit, _T("E&xit\tCtrl-X"), _T("Quit this program."));

    // About Menu
    // ----------
    HelpMenu = 0;
    HelpMenu = new wxMenu;
    // About
    HelpMenu->Append(wxOsmose_About, _T("&About...\tF1"), _T("Show about dialog."));
    
    // Video Menu
    // ----------
    VideoMenu = 0;
    VideoMenu = new wxMenu;
    // Full Screen
    VideoMenu->AppendCheckItem(wxOsmose_FullScreen, _T("&Full Screen\tAlt-F"), _T("Launch a ROM in Full Screen Mode / Window Mode."));
    if ( Config.IsFullScreenEnabled() )
    {
    	VideoMenu->Check(wxOsmose_FullScreen, TRUE);
    }
    
    // Scale2x
    VideoMenu->AppendCheckItem(wxOsmose_Scale2x, _T("Scale Video Size to 2&x\tAlt-X"), _T("Launch a ROM in 2x Scaled Mode / Normal Mode."));
    if ( Config.IsScale2xEnabled() )
    {
    	VideoMenu->Check(wxOsmose_Scale2x, TRUE);
    }
    
    // TV Effect
    VideoMenu->AppendCheckItem(wxOsmose_TVLike, _T("&TV Effect\tAlt-T"), _T("Set / Unset TV Mode."));
    if ( Config.IsTVModeEnabled() )
    {
    	VideoMenu->Check(wxOsmose_TVLike, TRUE);
    }
    
    // Monochrome
    VideoMenu->AppendCheckItem(wxOsmose_Monochrom, _T("&Monochrom\tAlt-M"), _T("Play in Monochrom / Color Mode."));
    if ( Config.IsMonochromEnabled() )
    {
    	VideoMenu->Check(wxOsmose_Monochrom, TRUE);
    }
    
    // FPS
    VideoMenu->AppendCheckItem(wxOsmose_FPS, _T("FP&S\tAlt-S"), _T("Show / Hide FPS in Title Bar."));
    if ( Config.IsFPSDisplayEnabled() )
    {
    	VideoMenu->Check(wxOsmose_FPS, TRUE);
    }
    
    // Dark Palette
    VideoMenu->AppendCheckItem(wxOsmose_DarkPalette, _T("&Dark Palette\tAlt-D"), _T("Enable / Disable Dark Palette."));
    if ( Config.IsDarkPaletteEnabled() )
    {
    	VideoMenu->Check(wxOsmose_DarkPalette, TRUE);
    }

    // Bilinear Filtering
    VideoMenu->AppendCheckItem(wxOsmose_Bilinear, _T("&Bilinear Filtering\tAlt-B"), _T("Enable / Disable Bilinear Filtering."));
    if ( Config.IsBilinearFilteringEnabled() )
    {
    	VideoMenu->Check(wxOsmose_Bilinear, TRUE);
    }

    
    // Audio Menu
    // ----------
    AudioMenu = 0;
    AudioMenu = new wxMenu;
    // Sound
    AudioMenu->AppendCheckItem(wxOsmose_Sound, _T("&Sound\tAlt-S"), _T("Activate / Deactivate Sound."));
    if ( Config.IsSoundEnabled() )
    {
    	AudioMenu->Check(wxOsmose_Sound, TRUE);
    }
    
    // Options Menu
    // ------------
    OptionsMenu = 0;
    OptionsMenu = new wxMenu;
    // Set Keys
    OptionsMenu->Append(wxOsmose_Keys, _T("&Keys...\tAlt-K"), _T("Set Emulator's Keys."));
    // Set Executable Path
    OptionsMenu->Append(wxOsmose_ExecutablePath, _T("Executable &Path...\tAlt-P"), _T("Set Emulator's Path."));
    
    #ifdef _DEBUG_CONSOLE_
    OptionsMenu->AppendSeparator();
    
    OptionsMenu->Append(wxOsmose_DebugConsole, _T("Debug &Console...\tAlt-C"), _T("Debug System For Bruno only !"));
    #endif
    
    // Zone Menu
    // ------------
    ZoneMenu = 0;
    ZoneMenu = new wxMenu;
    // Set Keys
    //ZoneMenu->Append(wxOsmose_Zone_Auto, _T("&Automatic\tAlt-A"), _T("Automatic Rom Type"));
    //ZoneMenu->AppendSeparator();
    ZoneMenu->AppendCheckItem(wxOsmose_Exported, _T("&Exported (Default)\tAlt-E"), _T("Set Rom Type to Exported (USA)."));
    ZoneMenu->AppendCheckItem(wxOsmose_Pal, _T("PA&L\tAlt-L"), _T("Set Rom Type to PAL."));
    ZoneMenu->AppendCheckItem(wxOsmose_Jap, _T("&Japanese\tAlt-J"), _T("Set Rom Type to Japanese."));
    
    if ( Config.IsExportedModeEnabled() )
    {
    	ZoneMenu->Check(wxOsmose_Exported, TRUE);
	ZoneMenu->Check(wxOsmose_Pal, FALSE);
	Config.SetPALModeEnabled(false);
	ZoneMenu->Check(wxOsmose_Jap, FALSE);
	Config.SetJapaneseModeEnabled(false);
    }
    
    if ( Config.IsPALModeEnabled() )
    {
    	ZoneMenu->Check(wxOsmose_Pal, TRUE);
	ZoneMenu->Check(wxOsmose_Exported, FALSE);
	Config.SetExportedModeEnabled(false);
	ZoneMenu->Check(wxOsmose_Jap, FALSE);
	Config.SetJapaneseModeEnabled(false);
    }
    
    if ( Config.IsJapaneseModeEnabled() )
    {
    	ZoneMenu->Check(wxOsmose_Jap, TRUE);
	ZoneMenu->Check(wxOsmose_Pal, FALSE);
	Config.SetPALModeEnabled(false);
	ZoneMenu->Check(wxOsmose_Exported, FALSE);
	Config.SetExportedModeEnabled(false);
    }
    
    // And if none of these are enabled (including Auto), enable Exported before Auto mode
    if ( (!Config.IsExportedModeEnabled()) && (!Config.IsJapaneseModeEnabled()) && (!Config.IsPALModeEnabled()) )
    {
	ZoneMenu->Check(wxOsmose_Exported, TRUE);
	ZoneMenu->Check(wxOsmose_Pal, FALSE);
	Config.SetPALModeEnabled(false);
	ZoneMenu->Check(wxOsmose_Jap, FALSE);
	Config.SetJapaneseModeEnabled(false);
    }

    // now append the freshly created menu to the menu bar...
    MenuBar = 0;
    MenuBar = new wxMenuBar();
    MenuBar->Append(FileMenu, _T("&File"));
    MenuBar->Append(VideoMenu, _T("&Video"));
    MenuBar->Append(AudioMenu, _T("&Audio"));
    MenuBar->Append(OptionsMenu, _T("&Options"));
    MenuBar->Append(ZoneMenu, _T("&Zone"));
    MenuBar->Append(HelpMenu, _T("&Help"));

    // ... and attach this menu bar to the frame
    SetMenuBar(MenuBar);
#endif // wxUSE_MENUS

#if wxUSE_STATUSBAR
    // create a status bar just for fun (by default with 1 pane only)
    CreateStatusBar(1);
    SetStatusText(_T("Welcome to wxOsmose !"));
#endif // wxUSE_STATUSBAR

	// The list Ctrl
	// -------------
	wxSize FrameSize = GetClientSize();
	MyRomListCtrl = 0;
	MyRomListCtrl = new RomListCtrl(this, wxOsmose_List_Rom, wxPoint(0,0), wxSize(FrameSize.GetWidth() - 100, FrameSize.GetHeight()), wxLC_VRULES | wxLC_HRULES
	| wxLC_REPORT | wxLC_SINGLE_SEL );
	MyRomListCtrl->InsertColumn(0, wxString::Format(_T("Name")), wxLIST_FORMAT_LEFT, 150 /* Width */);
	MyRomListCtrl->InsertColumn(1, wxString::Format(_T("Type")), wxLIST_FORMAT_LEFT, 100 /* Width */);
	MyRomListCtrl->InsertColumn(2, wxString::Format(_T("Region")), wxLIST_FORMAT_LEFT, 50 /* Width */);
	
	// Grab the config
	MyRomListCtrl->GetConfig(&Config);
	
	// The Image presenting the rom
	// RomImage = ...;

}


// event handlers

void MyFrame::OnQuit(wxCommandEvent& WXUNUSED(event))
{
    // TRUE is to force the frame to close
    Close(TRUE);
}

void MyFrame::OnAbout(wxCommandEvent& WXUNUSED(event))
{
   MyAboutDialog * AnAboutDialog = 0;
   AnAboutDialog = new MyAboutDialog(this, _T("About wxOsmOse"), wxPoint(-1,-1), wxSize(-1,-1));
   AnAboutDialog->ShowModal();
}

void MyFrame::OnOpenRom(wxCommandEvent& WXUNUSED(event))
{
	#ifdef _DEBUG_
		wxLogMessage(Config.GetUserHomeDirectory());
		wxLogMessage(Config.GetExecutableFilename());
		wxLogMessage(Config.GetLastOpenedDirectory());
	#endif
	
	// Check line-command Options
	wxString Options = _T("");
	
	// Video
	if ( VideoMenu->IsChecked(wxOsmose_FullScreen) )
	{
		Options += _T(" -fs");
	}
	
	if ( VideoMenu->IsChecked(wxOsmose_TVLike) )
	{
		Options += _T(" -tv");
	}
	if ( VideoMenu->IsChecked(wxOsmose_DarkPalette) )
	{
		Options += _T(" -dp");
	}
	
	if ( VideoMenu->IsChecked(wxOsmose_Monochrom) )
	{
		Options += _T(" -monochrom");
	}
	
	if ( VideoMenu->IsChecked(wxOsmose_FPS) )
	{
		Options += _T(" -fps");
	}
	
	if ( VideoMenu->IsChecked(wxOsmose_Scale2x) )
	{
		Options += _T(" -scale2x");
	}

	if ( VideoMenu->IsChecked(wxOsmose_Bilinear) )
	{
		Options += _T(" -bilinear");
	}
	
	// Audio
	if ( !( AudioMenu->IsChecked(wxOsmose_Sound) ) )
	{
		Options += _T(" -nosound");
	}

	// Options
	if ( Config.IsMousepadEnabled() )
	{
		Options += _T(" -paddle");
		Options += wxString::Format(_T(" -acceleration %0.00f"), float(Config.GetMousepadAcceleration()) );
		
	}

	// Zone
	if ( ZoneMenu->IsChecked(wxOsmose_Jap) )
	{
		Options += _T(" -jap");
	}
	
	if ( ZoneMenu->IsChecked(wxOsmose_Pal) )
	{
		Options += _T(" -pal");
	}
	
	// Check if ini file exists
	wxString IniFile = _T("");
	wxString wxOsmoseIniFullName = _T("");
	#ifdef __USE_UNIX98
	wxOsmoseIniFullName = Config.GetUserHomeDirectory();
	wxOsmoseIniFullName += _T("/.osmose/wxosmose.ini");
	#endif
	#ifdef _WIN32
	wxOsmoseIniFullName = wxGetCwd(); // The current directory...
	wxOsmoseIniFullName += _T("\wxosmose.ini"); 
	#endif
	
	if ( !( wxFileExists( wxOsmoseIniFullName ) ) )
	{
		IniFile = _T(""); // Use defaults
	}
	else
	{
		IniFile = _T(" -inifile \"");
		IniFile += wxOsmoseIniFullName;
		IniFile += _T("\"");
	}
	
	wxString RomPath = _T("");
	wxString Command = _T("\"");
	Command += Config.GetExecutableFilename();
	Command += _T("\"");
	if ( Command == wxString::Format(_T("\"\"")) )
	{
		Command = _T("./osmose");
	}
	
	wxFileDialog RomDialog(this, _T("Choose the ROM to open"), Config.GetLastOpenedDirectory(), _T(""), _T("ZIP files with SMS file within (*.zip)|*.zip|SMS files (*.sms)|*.sms"), wxOPEN | wxHIDE_READONLY );
	
	if ( RomDialog.ShowModal() == wxID_OK )
	{
		RomPath = _T(" \"");
		RomPath += RomDialog.GetPath();
		RomPath += _T("\"");
		Config.SetLastOpenedDirectory( wxPathOnly( RomDialog.GetPath() ) );
		Command += RomPath + Options + IniFile; 
		#ifdef _DEBUG_
			wxLogMessage(Command);
		#endif
		/*int ReturnCode =*/ wxExecute(Command, wxEXEC_SYNC);
		UpdateRomList();
	}
}

// Video
void MyFrame::OnChangeFullScreen(wxCommandEvent& WXUNUSED(event))
{

	if ( VideoMenu->IsChecked(wxOsmose_FullScreen) )
	{
		Config.SetFullScreenEnabled(true);
	}
	else
	{
		Config.SetFullScreenEnabled(false);
	}

}

void MyFrame::OnChangeScale2x(wxCommandEvent& WXUNUSED(event))
{

	if ( VideoMenu->IsChecked(wxOsmose_Scale2x) )
	{
		Config.SetScale2xEnabled(true);
	}
	else
	{
		Config.SetScale2xEnabled(false);
	}

}

void MyFrame::OnChangeTVMode(wxCommandEvent& WXUNUSED(event))
{
	if ( VideoMenu->IsChecked(wxOsmose_TVLike) )
	{
		Config.SetTVModeEnabled(true);
	}
	else
	{
		Config.SetTVModeEnabled(false);
	}
}

void MyFrame::OnChangeDarkPaletteMode(wxCommandEvent& WXUNUSED(event))
{
	if ( VideoMenu->IsChecked(wxOsmose_DarkPalette) )
	{
		Config.SetDarkPaletteEnabled(true);
	}
	else
	{
		Config.SetDarkPaletteEnabled(false);
	}
}

void MyFrame::OnChangeMonochromMode(wxCommandEvent& WXUNUSED(event))
{
	if ( VideoMenu->IsChecked(wxOsmose_Monochrom) )
	{
		Config.SetMonochromEnabled(true);
	}
	else
	{
		Config.SetMonochromEnabled(false);
	}
}

void MyFrame::OnChangeFPSDisplay(wxCommandEvent& WXUNUSED(event))
{
	if ( VideoMenu->IsChecked(wxOsmose_FPS) )
	{
		Config.SetFPSDisplayEnabled(true);
	}
	else
	{
		Config.SetFPSDisplayEnabled(false);
	}
}

void MyFrame::OnChangeBilinearFiltering(wxCommandEvent& WXUNUSED(event))
{
	if ( VideoMenu->IsChecked(wxOsmose_Bilinear) )
	{
		Config.SetBilinearFilteringEnabled(true);
	}
	else
	{
		Config.SetBilinearFilteringEnabled(false);
	}
}

// Audio
void MyFrame::OnChangeSound(wxCommandEvent& WXUNUSED(event))
{
	if ( AudioMenu->IsChecked(wxOsmose_Sound) )
	{
		Config.SetSoundEnabled(true);
	}
	else
	{
		Config.SetSoundEnabled(false);
	}
}

// Options & Paths
void MyFrame::OnChangeKeys(wxCommandEvent& WXUNUSED(event))
{
	KeyDialog ChangeKeysDialog(this, _T("Change Keys..."), wxPoint(-1,-1), wxSize(510,355));
	ChangeKeysDialog.ShowModal();
}

void MyFrame::OnChangeExecutablePath(wxCommandEvent& WXUNUSED(event))
{
	wxFileDialog ExecDialog(this, _T("Choose where the Osmose Executable is."), _T(""), _T(""), _T("Executable files (*)|*"), wxOPEN | wxHIDE_READONLY );
	
	if ( ExecDialog.ShowModal() == wxID_OK )
	{
		wxString ExecutablePath = ExecDialog.GetPath();
		Config.SetExecutableFilename(ExecutablePath);
		FileMenu->Enable(wxOsmose_OpenRom, TRUE);
		#ifdef _DEBUG_
			wxLogMessage(ExecutablePath);
		#endif
	}
}

#ifdef _DEBUG_CONSOLE_
void MyFrame::OnOpenDebugConsole(wxCommandEvent& WXUNUSED(event))
{
	DebugDialog * DebugConsole = new DebugDialog(this, _T("Debug Console --Osmose Developers Only--"), wxPoint(-1,-1), &Config);
	DebugConsole->ShowModal();
}
#endif

// Zone
void MyFrame::OnChangeExportedMode(wxCommandEvent& WXUNUSED(event))
{
	//wxLogMessage(_T("Exported"));
	
	if ( ZoneMenu->IsChecked(wxOsmose_Exported) )
	{
		Config.SetExportedModeEnabled(true);
		ZoneMenu->Check(wxOsmose_Pal, FALSE);
		ZoneMenu->Check(wxOsmose_Jap, FALSE);
	}
	else
	{
		Config.SetExportedModeEnabled(false);
		
		// And if none of these are enabled (including Auto), enable Exported before Auto mode
		if ( (!Config.IsExportedModeEnabled()) && (!Config.IsJapaneseModeEnabled()) && (!Config.IsPALModeEnabled()) )
		{
			ZoneMenu->Check(wxOsmose_Exported, TRUE);
			ZoneMenu->Check(wxOsmose_Pal, FALSE);
			Config.SetPALModeEnabled(false);
			ZoneMenu->Check(wxOsmose_Jap, FALSE);
			Config.SetJapaneseModeEnabled(false);
		}
	}
}

void MyFrame::OnChangePALMode(wxCommandEvent& WXUNUSED(event))
{
	//wxLogMessage(_T("PAL"));
	
	if ( ZoneMenu->IsChecked(wxOsmose_Pal) )
	{
		Config.SetPALModeEnabled(true);
		ZoneMenu->Check(wxOsmose_Exported, FALSE);
		ZoneMenu->Check(wxOsmose_Jap, FALSE);
	}
	else
	{
		Config.SetPALModeEnabled(false);
		
		// And if none of these are enabled (including Auto), enable Exported before Auto mode
		if ( (!Config.IsExportedModeEnabled()) && (!Config.IsJapaneseModeEnabled()) && (!Config.IsPALModeEnabled()) )
		{
			ZoneMenu->Check(wxOsmose_Exported, TRUE);
			ZoneMenu->Check(wxOsmose_Pal, FALSE);
			Config.SetPALModeEnabled(false);
			ZoneMenu->Check(wxOsmose_Jap, FALSE);
			Config.SetJapaneseModeEnabled(false);
		}
	}
}

void MyFrame::OnChangeJapaneseMode(wxCommandEvent& WXUNUSED(event))
{
	//wxLogMessage(_T("Japanese"));
	
	if ( ZoneMenu->IsChecked(wxOsmose_Jap) )
	{
		Config.SetJapaneseModeEnabled(true);
		ZoneMenu->Check(wxOsmose_Pal, FALSE);
		ZoneMenu->Check(wxOsmose_Exported, FALSE);
	}
	else
	{
		Config.SetJapaneseModeEnabled(false);
		
		// And if none of these are enabled (including Auto), enable Exported before Auto mode
		if ( (!Config.IsExportedModeEnabled()) && (!Config.IsJapaneseModeEnabled()) && (!Config.IsPALModeEnabled()) )
		{
			ZoneMenu->Check(wxOsmose_Exported, TRUE);
			ZoneMenu->Check(wxOsmose_Pal, FALSE);
			Config.SetPALModeEnabled(false);
			ZoneMenu->Check(wxOsmose_Jap, FALSE);
			Config.SetJapaneseModeEnabled(false);
		}
	}
}

// Private
void MyFrame::UpdateRomList()
{
	// We look into the latest opened directory
	if ( wxDirExists( Config.GetLastOpenedDirectory() ) )
	{
		// We clean the list
		MyRomListCtrl->DeleteAllItems();
		
		// We prepare the MD5 calculator
		MD5Sum MD5SumOfFile;
		
		// We load the known sms list.
		RomsList SmsList(_T("datas/MasterSystem/roms.list"));
		
		// A backend to get all the files in a alphabetical sorted way.
		wxArrayString SortedFoundSmsFiles;
		SortedFoundSmsFiles.Clear();
		
		// We search all the files and sort them
		wxString FileToSeek = wxFindFirstFile( wxString::Format(_T("%s/*.sms"), (Config.GetLastOpenedDirectory()).c_str()), wxFILE );
		while ( !FileToSeek.IsEmpty() )
		{
			SortedFoundSmsFiles.Add(FileToSeek);
			FileToSeek = wxFindNextFile();
		}
		FileToSeek = wxFindFirstFile( wxString::Format(_T("%s/*.zip"), (Config.GetLastOpenedDirectory()).c_str()), wxFILE );
		while ( !FileToSeek.IsEmpty() )
		{
			SortedFoundSmsFiles.Add(FileToSeek);
			FileToSeek = wxFindNextFile();
		}
		SortedFoundSmsFiles.Sort();
		
		long ListIndex = 0;
		long Count = SortedFoundSmsFiles.GetCount();
		while ( ListIndex < Count )
		{
			// We record the romfilename
			MyRomListCtrl->RomFilenames.Add(SortedFoundSmsFiles[ListIndex]);
			
			// Add items to list
			if ( SortedFoundSmsFiles[ListIndex].substr( SortedFoundSmsFiles[ListIndex].length() - 3, 3 ) == _T("sms") )
			{
				MD5SumOfFile.CalculateFromFile(SortedFoundSmsFiles[ListIndex]);
			}
			else if ( SortedFoundSmsFiles[ListIndex].substr( SortedFoundSmsFiles[ListIndex].length() - 3, 3 ) == _T("zip") )
			{
				MD5SumOfFile.CalculateFromZipFile(SortedFoundSmsFiles[ListIndex], wxString::Format(_T("sms")));
			}
			if ( SmsList.SetIndexToMD5Sum(MD5SumOfFile.GetHash()) ) // We found an occurence
			{
				MyRomListCtrl->InsertItem(ListIndex, SmsList.GetRomName());
				MyRomListCtrl->SetItem(ListIndex, 1, SmsList.GetRomType());
				MyRomListCtrl->SetItem(ListIndex, 2, SmsList.GetRomRegion());
			}
			else
			{
				wxFileName MyFile(SortedFoundSmsFiles[ListIndex]);
				wxString RealName = MyFile.GetName();
				RealName += _T(".");
				RealName += MyFile.GetExt();
				MyRomListCtrl->InsertItem(ListIndex, RealName);
				MyRomListCtrl->SetItem(ListIndex, 1, wxString::Format(_T("Unknown")));
				MyRomListCtrl->SetItem(ListIndex, 2, wxString::Format(_T("Unknown")));
			}
			ListIndex++;
		} // While there is files to identify
	} // If last directory exists
}
