#include <string.h>
#include "gba_defs.h"

extern u32 gbarotscale[];
void get_new_keyconfig(void);

void LZ77UnCompVram(void *src,void *dest) {
	__asm{swi 0x12}
}

//return ptr to end of gfx data (start of SNES rom data)
u32 *unpack_gfx() {
	u32 *p=userdata;

	LZ77UnCompVram(p+1,AGB_VRAM);	//bg
	p=(u32*)((u32)p+*p+4);
	LZ77UnCompVram(p+1,AGB_PALETTE);		//bgpal
	p=(u32*)((u32)p+*p+4);
	LZ77UnCompVram(p+1,AGB_VRAM+0x14000);	//font
	p=(u32*)((u32)p+*p+4);
	memcpy(AGB_PALETTE+0x300,p+1,32);	//fontpal
	p=(u32*)((u32)p+*p+4);

	romdata=p;

	return p;
}

void change_to_menu_screenmode() {
	REG_DISPCNT=BG2_EN|OBJ_EN|MODE4|OBJ_1D;
	REG_BG2CNT=0x41;		//mosaic, priority 1
	REG_BLDCNT=0x0440;
	REG_BLDALPHA=0x00a08;
	REG_MOSAIC=0;
	gbarotscale[0]=0x00000100;
	gbarotscale[1]=0x01000000;
	gbarotscale[2]=0x00000000;
	gbarotscale[3]=0x00000000;
}

//returns next available sprite ptr
const signed char wave[32]={0,1,2,3,3,3,2,1,0,-1,-2,-3,-3,-3,-2,-1,0,1,2,3,3,3,2,1,0,-1,-2,-3,-3,-3,-2,-1};
u16 *sinestring(u16 *oam,char *s,int x,int y,int waveoffset) {
	u32 c;
	waveoffset>>=1;
	while((c=*s)!=0 && x<240) {
		if(c!=' ' && x>-16) {
			oam[0]=y+wave[waveoffset&31];
			oam[1]=0x4000+(x&511);
			oam[2]=0x8200+(c-32)*4;
			oam+=4;
		}
		x+=16;
		s++;
		waveoffset++;
	}
	return oam;
}

//returns next available sprite ptr
u16 *spritestring(u16 *oam,char *s,int x,int y) {
	u32 c;
	y=(y&255)|0x400;
	while((c=*s)!=0 && x<240) {
		if(c!=' ' && x>-16) {
			oam[0]=y;
			oam[1]=0x4000+(x&511);
			oam[2]=0x8600+(c-32)*4;
			oam+=4;
		}
		x+=16;
		s++;
	}
	return oam;
}

typedef char* (*sfunc)(int,void*);
//getstring returns text at line (i,?)
//y=where to start drawing menu
//selected=which line is selected
//xscroll=x offset of selected line (for text too long to fit)
int drawmenu(sfunc getstring,void *param,int y,int selected,int xscroll) {
	static int wave=0;
	int line=0;
	u16 *oam=(u16*)AGB_OAM;
	int xtmp;
	int len;
	char *s;

	wave++;
	do {
		s=getstring(line,param);
		if(s && y>-16) {
			len=strlen(s);
			xtmp=120-len*8;
			if(len>15)
				xtmp=0;
			if(line==selected) {
				if(len>15) {
					xscroll++;
					xtmp-=xscroll;
					oam=sinestring(oam,s,xtmp,y,wave);
					xtmp+=len*16+64;
				}
				oam=sinestring(oam,s,xtmp,y,wave);
				if(xtmp==1) xscroll=0;	//loop
			} else {
				oam=spritestring(oam,s,xtmp,y);
			}
		}
		line++;
		y+=16;
	} while(s && y<160);
	while((u32)oam<0x7000400) {	//hide unused sprites
		oam[0]=0x0200;
		oam+=4;
	}
	return xscroll;
}

//p=ptr to rom list
char *findrom(int i,u32 *p) {
	if(*p<=i)		//if (requested rom) > (total roms)
		return 0;
	p++;
	while(i--)
		p=(u32*)((u32)p+((romheader*)p)->romsize+sizeof(romheader));
	return (char*)p;
}

const signed char scrolloffset[]={0,-1,-1,-2,-4,-6,-8,-10,-13,1,1,2,4,6,8,10,13};
//returns selected menuitem (|256 if start was pushed)
int do_menu(sfunc getstring,void *param,int selected,int menuitems) {
	int y,joypad,joymask,yscroll,xscroll;

	menuitems--;
	y=72-selected*16;
	joymask=readpad();
	yscroll=0;
	xscroll=0;
	do {
		waitframe();
		xscroll=drawmenu(getstring,param,y+scrolloffset[yscroll],selected,xscroll);
		joypad=readpad()^joymask;
		joymask^=joypad;
		joypad&=joymask;
		if(yscroll) {
			yscroll--;
			if(yscroll==8) yscroll=0;
		}
		if(joypad&DPAD_UP && selected) {
			xscroll=0;
			yscroll=8;
			y+=16;
			selected--;
		} else if(joypad&DPAD_DOWN && selected<menuitems) {
			xscroll=0;
			yscroll=16;
			y-=16;
			selected++;
		}
	} while(!(joypad&(B_BUTTON|A_BUTTON|START_BUTTON)));
	if(joypad&START_BUTTON)
		selected|=256;
	return selected;
}

void fade_to_black() {
	int i,j;
	REG_DISPCNT=BG2_EN|MODE4;
	REG_BLDCNT=0x00c4;
	REG_BLDY=0;
	for(i=0;i<16;i++) {		//fade to black w/ mosaic
		waitframe();
		j=i;
		REG_BLDY=j;
		REG_MOSAIC=j|j<<4;
	}
	REG_BLDCNT=0;
}

void white_blurry() {
	int i;
	REG_DISPCNT=BG2_EN|MODE4;
	REG_BLDCNT=0x0084;
	REG_BLDY=0;
	for(i=0;i<16;i++) {		//fade to black w/ mosaic
		waitframe();
		REG_BLDY=i;
		REG_MOSAIC=i|i<<4;
	}
	while(i>=0) {		//fade to black w/ mosaic
		waitframe();
		REG_BLDY=i;
		REG_MOSAIC=i|i<<4;
		i--;
	}
	REG_BLDCNT=0;
}

//show list of roms, returns ptr to selected rom
romheader *romchooser() {
	u32 *p;
	p=unpack_gfx();
	if(*p>1) {			//more than one rom?
		change_to_menu_screenmode();
		p=(u32*)findrom(do_menu((sfunc)findrom,p,0,*p)&255,p);
		fade_to_black();
	} else
		p++;
	return (romheader*)p;
}

typedef const char* cc;
typedef const cc* ccc;	//there MUST be an easier way of doing this..??
enum {GAMMA, THROTTLE, KEYCONFIG, RESET, GAMEMENU, SETTINGOPTIONS};
const cc settingtext[]={"BRIGHT=","THROTTLE=","KEY CONFIG","RESET GAME","ROM MENU"};
const cc bright[]={"LOW","HIGH"};
const cc vbl_txt[]={"VBLANK","NONE","5.00","5.99","6.14","6.55","7.37","7.68"};
const cc empty[]={""};
const ccc strset[]={bright,vbl_txt,empty,empty,empty};

typedef struct {
	char options[SETTINGOPTIONS];
	char n_settings;
	char tmpstr[32];
} setstruct;

char *lookup_settingstext(int i,setstruct *param) {
	if(i>=param->n_settings)
		return 0;
	else {
		strcpy(param->tmpstr,settingtext[i]);
		strcat(param->tmpstr,strset[i][param->options[i]]);
		return param->tmpstr;
	}
}

int settings(romheader *r) {
	int n_settings=SETTINGOPTIONS-(*romdata==1);	//minus one if #roms=1 (remove "GAME MENU" option)
	setstruct settings;
	int i;
	int selected=0;

	memset(&settings,0,sizeof(settings));
	settings.options[GAMMA]=gamma;
	settings.options[THROTTLE]=throttle_type;
	settings.n_settings=n_settings;
	unpack_gfx();
	change_to_menu_screenmode();
	do {
		selected=do_menu((sfunc)lookup_settingstext,&settings,selected,n_settings);
		switch(selected) {
			case GAMMA:	//brightness
				settings.options[GAMMA]^=1;
				break;
			case THROTTLE:
				i=settings.options[THROTTLE];
				i++;
				if(i>8)
					i=0;
				settings.options[THROTTLE]=i;
				break;
			case KEYCONFIG:
				get_new_keyconfig();
				break;
			case GAMEMENU:	//rom menu
				white_blurry();
				break;
			case RESET:	//reset
				hardreset(r+1,r->flags);
			default: //(start pushed)
				fade_to_black();
				selected=RESET;
				break;
		}
	} while(selected<RESET);
	gamma=settings.options[GAMMA];
	set_throttle_timer(settings.options[THROTTLE]);
	while(readpad());
	return selected-RESET;	//1=return to rom menu.  0=continue running current rom
}

u8 vblankreadkey() {
	u32 i;
	waitframe();
	i=readpad();
	return (i&0x3f)|((i&0x300)>>2);
}

const char _y[]="Y";
const char _x[]="X";
const char _b[]="B";
const char _a[]="A";
const char _l[]="L";
const char _r[]="R";
const char _select[]="SEL";
const char _start[]="START";
const cc sneskeys[]={_y,_b,_x,_a,_l,_r,_start,_select};
const cc gbakeys[]={_a,_b,_select,_start,_r,_l,_r,_l};

char *keyconfigtext(int i,u8 *keys) {
	char *c;
	u8 k;
	if(!i)
		return "PUSH TO SET";
	if(i>8)
		return 0;
	i--;
	c=(char*)keys+8;
	strcpy(c,sneskeys[i]);
	k=keys[i];
	strcat(c,k?"=":"?");
	i=0;
	while(k) {
		if(k&1)
			strcat(c,gbakeys[i]);
		i++;
		k>>=1;
	}
	return c;
}

void get_new_keyconfig() {
	u8 keys[8+32];	//y x b a l r select start
	u8 c1,c2;
	int i;
	
	*(u32*)keys=0;	//clear keys
	*(u32*)&keys[4]=0;
	
	while(vblankreadkey());
	i=0;
	c2=0;
	do {
		drawmenu((sfunc)keyconfigtext,keys,8,i+1,0);
		c1=vblankreadkey();
		c2|=c1;
		if(!c1 && c2) {
			keys[i++]=c2;
			c2=0;
		}
	} while(i<8);
	set_keyconfig(keys);
}

//modifies readkeyinput() code in IO.S
void set_keyconfig(const u8 *keys) {
	int i;
	u8 key;
	u8 *kc=&keyconfig;
	for(i=0;i<8;i++) {
		key=keys[i];
		*kc=key;
		*(kc+4)=key;
		kc+=12;
	}
}