// GBA MMU emulator
// Mic, 2004

#include <windows.h>
#include <commctrl.h>
#include <string.h>
#include "zombie.h"
#include <stdio.h>

#define JUMP 0

#define NR10	0x60
#define NR11	0x62
#define NR12	0x63
#define NR13	0x64
#define NR14	0x65
#define NR21	0x68
#define NR22	0x69
#define NR23	0x6C
#define NR24	0x6D
#define NR30	0x70
#define NR31	0x72
#define NR32	0x73
#define NR33	0x74
#define NR34	0x75
#define NR41	0x78
#define NR42	0x79
#define NR43	0x7C
#define NR44	0x7D
#define NR50	0x80
#define NR51	0x81
#define NR52	0x84

#define TM0D	0x100
#define TM0CNT	0x102
#define TM1D	0x104
#define TM1CNT	0x106
#define TM2D	0x108
#define TM2CNT	0x10A
#define TM3D	0x10C
#define TM3CNT	0x10E
#define P1		0x130
#define P1CNT	0x132
#define IE		0x200
#define IF_		0x202
#define IME		0x208


typedef struct
{
	int sad,
		dad,
		cnt_l,
		cnt_h;
	bool enabled;
	unsigned int sad_int,dad_int,cnt_int;
}dma_channel;

typedef struct
{
	unsigned int tstart;
	unsigned short int tcurr,td,te;
	bool enabled,countup;
}timer_channel;

const unsigned char prescalar[4] = {0,6,8,10};
const int keymask[] = {0x040,0x080,0x020,0x010,0x001,0x002,0x200,0x100,0x008,0x004};

HINSTANCE hInst;
HANDLE hout;
unsigned char *ROM,*SYSROM,*IWRAM,*EXRAM,*cart,*VRAM,*IOREGS,*mmu_memory;
unsigned char *iomap,**mmap,*pageTable[16];
unsigned int pageLength[16],pageMask[16];
int keycodes[] = {38,40,37,39,88,90,76,82,13,32};

unsigned short int keys,irqFlags;
int *cpu_regs;
int romOffset,romSize;
int iwramOffset,iwramSize;
int exramOffset,exramSize;
int vramOffset,vramSize;
char mmuName[] = "GBA MMU";
int dummy;
unsigned int *cycle;
unsigned int addressLatch;
cpu_irq_callback cpu_irq;
apu_write_callback apu_write;
zombie_callback zombie;
char buffer[80];
FILE *log;
dma_channel dma[4];
timer_channel timer[4];

unsigned int sysromcode_0[8] = {
	0,			// 00000000
	0,			// 00000004
	0,			// 00000008
	0,			// 0000000C
	0,			// 00000010
	0,			// 00000014
	0xEA000042,	// 00000018: B $128
	0			// 0000001C
};

unsigned int sysromcode_128[6] = {
	0xE92D500F,	// 00000128: STMFD R13!,{R0-R3,R12,R14}
	0xE3A00301,	// 0000012C: MOV R0,#$04000000
	0xE28FE000,	// 00000130: ADD R14,R15,#0
	0xE510F004,	// 00000134: LDR R15,[R0,-4]
	0xE8BD500F,	// 00000138: LDMFD R13!,{R0-R3,R12,R14}
	0xE25EF004	// 0000013C: SUBS R15,R14,#4
};




BOOL CALLBACK DlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam);



bool __stdcall DllMain(HINSTANCE hDLL,DWORD reason,LPVOID unused)
{
	if (reason==DLL_PROCESS_ATTACH)
		hInst = hDLL;

	return true;
}



bool __stdcall mmu_init(zombie_callback commlink)
{
	int i;

	zombie = commlink;

	romOffset = 0x8000000;
	romSize = 0x8000000;

	exramOffset = 0x2000000;
	exramSize = 0x40000;

	iwramOffset = 0x3000000;
	iwramSize = 0x8000;

	vramOffset = 0x6000000;
	vramSize = 0x40000;

	mmu_memory = (unsigned char*)LocalAlloc(LPTR,iwramSize+exramSize+0x4000);
	IWRAM = mmu_memory;
	EXRAM = IWRAM + iwramSize;
	SYSROM = EXRAM + exramSize;
	//IOREGS = (unsigned char*)LocalAlloc(LPTR,0x400);

	pageTable[3] = IWRAM;
	pageTable[2] = EXRAM;
	pageTable[0] = SYSROM;

	pageLength[3] = 0x8000;
	pageLength[2] = 0x40000;
	pageLength[0] = 0x4000;

	for (i=0; i<4; i++)
	{
		dma[i].sad = 0xB0 + (i*0xC);
		dma[i].dad = 0xB4 + (i*0xC);
		dma[i].cnt_l = 0xB8 + (i*0xC);
		dma[i].cnt_h = 0xBA + (i*0xC);

		timer[i].enabled = false;
		timer[i].tcurr = 0;
		timer[i].tstart = 0;
	}

	/*iomap = new unsigned char[0x10000];

	for (i=0; i<0x10000; i++)
		iomap[i] = 0xFF;*/

	//log = fopen("mmulog.txt","wb");

	hout = GetStdHandle(STD_OUTPUT_HANDLE);
	return true;

}


bool __stdcall mmu_reset()
{
	int i;

	//cpu_inpb = (cpu_inpb_callback)zombie(ASK,CPU_INPB_CALLBACK,0,0);
	//cpu_outpb = (cpu_outpb_callback)zombie(ASK,CPU_OUTPB_CALLBACK,0,0);

	cpu_irq = (cpu_irq_callback)zombie(ASK,CPU_IRQ_CALLBACK,0,0);
	apu_write = (apu_write_callback)zombie(ASK,APU_WRITE_CALLBACK,0,0);
	cpu_regs = (int*)zombie(ASK,CPU_REG,0,0);

	cart = (unsigned char*)zombie(ASK,ZOMBIE_ROM_BASE,0,0);
	ROM = cart;

	zombie(TELL,GPU_VBLANK_ACTION,(void*)JUMP,(void*)0x38);
	zombie(TELL,GPU_HBLANK_ACTION,(void*)JUMP,(void*)0x38);

	zombie(TELL,CPU_REG_PC,(void*)0x8000000,0);
	VRAM = (unsigned char*)zombie(ASK,GPU_MEMORY,0,0);
	IOREGS = &VRAM[0x1D000];

	IOREGS[0x130] = 0xFF;
	keys = 0x3FF;

	pageTable[0x8] = ROM;
	pageTable[0xA] = ROM;
	pageTable[0xC] = ROM;
	pageTable[7] = VRAM+0x20000;
	pageTable[6] = VRAM;
	pageTable[5] = VRAM+0x1C000;
	pageTable[4] = IOREGS;

	pageLength[8] = (unsigned int)zombie(ASK,ZOMBIE_ROM_SIZE,0,0);
	pageLength[7] = 0x400;
	pageLength[6] = 0x18000;
	pageLength[5] = 0x400;
	pageLength[4] = 0x400;

	for (i=0; i<16; i++)
		pageMask[i] = pageLength[i] - 1;

	// Align the ROM size mask to the nearest 2^n boundary
	i = 0;
	while (true)
	{
		if (((1<<i)<=pageMask[8]) && ((1<<(i+1))>pageMask[8]))
			break;
		i++;
	}
	pageMask[8] = (1<<(i+1)) - 1;

	for (i=0; i<4; i++)
	{
		timer[i].enabled = false;
		timer[i].countup = false;
		timer[i].tcurr = 0;
		timer[i].tstart = 0;

		dma[i].sad_int = 0;
		dma[i].dad_int = 0;
		dma[i].enabled = false;
	}

	// Clear IWRAM
	memset(IWRAM,0,iwramSize);

	memcpy(SYSROM,sysromcode_0,8*4);
	memcpy(SYSROM+0x128,sysromcode_128,6*4);

	srand(123);

	irqFlags = 0;

	return true;
}


void __stdcall mmu_close()
{
	//fclose(log);
	LocalFree(mmu_memory);
}



int __stdcall mmu_config(HWND hwnd)
{
	return 0; /*DialogBoxParam(hInst,
            MAKEINTRESOURCE(IDD_MMU_CONFIG), hwnd,
            DlgProc, NULL);*/
}




unsigned char* __stdcall mmu_translate_address(unsigned int address)
{
	return pageTable[address>>24] + (address&0xFFFFFF);
}



void dma_transfer(void *src,void *dest,int cnt)
{
	memcpy(dest,src,cnt);
}



unsigned short int __stdcall mmu_read_word(unsigned int);
void __stdcall mmu_write_word(unsigned int,int);
void __stdcall mmu_irq(int);


void dma_setup(int channel,bool forceDMA)
{
	int i,cnt;
	unsigned char sadcnt;
	unsigned short int cnts;
	unsigned int src,dest,srcend,destend;

	if ( ((IOREGS[dma[channel].cnt_h+1] & 0xB0) == 0x80) ||
		 (((IOREGS[dma[channel].cnt_h+1] & 0xB0) >= 0x80) && forceDMA) )
	{
		if (dma[channel].enabled == false)
		{
			cnts = *(unsigned short*)(IOREGS + dma[channel].cnt_l);
			dma[channel].cnt_int = cnts;
		}
		cnt = dma[channel].cnt_int;
		cnt <<= ((IOREGS[dma[channel].cnt_h+1] & 4) == 4) ? 2 : 1;

		src = (*(unsigned int*)(IOREGS + dma[channel].sad)) & 0xFFFFFFF;
		dest = (*(unsigned int*)(IOREGS + dma[channel].dad)) & 0xFFFFFFF;
		if (dma[channel].enabled == false)
		{
			dma[channel].sad_int = src;
			dma[channel].dad_int = dest;
		}
		dma[channel].enabled = true;

		sadcnt = IOREGS[dma[channel].cnt_h] & 0x80;
		sadcnt |= IOREGS[dma[channel].cnt_h+1] & 0x01;

		/*wsprintf(buffer,"DMA:ing %d bytes from %08x to %08x SADCNT=%x (%04x) (PC=%08x)\n",
			cnt,src,dest,cnts,sadcnt,(unsigned int)zombie(ASK,CPU_REG_PC,0,0));
		WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);*/

		if (sadcnt == 0x00)
		{
			srcend = (dma[channel].sad_int + cnt) & 0xFFFFFF;
			if (srcend >= pageLength[dma[channel].sad_int >> 24])
				cnt -= (srcend - pageLength[dma[channel].sad_int >> 24]);
			if (cnt<=0) return;
		}

		destend = (dma[channel].dad_int + cnt) & 0xFFFFFF;
		if (destend >= pageLength[dma[channel].dad_int >> 24])
			cnt -= (destend-pageLength[dma[channel].dad_int >> 24]);
		if (cnt<=0) return;


		//dma_transfer((void*)mmu_translate_address(src),(void*)mmu_translate_address(dest),cnt);

		if (sadcnt == 0x00)
		{
			for (i=cnt; i>0; i-=2)
				mmu_write_word(dma[channel].dad_int+i-2,mmu_read_word(dma[channel].sad_int+i-2));
		} else if (sadcnt == 0x01)
		{
			for (i=cnt; i>0; i-=2)
				mmu_write_word(dma[channel].dad_int+i-2,mmu_read_word(dma[channel].sad_int));
		}

		switch (IOREGS[dma[channel].cnt_h] & 0x60)
		{
			case 0x00:
				dma[channel].dad_int += cnt;
				break;
			case 0x20:
				dma[channel].dad_int -= cnt;
				break;
			default:
				break;
		}
		//*(unsigned int*)(IOREGS + dma[channel].dad) = dma[channel].dad_int; //dest;


		switch (sadcnt)
		{
			case 0x00:
				dma[channel].sad_int += cnt;
				break;
			case 0x80:
				dma[channel].sad_int -= cnt;
				break;
			default:
				break;
		}
		//*(unsigned int*)(IOREGS + dma[channel].sad) = dma[channel].sad_int; //src;

		// Reset the DMA enable bit unless repeat mode is specified
		if ((IOREGS[dma[channel].cnt_h+1] & 0x02) == 0)
		{
			IOREGS[dma[channel].cnt_h+1] &= 0x7F;
			dma[channel].enabled = false;
		}

		// Generate an IRQ if needed
		if (IOREGS[dma[channel].cnt_h+1] & 0x40)
			if (IOREGS[IE+1]&(1<<channel))
				mmu_irq(0x100<<channel);
	} else
		dma[channel].enabled = false;
}



void timer_setup(int channel)
{
	timer[channel].tstart = *cycle;
	timer[channel].countup = ((IOREGS[TM0CNT+channel*4]&0x4)!=0);
	timer[channel].td = *(unsigned short int*)&IOREGS[TM0D + channel*4];

	if ((IOREGS[TM0CNT+channel*4]&0x80)&&(!timer[channel].enabled))
	{
		timer[channel].tcurr = timer[channel].td;
		timer[channel].te = timer[channel].td;
	}

	timer[channel].enabled = false;
	if ((IOREGS[TM0CNT+channel*4]&0x40) &&
		((IOREGS[TM0CNT+channel*4]&0x04)==0))
			zombie(TELL,ZOMBIE_EVENT,(void*)(EV_TIMER_OVERFLOW|(channel<<16)),
				   (void*)((*cycle)+((0x10000-timer[channel].tcurr)<<prescalar[IOREGS[TM0CNT+channel*4]&3])));
	else if ((channel<3) && ((IOREGS[TM0CNT+channel*4]&0x80)!=0))
		if (timer[channel+1].countup)
			zombie(TELL,ZOMBIE_EVENT,(void*)(EV_TIMER_OVERFLOW|(channel<<16)),
				   (void*)((*cycle)+((0x10000-timer[channel].tcurr)<<prescalar[IOREGS[TM0CNT+channel*4]&3])));

	if ((channel!=0) && ((IOREGS[TM0CNT+channel*4]&0x04)!=0))
	{
		if (timer[channel-1].enabled)
		{
		//WriteConsole(hout,"Starting countup timer\n",24,(LPDWORD)dummy,NULL);
			zombie(TELL,ZOMBIE_EVENT,(void*)(EV_TIMER_OVERFLOW|((channel-1)<<16)),
			   (void*)(timer[channel-1].tstart + ((0x10000-timer[channel-1].td)<<prescalar[IOREGS[TM0CNT+(channel-1)*4]&3])));
		}
	}
	if (IOREGS[TM0CNT+channel*4]&0x80)
		timer[channel].enabled = true;
}




void __stdcall mmu_irq(int mask)
{
	int i;

	if (mask & 2)
		for (i=1; i<4; i++)
			if ((IOREGS[dma[i].cnt_h+1] & 0xB0) == 0xA0)
				dma_setup(i,true);

	if (mask & 1)
		for (i=1; i<4; i++)
			if ((IOREGS[dma[i].cnt_h+1] & 0xB0) == 0x90)
				dma_setup(i,true);

	if (mask & 0x10)
	{
		//WriteConsole(hout,"IRQ on TM1\n",12,(LPDWORD)dummy,NULL);
		irqFlags |= 0x10;
		if ((IOREGS[TM1CNT]&0x4)==0)
		{
			timer[1].enabled = false;
			timer_setup(1);
		}
		cpu_irq(0x0000018);
	}
	if (mask & 0x20)
	{
		irqFlags |= 0x20;
		if ((IOREGS[TM2CNT]&0x4)==0)
		{
			timer[2].enabled = false;
			timer_setup(2);
		}
		cpu_irq(0x0000018);
	}
	if (mask & 0x40)
	{
		irqFlags |= 0x40;
		if ((IOREGS[TM3CNT]&0x4)==0)
		{
			timer[3].enabled = false;
			timer_setup(3);
		}
		cpu_irq(0x0000018);
	}
	
	if (mask & 0x800)
	{
		irqFlags |= 0x800;
		cpu_irq(0x0000018);
	}
}



unsigned char __stdcall mmu_read_byte(unsigned int address)
{
	unsigned int page = address>>24,offset; //=address&0xFFFFFF;

	if (pageTable[page])
		//if (pageLength[page]>offset)
			return *(unsigned char*)(pageTable[page] + (address&pageMask[page]));

	return rand()&0xFF;
}


unsigned short int __stdcall mmu_read_word(unsigned int address)
{
	unsigned int page = address>>24,offset=address&0xFFFFFF;
	unsigned int tlong;

	if (page == 4)
		switch (offset)
		{
			case P1:
				//wsprintf(buffer,"Reading keys (%06x)\n",keys&0x3FF);
				//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				return keys;

			case TM0D:
				if (timer[0].enabled)
				{
					tlong = *cycle;
					if (tlong < timer[0].tstart)
						tlong -= (timer[0].tstart-0x40000000);
					else
						tlong -= timer[0].tstart;
					timer[0].tcurr = tlong >> prescalar[IOREGS[TM0CNT]&3];
				}
				return timer[0].tcurr;

			case TM1D:
				if (timer[1].countup)
					return timer[1].te;
				if (timer[1].enabled)
				{
					tlong = *cycle;
					if (tlong < timer[1].tstart)
						tlong -= (timer[1].tstart-0x40000000);
					else
						tlong -= timer[1].tstart;
					timer[1].tcurr = tlong >> prescalar[IOREGS[TM1CNT]&3];
				}
				return timer[1].tcurr;

			case TM2D:
				if (timer[2].countup)
					return timer[2].te;
				if (timer[2].enabled)
				{
					tlong = *cycle;
					if (tlong < timer[2].tstart)
						tlong -= (timer[2].tstart-0x40000000);
					else
						tlong -= timer[2].tstart;
					timer[2].tcurr = tlong >> prescalar[IOREGS[TM2CNT]&3];
				}
				return timer[2].tcurr;

			case TM3D:
				if (timer[3].countup)
					return timer[3].te;
				if (timer[3].enabled)
				{
					tlong = *cycle;
					if (tlong < timer[3].tstart)
						tlong -= (timer[3].tstart-0x40000000);
					else
						tlong -= timer[3].tstart;
					timer[3].tcurr = tlong >> prescalar[IOREGS[TM3CNT]&3];
				}
				return timer[3].tcurr;

			case IF_:
				//wsprintf(buffer,"Reading IF (%04x)\n",irqFlags&0xFFFF);
				//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				return irqFlags;

			case 0x10:
			case 0x12:
			case 0x14:
			case 0x16:
			case 0x18:
			case 0x1A:
			case 0x1C:
			case 0x1E:
				page = cpu_regs[15];
				offset = page&0xFFFFFF;
				return *(unsigned short int*)(pageTable[page>>24] + offset);

			default:
				return *(unsigned short int*)(pageTable[page] + offset);
		}

	if (pageTable[page])
		return *(unsigned short int*)(pageTable[page] + offset);
	return rand()&0xFFFF;
}



int __stdcall mmu_read_dword(unsigned int address)
{
	unsigned int page = (address>>24)&0x0F,offset=address&0xFFFFFF;

	// DEBUG
	//addressLatch = address;

	if (page==8)
	{
		while (offset >= pageLength[page])
			offset -= pageLength[page];
		return *(int*)(pageTable[page] + offset);
	} else if (page == 4)
	{
		switch (offset)
		{
			case P1:
				return keys | IOREGS[P1CNT]<<16 | IOREGS[P1CNT+1]<<24;
			default:
				break;
		}
	}

	if (pageTable[page])
		//if (pageLength[page]>offset)
			return *(int*)(pageTable[page] + (address&pageMask[page]));

	return rand()&0xFFFFFFFF;
}



void __stdcall mmu_write_byte(unsigned int address,int byte)
{
	unsigned int page = (address>>24)&0x0F,offset=address&0xFFFFFF;

	/*if (page==6)
		if (((IOREGS[4] & 3) | (IOREGS[0] & 0x80)) == 0)
			return;
	if (page==7)
		if (((IOREGS[4] & 3) | (IOREGS[0] & 0x80)) == 0)
			return;*/

	if (pageTable[page])
		if (pageLength[page]>offset)
			*(char*)(pageTable[page] + offset) = (char)byte;

	if ((page==4)&&(offset>=NR10)&&(offset<=NR52))
		apu_write(offset,byte&0xFF);

	/*if (page == 6)
		if (offset&1)
			*(char*)(pageTable[page] + offset-1) = (char)byte;
		else
			*(char*)(pageTable[page] + offset+1) = (char)byte;*/
}



void __stdcall mmu_write_word(unsigned int address,int word)
{
	unsigned int page = address>>24,offset=address&0xFFFFFF;

	/*if (page==6)
		if (((IOREGS[4] & 3) | (IOREGS[0] & 0x80)) == 0)
			return;
	if (page==7)
		if (((IOREGS[4] & 3) | (IOREGS[0] & 0x80)) == 0)
			return;*/

	if (pageTable[page])
		if (pageLength[page]>offset)
			*(unsigned short int*)(pageTable[page] + offset) = (unsigned short int)word;


	if (page == 4)
	{
		if ((offset>=NR10)&&(offset<=NR52))
		{
			apu_write(offset,word&0xFF);
			apu_write(offset+1,(word>>8)&0xFF);
		}

		switch (offset)
		{
			/*case 0:
				wsprintf(buffer,"Video buffer %d\n",(word>>4)&1);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;*/

			case 0xC6:
				dma_setup(1,false);
				break;
			case 0xD2:
				dma_setup(2,false);
				break;
			case 0xDE:
				dma_setup(3,false);
				break;
			case TM0D:
				//timer[0].tcurr = word&0xFFFF;
				break;
			case TM1D:
				//timer[1].tcurr = word&0xFFFF;
				break;
			case TM2D:
				//timer[2].tcurr = word&0xFFFF;
				break;
			case TM3D:
				//timer[3].tcurr = word&0xFFFF;
				break;
			case TM0CNT:
				timer_setup(0);
				break;
			case TM1CNT:
				timer_setup(1);
				break;
			case TM2CNT:
				timer_setup(2);
				break;
			case TM3CNT:
				timer_setup(3);
				break;
			case IF_:
				irqFlags ^= (word&0xFFFF);
				break;
			default:
				break;
		}
	}
}



void __stdcall mmu_write_dword(unsigned int address,int dwrd)
{
	unsigned int page = address>>24,offset=address&0xFFFFFF;

	/*if (page==6)
		if (((IOREGS[4] & 3) | (IOREGS[0] & 0x80)) == 0)
			return;
	if (page==7)
		if (((IOREGS[4] & 3) | (IOREGS[0] & 0x80)) == 0)
			return;*/

	if (pageTable[page])
		if (pageLength[page]>offset)
			*(unsigned int*)(pageTable[page] + offset) = (unsigned int)dwrd;

	if (page == 4)
	{
		switch (address & 0xFFFFFF)
		{
			/*case 0:
				wsprintf(buffer,"Video buffer %d\n",(dwrd>>4)&1);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;*/

			case TM0D:
				timer_setup(0);
				break;
			case TM1D:
				timer_setup(1);
				break;
			case TM2D:
				timer_setup(2);
				break;
			case TM3D:
				timer_setup(3);
				break;

			case 0xC4:
				dma_setup(1,false);
				break;
			case 0xC6:
				dma_setup(1,false);
				break;
			case 0xD0:
				dma_setup(2,false);
				break;
			case 0xD2:
				dma_setup(2,false);
				break;
			/*case 0xD4:
				wsprintf(buffer,"DM3SAD = %08x\n",dwrd);
				WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);
				break;*/
			case 0xDC:
				dma_setup(3,false);
				break;
			case 0xDE:
				dma_setup(3,false);
				break;
			default:
				break;
		}
	}
}



int __stdcall mmu_get(int what)
{
	switch (what)
	{
		case MMU_READ_BYTE_CALLBACK:
			return (int)mmu_read_byte;
		case MMU_READ_WORD_CALLBACK:
			return (int)mmu_read_word;
		case MMU_READ_DWORD_CALLBACK:
			return (int)mmu_read_dword;
		case MMU_WRITE_BYTE_CALLBACK:
			return (int)mmu_write_byte;
		case MMU_WRITE_WORD_CALLBACK:
			return (int)mmu_write_word;
		case MMU_WRITE_DWORD_CALLBACK:
			return (int)mmu_write_dword;
		case MMU_TRANSLATE_ADDRESS_CALLBACK:
			return (int)mmu_translate_address;
		case MMU_IRQ_CALLBACK:
			return (int)mmu_irq;
		case PLUGIN_NAME:
			return (int)&mmuName;
		case 0x700:
			return (int)mmu_translate_address((unsigned int)0x8000288);
		default:
			break;
	}
	return 0;
}


void __stdcall mmu_set(int what,void* param1,void* param2)
{
	static char buffer[64];
	int i,n;

	switch (what)
	{
		case CPU_CYCLE_PTR:
			cycle = (unsigned int *)param1;
			break;

		case MMU_TIMER_OVERFLOW:
			n = ((int)param1)>>16;

			//wsprintf(buffer,"Overflow on timer %d\n",n);
			//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);

			if (IOREGS[TM0CNT+n*4]&0x40)
				mmu_irq(0x08<<n);
			else
				timer_setup(n);

			if (n<3)
				if (IOREGS[TM0CNT+((n+1)*4)]&0x04)
				{
					//wsprintf(buffer,"Doing count-up on timer %d (%d)\n",n+1,timer[n+1].tcurr+1);
					//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);

					timer[n+1].te++;
					if (timer[n+1].te==0)
					{
					//wsprintf(buffer,"Timer %d overflowed, reloading with %04x\n",n+1,timer[n+1].td);
					//WriteConsole(hout,buffer,lstrlen(buffer),(unsigned long*)&dummy,NULL);

						timer[n+1].te = timer[n+1].td;
						if (IOREGS[TM0CNT+((n+1)*4)]&0x40)
							mmu_irq(0x08<<(n+1));			
					}
				}
			break;


		case ZOMBIE_KEYSTATE:
			if ((int)param2==1)
			{
				for (i=0; i<10; i++)
					if ((int)param1 == keycodes[i])
					{
						keys &= (keymask[i] ^ 0x3FF);
						break;
					}

				/*switch ((int)param1)
				{
					case 38:	// Up
						keys &= 0x3BF;
						break;
					case 40:
						keys &= 0x37F;
						break;
					case 37:	// Left
						keys &= 0x3DF;
						break;
					case 39:
						keys &= 0x3EF;
						break;
					case 13:
						keys &= 0x3F7;
						break;
					case 'X':
						keys &= 0x3FE;
						break;
					case 'Z':
						keys &= 0x3FD;
						break;
					case 'L':
						keys &= 0x1FF;
						break;
					case 'R':
						keys &= 0x2FF;
						break;
					case ' ':
						keys &= 0x3FB;
						break;
				}*/

			} else
			{
				for (i=0; i<10; i++)
					if ((int)param1 == keycodes[i])
					{
						keys |= (keymask[i]);
						break;
					}

				/*switch ((int)param1)
				{
					case 38:	// Up
						keys |= 0x40;
						break;
					case 40:
						keys |= 0x80;
						break;
					case 37:	// Left
						keys |= 0x20;
						break;
					case 39:
						keys |= 0x10;
						break;
					case 13:
						keys |= 0x08;
						break;
					case 'X':
						keys |= 0x01;
						break;
					case 'Z':
						keys |= 0x02;
						break;
					case 'L':
						keys |= 0x200;
						break;
					case 'R':
						keys |= 0x100;
						break;
					case ' ':
						keys |= 0x04;
						break;
				}*/
			}
			if (!IOREGS) return;

			if (IOREGS[IME]&1)
			if (IOREGS[IE+1]&0x10)
				if (IOREGS[P1CNT+1]&0x40)
					if (IOREGS[P1CNT+1]&0x80)
					{
					// Logical multiplication (AND)
						if (((keys^0xFFFF)&0x3FF) == (IOREGS[P1CNT]&0x3FF)) 
						{
							//IOREGS[0x201] |= 0x20;
							irqFlags |= 0x1000;
							cpu_irq(0x00000018);
						}
					}
					//else if ((keys&IOREGS[P1CNT])!=IOREGS[P1CNT])
					// Logical addition (OR)
					else if ((keys^0xFFFF)&(IOREGS[P1CNT]&0x3FF))
					{
						irqFlags |= 0x1000;
						cpu_irq(0x00000018);
					}
			break;

		case PLUGIN_SETTING:
			switch ((int)param1)
			{
				case 0:
				case 1:
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
				case 8:
				case 9:
					keycodes[(int)param1] = (int)param2;
					break;
				default:
					break;
			}
			break;

		default:
			break;
	}
}


