/************************************************************************

		Copyright (c) 2002 Brad Martin.

This file is part of OpenSPC.

OpenSPC is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

OpenSPC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with OpenSPC; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA



main.c: implements functions intended for external use of the libopenspc
library.

 ************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_ZLIB
#include <zlib.h>
#endif

#include "dsp.h"
#include "spc.h"
#include "openspc.h"

#undef NO_CLEAR_ECHO

extern int TS_CYC;

#ifdef HAVE_ZLIB
class gzreader
{
private:
	z_streamp zsp;
public:
	gzreader ()
	{
		zsp = 0;
	}

	~gzreader ()
	{
		if (zsp)
			delete zsp;
	}

	int read (void *buf, size_t size)
	{
		zsp->next_out = (unsigned char *) buf;
		zsp->avail_out = size;
		return inflate (zsp, Z_SYNC_FLUSH);
	}

	int open (unsigned char *buf, size_t size)
	{
		struct gz_header
		{
			unsigned char id1,
			              id2,
			              cm,
			              flg,
			              junk[6];
		} *gzh=(struct gz_header *)buf;
		size_t skip = sizeof (struct gz_header);
		/* First, verify the GZ header */
		if ((gzh->id1 != 0x1F) || (gzh->id2 != 0x8B) || (gzh->cm != 0x08) ||
			(gzh->flg & 0xE0))
			return 0;
		if (gzh->flg & 0x04)
			skip += (int) buf[skip] + (int) (buf[skip + 1] << 8) + 2;
		if (gzh->flg & 0x08)
		{
			while ((skip < size) && (buf[skip] != '\0'))
			{
				skip++;
			}
		}
		if (gzh->flg & 0x10)
		{
			while ((skip < size) && (buf[skip] != '\0'))
			{
				skip++;
			}
		}
		if (gzh->flg & 0x02)
			skip += 2;
		if (skip >= size)
			return 0;
		zsp = new z_stream;
		zsp->next_in = buf + skip;
		zsp->avail_in = size - skip;
		zsp->zalloc = Z_NULL;
		zsp->zfree = Z_NULL;
		zsp->opaque = Z_NULL;
		if (inflateInit2 (zsp, -MAX_WBITS) != Z_OK)
		{
			fprintf (stderr, "ZLib init error: '%s'\n", zsp->msg);
			return 0;
		}
		return 1;
	}
};
#endif

/**** Internal (static) functions ****/

int
openspc::Load_SPC (void *buf, size_t size)
{
	const char ident[] = "SNES-SPC700 Sound File Data";
	struct SPC_FILE
	{
		unsigned char ident[37],
		              PCl,PCh,
		              A,
		              X,
		              Y,
		              P,
		              SP,
		              junk[212],
		              RAM[65536],
		              DSP[128];
	}  *spc_file;
	if (size < sizeof (spc_file))
		return 1;
	spc_file = (struct SPC_FILE *) buf;

	if (memcmp (buf, ident, strlen (ident)))
		return 1;
	memcpy (SPC_RAM, spc_file->RAM, 65536);
	pspc->setstate (((int)spc_file->PCh<<8)+spc_file->PCl, spc_file->A,
	 spc_file->X, spc_file->Y, spc_file->P, 0x100 + spc_file->SP);
	memcpy (DSPregs, spc_file->DSP, 128);
	return 0;
}

#ifndef NO_ZST
int
openspc::Load_ZST (void *buf, size_t size)
{
	int p;
	const char ident[] = "ZSNES Save State File";
	struct ZST_FILE
	{
		unsigned char ident[26],
		         junk[199673],
		         RAM[65536],
		         junk2[16],
		         PCl,PCh,PCj[2],
		         A,Aj[3],
		         X,Xj[3],
		         Y,Yj[3],
		         P,Pj[3],
		         P2[4],
		         SP,SPj[3],
		         junk3[420],
		         v_on[8],
		         junk4[916],
		         DSP[256];
	}  *zst_file;

	if (size < sizeof (struct ZST_FILE))
		return 1;
	zst_file = (struct ZST_FILE *) buf;

	if (memcmp (buf, ident, strlen (ident)))
		return 1;
	p = zst_file->P;
	if ((zst_file->P2[0]|zst_file->P2[1]|zst_file->P2[2]|zst_file->P2[3])
	 == 0)
		p |= 2;
	else
		p &= ~2;
	if (zst_file->P2 & 0x80)
		p |= 0x80;
	else
		p &= ~0x80;
	memcpy (SPC_RAM, zst_file->RAM, 65536);
	pspc->setstate (((int)zst_file->PCh<<8)+zst_file->PCl, zst_file->A,
	 zst_file->X, zst_file->Y, p, 0x100+zst_file->SP);
	memcpy (DSPregs, zst_file->DSP, 256);
	/* Little hack to turn on voices that were already on when state
	   was saved.  Doesn't restore the entire state of the voice, just
	   starts it over from the beginning. */
	for (p = 0; p < 8; p++)
		if (zst_file->v_on[p])
			DSPregs[0x4C] |= (1 << p);
	return 0;
}
#endif

#ifdef HAVE_ZLIB
int
openspc::Load_S9X (void *buf, size_t size)
{
	struct S9X_APU_BLOCK
	{
		unsigned char junk1[11],
		              DSP[0x80],
		              junk2[82];
	};
	struct S9X_APUREGS_BLOCK
	{
		unsigned char P,
		              A,
		              Y,
		              X,
		              S,
		              PCh,
		              PCl;
	} SnapAPURegisters;
	const char ident[] = "#!snes9";
	const int bufsize = 65536;
	mem_block_t < char >obuf_block, RAM_block;
	char *obuf = obuf_block.set_size (bufsize), *RAM =
		obuf_block.set_size (65536);
	gzreader gz;
	int i, blen, foundRAM = 0, foundRegs = 0;

	if (!gz.open ((unsigned char *) buf, size))
		return 1;
	i = gz.read (obuf, 14);
	if (memcmp (ident, obuf, strlen (ident)))
	{
		return 1;
	}
	while (gz.read (obuf, 11) != Z_STREAM_END)
	{
		for (i = 0; (i < 11) && (obuf[i] != ':'); i++);
		blen = strtol (&obuf[i + 1], NULL, 10);
		if (!memcmp (obuf, "APU", 3))
		{
			if (blen >= bufsize)
			{
				gz.read (obuf, bufsize);
				blen -= bufsize;
			}
			else
			{
				gz.read (obuf, blen);
				blen = 0;
			}
			memcpy (DSPregs, ((struct S9X_APU_BLOCK *) obuf)->DSP, 0x80);
		}
		else if (!memcmp (obuf, "ARE", 3))
		{
			if (blen >= (int) sizeof (struct S9X_APUREGS_BLOCK))
			{
				gz.read (&SnapAPURegisters,
						 sizeof (struct S9X_APUREGS_BLOCK));
				blen -= sizeof (struct S9X_APUREGS_BLOCK);
			}
			else
			{
				gz.read (&SnapAPURegisters, blen);
				blen = 0;
			}
			foundRegs = 1;
		}
		else if (!memcmp (obuf, "ARA", 3))
		{
			if (blen >= 65536)
			{
				gz.read (RAM, 65536);
				blen -= 65536;
			}
			else
			{
				gz.read (RAM, blen);
				blen = 0;
			}
			foundRAM = 1;
		}

		while (blen > bufsize)
		{
			gz.read (obuf, bufsize);
			blen -= bufsize;
		}
		gz.read (obuf, blen);
	}

	if (!foundRAM || !foundRegs)
	{
		return 1;
	}
	/* Now that we have all the info, load the state */
	memcpy (SPC_RAM, RAM, 65536);
	pspc->setstate (((int) SnapAPURegisters.PCh << 8) +
					SnapAPURegisters.PCl, SnapAPURegisters.A,
					SnapAPURegisters.X, SnapAPURegisters.Y,
					SnapAPURegisters.P, (int) SnapAPURegisters.S + 0x100);
	return 0;
}
#endif

/**** Exported library interfaces ****/

int
openspc::init (void *buf, size_t size)
{
	int ret;
#ifndef NO_CLEAR_ECHO
	int start, len;
#endif
	mix_left = 0;
	pspc->reset ();
	pdsp->reset ();
	ret = Load_SPC (buf, size);
#ifndef NO_ZST
	if (ret == 1)				/* Return 1 means wrong format */
		ret = Load_ZST (buf, size);
#endif
#ifdef HAVE_ZLIB
	if (ret == 1)
		ret = Load_S9X (buf, size);
#endif

/* New file formats could go on from here, for example:
	if(ret==1)
		ret=Load_FOO(buf,size);
	...
*/
#ifndef NO_CLEAR_ECHO
	/* Because the emulator that generated the SPC file most likely did
	   not correctly support echo, it is probably necessary to zero out
	   the echo region of memory to prevent pops and clicks as playback
	   begins. */
	if (!(DSPregs[0x6C] & 0x20))
	{
		start = (unsigned char) DSPregs[0x6D] << 8;
		len = (unsigned char) DSPregs[0x7D] << 11;
		if (start + len > 0x10000)
			len = 0x10000 - start;
		memset (&SPC_RAM[start], 0, len);
	}
#endif
	return ret;
}

int
openspc::run (int cyc, short *s_buf, int s_size)
{
	int i;
	if ((cyc < 0) && (s_size) && (s_buf == NULL))
	{
		s_size &= ~3;
		if (mix_left)
			pspc->run (mix_left);
		pspc->run (TS_CYC * (s_size >> 2));
		mix_left = 0;
		return s_size;
	}
	if ((cyc < 0)
		|| ((s_buf != NULL) && (cyc >= (s_size >> 2) * TS_CYC + mix_left)))
	{
		s_size &= ~3;
		if (mix_left)
			pspc->run (mix_left);
		for (i = 0; i < s_size; i += 4, s_buf += 2)
		{
			pdsp->update (s_buf);
			pspc->run (TS_CYC);
		}
		mix_left = 0;
		return s_size;
	}
	if (cyc < mix_left)
	{
		pspc->run (cyc);
		mix_left -= cyc;
		return 0;
	}
	if (mix_left)
	{
		pspc->run (mix_left);
		cyc -= mix_left;
	}
	for (i = 0; cyc >= TS_CYC; i += 4, cyc -= TS_CYC, s_buf += 2)
	{
		pdsp->update (s_buf);
		pspc->run (TS_CYC);
	}
	if (cyc)
	{
		pdsp->update (s_buf);
		pspc->run (cyc);
		mix_left = TS_CYC - cyc;
		i += 4;
	}
	return i;
}

void
openspc::setmute (int m)
{
	pdsp->setmask (~m & 0xFF);
}

void
openspc::writeport (int port, char data)
{
	pspc->writeport (port, data);
}

char
openspc::readport (int port)
{
	return pspc->readport (port);
}

openspc::openspc ()
{
	memset(&SPC_RAM, 0, 65536);
	memset(&DSPregs, 0, 256);
	pspc = new spc (&SPC_RAM, &DSPregs);
	pdsp = new dsp (&SPC_RAM, &DSPregs);
}

openspc::~openspc ()
{
	delete pdsp;
	delete pspc;
}
