/*  Copyright 2004 Lawrence Sebald
    Copyright 2004 Theo Berkau

    This file is part of Yabause.

    Yabause is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Yabause is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Yabause; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*  Some of the code contained in this file is based upon the
    cdrom.c file from KallistiOS. The code from this file was written by
    Dan Potter, and is available unchanged from http://gamedev.allusion.net/
    under a BSD-like license.
    Also, some code in this file was based on information provided by
    Thierry Vermeylen on the KallistiOS mailing list.
*/

#include <kos/thread.h>
#include <dc/cdrom.h>
#include <iostream>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

using namespace std;

extern "C" int cd_reinit();

static string ets(int e)	{
	switch(e)	{
		case ERR_OK:
			return "No Error";
		case ERR_NO_DISC:
			return "No Disc";
		case ERR_DISC_CHG:
			return "Disc Changed";
		case ERR_SYS:
			return "Unknown System Error";
	}
}

int CDInit(char *cdrom_name)	{
	cdrom_init();
	int r = cd_reinit();
	if(r == ERR_SYS)	{
#ifdef CDDEBUG
		cerr << "cdblock: Couldn't initialize CDROM drive!" << endl;
#endif
		return -1;
	}
	return 0;
}

int CDDeInit()	{
	/* Play nice with other programs, reinit to known KOS state. */
	cdrom_reinit();
	return 0;
}


bool CDIsCDPresent()	{
	int status, disc_type;
	int rv = cdrom_get_status(&status, &disc_type);
	if(rv < 0)	{
#ifdef CDDEBUG
		cerr << "cdblock: Could not get CDROM status!" << endl;
#endif
		return false;
	}
	return (status != CD_STATUS_NO_DISC);
}

long CDReadToc(unsigned long *TOC)	{
	CDROM_TOC tocbuf;
	int i;
	int rv = cdrom_read_toc(&tocbuf, 0);
	if(rv == ERR_DISC_CHG)	{
#ifdef CDDEBUG
		cerr << "cdblock: Disc changed, reiniting." << endl;
#endif
		cdrom_reinit();
		cd_reinit();
		rv = cdrom_read_toc(&tocbuf, 0);
	}
	if(rv)	{
#ifdef CDDEBUG
		cerr << "cdblock: Could not read TOC." << endl;
#endif
		return 0;
	}

	memset(TOC, 0xFF, 0xCC * 2);

	int lasttrack = TOC_TRACK(tocbuf.last);

	// convert TOC to saturn format
	for (i = 0; i < lasttrack; i++)	{      
		TOC[i] = tocbuf.entry[i];
	}

	// Do First, Last, and Lead out sections here
	TOC[99] = tocbuf.first;
	TOC[100] = tocbuf.last;
	/* Hack to get around wacky leadout CTRL Values.
	   Possibly a Dreamcast CD System/BIOS bug? */
	TOC[101] = (tocbuf.leadout_sector & 0x0FFFFFFF) | TOC_CTRL(tocbuf.last);

	return (0xCC * 2);
}

int CDGetStatus(void)	{
	// 0 - CD Present, disc spinning
	// 1 - CD Present, disc not spinning
	// 2 - CD not present
	// 3 - Tray open
	// see ../windows/cd.cc for more info
	int status, disc_type;
	int rv = cdrom_get_status(&status, &disc_type);
	if(rv < 0)	{
#ifdef CDDEBUG
		cerr << "cdblock: Could not get CDROM status!" << endl;
#endif
		return 2;
	}

	switch(status)	{
		case CD_STATUS_NO_DISC:
			return 2;
		case CD_STATUS_OPEN:
			return 3;
		case CD_STATUS_STANDBY:
			return 1; // maybe?
		default:
			return 0;
	}
}

bool CDReadSectorFAD(unsigned long FAD, void *buffer)	{
	int rv = cdrom_read_sectors(buffer, FAD, 1);
	if(rv == ERR_DISC_CHG)	{
#ifdef CDDEBUG
		cerr << "cdblock: Disc changed, reiniting." << endl;
#endif
		cdrom_reinit();
		cd_reinit();
		rv = cdrom_read_sectors(buffer, FAD, 1);
	}
	if(rv)	{
#ifdef CDDEBUG
		cerr << "cdblock: Could not read Sector " << FAD << endl;
		cerr << "         Error: " << ets(rv) << endl;
#endif
		return false;
	}
	return true;
}
