/*
 * ============================================================================
 *  Title:    Graphics Interface Routines
 *  Author:   J. Zbiciak
 *  $Id: gfx.c,v 1.18 2001/02/03 02:34:21 im14u2c Exp $
 * ============================================================================
 *  GFX_INIT         -- Initializes a gfx_t object.
 *  GFX_TICK         -- Services a gfx_t tick.
 *  GFX_VID_ENABLE   -- Alert gfx that video has been enabled or blanked
 *  GFX_SET_BORD     -- Set the border / offset parameters for the display
 * ============================================================================
 *  GFX_T            -- Graphics subsystem object.
 *  GFX_PVT_T        -- Private internal state to gfx_t structure.
 *  GFX_STIC_PALETTE -- The STIC palette.
 * ============================================================================
 *  The graphics subsystem provides an abstraction layer between the 
 *  emulator and the graphics library being used.  Theoretically, this 
 *  should allow easy porting to other graphics libraries.
 *
 *  TODO:  
 *   -- Make use of dirty rectangle updating for speed.
 * ============================================================================
 */
static const char rcs_id[]="$Id: gfx.c,v 1.18 2001/02/03 02:34:21 im14u2c Exp $";

#include <SDL/SDL.h>
#include "config.h"
#include "periph/periph.h"
#include "gfx.h"
#include "file/file.h"

/*
 * ============================================================================
 *  GFX_PVT_T        -- Private internal state to gfx_t structure.
 * ============================================================================
 */
typedef struct gfx_pvt_t
{
    SDL_Surface *scr;               /*  Screen surface.                 */
    SDL_Color   pal_on [32];        /*  Palette when video is enabled.  */
    SDL_Color   pal_off[32];        /*  Palette when video is blanked.  */
    int         vid_enable;         /*  Video enable flag.              */
} gfx_pvt_t;

/*
 * ============================================================================
 *  GFX_STIC_PALETTE -- The STIC palette.
 * ============================================================================
 */
static uint_8 gfx_stic_palette[32][3] = 
{
    /* -------------------------------------------------------------------- */
    /*  I generated these colors by directly eyeballing my television       */
    /*  while it was next to my computer monitor.  I then tweaked each      */
    /*  color until it was pretty close to my TV.  Bear in mind that        */
    /*  NTSC (said to mean "Never The Same Color") is highly susceptible    */
    /*  to Tint/Brightness/Contrast settings, so your mileage may vary      */
    /*  with this particular pallete setting.                               */
    /* -------------------------------------------------------------------- */
    { 0x00, 0x00, 0x00 },
    { 0x00, 0x2D, 0xFF },
    { 0xFF, 0x3D, 0x10 },
    { 0xC9, 0xCF, 0xAB },
    { 0x38, 0x6B, 0x3F },
    { 0x00, 0xA7, 0x56 },
    { 0xFA, 0xEA, 0x50 },
    { 0xFF, 0xFC, 0xFF },
    { 0xBD, 0xAC, 0xC8 },
    { 0x24, 0xB8, 0xFF },
    { 0xFF, 0xB4, 0x1F },
    { 0x54, 0x6E, 0x00 },
    { 0xFF, 0x4E, 0x57 },
    { 0xA4, 0x96, 0xFF },
    { 0x75, 0xCC, 0x80 },
    { 0xB5, 0x1A, 0x58 },

    /* -------------------------------------------------------------------- */
    /*  This pink color is used for drawing rectangles around sprites.      */
    /*  It's a temporary hack.                                              */
    /* -------------------------------------------------------------------- */
    { 0xFF, 0x80, 0x80 },
    /* -------------------------------------------------------------------- */
    /*  Grey shades used for misc tasks (not currently used).               */
    /* -------------------------------------------------------------------- */
    { 0x11, 0x11, 0x11 },
    { 0x22, 0x22, 0x22 },
    { 0x33, 0x33, 0x33 },
    { 0x44, 0x44, 0x44 },
    { 0x55, 0x55, 0x55 },
    { 0x66, 0x66, 0x66 },
    { 0x77, 0x77, 0x77 },
    { 0x88, 0x88, 0x88 },
    { 0x99, 0x99, 0x99 },
    { 0xAA, 0xAA, 0xAA },
    { 0xBB, 0xBB, 0xBB },
    { 0xCC, 0xCC, 0xCC },
    { 0xDD, 0xDD, 0xDD },
    { 0xEE, 0xEE, 0xEE },
    { 0xFF, 0xFF, 0xFF },
};


/*
 * ============================================================================
 *  GFX_SDL_ABORT    -- Abort due to SDL errors.
 * ============================================================================
 */
static void gfx_sdl_abort(void)
{
    fprintf(stderr, "gfx/SDL Error:%s\n", SDL_GetError());
    exit(1);
}

/*
 * ============================================================================
 *  GFX_INIT         -- Initializes a gfx_t object.
 * ============================================================================
 */
int gfx_init(gfx_t *gfx, int desire_x, int desire_y, int desire_bpp, int fs)
{
    int i;
    periph_tick_t gfx_tick = NULL;

    /* -------------------------------------------------------------------- */
    /*  Sanity checks and cleanups.                                         */
    /* -------------------------------------------------------------------- */
    assert(gfx);
    memset((void*)gfx, 0, sizeof(gfx_t));
    

    /* -------------------------------------------------------------------- */
    /*  Select the appropriate tick function based on our display res.      */
    /*  For now, only support 320x200x8bpp or 640x480x8bpp.                 */
    /* -------------------------------------------------------------------- */
    if (desire_x == 320 && desire_y == 200 && desire_bpp == 8)
        gfx_tick = gfx_tick_320x200;

    if (desire_x == 640 && desire_y == 480 && desire_bpp == 8)
        gfx_tick = gfx_tick_640x480;

    if (gfx_tick == NULL)
    {
        fprintf(stderr, 
                "gfx panic:  Right now, only 320x200x8bpp and "
                "640x480x8bpp modes \n            are supported.\n" );
        return -1;
    }


    /* -------------------------------------------------------------------- */
    /*  Allocate memory for the gfx_t.                                      */
    /* -------------------------------------------------------------------- */
    gfx->vid = calloc(160, 200);
    gfx->pvt = calloc(1, sizeof(gfx_pvt_t));

    if (!gfx->vid && !gfx->pvt)
    {
        fprintf(stderr, "gfx panic:  Could not allocate memory.\n");
        return -1;
    }

#if 0
    /* -------------------------------------------------------------------- */
    /*  Initialize the SDL library.  Note that we need to ask for audio     */
    /*  initialization here as well in order to work properly under         */
    /*  DirectX (if we ever port this to Winblows).                         */
    /* -------------------------------------------------------------------- */
    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO) < 0)
        gfx_sdl_abort();

    /* -------------------------------------------------------------------- */
    /*  Register SDL_Quit to run when we shut down, in order to clean       */
    /*  everything up.                                                      */
    /* -------------------------------------------------------------------- */
    atexit(SDL_Quit);
#endif

    /* -------------------------------------------------------------------- */
    /*  Try to allocate a screen surface at the desired size, etc.          */
    /* -------------------------------------------------------------------- */
    /*  NOTE:  This eventually should do better things about finding        */
    /*  resolutions / color depths that we like, etc.  For now just be      */
    /*  braindead, even if it means SDL will run our video in "emulation."  */
    /* -------------------------------------------------------------------- */
    printf("gfx: Setting video mode: %dx%dx%d (%s)\n", 
            desire_x, desire_y, desire_bpp, fs ? "Full Screen" : "Windowed"); 
    fflush(stdout);
#if 1
    gfx->pvt->scr = SDL_SetVideoMode(desire_x, desire_y, desire_bpp, 
                        SDL_DOUBLEBUF|SDL_HWPALETTE|SDL_HWSURFACE|(fs?SDL_FULLSCREEN:0));
#else
    gfx->pvt->scr = SDL_SetVideoMode(desire_x, desire_y, desire_bpp, 
                        SDL_DOUBLEBUF|SDL_SWSURFACE|(fs?SDL_FULLSCREEN:0));
#endif

    if (!gfx->pvt->scr)
        gfx_sdl_abort();

    /* -------------------------------------------------------------------- */
    /*  TEMPORARY: Verify that the surface's format is as we expect.  This  */
    /*  is just a temporary bit of paranoia to ensure that scr->pixels      */
    /*  is in the format I _think_ it's in.                                 */
    /* -------------------------------------------------------------------- */
    if (gfx->pvt->scr->format->BitsPerPixel  != 8 ||
        gfx->pvt->scr->format->BytesPerPixel != 1)
    {
        fprintf(stderr,"gfx panic: BitsPerPixel = %d, BytesPerPixel = %d\n",
                gfx->pvt->scr->format->BitsPerPixel,
                gfx->pvt->scr->format->BytesPerPixel);
        return -1;
    }

    /* -------------------------------------------------------------------- */
    /*  Set up our color palette.  We start with video blanked.             */
    /* -------------------------------------------------------------------- */
    for (i = 0; i < 16; i++)
    {
        gfx->pvt->pal_on [i].r = gfx_stic_palette[i][0];
        gfx->pvt->pal_on [i].g = gfx_stic_palette[i][1];
        gfx->pvt->pal_on [i].b = gfx_stic_palette[i][2];
        gfx->pvt->pal_off[i].r = gfx_stic_palette[i][0] >> 1;
        gfx->pvt->pal_off[i].g = gfx_stic_palette[i][1] >> 1;
        gfx->pvt->pal_off[i].b = gfx_stic_palette[i][2] >> 1;
    }
    for (i = 16; i < 32; i++)
    {
        gfx->pvt->pal_on [i].r = gfx_stic_palette[i][0];
        gfx->pvt->pal_on [i].g = gfx_stic_palette[i][1];
        gfx->pvt->pal_on [i].b = gfx_stic_palette[i][2];
        gfx->pvt->pal_off[i].r = gfx_stic_palette[i][0];
        gfx->pvt->pal_off[i].g = gfx_stic_palette[i][1];
        gfx->pvt->pal_off[i].b = gfx_stic_palette[i][2];
    }
    gfx->pvt->vid_enable = 0;
    SDL_SetColors(gfx->pvt->scr, gfx->pvt->pal_off, 0, 32);

    /* -------------------------------------------------------------------- */
    /*  Hide the mouse.                                                     */
    /* -------------------------------------------------------------------- */
    SDL_ShowCursor(0);

    /* -------------------------------------------------------------------- */
    /*  Set up the gfx_t's internal structures.                             */
    /* -------------------------------------------------------------------- */
    gfx->periph.read        = NULL;
    gfx->periph.write       = NULL;
    gfx->periph.peek        = NULL;
    gfx->periph.poke        = NULL;
    gfx->periph.tick        = gfx_tick;
    gfx->periph.min_tick    = 14934;
    gfx->periph.max_tick    = 14934;
    gfx->periph.addr_base   = 0;
    gfx->periph.addr_mask   = 0;


    return 0;
}

/*
 * ============================================================================
 *  GFX_SET_TITLE    -- Sets the window title 
 * ============================================================================
 */
int gfx_set_title(gfx_t *gfx, const char *title)
{
    (void)gfx;
    SDL_WM_SetCaption(title, title);
    return 0;
}

/*
 * ============================================================================
 *  GFX_TICK_640X480 -- Services a gfx_t tick (in 640x480 mode)
 * ============================================================================
 */
uint_32 gfx_tick_640x480(periph_p gfx_periph, uint_32 len)
{
    gfx_t   *gfx = (gfx_t*) gfx_periph;
    uint_32 *scr0, *scr1, pix;
    uint_8  *vid;
    int j;
    int i;
    int r_step;
#ifdef BENCHMARK_GFX
    double start, end;

    start = get_time();
#endif

    gfx->tot_frames++;

    /* -------------------------------------------------------------------- */
    /*  Drop a frame if we need to.                                         */
    /* -------------------------------------------------------------------- */
    if (gfx->drop_frame)
    {
        gfx->drop_frame--;
        if (gfx->dirty) gfx->dropped_frames++;
        return len;
    }

    /* -------------------------------------------------------------------- */
    /*  Don't bother if display isn't dirty or if we're iconified.          */
    /* -------------------------------------------------------------------- */
    if (!gfx->scrshot && (!gfx->dirty || gfx->hidden))
    {
        return len;
    }

    /* -------------------------------------------------------------------- */
    /*  DEBUG: Report blocks of dropped frames.                             */
    /* -------------------------------------------------------------------- */
    if (gfx->dropped_frames)
    {
#if 0
        printf("Dropped %d frames.\n", gfx->dropped_frames);
        fflush(stdout);
#endif
        gfx->tot_dropped_frames += gfx->dropped_frames;
        gfx->dropped_frames = 0;
    }

    /* -------------------------------------------------------------------- */
    /*  Draw the frame to the screen surface.                               */
    /* -------------------------------------------------------------------- */

    if (SDL_MUSTLOCK(gfx->pvt->scr))
        SDL_LockSurface(gfx->pvt->scr);

    scr0 = (uint_32 *) gfx->pvt->scr->pixels;
    vid  = (uint_8  *) gfx->vid;
    if (gfx->b_dirty > 0)
    {
        static SDL_Rect top = { 0, 0, 640, 40 }, bot = { 0, 440, 640, 40 } ;

        gfx->b_dirty--;
        SDL_FillRect(gfx->pvt->scr, &top, gfx->b_color);
        SDL_FillRect(gfx->pvt->scr, &bot, gfx->b_color);
    }

    r_step = (2*gfx->pvt->scr->pitch - 640) / sizeof(uint_32);
    if (gfx->pvt->scr->pitch % sizeof(uint_32))
    {
        printf("error: can't blit to pitch %% 4 != 0\n");
        exit(1);
    }
    scr0 +=  40 * (gfx->pvt->scr->pitch) / sizeof(uint_32);
    scr1 = scr0 + (gfx->pvt->scr->pitch) / sizeof(uint_32);

    for (j = 0; j < 200; j++)
    {
        for (i = 0; i < 160; i++)
        {
            pix = *vid++ * 0x01010101;
            *scr0++ = *scr1++ = pix;
        }
        scr0 += r_step;
        scr1 += r_step;
    }

    if (SDL_MUSTLOCK(gfx->pvt->scr))
        SDL_UnlockSurface(gfx->pvt->scr);

    /* -------------------------------------------------------------------- */
    /*  Actually update the display.                                        */
    /* -------------------------------------------------------------------- */
    //SDL_UpdateRect(gfx->pvt->scr, 0, 0, 0, 0);
    SDL_Flip(gfx->pvt->scr);
    gfx->dirty = 0;

    /* -------------------------------------------------------------------- */
    /*  If a screen-shot was requested, go write out a PPM file of the      */
    /*  screen right now.  Screen-shot PPMs are always 320x200.             */
    /* -------------------------------------------------------------------- */
    if (gfx->scrshot)
    {
        if ((gfx->scrshot & 1) || 
            ((gfx->tot_frames % (gfx->scrshot >> 1)) == 0))
            gfx_scrshot(gfx->vid);
        gfx->scrshot &= ~1;
    }
#ifdef BENCHMARK_GFX
    end = get_time();
    printf("gfx_tick = %8.3fms\n", (end-start)*1000.);
#endif

    return len;
}

/*
 * ============================================================================
 *  GFX_TICK_320X200 -- Services a gfx_t tick (in 320x200 mode)
 * ============================================================================
 */
uint_32 gfx_tick_320x200(periph_p gfx_periph, uint_32 len)
{
    gfx_t   *gfx = (gfx_t*) gfx_periph;
    uint_8  *vid, *scr, pix;
    int x, y;
#ifdef BENCHMARK_GFX
    double start, end;

    start = get_time();
#endif

    gfx->tot_frames++;

    /* -------------------------------------------------------------------- */
    /*  Drop a frame if we need to.                                         */
    /* -------------------------------------------------------------------- */
    if (gfx->drop_frame)
    {
        gfx->drop_frame--;
        if (gfx->dirty) gfx->dropped_frames++;
        return len;
    }

    /* -------------------------------------------------------------------- */
    /*  Don't bother if display isn't dirty or if we're iconified.          */
    /* -------------------------------------------------------------------- */
    if (!gfx->dirty || gfx->hidden)
    {
        return len;
    }

    /* -------------------------------------------------------------------- */
    /*  DEBUG: Report blocks of dropped frames.                             */
    /* -------------------------------------------------------------------- */
    if (gfx->dropped_frames)
    {
#if 0
        printf("Dropped %d frames.\n", gfx->dropped_frames);
        fflush(stdout);
#endif
        gfx->tot_dropped_frames += gfx->dropped_frames;
        gfx->dropped_frames = 0;
    }

    /* -------------------------------------------------------------------- */
    /*  Draw the frame to the screen surface.                               */
    /* -------------------------------------------------------------------- */

    if (SDL_MUSTLOCK(gfx->pvt->scr))
        SDL_LockSurface(gfx->pvt->scr);

    scr = (uint_8 *) gfx->pvt->scr->pixels;
    vid = (uint_8 *) gfx->vid;
    for (y = 0; y < 200; y++)
    {
        for (x = 0; x < 160; x++)
        {
            pix    = *vid++;
            scr[0] = scr[1] = pix;
            scr   += 2;
        }
        scr += gfx->pvt->scr->pitch - 320;
    }
    if (SDL_MUSTLOCK(gfx->pvt->scr))
        SDL_UnlockSurface(gfx->pvt->scr);

    /* -------------------------------------------------------------------- */
    /*  Actually update the display.                                        */
    /* -------------------------------------------------------------------- */
    //SDL_UpdateRect(gfx->pvt->scr, 0, 0, 0, 0);
    SDL_Flip(gfx->pvt->scr);
#if 0
    {
        static double now, then = 0;
        now = get_time();
        if (then > 0.0)
            printf("flip after %8.0fus\n", (now - then) * 1e6);
        then  = now;
    }
#endif
    gfx->dirty = 0;

    /* -------------------------------------------------------------------- */
    /*  If a screen-shot was requested, go write out a PPM file of the      */
    /*  screen right now.  Screen-shot PPMs are always 320x200.             */
    /* -------------------------------------------------------------------- */
    if (gfx->scrshot)
    {
        if ((gfx->scrshot & 1) || 
            ((gfx->tot_frames % (gfx->scrshot >> 1)) == 0))
            gfx_scrshot(gfx->vid);
        gfx->scrshot &= ~1;
    }

#ifdef BENCHMARK_GFX
    end = get_time();
    printf("gfx_tick = %8.3fms\n", (end-start)*1000.);
#endif

    return len;
}

/*
 * ============================================================================
 *  GFX_VID_ENABLE   -- Alert gfx that video has been enabled or blanked
 * ============================================================================
 */
void gfx_vid_enable(gfx_t *gfx, int enabled)
{
    /* -------------------------------------------------------------------- */
    /*  Force 'enabled' to be 0 or 1.                                       */
    /* -------------------------------------------------------------------- */
    enabled = enabled != 0;

    /* -------------------------------------------------------------------- */
    /*  Update the palette if there's been a change in blanking state.      */
    /* -------------------------------------------------------------------- */
    if (gfx->pvt->vid_enable ^ enabled)
    {
        SDL_SetColors(gfx->pvt->scr, 
                      enabled ? gfx->pvt->pal_on : gfx->pvt->pal_off, 0, 16);
    }

    gfx->pvt->vid_enable = enabled;
}

/*
 * ============================================================================
 *  GFX_SET_BORD     -- Set the border color for the display
 * ============================================================================
 */
void gfx_set_bord
(
    gfx_t *gfx,         /*  Graphics object.                        */
    int b_color
)
{
    int dirty = 0;

    /* -------------------------------------------------------------------- */
    /*  Set up the display parameters.                                      */
    /* -------------------------------------------------------------------- */
    if (gfx->b_color != b_color) { gfx->b_color = b_color; dirty = 3; }

    /* -------------------------------------------------------------------- */
    /*  If we're using the normal STIC blanking behavior, set our "off"     */
    /*  colors to the currently selected border color.  The alternate mode  */
    /*  (which is useful for debugging) sets the blanked colors to be       */
    /*  dimmed versions of the normal palette.                              */
    /* -------------------------------------------------------------------- */
    if (gfx->debug_blank == 0)
    {
        int i;

        for (i = 0; i < 16; i++)
            gfx->pvt->pal_off[i] = gfx->pvt->pal_on[b_color];
    }

    if (dirty)     { gfx->dirty   = 1; }
    if (dirty & 2) { gfx->b_dirty = 2; }
}

/*
 * ============================================================================
 *  GFX_SCRSHOT      -- Write a 320x200 screen shot to a PPM file.
 * ============================================================================
 */
void gfx_scrshot(uint_8 *scr)
{
    static int last = -1;
    FILE * f;
    char f_name[32];
    int num = last, i, j;
    uint_8 buf[320 * 3], *line, *disp, pix, r, g, b;


    /* -------------------------------------------------------------------- */
    /*  Search for an unused screen-shot file name.                         */
    /* -------------------------------------------------------------------- */
    do 
    {
        num = (num + 1) % 1000;

        sprintf(f_name, "shot_%.3d.ppm", num);

        if (!file_exists(f_name)) 
            break;

    } while (num != last);

    /* -------------------------------------------------------------------- */
    /*  Warn the user if we wrapped all 1000 screen shots...                */
    /* -------------------------------------------------------------------- */
    if (num == last)
    {
        num = (num + 1) % 1000;
        sprintf(f_name, "shot_%.3d.ppm", num);
        fprintf(stderr, "Warning:  Overwriting %s...\n", f_name);
    }

    /* -------------------------------------------------------------------- */
    /*  Update our 'last' pointer and open the file and dump the PPM.       */
    /* -------------------------------------------------------------------- */
    last = num;
    f    = fopen(f_name, "wb");

    if (!f)
    {
        fprintf(stderr, "Error:  Could not open '%s' for screen dump.\n",
                f_name);
        return; 
    }


    /* -------------------------------------------------------------------- */
    /*  Do the screen dump.  All screen dumps are always 320x200x8bpp       */
    /*  binary PPM files.  We 'image' the display to a line-buffer and      */
    /*  fwrite it line-by-line.  This is reasonably efficient.              */
    /* -------------------------------------------------------------------- */
    fprintf(f, "P6\n# jzIntv screen shot\n320 200 255\n");

    disp = scr;
    for (i = 0; i < 200; i++)
    {
        line = buf;

        for (j = 0; j < 160; j++)
        {
            pix = *disp++;

            r = gfx_stic_palette[pix][0];
            g = gfx_stic_palette[pix][1];
            b = gfx_stic_palette[pix][2];

            line[0] = line[3] = r;
            line[1] = line[4] = g;
            line[2] = line[5] = b;

            line += 6;
        }

        j = fwrite(buf, 3, 320, f);

        if (j != 320)
        {
            perror("gfx_scrshot:fwrite()");
            fprintf(stderr, "Error:  Screen-dump failed.  Disk full?");
            break;
            return;
        }
    }

    if (i == 200)
    {
        printf("\nWrote screen shot to '%s'\n", f_name);
        fflush(stdout);
    }

    fclose(f);

    return;
}

/* ======================================================================== */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 2 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       */
/*  General Public License for more details.                                */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/* ======================================================================== */
/*                 Copyright (c) 1998-2000, Joseph Zbiciak                  */
/* ======================================================================== */
