/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"

#include <crtdbg.h>

#include <ddraw.h>
#include <d3dx.h>
#include "resource.h"

#include "DBGConsole.h"

#ifndef SAFE_DELETE
#define SAFE_DELETE(p)  { if(p) { delete (p);     (p)=NULL; } }
#endif

#include "SurfaceHandler.h"
#include "GfxContext.h"
#include "PngUtil.h"

//////////////////////////////////////////
// Constructors / Deconstructors

// Probably shouldn't need more than 4096 * 4096
static BYTE g_ucTempBuffer[1024*1024];

SurfaceHandler::SurfaceHandler(DWORD dwWidth, DWORD dwHeight) :
	m_lpDDS(NULL),
	m_SurfFmt(D3DX_SF_UNKNOWN),
	m_dwWidth(0),
	m_dwHeight(0),
	m_dwCorrectedWidth(0),
	m_dwCorrectedHeight(0)

{
	Resize(dwWidth, dwHeight);
}


SurfaceHandler::~SurfaceHandler(void)
{

	if (m_lpDDS)
	{
		m_lpDDS->Release();
		m_lpDDS = NULL;
	}
	m_dwWidth = 0;
	m_dwHeight = 0;

}


static void Init()
{
	HRESULT hr;
	LPDIRECT3DDEVICE7 pD3DDev;
	DWORD dwFlags;
	DWORD dwWidth;
	DWORD dwHeight;
	D3DX_SURFACEFORMAT pf;

	g_GfxContext.Lock();
	pD3DDev = g_GfxContext.GetD3DDev();

	// Don't worry about Null pointer
	dwFlags = 0;
	dwWidth = 64;
	dwHeight = 64;
	pf = D3DX_SF_A8R8G8B8;
	hr = D3DXCheckTextureRequirements(pD3DDev, &dwFlags, &dwWidth, &dwHeight, &pf);
	if (FAILED(hr))
	{	
			
		
	}
	g_GfxContext.Unlock();

}

//////////////////////////////////////////
// Resize the DIB. Just passes control to
// CreateDIB()
BOOL
SurfaceHandler::Resize(DWORD dwWidth, DWORD dwHeight)
{
	LPDIRECTDRAWSURFACE7 lpSurf;

	if (dwWidth < 1)
		dwWidth = 1;
	else if (dwWidth > 1024)
		dwWidth = 1024;

	if (dwHeight < 1)
		dwHeight = 1;
	else if (dwHeight > 1024)
		dwHeight = 1024;

	// Check if surface is already of required dimensions etc
	if (m_lpDDS && dwWidth == m_dwWidth && dwHeight == m_dwHeight)
		return TRUE;

	if (dwWidth < m_dwWidth)
		DBGConsole_Msg(0, "New width (%d) < Old Width (%d)", dwWidth, m_dwWidth);
	if (dwHeight < m_dwHeight)
		DBGConsole_Msg(0, "New height (%d) < Old height (%d)", dwHeight, m_dwHeight);
	
	lpSurf = CreateSurface(dwWidth, dwHeight);
	if (lpSurf == NULL)
		return FALSE;

	// Copy from old surface to new surface
	if (m_lpDDS != NULL)
	{
		m_lpDDS->Release();
		m_lpDDS = NULL;
	}

	m_dwWidth = dwWidth;
	m_dwHeight = dwHeight;
	m_lpDDS = lpSurf;

	//GetSurfaceInfo();

	return TRUE;

}
/* Eventually we may need to reimplment this if
   we don't let D3DX do our texture handling.
// Returns the next highest power of 2 (unless dwX is a power of 2)
// e.g. F(3) = 4
// e.g. F(8) = 8
DWORD TH_GetHigherPowerOf2(DWORD dwX)
{
	DWORD temp = dwX;
	DWORD dwNewX;

	for (DWORD w = 0; w < 12; w++)
	{
		dwNewX = 1<<w;
		if (dwNewX >= dwX)
			break;
	}
	return dwNewX;

}*/


/////////////////////////////////////////////
// Create the surface. Destroys old surface if
// necessary
LPDIRECTDRAWSURFACE7
SurfaceHandler::CreateSurface(DWORD dwWidth, DWORD dwHeight)
{
	HRESULT hr;
	LPDIRECTDRAWSURFACE7 lpSurf;
	LPDIRECT3DDEVICE7 pD3DDev;


	DWORD dwFlags = D3DX_TEXTURE_NOMIPMAP;		// D3DX_TEXTURE_STAGE0
	D3DX_SURFACEFORMAT pf = D3DX_SF_A8R8G8B8;
	DWORD dwNumMaps = 0;

	m_dwCorrectedWidth = dwWidth;
	m_dwCorrectedHeight = dwHeight;

	// Uncomment this to simulate Voodoo 256x256 limitation
	/*if (m_dwCorrectedWidth > 256)
		m_dwCorrectedWidth = 256;
	if (m_dwCorrectedHeight > 256)
		m_dwCorrectedHeight = 256;*/

	g_GfxContext.Lock();
		pD3DDev = g_GfxContext.GetD3DDev();
		hr = D3DXCreateTexture(pD3DDev, &dwFlags, &m_dwCorrectedWidth, &m_dwCorrectedHeight, &pf, NULL, &lpSurf, &dwNumMaps);
	g_GfxContext.Unlock();

	if (pf != D3DX_SF_A8R8G8B8 &&
		pf != D3DX_SF_A4R4G4B4)
	{
		static BOOL bWarned = FALSE;
		if (!bWarned)
		{
			TCHAR szMsg[300+1];

			// IDS_UNSUPPORTEDTEXFMT takes one %d arg
			wsprintf(szMsg, CResourceString(IDS_UNSUPPORTEDTEXFMT), pf);
			MessageBox(g_hMainWindow, szMsg, g_szDaedalusName, MB_ICONEXCLAMATION|MB_OK);
			bWarned = TRUE;
		}
	}

	// D3D likes texture w/h to be a power of two, so the condition below
	// will almost always hold. 
	// D3D should usually create textures large enough (on nVidia cards anyway),
	// and so there will usually be some "slack" left in the texture (blank space
	// that isn't used). The D3DRender code takes care of this (mostly)
	// Voodoo cards are limited to 256x256 and so often textures will be
	// created that are too small for the required dimensions. We compensate for
	// this by passing in a dummy area of memory when the surface is locked,
	// and copying across pixels to the real surface when the surface is unlocked.
	// In this case there will be no slack and D3DRender takes this into account.
	// 
	/*if (dwWidth != m_dwCorrectedWidth ||
		dwHeight != m_dwCorrectedHeight)
	{
		DBGConsole_Msg(0, "Couldn't create texture of size %d x %d (get %d x %d)",
			dwWidth, dwHeight, m_dwCorrectedWidth, m_dwCorrectedHeight);
	}*/

	// HACK - we should only assign this when m_lpDDS is assigned!
	m_SurfFmt = pf;
	

	if (FAILED(hr))
	{
		char szError[200+1];

		D3DXGetErrorString(hr, 200, szError);
		
		DBGConsole_Msg(0, "!!Unable to create surface!! %d x %d (%s)", dwWidth, dwHeight, szError);
		
		return NULL;
	}
	
	return lpSurf;		
}

HRESULT 
SurfaceHandler::GetSurfaceInfo()
{
	HRESULT hr;
	DDPIXELFORMAT pf;

	if (m_lpDDS == NULL)
		return E_FAIL;

	ZeroMemory(&pf, sizeof(DDPIXELFORMAT));
	pf.dwSize = sizeof(DDPIXELFORMAT);
	pf.dwFlags = (DDPF_ALPHAPIXELS|DDPF_RGB);
	hr = m_lpDDS->GetPixelFormat(&pf);
	if (FAILED(hr))
		return hr;

	//DBGConsole_Msg(0, "Pixel Format: bpp: %d", pf.dwRGBBitCount);
	//DBGConsole_Msg(0, "         Red: 0x%08x", pf.dwRBitMask);
	//DBGConsole_Msg(0, "       Green: 0x%08x", pf.dwGBitMask);
	//DBGConsole_Msg(0, "        Blue: 0x%08x", pf.dwBBitMask);
	//DBGConsole_Msg(0, "       Alpha: 0x%08x", pf.dwRGBAlphaBitMask);


	return S_OK;
}


HRESULT
SurfaceHandler::GetSurfaceFormat(D3DX_SURFACEFORMAT * pSurfFmt)
{
	if (m_lpDDS == NULL)
		return E_FAIL;
	
	*pSurfFmt = m_SurfFmt;

	return S_OK;
}

DWORD SurfaceHandler::GetPixelSize()
{
	switch (m_SurfFmt)
	{
	case D3DX_SF_A8R8G8B8: return 4;
	case D3DX_SF_A4R4G4B4: return 2;
	default: return 1;		// Safe?
	}

}

//////////////////////////////////////////////////
// Get information about the DIBitmap
// This locks the bitmap (and stops 
// it from being resized). Must be matched by a
// call to EndUpdate();
BOOL
SurfaceHandler::StartUpdate(DrawInfo *di)
{
	DDSURFACEDESC2 ddsd;
	
	if (m_lpDDS == NULL)
		return FALSE;

    ZeroMemory( &ddsd, sizeof(DDSURFACEDESC2) );
    ddsd.dwSize = sizeof(DDSURFACEDESC2);

	if (m_dwCorrectedWidth < m_dwWidth ||
		m_dwCorrectedHeight < m_dwHeight)
	{
		// Return a temporary buffer to use
		di->lpSurface = g_ucTempBuffer;
		di->dwWidth = m_dwWidth;
		di->dwHeight = m_dwHeight;
		di->lPitch = m_dwWidth * GetPixelSize();
	}
	else
	{
		if (SUCCEEDED(m_lpDDS->Lock(NULL, &ddsd, DDLOCK_NOSYSLOCK|DDLOCK_WAIT, NULL)))
		{
			di->dwWidth = ddsd.dwWidth;
			di->dwHeight = ddsd.dwHeight;
			di->lpSurface = ddsd.lpSurface;
			di->lPitch    = ddsd.lPitch;
		}
		else
		{
			return FALSE;
		}
	}

	return TRUE;
}

///////////////////////////////////////////////////
// This releases the DIB information, allowing it
// to be resized again
void
SurfaceHandler::EndUpdate(DrawInfo *di)
{
	
	if (m_lpDDS == NULL)
		return;

	if (m_dwCorrectedWidth < m_dwWidth ||
		m_dwCorrectedHeight < m_dwHeight)
	{
		ScaleTempBufferToSurface();
	}
	else
	{
		m_lpDDS->Unlock(NULL);
	}

}


void SurfaceHandler::ScaleTempBufferToSurface()
{
	DWORD xDst, yDst;
	DWORD xSrc, ySrc;
	DDSURFACEDESC2 ddsd;
	
	if (m_lpDDS == NULL)
		return;

    ZeroMemory( &ddsd, sizeof(DDSURFACEDESC2) );
    ddsd.dwSize = sizeof(DDSURFACEDESC2);

	if (FAILED(m_lpDDS->Lock(NULL, &ddsd, DDLOCK_NOSYSLOCK|DDLOCK_WAIT, NULL)))
		return;

	// Copy across from the temp buffer to the surface
	switch (GetPixelSize())
	{
	case 4:
		{
			DWORD * pDst;
			DWORD * pSrc;

			for (yDst = 0; yDst < m_dwCorrectedHeight; yDst++)
			{
				// ySrc ranges from 0..m_dwHeight
				// I'd rather do this but sometimes very narrow (i.e. 1 pixel)
				// surfaces are created which results in  /0...
				//ySrc = (yDst * (m_dwHeight-1)) / (m_dwCorrectedHeight-1);
				ySrc = (yDst * (m_dwHeight)) / (m_dwCorrectedHeight);

				pSrc = (DWORD*)((BYTE*)g_ucTempBuffer + (ySrc * m_dwWidth * 4));
				pDst = (DWORD*)((BYTE*)ddsd.lpSurface + (yDst * ddsd.lPitch));

				for (xDst = 0; xDst < m_dwCorrectedWidth; xDst++)
				{
					xSrc = (xDst * (m_dwWidth)) / (m_dwCorrectedWidth);
					pDst[xDst] = pSrc[xSrc];
				}
			}
		}

		break;
	case 2:
		{
			WORD * pDst;
			WORD * pSrc;

			for (yDst = 0; yDst < m_dwCorrectedHeight; yDst++)
			{
				// ySrc ranges from 0..m_dwHeight
				ySrc = (yDst * (m_dwHeight)) / (m_dwCorrectedHeight);

				pSrc = (WORD*)((BYTE*)g_ucTempBuffer + (ySrc * m_dwWidth * 2));
				pDst = (WORD*)((BYTE*)ddsd.lpSurface + (yDst * ddsd.lPitch));

				for (xDst = 0; xDst < m_dwCorrectedWidth; xDst++)
				{
					xSrc = (xDst * (m_dwWidth)) / (m_dwCorrectedWidth);
					pDst[xDst] = pSrc[xSrc];
				}
			}
		}
		break;

	}

	m_lpDDS->Unlock(NULL);
}


//////////////////////////////////////////////////////
// Set all pixels to (0,0,0) (also clears alpha)
void
SurfaceHandler::Clear(void)
{
	DDBLTFX ddbltfx;
	
	if (m_lpDDS == NULL)
		return;

	ZeroMemory(&ddbltfx, sizeof(ddbltfx));
	ddbltfx.dwSize = sizeof(ddbltfx);
	ddbltfx.dwFillColor = 0;

	m_lpDDS->Blt(NULL, NULL, NULL, DDBLT_COLORFILL|DDBLT_WAIT, &ddbltfx);

}


// Make the pixels of the texture blue
HRESULT 
SurfaceHandler::MakeBlue(DWORD dwWidth, DWORD dwHeight)
{
	DrawInfo di;
	
	if (!StartUpdate(&di))
		return E_FAIL;
	
	DWORD * pSrc = (DWORD *)di.lpSurface;
	LONG lPitch = di.lPitch;

	for (DWORD y = 0; y < dwHeight; y++)
	{
		DWORD * dwSrc = (DWORD *)((BYTE *)pSrc + y*lPitch);
		for (DWORD x = 0; x < dwWidth; x++)
		{
			dwSrc[x] = 0x000000FF | ((rand()% 255)<<8) | ((rand()%255)<<24);
		}
	}
	
	EndUpdate(&di);

	return S_OK;

}


BOOL SurfaceHandler::DumpImageAsRAW(LPCTSTR szFileName, DWORD dwWidth, DWORD dwHeight)
{
	DrawInfo di;
	FILE * fp = fopen(szFileName, "wb");
	if (fp == NULL)
		return FALSE;
	
	if (!StartUpdate(&di))
	{
		fclose(fp);
		return FALSE;
	}
	
	DWORD *pSrc = (DWORD *)di.lpSurface;
	LONG lPitch = di.lPitch;

	for (DWORD y = 0; y < dwHeight; y++)
	{
		DWORD * dwSrc = (DWORD *)((BYTE *)pSrc + y*lPitch);
		for (DWORD x = 0; x < dwWidth; x++)
		{
			DWORD dw = dwSrc[x];

			BYTE dwRed   = (BYTE)((dw & 0x00FF0000)>>16);
			BYTE dwGreen = (BYTE)((dw & 0x0000FF00)>>8 );
			BYTE dwBlue  = (BYTE)((dw & 0x000000FF)    );
			BYTE dwAlpha = (BYTE)((dw & 0xFF000000)>>24);

			fwrite(&dwRed, 1,1,fp);
			fwrite(&dwGreen, 1,1,fp);
			fwrite(&dwBlue, 1,1,fp);
			fwrite(&dwAlpha, 1,1,fp);
		}
	}
	EndUpdate(&di);

	fclose(fp);

	return TRUE;
}

#ifdef LIBPNG_SUPPORT

BOOL SurfaceHandler::DumpImageAsPNG(LPCTSTR szFileName, DWORD dwWidth, DWORD dwHeight)
{
	DWORD * pB = new DWORD [dwWidth * dwHeight];
	if (pB == NULL)
	{
		return FALSE;
	}

	DrawInfo di;
	
	if (!StartUpdate(&di))
	{
		delete [] pB;
		return FALSE;
	}
	
	DWORD *pSrc = (DWORD *)di.lpSurface;
	LONG lPitch = di.lPitch;

	DWORD index = 0;

	for (DWORD y = 0; y < dwHeight; y++)
	{
		DWORD * dwSrc = (DWORD *)((BYTE *)pSrc + y*lPitch);
		for (DWORD x = 0; x < dwWidth; x++)
		{
			pB[index++] = dwSrc[x];
		}
	}
	EndUpdate(&di);
	
	PngSaveImage(szFileName, (png_byte *)pB, dwWidth, dwHeight);

	delete [] pB;

	return TRUE;
}

#endif // LIBPNG_SUPPORT
