/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"

#include <commctrl.h>
#include <shlwapi.h>
#include <shlobj.h>

#include "AudioDialog.h"
#include "DInputHandler.h"
#include "Debug.h"
#include "GraphicsDialog.h"
#include "InputSelectDialog.h"
#include "Interrupt.h"
#include "MainWindow.h"
#include "CPU.h"
#include "SR.h"
#include "GfxContext.h"
#include "RDP.h"				// g_dwNumFrames
#include "RDP_GFX.h"			// RDP_GFX_DumpNextDisplayList
#include "RomSettingsDialog.h"
#include "ultra_rcp.h"
#include "ultra_r4300.h"
#include "ultra_os.h"			// Controller buttons defs
#include "Controller.h"

#include "DBGConsole.h"

#include "ROM.h"
#include "FileNameHandler.h"

#include "resource.h"



//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
//                       Eternals
//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
extern DWORD g_dwTLBRHit;
extern DWORD g_dwTLBWHit;

//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
//                        Globals
//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
CGfxContext g_GfxContext;


//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
//                        Locals
//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
static HWND	g_hWndList	   = NULL;
HWND	g_hWndStatus   = NULL;
static BOOL	g_bActive	   = TRUE;

static void Main_SizeForDisplay();
static void Main_SaveWindowPos(HWND hWnd);

static void Main_PauseDrawing();
static void Main_RestartDrawing();

static BOOL Main_CreateList(HWND hWndParent);
static BOOL Main_InitList();
static BOOL Main_FillList();
static BOOL Main_AddDirectory(LPCTSTR szDirectory);
static BOOL Main_AddItem(LPCTSTR szFileName);

static BOOL Main_CreateStatus(HWND hWndParent);

static LRESULT CALLBACK MainWndProc(HWND, UINT, WPARAM, LPARAM);

static LRESULT Main_OnCreate(HWND hWnd, WPARAM wParam, LPARAM lParam);
static LRESULT Main_OnNotify(HWND hWnd, WPARAM wParam, LPARAM lParam);

static LRESULT Main_OnListView_DeleteItem(int idCtrl, LPNMHDR pnmh);
static LRESULT Main_OnListView_GetDispInfo(int idCtrl, LPNMHDR pnmh);
static LRESULT Main_OnListView_ColumnClick(int idCtrl, LPNMHDR pnmh);
static LRESULT Main_OnListView_DblClick(int idCtrl, LPNMHDR pnmh);
static LRESULT Main_OnListView_RClick(int idCtrl, LPNMHDR pnmh);
static LRESULT Main_OnListView_Return(int idCtrl, LPNMHDR pnmh);

static HRESULT Main_InitD3DX(HWND hWndParent);
static HRESULT Main_ReleaseD3DX();
static HRESULT Main_InitRenderer();

static void Main_AskForRomDirectory(HWND hWndParent);

static const char c_szRomFileStr[] = "ROM Files\0*.v64;*.z64;*.n64;*.rom;*.jap;*.pal;*.usa;*.zip\0All Files (*.*)\0";

static const TCHAR	sc_szMainWindowClassname[256] = "DaedalusWindowClass";


///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
BOOL InitMainWindow()
{ 
    WNDCLASS wc; 
 
    // Register the frame window class. 
    wc.style         = 0; 
    wc.hInstance     = g_hInstance; 
    wc.cbClsExtra    = 0; 
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW); 
    wc.lpfnWndProc   = (WNDPROC)MainWndProc; 
    wc.cbWndExtra    = 0; 
    wc.hIcon         = LoadIcon(g_hInstance, MAKEINTRESOURCE(IDI_DAEDALUS_ICON)); 
    wc.hbrBackground = (HBRUSH)NULL;// (COLOR_APPWORKSPACE + 1); //(HBRUSH)GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName  = MAKEINTRESOURCE(IDR_APP_MENU); 
    wc.lpszClassName = sc_szMainWindowClassname; 
 
    if (!RegisterClass (&wc) ) 
        return FALSE; 

    return TRUE; 
} 

HWND Main_Create(INT nWinMode) 
{
	HWND hWnd = CreateWindow(sc_szMainWindowClassname,
								g_szDaedalusName,
								WS_OVERLAPPEDWINDOW,
								g_rcMainWindow.left,
								g_rcMainWindow.top,
								g_rcMainWindow.right - g_rcMainWindow.left,
								g_rcMainWindow.bottom - g_rcMainWindow.top,
								HWND_DESKTOP,
								LoadMenu(g_hInstance, MAKEINTRESOURCE(IDR_APP_MENU)),
								g_hInstance,
								NULL,
								);

	// Get the window pos, if CW_USEDEFAULT has been used for left/top
	if (IsWindow(hWnd))
	{
		ShowWindow(hWnd, nWinMode);
		UpdateWindow(hWnd);

		Main_SaveWindowPos(hWnd);
		Main_FillList();

	}

	return hWnd;
}

void Main_SizeForDisplay()
{
	RECT rcScreen;
	RECT rcStatus;

	// This might not be the best flag to switch on - ]
	// maybe we could check g_bD3DXReady?
	if (g_bCPURunning)
	{
		SetRect(&rcScreen, 0,0, g_dwDisplayWidth, g_dwDisplayHeight);

		// Calculate window size to give desired window size...
		AdjustWindowRectEx(&rcScreen, WS_OVERLAPPEDWINDOW, TRUE, 0);

		// Add on enough space for the status bar
		GetClientRect(g_hWndStatus, &rcStatus);
		rcScreen.bottom += (rcStatus.bottom - rcStatus.top);

		SetWindowPos(g_hMainWindow, 0,
			rcScreen.left, rcScreen.top,
			rcScreen.right-rcScreen.left, rcScreen.bottom-rcScreen.top,
			SWP_NOMOVE|SWP_NOACTIVATE|SWP_NOZORDER);

	}
	else
	{
		rcScreen = g_rcMainWindow;

		SetWindowPos(g_hMainWindow, 0,
			rcScreen.left, rcScreen.top,
			rcScreen.right-rcScreen.left, rcScreen.bottom-rcScreen.top,
			SWP_NOACTIVATE|SWP_NOZORDER);
	}


}

void Main_SaveWindowPos(HWND hWnd)
{
	/*bResult = */GetWindowRect(hWnd, &g_rcMainWindow);
}

static BOOL __stdcall AboutDlgProc(HWND hWndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{

	switch (msg)
	{
	case WM_INITDIALOG:
		SendMessage(GetDlgItem(hWndDlg, IDC_DAEDALUS_LOGO), STM_SETIMAGE, IMAGE_BITMAP, (LPARAM)LoadBitmap(g_hInstance, MAKEINTRESOURCE(IDB_DAEDALUS)));
		return TRUE;
	case WM_COMMAND:
		switch (LOWORD(wParam))
		{
		case IDOK:
			EndDialog(hWndDlg, 0);
			break;
		}
		break;	
	}

	return 0;

}

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
LRESULT CALLBACK MainWndProc(HWND hWnd, UINT message,
							WPARAM wParam, LPARAM lParam)
{
	static DWORD dwLastTick = 0;
	static BOOL  bSetTitle = FALSE;
	static HWND hWndList = NULL;

	switch (message)
	{

		case WM_ENTERSIZEMOVE:
			Input_Unaquire();
	        break;
		case WM_ENTERMENULOOP:
			Input_Unaquire();
	        Main_PauseDrawing();
	        break;
	    case WM_EXITMENULOOP:
	        Main_RestartDrawing();
	        break;


		case WM_CREATE:
			return Main_OnCreate(hWnd, wParam, lParam);

		case WM_DESTROY:
			{
				Main_SaveWindowPos(hWnd);

				StopCPUThread();
				g_hWndList = NULL;
				g_hWndStatus = NULL;

				Input_Finalise();		// Do this after CPU thread has finished
				PostQuitMessage(0);
			}
			return 0;

		//case MW_PAINT:
		// g_pGfxPlugin->DrawScreen();


		case WM_TIMER:
			{
				static u64 qwLastCount = 0;
				static LARGE_INTEGER liLastCount = {0,0};
				s64 qwNum = g_qwCPR[0][C0_COUNT] - qwLastCount;
				static DWORD dwLastNumFrames = 0;
				static DWORD dwLastNumVIs = 0;
				static DWORD dwLastTLBRHit = 0;
				static DWORD dwLastTLBWHit = 0;
				static DWORD dwLastNumEx = 0;

				static DWORD dwNumSRCompiled = 0;
				static DWORD dwNumSROptimised = 0;
				static DWORD dwNumSRFailed = 0;

				Input_CheckKeyboard(hWnd);


				if (g_bCPURunning)
				{
					LARGE_INTEGER li;
					LARGE_INTEGER liDiff;
					LARGE_INTEGER liFreq;
					float fElapsedTime;
					float fMIPS;
					float fFPS;
					float fVIs;
					float fTLBRS;
					float fTLBWS;
					float fExS;
					float fSRCS;
					float fSROS;
					float fSRFS;

					if (QueryPerformanceCounter(&li) && QueryPerformanceFrequency(&liFreq))
					{
						if (liLastCount.QuadPart == 0)
							liLastCount = li;
						if (dwLastNumFrames == 0)
							dwLastNumFrames = g_dwNumFrames;
						if (dwLastNumVIs == 0)
							dwLastNumVIs = g_dwVIs;

						liDiff.QuadPart = li.QuadPart - liLastCount.QuadPart;
						liLastCount = li;

						fElapsedTime = (float)liDiff.QuadPart / (float)liFreq.QuadPart;
						fMIPS = ((float)qwNum / 1000000.0f)/fElapsedTime;
						fFPS = ((float)(g_dwNumFrames-dwLastNumFrames)/fElapsedTime);
						fVIs = ((float)(g_dwVIs - dwLastNumVIs)/fElapsedTime);
						fTLBRS = ((float)(g_dwTLBRHit - dwLastTLBRHit)/fElapsedTime);
						fTLBWS = ((float)(g_dwTLBWHit - dwLastTLBWHit)/fElapsedTime);
						fExS = ((float)(g_dwNumExceptions - dwLastNumEx)/fElapsedTime);

						fSRCS = ((float)(g_dwNumSRCompiled - dwNumSRCompiled)/fElapsedTime);
						fSROS = ((float)(g_dwNumSROptimised - dwNumSROptimised)/fElapsedTime);
						fSRFS = ((float)(g_dwNumSRFailed - dwNumSRFailed)/fElapsedTime);

						DBGConsole_Stats(0, "MIPS: %#.2f  FPS: %#.2f", fMIPS, fFPS);
						DBGConsole_Stats(1, "VIs/s: %#.2f",fVIs);
						DBGConsole_Stats(3, "Sync: %#.2f%%",	100 * DESIRED_VI_INTERVAL / g_fCurrentRate);
						DBGConsole_Stats(5, "TLB: R %#.2f W: %#.2f", fTLBRS, fTLBWS);
						DBGConsole_Stats(6, "Ex/S %#.2f", fExS);
						DBGConsole_Stats(7, "DR:  C%d O%d F%d", (DWORD)fSRCS, (DWORD)fSROS, (DWORD)fSRFS);

						Main_SetStatus(1, "VIs/s: %#.2f FPS: %#.2f",fVIs, fFPS);

					}

				}
				qwLastCount = g_qwCPR[0][C0_COUNT];
				dwLastNumFrames = g_dwNumFrames;
				dwLastNumVIs = g_dwVIs;
				dwLastTLBRHit = g_dwTLBRHit;
				dwLastTLBWHit = g_dwTLBWHit;
				dwLastNumEx = g_dwNumExceptions;
				dwNumSRCompiled = g_dwNumSRCompiled;
				dwNumSROptimised = g_dwNumSROptimised;
				dwNumSRFailed = g_dwNumSRFailed;

			}
			return 0;

		case WM_NOTIFY:
			return Main_OnNotify(hWnd, wParam, lParam);

		case WM_INITMENU:
			{
				// Only need this for Display menu
				HMENU hMenu;
				HMENU hConfigMenu;
				HMENU hDisplayMenu;

				hMenu = GetMenu(hWnd);
				hConfigMenu = GetSubMenu(hMenu, 3);
				hDisplayMenu = GetSubMenu(hConfigMenu, 4);
				switch (g_dwDisplayWidth)
				{
				case 320:
					CheckMenuItem(hDisplayMenu, IDM_320_240, MF_BYCOMMAND|MF_CHECKED);
					CheckMenuItem(hDisplayMenu, IDM_640_480, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_800_600, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_1024_768, MF_BYCOMMAND|MF_UNCHECKED);
					break;
				case 640:
					CheckMenuItem(hDisplayMenu, IDM_320_240, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_640_480, MF_BYCOMMAND|MF_CHECKED);
					CheckMenuItem(hDisplayMenu, IDM_800_600, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_1024_768, MF_BYCOMMAND|MF_UNCHECKED);
					break;
				case 800:
					CheckMenuItem(hDisplayMenu, IDM_320_240, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_640_480, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_800_600, MF_BYCOMMAND|MF_CHECKED);
					CheckMenuItem(hDisplayMenu, IDM_1024_768, MF_BYCOMMAND|MF_UNCHECKED);
					break;
				case 1024:
					CheckMenuItem(hDisplayMenu, IDM_320_240, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_640_480, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_800_600, MF_BYCOMMAND|MF_UNCHECKED);
					CheckMenuItem(hDisplayMenu, IDM_1024_768, MF_BYCOMMAND|MF_CHECKED);
					break;
				}

				if (g_bShowDebug)
					CheckMenuItem(hConfigMenu, IDM_DEBUG_CONSOLE, MF_BYCOMMAND | MF_CHECKED);
				else
					CheckMenuItem(hConfigMenu, IDM_DEBUG_CONSOLE, MF_BYCOMMAND | MF_UNCHECKED);

				if (g_bRunAutomatically)
					CheckMenuItem(hConfigMenu, IDM_RUNAUTO, MF_BYCOMMAND | MF_CHECKED);
				else
					CheckMenuItem(hConfigMenu, IDM_RUNAUTO, MF_BYCOMMAND | MF_UNCHECKED);



				if (g_bDisplayFullscreen)
					CheckMenuItem(hDisplayMenu, IDM_DISPLAY_FULLSCREEN, MF_BYCOMMAND | MF_CHECKED);
				else
					CheckMenuItem(hDisplayMenu, IDM_DISPLAY_FULLSCREEN, MF_BYCOMMAND | MF_UNCHECKED);


					
				
			}
			return 0;

		case WM_MOVE:
			if (g_pGfxPlugin != NULL)
			{
				g_pGfxPlugin->MoveScreen((short) LOWORD(lParam),(short) HIWORD(lParam));
			}

			return 0;

		case WM_SIZE:
			{

				WORD wWidth = LOWORD(lParam);
				WORD wHeight = HIWORD(lParam);
				WORD wTop = 0;
				LONG nStatusHeight;

				if (IsWindow(g_hWndStatus))
				{
					RECT rc;

					SendMessage(g_hWndStatus, WM_SIZE, wParam, lParam);

					GetWindowRect(g_hWndStatus, &rc);
					nStatusHeight = rc.bottom - rc.top;
				}
				else
				{
					nStatusHeight = 0;
				}

				wHeight -= nStatusHeight;

				//resize the ListView to fit our window
				if (IsWindow(g_hWndList))
				{
					MoveWindow(g_hWndList, 0, wTop, wWidth, wHeight, TRUE);
				}

				g_GfxContext.Lock();
				if ( g_GfxContext.Ready() &&
					 wWidth > 0 &&
					 wHeight > 0 &&
					 !g_bDisplayFullscreen)
				{
					HRESULT hr;

					if (wWidth != g_dwDisplayWidth ||
						wHeight != g_dwDisplayHeight)
					{
						ID3DXContext* pD3DX;

						pD3DX = g_GfxContext.GetD3DX();
						if (pD3DX != NULL)
						{
							hr = pD3DX->Resize(wWidth,wHeight);
							if (FAILED(hr))
							{
								TCHAR szError[512+1];
								TCHAR szMsg[512+1];

								D3DXGetErrorString(hr, 512, szError);

								// IDS_D3DERROR has one %s for the error message
								wsprintf(szMsg, CResourceString(IDS_D3DERROR), szError);

								MessageBox(NULL, szMsg, g_szDaedalusName, MB_ICONERROR|MB_OK);

								g_GfxContext.SetReady(FALSE);
								//PostQuitMessage(0);
							}
						}
					}
				
				}
				/*else
				{
					g_dwMainWindowWidth = wWidth;
					g_dwMainWindowHeight = wHeight;
				}*/
				g_GfxContext.Unlock();
				//return DefWindowProc( hWnd, WM_SIZE, wParam, lParam );

			}
			return 0;



		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
		
				// File menu commands
				case IDM_OPEN:
					{
						TCHAR szFileName[MAX_PATH+1];

						FileNameHandler * pfn;

						pfn = new FileNameHandler("Daedalus", c_szRomFileStr, 1, "v64");
						if (pfn != NULL)
						{
							BOOL bSelected;
							
							bSelected = pfn->GetOpenName(hWnd, szFileName);
							if (bSelected)
							{
								/*bSuccess = */ROM_LoadFile(szFileName);
								pfn->GetCurrentDirectory(g_szRomsDir);
							}
							SAFE_DELETE(pfn);
						}

					}
					return 0;
				case IDM_EXIT:
					DestroyWindow(hWnd);
					return 0;

				case IDM_CPUSTART:
					{
						CHAR szReason[300+1];

						if (!StartCPUThread(szReason, 300))
							MessageBox(hWnd, szReason, g_szDaedalusName, MB_OK);
					}
					return 0;

				case IDM_CPUSTOP:
					StopCPUThread();
					return 0;

				case IDM_REFRESH:
					// Refresh
					Main_FillList();
					return 0;
				case IDM_SELECTDIR:
					{
						Main_AskForRomDirectory(hWnd);

						// Refresh
						Main_FillList();
					}
					return 0;

				case IDM_GRAPHICSCONFIG:
					GraphicsDialog_DoModal(hWnd);
					break;
				case IDM_AUDIOCONFIG:
					AudioDialog_DoModal(hWnd);
					break;
				case IDM_INPUTCONFIG:
					InputSelectDialog_DoModal(hWnd);
					break;

				case IDM_320_240:
					g_dwDisplayWidth = 320;
					g_dwDisplayHeight = 240;
					Main_SizeForDisplay();
					break;
				case IDM_640_480:
					g_dwDisplayWidth = 640;
					g_dwDisplayHeight = 480;
					Main_SizeForDisplay();
					break;
				case IDM_800_600:
					g_dwDisplayWidth = 800;
					g_dwDisplayHeight = 600;
					Main_SizeForDisplay();
					break;
				case IDM_1024_768:
					g_dwDisplayWidth = 1024;
					g_dwDisplayHeight = 768;
					Main_SizeForDisplay();
					break;

				case IDM_DISPLAY_FULLSCREEN:
					// If running, need to recreate the context
					g_bDisplayFullscreen = !g_bDisplayFullscreen;
					break;

				case IDM_SELECTSAVEDIR:
					Main_SelectSaveDir(hWnd);
					break;

				case IDM_DEBUG_CONSOLE:
					DBGConsole_Enable(!g_bShowDebug);
					break;

				case IDM_RUNAUTO:

					g_bRunAutomatically = !g_bRunAutomatically;
					break;

				case IDM_ABOUT:
					DialogBox(g_hInstance, MAKEINTRESOURCE(IDD_ABOUT), hWnd, AboutDlgProc);
					break;
				case IDM_SCREENSHOT:
					RDP_GFX_DumpNextScreen();
					break;

			}
			break;


		case WM_KEYDOWN:
			{
				switch ((int)wParam)
				{
					// None of this will work, because we're using exclusive dinput:
					case VK_F1:
						RDP_GFX_DumpNextDisplayList();
						return 0;	

					case VK_F2:
						#ifdef DAEDALUS_LOG
						g_bLog = !g_bLog;
						if (g_bLog)
							DBGConsole_Msg(0, "Debugging Output On");
						else
							DBGConsole_Msg(0, "Debugging Output Off");
						#endif
						return 0;	
					case VK_F5:
						// Refresh
						Main_FillList();

						return 0;

					case VK_F12:
						RDP_GFX_DumpNextScreen();
						return 0;

					case VK_ESCAPE:
						StopCPUThread();
						return 0;					
				}
			}
			break;


		case MWM_STARTEMU:
			{

				TCHAR szName[256+1];
				// Save the current window size
				Main_SaveWindowPos(hWnd);

				g_GfxContext.SetReady(FALSE);

				// Set the title text to include the rom name
				wsprintf(szName, "Daedalus - %s", g_ROM.szGameName);
				SetWindowText(hWnd, szName);

				EnableWindow(g_hWndList, FALSE);
				ShowWindow(g_hWndList, SW_HIDE);

				Main_SizeForDisplay();


				if (g_pGfxPlugin == NULL)
				{
					HRESULT hr;
					hr = Main_InitD3DX(hWnd);
					if (FAILED(hr)) 
					{

						TCHAR szError[512+1];
						TCHAR szMsg[512+1];

						D3DXGetErrorString(hr, 512, szError);

						// IDS_D3DERROR has one %s for the error message
						wsprintf(szMsg, CResourceString(IDS_D3DERROR), szError);

						MessageBox(hWnd, szMsg, g_szDaedalusName, MB_ICONERROR|MB_OK);

						// Disable rendering output
						g_GfxContext.SetReady(FALSE);
					}
					g_GfxContext.Clear(D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER);
					/*hr = */MainWindow_UpdateScreen();
				}
				else
				{
					RECT rc;
					GetWindowRect(hWnd, &rc);
					g_pGfxPlugin->MoveScreen(rc.left,rc.top);
				}
			}
			return 0;
		case MWM_ENDEMU:
			// Release the keyboard!
			Input_Unaquire();

		    Main_ReleaseD3DX();

			ShowWindow(g_hWndList, SW_SHOW);
			EnableWindow(g_hWndList, TRUE);

			// Restore the title text
			SetWindowText(hWnd, "Daedalus");

			Main_SizeForDisplay();
			return 0;
			
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

void __cdecl Main_SetStatus(INT nPiece, LPCTSTR szFormat, ...)
{
    va_list va;
	TCHAR szBuffer[2048+1];

	// HACK: With D3DX I can't seem to avoid splattering the status bar
	// this should help avoid the horrible flickering
	if (g_pGfxPlugin == NULL)
	{
		return;
	}

	// Format the output
	va_start(va, szFormat);
	// Don't use wvsprintf as it doesn't handle floats!
	vsprintf(szBuffer, szFormat, va);
	va_end(va);

	SendMessage(g_hWndStatus, SB_SETTEXT, nPiece | 0, (LPARAM)szBuffer);
}


void Main_PauseDrawing()
{
	LPDIRECTDRAW7 pDD;

    g_bActive = FALSE;

	g_GfxContext.Lock();
		pDD = g_GfxContext.GetDD();
		if( pDD )
			pDD->FlipToGDISurface();
	g_GfxContext.Unlock();


    DrawMenuBar(g_hMainWindow);
    RedrawWindow(g_hMainWindow, NULL, NULL, RDW_FRAME);
    ShowCursor(TRUE);
}

void Main_RestartDrawing()
{
    g_bActive = TRUE;
    ShowCursor(FALSE);
}


LRESULT Main_OnCreate(HWND hWnd, WPARAM wParam, LPARAM lParam)
{
	HRESULT hr;

	if (Main_CreateList(hWnd))
	{
		if(Main_InitList())
		{
			
		}
	}

	Main_CreateStatus(hWnd);

	SetTimer(hWnd, 0, 2000, NULL);


	hr = Input_Initialise(hWnd);
	if (FAILED(hr))
	{
		MessageBox(hWnd, CResourceString(IDS_UNABLETOINITINPUT), g_szDaedalusName, MB_OK);
	}

	return 0;
}


LRESULT Main_OnNotify(HWND hWnd, WPARAM wParam, LPARAM lParam)
{
	INT id = (INT)wParam;
	LPNMHDR pnmh = (LPNMHDR)lParam;
	LRESULT lResult = 1;
	BOOL bHandled;

	bHandled = FALSE;

	if (pnmh->idFrom == IDC_ROMLISTVIEW)
	{
		bHandled = TRUE;
		switch (pnmh->code)
		{
		case LVN_DELETEITEM:	lResult = Main_OnListView_DeleteItem((int)wParam, pnmh); break;
		case LVN_GETDISPINFO:	lResult = Main_OnListView_GetDispInfo((int)wParam, pnmh); break;
		case LVN_COLUMNCLICK:	lResult = Main_OnListView_ColumnClick((int)wParam, pnmh); break;
		case NM_DBLCLK:			lResult = Main_OnListView_DblClick((int)wParam, pnmh); break;
		case NM_RCLICK:			lResult = Main_OnListView_RClick((int)wParam, pnmh); break;
		case NM_RETURN:			lResult = Main_OnListView_Return((int)wParam, pnmh); break;
		default:
			bHandled = FALSE;
			break;
		}
	}
	if (bHandled)
		return lResult;
	else
		return DefWindowProc(hWnd, WM_NOTIFY, wParam, lParam);
}



BOOL Main_CreateStatus(HWND hWndParent)
{
	DWORD dwStyle;
	INT sizes[2] = { 340, -1 };

	dwStyle  = WS_CHILD|WS_VISIBLE;
	dwStyle |= SBARS_SIZEGRIP;

	g_hWndStatus = CreateStatusWindow(dwStyle, TEXT("Daedalus"), hWndParent, IDC_STATUSBAR);
	if (!IsWindow(g_hWndStatus))
		return FALSE;

	SendMessage(g_hWndStatus, SB_SETPARTS, 2, (LPARAM)sizes);

	return TRUE;
}

BOOL Main_CreateList(HWND hWndParent)
{
	DWORD dwStyle;
	DWORD dwExStyle;
	
	dwStyle =   WS_TABSTOP | 
        WS_VISIBLE |
        WS_CHILD | 
        WS_BORDER | 
        LVS_SINGLESEL |
        LVS_REPORT | 
       // LVS_NOSORTHEADER |
        LVS_SHAREIMAGELISTS;

	dwExStyle = WS_EX_CLIENTEDGE;

	g_hWndList = ::CreateWindowEx( dwExStyle,
								WC_LISTVIEW,
								NULL,
								dwStyle,
								0,
								0,
								0,
								0,
								hWndParent,
								(HMENU)IDC_ROMLISTVIEW, // Child window id
								g_hInstance,
								NULL);
	
	return IsWindow(g_hWndList);

}

/*enum
{
	COLUMN_NAME = 0,
	COLUMN_COUNTRY,
	COLUMN_SIZE,
	COLUMN_BOOT,
	COLUMN_COMMENTS,
	COLUMN_INFO
};*/
//BOOL g_bColumnEnabled[] = { TRUE, TRUE, TRUE, TRUE, TRUE, TRUE };
//LONG g_nColumnOrder[] = { 0, 1, 2, 3, 4, 5 };


BOOL Main_InitList()
{
	LV_COLUMN   lvColumn;
	TCHAR       szString[MAX_PATH];

	// Allow the columns to be rearranged
	//ListView_SetExtendedListViewStyle(g_hWndList, LVS_EX_HEADERDRAGDROP );
	
	//initialize the columns
	lvColumn.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
	lvColumn.fmt = LVCFMT_LEFT;
	lvColumn.pszText = szString;
	
	lvColumn.cx = 180;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_NAME), sizeof(szString));
	ListView_InsertColumn(g_hWndList, 0, &lvColumn);
		
	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_COUNTRY), sizeof(szString));
	ListView_InsertColumn(g_hWndList, 1, &lvColumn);
	
	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_SIZE), sizeof(szString));
	ListView_InsertColumn(g_hWndList, 2, &lvColumn);

	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_BOOT), sizeof(szString));
	ListView_InsertColumn(g_hWndList, 3, &lvColumn);

	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_COMMENTS), sizeof(szString));
	ListView_InsertColumn(g_hWndList, 4, &lvColumn);

	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_INFO), sizeof(szString));
	ListView_InsertColumn(g_hWndList, 5, &lvColumn);
/*

	ListView_SetColumnWidth(g_hWndList, 0, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(g_hWndList, 1, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(g_hWndList, 2, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(g_hWndList, 3, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(g_hWndList, 4, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(g_hWndList, 5, LVSCW_AUTOSIZE_USEHEADER);
*/
	return TRUE;
}

static int CALLBACK Main_ListView_CompareItems(LPARAM lParam1, LPARAM lParam2, LPARAM lpData)
{
	// lpData is column to sort on:

	LPROMINFO pRomInfo1 = (LPROMINFO)lParam1;
	LPROMINFO pRomInfo2 = (LPROMINFO)lParam2;

	switch ((LONG)lpData)
	{
	case 0: return (int)lstrcmpi(pRomInfo1->szGameName, pRomInfo2->szGameName);
	case 1: return (int)pRomInfo1->nCountryID - (int)pRomInfo2->nCountryID;
	case 2: return (int)pRomInfo1->dwRomSize - (int)pRomInfo2->dwRomSize;
	case 3: return (int)lstrcmpi(pRomInfo1->szBootName, pRomInfo2->szBootName);
	case 4: return (int)lstrcmpi(pRomInfo1->szComment, pRomInfo2->szComment);
	case 5: return (int)lstrcmpi(pRomInfo1->szInfo, pRomInfo2->szInfo);
	}

	return 0;
}


void Main_SelectSaveDir(HWND hWndParent)
{
	BROWSEINFO bi;
	LPITEMIDLIST lpidl;
	TCHAR szDisplayName[MAX_PATH+1];

	bi.hwndOwner = hWndParent;
	bi.pidlRoot = NULL;
	bi.pszDisplayName = szDisplayName;
	bi.lpszTitle = CResourceString(IDS_SELECTSAVEFOLDER);
	bi.ulFlags = BIF_RETURNONLYFSDIRS;
	bi.lpfn = NULL;
	bi.lParam = NULL;
	bi.iImage = 0;

	lpidl = SHBrowseForFolder(&bi);
	if (lpidl)
	{
		/*bResult = */SHGetPathFromIDList(lpidl, g_szSaveDir);
	}
	else
	{

	}

}

void Main_AskForRomDirectory(HWND hWndParent)
{
	BROWSEINFO bi;
	LPITEMIDLIST lpidl;
	TCHAR szDisplayName[MAX_PATH+1];

	bi.hwndOwner = hWndParent;
	bi.pidlRoot = NULL;
	bi.pszDisplayName = szDisplayName;
	bi.lpszTitle = CResourceString(IDS_SELECTFOLDER);
	bi.ulFlags = BIF_RETURNONLYFSDIRS;
	bi.lpfn = NULL;
	bi.lParam = NULL;
	bi.iImage = 0;

	lpidl = SHBrowseForFolder(&bi);
	if (lpidl)
	{
		/*bResult = */SHGetPathFromIDList(lpidl, g_szRomsDir);
	}
	else
	{

	}
}


BOOL Main_FillList()
{
	if (!PathIsDirectory(g_szRomsDir))
	{
		// Don't pass in g_hMainWindow - as it may not have been set up yet!
		Main_AskForRomDirectory(g_hWndList);
		if (!PathIsDirectory(g_szRomsDir))
		{
			return FALSE;
		}
	}

	//empty the list
	ListView_DeleteAllItems(g_hWndList);

	/* bResult = */Main_AddDirectory(g_szRomsDir);

	//sort the items by name
	ListView_SortItems(g_hWndList, Main_ListView_CompareItems, (LPARAM)0);

	return TRUE;
}

// Return FALSE on critical error (stop recursing)
BOOL Main_AddDirectory(LPCTSTR szDirectory)
{
	HANDLE hFind;
	WIN32_FIND_DATA wfd;
	TCHAR szFullPath[MAX_PATH+1];
	TCHAR szSearchSpec[MAX_PATH+1];
	LPTSTR szExt;

	if (!PathIsDirectory(szDirectory))
		return FALSE;

	// Recurse, adding all roms that we find...
	PathCombine(szSearchSpec, szDirectory, TEXT("*.*"));
	
	hFind = FindFirstFile(szSearchSpec, &wfd);
	if (hFind == INVALID_HANDLE_VALUE)
		return FALSE;

	do 
	{
		// Ignore current/parent directories
		if (lstrcmp(wfd.cFileName, TEXT(".")) == 0 ||
			lstrcmp(wfd.cFileName, TEXT("..")) == 0)
			continue;

		PathCombine(szFullPath, szDirectory, wfd.cFileName);

		if (PathIsDirectory(szFullPath))
		{
			// Recurse - add the contents of this directory
			if (!Main_AddDirectory(szFullPath))
			{
				// Break/Exit??
			}
		}
		else
		{
			szExt = PathFindExtension(wfd.cFileName);

			if (lstrcmpi(szExt, TEXT(".v64")) == 0 ||
				lstrcmpi(szExt, TEXT(".z64")) == 0 ||
				lstrcmpi(szExt, TEXT(".n64")) == 0 ||
				lstrcmpi(szExt, TEXT(".rom")) == 0 ||
				lstrcmpi(szExt, TEXT(".jap")) == 0 ||
				lstrcmpi(szExt, TEXT(".pal")) == 0 ||
				lstrcmpi(szExt, TEXT(".usa")) == 0 ||
				lstrcmpi(szExt, TEXT(".zip")) == 0)
			{
				// Add to the list
				Main_AddItem(szFullPath);
			}
		}
		
	} while (FindNextFile(hFind, &wfd));

	FindClose(hFind);

	return TRUE;
}


BOOL Main_AddItem(LPCTSTR szFileName)
{
	LV_ITEM  lvItem;
	LPROMINFO pRomInfo;

	pRomInfo = new RomInfo;
	if (pRomInfo == NULL)
		return FALSE;

	ZeroMemory(pRomInfo, sizeof(RomInfo));	
	ZeroMemory(&lvItem, sizeof(lvItem));

	lstrcpyn(pRomInfo->szFileName, szFileName, MAX_PATH);

	if (!ROM_InitRomInfo(pRomInfo))
		return FALSE;
	
	lvItem.mask = LVIF_TEXT | /*LVIF_IMAGE |*/ LVIF_PARAM;		
	lvItem.iItem = ListView_GetItemCount(g_hWndList);	//add the item to the end of the list
	lvItem.lParam = (LPARAM)pRomInfo;
	lvItem.pszText = LPSTR_TEXTCALLBACK;
	//lvItem.iImage = I_IMAGECALLBACK;
	
	ListView_InsertItem(g_hWndList, &lvItem);

	return TRUE;
}

LRESULT Main_OnListView_DeleteItem(int idCtrl, LPNMHDR pnmh)
{
	NM_LISTVIEW *lpnmlv = (NM_LISTVIEW*)pnmh;
	LPROMINFO pRomInfo = (LPROMINFO)lpnmlv->lParam;
	
	//delete the pidl because we made a copy of it
	if (pRomInfo)
		delete (pRomInfo);

	return 0;
}




LRESULT Main_OnListView_GetDispInfo(int idCtrl, LPNMHDR pnmh)
{	
	LV_DISPINFO * lpdi = (LV_DISPINFO *)pnmh;
	LPROMINFO  pRomInfo = (LPROMINFO)lpdi->item.lParam;

	// Which column is being requested?
	if (lpdi->item.iSubItem == 0)
	{
		//is the text being requested?
		if(lpdi->item.mask & LVIF_TEXT)
		{
			lstrcpyn(lpdi->item.pszText, pRomInfo->szGameName, lpdi->item.cchTextMax);			
		}
		
		//is the image being requested?
		if(lpdi->item.mask & LVIF_IMAGE)
		{
			lpdi->item.iImage = -1;
		}
	}

	// Country
	else if (lpdi->item.iSubItem == 1)
	{
		//is the text being requested?
		if (lpdi->item.mask & LVIF_TEXT)
		{
			LONG i;
			//if the data text is NULL, get the no value string
			//if(lstrlen(lpdi->item.pszText) == 0)
			//	LoadString(_Module.GetModuleInstance(), IDS_UNKNOWN, lpdi->item.pszText, lpdi->item.cchTextMax);

			// Default (if not found) is character string
			wsprintf(lpdi->item.pszText, "%c", pRomInfo->nCountryID);
			for (i = 0; g_CountryCodeInfo[i].szName != NULL; i++)
			{
				if (g_CountryCodeInfo[i].nCountryID == pRomInfo->nCountryID)
				{
					lstrcpyn(lpdi->item.pszText, g_CountryCodeInfo[i].szName, lpdi->item.cchTextMax);
					break;
				}
			}

		}
	}
	// Size
	else if (lpdi->item.iSubItem == 2)
	{
		//is the text being requested?
		if (lpdi->item.mask & LVIF_TEXT)
		{
			StrFormatByteSize(pRomInfo->dwRomSize, lpdi->item.pszText, lpdi->item.cchTextMax);
			//lstrcpyn(lpdi->item.pszText, TEXT(""), lpdi->item.cchTextMax);
		}
	}
	// Comments
	else if (lpdi->item.iSubItem == 3)
	{
		//is the text being requested?
		if (lpdi->item.mask & LVIF_TEXT)
		{
			// Don't give sizes for directories:
			lstrcpyn(lpdi->item.pszText, pRomInfo->szBootName, lpdi->item.cchTextMax);
		}
	}
	// Comments
	else if (lpdi->item.iSubItem == 4)
	{
		//is the text being requested?
		if (lpdi->item.mask & LVIF_TEXT)
		{
			// Don't give sizes for directories:
			lstrcpyn(lpdi->item.pszText, pRomInfo->szComment, lpdi->item.cchTextMax);
		}
	}
	// Info
	else if (lpdi->item.iSubItem == 5)
	{
		//is the text being requested?
		if (lpdi->item.mask & LVIF_TEXT)
		{
			// Don't give sizes for directories:
			lstrcpyn(lpdi->item.pszText, pRomInfo->szInfo, lpdi->item.cchTextMax);
		}
	}
	//the item text is being requested
	else
	{
		// Huh?
	}
	return 0;
}

LRESULT Main_OnListView_ColumnClick(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	// Process LVN_COLUMNCLICK to sort items by column. 
	ListView_SortItems(lpnmlv->hdr.hwndFrom, Main_ListView_CompareItems, (LPARAM)(lpnmlv->iSubItem));

	return 0;

}

LRESULT Main_OnListView_DblClick(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	if (lpnmlv->iItem != -1)
	{
		LV_ITEM lvItem;

		ZeroMemory(&lvItem, sizeof(LV_ITEM));
		lvItem.mask = LVIF_PARAM;
		lvItem.iItem = lpnmlv->iItem;

		if (ListView_GetItem(g_hWndList, &lvItem))
		{
			LPROMINFO pRomInfo;
			pRomInfo = (LPROMINFO)lvItem.lParam;

			if (pRomInfo != NULL)
			{
				/*bSuccess = */ROM_LoadFile(pRomInfo->szFileName);
			}
		}
	}

	return 0;

}


LRESULT Main_OnListView_RClick(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	if (lpnmlv->iItem != -1)
	{
		LV_ITEM lvItem;

		ZeroMemory(&lvItem, sizeof(LV_ITEM));
		lvItem.mask = LVIF_PARAM;
		lvItem.iItem = lpnmlv->iItem;

		if (ListView_GetItem(g_hWndList, &lvItem))
		{
			LPROMINFO pRomInfo;
			pRomInfo = (LPROMINFO)lvItem.lParam;

			if (pRomInfo != NULL)
			{
				RomSettingsDialog_DoModal(g_hMainWindow, pRomInfo);
				// Refresh
				ListView_Update(g_hWndList, lpnmlv->iItem);
			}

		}
	}

	return 0;

}

LRESULT Main_OnListView_Return(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;
	LONG iItem;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	// Get the currently highlighted item
	iItem = ListView_GetNextItem(g_hWndList, -1, LVNI_SELECTED);

	if (iItem != -1)
	{
		LV_ITEM lvItem;

		ZeroMemory(&lvItem, sizeof(LV_ITEM));
		lvItem.mask = LVIF_PARAM;
		lvItem.iItem = iItem;

		if (ListView_GetItem(g_hWndList, &lvItem))
		{
			LPROMINFO pRomInfo;
			pRomInfo = (LPROMINFO)lvItem.lParam;

			if (pRomInfo != NULL)
			{
				/*bSuccess = */ROM_LoadFile(pRomInfo->szFileName);
			}

		}
	}

	return 0;
}


static HRESULT HandleWindowedModeChanges();


///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////



HRESULT MainWindow_UpdateScreen()
{
	HRESULT hr;

    // Show the frame on the primary surface.
    if( !g_GfxContext.Ready() )
        return E_FAIL;

    if( !g_bActive )
        return S_OK;

	hr = g_GfxContext.UpdateFrame( 0 );
    if ( hr == DDERR_SURFACELOST || hr == DDERR_SURFACEBUSY )
	{
		DBGConsole_Msg(0, "UpdateFrame() returned 0x%08x", hr);
        hr = HandleWindowedModeChanges();
	}



    return hr;
}

//////////////////////////////////////////////////////////////////



HRESULT HandleWindowedModeChanges()
{
    HRESULT hr;
    hr = g_GfxContext.TestCooperativeLevel();

    if (SUCCEEDED(hr))
    {
        // This means that mode changes had taken place, surfaces
        // were lost but still we are in the original mode, so we
        // simply restore all surfaces and keep going.
        hr = g_GfxContext.RestoreAllSurfaces();
		if (FAILED(hr))
            return hr;
    }
    else if (hr == DDERR_WRONGMODE)
    {
        // This means that the desktop mode has changed
        // we can destroy and recreate everything back again.
        if (FAILED(hr = Main_ReleaseD3DX()))
            return hr;
        if (FAILED(hr = Main_InitD3DX(g_hMainWindow)))
            return hr;
    }
    else if (hr == DDERR_EXCLUSIVEMODEALREADYSET)
    {
        // This means that some app took exclusive mode access
        // we need to sit in a loop till we get back to the right mode.
        do
        {
            Sleep( 500 );
        } while( DDERR_EXCLUSIVEMODEALREADYSET == 
                 (hr = g_GfxContext.TestCooperativeLevel()) );
        if (SUCCEEDED(hr))
        {
            // This means that the exclusive mode app relinquished its 
            // control and we are back to the safe mode, so simply restore
            if (FAILED( g_GfxContext.RestoreAllSurfaces()))
                return hr;
        }
        else if (hr == DDERR_WRONGMODE)
        {
            // This means that the exclusive mode app relinquished its 
            // control BUT we are back to some strange mode, so destroy
            // and recreate.
            if (FAILED(hr = Main_ReleaseD3DX()))
                return hr;
            if (FAILED(hr = Main_InitD3DX(g_hMainWindow)))
                return hr;
        }
        else
        {
            // Busted!!
            return hr;
        }
    }
    else
    {
        // Busted!!
        return hr;
    }
    return S_OK;
}



HRESULT Main_InitD3DX(HWND hWndParent)
{
    HRESULT hr;
	DWORD dwFlags;
	ID3DXContext * pD3DX;

	// Release D3DX if not already released?

    if (FAILED(hr = D3DXInitialize()))
        return hr;

	g_GfxContext.Lock();

	g_GfxContext.Cleanup();


	pD3DX = NULL;
	dwFlags = 0;
	if (g_bDisplayFullscreen)
		dwFlags |= D3DX_CONTEXT_FULLSCREEN;
    
/*
	// By Orkin - in windowed mode the display was getting scaled, this should fix it
	RECT clientRect;
	GetClientRect(hWndParent, &clientRect);
	    
    hr = D3DXCreateContext( D3DX_DEFAULT,		// D3DX handle
                            dwFlags,			// flags
                            g_hMainWindow,
                            clientRect.right - clientRect.left, // Set the size to the client size
                            clientRect.bottom - clientRect.top,
                            &pD3DX // returned D3DX interface
                            );*/

    hr = D3DXCreateContext( D3DX_DEFAULT,		// D3DX handle
                            dwFlags,			// flags
                            hWndParent,
                            g_dwDisplayWidth,
                            g_dwDisplayHeight,
                            &pD3DX // returned D3DX interface
                            );
		/*
		hr =  D3DXCreateContextEx(
			  D3DX_DEFAULT,
			  0,//D3DX_CONTEXT_FULLSCREEN,
			  hWndParent,
			  hWndParent,
			  D3DX_DEFAULT,
			  D3DX_DEFAULT,
			  D3DX_DEFAULT,
			  D3DX_DEFAULT,
			  1,
			  g_dwDisplayWidth,
			  g_dwDisplayHeight,
			  D3DX_DEFAULT,
			  &g_pD3DX
			); 
			*/

	if (SUCCEEDED(hr))
		hr = g_GfxContext.SetD3DX(pD3DX);

	g_GfxContext.Unlock();

    if (FAILED(hr))
        return hr;

    return Main_InitRenderer();
}

HRESULT Main_InitRenderer()
{
    HRESULT hr;

    if( !g_GfxContext.Ready() )
        return E_FAIL;

	// Set the device's states for rendering
	hr = RDPInit();
    if ( FAILED(hr) )
        return hr;

    return S_OK;
}


HRESULT Main_ReleaseD3DX()
{
	RDPCleanup();

	g_GfxContext.Cleanup();

    D3DXUninitialize();
    return S_OK;
}



