/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Taken from DX8 sample

#include "stdafx.h"
#include <dinput.h>
#include <atlbase.h>		// CComBSTR

#include "ConfigHandler.h"
#include "DInputHandler.h"
#include "DBGConsole.h"


static LPDIRECTINPUT8       g_pDI       = NULL;

#define ANALOGUE_STICK_RANGE		80

#define INPUT_CONFIG_VER0			0


static BOOL CALLBACK EnumJoysticksCallback( const DIDEVICEINSTANCE* pdidInstance, VOID* pContext );
static BOOL CALLBACK EnumAxesCallback( const DIDEVICEOBJECTINSTANCE* pdidoi, VOID* pContext );

static BOOL CALLBACK EnumKeyboardKeys( const DIDEVICEOBJECTINSTANCE* pdidoi, LPVOID pContext );

static HRESULT Input_IIDToAsciiString(REFGUID refguid, LPSTR szStr);
static HRESULT Input_AttemptDeviceCreate(REFGUID guid);

static void Input_ReadConfig(InputConfiguration & ic);
static void Input_WriteConfig(InputConfiguration & ic);
static HRESULT Input_CreateDevices(InputConfiguration & ic);
static void Input_FreeDevices();

static void Input_OptimiseConfig(InputConfiguration & ic);

static HWND	s_hWnd = NULL;
static std::vector< GUID > g_DeviceGuids;
static DWORD s_dwNumDevices = 0;
static LPDIRECTINPUTDEVICE8 * g_pDeviceInstances;

// As we may be called by several threads, we use this
// critical section to ensure that only one thread can 
// access the above arrays at once.
static CCritSect	s_InputSemaphore;

#define OFF_UNASSIGNED ~0

// This basically says if the n64 control is a button or a 
// axis, and gives the button bits if it is a button
N64Button g_N64Buttons[NUM_INPUT_IDS] =
{
	{ TEXT("Analogue Stick X"), 0, TRUE },		// Analogue x-axis
	{ TEXT("Analogue Stick Y"), 0, TRUE },		// Analogue y-axis
	{ TEXT("DPad Up"), U_JPAD, FALSE },			// Up Digital
	{ TEXT("DPad Left"), L_JPAD, FALSE },		// Left Digital
	{ TEXT("DPad Right"), R_JPAD, FALSE },		// Right Digital
	{ TEXT("DPad Down"), D_JPAD, FALSE },		// Down Digital
	{ TEXT("A Button"), A_BUTTON, FALSE },		// A
	{ TEXT("B Button"), B_BUTTON, FALSE },		// B
	{ TEXT("Z Trigger"), Z_TRIG, FALSE },		// Z
	{ TEXT("Left Pan"), L_TRIG, FALSE },		// L Trig
	{ TEXT("Right Pan"), R_TRIG, FALSE },		// R Trig
	{ TEXT("C Up"), U_CBUTTONS, FALSE },		// Up C
	{ TEXT("C Left"), L_CBUTTONS, FALSE },		// Left C
	{ TEXT("C Right"), R_CBUTTONS, FALSE },		// Right C
	{ TEXT("C Down"), D_CBUTTONS, FALSE },		// Down C
	{ TEXT("Start"), START_BUTTON, FALSE },		// Start
};




// In the default assignment, all buttons are assigned to the keyboard
static ButtonAssignment s_DefaultButtons[4][NUM_INPUT_IDS] =
{
	// Controller0
	{
		// 0 is for keyboard, by default input device 0
		{ 0, FALSE, DIK_LEFT,	DIK_RIGHT },
		{ 0, FALSE, DIK_UP,		DIK_DOWN },
		{ 0, FALSE, DIK_T,		0 },			// Up Digital
		{ 0, FALSE, DIK_F,		0 },			// Left Digital
		{ 0, FALSE, DIK_G,		0 },			// Right Digital
		{ 0, FALSE, DIK_V,		0 },			// Down Digital
		{ 0, FALSE, DIK_A,		0 },			// A
		{ 0, FALSE, DIK_S,		0 },			// B
		{ 0, FALSE, DIK_X,		0 },			// Z
		{ 0, FALSE, DIK_LBRACKET, 0 },			// L Trig
		{ 0, FALSE, DIK_RBRACKET, 0 },			// R Trig
		{ 0, FALSE, DIK_I,		0 },			// Up C
		{ 0, FALSE, DIK_J,		0 },			// Left C
		{ 0, FALSE, DIK_K,		0 },			// Right C
		{ 0, FALSE, DIK_M,		0 },			// Down C
		{ 0, FALSE, DIK_RETURN, 0, },			// Start
	}

};

// This is set up either by reading the config, or by rescanning the devices
InputConfiguration g_CurrInputConfig;

static std::vector<InputConfiguration> g_InputConfigs;




//-----------------------------------------------------------------------------
// Name: Input_Initialise()
// Desc: Initialize the DirectInput variables.
//-----------------------------------------------------------------------------
HRESULT Input_Initialise( HWND hWnd )
{
	LONG cont;
	LONG c;
    HRESULT hr;

	s_hWnd = hWnd;



	// Init controllers 1-3
	for (cont = 1; cont < 4; cont++)
	{
		for (c = 0; c < NUM_INPUT_IDS; c++)
		{
			s_DefaultButtons[cont][c].dwDevice = 0;
			s_DefaultButtons[cont][c].bIsAxis = FALSE;
			s_DefaultButtons[cont][c].dwOfsA = OFF_UNASSIGNED;
			s_DefaultButtons[cont][c].dwOfsB = OFF_UNASSIGNED;
		}
	}

    // Register with the DirectInput subsystem and get a pointer
    // to a IDirectInput interface we can use.
    // Create a DInput object
    if( FAILED( hr = DirectInput8Create( g_hInstance, DIRECTINPUT_VERSION, 
                                         IID_IDirectInput8, (VOID**)&g_pDI, NULL ) ) )
        return hr;


	s_InputSemaphore.Lock();


	Input_ReadConfig(g_CurrInputConfig);
	/*hr = */Input_CreateDevices(g_CurrInputConfig);
	if (s_dwNumDevices == 0)
	{
		// Reset config to default settings
		Input_ResetConfig(g_CurrInputConfig);
		/*hr = */Input_CreateDevices(g_CurrInputConfig);
	}

	s_InputSemaphore.Unlock();


	// At this point the devices will have been created, and 
	// the initial button assigments will have been created


    return S_OK;
}

BOOL CALLBACK EnumAxesCallback( const DIDEVICEOBJECTINSTANCE* pdidoi, LPVOID pContext )
{
    LPDIRECTINPUTDEVICE8 pDev = (LPDIRECTINPUTDEVICE8)pContext;

    DIPROPRANGE diprg; 
    diprg.diph.dwSize       = sizeof(DIPROPRANGE); 
    diprg.diph.dwHeaderSize = sizeof(DIPROPHEADER); 
    diprg.diph.dwHow        = DIPH_BYID; 
    diprg.diph.dwObj        = pdidoi->dwType; // Specify the enumerated axis

	// We set the range to -80 to +80 here
    diprg.lMin              = -ANALOGUE_STICK_RANGE; 
    diprg.lMax              = +ANALOGUE_STICK_RANGE; 
    
	// Set the range for the axis
	if( FAILED( pDev->SetProperty( DIPROP_RANGE, &diprg.diph ) ) )
		return DIENUM_STOP;

    return DIENUM_CONTINUE;
}



//-----------------------------------------------------------------------------
// Name: FreeDirectInput()
// Desc: Initialize the DirectInput variables.
//-----------------------------------------------------------------------------
void Input_Finalise()
{
	Input_WriteConfig(g_CurrInputConfig);
	Input_FreeDevices();

    SAFE_RELEASE( g_pDI );

	s_InputSemaphore.Unlock();

}


void Input_ReadConfig(InputConfiguration & ic)
{
	HRESULT hr;
	LONG cont;
	LONG i;
	LONG n;
	TCHAR szBuffer[200+1];
	ConfigHandler * pConfig = new ConfigHandler("Input");

	s_InputSemaphore.Lock();

	// Copy default values to current
	ic.guidDevices.clear();
	ic.bConnected[0] = TRUE;
	ic.bConnected[1] = FALSE;
	ic.bConnected[2] = FALSE;
	ic.bConnected[3] = FALSE;
	lstrcpyn(ic.szFileName, TEXT("<current>"), MAX_PATH);
	memcpy(ic.buttons, s_DefaultButtons, sizeof(ic.buttons));
	
	if (pConfig != NULL)
	{
		i = 0;
		do
		{
			pConfig->ReadString("Device", szBuffer, sizeof(szBuffer), "", i);

			if (lstrlen(szBuffer) > 0)
			{
				GUID guid;
				CComBSTR bstrGuid(szBuffer);

				hr = IIDFromString(bstrGuid, &guid);
				if (SUCCEEDED(hr))
					ic.guidDevices.push_back(guid);
			}

			i++;
		}
		while (lstrlen(szBuffer) > 0);

		// Now read in controls
		for (cont = 0; cont < 4; cont++)
		{
			for (i = 0; i < NUM_INPUT_IDS; i++)
			{
				TCHAR szName[200+1];
				ButtonAssignment ba;

				wsprintf(szName, "Controller %d %s", cont, g_N64Buttons[i].szName);
				pConfig->ReadString(szName, szBuffer, sizeof(szBuffer), "");

				n = sscanf(szBuffer, "%d %d %d %d",
					&ba.dwDevice, 
					&ba.bIsAxis,
					&ba.dwOfsA,
					&ba.dwOfsB);

				// Only assign if fields are valid
				if (n == 4 && ba.dwDevice < ic.guidDevices.size())
				{
					ic.buttons[cont][i] = ba;
				}
			}
		}
	}
	s_InputSemaphore.Unlock();
}

HRESULT Input_IIDToAsciiString(REFGUID refguid, LPSTR szStr)
{
	HRESULT hr;
	LPOLESTR szString;
	IMalloc * pMalloc;

	hr = StringFromIID(refguid, &szString);
	if (FAILED(hr))
		return hr;

	wsprintfA(szStr, "%S", szString);			// %S (not %s) is to convert to widechar

	// Free the buffer
	pMalloc = NULL;
	hr = CoGetMalloc(1, &pMalloc);
	if (SUCCEEDED(hr))
	{
		pMalloc->Free(szString);
		pMalloc->Release();
	}

	return S_OK;


}

void Input_WriteConfig(InputConfiguration & ic)
{
	HRESULT hr;
	LONG cont;
	LONG i;
	TCHAR szBuffer[200+1];

	ConfigHandler * pConfig = new ConfigHandler("Input");

	s_InputSemaphore.Lock();
	if (pConfig != NULL)
	{
		// Write out device data
		for (i = 0; i < ic.guidDevices.size(); i++)
		{
			hr = Input_IIDToAsciiString(ic.guidDevices[i], szBuffer);
			if (SUCCEEDED(hr))
			{
				pConfig->WriteString("Device", szBuffer, i);
			}
		}

		// Write out button assignments
		for (cont = 0; cont < 4; cont++)
		{
			for (i = 0; i < NUM_INPUT_IDS; i++)
			{
				TCHAR szName[200+1];

				wsprintf(szName, "Controller %d %s", cont, g_N64Buttons[i].szName);
				pConfig->ReadString(szName, szBuffer, sizeof(szBuffer), "");

				wsprintf(szBuffer, "%d %d %d %d",
					ic.buttons[cont][i].dwDevice, 
					ic.buttons[cont][i].bIsAxis,
					ic.buttons[cont][i].dwOfsA,
					ic.buttons[cont][i].dwOfsB);
				pConfig->WriteString(szName, szBuffer);
			}
		}

		delete pConfig;
	}
	s_InputSemaphore.Unlock();


}


BOOL CALLBACK EnumJoysticksCallback( const DIDEVICEINSTANCE* pdidInstance, LPVOID pContext )
{
	LONG i;
	InputConfiguration * pIC;

	// Maybe do some checks on the type here

	pIC = (InputConfiguration *)pContext;

	// Skip any devices we've already seen
	for (i = 0; i < pIC->guidDevices.size(); i++)
	{
		if (IsEqualIID(pIC->guidDevices[i], pdidInstance->guidInstance))
		    return DIENUM_CONTINUE;

	}

	pIC->guidDevices.push_back(pdidInstance->guidInstance);

    return DIENUM_CONTINUE;
}

// Rescan for any attached devices. This resets the current config
void Input_ResetConfig(InputConfiguration & ic)
{
	s_InputSemaphore.Lock();
	
	// Always include the KeyBoard!
	ic.guidDevices.clear();
	ic.guidDevices.push_back(GUID_SysKeyboard);

	// Copy default values to current
	ic.bConnected[0] = TRUE;
	ic.bConnected[1] = FALSE;
	ic.bConnected[2] = FALSE;
	ic.bConnected[3] = FALSE;
	// Leave as default
	//lstrcpyn(ic.szFileName, TEXT("<current>"), MAX_PATH);
	memcpy(ic.buttons, s_DefaultButtons, sizeof(ic.buttons));

	// Look for any attached joysticks - this basically
	// pushes the guids of any devices we disconver on
	// to ic.guidDevices
	/*hr = */g_pDI->EnumDevices( DI8DEVCLASS_GAMECTRL, 
								 EnumJoysticksCallback,
								 (LPVOID)&ic, DIEDFL_ATTACHEDONLY );
	
	s_InputSemaphore.Unlock();
}

// A bit of a hack to add any missing devices to the list of guids for the config
// This ensures that we see all the devices, even if the config has been "optimised"
// at some point
void Input_AddMissingDevices(InputConfiguration & ic)
{
	s_InputSemaphore.Lock();

	// Same call as above - it checks for duplicates
	/*hr = */g_pDI->EnumDevices( DI8DEVCLASS_GAMECTRL, 
								 EnumJoysticksCallback,
								 (LPVOID)&ic, DIEDFL_ATTACHEDONLY );

	s_InputSemaphore.Unlock();
}




void Input_FreeDevices()
{
	LONG i;

	s_InputSemaphore.Lock();
	if (g_pDeviceInstances != NULL)
	{
		for (i = 0; i < s_dwNumDevices; i++)
		{
			if (g_pDeviceInstances[i] != NULL)
			{
				g_pDeviceInstances[i]->Unacquire();
				g_pDeviceInstances[i]->Release();
			}
		}
		delete [] g_pDeviceInstances;
	}	
	s_dwNumDevices = 0;
	s_InputSemaphore.Unlock();
}

// IMPORTANT! Must be called from within the critical section!
HRESULT Input_CreateDevices(InputConfiguration & ic)
{
	LONG i;
	HRESULT hr;

	// Free any devices that we have acquired
	Input_FreeDevices();

	// Size g_DeviceInstances to the same size
	s_dwNumDevices = ic.guidDevices.size();
	g_pDeviceInstances = new LPDIRECTINPUTDEVICE8[s_dwNumDevices];
	if (g_pDeviceInstances == NULL)
	{
		s_dwNumDevices = 0;
		return E_OUTOFMEMORY;
	}

	// Clear array
	ZeroMemory(g_pDeviceInstances, sizeof(LPDIRECTINPUTDEVICE8) * s_dwNumDevices);

	for (i = 0; i < s_dwNumDevices; i++)
	{
		LPDIRECTINPUTDEVICE8 pDevice = NULL;

		// Device 1 is the first joystick
	    hr = g_pDI->CreateDevice( ic.guidDevices[i], &pDevice, NULL );

		if SUCCEEDED(hr)
		{
			if (IsEqualIID(ic.guidDevices[i], GUID_SysKeyboard))
			{
				hr = pDevice->SetDataFormat( &c_dfDIKeyboard );
				if (FAILED(hr))
				{
					pDevice->Release();
					continue;		// Try the next device
				}

			}
			else
			{
				// Set the data format to "simple joystick" - a predefined data format 
				//
				// A data format specifies which controls on a device we are interested in,
				// and how they should be reported. This tells DInput that we will be
				// passing a DIJOYSTATE2 structure to IDirectInputDevice::GetDeviceState().
				hr = pDevice->SetDataFormat( &c_dfDIJoystick2 );
				if (FAILED(hr))
				{
					pDevice->Release();
					continue;		// Try the next device
				}
			}

			// Set the cooperative level to let DInput know how this device should
			// interact with the system and with other DInput applications.
			hr = pDevice->SetCooperativeLevel( s_hWnd, DISCL_EXCLUSIVE | // DISCL_NONEXCLUSIVE
													   DISCL_FOREGROUND );
			if (FAILED(hr))
			{
				pDevice->Release();
				continue;
			}

			// Enumerate the axes of the joystick and set the range of each axis. 
			hr = pDevice->EnumObjects( EnumAxesCallback, 
									(VOID*)pDevice, DIDFT_AXIS );

			if (FAILED(hr))
			{
				pDevice->Release();
				continue;
			}
			g_pDeviceInstances[i] = pDevice;
		}

	}

	return S_OK;
}


// We need this struct to pass in both control list and the device instance info
typedef struct
{
	ControlList * pcl;
	DWORD dwDevice;

	TCHAR szDeviceName[MAX_PATH+1];

} EnumObjectInfo;


HRESULT Input_GenerateAllObjects(InputConfiguration & ic, ControlList * pCL)
{
	LONG b;
	LONG i;
	HRESULT hr;
	DIDEVICEINSTANCE di;
	DIDEVCAPS diDevCaps;

	if (pCL == NULL)
		return E_POINTER;

	pCL->controls.clear();

	s_InputSemaphore.Lock();
	for (i = 0; i < ic.guidDevices.size(); i++)
	{
		LPDIRECTINPUTDEVICE8 pDevice = NULL;

		// Create the device
	    hr = g_pDI->CreateDevice( ic.guidDevices[i], &pDevice, NULL );

		// Skip any devices we couldn't create
		if (FAILED(hr) || pDevice == NULL)
		{
			CHAR szBuf[100];
			Input_IIDToAsciiString(ic.guidDevices[i], szBuf);
			DBGConsole_Msg(0, "Couldn't create device %s!", szBuf);
			continue;
		}

		// Get the device info so that we can pass it into the enumeration function
		di.dwSize = sizeof(di);
		hr = pDevice->GetDeviceInfo(&di);
		if (FAILED(hr))
		{
			pDevice->Release();
			continue;
		}

		if (IsEqualIID(ic.guidDevices[i], GUID_SysKeyboard))
		{
			EnumObjectInfo eoi;
			// For the keyboard, enumerate all the buttons so that we
			// can get the diplay names
			lstrcpyn(eoi.szDeviceName, di.tszInstanceName, sizeof(eoi.szDeviceName));

			eoi.pcl = pCL;
			eoi.dwDevice = i;

			/*hr = */pDevice->EnumObjects( EnumKeyboardKeys, (LPVOID)&eoi, DIDFT_AXIS|DIDFT_BUTTON);
		}
		else
		{
			ControlID cid;

			diDevCaps.dwSize = sizeof(DIDEVCAPS);
			hr = pDevice->GetCapabilities(&diDevCaps);
			if (FAILED(hr))
			{
				pDevice->Release();
				continue;
			}

			cid.dwDevice = i;

			if (diDevCaps.dwAxes >= 1)
			{
				cid.dwOfs = DIJOFS_X;
				wsprintf(cid.szName, "%s.X Axis", di.tszInstanceName);
				cid.bIsAxis = TRUE;
				pCL->controls.push_back(cid);
			}
			if (diDevCaps.dwAxes >= 2)
			{
				cid.dwOfs = DIJOFS_Y;
				wsprintf(cid.szName, "%s.Y Axis", di.tszInstanceName);
				cid.bIsAxis = TRUE;
				pCL->controls.push_back(cid);
			}
			// Ignore Z axis

			for (b = 0; b < diDevCaps.dwButtons; b++)
			{
				cid.dwOfs = DIJOFS_BUTTON(b);
				cid.bIsAxis = FALSE;
				wsprintf(cid.szName, "%s.Button %d", di.tszInstanceName, b);
				pCL->controls.push_back(cid);
				// Add button
			}
		}

		// Free the device
		pDevice->Release();



	}
	s_InputSemaphore.Unlock();

	return S_OK;
}

static BOOL Input_IsObjectOnExcludeList( DWORD dwOfs )
{
	if (dwOfs == DIK_PREVTRACK  ||
	    dwOfs == DIK_NEXTTRACK  ||
	    dwOfs == DIK_MUTE       ||
	    dwOfs == DIK_CALCULATOR ||
	    dwOfs == DIK_PLAYPAUSE  ||
	    dwOfs == DIK_MEDIASTOP  ||
	    dwOfs == DIK_VOLUMEDOWN ||
	    dwOfs == DIK_VOLUMEUP   ||
	    dwOfs == DIK_WEBHOME    ||
	    dwOfs == DIK_SLEEP      ||
	    dwOfs == DIK_WEBSEARCH  ||
	    dwOfs == DIK_WEBFAVORITES ||
	    dwOfs == DIK_WEBREFRESH ||
	    dwOfs == DIK_WEBSTOP    ||
	    dwOfs == DIK_WEBFORWARD ||
	    dwOfs == DIK_WEBBACK    ||
	    dwOfs == DIK_MYCOMPUTER ||
	    dwOfs == DIK_MAIL       ||
		dwOfs == DIK_WAKE		||
		dwOfs == DIK_POWER		||
	    dwOfs == DIK_MEDIASELECT)
		return TRUE;

	return FALSE;
}

BOOL CALLBACK EnumKeyboardKeys( const DIDEVICEOBJECTINSTANCE* pdidoi, LPVOID pContext )
{
	ControlID cid;
	EnumObjectInfo * peoi;

	peoi = (EnumObjectInfo *)pContext;

	if (pdidoi->guidType == GUID_Key)
	{
		if (Input_IsObjectOnExcludeList(pdidoi->dwOfs))
			return DIENUM_CONTINUE;

		cid.dwDevice = peoi->dwDevice;
		cid.dwOfs = pdidoi->dwOfs;

		wsprintf(cid.szName, "%s.%s", peoi->szDeviceName, pdidoi->tszName);

		cid.bIsAxis = FALSE;
		peoi->pcl->controls.push_back(cid);
	}


    return DIENUM_CONTINUE;
}

// Unaquire the input focus, if we have it
void Input_Unaquire()
{
	LONG i;

	s_InputSemaphore.Lock();

	// Release any other devices we have acquired
	if (g_pDeviceInstances != NULL)
	{
		for (i = 0; i < s_dwNumDevices; i++)
		{
			if (g_pDeviceInstances[i] != NULL)
				g_pDeviceInstances[i]->Unacquire();
		}
	}	

	s_InputSemaphore.Unlock();
}

// Checks the keyboard for any input and passes VK_ messages on to the main window
// This is to prevent lockups in fullscreen mode where the game stops reading from the
// controller and so out checks for escape etc are missed
HRESULT Input_CheckKeyboard( HWND hWnd  )
{
	LONG i;
    HRESULT     hr;
	BYTE    diks[256];   // DirectInput keyboard state buffer 
	LONG nAttempts;

	s_InputSemaphore.Lock();

	for (i = 0; i < s_dwNumDevices; i++)
	{
		LPDIRECTINPUTDEVICE8 pDev;

		// Skip until we get the keyboard device
		if (!IsEqualIID(g_CurrInputConfig.guidDevices[i], GUID_SysKeyboard))
			continue;

		pDev = g_pDeviceInstances[i];
		// Skip any devices we couldn't create
		if (pDev == NULL)
			break;

		hr = pDev->Poll(); 
		if (FAILED(hr))
		{
			// DInput is telling us that the input stream has been
			// interrupted. We aren't tracking any state between polls, so
			// we don't have any special reset that needs to be done. We
			// just re-acquire and try again.
			hr = pDev->Acquire();
			nAttempts = 0;
			while ( hr == DIERR_INPUTLOST && nAttempts < 5 ) 
			{
				hr = pDev->Acquire();
				nAttempts++;
			}

			// hr may be DIERR_OTHERAPPHASPRIO or other errors.  This
			// may occur when the app is minimized or in the process of 
			// switching, so just try again later 
			break; 
		}

		// Treat keyboard differently - use the diks structure
		ZeroMemory( diks, sizeof(diks) );
		hr = pDev->GetDeviceState( sizeof(diks), diks );

		if (FAILED(hr)) 
			break;		// The device should have already been acquired

		if (diks[DIK_ESCAPE] & 0x80)
			PostMessage(hWnd, WM_KEYDOWN, (WPARAM)VK_ESCAPE, 0);
		if (diks[DIK_F1] & 0x80)
			PostMessage(hWnd, WM_KEYDOWN, (WPARAM)VK_F1, 0);
		if (diks[DIK_F2] & 0x80)
			PostMessage(hWnd, WM_KEYDOWN, (WPARAM)VK_F2, 0);
		if (diks[DIK_F12] & 0x80)
			PostMessage(hWnd, WM_KEYDOWN, (WPARAM)VK_F12, 0);
		if (diks[DIK_SYSRQ] & 0x80)
			PostMessage(hWnd, WM_KEYDOWN, (WPARAM)VK_F12, 0);

	}
	s_InputSemaphore.Unlock();
    return S_OK;
}


HRESULT Input_GetState( OSContPad pPad[4] )
{
	LONG i;
	LONG c;
	LONG cont;
    HRESULT     hr;
	BYTE    diks[256];   // DirectInput keyboard state buffer 
	DIJOYSTATE2 js;      // DirectInput joystick state 
	BYTE * pbInput;
	DWORD dwInputSize;
	LONG nAttempts;

	for (cont = 0; cont < 4; cont++)
	{
		pPad[cont].button = 0;
		pPad[cont].stick_x = 0;
		pPad[cont].stick_y = 0;
	}

	s_InputSemaphore.Lock();

	for (i = 0; i < s_dwNumDevices; i++)
	{
		LPDIRECTINPUTDEVICE8 pDev;

		pDev = g_pDeviceInstances[i];
		// Skip any devices we couldn't create
		if (pDev == NULL)
			continue;


		hr = pDev->Poll(); 
		if (FAILED(hr))
		{
			// DInput is telling us that the input stream has been
			// interrupted. We aren't tracking any state between polls, so
			// we don't have any special reset that needs to be done. We
			// just re-acquire and try again.
			/* switch (hr)
			{
			case DIERR_INPUTLOST:
				DBGConsole_Msg(0, "Poll returned DIERR_INPUTLOST");
				break;
			case DIERR_NOTACQUIRED:
				DBGConsole_Msg(0, "Poll returned DIERR_NOTACQUIRED");
				break;
			case DIERR_NOTINITIALIZED:
				DBGConsole_Msg(0, "Poll returned DIERR_NOTINITIALIZED");
				break;
			default:
				DBGConsole_Msg(0, "Poll returned 0x%08x", hr);
				break;
			}*/

			hr = pDev->Acquire();
			nAttempts = 0;
			while ( hr == DIERR_INPUTLOST && nAttempts < 5) 
			{
				DBGConsole_Msg(0, "Retrying...");
				hr = pDev->Acquire();
				nAttempts++;
			}

			/*switch (hr)
			{
			case DIERR_INPUTLOST:
				DBGConsole_Msg(0, "Acquire returned DIERR_INPUTLOST");
				break;
			case DIERR_NOTACQUIRED:
				DBGConsole_Msg(0, "Acquire returned DIERR_NOTACQUIRED");
				break;
			case DIERR_NOTINITIALIZED:
				DBGConsole_Msg(0, "Acquire returned DIERR_NOTINITIALIZED");
				break;
			case DIERR_OTHERAPPHASPRIO:
				DBGConsole_Msg(0, "Acquire returned DIERR_OTHERAPPHASPRIO ");
				break;
			default:
				DBGConsole_Msg(0, "Acquire returned 0x%08x", hr);
				break;
			}*/


			// hr may be DIERR_OTHERAPPHASPRIO or other errors.  This
			// may occur when the app is minimized or in the process of 
			// switching, so just try again later 
			continue; 
		}

		// Treat keyboard differently - use the diks structure
		if (IsEqualIID(g_CurrInputConfig.guidDevices[i], GUID_SysKeyboard))
		{
			pbInput = diks;
			dwInputSize = sizeof(diks);
		}
		else
		{
			pbInput = (BYTE*)&js;
			dwInputSize = sizeof(js);
		}

		ZeroMemory( pbInput, dwInputSize );
		hr = pDev->GetDeviceState( dwInputSize, pbInput );

		if (FAILED(hr)) 
			continue;		// The device should have already been acquired

		// Do this here too, as well as Input_CheckKeyboard
		// The displaylist (f1) key needs to be polled quite often to 
		// ensure that display lists are caught close to when the
		// user presses the key
		if (IsEqualIID(g_CurrInputConfig.guidDevices[i], GUID_SysKeyboard))
		{
			if (diks[DIK_ESCAPE] & 0x80)
				PostMessage(g_hMainWindow, WM_KEYDOWN, (WPARAM)VK_ESCAPE, 0);
			if (diks[DIK_F1] & 0x80)
				PostMessage(g_hMainWindow, WM_KEYDOWN, (WPARAM)VK_F1, 0);
			if (diks[DIK_F2] & 0x80)
				PostMessage(g_hMainWindow, WM_KEYDOWN, (WPARAM)VK_F2, 0);
			if (diks[DIK_F12] & 0x80)
				PostMessage(g_hMainWindow, WM_KEYDOWN, (WPARAM)VK_F12, 0);
			if (diks[DIK_SYSRQ] & 0x80)
				PostMessage(g_hMainWindow, WM_KEYDOWN, (WPARAM)VK_F12, 0);
		}

		for (cont = 0; cont < 4; cont++)
		{
			for (c = 0; c < NUM_INPUT_IDS; c++)
			{
				DWORD dwOfsA = g_CurrInputConfig.buttons[cont][c].dwOfsA;
				DWORD dwOfsB = g_CurrInputConfig.buttons[cont][c].dwOfsB;

				// Only process this input if it is assigned to the device we're looking at
				if (g_CurrInputConfig.buttons[cont][c].dwDevice != i)
					continue;

				// Check if this input is a button or a stick
				if (!g_N64Buttons[c].bIsAxis)
				{
					// The n64 control is a button - we 
					// should only use the first offset
					if (dwOfsA != OFF_UNASSIGNED &&
						pbInput[dwOfsA] & 0x80)
						pPad[cont].button |= g_N64Buttons[c].button;

				}
				else
				{
					LONG val;


					val = 0;
					// The n64 control is an axis. If bIsAxis
					// is set, then .dwOfsA refers to an axis value
					if (g_CurrInputConfig.buttons[cont][c].bIsAxis)
					{
						// Val should be in the range -80 to 80 already, 
						// As we set this property of the joystick during init
						if (dwOfsA != OFF_UNASSIGNED)
							val = ((LONG*)pbInput)[dwOfsA/4];
					}
					else
					{
						if (dwOfsA != OFF_UNASSIGNED &&
							pbInput[dwOfsA] & 0x80)
							val = -ANALOGUE_STICK_RANGE;
						else if (dwOfsB != OFF_UNASSIGNED &&
							pbInput[dwOfsB] & 0x80)
							val = +ANALOGUE_STICK_RANGE;
					}

					// Check which of the axis it was.
					// Later we might want the dpad and c buttons to be treated
					// as axis too
					if (c == INPUT_ID_STICK_X)
						pPad[cont].stick_x = val;
					else if (c == INPUT_ID_STICK_Y)
						pPad[cont].stick_y = -val;
				}
			}
		}
	}
	s_InputSemaphore.Unlock();
    return S_OK;
}

// Remove unused devices from the list
// If only the keyboard and one gamepad is referenced, all other devices will
// be stripped from the config
void Input_OptimiseConfig(InputConfiguration & ic)
{
	LONG cont;
	LONG c;
	LONG e;
	BOOL bFound;
	DWORD iOrig;
	DWORD iFound;
	std::vector<DWORD> map;
	std::vector<GUID> newguids;

	// This was to ensure keyboard was always 0 - not needed now as we just check guids
	//map.push_back(0);		// map 0 -> 0
	//newguids.push_back(ic.guidDevices[0]);

	for (cont = 0; cont < 4; cont++)
	{
		for (c = 0; c < NUM_INPUT_IDS; c++)
		{
			bFound = FALSE;
			iOrig = ic.buttons[cont][c].dwDevice;
			
			for (e = 0; e < map.size(); e++)
			{
				if (map[e] == iOrig)
				{
					iFound = e;
					bFound = TRUE;
					break;
				}
			}


			if (!bFound)
			{
				newguids.push_back(ic.guidDevices[iOrig]);
				map.push_back(iOrig);
				iFound = map.size() - 1;
			}
			ic.buttons[cont][c].dwDevice = iFound;

			
		}
	}

	DBGConsole_Msg(0, "Optimized map: %d -> %d",
		ic.guidDevices.size(), 
		newguids.size());

	// Copy new guids across
	ic.guidDevices.clear();
	for (e = 0; e < newguids.size(); e++)
	{
		ic.guidDevices.push_back(newguids[e]);
	}
}



// Load config from specified filename
HRESULT Input_LoadConfig(InputConfiguration & ic)
{
	HRESULT hr;
	FILE * fh;
	LONG n;
	LONG i;
	DWORD dwNumDevices;
	DWORD dwVersion;

	// Attempt to open
	fh = fopen(ic.szFileName, "rb");
	if (fh == NULL)
		return E_FAIL;

	n = fread(&dwVersion, sizeof(dwVersion), 1, fh);
	if (n != 1)
		goto fail_cleanup;

	if (dwVersion != INPUT_CONFIG_VER0)
	{
		DBGConsole_Msg(0, "Unknown version: %d", dwVersion);
		goto fail_cleanup;
	}

	// Read in devices
	n = fread(&dwNumDevices, sizeof(dwNumDevices), 1, fh);
	if (n != 1)
		goto fail_cleanup;

	ic.guidDevices.clear();
	for (i = 0; i < dwNumDevices; i++)
	{
		GUID guid;

		n = fread(&guid, sizeof(guid), 1, fh);
		if (n != 1)
			goto fail_cleanup;

		hr = Input_AttemptDeviceCreate(guid);
		if (FAILED(hr))
			goto fail_cleanup;

		ic.guidDevices.push_back(guid);
	}	

	// Read in connected/unconnected
	n = fread(ic.bConnected, sizeof(BOOL), 4, fh);
	if (n != 4)
		goto fail_cleanup;

	// Read in buttons
	n = fread(ic.buttons, sizeof(ic.buttons), 1, fh);
	if (n != 1)
		goto fail_cleanup;

	// Remove any devices that are needed
	Input_OptimiseConfig(ic);


	fclose(fh);
	return S_OK;

fail_cleanup:
	fclose(fh);
	return E_FAIL;
}

// Attempt to create the specified device
// Display a warning if we couldn't create an instance of the specified device
HRESULT Input_AttemptDeviceCreate(REFGUID guid)
{
	HRESULT hr;
	LPDIRECTINPUTDEVICE8 pDevice = NULL;

	// Create the device
	hr = g_pDI->CreateDevice( guid, &pDevice, NULL );

	// Skip any devices we couldn't create
	if (FAILED(hr) || pDevice == NULL)
	{
		CHAR szBuf[100];
		Input_IIDToAsciiString(guid, szBuf);
		DBGConsole_Msg(0, "Couldn't create device %s!", szBuf);
		return hr;
	}

	pDevice->Release();
	return S_OK;

}



HRESULT Input_SaveConfig(LONG iConfig)
{
	FILE * fh;
	LONG i;
	DWORD dwNumDevices;
	DWORD dwVersion;

	// Check bounds!
	if (iConfig < 0 || iConfig >= g_InputConfigs.size())
		return E_FAIL;

	InputConfiguration & ic = g_InputConfigs[iConfig];

	DBGConsole_Msg(0, "Saving Config %s", ic.szFileName);

	fh = fopen(ic.szFileName, "wb");
	if (fh == NULL)
		return E_FAIL;

	// Strip any unused devices
	Input_OptimiseConfig(ic);

	dwVersion = INPUT_CONFIG_VER0;
	fwrite(&dwVersion, sizeof(dwVersion), 1, fh);

	// Write devices
	dwNumDevices = ic.guidDevices.size();
	fwrite(&dwNumDevices, sizeof(dwNumDevices), 1, fh);

	// Write each device
	for (i = 0; i < ic.guidDevices.size(); i++)
	{
		GUID guid;

		guid = ic.guidDevices[i];
		fwrite(&guid, sizeof(guid), 1, fh);
	}	

	fwrite(ic.bConnected, sizeof(BOOL), 4, fh);
	fwrite(ic.buttons, sizeof(ic.buttons), 1, fh);

	fclose(fh);
	return S_OK;
}


// Iterate through all files in g_szDaedalueExeDir/Input
void Input_LoadAllInputConfigs()
{
	HRESULT hr;
	TCHAR szSearch[MAX_PATH+1];
	TCHAR szInputDir[MAX_PATH+1];
	WIN32_FIND_DATA fd;
	HANDLE hFind;
	InputConfiguration ic;

	g_InputConfigs.clear();

	PathCombine(szInputDir, g_szDaedalusExeDir, "Input");
	PathCombine(szSearch, szInputDir, "*.din");

	hFind = FindFirstFile(szSearch, &fd);
	if (hFind != INVALID_HANDLE_VALUE)
	{
		do
		{
			// Skip current/parent dirs
			if (lstrcmpi(fd.cFileName, TEXT(".")) == 0 ||
				lstrcmpi(fd.cFileName, TEXT("..")) == 0)
				continue;

			PathCombine(ic.szFileName, szInputDir, fd.cFileName);

			{
				//DBGConsole_Msg(0, "Loading input config %s", szFileName);
				hr = Input_LoadConfig(ic);
				if (SUCCEEDED(hr))
				{
					g_InputConfigs.push_back(ic);
				}
				else
				{	
					DBGConsole_Msg(0, "Unable to load input config [C%s]", ic.szFileName);
				}	
			}

		} while (FindNextFile(hFind, &fd));

		FindClose(hFind);
	}
}

// Generate a new config based on the current config
void Input_NewConfig(InputConfiguration & ic)
{
	// Copy the default config
	ic = g_CurrInputConfig;

	PathCombine(ic.szFileName, g_szDaedalusExeDir, "Input");
	PathAppend(ic.szFileName, TEXT("NewConfig.din"));

}

LONG Input_AddConfig(InputConfiguration & ic)
{
	// Name properly!
	TCHAR szNewFileName[MAX_PATH+1];

	PathCombine(szNewFileName, g_szDaedalusExeDir, "Input");
	PathAppend(szNewFileName, ic.szFileName);
	PathAddExtension(szNewFileName, TEXT(".din"));

	lstrcpyn(ic.szFileName, szNewFileName, MAX_PATH);
	g_InputConfigs.push_back(ic);

	// Return the index we inserted at
	return g_InputConfigs.size() - 1;
}

LONG Input_GetNumConfigs()
{
	return g_InputConfigs.size();
}

// Utility to change the name of the specified config
// This essentially updates the field in ic, and
// renames the config file on disk
void Input_RenameConfig(InputConfiguration & ic, LPCTSTR szNewName)
{
	TCHAR szOrigName[MAX_PATH+1];
	TCHAR szNewFileName[MAX_PATH+1];

	PathCombine(szOrigName, g_szDaedalusExeDir, "Input");
	PathAppend(szOrigName, ic.szFileName);
	PathAddExtension(szOrigName, TEXT(".din"));

	PathCombine(szNewFileName, g_szDaedalusExeDir, "Input");
	PathAppend(szNewFileName, szNewName);
	PathAddExtension(szNewFileName, TEXT(".din"));

	// Rename file on disk, if it exists
	MoveFile(szOrigName, szNewFileName);

	DBGConsole_Msg(0, "Renaming input config\n[C%s] -> [C%s]",
		szOrigName, szNewFileName);


	lstrcpyn(ic.szFileName, szNewFileName, MAX_PATH);

}

BOOL Input_GetConfig(LONG iConfig, InputConfiguration & ic)
{
	if (iConfig == -1)
	{
		ic = g_CurrInputConfig;
		return TRUE;
	}
	else if (iConfig < g_InputConfigs.size())
	{
		ic = g_InputConfigs[iConfig];
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

void Input_SetConfig(LONG iConfig, InputConfiguration & ic)
{
	if (iConfig == -1)
	{
		g_CurrInputConfig = ic;

		// Preserve name!
		lstrcpyn(g_CurrInputConfig.szFileName, TEXT("<current>"), MAX_PATH);

		// Recreate devices!
		/*hr = */Input_CreateDevices(g_CurrInputConfig);
	}
	else if (iConfig < g_InputConfigs.size())
	{
		// Hmm - need to find a better way of detecting changes
		// This picks them all up now because the dialog always
		// writes the display name to the filename field
		if (lstrcmpi(g_InputConfigs[iConfig].szFileName, ic.szFileName) != 0)
		{
			Input_RenameConfig(g_InputConfigs[iConfig], ic.szFileName);

			// Hmm hack to make sure filename is the same when it's synchronised below
			// Really should sort out the filename gubbins!
			lstrcpyn(ic.szFileName, g_InputConfigs[iConfig].szFileName, MAX_PATH);
		}
		g_InputConfigs[iConfig] = ic;
	}

}