/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Lots of additional work by Lkb - thanks!

#include "stdafx.h"


#include "ultra_r4300.h"

#include "DBGConsole.h"

#include "Registers.h"	// For RegNames
#include "CPU.h"
#include "RSP.h"
#include "RDP.h"
#include "RDP_GFX.h"
#include "OpCode.h"
#include "Patch.h"		// To get patch names
#include "Dump.h"
#include "Controller.h"

// Size of the background text buffer
static const COORD sc_outbufsize = { 80, 120 };

static HANDLE g_hStdOut = NULL;
static HANDLE g_hOutputBuffer = INVALID_HANDLE_VALUE;
static HANDLE g_hConsoleThread = NULL;

static BOOL DBGConsole_Init();
static void DBGConsole_Fini();

static BOOL DBGConsole_ResizeConBufAndWindow(HANDLE hConsole, SHORT xSize, SHORT ySize);
static DWORD __stdcall DBGConsole_ConsoleFunc(LPVOID pArg);
static void DBGConsole_ProcessKeyEvent(KEY_EVENT_RECORD kr);

static void DBGConsole_ParseStringHighlights(LPSTR szString, WORD * arrwAttributes, WORD wAttr);
static BOOL DBGConsole_WriteString(LPCTSTR szString, BOOL bParse, SHORT x, SHORT y, WORD wAttr, SHORT width);
static void DBGConsole_DisplayOutput(LONG nOffset);

enum DBGCON_MIDPANETYPE
{
	DBGCON_CPU_CODE_VIEW = 0,
	DBGCON_RSP_CODE_VIEW,
};

CHAR * s_szPaneNameLabels[] = { "Disassembly", "RSP Disassembly", "Memory" };

enum DBGCON_PANETYPE
{
	DBGCON_REG_PANE = 0,		// Register view
	DBGCON_MEM_PANE,
	DBGCON_MID_PANE,			// Disasm/Mem area
	DBGCON_BOT_PANE				// Output area
};

static DWORD g_dwMemAddress = 0xb0000000;
static DWORD g_dwDisplayPC = ~0;
static DWORD g_dwDisplayRSP_PC = ~0;
static LONG  g_nOutputBufOffset = 0;

static DBGCON_MIDPANETYPE s_nMidPaneType = DBGCON_CPU_CODE_VIEW;

static DBGCON_PANETYPE s_nActivePane = DBGCON_MID_PANE;

static void DBGConsole_SetActivePane(DBGCON_PANETYPE pane);
static void DBGConsole_RedrawPaneLabels();

static void DBGConsole_ProcessInput();
static void DBGConsole_HelpCommand(LPCTSTR szCommand);
static void DBGConsole_GoCommand();
static void DBGConsole_StopCommand();

//////////////////////////////////////////////////////////////////////////
static void DBGConsole_DumpContext(DWORD dwPC);
static void DBGConsole_DumpContext_NextLine();
static void DBGConsole_DumpContext_NextPage();
static void DBGConsole_DumpContext_PrevLine();
static void DBGConsole_DumpContext_PrevPage();

static void DBGConsole_DumpMemory(DWORD dwAddress);
static void DBGConsole_DumpCPUContext(DWORD dwPC);
static void DBGConsole_DumpRSPContext(DWORD dwPC);


//////////////////////////////////////////////////////////////////////////
static void DBGConsole_DumpRegisters();
static void DBGConsole_DumpCPURegisters();
static void DBGConsole_DumpRSPRegisters();



//////////////////////////////////////////////////////////////////////////
static const WORD sc_wAttrWhite = FOREGROUND_RED|FOREGROUND_GREEN|FOREGROUND_BLUE;
static const WORD sc_wAttrBoldWhite = FOREGROUND_INTENSITY|FOREGROUND_RED|FOREGROUND_GREEN|FOREGROUND_BLUE;
static const WORD sc_wAttrRed = FOREGROUND_RED;
static const WORD sc_wAttrBoldRed = FOREGROUND_INTENSITY|FOREGROUND_RED;

static TCHAR g_szInput[1024+1];

static const int sc_nTopPaneTextOffset = 0;  static const sc_nTopPaneLines = 11;
static const int sc_nMemPaneTextOffset = 12; static const sc_nMemPaneLines = 7;
static const int sc_nMidPaneTextOffset = 20; static const sc_nMidPaneLines = 13;
static const int sc_nBotPaneTextOffset = 34; static const sc_nBotPaneLines = 24;
static const int sc_nCommandTextOffset = 59 ;



typedef struct tagDebugCommandInfo
{
	LPCTSTR szCommand;
	void (* pArglessFunc)();
	void (* pArgFunc)(LPCTSTR);
	LPCTSTR szHelpText;
} DebugCommandInfo;

typedef struct tagHelpTopicInfo
{
	LPCTSTR szHelpTopic;
	LPCTSTR szHelpText;
} HelpTopicInfo;

static void DBGConsole_IntPi() { DBGConsole_Msg(0, "Pi Interrupt"); Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_PI); AddCPUJob(CPU_CHECK_INTERRUPTS); }
static void DBGConsole_IntAi() { DBGConsole_Msg(0, "Ai Interrupt"); Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_AI); AddCPUJob(CPU_CHECK_INTERRUPTS); }
static void DBGConsole_IntDp() { DBGConsole_Msg(0, "Dp Interrupt"); Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_DP); AddCPUJob(CPU_CHECK_INTERRUPTS); }
static void DBGConsole_IntSp() { DBGConsole_Msg(0, "Sp Interrupt"); Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_SP); AddCPUJob(CPU_CHECK_INTERRUPTS); }
static void DBGConsole_IntSi() { DBGConsole_Msg(0, "Si Interrupt"); Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_SI); AddCPUJob(CPU_CHECK_INTERRUPTS); }
static void DBGCommand_FP(LPCTSTR szCommand);
static void DBGCommand_Vec(LPCTSTR szCommand);
static void DBGCommand_Mem(LPCTSTR szCommand);
static void DBGCommand_List(LPCTSTR szCommand);
static void DBGCommand_ListOS(LPCTSTR szCommand);
static void DBGCommand_BPX(LPCTSTR szCommand);
static void DBGCommand_BPD(LPCTSTR szCommand);
static void DBGCommand_BPE(LPCTSTR szCommand);
static void DBGCommand_ShowCPU();
static void DBGCommand_ShowRSP();

static void DBGCommand_Write8(LPCTSTR szCommand);
static void DBGCommand_Write16(LPCTSTR szCommand);
static void DBGCommand_Write32(LPCTSTR szCommand);
static void DBGCommand_Write64(LPCTSTR szCommand);
static void DBGCommand_WriteReg(LPCTSTR szCommand);

static void DBGCommand_Dis(LPCTSTR szCommand);
static void DBGCommand_RDis(LPCTSTR szCommand);

static void DBGCommand_Close();
static void DBGCommand_Quit();

#define BEGIN_DBGCOMMAND_MAP(name) \
static const DebugCommandInfo name[] =	\
{
#define DBGCOMMAND_ARGLESS(cmd, func) \
{ cmd, func, NULL, NULL },
#define DBGCOMMAND_ARG(cmd, func) \
{ cmd, NULL, func, NULL },

#define DBGCOMMAND_ARGLESS_HELP(cmd, func, help) \
{ cmd, func, NULL, help },
#define DBGCOMMAND_ARG_HELP(cmd, func, help) \
{ cmd, NULL, func, help },

#define BEGIN_DBGCOMMAND_ARGLESS_HELP(cmd, func) \
{ cmd, func, NULL, 
#define BEGIN_DBGCOMMAND_ARG_HELP(cmd, func) \
{ cmd, NULL, func,

#define BEGIN_DBGCOMMAND() },

#define END_DBGCOMMAND_MAP()		\
	{ NULL, NULL }					\
};

#define BEGIN_HELPTOPIC_MAP(name) \
static const HelpTopicInfo name[] =	\
{

#define HELPTOPIC(topic, text) \
{ topic, text },

#define BEGIN_HELPTOPIC(topic) { topic, 

#define END_HELPTOPIC() },

#define END_HELPTOPIC_MAP()		\
	{ NULL, NULL }				\
};

BEGIN_DBGCOMMAND_MAP(g_DBGCommands)
	DBGCOMMAND_ARG_HELP( "help", DBGConsole_HelpCommand, "Displays commands and help topics or shows help text for one of them" ) // Implemented by Lkb
	DBGCOMMAND_ARGLESS_HELP( "go", DBGConsole_GoCommand, "Starts the CPU" )
	DBGCOMMAND_ARGLESS_HELP( "stop", DBGConsole_StopCommand, "Stops the CPU" )
	DBGCOMMAND_ARGLESS_HELP( "skip", CPUSkip, "When the CPU is stopped, skips the current instruction" )
	DBGCOMMAND_ARGLESS_HELP( "dump dlist", RDP_GFX_DumpNextDisplayList, "Dumps the next display list to disk" )
	DBGCOMMAND_ARGLESS_HELP( "dump textures", RDP_GFX_DumpTextures, "Dumps existing/new textures to disk as .png files" )
	DBGCOMMAND_ARGLESS_HELP( "nodump textures", RDP_GFX_NoDumpTextures, "Stops dumping textures" )
	DBGCOMMAND_ARGLESS_HELP( "drop textures", RDP_GFX_DropTextures, "Forces Daedalus to recreate the textures" )
	DBGCOMMAND_ARGLESS_HELP( "make textures blue", RDP_GFX_MakeTexturesBlue, "Makes all the textures in the scene blue" )
	DBGCOMMAND_ARGLESS_HELP( "make textures normal", RDP_GFX_MakeTexturesNormal, "Restores textures after being made blue" )
	DBGCOMMAND_ARGLESS_HELP( "enablegfx", RDP_EnableGfx, "Reenables display list processing" )
	DBGCOMMAND_ARGLESS_HELP( "disablegfx", RDP_DisableGfx, "Turns off Display List processing" )
	
	DBGCOMMAND_ARGLESS_HELP( "stat dr", SR_Stats, "Displays some statistics on the dynamic recompiler" )

	DBGCOMMAND_ARGLESS( "disable si", Memory_SiDisable )
	DBGCOMMAND_ARGLESS( "enable si", Memory_SiEnable )

	DBGCOMMAND_ARGLESS_HELP( "int pi", DBGConsole_IntPi, "Generates a PI interrupt" )
	DBGCOMMAND_ARGLESS_HELP( "int ai", DBGConsole_IntAi, "Generates a AI interrupt" )
	DBGCOMMAND_ARGLESS_HELP( "int dp", DBGConsole_IntDp, "Generates a DP interrupt" )
	DBGCOMMAND_ARGLESS_HELP( "int sp", DBGConsole_IntSp, "Generates a SP interrupt" )
	DBGCOMMAND_ARGLESS_HELP( "int si", DBGConsole_IntSi, "Generates a SI interrupt" )

	DBGCOMMAND_ARG_HELP( "dis", DBGCommand_Dis, "Dumps disassembly of specified region to disk" )
	DBGCOMMAND_ARG_HELP( "rdis", DBGCommand_RDis, "Dumps disassembly of the rsp imem/dmem to disk" )
		
	DBGCOMMAND_ARG_HELP( "listos", DBGCommand_ListOS, "Shows the os symbol in the code view (e.g. __osDisableInt)" )
	DBGCOMMAND_ARG_HELP( "list", DBGCommand_List, "Shows the specified address in the code view" )
	DBGCOMMAND_ARGLESS_HELP( "cpu", DBGCommand_ShowCPU, "Show the CPU code view" )
	DBGCOMMAND_ARGLESS_HELP( "rsp", DBGCommand_ShowRSP, "Show the RSP code view" )
	DBGCOMMAND_ARG_HELP( "fp", DBGCommand_FP, "Displays the specified floating point register" )
	DBGCOMMAND_ARG_HELP( "vec", DBGCommand_Vec, "Dumps the specified rsp vector" )

	DBGCOMMAND_ARG_HELP( "mem", DBGCommand_Mem, "Shows the memory at the specified address" )
	DBGCOMMAND_ARG_HELP( "bpx", DBGCommand_BPX, "Sets a breakpoint at the specified address" )
	DBGCOMMAND_ARG_HELP( "bpd", DBGCommand_BPD, "Disables a breakpoint at the specified address" )
	DBGCOMMAND_ARG_HELP( "bpe", DBGCommand_BPE, "Enables a breakpoint at the specified address" )

	DBGCOMMAND_ARG_HELP( "w8", DBGCommand_Write8, "Writes the specified 8-bit value at the specified address. Type \"help write\" for more info.") // Added and implemented by Lkb
	DBGCOMMAND_ARG_HELP( "w16", DBGCommand_Write16, "Writes the specified 16-bit value at the specified address. Type \"help write\" for more info." ) // Added and implemented by Lkb
	DBGCOMMAND_ARG_HELP( "w32", DBGCommand_Write32, "Writes the specified 32-bit value at the specified address. Type \"help write\" for more info." ) // Added and implemented by Lkb
	DBGCOMMAND_ARG_HELP( "w64", DBGCommand_Write64, "Writes the specified 64-bit value at the specified address. Type \"help write\" for more info." ) // Added and implemented by Lkb
	DBGCOMMAND_ARG_HELP( "wr", DBGCommand_WriteReg, "Writes the specified value in the specified register. Type \"help write\" for more info." ) // Added and implemented by Lkb

	DBGCOMMAND_ARGLESS_HELP( "osinfo threads", Patch_DumpOsThreadInfo, "Displays a list of active threads" )
	DBGCOMMAND_ARGLESS_HELP( "osinfo queues", Patch_DumpOsQueueInfo, "Display a list of queues (some are usually invalid)" )
	DBGCOMMAND_ARGLESS_HELP( "osinfo events", Patch_DumpOsEventInfo, "Display the system event details" )

	DBGCOMMAND_ARGLESS_HELP( "close", DBGCommand_Close, "Closes the debug console") // Added and implemented by Lkb
	DBGCOMMAND_ARGLESS_HELP( "quit", DBGCommand_Quit, "Quits Daedalus") // Added and implemented by Lkb
	DBGCOMMAND_ARGLESS_HELP( "exit", DBGCommand_Quit, "Quits Daedalus") // Added and implemented by Lkb
	
END_DBGCOMMAND_MAP()


BEGIN_HELPTOPIC_MAP(g_DBGHelpTopics)

	BEGIN_HELPTOPIC("numbers")
	"In Daedalus you can specify numbers with the following notations:\n"
	"\tDecimal: 0t12345678\n"
	"\tHexadecimal: 0xabcdef12 or abcdef12\n"
	"\tRegister: %sp\n"
	"\tRegister+-value: %sp+2c\n"
	"\tDon't add spaces when using the reg+-value notation: \"%sp + 4\" is incorrect\n"
	"\tIn this release, \"mem [[sp+4[]\" is replaced by \"mem %sp+4\""
	END_HELPTOPIC()

	BEGIN_HELPTOPIC("write")
	"Syntax:\n"
	"w{8|16|32|64} <address> [[<operators>[] <value>\n"
	"wr <regname> [[<operators>[] <value>\n"
	"\n"
	"<address> - Address to modify\n"
	"<regname> - Register to modify\n"
	"<value> - Value to use\n"
	"<operators> - One or more of the following symbols:\n"
	"\t$ - Bit mode: uses (1 << <value>) instead of <value>\n"
	"\t~ - Not mode: negates <value> before using it\n"
	"\t| - Or operator: combines [[<address>[] with <value> using Or\n"
	"\t^ - Xor operator: combines [[<address>[] with <value> using Xor\n"
	"\t& - And operator: combines [[<address>[] with <value> using And\n"
	"For information on the format of <address> and <value>, type \"help numbers\""
	END_HELPTOPIC()

	BEGIN_HELPTOPIC("rspmem")
	"To read or write form RSP code memory you must replace the first zero with 'a'\n"
	"Example: to write NOP to RSP 0x04001090 use: w32 a4001090 0"
	END_HELPTOPIC()

	BEGIN_HELPTOPIC("dis")
	"Disassemble a region of memory to file\n"
	"Syntax:\n"
	"dis <address1> <address2> [[<filename>[]\n"
	"\n"
	"<address1> - Starting address\n"
	"<address2> - Ending address\n"
	"<filename> - Optional filename (default is dis.txt)\n"
	"Example: Bootrom   : dis 0xB0000000 0xB0001000\n"
	"         ExcVectors: dis 0x80000000 0x80000200"
	END_HELPTOPIC()

	BEGIN_HELPTOPIC("rdis")
	"Disassemble the rsp dmem/imem to file\n"
	"Syntax:\n"
	"dis [[<filename>[]\n"
	"\n"
	"<filename> - Optional filename (default is rdis.txt)"
	END_HELPTOPIC()


END_HELPTOPIC_MAP()

static const DWORD sc_Cop0RegsToShow[] =
{
	C0_INX,
	//C0_RAND,
	C0_ENTRYLO0,
	C0_ENTRYLO1,
	C0_CONTEXT,
	C0_PAGEMASK,
	C0_WIRED,
	C0_BADVADDR,
	C0_COUNT,
	C0_ENTRYHI,
	C0_SR,
	C0_CAUSE,
	C0_EPC,
	//C0_PRID,
	C0_COMPARE,
	C0_CONFIG,
	C0_LLADDR,
	C0_WATCHLO,
	C0_WATCHHI,
	C0_ECC,			// PERR
	C0_CACHE_ERR,
	C0_TAGLO,
	C0_TAGHI,
	C0_ERROR_EPC,
	~0
};

static BOOL s_bConsoleAllocated = FALSE;

void DBGConsole_Enable(BOOL bEnable)
{
	// Shut down
	DBGConsole_Fini();

	if (bEnable)
	{
		g_bShowDebug = DBGConsole_Init();
	}
	else
	{
		g_bShowDebug = FALSE;
	}

}


BOOL DBGConsole_Init()
{
	BOOL bRetVal;

	// Don't re-init!
	if (s_bConsoleAllocated)
		return TRUE;
	
	bRetVal = AllocConsole();
	if (!bRetVal)
		return FALSE;

	s_bConsoleAllocated = TRUE;
 
    // Get a handle to the STDOUT screen buffer to copy from and 
    // create a new screen buffer to copy to. 
 
    g_hStdOut = GetStdHandle(STD_OUTPUT_HANDLE); 
    if (g_hStdOut == INVALID_HANDLE_VALUE) 
        return FALSE;
 
    g_hOutputBuffer = CreateConsoleScreenBuffer( 
       GENERIC_READ |           // read-write access 
       GENERIC_WRITE, 
       0,                       // not shared 
       NULL,                    // no security attributes 
       CONSOLE_TEXTMODE_BUFFER, // must be TEXTMODE 
       NULL);                   // reserved; must be NULL 
    if (g_hOutputBuffer == INVALID_HANDLE_VALUE) 
        return FALSE;

	// Resize to some reasonable size
	if (!SetConsoleScreenBufferSize(g_hOutputBuffer, sc_outbufsize))
		return FALSE;

	DBGConsole_ResizeConBufAndWindow(g_hStdOut, 80,60);

	DBGConsole_SetActivePane(DBGCON_BOT_PANE);

	DBGConsole_DumpRegisters();
	DBGConsole_DumpMemory(~0);
	DBGConsole_DumpContext(~0);


	DWORD dwThreadID;

	g_hConsoleThread = CreateThread(NULL, 0, DBGConsole_ConsoleFunc, NULL, CREATE_SUSPENDED , &dwThreadID);
	if (g_hConsoleThread == NULL)
		return FALSE;

	ResumeThread(g_hConsoleThread);

	return TRUE;
}

void DBGConsole_Fini()
{
	if (g_hConsoleThread != NULL)
	{
		CloseHandle(g_hConsoleThread);
		g_hConsoleThread = NULL;
	}

	// Maybe preserve this - we might want to use the contents later
	if (g_hOutputBuffer != INVALID_HANDLE_VALUE)
	{
		CloseHandle(g_hOutputBuffer);
		g_hOutputBuffer = INVALID_HANDLE_VALUE;
	}

	if (s_bConsoleAllocated)
	{
		FreeConsole();
		s_bConsoleAllocated = FALSE;
	}
}

void DBGConsole_UpdateDisplay()
{
	// Return if we don't have a console!
	if (!g_bShowDebug || !s_bConsoleAllocated)
		return;

	DBGConsole_DumpRegisters();
	DBGConsole_DumpMemory(~0);
	DBGConsole_DumpContext(~0);
}


DWORD __stdcall DBGConsole_ConsoleFunc(LPVOID pArg)
{
    DWORD cNumRead, fdwMode, fdwSaveOldMode, i; 
    INPUT_RECORD irInBuf[128]; 
	COORD curPos;

    HANDLE hInput = GetStdHandle(STD_INPUT_HANDLE); 
    if (hInput == INVALID_HANDLE_VALUE) 
        return 1;


    // Save the current input mode, to be restored on exit. 
    if (!GetConsoleMode(hInput, &fdwSaveOldMode) ) 
        return 1; 
 
    // Enable the window and mouse input events. 
    fdwMode = ENABLE_WINDOW_INPUT | ENABLE_MOUSE_INPUT; 
    if (! SetConsoleMode(hInput, fdwMode) ) 
        return 1; 

	curPos.X = 0;
	curPos.Y = sc_nCommandTextOffset;
	SetConsoleCursorPosition(g_hStdOut, curPos);
 	

	// Handle set to null on exit
	while (g_hConsoleThread != NULL)
	{
		
        if (! ReadConsoleInput( 
                hInput,      // input buffer handle 
                irInBuf,     // buffer to read into 
                128,         // size of read buffer 
                &cNumRead) ) // number of records read 
            break; 
 
        // Dispatch the events to the appropriate handler. 
 
        for (i = 0; i < cNumRead; i++) 
        {
            switch(irInBuf[i].EventType) 
            { 
                case KEY_EVENT: // keyboard input 
					
                    DBGConsole_ProcessKeyEvent(irInBuf[i].Event.KeyEvent); 

                    break; 
 
               /* case MOUSE_EVENT: // mouse input 
                    MouseEventProc(irInBuf[i].Event.MouseEvent); 
                    break; */
 
                /*case WINDOW_BUFFER_SIZE_EVENT: // scrn buf. resizing 
                    ResizeEventProc( 
                        irInBuf[i].Event.WindowBufferSizeEvent); 
                    break; */
 
                case FOCUS_EVENT:  // disregard focus events 
 
                case MENU_EVENT:   // disregard menu events 
                    break; 
 
                default: 
                    break; 
            } 
		}

	}

	return 0;
}
	


BOOL DBGConsole_ResizeConBufAndWindow(HANDLE hConsole, SHORT xSize, SHORT ySize)
{
	CONSOLE_SCREEN_BUFFER_INFO csbi; /* hold current console buffer info */
	BOOL bSuccess;
	SMALL_RECT srWindowRect; /* hold the new console size */
	COORD coordScreen;

	bSuccess = GetConsoleScreenBufferInfo(hConsole, &csbi);
	if (!bSuccess)
		return FALSE;

	/* get the largest size we can size the console window to */

	coordScreen = GetLargestConsoleWindowSize(hConsole); 
	if (!bSuccess)
		return FALSE;

	/* define the new console window size and scroll position */
	srWindowRect.Right = (SHORT) (min(xSize, coordScreen.X) - 1);
	srWindowRect.Bottom = (SHORT) (min(ySize, coordScreen.Y) - 1);

	srWindowRect.Left = srWindowRect.Top = (SHORT) 0;

	/* define the new console buffer size */
	coordScreen.X = xSize;
	coordScreen.Y = ySize;

	/* if the current buffer is larger than what we want, resize the */
	/* console window first, then the buffer */
	if ((DWORD) csbi.dwSize.X * csbi.dwSize.Y > (DWORD) xSize * ySize)
	{
		bSuccess = SetConsoleWindowInfo(hConsole, TRUE, &srWindowRect);
		if (!bSuccess)
			return FALSE;

		bSuccess = SetConsoleScreenBufferSize(hConsole, coordScreen);
		if (!bSuccess)
			return FALSE;
	}

	/* if the current buffer is smaller than what we want, resize the */
	/* buffer first, then the console window */
	if ((DWORD) csbi.dwSize.X * csbi.dwSize.Y < (DWORD) xSize * ySize) 
	{
		bSuccess = SetConsoleScreenBufferSize(hConsole, coordScreen);
		if (!bSuccess)
			return FALSE;

		bSuccess = SetConsoleWindowInfo(hConsole, TRUE, &srWindowRect);
		if (!bSuccess)
			return FALSE;
	}

	/* if the current buffer *is* the size we want, don't do anything! */

	return TRUE;
} 

void DBGConsole_DumpRegisters()
{
	if (s_nMidPaneType == DBGCON_RSP_CODE_VIEW)
		DBGConsole_DumpRSPRegisters();
	else
		DBGConsole_DumpCPURegisters();

}

void DBGConsole_ClearRegisters()
{
	BOOL bSuccess;
    COORD coord; 
	DWORD cWritten;

	for (coord.Y = 0; coord.Y < 21; coord.Y++)
	{
		coord.X = 0;            // start at first cell 

		// Write a string of colors to a screen buffer. 
		bSuccess = FillConsoleOutputAttribute(
			g_hStdOut,
			sc_wAttrWhite,
			80,		// Clear to end
			coord,
			&cWritten);

		bSuccess = FillConsoleOutputCharacter(g_hStdOut,
			' ', 80, coord, &cWritten);


	}


}

void DBGConsole_DumpCPURegisters()
{
	char msg[200];

	DWORD base;
	DWORD dwReg;
	SHORT wX;
	SHORT wY;
	static BOOL bInit = FALSE;
	static u64 qwGPRCopy[32];

	static const DWORD dwRegDisplayWidth = 11;

	// Make sure that we render all registers - set all regs to 
	if (bInit == FALSE)
	{
		for (dwReg = 0; dwReg < 32; dwReg++)
		{
			qwGPRCopy[dwReg] = g_qwGPR[dwReg];
		}
		bInit = TRUE;
	}

	wX = 0;
	wY = sc_nTopPaneTextOffset+1;

	sprintf(msg, "PC:%08x", g_dwPC);
	DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrWhite, 19); wY ++;
	sprintf(msg, "MH:%016I64x", g_qwMultHI);
	DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrWhite, 19); wY ++;
	sprintf(msg, "ML:%016I64x", g_qwMultLO);
	DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrWhite, 19); wY ++;

	// Do main registers
	for (base = 0; base < 32; base += 8)
	{
		wY = sc_nTopPaneTextOffset+1+3;
		for (dwReg = base; dwReg < base+8; dwReg++)
		{
			// Not wsprintf as we want long int output
			wsprintf(msg, "%s:%08x", RegNames[dwReg], (DWORD)g_qwGPR[dwReg]);
			
			// Only draw if it's changed
			if (qwGPRCopy[dwReg] != g_qwGPR[dwReg])
			{
				DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrBoldRed, dwRegDisplayWidth);
				qwGPRCopy[dwReg] = g_qwGPR[dwReg];
			}
			else
			{
				DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrWhite, dwRegDisplayWidth);
			}
			wY++;
		}
		wX += dwRegDisplayWidth + 1;
	}
	
	// Do CoPro1 regs
	wY = sc_nTopPaneTextOffset+1;
	DWORD i;
	for (i = 0; ; i++)
	{
		dwReg = sc_Cop0RegsToShow[i];
		if (dwReg == ~0)
			break;

		sprintf(msg, "%4.4s:%08x", ShortCop0RegNames[dwReg], (DWORD)g_qwCPR[0][dwReg]);
		DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrWhite, 13);
		wY ++;
		if (i == 10)
		{
			wX += 15;
			wY = sc_nTopPaneTextOffset+1;
		}
	}


}


void DBGConsole_DumpRSPRegisters()
{
	char msg[200];

	DWORD dwReg;
	DWORD base;
	WORD wX = 0;
	WORD wY = 0;
	static BOOL bInit = FALSE;
	static DWORD dwRSPGPRCopy[32];

	static const DWORD dwRegDisplayWidth = 15;


	if (bInit == FALSE)
	{
		// Make sure that we render all registers
		for (dwReg = 0; dwReg < 32; dwReg++) {
			dwRSPGPRCopy[dwReg] = g_dwRSPGPR[dwReg];
		}
		bInit = TRUE;
	}

	wY = sc_nTopPaneTextOffset+1;
	wsprintf(msg, "PC: 0x%08x", RSPPC());
	DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrWhite, dwRegDisplayWidth+8); wY++;

	// Do main registers
	for (base = 0; base < 32; base += 8)
	{
		wY = sc_nTopPaneTextOffset+1+3;
		for (dwReg = base; dwReg < base+8; dwReg++) {
			wsprintf(msg, "%s: 0x%08x", RegNames[dwReg], g_dwRSPGPR[dwReg]);

			// Only draw if changed
			if (dwRSPGPRCopy[dwReg] != g_dwRSPGPR[dwReg])
			{
				DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrBoldRed, 20);
				dwRSPGPRCopy[dwReg] = g_dwRSPGPR[dwReg];
			}
			else
			{
				DBGConsole_WriteString(msg, FALSE, wX, wY, sc_wAttrWhite, 20);
			}
			wY++;
		}
		wX += dwRegDisplayWidth + 1;
	}
	
}


void DBGConsole_SetActivePane(DBGCON_PANETYPE pane)
{
	
	s_nActivePane = pane;

	DBGConsole_RedrawPaneLabels();

}

void DBGConsole_RedrawPaneLabels()
{
	int i;
	CHAR szLine[80+1];
	CHAR szPaneName[80+1];
	CHAR * s, * d;
	static const CHAR cLine = (char)196;
	static const CHAR cDownArrow = 31;		

	// Top pane
	lstrcpyn(szPaneName, "Registers", 80);
	for (i = 0; i < 80; i++)
		szLine[i] = cLine;

	if (s_nActivePane == DBGCON_REG_PANE)
	{
		szLine[1] = szLine[2] = szLine[3] = szLine[4] = cDownArrow;
	}
	szLine[5] = 'F'; szLine[6] = '1';
	for (s = szPaneName, d = &szLine[8]; s[0] != 0; s++,d++)
		d[0] = s[0];
	szLine[80] = 0;
	DBGConsole_WriteString(szLine, FALSE, 0, sc_nTopPaneTextOffset, BACKGROUND_GREEN, 80);

	// Mem pane
	lstrcpyn(szPaneName, "Memory", 80);
	for (i = 0; i < 80; i++)
		szLine[i] = cLine;

	if (s_nActivePane == DBGCON_MEM_PANE)
	{
		szLine[1] = szLine[2] = szLine[3] = szLine[4] = cDownArrow;
	}
	szLine[5] = 'F'; szLine[6] = '2';
	for (s = szPaneName, d = &szLine[8]; s[0] != 0; s++,d++)
		d[0] = s[0];
	szLine[80] = 0;
	DBGConsole_WriteString(szLine, FALSE, 0, sc_nMemPaneTextOffset, BACKGROUND_GREEN, 80);


	// Mid pane
	wsprintf(szPaneName, "%s", s_szPaneNameLabels[s_nMidPaneType]);
	for (i = 0; i < 80; i++)
		szLine[i] = cLine;

	if (s_nActivePane == DBGCON_MID_PANE)
	{
		szLine[1] = szLine[2] = szLine[3] = szLine[4] = cDownArrow;
	}
	szLine[5] = 'F'; szLine[6] = '3';
	for (s = szPaneName, d = &szLine[8]; s[0] != 0; s++,d++)
		d[0] = s[0];
	szLine[80] = 0;
	DBGConsole_WriteString(szLine, FALSE, 0, sc_nMidPaneTextOffset, BACKGROUND_GREEN, 80);


	// Bottom pane:
	lstrcpyn(szPaneName, "Output", 80);
	for (i = 0; i < 80; i++)
		szLine[i] = cLine;

	if (s_nActivePane == DBGCON_BOT_PANE)
	{
		szLine[1] = szLine[2] = szLine[3] = szLine[4] = cDownArrow;
	}
	szLine[5] = 'F'; szLine[6] = '4';
	for (s = szPaneName, d = &szLine[8]; s[0] != 0; s++,d++)
		d[0] = s[0];
	szLine[80] = 0;
	DBGConsole_WriteString(szLine, FALSE, 0, sc_nBotPaneTextOffset, BACKGROUND_GREEN, 80);

}

void DBGConsole_DumpContext(DWORD dwPC)
{
	switch (s_nMidPaneType)
	{
	case DBGCON_CPU_CODE_VIEW:
		DBGConsole_DumpCPUContext(dwPC);
		break;
	case DBGCON_RSP_CODE_VIEW:
		DBGConsole_DumpRSPContext(dwPC);
		break;
	}

}

void DBGConsole_DumpContext_PrevLine()
{
	switch (s_nActivePane)
	{
	case DBGCON_MEM_PANE:
		DBGConsole_DumpMemory(g_dwMemAddress - 4*4);
		break;
	case DBGCON_MID_PANE:
		switch (s_nMidPaneType)
		{
		case DBGCON_CPU_CODE_VIEW:
			DBGConsole_DumpCPUContext(g_dwDisplayPC - 4);
			break;
		case DBGCON_RSP_CODE_VIEW:
			DBGConsole_DumpRSPContext(g_dwDisplayRSP_PC - 4);
			break;

		}
		break;
	case DBGCON_BOT_PANE:
		DBGConsole_DisplayOutput(g_nOutputBufOffset - 1);
		break;
	}
}

void DBGConsole_DumpContext_PrevPage()
{
	switch (s_nActivePane)
	{
	case DBGCON_MEM_PANE:
		DBGConsole_DumpMemory(g_dwMemAddress - 4*4*sc_nMidPaneLines);
		break;
	case DBGCON_MID_PANE:
		switch (s_nMidPaneType)
		{
		case DBGCON_CPU_CODE_VIEW:
			DBGConsole_DumpCPUContext(g_dwDisplayPC - 4*sc_nMidPaneLines);
			break;
		case DBGCON_RSP_CODE_VIEW:
			DBGConsole_DumpRSPContext(g_dwDisplayRSP_PC - 4*sc_nMidPaneLines);
			break;
		}
		break;
	case DBGCON_BOT_PANE:
		DBGConsole_DisplayOutput(g_nOutputBufOffset - sc_nBotPaneLines);
		break;
	}
}

void DBGConsole_DumpContext_NextLine()
{
	switch (s_nActivePane)
	{
	case DBGCON_MEM_PANE:
		DBGConsole_DumpMemory(g_dwMemAddress + 4*4);
		break;
	case DBGCON_MID_PANE:
		switch (s_nMidPaneType)
		{
		case DBGCON_CPU_CODE_VIEW:
			DBGConsole_DumpCPUContext(g_dwDisplayPC + 4);
			break;
		case DBGCON_RSP_CODE_VIEW:
			DBGConsole_DumpRSPContext(g_dwDisplayRSP_PC + 4);
			break;
		}
		break;
	case DBGCON_BOT_PANE:
		DBGConsole_DisplayOutput(g_nOutputBufOffset + 1);
		break;
	}

}

void DBGConsole_DumpContext_NextPage()
{
	switch (s_nActivePane)
	{
	case DBGCON_MEM_PANE:
		DBGConsole_DumpMemory(g_dwMemAddress + 4*4*sc_nMidPaneLines);
		break;
	case DBGCON_MID_PANE:
		switch (s_nMidPaneType)
		{
		case DBGCON_CPU_CODE_VIEW:
			DBGConsole_DumpCPUContext(g_dwDisplayPC + 4*sc_nMidPaneLines);
			break;
		case DBGCON_RSP_CODE_VIEW:
			DBGConsole_DumpRSPContext(g_dwDisplayRSP_PC + 4*sc_nMidPaneLines);
			break;
		}
		break;
	case DBGCON_BOT_PANE:
		DBGConsole_DisplayOutput(g_nOutputBufOffset + sc_nBotPaneLines);
		break;
	}
}

// Pass in ~0 to display current mem pointer
void DBGConsole_DumpMemory(DWORD dwAddress)
{
	WORD wY;
	char buf[120];

	if (dwAddress == ~0)
		dwAddress = g_dwMemAddress;

	g_dwMemAddress = dwAddress;

	// Clear the background area...

	wY = sc_nMemPaneTextOffset+1;
	for (DWORD dwIndex = 0; dwIndex < sc_nMemPaneLines; dwIndex++)
	{
		DWORD * pMem;
		DWORD dwBase = InternalReadAddress(dwAddress, (LPVOID*)&pMem);

		// TODO - Lookup a base for each of pMem[0], pMem[1] etc - this
		// is significant if we cross a boundary, but not *really* important

		if (dwBase != MEM_UNUSED)
		{
			wsprintf(buf, "    0x%08x: %08x %08x %08x %08x ", dwAddress, pMem[0], pMem[1], pMem[2], pMem[3]);					

			char * p = buf + lstrlen(buf);
			// Concat characters
			for (int i = 0; i < 16; i++)
			{
				char c = ((BYTE*)pMem)[i ^ 0x3];
				// The cast to unsigned int avoids widening char to a signed int.
				// This cast avoids a failed assertion popup in MSVC 7.0
				if (isprint((unsigned int)(unsigned char)c))
					*p++ = c;
				else 
					*p++ = '.';

				if ((i%4)==3)
					*p++ = ' ';
			}
			*p++ = 0;
		}
		else
		{
			wsprintf(buf, "    0x%08x: ???????? ???????? ???????? ???????? .... .... .... ....", dwAddress);
		}

		DBGConsole_WriteString(buf, FALSE, 0, wY, sc_wAttrWhite, 80);
		wY++;

		dwAddress+=4*4;
	}
}



// Pass in ~0 to display current PC
void DBGConsole_DumpCPUContext(DWORD dwPC)
{
	WORD wY;
	DWORD * pPC;
	char opinfo[120];
	char msg[200];
	DWORD dwBase;
	WORD wAttr;

	if (dwPC == ~0)
		dwPC = g_dwPC;

	g_dwDisplayPC = dwPC;

	dwPC -= (4*(sc_nMidPaneLines/2));	// Show previous instructions

	// Clear the background area...

	wY = sc_nMidPaneTextOffset+1;
	for (DWORD dwIndex = 0; dwIndex < sc_nMidPaneLines; dwIndex++)
	{
		dwBase = InternalReadAddress(dwPC, (LPVOID*)&pPC);

		if (dwPC == g_dwPC) 
			wAttr = sc_wAttrBoldWhite;
		else
			wAttr = sc_wAttrWhite;

		if (dwBase != MEM_UNUSED)
		{
			DWORD dwOpCode = pPC[0];
			LPCSTR pszSymbol;

			SprintOpCodeInfo(opinfo, dwPC, dwOpCode);


			// Additional info for our hacks:
			if (op(dwOpCode) == OP_DBG_BKPT)
			{
				DWORD dwBreakPoint = dwOpCode & 0x03FFFFFF;

				if (dwBreakPoint < g_BreakPoints.size())
				{
					if (g_BreakPoints[dwBreakPoint].bEnabled)
						wsprintf(msg, "[R!]   0x%08x: %s", dwPC, opinfo);
					else
						wsprintf(msg, "[G!]   0x%08x: %s", dwPC, opinfo);
				}
				else
				{
					wsprintf(msg, "    0x%08x: %s", dwPC, opinfo);
				}
			}
			else if (op(dwOpCode) == OP_SRHACK_UNOPT)
			{

				DWORD dwEntry = dwOpCode & 0x03FFFFFF;

				if (dwEntry < g_dwNumStaticEntries)
				{
					CDynarecCode * pCode = g_pDynarecCodeTable[dwEntry];

					wsprintf(msg, "u%2d 0x%08x: %s", pCode->dwNumOps, dwPC, opinfo);
				}
				else
				{
					wsprintf(msg, "    0x%08x: %s", dwPC, opinfo);
				}
			}
			else if (op(dwOpCode) == OP_SRHACK_OPT)
			{

				DWORD dwEntry = dwOpCode & 0x03FFFFFF;

				if (dwEntry < g_dwNumStaticEntries)
				{
					CDynarecCode * pCode = g_pDynarecCodeTable[dwEntry];

					wsprintf(msg, "o%2d 0x%08x: %s", pCode->dwNumOps, dwPC, opinfo);
				}
				else
				{
					wsprintf(msg, "    0x%08x: %s", dwPC, opinfo);
				}
			}
			else if (op(dwOpCode) == OP_SRHACK_NOOPT)
			{

				DWORD dwEntry = dwOpCode & 0x03FFFFFF;

				if (dwEntry < g_dwNumStaticEntries)
				{
					wsprintf(msg, "n   0x%08x: %s", dwPC, opinfo);
				}
				else
				{
					wsprintf(msg, "    0x%08x: %s", dwPC, opinfo);
				}
			}
			else
			{
				wsprintf(msg, "    0x%08x: %s", dwPC, opinfo);
			}
						
			//if (op(dwOpCode) == OP_PATCH)
			{
				pszSymbol = Patch_GetJumpAddressName(dwPC);
				if (pszSymbol[0] != '?')
				{
					// This is actually a patch target
					DBGConsole_WriteString("", FALSE, 0, wY, wAttr, 80);
					wY++;
					dwIndex++;

					if (dwIndex < sc_nMidPaneLines)
					{
						TCHAR szFuncInfo[30];
						wsprintf(szFuncInfo, "[G%s]", pszSymbol);
						DBGConsole_WriteString(szFuncInfo, TRUE, 0, wY, wAttr, 80);
						wY++;
						dwIndex++;
					}
				}
			}
		}
		else
		{
			wsprintf(msg, "    0x%08x: ???", dwPC);
		}


		if (dwIndex < sc_nMidPaneLines)
		{
			DBGConsole_WriteString(msg, TRUE, 0, wY, wAttr, 80);
			wY++;
		}

		dwPC+=4;
	}
}

void DBGConsole_DumpRSPContext(DWORD dwPC)
{
	char opinfo[256];
	char msg[200];
	WORD wY;

	DWORD dwRSPPC = (RSPPC() & 0x0FFF) | 0x04001000;

	if (dwPC == ~0)
		dwPC = dwRSPPC;

	g_dwDisplayRSP_PC = dwPC;

	dwPC -= (4*(sc_nMidPaneLines/2));	// Show previous 4 instructions

	// Clear the background area...

	wY = sc_nMidPaneTextOffset+1;
	for (DWORD dwIndex = 0; dwIndex < sc_nMidPaneLines; dwIndex++)
	{
		if (dwPC >= 0x04002000)
			break;

		//- TODO - Translate to InternalRead
		DWORD dwOpCode = Read32Bits(PHYS_TO_K1(dwPC));

		SprintRSPOpCodeInfo(opinfo, dwPC, dwOpCode);
		wsprintf(msg, "    0x%08x: %s", dwPC, opinfo);

		if ((dwPC&0x1FFF) == (dwRSPPC&0x1FFF))
		{
			DBGConsole_WriteString(msg, FALSE, 0, wY, sc_wAttrBoldWhite, 80);
		} else {
			DBGConsole_WriteString(msg, FALSE, 0, wY, sc_wAttrWhite, 80);
		}
		wY++;
		dwPC += 4;
		
	}
}



// Write a string of characters to a screen buffer. 
BOOL DBGConsole_WriteString(LPCTSTR szString, BOOL bParse, SHORT x, SHORT y, WORD wAttr, SHORT width)
{
    DWORD cWritten; 
    BOOL bSuccess; 
    COORD coord; 
	WORD wNumToWrite;
	TCHAR szBuffer[2048+1];
	WORD arrwAttributes[2048];

    coord.X = x;            // start at first cell 
    coord.Y = y;            //   of first row      

	lstrcpyn(szBuffer, szString, 2048);

	if (bParse)
		DBGConsole_ParseStringHighlights(szBuffer, arrwAttributes, wAttr);

	wNumToWrite = lstrlen(szBuffer);
	if (wNumToWrite > width)
		wNumToWrite = width;

    bSuccess = WriteConsoleOutputCharacter( 
        g_hStdOut,              // screen buffer handle 
        szBuffer,				// pointer to source string 
        wNumToWrite,			// length of string 
        coord,					// first cell to write to 
        &cWritten);				// actual number written 
    if (!bSuccess) 
        return FALSE; 
 
	// Copy the visible portion to the display
	if (bParse)
	{
		/*bSuccess = */WriteConsoleOutputAttribute(
			g_hStdOut,
			arrwAttributes,
			wNumToWrite,
			coord,
			&cWritten);
	}
	else
	{
		// Just clear
		/*bSuccess = */FillConsoleOutputAttribute(
			g_hStdOut,
			wAttr,
			wNumToWrite,		// Clear to end
			coord,
			&cWritten);
	}

	// Clear the end of the buffer
	if (coord.X + wNumToWrite < width)
	{
		coord.X += wNumToWrite;
		/*bSuccess = */FillConsoleOutputAttribute(
			g_hStdOut,
			wAttr,
			width - wNumToWrite,		// Clear to end
			coord,
			&cWritten);

		/*bSuccess = */FillConsoleOutputCharacter(g_hStdOut,
			' ', width - coord.X, coord, &cWritten);
	}
	
	return TRUE;
}





void DBGConsole_ProcessKeyEvent(KEY_EVENT_RECORD kr)
{
	TCHAR szInputChar[1+1];
	COORD curPos;
	DWORD dwInputLen;

	dwInputLen = lstrlen(g_szInput);

	if (kr.bKeyDown)
	{
		switch (kr.wVirtualKeyCode)
		{
		case VK_RETURN:
			// Process the current input
			if (lstrlen(g_szInput) > 0)
			{
				DBGConsole_ProcessInput();

				// Reset the input buffer
				lstrcpy(g_szInput, "");

				DBGConsole_WriteString(g_szInput, FALSE, 0, sc_nCommandTextOffset, BACKGROUND_GREEN | BACKGROUND_BLUE, 80);
			}
			break;
		case VK_BACK:
			if (dwInputLen > 0)
			{
				g_szInput[dwInputLen-1] = '\0';
			}
			DBGConsole_WriteString(g_szInput, FALSE, 0, sc_nCommandTextOffset, BACKGROUND_GREEN | BACKGROUND_BLUE, 80);
			break;
		case VK_PRIOR:
			DBGConsole_DumpContext_PrevPage();
			break;
		case VK_UP:
			DBGConsole_DumpContext_PrevLine();
			break;
		case VK_HOME:
		/*	if (s_bShowCPU)
				DBGConsole_DumpCPUContext(~0);
			else
				DBGConsole_DumpRSPContext(~0);*/
			break;
		case VK_DOWN:
			DBGConsole_DumpContext_NextLine();
			break;
		case VK_NEXT:
			DBGConsole_DumpContext_NextPage();
			break;

		case VK_F1:
			DBGConsole_SetActivePane(DBGCON_REG_PANE);
			break;
		case VK_F2:
			DBGConsole_SetActivePane(DBGCON_MEM_PANE);
			break;
		case VK_F3:
			DBGConsole_SetActivePane(DBGCON_MID_PANE);
			break;
		case VK_F4:
			DBGConsole_SetActivePane(DBGCON_BOT_PANE);
			break;

		// Step into
		case VK_F10:
			CPUStep();
			DBGConsole_UpdateDisplay();
			break;


		default:
			if (kr.dwControlKeyState & LEFT_CTRL_PRESSED ||
				kr.dwControlKeyState & RIGHT_CTRL_PRESSED)
			{
				if (kr.wVirtualKeyCode == 'D' ||
					kr.wVirtualKeyCode == 'd')		// Not sure if lowercase is necessary
				{
					DBGConsole_Enable(!g_bShowDebug);
				}
			}

			if(kr.uChar.AsciiChar >= 32)
			{
			
				wsprintf(szInputChar, "%c", kr.uChar.AsciiChar);
				lstrcat(g_szInput, szInputChar);

				DBGConsole_WriteString(g_szInput, FALSE, 0, sc_nCommandTextOffset, BACKGROUND_GREEN | BACKGROUND_BLUE, 80);
			}

			break;
		}
	}

	curPos.X = lstrlen(g_szInput);
	curPos.Y = sc_nCommandTextOffset;
						
	SetConsoleCursorPosition(g_hStdOut, curPos);

}



void DBGConsole_ProcessInput()
{
	DWORD i;
	BOOL bProcessed;

	bProcessed = FALSE;

	for (i = 0; ; i++)
	{
		if (g_DBGCommands[i].szCommand == NULL)
			break;

		DWORD dwCmdLen = lstrlen(g_DBGCommands[i].szCommand);
		DWORD dwInputLen = lstrlen(g_szInput);

		if (strnicmp(g_DBGCommands[i].szCommand, g_szInput, dwCmdLen) == 0)
		{
			if (g_DBGCommands[i].pArglessFunc != NULL)
				g_DBGCommands[i].pArglessFunc();
			else if (g_DBGCommands[i].pArgFunc != NULL)
			{
				// Trim spaces...
				LPTSTR szArgs = g_szInput + dwCmdLen;
				while (*szArgs == ' ')
					szArgs++;

				g_DBGCommands[i].pArgFunc(szArgs);
			}
			bProcessed = TRUE;
			break;
		}
	}

	if (bProcessed == FALSE)
		DBGConsole_Msg(0, "[YUnknown command: %s]", g_szInput);
}

typedef struct tagMiscRegInfo
{
	LPCTSTR szName;
	void* pReg;
	DWORD nBits;
} MiscRegInfo;

#define BEGIN_MISCREG_MAP(name) \
static const MiscRegInfo name[] =	\
{
#define MISCREG(name, reg, bits) \
{ name, &(reg), bits },

#define END_MISCREG_MAP()		\
	{ NULL, NULL }					\
};

BEGIN_MISCREG_MAP(g_MiscRegNames)
	MISCREG("pc", g_dwPC, 32)
	MISCREG("hi", g_qwMultHI, 64)
	MISCREG("lo", g_qwMultLO, 64)
	MISCREG("fcr0", g_qwFCR0, 64)
	MISCREG("fcr1", g_qwFCR1, 64)
END_MISCREG_MAP()

// returns size of the register in bits. 0 = register not found
void* DBGCommandUtil_GetAddressOfRegister(LPCTSTR szRegName, DWORD* pdwSize = NULL)
{
	void* pReg = NULL;
	DWORD dwSize = 0;

	int i;

	for (i = 0; ; i++)
	{
		if (g_MiscRegNames[i].szName == NULL)
			break;

		if (stricmp(g_MiscRegNames[i].szName, szRegName) == 0)
		{
			dwSize = g_MiscRegNames[i].nBits;
			pReg = g_MiscRegNames[i].pReg;
		}
	}
	if(pReg == NULL)
	{
		for (i = 0; i < 32; i++)
		{
			if (lstrcmpi(szRegName, RegNames[i]) == 0)
			{
				pReg = &g_qwGPR[i];
				dwSize = 64;
				break;
			}
			else if (lstrcmpi(szRegName, Cop0RegNames[i]) == 0)
			{
				pReg = &g_qwCPR[0][i];
				dwSize = 64;
				break;
			}
			else if (lstrcmpi(szRegName, ShortCop0RegNames[i]) == 0)
			{
				pReg = &g_qwCPR[0][i];
				dwSize = 64;
				break;
			}
		}
	}	
	if(pReg == NULL)
	{
		DBGConsole_Msg(0, "[YInvalid register name in numeric argument! Type \"help numbers\" for more info]");
		//DBGConsole_Msg(0, "[Yreg %s not found]", szRegName);
		return NULL;
	}
	else
	{
		if(pdwSize != NULL)
		{
			*pdwSize = dwSize;
		}
		return pReg;
	}
}

ULONGLONG DBGCommandUtil_ReadRegister(void* pReg, DWORD regSize = 0)
{
	if(pReg == NULL)
		return 0;

	if(regSize == 0)
	{
		regSize = (pReg == &g_dwPC) ? 32 : 64;
	}
	//ULONGLONG regValue = *(ULONGLONG*)pReg;
	/*switch(regSize)
	{
		case 8:
			return regValue & (u64)~((u64)(1 << 8));
		case 16:
			return regValue & (u64)~((u64)(1 << 16));
		case 32:
			return regValue & (u64)~((u64)(1 << 32));
		default:
			return regValue;
	}*/

	// STRMNRMN - bugfix - these were & (u64)~((u64)1<<n), which masked with ..111101111111 etc
	// It's easier just to read the correct size to start with?
	switch(regSize)
	{
		case 8:
			return *(BYTE*)pReg;
		case 16:
			return *(WORD*)pReg;
		case 32:
			return *(DWORD*)pReg;
		default:
			return *(ULONGLONG*)pReg;
	}

}

void DBGCommandUtil_WriteRegister(void* pReg, ULONGLONG nValue, DWORD regSize = 0)
{
	if(pReg == NULL)
		return;

	if(regSize == 0)
	{
		regSize = (pReg == &g_dwPC) ? 32 : 64;
	}
	ULONGLONG pcTemp = 0;
	if(pReg == &g_dwPC)
	{
		pReg = &pcTemp;
	}
	ULONGLONG regValue = *(ULONGLONG*)pReg;
	switch(regSize)
	{
		case 8:
			*(u8*)pReg = (u8)nValue;
			break;
		case 16:
			*(u16*)pReg = (u16)nValue;
			break;
		case 32:
			*(u32*)pReg = (u32)nValue;
			break;
		default:
			*(u64*)pReg = nValue;
			break;
	}
	if(pReg == &pcTemp)
	{
		CPU_SetPC((DWORD)pcTemp);
	}
}

// if bHex is TRUE non-0x values are interpreted as hex, else they are interpreted as decimal
// 0xaaaaaaa -> Hex
// 0t3465767 -> Dec (t -> ten :) )

// The fact the some commands use decimal and others hex as the default may lead to confusion
// However I'm not sure on what the best solution is

LPCTSTR DBGCommandUtil_SimpleGetValue64(LPCTSTR szCommand, ULONGLONG* pValue)
{
	// PRB: What happens if %i is specified and the number is > 2^31? I hope it's read as an unsigned integer

	int n = -1;
	n = sscanf(szCommand, "0x%I64x", pValue);
	if (n != 1)
	{
		n = sscanf(szCommand, "0t%I64i", pValue);
	}
	if (n != 1)
	{
		// Check for absence of 0x or 0t prefixes - default is hex!
		//if(bHex)
		//{
			n = sscanf(szCommand, "%I64x", pValue);
		//}
		//else
		//{
		//	n = sscanf(szCommand, "%I64i", pValue);
		//}
	}
	if (n != 1)
	{
		// TODO: implement "help numbers"
		DBGConsole_Msg(0, "[YInvalid numeric argument! Type \"help numbers\" for more info]");
		return NULL;
	}

	LPCTSTR pszWhiteSpaceAfterScannedValue = strchr(szCommand, ' ');
	if(pszWhiteSpaceAfterScannedValue != NULL)
	{
		return pszWhiteSpaceAfterScannedValue;
	}
	else
	{
		return szCommand + strlen(szCommand);
	}
}

// Converts the first value in the string into a number, returns the
// character just after the string
LPCTSTR DBGCommandUtil_GetValue64(LPCTSTR szCommand, ULONGLONG* pValue)
{
	if(szCommand == NULL)
		return NULL;

	while(*szCommand == ' ') szCommand++;

	LPCTSTR pszCharAfterValue = szCommand + strlen(szCommand);

	ULONGLONG qwAddress = 0;
	
	// Check if this is a register, e.g. "mem [sp+30]"
	if (szCommand[0] == '%')
	{
		TCHAR szRegName[100];

		LPCTSTR temp_pszRegisterNameEnd = strpbrk(szCommand + 1, "+- ");
		if(temp_pszRegisterNameEnd != NULL)
			pszCharAfterValue = temp_pszRegisterNameEnd;

		DWORD dwRegNameLength = pszCharAfterValue - szCommand - 1;
		if(dwRegNameLength == 0)
		{
			DBGConsole_Msg(0, "[YInvalid register name in numeric argument! Type \"help numbers\" for more info]");
			return NULL;
		}

		memcpy(szRegName, szCommand + 1, dwRegNameLength);
		szRegName[dwRegNameLength] = 0;

		DWORD regSize;
		void* pReg = DBGCommandUtil_GetAddressOfRegister(szRegName, &regSize);
		if(pReg != NULL)
		{
			qwAddress = DBGCommandUtil_ReadRegister(pReg, regSize);
		}

		if((*pszCharAfterValue != 0) && (strpbrk(pszCharAfterValue, "+-") != NULL))
		{
			ULONGLONG qwOffset;
			if((pszCharAfterValue = DBGCommandUtil_SimpleGetValue64(pszCharAfterValue, &qwOffset)) == NULL)
				return NULL;
			qwAddress += qwOffset;
		}
	}
	else
	{
		if((pszCharAfterValue = DBGCommandUtil_SimpleGetValue64(szCommand, &qwAddress)) == NULL)
			return NULL;
	}

	*pValue = qwAddress;

	return pszCharAfterValue;
}

LPCTSTR DBGCommandUtil_GetValue32(LPCTSTR szCommand, DWORD* pValue)
{
	ULONGLONG temp64;
	LPCTSTR retVal = DBGCommandUtil_GetValue64(szCommand, &temp64);
	*pValue = (DWORD)temp64;
	return retVal;
}

void DBGCommand_FP(LPCTSTR szCommand)
{
	LONG n;
	DWORD dwReg;

	// We need the decimal value
	n = sscanf(szCommand, "%d", &dwReg);
	if (n == 1)
	{
		// TODO show long and double values
		DBGConsole_Msg(0, "FP%02d", dwReg);
		DBGConsole_Msg(0, "w: 0x%08x = %d", (u32)g_qwCPR[1][dwReg], (u32)g_qwCPR[1][dwReg]);
		DBGConsole_Msg(0, "f: %f", ToFloat(g_qwCPR[1][dwReg]));
	}

}

void DBGCommand_Vec(LPCTSTR szCommand)
{
	LONG n;
	DWORD dwReg1, dwReg2, dwReg3;

	// We need the decimal value
	n = sscanf(szCommand, "%d %d %d", &dwReg1, &dwReg2, &dwReg3);
	if (n == 1)
	{
		RSP_DumpVector(dwReg1);
	}
	else if (n == 3)
	{
		RSP_DumpVectors(dwReg1, dwReg2, dwReg3);
	}
}

void DBGCommand_Mem(LPCTSTR szCommand)
{
	DWORD dwAddress;
	if(DBGCommandUtil_GetValue32(szCommand, &dwAddress) != NULL)
	{
		DBGConsole_DumpMemory(dwAddress);
	}
}

void WriteBits(DWORD dwAddress, ULONGLONG nValue, int nBits)
{
	switch(nBits)
	{
		case 8:
			Write8Bits(dwAddress, (u8)nValue);
			break;
		case 16:
			Write16Bits(dwAddress, (u16)nValue);
			break;
		case 32:
			Write32Bits(dwAddress, (u32)nValue);
			break;
		case 64:
			Write64Bits(dwAddress, (u64)nValue);
			break;
	}
}

ULONGLONG ReadBits(DWORD dwAddress, int nBits)
{
	switch(nBits)
	{
		case 8:
			return (ULONGLONG)Read8Bits(dwAddress);
		case 16:
			return (ULONGLONG)Read16Bits(dwAddress);
		case 32:
			return (ULONGLONG)Read32Bits(dwAddress);
		case 64:
			return (ULONGLONG)Read64Bits(dwAddress);
	}
	return 0;
}

LPCTSTR DBGCommandUtil_GetValueToWrite(LPCTSTR szCommand, ULONGLONG nReadBits, ULONGLONG* pnValueToWrite)
{
	ULONGLONG nRValue;
	BOOL bBitMode = FALSE;
	BOOL bNotMode = FALSE;
	TCHAR cOperator = '=';
	LPCTSTR psz = szCommand;

	if(psz == NULL)
		return NULL;
	
	while(*psz == ' ') psz++;

	while(TRUE)
	{
		switch(*psz)
		{
			case '$':
				bBitMode = TRUE;
				break;
			case '^':
			case '|':
			case '&':
			case '=':
			case '+':
			case '-':
				cOperator = *psz;
				break;
			case '~':
				bNotMode = TRUE;
				break;
			default:
				goto endoperators;
		}
		psz++;
	}
endoperators:
	
	if(DBGCommandUtil_GetValue64(psz, &nRValue) == NULL)
		return NULL;

	if(bBitMode)
	{
		nRValue = 1 << nRValue;
	}
	if(bNotMode)
	{
		nRValue = ~nRValue;
	}
	switch(cOperator)
	{
		case '^':
			nRValue ^= nReadBits;
			break;
		case '|':
			nRValue |= nReadBits;
			break;
		case '&':
			nRValue &= nReadBits;
			break;
		case '+':
			nRValue += nReadBits;
			break;
		case '-':
			nRValue = nReadBits - nRValue;
			break;
	}
	*pnValueToWrite = nRValue;
	return psz;
}

void DBGCommandImpl_Write(LPCTSTR szCommand, int nBits)
{
	ULONGLONG nRValue;
	DWORD dwAddress;
	LPCTSTR psz = DBGCommandUtil_GetValue32(szCommand, &dwAddress);
	if(DBGCommandUtil_GetValueToWrite(psz, ReadBits(dwAddress, nBits), &nRValue) != NULL)
	{
		WriteBits(dwAddress, nRValue, nBits);
		DBGConsole_DumpMemory(~0);
		DBGConsole_DumpContext(~0);
	}
}

void DBGCommand_Dis(LPCTSTR szCommand)
{
	LONG n;
	TCHAR szStart[300];
	TCHAR szEnd[300];
	TCHAR szFileName[MAX_PATH+1] = "";		// Initialise to empty string
	DWORD dwStart;
	DWORD dwEnd;

	n = sscanf(szCommand, "%s %s %s", szStart, szEnd, szFileName);
	// Note - if the filename is left blank, Dump_Disassemble defaults
	if (n < 2)
	{
		DBGConsole_Msg(0, "[YInvalid argument! Type \"help dis\" for more info]");
		return;
	}

	DBGCommandUtil_GetValue32(szStart, &dwStart);
	DBGCommandUtil_GetValue32(szEnd, &dwEnd);

	Dump_Disassemble(dwStart, dwEnd, szFileName);

}

void DBGCommand_RDis(LPCTSTR szCommand)
{
	Dump_RSPDisassemble(szCommand);

}
#define IMPLEMENT_DBGCommand_Write(bits) void DBGCommand_Write##bits (LPCTSTR szCommand) {DBGCommandImpl_Write(szCommand, bits );}

IMPLEMENT_DBGCommand_Write(8)
IMPLEMENT_DBGCommand_Write(16)
IMPLEMENT_DBGCommand_Write(32)
IMPLEMENT_DBGCommand_Write(64)

#undef IMPLEMENT_DBGCommand_Write



void DBGCommand_WriteReg(LPCTSTR szCommand)
{
	while(*szCommand == '%') szCommand++;
	LPCTSTR pszCharAfterValue = strchr(szCommand, ' ');
	if(pszCharAfterValue == NULL)
	{
		DBGConsole_Msg(0, "[YInvalid numeric argument! Type \"help numbers\" for more info]");
		return;
	}
	
	DWORD dwRegNameLength = (DWORD)pszCharAfterValue - (DWORD)szCommand;
	if(dwRegNameLength == 0)
	{
		DBGConsole_Msg(0, "[YInvalid register name in numeric argument! Type \"help numbers\" for more info]");
		return;
	}

	TCHAR szRegName[100];
	memcpy(szRegName, szCommand, dwRegNameLength);
	szRegName[dwRegNameLength] = 0;

	DWORD nBits;
	void* pReg = DBGCommandUtil_GetAddressOfRegister(szRegName, &nBits);
	if(pReg == NULL)
		return;

	ULONGLONG nRValue;
	/*
	if(DBGCommandUtil_GetValue64(pszCharAfterValue, &nRValue) == NULL)
		return;
	*/
	if(DBGCommandUtil_GetValueToWrite(pszCharAfterValue, DBGCommandUtil_ReadRegister(pReg, nBits), &nRValue) != NULL)
	{
		DBGCommandUtil_WriteRegister(pReg, nRValue, nBits);

		DBGConsole_DumpRegisters();
	}	
}

void DBGCommand_List(LPCTSTR szCommand)
{
	DWORD dwAddress;
	if(DBGCommandUtil_GetValue32(szCommand, &dwAddress) != NULL)
	{
		s_nMidPaneType = DBGCON_CPU_CODE_VIEW;
		DBGConsole_RedrawPaneLabels();
		DBGConsole_DumpCPUContext(dwAddress);
	}

}


void DBGCommand_ListOS(LPCTSTR szCommand)
{
	if (lstrlen(szCommand) == 0)
	{
		DBGConsole_Msg(0, "Invalid argument: %s", szCommand);
	}
	else
	{
		s_nMidPaneType = DBGCON_CPU_CODE_VIEW;
		DBGConsole_RedrawPaneLabels();

		DWORD dwAddress = Patch_GetSymbolAddress(szCommand);
		if (dwAddress != ~0)
			DBGConsole_DumpCPUContext(dwAddress);
		else
			DBGConsole_Msg(0, "Symbol %s not found", szCommand);
	}

}

void DBGCommand_BPX(LPCTSTR szCommand)
{
	DWORD dwAddress;
	if(DBGCommandUtil_GetValue32(szCommand, &dwAddress) != NULL)
	{		
		CPU_AddBreakPoint(dwAddress);
		DBGConsole_UpdateDisplay();
	}
}

void DBGCommand_BPD(LPCTSTR szCommand)
{
	DWORD dwAddress;
	if(DBGCommandUtil_GetValue32(szCommand, &dwAddress) != NULL)
	{		
		CPU_EnableBreakPoint(dwAddress, FALSE);
		DBGConsole_UpdateDisplay();
	}
}

void DBGCommand_BPE(LPCTSTR szCommand)
{
	DWORD dwAddress;
	if(DBGCommandUtil_GetValue32(szCommand, &dwAddress) != NULL)
	{		
		CPU_EnableBreakPoint(dwAddress, TRUE);
		DBGConsole_UpdateDisplay();
	}
}


void DBGCommand_ShowCPU()
{
	s_nMidPaneType = DBGCON_CPU_CODE_VIEW;
	DBGConsole_ClearRegisters();
	DBGConsole_DumpContext(~0);
	DBGConsole_DumpRegisters();
	DBGConsole_RedrawPaneLabels();
}
void DBGCommand_ShowRSP()
{
	s_nMidPaneType = DBGCON_RSP_CODE_VIEW;
	DBGConsole_ClearRegisters();
	DBGConsole_DumpContext(~0);
	DBGConsole_DumpRegisters();
	DBGConsole_RedrawPaneLabels();
}


void DBGConsole_HelpCommand(LPCTSTR szCommand)
{
	if(*szCommand == 0)
	{
		int i;

		for(i = 0;; i++)
		{
			if(g_DBGHelpTopics[i].szHelpTopic == NULL)
				break;
		}

		DBGConsole_Msg(0, "[YSupported commands (help is available for light blue commands)]");
		
		for (i = 0; ; i++)
		{
			if (g_DBGCommands[i].szCommand == NULL)
				break;

			DBGConsole_Msg(0, "\t[%c%s%s]", (g_DBGCommands[i].szHelpText != NULL) ? 'C' : 'W', g_DBGCommands[i].szCommand, (g_DBGCommands[i].pArgFunc != NULL) ? " <parameters>" : "");
		}

		DBGConsole_Msg(0, "\n[YAvailable help topics]");

		for (i = 0; ; i++)
		{
			if (g_DBGHelpTopics[i].szHelpTopic == NULL)
				break;

			DBGConsole_Msg(0, "\t[%c%s]", (g_DBGHelpTopics[i].szHelpText != NULL) ? 'C' : 'W', g_DBGHelpTopics[i].szHelpTopic);
		}

		DBGConsole_Msg(0, "\n[YUse \"help <topic>\" to get help on a specific topic or command\n]");
	}
	else
	{
		BOOL bHelpDisplayed = FALSE;
		for (int i = 0; ; i++)
		{
			if (g_DBGHelpTopics[i].szHelpTopic == NULL)
			{
				break;
			}

			if((g_DBGHelpTopics[i].szHelpTopic == NULL) || (g_DBGHelpTopics[i].szHelpText == NULL))
				continue;

			if (stricmp(g_DBGHelpTopics[i].szHelpTopic, szCommand) == 0)
			{
				DBGConsole_Msg(0, "[YHelp text for \"%s\":]", g_DBGHelpTopics[i].szHelpTopic);
				
				DBGConsole_Msg(0, "[C%s\n]", g_DBGHelpTopics[i].szHelpText);
				
				bHelpDisplayed = TRUE;
				break;
			}
		}
		if(!bHelpDisplayed)
		{
			for (int i = 0; ; i++)
			{
				if (g_DBGCommands[i].szCommand == NULL)
				{
					break;
				}

				if(g_DBGCommands[i].szHelpText == NULL)
					continue;

				if (stricmp(g_DBGCommands[i].szCommand, szCommand) == 0)
				{
					DBGConsole_Msg(0, "[YHelp text for \"%s\":]", g_DBGCommands[i].szCommand);
					
					DBGConsole_Msg(0, "[C%s\n]", g_DBGCommands[i].szHelpText);
					
					bHelpDisplayed = TRUE;
					break;
				}
			}
		}
		if(!bHelpDisplayed)
		{
			DBGConsole_Msg(0, "[YNo help available on \"%s\"]", szCommand);
		}
	}
}


void DBGConsole_GoCommand()
{
	CHAR szReason[300+1];

	if (!StartCPUThread(szReason, 300))
	{
		DBGConsole_Msg(0, "[YGo: %s]", szReason);
	}
}

void DBGConsole_StopCommand()
{
	StopCPUThread();
}

int DBGConsole_GetStringHighlight(char c)
{
	switch(c)
	{
		case 'r': return FOREGROUND_RED;
		case 'g': return FOREGROUND_GREEN;
		case 'b': return FOREGROUND_BLUE;
		case 'c': return FOREGROUND_GREEN|FOREGROUND_BLUE;
		case 'm': return FOREGROUND_RED|FOREGROUND_BLUE;
		case 'y': return FOREGROUND_RED|FOREGROUND_GREEN;
		case 'w': return sc_wAttrWhite;
		case 'R': return FOREGROUND_RED|FOREGROUND_INTENSITY;
		case 'G': return FOREGROUND_GREEN|FOREGROUND_INTENSITY;
		case 'B': return FOREGROUND_BLUE|FOREGROUND_INTENSITY;
		case 'C': return FOREGROUND_GREEN|FOREGROUND_BLUE|FOREGROUND_INTENSITY;
		case 'M': return FOREGROUND_RED|FOREGROUND_BLUE|FOREGROUND_INTENSITY;
		case 'Y': return FOREGROUND_RED|FOREGROUND_GREEN|FOREGROUND_INTENSITY;
		case 'W': return sc_wAttrBoldWhite;
		default: return -1;
	}
}

void DBGCommand_Close()
{
	DBGConsole_Enable(FALSE);
}

void DBGCommand_Quit()
{
	SendMessage(g_hMainWindow, WM_CLOSE, 0, 0);
	//DestroyWindow(g_hMainWindow);
}

void DBGConsole_ParseStringHighlights(LPSTR szString, WORD * arrwAttributes, WORD wAttr)
{
	WORD wCurrAttribute = wAttr;
	int iIn, iOut;
	int nMax = lstrlen(szString);

	for (iIn = 0, iOut = 0; iIn < nMax; iIn++)
	{
		if (szString[iIn] == '[')
		{
			int highlight = DBGConsole_GetStringHighlight(szString[iIn+1]);
			if(highlight != -1)
			{
				wCurrAttribute = (WORD)highlight;
			}
			else
			{
				switch (szString[iIn+1])
				{
				case '[':
				case ']':
					szString[iOut] = szString[iIn+1];
					arrwAttributes[iOut] = wCurrAttribute;

					iOut++;
					break;
				}
			}

			// Skip colour character
			iIn++;
		}
		else if (szString[iIn] == ']')
		{
			wCurrAttribute = wAttr;
		}
		else
		{
			szString[iOut] = szString[iIn];
			arrwAttributes[iOut] = wCurrAttribute;

			iOut++;
		}
	}
	szString[iOut] = '\0';

}

void __cdecl DBGConsole_Stats(DWORD dwRow, LPCTSTR szFormat, ...)
{

    DWORD cWritten; 
    BOOL bSuccess; 
    COORD coord; 
	WORD wNumToWrite;
    va_list va;
	TCHAR szBuffer[2048+1];

	// Return if we don't have a console!
	if (!g_bShowDebug || !s_bConsoleAllocated)
		return;


    coord.X = 55;
    coord.Y = (SHORT)(sc_nMidPaneTextOffset+1+dwRow);
	
	WORD wMaxWidth = 80 - coord.X;

	// Format the output
	va_start(va, szFormat);
	// Don't use wvsprintf as it doesn't handle floats!
	vsprintf(szBuffer, szFormat, va);
	va_end(va);


	wNumToWrite = lstrlen(szBuffer);
	if (wNumToWrite > wMaxWidth)
		wNumToWrite = wMaxWidth;

    bSuccess = WriteConsoleOutputCharacter( 
        g_hStdOut,              // screen buffer handle 
        szBuffer,								// pointer to source string 
        wNumToWrite,							// length of string 
        coord,									// first cell to write to 
        &cWritten);								// actual number written 
    if (!bSuccess) 
        return; 
 
	// Write a string of colors to a screen buffer. 
	bSuccess = FillConsoleOutputAttribute(
		g_hStdOut,
		sc_wAttrWhite,
		wMaxWidth,		// Clear to end
		coord,
		&cWritten);
    if (!bSuccess) 
        return;  

	// Clear the end of the buffer
	if (wNumToWrite < wMaxWidth)
	{
		coord.X += wNumToWrite;
		FillConsoleOutputCharacter(g_hStdOut,
			' ', wMaxWidth - wNumToWrite, coord, &cWritten);
	}
	
	return;


}

DWORD g_nTabSpaces = 4;

LPCTSTR DBGConsole_ParseTabs(LPCTSTR psz, LPWORD pwAttributes)
{
	int len = strlen(psz);
	int tabCount = 0;

	for(int i = 0; i < len; i++)
	{
		if(psz[i] == '\t')
			tabCount++;
	}

	int tabbedLen = len + (tabCount * (g_nTabSpaces - 1));
	LPTSTR pszOutput = new TCHAR[tabbedLen + 1];

	if(tabCount > 0)
	{
		LPWORD pwOutAttributes = new WORD[tabbedLen];

		int iOut = 0;

		for(int i = 0; i < len; i++)
		{
			if(psz[i] == '\t')
			{
				int tabsToAdd = g_nTabSpaces - (iOut % g_nTabSpaces);
				memset(pszOutput + iOut, ' ', tabsToAdd);
				for(int iTab = 0; iTab < tabsToAdd; iTab++)
				{
					pwOutAttributes[iOut + iTab] = pwAttributes[iTab];
				}
				iOut += tabsToAdd;
			}
			else
			{
				pszOutput[iOut] = psz[i];
				pwOutAttributes[iOut] = pwAttributes[i];
				iOut++;
			}
		}

		pszOutput[iOut] = 0;

		memcpy(pwAttributes, pwOutAttributes, tabbedLen * sizeof(WORD));

		delete [] pwOutAttributes;
	}
	else
	{
		strcpy(pszOutput, psz);
	}

	return pszOutput;
}

// do not call this directly
void DBGConsole_Internal_SingleLineMsg(DWORD dwType, LPTSTR pszBuffer, LPWORD arrwAttributes)
{
	BOOL bSuccess; 
	CONSOLE_SCREEN_BUFFER_INFO csbiInfo; 
	SMALL_RECT srctScrollRect, srctClipRect; 
	CHAR_INFO chiFill; 
	COORD coordDest; 
	DWORD cWritten;
	LONG nOutputLength;

	bSuccess = GetConsoleScreenBufferInfo(g_hOutputBuffer, &csbiInfo); 
	if (!bSuccess) 
		return;

	// Scan through szBuffer and set up attributes
	//DBGConsole_ParseStringHighlights(pszBuffer, arrwAttributes, sc_wAttrWhite);

	nOutputLength = lstrlen(pszBuffer);

	// Scroll the contents of the buffer up by one line
	// Define the scrolling rectangle (we lose the top line)
	srctScrollRect.Top = 1; 
	srctScrollRect.Bottom = csbiInfo.dwSize.Y - 1; 
	srctScrollRect.Left = 0; 
	srctScrollRect.Right = csbiInfo.dwSize.X - 1; 
	
	// The destination for the scroll rectangle is one row up. 
	coordDest.X = 0; 
	coordDest.Y = 0; 
	
	// The clipping rectangle is the same as the scrolling rectangle. 
	// The destination row is left unchanged.
	srctClipRect = srctScrollRect; 
	
	// Fill the bottom row
	chiFill.Attributes = sc_wAttrWhite; 
	chiFill.Char.AsciiChar = ' '; 
	
	// Scroll up one line. 
	/*bSuccess = */ScrollConsoleScreenBuffer( 
		g_hOutputBuffer,         // screen buffer handle 
		&srctScrollRect, // scrolling rectangle 
		&srctClipRect,   // clipping rectangle 
		coordDest,       // top left destination cell 
		&chiFill);       // fill character and color 


	// Write the output to the buffer
	COORD coordCursor;

	// Write the new string to the bottom of the buffer
	coordCursor.X = 0;
	coordCursor.Y = csbiInfo.dwSize.Y - 1; 

	/*bSuccess = */WriteConsoleOutputCharacter( 
		g_hOutputBuffer,            // screen buffer handle 
		pszBuffer,					// pointer to source string 
		nOutputLength,				// length of string 
		coordCursor,				// first cell to write to 
		&cWritten);					// actual number written 

	/*bSuccess = */WriteConsoleOutputAttribute(
		g_hOutputBuffer,
		arrwAttributes,
		nOutputLength,
		coordCursor,
		&cWritten);

	DBGConsole_DisplayOutput(0);

}

void __cdecl DBGConsole_Msg(DWORD dwType, LPCTSTR szFormat, ...)
{
    va_list va;

	// Return if we don't have a console!
	if (!g_bShowDebug || !s_bConsoleAllocated)
		return;

	if (szFormat == NULL)
		return;

	TCHAR szBuffer[65536+1];
	LPTSTR pszBuffer = szBuffer;
	
	// Parse the buffer:
	try {
		// Format the output
		va_start(va, szFormat);
		// Don't use wvsprintf as it doesn't handle floats!
		vsprintf(pszBuffer, szFormat, va);
		va_end(va);
	}
	catch (...)
	{
		// Ignore g_bTrapExceptions
		return;
	}

	int nOutputLength = lstrlen(pszBuffer);

	WORD arrwAttributes[2048];
	WORD wCurrAttribute = sc_wAttrWhite;
	
	char cLastNewLine = '\0';
	int iOut = 0;

	for(int i = 0; i <= nOutputLength; i++)
	{
		char c = pszBuffer[i];
		if(
			((c == '\n') && (cLastNewLine == '\r')) ||
			((c == '\r') && (cLastNewLine == '\n'))
			)
		{
			cLastNewLine = 0;
			//pszLineBeginning++;
		}
		else if((c == '\n') || (c == '\r') || (c == '\0'))
		{
			pszBuffer[iOut] = '\0';
			// ParseTabs return a newed string. It never returns the parameter even if there are no tabs.
			//MessageBox(NULL, pszBuffer, "Buf", 0);
			LPTSTR pszTabParsed = (LPTSTR)DBGConsole_ParseTabs(pszBuffer, arrwAttributes);
			DBGConsole_Internal_SingleLineMsg(dwType, pszTabParsed, arrwAttributes);
			delete [] pszTabParsed;
			iOut = 0;

			cLastNewLine = c;
			//pszLineBeginning = pszBuffer + i + 1;
		}
		else if (c == '[')
		{
			int highlight = DBGConsole_GetStringHighlight(pszBuffer[i+1]);
			if(highlight != -1)
			{
				wCurrAttribute = (WORD)highlight;
			}
			else
			{
				switch (pszBuffer[i+1])
				{
				case '[':
				case ']':
					pszBuffer[iOut] = pszBuffer[i+1];
					arrwAttributes[iOut] = wCurrAttribute;

					iOut++;
					break;
				}
			}

			// Skip colour character
			i++;
		}
		else if (c == ']')
		{
			wCurrAttribute = sc_wAttrWhite;
		}
		else
		{
			pszBuffer[iOut] = c;
			arrwAttributes[iOut] = wCurrAttribute;

			iOut++;
		}			
	}
	//pszBuffer[iOut] = '\0';
	//DBGConsole_Internal_SingleLineMsg(dwType, pszBuffer, arrwAttributes);
}

// Used to overwrite previous lines - like for doing % complete indicators
void __cdecl DBGConsole_MsgOverwrite(DWORD dwType, LPCTSTR szFormat, ...)
{
	BOOL bSuccess; 
	CONSOLE_SCREEN_BUFFER_INFO csbiInfo; 
	DWORD cWritten;
    va_list va;
	TCHAR szBuffer[2048+1];
	LONG nOutputLength;

	WORD arrwAttributes[2048];

	// Return if we don't have a console!
	if (!g_bShowDebug || !s_bConsoleAllocated)
		return;

    
	bSuccess = GetConsoleScreenBufferInfo(g_hOutputBuffer, &csbiInfo); 
	if (!bSuccess) 
		return;

	// Parse the buffer:
	{
		// Format the output
		va_start(va, szFormat);
		// Don't use wvsprintf as it doesn't handle floats!
		vsprintf(szBuffer, szFormat, va);
		va_end(va);
		// Scan through szBuffer and set up attributes
		DBGConsole_ParseStringHighlights(szBuffer, arrwAttributes, sc_wAttrWhite);

		nOutputLength = lstrlen(szBuffer);

	}
	
	// Write the output to the buffer
	{
		COORD coord;

		// Write the new string to the bottom of the buffer
		coord.X = 0;
		coord.Y = csbiInfo.dwSize.Y - 1; 

		/*bSuccess = */WriteConsoleOutputCharacter( 
			g_hOutputBuffer,            // screen buffer handle 
			szBuffer,					// pointer to source string 
			nOutputLength,				// length of string 
			coord,						// first cell to write to 
			&cWritten);					// actual number written 

		/*bSuccess = */WriteConsoleOutputAttribute(
			g_hOutputBuffer,
			arrwAttributes,
			nOutputLength,
			coord,
			&cWritten);

			// Clear the end of the buffer
			if (nOutputLength < 80)
			{
				coord.X += nOutputLength;
				FillConsoleOutputCharacter(g_hOutputBuffer,
					' ', 80 - nOutputLength, coord, &cWritten);
			}
	}

	DBGConsole_DisplayOutput(0);


}



// Copy the visible portion to the display
void DBGConsole_DisplayOutput(LONG nOffset)
{
	CONSOLE_SCREEN_BUFFER_INFO csbiInfo; 
	SMALL_RECT srctReadRect; 
	SMALL_RECT srctWriteRect; 
	CHAR_INFO chiBuffer[sc_nBotPaneLines*80];
	COORD coordBufSize; 
	COORD coordBufCoord; 
	BOOL bSuccess; 

	bSuccess = GetConsoleScreenBufferInfo(g_hOutputBuffer, &csbiInfo); 
	if (!bSuccess) 
		return;


	// We can only have negative offsets - an offset of 0 shows the last line!
	if (nOffset > 0)
		nOffset = 0;

	g_nOutputBufOffset = nOffset;

	// Offset top by 
	srctReadRect.Top = (SHORT)(csbiInfo.dwSize.Y - (sc_nBotPaneLines - g_nOutputBufOffset));
	if (srctReadRect.Top < 0)
		srctReadRect.Top = 0;
	srctReadRect.Left = 0; 
	srctReadRect.Bottom = srctReadRect.Top + sc_nBotPaneLines;
	srctReadRect.Right = csbiInfo.dwSize.X - 1; 

	// The temporary buffer size is sc_nBotPaneLines rows x 80 columns. 
	coordBufSize.X = 80; 
	coordBufSize.Y = sc_nBotPaneLines; 

	// The top left destination cell of the temporary buffer is 
	// row 0, col 0. 
	coordBufCoord.X = 0; 
	coordBufCoord.Y = 0; 

	// Copy the block from the screen buffer to the temp. buffer. 

	bSuccess = ReadConsoleOutput( 
	   g_hOutputBuffer,        // screen buffer to read from 
	   chiBuffer,      // buffer to copy into 
	   coordBufSize,   // col-row size of chiBuffer 
	   coordBufCoord,  // top left dest. cell in chiBuffer 
	   &srctReadRect); // screen buffer source rectangle 
	if (!bSuccess) 
		return; 

	// Set the destination rectangle. 
	srctWriteRect.Top = sc_nBotPaneTextOffset+1; 
	srctWriteRect.Left = 0; 
	srctWriteRect.Bottom = (srctWriteRect.Top+sc_nBotPaneLines)-1; 
	srctWriteRect.Right = 80-1; 

	// Copy from the temporary buffer to the new screen buffer. 

	bSuccess = WriteConsoleOutput( 
		g_hStdOut, // screen buffer to write to 
		chiBuffer,        // buffer to copy from 
		coordBufSize,     // col-row size of chiBuffer 
		coordBufCoord,    // top left src cell in chiBuffer 
		&srctWriteRect);  // dest. screen buffer rectangle 
	if (!bSuccess) 
		return;

}


