/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifndef __DAEDALUS_D3DRENDER_H__
#define __DAEDALUS_D3DRENDER_H__

#include <d3dx.h>

#include "RDP_GFX.h"		// DL_PF

#include "DBGConsole.h"
#include "Memory.h"

#define CLIPPING 0

#define DAEDALUS_MATRIX_STACK		60

#define DAEDALUSFVF_TLITVERTEX (D3DFVF_XYZRHW | D3DFVF_DIFFUSE | D3DFVF_TEX1) 
 typedef struct _TLITVERTEX {
	D3DXVECTOR3 pos;
	float rhw;
	D3DCOLOR  dcDiffuse;
	D3DVALUE  tu, tv;
} TLITVERTEX, *LPTLITVERTEX;

#define DAEDALUSFVF_LITVERTEX (D3DFVF_XYZRHW | D3DFVF_DIFFUSE ) 
 typedef struct _LITVERTEX {
	D3DXVECTOR3 pos;
	float rhw;
	D3DCOLOR  dcDiffuse;
} LITVERTEX, *LPLITVERTEX;

typedef struct
{
	D3DXVECTOR3 dir;		// Should be normalised
	D3DXCOLOR col;

} DaedalusLight;




/*n.x = (mn.x * matWorld.m00) + (mn.y * matWorld.m10) + (mn.z * matWorld.m20);
n.y = (mn.x * matWorld.m01) + (mn.y * matWorld.m11) + (mn.z * matWorld.m21);
n.z = (mn.x * matWorld.m02) + (mn.y * matWorld.m12) + (mn.z * matWorld.m22);*/

// Multiply (x,y,z,0) by matrix m, then normalize
#define Vec3TransformNormal(vec, m) __asm					\
	{														\
		__asm fld		dword ptr [vec + 0]							\
		__asm fmul	dword ptr [m + 0]			/* x m00*/		\
		__asm fld		dword ptr [vec + 0]							\
		__asm fmul	dword ptr [m + 4] 	/* x m01  x m00*/			\
		__asm fld		dword ptr [vec + 0]								\
		__asm fmul	dword ptr [m + 8] 	/* x m02  x m01  x m00*/	\
																\
		__asm fld		dword ptr [vec + 4]								\
		__asm fmul	dword ptr [m + 16] 	/* y m10  x m02  x m01  x m00*/	\
		__asm fld		dword ptr [vec + 4]									\
		__asm fmul	dword ptr [m + 20] 	/* y m11  y m10  x m02  x m01  x m00*/		\
		__asm fld		dword ptr [vec + 4]												\
		__asm fmul	dword ptr [m + 24]	/* y m12  y m11  y m10  x m02  x m01  x m00*/	\
																						\
		__asm fxch	st(2)				/* y m10  y m11  y m12  x m02  x m01  x m00*/			\
		__asm faddp	st(5), st(0)		/* y m11  y m12  x m02  x m01  (x m00 + y m10)*/		\
		__asm faddp	st(3), st(0)		/* y m12  x m02  (x m01 + ym11)  (x m00 + y m10)*/	\
		__asm faddp	st(1), st(0)		/* (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/	\
																						\
		__asm fld		dword ptr [vec + 8]														\
		__asm fmul	dword ptr [m + 32] /* z m20  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/	\
		__asm fld		dword ptr [vec + 8]																\
		__asm fmul	dword ptr [m + 36] /* z m21  z m20  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/				\
		__asm fld		dword ptr [vec + 8]																				\
		__asm fmul	dword ptr [m + 40] /* z m22  z m21  z m20  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/		\
																												\
		__asm fxch	st(2)				/* z m20  z m21  z m22  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/		\
		__asm faddp	st(5), st(0)		/* z m21  z m22  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10 + z m20)*/	\
		__asm faddp	st(3), st(0)		/* z m22  (x m02 + y m12) (x m01 + ym11 + z m21)  (x m00 + y m10 + z m20)*/	\
		__asm faddp	st(1), st(0)		/* (x m02 + y m12 + z m 22) (x m01 + ym11 + z m21)  (x m00 + y m10 + z m20)*/	\
																						\
		__asm fxch	st(2)				/* (x m00 + y m10 + z m20) (x m01 + ym11 + z m21) (x m02 + y m12 + z m 22) */	\
																														\
		__asm fld1                      /* 1 x y z */ \
		__asm fld   st(1)				/* x 1 x y z */	\
		__asm fmul  st(0),st(0)			/* xx 1 x y z */  \
		__asm fld   st(3)               /* y xx 1 x y z */ \
		__asm fmul  st(0),st(0)			/* yy xx 1 x y z */ \
		__asm fld   st(5)				/* z yy xx 1 x y z */ \
		__asm fmul  st(0),st(0)			/* zz yy xx 1 x y z */ \
																														\
		__asm fxch  st(2)				/* xx yy zz 1 x y z */ \
																														\
		__asm faddp st(1),st(0)			/* (xx+yy) zz 1 x y z */ \
		__asm faddp st(1),st(0)			/* (xx+yy+zz) 1 x y z */ \
																														\
		__asm fsqrt						/* l 1 x y z */ \
																														\
		__asm fdivp st(1),st(0)			/* (1/l) x y z */ \
																														\
		__asm fmul  st(3),st(0)		    /* f x y fz */										\
		__asm fmul  st(2),st(0)			/* f x fy fz */										\
		__asm fmulp st(1),st(0)			/* fx fy fz */										\
																														\
		__asm fstp	dword ptr [vec + 0]	/* fy fz*/							\
		__asm fstp	dword ptr [vec + 4]	/* fz	*/			\
		__asm fstp	dword ptr [vec + 8]	/* done	*/																		\
	}		\



// Multiply (x,y,z,1) by matrix m and project onto w = 1
#define Vec3TransformCoord(vecout, vec, m) __asm			\
	{														\
		__asm fld	dword ptr [vec + 0]							\
		__asm fmul	dword ptr [m + 0]			/* x m00*/		\
		__asm fld	dword ptr [vec + 0]							\
		__asm fmul	dword ptr [m + 4] 	/* x m01  x m00*/			\
		__asm fld	dword ptr [vec + 0]								\
		__asm fmul	dword ptr [m + 8] 	/* x m02  x m01  x m00*/	\
																\
		__asm fld	dword ptr [vec + 4]								\
		__asm fmul	dword ptr [m + 16] 	/* y m10  x m02  x m01  x m00*/	\
		__asm fld	dword ptr [vec + 4]									\
		__asm fmul	dword ptr [m + 20] 	/* y m11  y m10  x m02  x m01  x m00*/		\
		__asm fld	dword ptr [vec + 4]												\
		__asm fmul	dword ptr [m + 24]	/* y m12  y m11  y m10  x m02  x m01  x m00*/	\
																						\
		__asm fxch	st(2)				/* y m10  y m11  y m12  x m02  x m01  x m00*/			\
		__asm faddp	st(5), st(0)		/* y m11  y m12  x m02  x m01  (x m00 + y m10)*/		\
		__asm faddp	st(3), st(0)		/* y m12  x m02  (x m01 + ym11)  (x m00 + y m10)*/	\
		__asm faddp	st(1), st(0)		/* (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/	\
																						\
		__asm fld	dword ptr [vec + 8]														\
		__asm fmul	dword ptr [m + 32] /* z m20  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/	\
		__asm fld	dword ptr [vec + 8]																\
		__asm fmul	dword ptr [m + 36] /* z m21  z m20  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/				\
		__asm fld	dword ptr [vec + 8]																				\
		__asm fmul	dword ptr [m + 40] /* z m22  z m21  z m20  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/		\
																												\
		__asm fxch	st(2)				/* z m20  z m21  z m22  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10)*/		\
		__asm faddp	st(5), st(0)		/* z m21  z m22  (x m02 + y m12) (x m01 + ym11)  (x m00 + y m10 + z m20)*/	\
		__asm faddp	st(3), st(0)		/* z m22  (x m02 + y m12) (x m01 + ym11 + z m21)  (x m00 + y m10 + z m20)*/	\
		__asm faddp	st(1), st(0)		/* (x m02 + y m12 + z m 22) (x m01 + ym11 + z m21)  (x m00 + y m10 + z m20)*/	\
																						\
		__asm fxch	st(2)				/* (x m00 + y m10 + z m20) (x m01 + ym11 + z m21) (x m02 + y m12 + z m 22) */	\
		__asm fadd	dword ptr [m + 48]	/* (xm00+ym10+zm20+m30)	 (xm01+ym11+zm21) (xm02+ym12+zm22) */		\
		__asm fxch	st(1)				/* (xm01+ym11+zm21) (xm00+ym10+zm20+m30)  (xm02+ym12+zm22) */		\
		__asm fadd	dword ptr [m + 52] 	/* (xm01+ym11+zm21+m31) (xm00+ym10+zm20+m30)  (xm02+ym12+zm22) */		\
		__asm fxch	st(2)				/* (xm02+ym12+zm22) (xm00+ym10+zm20+m30)(xm01+ym11+zm21+m31)   */	\
		__asm fadd	dword ptr [m + 56]	/* (xm02+ym12+zm22+m32) (xm00+ym10+zm20+m30)(xm01+ym11+zm21+m31)   */	\
		__asm fxch	st(1)				/* (xm00+ym10+zm20+m30)(xm02+ym12+zm22+m32) (xm01+ym11+zm21+m31)   */	\
																							\
		__asm fld1						/* 1 abc */											\
		__asm fld	dword ptr [vec + 0]	/* x 1 abc*/										\
		__asm fmul  dword ptr [m + 12]  /* xm03 1 abc*/										\
		__asm fld	dword ptr [vec + 4]	/* y xm03 1 abc*/									\
		__asm fmul  dword ptr [m + 28]	/* ym13 xm03 1 abc*/								\
		__asm fld	dword ptr [vec + 8]	/* z ym13 xm03 1 abc*/								\
		__asm fmul  dword ptr [m + 44]  /* zm23 ym13 xm03 1 abc*/							\
																							\
		__asm fxch	st(2)				/* xm03 ym13 zm23 1 abc*/							\
		__asm faddp st(1), st(0)		/* (xm03 + ym13) zm23 1 abc*/						\
		__asm faddp	st(1), st(0)		/* (xm03 + ym13 + zm23) 1 abc*/						\
																							\
		__asm fadd  dword ptr [m + 60]	/* (xm03+ym13+zm23+m33) 1 abc*/						\
																							\
		__asm fdivp  st(1), st(0)		/* 1.0 / (xm03+ym13+zm23+m33) abc*/					\
																							\
		__asm fmul  st(3),st(0)		    /* f a b fc */										\
		__asm fmul  st(2),st(0)			/* f a fb fc */										\
		__asm fmulp st(1),st(0)			/* fx fz fy */										\
																							\
		__asm fstp	dword ptr [vecout + 0]	/* (xm02+ym12+zm22+m32) (xm01+ym11+zm21+m31)*/	\
		__asm fstp	dword ptr [vecout + 8]	/* (xm01+ym11+zm21+m31)	*/						\
		__asm fstp	dword ptr [vecout + 4]	/* done	*/										\
	}																						\



typedef struct tagTSSInfo
{
	DWORD dwColorOp;
	DWORD dwColorArg1;
	DWORD dwColorArg2;
	DWORD dwAlphaOp;
	DWORD dwAlphaArg1;
	DWORD dwAlphaArg2;
	
	DWORD dwMinFilter;
	DWORD dwMagFilter;

	DWORD dwAddressUMode;
	DWORD dwAddressVMode;
	
} TSSInfo;

enum
{
	D3DRENDER_LOAD = 0,
	D3DRENDER_MUL = 1
};


#pragma pack(1)
	typedef struct _FiddledVtx
	{
		short y;
		short x;

		short flag;
		short z;

		short tv;
		short tu;

		union {
			struct _rgba {
				BYTE a;
				BYTE b;
				BYTE g;
				BYTE r;
			} rgba;
			struct _norma {
				char na;
				char nz;	// b
				char ny;	//g
				char nx;	//r
			} norma;
		};
	} FiddledVtx;

	typedef struct _FiddledVtxDKR
	{
		s16 y;
		s16 x;

		u8 a;
		u8 b;
		s16 z;

		u8 g;
		u8 r;

	} FiddledVtxDKR;


#pragma pack()

class D3DRender 
{
public:
	D3DRender();
	~D3DRender();

	HRESULT Initialize();


	inline void SetScreenMult(float fMultX, float fMultY)
	{
		m_fScreenMultX = fMultX;
		m_fScreenMultY = fMultY;
	}

	inline void SetTextureEnable(BOOL bEnable)
	{
		m_bTextureEnable = bEnable;
	}
	

	inline void SetMinFilter(DWORD dwStage, DWORD dwMinFilter)
	{
		if (m_TSSInfo[dwStage].dwMinFilter != dwMinFilter)
		{
			m_TSSInfo[dwStage].dwMinFilter = dwMinFilter;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_MINFILTER, dwMinFilter );
		}
	}

	inline void SetMagFilter(DWORD dwStage, DWORD dwMagFilter)
	{
		if (m_TSSInfo[dwStage].dwMagFilter != dwMagFilter)
		{
			m_TSSInfo[dwStage].dwMagFilter = dwMagFilter;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_MAGFILTER, dwMagFilter );
		}
	}


	inline void SetAddressU(DWORD dwStage, DWORD dwMode)
	{
		if (m_TSSInfo[dwStage].dwAddressUMode != dwMode)
		{
			m_TSSInfo[dwStage].dwAddressUMode = dwMode;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_ADDRESSU, dwMode );
		}
	}
	inline void SetAddressV(DWORD dwStage, DWORD dwMode)
	{
		if (m_TSSInfo[dwStage].dwAddressVMode != dwMode)
		{
			m_TSSInfo[dwStage].dwAddressVMode = dwMode;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_ADDRESSV, dwMode );
		}
	}

	inline void SetNumStages(DWORD dwMaxStage)
	{
		if (dwMaxStage < m_dwNumStages)
		{
			// Number of stages has been reduced
			// Disable dwNum+1
			if (dwMaxStage < 7)
			{
				m_pD3DDev->SetTextureStageState( dwMaxStage+1, D3DTSS_COLOROP,   D3DTOP_DISABLE );
				m_pD3DDev->SetTextureStageState( dwMaxStage+1, D3DTSS_ALPHAOP,   D3DTOP_DISABLE );
			}
		}
		else if (dwMaxStage > m_dwNumStages)
		{
			// Number of stages has been increased
			// Enable dwNum
			// TODO this probably wont work with dwMaxStage>1
			if (dwMaxStage <= 7)
			{
				m_pD3DDev->SetTextureStageState( dwMaxStage, D3DTSS_COLOROP,   m_TSSInfo[dwMaxStage].dwColorOp );
				m_pD3DDev->SetTextureStageState( dwMaxStage, D3DTSS_ALPHAOP,   m_TSSInfo[dwMaxStage].dwAlphaOp );
			}

		}
		m_dwNumStages = dwMaxStage;
	}

	inline void SetAmbientLight(DWORD dwLight)
	{
		//if (m_dwAmbientLight != dwLight)
		{
			m_dwAmbientLight = dwLight;
		//	m_pD3DDev->SetRenderState(  D3DRENDERSTATE_AMBIENT, dwLight );			
		}
	}

	inline void SetPrimitiveColor(DWORD dwCol)
	{
		m_dwPrimitiveColor = dwCol;
		//SetTextureFactor(dwCol);
	}


	inline void SetEnvColor(DWORD dwCol)
	{
		m_dwEnvColor = dwCol;
		//SetTextureFactor(dwCol);
	}

	inline void SetTextureFactor(DWORD dwCol)
	{
		if (m_dwTextureFactor != dwCol)
		{
			m_dwTextureFactor = dwCol;
			m_pD3DDev->SetRenderState(D3DRENDERSTATE_TEXTUREFACTOR, dwCol);
		}
	}


	inline void SetCullMode(BOOL bCullFront, BOOL bCullBack)
	{
		/*if (bCullFront && bCullBack)	g_Renderer->SetCullMode(D3DCULL_CCW);	// Cull everything....?
		else if (bCullFront)			g_Renderer->SetCullMode(D3DCULL_CCW);
		else if (bCullBack)				g_Renderer->SetCullMode(D3DCULL_CW);
		else							g_Renderer->SetCullMode(D3DCULL_NONE);*/
		m_bCullFront = bCullFront;
		m_bCullBack = bCullBack;
			
		/*if (m_dwCullMode != dwCullMode)
		{
			m_dwCullMode = dwCullMode;
			m_pD3DDev->SetRenderState( D3DRENDERSTATE_CULLMODE, dwCullMode );
		}*/
	}

	inline void SetShadeMode(DWORD dwShadeMode)
	{
		if (m_dwShadeMode != dwShadeMode)
		{
			m_dwShadeMode = dwShadeMode;
			m_pD3DDev->SetRenderState( D3DRENDERSTATE_SHADEMODE, dwShadeMode );
		}
	}

	inline void SetLighting(BOOL bLighting)
	{
		//if (m_bLightingEnable != bLighting)
		{
			m_bLightingEnable = bLighting;
			//m_pD3DDev->SetRenderState( D3DRENDERSTATE_LIGHTING, bLighting );
		}
	}


	
	inline void ZBufferEnable(BOOL bZBuffer)
	{
		if (m_bZBuffer != bZBuffer)
		{
			m_bZBuffer = bZBuffer;
			if (bZBuffer)
			{
				SetD3DRSZEnable( m_bZCompare ? D3DZB_TRUE : D3DZB_FALSE );
				SetD3DRSZWriteEnable( m_bZUpdate );
			}
			else
			{
				SetD3DRSZEnable( D3DZB_FALSE );
				SetD3DRSZWriteEnable( FALSE );
			}
		}
	}
	
	inline void SetZCompare(BOOL bZCompare)
	{
		if (m_bZCompare != bZCompare)
		{
			m_bZCompare = bZCompare;

			if (m_bZBuffer)
				SetD3DRSZEnable( bZCompare ? D3DZB_TRUE : D3DZB_FALSE );
		}
	}

	inline void SetZUpdate(BOOL bZUpdate)
	{
		if (m_bZUpdate != bZUpdate)
		{
			m_bZUpdate = bZUpdate;

			// Only update if ZBuffer is enabled
			if (m_bZBuffer)
				SetD3DRSZWriteEnable( bZUpdate );
		}
	}




	
	inline void SetTexturePerspective(DWORD dwTexturePerspective)
	{
		if (m_dwTexturePerspective != dwTexturePerspective)
		{
			m_dwTexturePerspective = dwTexturePerspective;
			m_pD3DDev->SetRenderState( D3DRENDERSTATE_TEXTUREPERSPECTIVE, dwTexturePerspective );
		}
	}

	inline void SetAlphaTestEnable(BOOL bAlphaTestEnable)
	{
		/*if (m_bAlphaTestEnable != bAlphaTestEnable)
		{
			m_bAlphaTestEnable = bAlphaTestEnable;
			m_pD3DDev->SetRenderState( D3DRENDERSTATE_ALPHATESTENABLE, bAlphaTestEnable );
		}*/
	}

	inline void SetTexture(LPDIRECTDRAWSURFACE7 lpsTexture,
						   LONG  nTileLeft, LONG nTileTop,
						   DWORD dwTileWidth, DWORD dwTileHeight)
	{
		//lpsTexture = NULL; - For wireframe
		if (m_lpsTexture != lpsTexture)
		{
			m_lpsTexture = lpsTexture;

			if (m_lpsTexture != NULL)
			{
				HRESULT hr;
				DDSURFACEDESC2 ddsd;
				
				m_lpsTexture->AddRef();

				m_dwTileWidth = dwTileWidth;
				m_dwTileHeight = dwTileHeight;

				// Update the bogus scale parameters
				ZeroMemory(&ddsd, sizeof(ddsd));
				ddsd.dwSize = sizeof(ddsd);
				ddsd.dwFlags = DDSD_WIDTH | DDSD_HEIGHT;

				m_fTexLeft = (float)nTileLeft;
				m_fTexTop  = (float)nTileTop;

				hr = m_lpsTexture->GetSurfaceDesc(&ddsd);
				if (FAILED(hr))
				{
					m_fTexWidth = (float)m_dwTileWidth;
					m_fTexHeight = (float)m_dwTileHeight;
				}
				else
				{
					m_fTexWidth = (float)ddsd.dwWidth;
					m_fTexHeight = (float)ddsd.dwHeight;
				}
			}
		}
	}

	// Generate texture coords?
	inline void SetTextureGen(BOOL bTextureGen)
	{
		m_bTextureGen = bTextureGen;
	}

	inline void SetTextureScale(float fScaleX, float fScaleY)
	{
		m_fTexScaleX = fScaleX;
		m_fTexScaleY = fScaleY;
	}

	inline void SetAlphaRef(DWORD dwAlpha)
	{
		/*if (m_dwAlpha != dwAlpha)
		{
			m_dwAlpha = dwAlpha;
			m_pD3DDev->SetRenderState(D3DRENDERSTATE_ALPHAREF, dwAlpha);
		}*/

	}
	void SetCycleType(DWORD dwCycleType)
	{
		m_dwCycleType = dwCycleType;
		// m_bBlendModeValid = FALSE;
	}

	void SetMux(DWORD dwMux0, DWORD dwMux1)
	{
		m_Mux = (((u64)dwMux0) << 32) | (u64)dwMux1;
		// m_bBlendModeValid = FALSE;
	}


	inline void SetFogMultOff(float fFogMult, float fFogOffset)
	{
		m_fFogMult = fFogMult;
		m_fFogOffset = fFogOffset;
		DL_PF("SetFogMultOff %f %f", fFogMult, fFogOffset);
		//DBGConsole_Msg(0, "SetFogMultOff %f %f", fFogMult, fFogOffset);
	}

	inline void SetFogEnable(BOOL bEnable)
	{
		m_bFogEnable = bEnable;
	}

	inline void SetFogColor(DWORD dwCol)
	{
		m_dwFogColor = dwCol;
	}

	inline void SetZmodeDecal(BOOL bZModeDecal)
	{
		m_bZModeDecal = bZModeDecal;
	}


// Bits
// +-+-+-
// xxyyzz
#define Z_NEG  0x01
#define Z_POS  0x02
#define Y_NEG  0x04
#define Y_POS  0x08
#define X_NEG  0x10
#define X_POS  0x20


	// Assumes dwAddress has already been checked!	
	// Don't inline - it's too big with the transform macros
	void SetNewVertexInfo(DWORD dwAddress, DWORD dwV0, DWORD dwNum)
	{
		LONG nFogR, nFogG, nFogB, nFogA;
		LONG nR, nG, nB, nA;

		if (m_bFogEnable)
		{
			nFogR = RGBA_GETRED(m_dwFogColor);
			nFogG = RGBA_GETGREEN(m_dwFogColor);
			nFogB = RGBA_GETBLUE(m_dwFogColor);
			nFogA = RGBA_GETALPHA(m_dwFogColor);
		}


		FiddledVtx * pVtxBase = (FiddledVtx*)(g_pu8RamBase + dwAddress);

		D3DXVECTOR4 vec4;

		D3DXMATRIX matWorldProject = m_mModelView[m_dwModelViewTop] * m_mProjection[m_dwProjectionTop];

		DWORD i;
		for (i = dwV0; i < dwV0 + dwNum; i++)
		{
			D3DXVECTOR3 w, wp;
			DWORD dwFlags;

			FiddledVtx & vert = pVtxBase[i - dwV0];

			w.x = (float)vert.x;
			w.y = (float)vert.y;
			w.z = (float)vert.z;

			D3DXVec3Transform(&vec4, &w, &matWorldProject);	// Convert to w=1
			m_fRHW[i] = 1.0f / vec4.w;

			m_vecProjected[i].x = vec4.x * m_fRHW[i];
			m_vecProjected[i].y = vec4.y * m_fRHW[i];
			m_vecProjected[i].z = vec4.z * m_fRHW[i];

			dwFlags = 0;
#if CLIPPING
			//if (m_vecProjected[i].z < 1)
			{
				D3DXVECTOR3 vecW1;

				if (m_fRHW[i] < 0)
					vecW1 = -m_vecProjected[i];
				else
					vecW1 = m_vecProjected[i];

				if (vecW1.x > 1.000f) dwFlags |= X_POS;
				else if (vecW1.x < -1.000f) dwFlags |= X_NEG;

				if (vecW1.y > 1.000f) dwFlags |= Y_POS;
				else if (vecW1.y < -1.000f) dwFlags |= Y_NEG;

				if (vecW1.z > 1.000f) dwFlags |= Z_POS;
				else if (vecW1.z < -1.000f) dwFlags |= Z_NEG;
			}
#endif

			m_dwVecFlags[i] = dwFlags;


			if (m_bLightingEnable)
			{
				D3DXVECTOR3 mn;		// modelnorm
				DWORD dwCol;
				D3DXMATRIX matWorld = m_mModelView[m_dwModelViewTop];

				mn.x = (float)vert.norma.nx;
				mn.y = (float)vert.norma.ny;
				mn.z = (float)vert.norma.nz;

				Vec3TransformNormal(mn, matWorld)
				m_vecTransformedNormal[i] = mn;

				// Do lighting 
				dwCol = LightVert(mn);

				nR = RGBA_GETRED(dwCol);
				nG = RGBA_GETGREEN(dwCol);
				nB = RGBA_GETBLUE(dwCol);
				nA = RGBA_GETALPHA(dwCol);
			}
			else
			{
				nR = vert.rgba.r;
				nG = vert.rgba.g;
				nB = vert.rgba.b;
				nA = vert.rgba.a;
			}

			/* I decided to take this out for v0.06b release - it's not quite
			// stable enough yet
			if (m_bFogEnable)
			{
				float fFog;
				float fEyespaceZ;

				fEyespaceZ = m_vecProjected[i].z;

				fFog = (fEyespaceZ * m_fFogMult) + m_fFogOffset;

				if (fFog < 0.0f)
					fFog = 0.0f;
				else if (fFog > 1.0f)
					fFog = 1.0f;

				//DBGConsole_Msg(0, "FogVal = %f", fFog);
				DL_PF("%d FogVal = %f", i, fFog);

				nR = ((1-fFog) * nR) + (fFog * nFogR);
				nG = ((1-fFog) * nG) + (fFog * nFogG);
				nB = ((1-fFog) * nB) + (fFog * nFogB);
				nA = ((1-fFog) * nA) + (fFog * nFogA);
			}*/

			// Assign true vert colour after lighting/fogging
			m_dwVecCol[i] = RGBA_MAKE(nR, nG, nB, nA);

			if (m_bTextureEnable)
			{
				// Update texture coords n.b. need to divide tu/tv by bogus scale on addition to buffer
				D3DXVECTOR2 & t = m_vecTexture[i];

				// If the vert is already lit, then there is no normal (and hence we
				// can't generate tex coord)
				if (m_bTextureGen && m_bLightingEnable)
				{
					D3DXMATRIX & matWV = m_mModelView[m_dwModelViewTop];
					D3DXVECTOR3 & norm = m_vecTransformedNormal[i];

					// Assign the spheremap's texture coordinates
					t.x = (0.5f * ( 1.0f + ( norm.x*matWV.m00 +
						                     norm.y*matWV.m10 +
											 norm.z*matWV.m20 ) ))/* * m_fTexScaleX*/;

					t.y = (0.5f * ( 1.0f - ( norm.x*matWV.m01 +
						                     norm.y*matWV.m11 +
											 norm.z*matWV.m21 ) ))/* * m_fTexScaleY*/;
				}
				else
				{
					t.x = (float)vert.tu * m_fTexScaleX;
					t.y = (float)vert.tv * m_fTexScaleY;
				}
			}
		}
		
	}


	// Assumes dwAddress has already been checked!	
	// Don't inline - it's too big with the transform macros
	// DKR seems to use longer vert info
	void SetNewVertexInfoDKR(DWORD dwAddress, DWORD dwV0, DWORD dwNum)
	{
		LONG nFogR, nFogG, nFogB, nFogA;
		LONG nR, nG, nB, nA;

		if (m_bFogEnable)
		{
			nFogR = RGBA_GETRED(m_dwFogColor);
			nFogG = RGBA_GETGREEN(m_dwFogColor);
			nFogB = RGBA_GETBLUE(m_dwFogColor);
			nFogA = RGBA_GETALPHA(m_dwFogColor);
		}


		s16 * pVtxBase = (s16*)(g_pu8RamBase + dwAddress);

		D3DXVECTOR4 vec4;

		D3DXMATRIX matWorldProject = m_mModelView[m_dwModelViewTop] * m_mProjection[m_dwProjectionTop];

		DWORD i;
		LONG nOff;

		nOff = 0;
		for (i = dwV0; i < dwV0 + dwNum; i++)
		{
			D3DXVECTOR3 w, wp;
			DWORD dwFlags;


			w.x = (float)pVtxBase[(nOff + 0) ^ 1];
			w.y = (float)pVtxBase[(nOff + 1) ^ 1];
			w.z = (float)pVtxBase[(nOff + 2) ^ 1];

			D3DXVec3Transform(&vec4, &w, &matWorldProject);	// Convert to w=1
			m_fRHW[i] = 1.0f / vec4.w;

			m_vecProjected[i].x = vec4.x * m_fRHW[i];
			m_vecProjected[i].y = vec4.y * m_fRHW[i];
			m_vecProjected[i].z = vec4.z * m_fRHW[i];

			dwFlags = 0;

			// Clip
			m_dwVecFlags[i] = dwFlags;

			s16 wA = pVtxBase[(nOff + 3) ^ 1];
			s16 wB = pVtxBase[(nOff + 4) ^ 1];

			s8 r = (s8)(wA >> 8);
			s8 g = (s8)(wA);
			s8 b = (s8)(wB >> 8);
			s8 a = (s8)(wB);

			if (m_bLightingEnable)
			{
				D3DXVECTOR3 mn;		// modelnorm
				DWORD dwCol;
				D3DXMATRIX matWorld = m_mModelView[m_dwModelViewTop];

				mn.x = (float)r; //norma.nx;
				mn.y = (float)g; //norma.ny;
				mn.z = (float)b; //norma.nz;

				Vec3TransformNormal(mn, matWorld)
				m_vecTransformedNormal[i] = mn;

				// Do lighting 
				dwCol = LightVert(mn);

				nR = RGBA_GETRED(dwCol);
				nG = RGBA_GETGREEN(dwCol);
				nB = RGBA_GETBLUE(dwCol);
				nA = RGBA_GETALPHA(dwCol);
			}
			else
			{
				nR = r;
				nG = g;
				nB = b;
				nA = a;
			}

			// Assign true vert colour after lighting/fogging
			m_dwVecCol[i] = RGBA_MAKE(nR, nG, nB, nA);

			/*if (m_bTextureEnable)
			{
				// Update texture coords n.b. need to divide tu/tv by bogus scale on addition to buffer
				D3DXVECTOR2 & t = m_vecTexture[i];

				// If the vert is already lit, then there is no normal (and hence we
				// can't generate tex coord)
				if (m_bTextureGen && m_bLightingEnable)
				{
					D3DXMATRIX & matWV = m_mModelView[m_dwModelViewTop];
					D3DXVECTOR3 & norm = m_vecTransformedNormal[i];

					// Assign the spheremap's texture coordinates
					t.x = (0.5f * ( 1.0f + ( norm.x*matWV.m00 +
						                     norm.y*matWV.m10 +
											 norm.z*matWV.m20 ) ))/* * m_fTexScaleX* /;

					t.y = (0.5f * ( 1.0f - ( norm.x*matWV.m01 +
						                     norm.y*matWV.m11 +
											 norm.z*matWV.m21 ) ))/* * m_fTexScaleY * /;
				}
				else
				{
					t.x = (float)vert.tu * m_fTexScaleX;
					t.y = (float)vert.tv * m_fTexScaleY;
				}
			}*/

			nOff += 5;
		}
		
	}


	void SetVtxTextureCoord(DWORD dwV, short tu, short tv)
	{
		m_vecTexture[dwV].x = (float)tu * m_fTexScaleX;
		m_vecTexture[dwV].y = (float)tv * m_fTexScaleY;
	}

	HRESULT TexRect(LONG nX0, LONG nY0, LONG nX1, LONG nY1, float fS0, float fT0, float fS1, float fT1);
	HRESULT TexRectFlip(LONG nX0, LONG nY0, LONG nX1, LONG nY1, float fS0, float fT0, float fS1, float fT1);
	HRESULT FillRect(LONG nX0, LONG nY0, LONG nX1, LONG nY1, DWORD dwColor);



	inline void SetProjection(const D3DXMATRIX & mat, BOOL bPush, LONG nLoadReplace) 
	{
		if (bPush)
		{
			if (m_dwProjectionTop >= (DAEDALUS_MATRIX_STACK-1))
				DBGConsole_Msg(0, "Pushing past proj stack limits!");
			else
				m_dwProjectionTop++;

			if (nLoadReplace == D3DRENDER_LOAD)
				// Load projection matrix
				m_mProjection[m_dwProjectionTop] = mat;
			else
				m_mProjection[m_dwProjectionTop] = mat * m_mProjection[m_dwProjectionTop-1];
			
		}
		else
		{
			if (nLoadReplace == D3DRENDER_LOAD)
				// Load projection matrix
				m_mProjection[m_dwProjectionTop] = mat;
			else
				m_mProjection[m_dwProjectionTop] = mat * m_mProjection[m_dwProjectionTop];

		}

	}

	inline void SetWorldView(const D3DXMATRIX & mat, BOOL bPush, LONG nLoadReplace)
	{

		// ModelView
		if (bPush)
		{
			if (m_dwModelViewTop >= (DAEDALUS_MATRIX_STACK-1))
				DBGConsole_Msg(0, "Pushing past modelview stack limits!");
			else
				m_dwModelViewTop++;

			// We should store the current projection matrix...
			if (nLoadReplace == D3DRENDER_LOAD)
			{
				// Load projection matrix
				m_mModelView[m_dwModelViewTop] = mat;
			}
			else			// Multiply projection matrix
			{
				m_mModelView[m_dwModelViewTop] = mat * m_mModelView[m_dwModelViewTop-1];
			}
		}
		else	// NoPush
		{
			if (nLoadReplace == D3DRENDER_LOAD)
			{
				// Load projection matrix
				m_mModelView[m_dwModelViewTop] = mat;
			}
			else
			{
				// Multiply projection matrix
				m_mModelView[m_dwModelViewTop] = mat * m_mModelView[m_dwModelViewTop];
			}

		}
	}

	inline void PopProjection()
	{
		if (m_dwProjectionTop > 0)
			m_dwProjectionTop--;
		//else
		//	DBGConsole_Msg(0, "Popping past projection stack limits");
	}

	inline void PopWorldView()
	{
		if (m_dwModelViewTop > 0)
			m_dwModelViewTop--;
		//else
		//	DBGConsole_Msg(0, "Popping past worldview stack limits");
	}

	void SetViewport(LONG nLeft, LONG nTop, LONG nRight, LONG nBottom)
	{
		m_nVPLeft = nLeft * m_fScreenMultX;
		m_nVPTop = nTop  * m_fScreenMultY;
		m_nVPWidth = (nRight - nLeft) * m_fScreenMultX;
		m_nVPHeight = (nBottom - nTop) * m_fScreenMultY;
	}

	// Returns TRUE if ok, FALSE if clipped
	BOOL TestTri(DWORD dwV0, DWORD dwV1, DWORD dwV2);
	void AddTri(DWORD dwV0, DWORD dwV1, DWORD dwV2);
	HRESULT FlushTris();

	void SetLightCol(DWORD dwLight, DWORD dwCol)
	{
		m_Lights[dwLight].col.r = (float)((dwCol >> 24)&0xFF);
		m_Lights[dwLight].col.g = (float)((dwCol >> 16)&0xFF);
		m_Lights[dwLight].col.b = (float)((dwCol >>  8)&0xFF);
		m_Lights[dwLight].col.a = 255.0f;	// Ignore light alpha
	}

	void SetLightDirection(DWORD dwLight, float x, float y, float z)
	{
		// Invert the x/y/z
		m_Lights[dwLight].dir.x = x;
		m_Lights[dwLight].dir.y = y;
		m_Lights[dwLight].dir.z = z;

		D3DXVec3Normalize(&m_Lights[dwLight].dir,
						  &m_Lights[dwLight].dir);

	}

	void SetNumLights(DWORD dwNumLights)
	{
		m_dwNumLights = dwNumLights;
	}


protected:
	LPDIRECT3DDEVICE7 m_pD3DDev;

	TSSInfo m_TSSInfo[8];

	float m_fScreenMultX;
	float m_fScreenMultY;

	LONG m_nVPLeft;
	LONG m_nVPTop;
	LONG m_nVPWidth;
	LONG m_nVPHeight;

	BOOL m_bTextureEnable;
	BOOL m_bLightingEnable;
	
	DWORD m_dwNumStages;

	DWORD m_dwAmbientLight;
	LONG  m_dwNumLights;
	//DWORD	m_dwPrimativeColor;
	DWORD m_dwTextureFactor;

	DWORD m_dwCycleType;
	u64 m_Mux;

	DWORD m_dwShadeMode;
	DWORD m_dwTexturePerspective;
	BOOL m_bAlphaTestEnable;

	BOOL m_bZBuffer;
	BOOL m_bZUpdate;
	BOOL m_bZCompare;

	DWORD m_dwrsZEnable;
	DWORD m_dwrsZWriteEnable;

	BOOL m_bCullFront;
	BOOL m_bCullBack;

	
	DWORD m_dwAlpha;

	float m_fFogMult;
	float m_fFogOffset;

	BOOL  m_bFogEnable;
	DWORD m_dwFogColor;
	DWORD m_dwPrimitiveColor;
	DWORD m_dwEnvColor;

	// Texutring
	LPDIRECTDRAWSURFACE7 m_lpsTexture;
	DaedalusLight m_Lights[8];

	DWORD m_dwTileWidth;
	DWORD m_dwTileHeight;
	float m_fTexWidth;
	float m_fTexHeight;		// Float to avoid converts when processing verts
	float m_fTexLeft;
	float m_fTexTop;		// 
	float m_fTexScaleX;
	float m_fTexScaleY;

	BOOL m_bTextureGen;			// Generate texture coords?
	BOOL m_bZModeDecal;			// Decal stuff?

	D3DXMATRIX m_mProjection[DAEDALUS_MATRIX_STACK];
	D3DXMATRIX m_mModelView[DAEDALUS_MATRIX_STACK];
	DWORD	  m_dwProjectionTop;
	DWORD	  m_dwModelViewTop;
		
	BYTE m_ucVertexBuffer[1000*32];
	DWORD m_dwNumVertices;

public:			// For DL_PF
	D3DXVECTOR3 m_vecProjected[32];
	float		m_fRHW[32];
	DWORD		m_dwVecFlags[32];			// Z_POS Z_NEG etc
	D3DXVECTOR3 m_vecTransformedNormal[32];
	D3DXVECTOR2 m_vecTexture[32];
	DWORD		m_dwVecCol[32];

	// Reset for a new frame
	inline void Reset()
	{
		ResetMatrices();
		m_dwNumVertices = 0;
	}

protected:

	inline void InitTLitVertex(TLITVERTEX & v, DWORD dwV);
	inline void InitLitVertex(LITVERTEX & v, DWORD dwV);

	DWORD LightVert(D3DXVECTOR3 & norm);

	
	//BOOL FindCombineMode(u64 mux, SetCombineInfo & sci);
	void InitBlendMode();
	
	HRESULT InitTUnlitVertices(DWORD dwMin, DWORD dwMax);
	HRESULT InitTLitVertices(DWORD dwMin, DWORD dwMax);
	
	HRESULT InitUnlitVertices(DWORD dwMin, DWORD dwMax);
	HRESULT InitLitVertices(DWORD dwMin, DWORD dwMax);
	

	inline void SetColorOp(DWORD dwStage, DWORD dwOp)
	{
		if (m_TSSInfo[dwStage].dwColorOp != dwOp)
		{
			m_TSSInfo[dwStage].dwColorOp = dwOp;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_COLOROP, dwOp );
		}
	}
	inline void SetColorArg1(DWORD dwStage, DWORD dwArg1)
	{
		if (m_TSSInfo[dwStage].dwColorArg1 != dwArg1 && dwArg1 != ~0)
		{
			m_TSSInfo[dwStage].dwColorArg1 = dwArg1;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_COLORARG1, dwArg1 );
		}
	}
	inline void SetColorArg2(DWORD dwStage, DWORD dwArg2)
	{
		if (m_TSSInfo[dwStage].dwColorArg2 != dwArg2 && dwArg2 != ~0)
		{
			m_TSSInfo[dwStage].dwColorArg2 = dwArg2;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_COLORARG2, dwArg2 );
		}
	}

	inline void SetAlphaOp(DWORD dwStage, DWORD dwOp)
	{
		if (m_TSSInfo[dwStage].dwAlphaOp != dwOp)
		{
			m_TSSInfo[dwStage].dwAlphaOp = dwOp;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_ALPHAOP, dwOp );
		}
	}

	inline void SetAlphaArg1(DWORD dwStage, DWORD dwArg1)
	{
		if (m_TSSInfo[dwStage].dwAlphaArg1 != dwArg1 && dwArg1 != ~0)
		{
			m_TSSInfo[dwStage].dwAlphaArg1 = dwArg1;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_ALPHAARG1, dwArg1 );
		}
	}

	inline void SetAlphaArg2(DWORD dwStage, DWORD dwArg2)
	{
		if (m_TSSInfo[dwStage].dwAlphaArg2 != dwArg2 && dwArg2 != ~0)
		{
			m_TSSInfo[dwStage].dwAlphaArg2 = dwArg2;
			m_pD3DDev->SetTextureStageState( dwStage, D3DTSS_ALPHAARG2, dwArg2 );
		}
	}

	inline void SetD3DRSZEnable(DWORD dwrsZEnable)
	{
		if (m_dwrsZEnable != dwrsZEnable)
		{
			m_dwrsZEnable = dwrsZEnable;
			m_pD3DDev->SetRenderState( D3DRENDERSTATE_ZENABLE, dwrsZEnable );	
		}
	}
	inline void SetD3DRSZWriteEnable(DWORD dwrsZWriteEnable)
	{
		if (m_dwrsZWriteEnable != dwrsZWriteEnable)
		{
			m_dwrsZWriteEnable = dwrsZWriteEnable;
			m_pD3DDev->SetRenderState( D3DRENDERSTATE_ZWRITEENABLE, dwrsZWriteEnable );	
		}
	}

public:
	// Init matrix stack to identity matrices
	void ResetMatrices()
	{
		D3DXMATRIX mat;

		D3DXMatrixIdentity(&mat);

		m_dwProjectionTop = 0;
		m_dwModelViewTop = 0;
		m_mProjection[0] = mat;
		m_mModelView[0] = mat;
	}

	void PrintActive()
	{
		if (g_hRDPDumpHandle != INVALID_HANDLE_VALUE)
		{
			D3DXMATRIX mat = m_mModelView[m_dwModelViewTop] * m_mProjection[m_dwProjectionTop];
			DL_PF(
				" %#+12.5f %#+12.5f %#+12.5f %#+12.5f\n"
				" %#+12.5f %#+12.5f %#+12.5f %#+12.5f\n"
				" %#+12.5f %#+12.5f %#+12.5f %#+12.5f\n"
				" %#+12.5f %#+12.5f %#+12.5f %#+12.5f\n",
				mat.m[0][0], mat.m[0][1], mat.m[0][2], mat.m[0][3],
				mat.m[1][0], mat.m[1][1], mat.m[1][2], mat.m[1][3],
				mat.m[2][0], mat.m[2][1], mat.m[2][2], mat.m[2][3],
				mat.m[3][0], mat.m[3][1], mat.m[3][2], mat.m[3][3]);
		}
	}

};



#endif // __DAEDALUS_D3DRENDER_H__