// GB Enhanced Copyright Daniel Baxter 2013
// Licensed under the GPLv2
// See LICENSE.txt for full license text

// File : mbc5.cpp
// Date : November 4, 2013
// Description : Game Boy Memory Bank Controller 5 I/O handling
//
// Handles reading and writing bytes to memory locations for MBC5
// Used to switch ROM and RAM banks in MBC5
// Also used for Rumble functionality

#include "mmu.h"

/****** Performs write operations specific to the MBC5 ******/
void MMU::mbc5_write(u16 address, u8 value)
{
	//Write to External RAM or RTC register
	if((address >= 0xA000) && (address <= 0xBFFF))
	{
		if(ram_banking_enabled) { random_access_bank[bank_bits][address - 0xA000] = value; }
	}

	//MBC register - Enable or Disable RAM Banking
	if(address <= 0x1FFF)
	{
		if((value & 0xF) == 0xA) 
		{ 
			if(cart_ram) { ram_banking_enabled = true; } 
		}
		else { ram_banking_enabled = false; }
	}

	//MBC register - Select ROM bank - Lower 8 bits
	if((address >= 0x2000) && (address <= 0x2FFF)) 
	{
		rom_bank &= ~0xFF;
		rom_bank |= value;
	}

	//MBC register - Select ROM Bank - High 9th bit
	if((address >= 0x3000) && (address <= 0x3FFF))
	{
		if((value & 0x1) == 1) { rom_bank |= 0x100; }
		else { rom_bank &= ~0x100; }
	}

	//MBC register - Select RAM bank
	if((address >= 0x4000) && (address <= 0x5FFF)) 
	{ 
		bank_bits = (value & 0xF);

		//Mirror RAM R-W to Bank 0 when using only 1 Bank
		//Most MBC5 games don't rely on mirroring, but DWI&II does
		//Nintendo specifically says to avoid it though (nice one Capcom...)
		if(memory_map[ROM_RAMSIZE] <= 2) { bank_bits = 0; }
	}
}

/****** Performs write operations specific to the MBC5 ******/
u8 MMU::mbc5_read(u16 address)
{
	//Read using ROM Banking
	if((address >= 0x4000) && (address <= 0x7FFF))
	{
		if(rom_bank >= 2) 
		{ 
			return read_only_bank[rom_bank - 2][address - 0x4000];
		}

		//When reading from Banks 0-1, just use the memory map
		else { return memory_map[address]; }
	}

	//Read using RAM Banking
	else if((address >= 0xA000) && (address <= 0xBFFF))
	{
		if(ram_banking_enabled) { return random_access_bank[bank_bits][address - 0xA000]; }
		else { return 0x00; }
	}
}