#ifndef MOBILE


// Declare Imports


   import java.net.*;
   import java.io.*;
   import java.util.*;



/** 
 *
 * Class to handle Internet Access required by the NESCafe NES Emulator.
 *
 * @author   David de Niese
 * @version  0.56f
 * @final    TRUE
 *
 */



public final class URLReader 
{

#ifndef APPLET

     /**
      *
      * Online Nintendo ROM File Unavailable
      *
      */

      protected static final int ERROR_NES_ROMLIST_UNAVAILABLE = 0x00;



     /**
      *
      * Online Content not Readable
      *
      */

      protected static final int ERROR_CONTENT_NOT_READABLE = 0x01;



     /**
      *
      * Online Content Manually Cancelled
      *
      */

      protected static final int ERROR_MANUALLY_CANCELLED = 0x02;


     /**
      *
      * Proxy Server Invalid
      *
      */

      protected static final int ERROR_PROXY_INVALID = 0x03;


     /**
      *
      * Proxy Server Authentication Invalid 
      *
      */

      protected static final int ERROR_PROXY_AUTHENTICATION_INVALID = 0x04;



     /**
      *
      * File Not Found
      *
      */

      protected static final int ERROR_FILE_NOT_FOUND = 0x06;



     /**
      *
      * Connection Error
      *
      */

      protected static final int ERROR_CONNECTION = 0x07;



     /**
      *
      * Protocol Error
      *
      */

      protected static final int ERROR_PROTOCOL_ERROR = 0x08;



     /**
      *
      * Unknown Error
      *
      */

      protected static final int ERROR_UNKNOWN = 0x0A;




     /**
      *
      * <P>The file to be downloaded.</P>
      *
      */

      private URL remoteFile; 


     /**
      *
      * <P>The description of the last recorded error.</P>
      *
      */

      private String lastError = "";


     /**
      *
      * <P>The number of the last recorded error.</P>
      *
      */

      private int lastErrorNum = 0; 


     /**
      *
      * <P>The number of bytes read from the current download.</P>
      *
      */

      private int contentIndex = 0; 


     /**
      *
      * <P>The content length of the current download.</P>
      *
      */

      private int contentLength = 0; 


     /**
      *
      * <P>The content as a String.</P>
      *
      */

      private String contentString; 


     /**
      *
      * <P>The content as a array of bytes.</P>
      *
      */

      private byte[] contentBuffer; 


     /**
      *
      * <P>Whether or not the URLReader should report its progress to the GUI.</P>
      *
      */

      private boolean reportOnProgress = false; 


     /**
      *
      * <P>Reference to the current GUI.</P>
      *
      */

      private GUI gui; 


     /**
      *
      * <P>Create a new URLReader object.</P>
      *
      * @param currentGUI The current NESCafe Graphical User Interface. 
      *
      */

      public URLReader(GUI gui) 
      {


         // Reference the current GUI

            this.gui = gui;
      
  
         // Initialise the Content Stream

            contentLength = 0; 
            contentIndex  = 0; 
            contentString = "";


         // Initialise the Error Recording

            lastError = "";
            lastErrorNum = 0; 


         // By Default don't Report Progress

            reportOnProgress = false; 

      }


     /**
      *
      * <P>Clears the current error status.</P>
      *
      */

      public final void clearErrorStatus() 
      {

         lastError    = "";
         lastErrorNum = 0; 

      }


     /**
      * 
      * <P>Download a file from the specified URL.</P>
      *
      * @param fileName The URL to download from. 
      *
      * @return Returns True if successfull
      *
      */

      public final boolean download(URL fileName) {



         // Grab Local Copy of Internet Settings

            boolean useProxyServer = gui.nes.usersettings.useProxyServer;


         // Grab Proxy Infomation

            String proxy, proxyPort, proxyUsername, proxyPassword;

            if (useProxyServer)
            {
               proxy = gui.nes.usersettings.proxyServer;
               proxyPort = gui.nes.usersettings.proxyServerPort;
               proxyUsername = gui.nes.usersettings.proxyServerUsername;
               proxyPassword = gui.nes.usersettings.proxyServerPassword;
            }
            else
            {
               proxy = "";
               proxyPort = "";
               proxyUsername = "";
               proxyPassword = "";
            }



         // Clear the Content Index Variable

            int contentIndex  = 0; 


         // Set and Record the Destination URL

            remoteFile = fileName; 


         // Keep Track of the Last Percentage Shown Reported

            int lastPercentShown = 0; 


         // Reset Error Information
            
            lastError = "";
            lastErrorNum = 0; 


         // Catch Exceptions

            try 
            {


               // Check if Proxy Authentication Required

                  if (useProxyServer)
                  {

                     // Set Username and Password

                        Authenticator.setDefault(new SimpleAuthenticator(proxyUsername,proxyPassword));


                     // Set Proxy Information

                        Properties systemProperties = System.getProperties();
                        systemProperties.setProperty("http.proxyHost",proxy);
                        systemProperties.setProperty("http.proxyPort",proxyPort);

                  }
                  else
                  {
 
                     // Clear Information

                        Authenticator.setDefault(new SimpleAuthenticator("",""));


                     // Clear Proxy Information

                        Properties systemProperties = System.getProperties();
                        systemProperties.setProperty("http.proxyHost","");
                        systemProperties.setProperty("http.proxyPort","");

                  }


               // Make Connection

                  URL webpage = remoteFile;
                  HttpURLConnection connection = (HttpURLConnection)webpage.openConnection();
                  connection.setUseCaches(false);
                  connection.connect();



               // Determine the Content Length

                  contentLength = connection.getContentLength();
                

               // Check that the Content Length is Reasonable

                  if (contentLength < 0) throw (new ConnectException());

                  if (contentLength == 0)
                  {

                     lastError = "The Nintendo ROM file is currently not available.";
                     lastErrorNum = URLReader.ERROR_NES_ROMLIST_UNAVAILABLE;
                     return false;

                  }



               // Create a Reference to the InputStream from the Remote File

                  InputStream input = connection.getInputStream();


               // Create the Content Buffer Large enough to Hold Content

                  contentBuffer = new byte[contentLength]; 


               // Buffer the Entire Content to the Array

                  for (int i = 0; i< contentLength; i++) {

                     // Read the Data Value

                        int data = input.read();

                        if (data == -1) 
                        { 

                           lastError = "Unexpected End of Input";
                           lastErrorNum = URLReader.ERROR_CONTENT_NOT_READABLE; 
                           return false; 

                        }


                     // Store in Buffer and Update Pointer

                        contentBuffer[i] = (byte)data; 
                        contentIndex = i; 


                     // Report on Progress if Request was Made

                        if (reportOnProgress) { 


                           // Calculate the Percentage

                              double percent = 100*((double)contentIndex)/((double)contentLength); 


                           // Check if Increment is enough to Warrant Displaying

                              if (percent >= (lastPercentShown+5)) {

                                 lastPercentShown = (int)percent; 
                                 String newText = "Downloading " + ((int)percent) + "% complete";
                                 gui.writeToScreen(newText); 

                              }


                           // Check if the User wants to Cancel the Download

                              if (gui.cancelDownload) 
                              {

                                 gui.writeToScreen("User Cancelled Download...");
                                 gui.serverMenu1.setEnabled(true);
                                 gui.serverMenu1.setEnabled(false);

                                 lastError = "The download has been manually cancelled";
                                 lastErrorNum = URLReader.ERROR_MANUALLY_CANCELLED; 
                                 input.close();
                                 return false; 

                              }

                        }

                  }


               // Close the Input Stream

                  input.close();


               // Create the Content String from the Content

                  contentString = new String(contentBuffer,0,contentLength); 


               // Signal a Successfull Download

                  return true; 



           } catch (NoRouteToHostException e) {

               lastError = "Proxy settings are invalid, proxy server or port is incorrect.";
               lastErrorNum = URLReader.ERROR_PROXY_INVALID; 
               return false; 

           } catch (ProtocolException e) {

               lastError = "Proxy settings are invalid, username and password incorrect.";
               lastErrorNum = URLReader.ERROR_PROXY_AUTHENTICATION_INVALID; 
               return false; 

           } catch (ConnectException myException) {

               lastError = "Could not establish connection to the Internet";
               lastErrorNum = URLReader.ERROR_CONNECTION; 
               return false; 

           } catch (FileNotFoundException myException) {

               lastError = "The Nintendo game could not be located";
               lastErrorNum = URLReader.ERROR_FILE_NOT_FOUND; 
               return false; 

           } catch (UnknownHostException myException) {

               lastError = "There is currently no connection to the Internet";
               lastErrorNum = URLReader.ERROR_CONNECTION; 
               return false; 

           } catch (MalformedURLException myException) {

               lastError = "URL has Invalid Protocol";
               lastErrorNum = URLReader.ERROR_PROTOCOL_ERROR; 
               return false; 
 
           } catch (SocketException myException) {

               lastError = "There is currently no connection to the Internet";
               lastErrorNum = URLReader.ERROR_CONNECTION;
               return false; 

           } catch (IOException myException) {

               lastError = myException.toString();
               lastErrorNum = URLReader.ERROR_UNKNOWN;
               return false; 

           } catch (Exception myException) {

               lastError = myException.toString();
               lastErrorNum = URLReader.ERROR_UNKNOWN;
               return false; 

           }

      }


     /**
      *
      * <P>Returns the content of the downloaded file.</P>
      *
      * @return Returns the content of the downloaded file as an array of bytes. 
      *
      */

      public final byte[] getBytes() {

         return contentBuffer; 

      }


     /**
      *
      * <P>Returns the content of the downloaded file.</P>
      *
      * @return Returns the content of the downloaded file as a String. 
      *
      */

      public final String getContent() {

         return contentString; 

      }


     /**
      *
      * <P>Returns the current URL.</P>
      *
      * @return The Returns the current URL as a String. 
      *
      */

      public final String getCurrentURL() {


         // Checks for a Valid URL Object

            if (remoteFile == null) return "";

 
         // Returns the URL Object as a String

            return remoteFile.toString();

      }


     /**
      * 
      * <P>Returns the last recorded error description.</P>
      *
      * @return The error description. 
      *
      */

      public final String getLastErrorString() { 

         return lastError; 

      }


     /**
      * 
      * <P>Returns the last recorded error number.</P>
      *
      * @return The error number. 
      *
      */

      public final int getLastErrorNumber() { 

        return lastErrorNum; 

      }


     /**
      * 
      * <P>Sets whether to report progress whilst downloading.</P>
      *
      * @param report True if you wish the URLReader to report progress to the GUI. 
      *
      */

      public final void setReportProgress(boolean report) {

         reportOnProgress = report; 

      }

#endif


}


#ifndef APPLET


/** 
 *
 * Authentication Class
 *
 * @author   David de Niese
 * @version  0.53
 * @final    TRUE
 *
 */

class SimpleAuthenticator extends Authenticator
{

	private String username;
	private String password;
                     

	public SimpleAuthenticator(String username,String password)
	{
		this.username = username;
		this.password = password;
	}
   
	protected PasswordAuthentication getPasswordAuthentication()
	{
		return new PasswordAuthentication(username,password.toCharArray());
	}

}


#endif

#endif
