#ifndef MOBILE


// Declare Imports

   import java.awt.*;
   import java.awt.image.*;
   import java.io.*;
   import java.util.jar.*;
   import java.net.*;


/** 
 *
 * Class for the TV Controller used by the NESCafe NES Emulator.
 *
 * @author   David de Niese
 * @version  0.56f
 * @final    TRUE
 *
 */

public final class TVController extends Frame implements Runnable 
{


     /**
      *
      * <P>Number of Frames drawn since last FPS Count.</P>
      *
      */

      private int actualFrameNum = 0;


     /**
      *
      * This Temp File must be Deleted on Virtual Machine Exit.
      *
      */

      private File filePause;


     /**
      *
      * This Temp File must be Deleted on Virtual Machine Exit.
      *
      */

      private File fileWait;


     /**
      *
      * <P>The current actual frame rate.</P>
      *
      */

      protected int frameRate = 0;


     /**
      *
      * <P>The current Frame Skip value.</P>
      *
      */
    
      protected int frameSkip = 0;


     /**
      *
      * <P>The current scanline number.</P>
      *
      */

      protected int scanLine = 0;


     /**
      *
      * <P>Screen Width.</P>
      *
      */

      protected static int screenWidth = 256;


     /**
      *
      * <P>Screen Height.</P>
      *
      */

      protected static int screenHeight = 240;


     /**
      *
      * <P>Whether FPS are currently being shown.</P>
      *
      */

      protected boolean showFPS = false;



     /**
      *
      * <P>The Screen Palette.</P>
      *
      */

      protected int palette[];



     /**
      *
      * <P>Buffer of Colour Entries for Screen.</P>
      *
      */

      protected int videoBuffer[] = new int
           [TVController.screenWidth * TVController.screenHeight];



     /**
      *
      * <P>The Off-Screen Image.</P>
      *
      */

      private Image offScreenImage;


     /**
      *
      * <P>The Graphics Object to paint the Image onto.</P>
      *
      */

      private Graphics screen;



     /**
      *
      * <P>The current Graphical User Interface.</P>
      *
      */

      private GUI gui;


     /**
      *
      * <P>The current NES Engine.</P>
      *
      */

      private NES nes;



     /** 
      *
      * The NESCafe Standard Logo as an Image
      *
      */

      private java.awt.Image nescafeImage;



#ifdef PUNCHOUT

     /** 
      *
      * The NESCafe Logo for Punchout Special Edition
      *
      */

      private java.awt.Image nescafeImagePO;


#endif


     /**
      *
      * <P>The "Please Wait" Image.</P>
      *
      */

      private java.awt.Image waitImage;


     /**
      *
      * <P>The "Paused" Image.</P>
      *
      */

      private java.awt.Image pauseImage;



     /**
      *
      * <P>The "Activate" Image.</P>
      *
      */

      private java.awt.Image activateImage;



     /**
      *
      * <P>The Frame Counter Thread.</P>
      *
      */

      private Thread counter;




     /**
      *
      * <P>True if the current frame is being skipped.</P>
      *
      */

      private boolean skipThisFrame = false;


     /**
      *
      * <P>The number of frames skipped since last frame draw.</P>
      *
      */

      private int framesSkipped = 0;



     /**
      *
      * <P>Whether the Screen is in Black and White Mode.</P>
      *
      */

      protected boolean BW = false;


     /**
      *
      * <P>The Percentage Tint for the Palette</P>
      *
      */

      protected float tint = 128.0f;


     /**
      *
      * <P>The Percentage Hue for the Palette</P>
      *
      */

      protected float hue = 128.0f;


     /**
      *
      * <P>The GUI Width Last Time we Drawed</P>
      *
      */

      private int guiLastW = 0;


     /**
      *
      * <P>The GUI Height Last Time we Drawed</P>
      *
      */

      private int guiLastH = 0;


     /**
      *
      * <P>The GUI X Last Time we Drawed (for LightGun)</P>
      *
      */

      protected int guiLastX = 0;


     /**
      *
      * <P>The GUI Y Last Time we Drawed (for LightGun)</P>
      *
      */

      protected int guiLastY = 0;



     /**
      *
      * <P>The Display Factor (for LightGun)</P>
      *
      */

      protected double tvFactor = 1;



     /**
      *
      * <P>Applet Graphics Context</P>
      *
      */

      private Graphics appletGraphics = null;



     /**
      *
      * Image Counter Latch Value
      *
      */

      protected int imageLatchCounter = 20;


     /**
      *
      * Counter to Allow Images to Flash
      *
      */

      public int imageCounter = imageLatchCounter;


     /**
      *
      * Counter to Allow Images to Flash
      *
      */

      private Image imageBuffer = null;
      
      
     /**
      *
      * The Message to Appear in the Status Bar
      *
      */
      
      private  String  statusMessage = "";


     /**
      *
      * Whether to remove State Bar message
      *
      */

      private  boolean statusBarOff = false;


      
     /**
      *
      * The Message to Appear in the Top Status Bar
      *
      */
      
      private  String  topstatusMessage = "";
      
      
      
     /**
      *
      * Whether to remove Top State Bar message
      *
      */

      private  boolean topstatusBarOff = false;


 
     /**
      *
      * Clean Screen Up when Zero
      *
      */

      private int cleanScreen = 0;



      
     /**
      *
      * <P>Create a new TVController Object.</P>
      *
      * @param NES The current NES Machine.
      * @param GUI The current Graphical User Interface.
      *
      */

      public TVController(NES nes, GUI gui) 
      {


         // Declare References to Current NES and GUI

            this.nes = nes;
            this.gui = gui;


         // Connect to the Screen (controlled by the GUI)

#ifdef STAND

            screen = gui.connect();

#else

            screen = null;

#endif



         // Create the Secondary Buffer (the Offscreen Image)

            int w = TVController.screenWidth;
            int h = TVController.screenHeight;

            offScreenImage = createImage(new MemoryImageSource(w, h, videoBuffer, 0, 256));


         // Set the Palette

            nes.palette.calcPalette(tint,hue,BW,0);
            palette = nes.palette.palette;



         // Start the Frames per Second Polling

            startFPSPolling();


         // Grab the Graphics

            try 
            {


               nescafeImage = loadImage("NESCafe.ddnb");
               nescafeImage.flush();

               waitImage = loadImage("Wait.ddnb");
               waitImage.flush();

               pauseImage = loadImage("Pause.ddnb");
               pauseImage.flush();

               activateImage = loadImage("Activate.ddnb");
               activateImage.flush();


#ifdef PUNCHOUT

               nescafeImagePO = loadImage("NESCafePO.ddnb");
               nescafeImagePO.flush();
               
#endif


            }
            catch (Exception e) {}



         // Clean Up

            System.gc();


      }




     /**
      *
      * Load an Image
      *
      */

      public final Image loadImage(String imgFile) throws Exception 
      {


         // The New Method Reads Directly from JAR File
         
            Image img = loadImageFromJar(imgFile);
        


         // Ensure Image is Fully Downloaded

            MediaTracker tracker = new MediaTracker(this);
            tracker.addImage(img, 0);
            tracker.waitForID(0);



         // Check for Errors

            if (tracker.isErrorID(0))
               return null;

            return img;


      }




     /**
      *
      * Load Image from JAR File
      *
      */

      public Image loadImageFromJar(String name) 
      {
         
         // Declare Input Stream
         
            DDNBInputStream in;
         
         
         
         // Declare Read Buffer
         
            byte[] _array;
            int _array_size;


         // Grab Toolkit
                  
            Toolkit toolkit = Toolkit.getDefaultToolkit();
         
         
         
         // Attempt to Read Content
         
            try 
            {
         
               // Grab InputStream to Resource
               
                  in = new DDNBInputStream(getClass().getResourceAsStream(name));
         
         
               // Configure Input Buffer
               
                  ByteArrayOutputStream bytes;
            
                  bytes = new ByteArrayOutputStream();
                  _array_size = 2048;
                  _array = new byte[_array_size];
            
            
               // Read Data into Byte Array
            
                  int rb;
            
                  while ((rb = in.read(_array, 0, _array_size)) > -1) 
                  {
                     bytes.write(_array, 0, rb);
                  }

                  bytes.close();
                  _array = bytes.toByteArray();
                  in.close();


               // Create Image

                  Image image = toolkit.createImage(_array);                  
                  return image;
                  
                  
            }
            catch (Exception e) 
            {
            	System.out.println(e);
               return null;
            }

      }
         

     /**
      *
      * <P>Blank the screen (show Waiting Screen).</P>
      *
      */

      public final void blankScreen() 
      {


         // Check that the GUI is valid

            if (gui == null || nes.memory.isNESCafeROM()) return;


         // Draw the Screen

            if (waitImage == null) return;
            waitImage.flush();
            drawImage(waitImage);  


      }




     /**
      *
      * <P>Blank the NESCafe Logo</P>
      *
      */

      public final void showNESCafeLogo() 
      {


         // Draw the Screen

#ifdef PUNCHOUT

            if (nescafeImagePO == null) return;
            nescafeImagePO.flush();
            drawImage(nescafeImagePO);  

#else

            if (nescafeImage == null) return;
            nescafeImage.flush();
            drawImage(nescafeImage);  

#endif


      }


     /**
      *
      * <P>Draw Blank Screen (useful for refreshing status when no ROM loaded)</P>
      *
      */

      public final void drawBlankScreen()
      {
         drawImage(null);
      }



     /**
      *
      * <P>Draw an Image File and Cater for Applet Mode></P>
      *
      */

      private final void drawImage(Image img)
      {

            
         try
         {
            
            // Declare Local Variables
   
               int h=256;
               int w=256;
               int x=0;
               int y=0;
               
               tvFactor = 1;
      

            // Get Display Dimensions

#ifdef APPLET


                  h = nes.applet.getDispHeight();
                  w = nes.applet.getDispWidth();
                     

#endif

#ifdef STAND
                     
   
                  h = gui.getDispHeight();
                  w = gui.getDispWidth();
   

#endif

   
   
               
            // Adjust Factor
            
               if (w > 320 && h > 320)
               {
               
                  double factorw = (w-20)/256.0;
                  double factorh = (h-20)/240.0;
                  
                  if (factorw > factorh)
                     tvFactor = factorh;
                  else
                     tvFactor = factorw;
                  
               }



            // Check Factor Limit
            
               if (tvFactor<1) tvFactor=1;
               if (tvFactor>2) tvFactor=2;
   
   

            // Determine if Screen Size Changed
            
               boolean sizeChanged = ((h != guiLastH) || (w != guiLastW));

               if (sizeChanged)
               {

                  // Determine how to Center the Screen on the GUI Panel
                     
                     x = w/2 - (int)(tvFactor * TVController.screenWidth )/2;
                     y = h/2 - (int)(tvFactor * TVController.screenHeight)/2;

                     guiLastX = x;
                     guiLastY = y;
                     
               }
               else
               {


                  // Copy Old Settings
                  
                     x = guiLastX;
                     y = guiLastY;
                     
               }

               

            // Check for Image Buffer (for mixing and matching)
            
               if (imageBuffer == null || sizeChanged)
               {

#ifdef APPLET

                     imageBuffer = nes.applet.createImage(w,h);
        
#endif
             
#ifdef STAND
                     
                     imageBuffer = nes.gui.createImage(w,h);
                     
#endif
                  
                  if (imageBuffer == null)
                     return;
                     
               }
         
   
   
   

            // Create if we Need to Super-impose Something
            
               boolean superimpose = sizeChanged || statusBarOff || topstatusBarOff || (statusMessage!=null && statusMessage!="") || (topstatusMessage!=null && topstatusMessage!="");
              
               if (superimpose)
               {



                  // Get Graphics Pointer
                  
                     Graphics g = imageBuffer.getGraphics();



                  // Clean the Screen Borders
                  
                     if (cleanScreen==0)
                     {
                     
                        g.setColor(Color.black);
                        g.fillRect(0,0,x,h);
                        g.fillRect(x,0,w-(2*x),y);
                        g.fillRect(w-x,0,x,h);
                        g.fillRect(x,h-y,w-(2*x),y);

                     }
                        
                        
                        

                  // Blank Screen if Size Changed
                 
                     if (sizeChanged || statusBarOff || topstatusBarOff || img == null)
                     {
                     
                        // Set Black to Blank
                        
                           g.setColor(Color.black);
                        
                        
                        // Blank Top
                        
                           if (statusBarOff)
                           {
                           
                              g.fillRect(0,h-16,w,16);
                              statusBarOff = false;
                              
                           }
                       
                       
                        // Blank Bottom
                        
                           if (topstatusBarOff)
                           {
                           
                              g.fillRect(0,0,w,16);
                              topstatusBarOff = false;
                              
                           }
                           
                     }
      
   
   
                  // Draw Image
                     
                     if (img != null)
                     {
                     
                        if (tvFactor == 1)
                        {
                        
                           if (!g.drawImage(img,x,y,Color.black,null))
                              g.drawImage(img,x,y,Color.black,this);
                           
                        }
                        else
                        {
                           
                           if (!g.drawImage(img,x,y,(int)(256 * tvFactor ),(int)(240 * tvFactor),Color.black,null))
                              g.drawImage(img,x,y,(int)(256 * tvFactor ),(int)(240 * tvFactor),Color.black,this);
                        
                        }
         
         
                     }
                     
   
   
                  // Write Status Message onto Screen
                  
                     if (statusMessage!=null && statusMessage!="")
                     {
                     
                        g.setColor(new Color(128,128,255));
                        g.fillRect(0,h - 16,w-1,15);
                        g.setColor(new Color(0,0,255));
                        g.drawRect(0,h - 16, w-1,15);
                     
                        g.setFont(new Font("Helvetica",Font.PLAIN,10));
                        g.setColor(new Color(0,0,128));
                        g.drawString(statusMessage,0+5,h - 4);
                     
                     }
         
         
                  // Write Status Message onto Screen
                  
                     if (topstatusMessage!=null && topstatusMessage!="")
                     {
                     
                        g.setColor(new Color(128,128,255));
                        g.fillRect(0,0,w-1,15);
                        g.setColor(new Color(0,0,255));
                        g.drawRect(0,0, w-1,15);
                     
                        g.setFont(new Font("Helvetica",Font.PLAIN,10));
                        g.setColor(new Color(0,0,128));
                        g.drawString(topstatusMessage,0+5,12);
                     
                     
                     }
         
         
         
                  // Dispose of Graphics Context
                  
                     g.dispose();



                  // Blitz the Screen

#ifdef APPLET

                     if (appletGraphics!=null)
                        if (!appletGraphics.drawImage(imageBuffer,0,0,Color.black,null))
                           appletGraphics.drawImage(imageBuffer,0,0,Color.black,this);

#endif


#ifdef STAND
                              
                     if (!screen.drawImage(imageBuffer,0,0,Color.black,null))
                        screen.drawImage(imageBuffer,0,0,Color.black,this);
                        
#endif


               }
               else
               {
                  
                  // No Super-Imposing

#ifdef APPLET

                     Graphics g = appletGraphics;
                     
#endif

#ifdef STAND
                     
                     Graphics g = screen;
                     
#endif

                     if (img != null && g != null)
                     {
                     
                        if (tvFactor == 1)
                        {
                        
                           if (!g.drawImage(img,x,y,Color.black,null))
                              g.drawImage(img,x,y,Color.black,this);
                           
                        }
                        else
                        {
                           
                           if (!g.drawImage(img,x,y,(int)(256 * tvFactor ),(int)(240 * tvFactor),Color.black,null))
                              g.drawImage(img,x,y,(int)(256 * tvFactor ),(int)(240 * tvFactor),Color.black,this);
                        
                        }
         
            
            
                     // Clean the Screen Borders
                     
                        if (cleanScreen==0)
                        {
                        
                           g.setColor(Color.black);
                           g.fillRect(0,0,x,h);
                           g.fillRect(x,0,w-(2*x),y);
                           g.fillRect(w-x,0,x,h);
                           g.fillRect(x,h-y,w-(2*x),y);
   
                        }
                                                   
         
                     }

                  
                  
               }
                                       
 
            // Decrease cleanScreen
            
               cleanScreen--;
               
               if (cleanScreen<0)
                  cleanScreen = 120;
                  
   
            // Record Last Width and Height When we Drawed
            
               guiLastH = h;
               guiLastW = w;
                              
               
         }
         catch (Exception e)
         {}
                  
      }



     /**
      * 
      * <P>Set Status Bar Message</P>
      *
      */

      public synchronized void setStatusBar(String message)
      {
         
         statusMessage = message;
         statusBarOff  = true;
      }
     
     
     /**
      * 
      * <P>Get Status Bar Message</P>
      *
      */

      public synchronized String getStatusBar()
      {
         
         return statusMessage;
      } 
      
      
      
     /**
      * 
      * <P>Set Top Status Bar Message</P>
      *
      */

      public synchronized void setTopStatusBar(String message)
      {
         
         topstatusMessage = message;
         topstatusBarOff  = true;
      }
     
        
     /**
      * 
      * <P>Get Top Status Bar Message</P>
      *
      */

      public synchronized String getTopStatusBar()
      {
         
         return topstatusMessage;
      } 
      
      

     /**
      * 
      * <P>Clear Pixel Information and Display Black Screen.</P>
      *
      */

      public final void deleteDisplay() 
      {


         // Check that the GUI is valid

            //if (gui == null) return;


         // Delete the Data in the Video Buffer

            for (int i = 0; i < videoBuffer.length; i++)
               videoBuffer[i] = 0xFF000000;


         // Flush the Object to Prepare it for Display

            offScreenImage.flush();
            drawImage(offScreenImage);


      }



     /**
      *
      * <P>Blitz the Screen to Canvas.</P>
      *
      */

      private final void blitzScreen() 
      {


         // Check for Applet Mode and No Focus

#ifdef APPLET

            if (!nes.applet.hasFocus && activateImage != null) 
            {

               if (imageCounter-- > 5)
               {

                  activateImage.flush();
                  drawImage(activateImage);

               }
               else
               {

                  nescafeImage.flush();
                  drawImage(nescafeImage);
                  if (imageCounter < 0) imageCounter = imageLatchCounter;

               }
               return;

            }

#endif


         // If in Paused Mode then Display Paused Screen

            if (nes.cpu.getCPUPause() && !nes.memory.isNESCafeROM()) 
            {

               if (pauseImage == null) 
               {

                  return;

               }

               if (imageCounter-- > 5)
               {

                  pauseImage.flush();
                  drawImage(pauseImage);

               }
               else
               {

                  nescafeImage.flush();
                  drawImage(nescafeImage);
                  if (imageCounter < 0) imageCounter = imageLatchCounter;

               }
               return;

            }


         // If in WaitMode then Draw "Please Wait" Screen

            if (gui.waitMode && !nes.memory.isNESCafeROM()) {

               if (waitImage == null)
               {
                  deleteDisplay();
               }
               else
               {
                  waitImage.flush();
                  drawImage(waitImage);
               }
               return;

            }


         // If no Cart is loaded then Blank the Screen

            if (!nes.isCartRunning()) 
            {

               deleteDisplay();
               return;

            }


         
         // Check if Halted (such as in NESCafe Punchout Special Edition)
         
            if (nes.cpu.isCPUHalted())
            {
               
               showNESCafeLogo();
               return;
               
            }
            

            
         // Draw the Screen from the Offscreen Buffer

            offScreenImage.flush();
			   drawImage(offScreenImage);


      }


     /**
      * 
      * <P>Draws the Screen.</P>
      *
      * @param force Forces a draw despite frame-skipping.
      *
      */

      public final void drawScreen(boolean force) 
      {

         if (!isSkipFrame() | force) blitzScreen();
         actualFrameNum++;

      }


     /**
      * 
      * <P>Grab Applet Graphics Context</P>
      *
      * @param force Forces a draw despite frame-skipping.
      *
      */


      public final void passAppletGraphics(Graphics g)
      {
		
         if (appletGraphics == null)
         {
            appletGraphics = g.create();
         }

      }



#ifdef STAND

     /**
      *
      * <P>Hides Frame per Second count.</P>
      *
      */

      public final void fpsHide() 
      {

         showFPS = false;
         gui.writeToTopBar("");
 
      }


     /**
      *
      * <P>Displays Frame per Second count.</P>
      *
      */

      public final void fpsShow() 
      {

         showFPS = true;
         gui.writeToTopBar("Calculating FPS...");
   
      }

#endif


     /**
      *
      * <P>Set the Scanline Manually</P>
      *
      */

      public final boolean setScanLineNum(int sl) 
      {


         // Set the Scanline

            scanLine = sl;


         // Check if we have reached a new Frame

            if (scanLine == 0) 
            {

               // Assume we are going to Skip and increment Counter

                  framesSkipped++;


               // Check if we should actually skip this Frame

                  if (framesSkipped <= frameSkip) 
                  {

                     skipThisFrame = true;

                  } 
                  else 
                  {

                     framesSkipped = 0;
                     skipThisFrame = false;

                  }

            }


         // Check for Palette Changes

            if (nes.palette.changed)
            {
               palette = nes.palette.palette;
               nes.palette.changed = false;
            }


            return skipThisFrame;



      }



     /**
      *
      * <P>Determine if the current frame is being skipped.</P>
      *
      * @return True if the frame is being skipped.
      *
      */

      private final boolean isSkipFrame() 
      {

         return skipThisFrame;

      }



     /**
      *
      * <P>The FPS Poller Thread.</P>
      *
      */

      public void run() 
      {

#ifdef STAND

         while (true) 
         {


            // Reset the Frame Number to 0
    
               actualFrameNum = 0;


            // Sleep for 5 seconds

               try 
               {

                  counter.sleep (5000);

               } catch (Exception e) {}



            // Calculate the Frame Rate

                if (showFPS)
                {

                   frameRate = (actualFrameNum)/5;
                   gui.writeToTopBar("Running at " + frameRate + " FPS");

                }

         }

#endif

      }


#ifdef STAND

     /**
      *
      * <P>Save a snapshot of the screen to file.</P>
      *
      */

      public void saveScreen() 
      {


         // Load SaveImage Class 

            try 
            {

               new SaveImage(offScreenImage);
               gui.writeToScreen("Screenshot saved...");
   
            } 
            catch (Exception e) 
            {

               gui.writeToScreen("Screenshot failed!");

            }


      }


#endif


     /**
      *
      * <P>Sets the frame skip for the TV Controller.</P>
      *
      * @param value The number of frames to skip before drawing.
      *
      */

      public final void setFrameSkip(int value) 
      {

#ifndef PUNCHOUT

         frameSkip = value;

#endif

      }



     /**
      *
      * <P>Set a pixel on Screen.</P>
      *
      * @param x The x-coordinate on the current scanline.
      * @param palEntry The palette entry number.
      *
      */
/*
      public final void setPixel(int x, int palEntry)  
      {

         // Set Pixel in Video Buffer

            videoBuffer[(scanLine << 8) | x] = palette[palEntry % 0x40];

      }

*/

     /**
      *
      * <P>Sets an Array of Pixels representing the current Scanline.</P>
      *
      */

      public final void setPixels(int [] palEntries) 
      {


         // Check Not First Line
         
            if (scanLine == 0)
               return;
               
               
         // Check Length

            if (palEntries.length < 256) 
               return;


         // Set Pixels

            for (int x = 0; x < 256; x++)
            {
                videoBuffer[(scanLine << 8) | x] = palette[palEntries[x]];
            }


      }



     /**
      *
      * <P>Returns a Pixel at the given Point.</P>
      *
      */

      public final int getPixel(int x, int y) 
      {


         // Determine the Offset

            int offset = (y<<8)|x;


         // Check Range

            if (offset >= videoBuffer.length) return 0;


         // Return the Pixel

            return videoBuffer[offset];

      }


     /**
      *
      * <P>Starts FPS Polling.</P>
      *
      */

      private final void startFPSPolling() 
      {

         counter = new Thread(this);
         counter.start();

      }




     /**
      *
      * <P>Increment the Tint.</P>
      *
      */

      public final void incTint() 
      {


         // Increment the Tint Value

            tint += 5;
            if (tint > 0xFF) tint = 0xFF;


         // Calculate the Palette

            nes.palette.calcPalette(tint,hue,BW,nes.memory.ppu.REG_2001);


         // Display the Selected Tint Value as a Percentage

            int percent = (int)((tint/256.0f) * 100);
            gui.writeToScreen("Tint = " + percent + "%" );


      }
      


     /**
      *
      * <P>Decrement the Tint.</P>
      *
      */

      public final void decTint() 
      {


         // Decrement the Tint Value

            tint -= 5;
            if (tint < 0) tint = 0x00;


         // Calculate the Palette

            nes.palette.calcPalette(tint,hue,BW,nes.memory.ppu.REG_2001);


         // Display the Selected Tint Value as a Percentage

            int percent = (int)((tint/256.0f) * 100);
            gui.writeToScreen("Tint = " + percent + "%" );

      }



     /**
      *
      * <P>Increment the Hue.</P>
      *
      */

      public final void incHue() 
      {


         // Increment the Hue Value

            hue += 5;
            if (hue > 0xFF) hue = 0xFF;


         // Calculate the Palette

            nes.palette.calcPalette(tint,hue,BW,nes.memory.ppu.REG_2001);


         // Display the Selected Hue Value as a Percentage

            int percent = (int)((hue/256.0f) * 100);
            gui.writeToScreen("Hue = " + percent + "%" );

      }



     /**
      *
      * <P>Decrement the Hue.</P>
      *
      */

      public final void decHue() 
      {


         // Decrement the Hue Value

            hue -= 5;
            if (hue < 0) hue = 0x00;


         // Calculate the Palette

            nes.palette.calcPalette(tint,hue,BW,nes.memory.ppu.REG_2001);


         // Display the Selected Hue Value as a Percentage

            int percent = (int)((hue/256.0f) * 100);
            gui.writeToScreen("Hue = " + percent + "%" );


      }



     /**
      *
      * <P>Toggle Black and White Mode.</P>
      *
      */

      public final void toggleBlackWhite() 
      {


         // Toggle Black and White Mode

            BW = !BW;


         // Calculate Palette

            nes.palette.calcPalette(tint,hue,BW,nes.memory.ppu.REG_2001);


         // Display Colour Mode

            gui.writeToScreen ( (BW ? "Black and White Mode" : "Colour Mode") );


      }



     /**
      *
      * <P>Image Observer Interface for Waiting for Dithering and Drawing of Images.</P>
      *
      */

      public boolean imageUpdate(Image img, int flags, int x, int y, int w, int h)
      {

         // Check if Image Content if Completely Ready Yet

            if ((flags & ALLBITS) != 0)
            {
                  drawImage(img);

            }
	
            return (flags & (ALLBITS | ABORT | ERROR)) == 0;

      }
      
      
      
#ifdef PUNCHOUT
#ifdef FUTURE


     /**
      *
      * Allow Punch Out Online Users to Post their Success Screens to PunchOut Server
      *
      */
      
      public void punchoutPostScreen()
      {
                     
            
            // Inform User of Intentions

               gui.writeToScreen("Attempting to submit screenshot to DavieBoy...");


            // Make TGA Image 
            
               byte[] buffer = SaveImage.makeTGA(offScreenImage);
            
            
            // Encode the Image as Hex
               
               StringBuffer a = new StringBuffer();
               
               for (int i=0 ; i<buffer.length; i++)
               {
                  a.append(DavOS.hex((int)(buffer[i]+256) & 0xFF,2));
               }
               
               String imageData = a.toString();
               
            
            // Construct Data

               String data = URLEncoder.encode("data", "UTF-8") + "=" + URLEncoder.encode(imageData, "UTF-8");

#ifdef APPLET

               data += "&" + URLEncoder.encode("user", "UTF-8") + "=" + URLEncoder.encode(nes.applet.username, "UTF-8");

#endif

               

               ServerConnection.postData(data);
               

         
      }


#endif
#endif



}

#endif